/*
 *  Platform depdenent initialization
 *  
 *  Copyright (C) 2009 Faraday Corp. (http://www.faraday-tech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/platform_device.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/spi/spi.h>
#include <linux/spi/flash.h>
#include <asm/setup.h>
#include <linux/module.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/mach-types.h>
#include <asm/mach/time.h>
#include <mach/spec.h>
#include <mach/ftintc010.h>
#include <mach/fttmr010.h>
#include <mach/platform/pmu.h>
#include <mach/ftpmu010.h>

static struct meminfo   mem_info, gmmem_info;
/******************************************************************************
 * platform devices
 * All IP module resource defined here
 *****************************************************************************/

/******************************************************************************
 * USB devices
 *****************************************************************************/
/* OTG:0 */ 
static struct resource fotg210_0_resources[] = {
	{
		.start  = USB_FOTG2XX_0_PA_BASE,
		.end    = USB_FOTG2XX_0_PA_LIMIT,
		.flags  = IORESOURCE_MEM,
	},
	{
		.start	= USB_FOTG2XX_0_IRQ,
		.end	= USB_FOTG2XX_0_IRQ,
		.flags	= IORESOURCE_IRQ,
	},
};

static u64 fotg210_0_dmamask = 0xFFFFFFUL;
static struct platform_device fotg210_0_device = {
	.name		= "fotg210",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(fotg210_0_resources),
	.resource	= fotg210_0_resources,
	.dev = {
		.dma_mask = &fotg210_0_dmamask,
		.coherent_dma_mask = 0xFFFFFFFF,
	},
};

/******************************************************************************
 * I2C devices
 *****************************************************************************/
/* i2c:0 */ 
static struct resource ftiic010_0_resources[] = {
	{
		.start  = I2C_FTI2C010_0_PA_BASE,
		.end    = I2C_FTI2C010_0_PA_LIMIT,
		.flags  = IORESOURCE_MEM,
	},
	{
		.start	= I2C_FTI2C010_0_IRQ,
		.end	= I2C_FTI2C010_0_IRQ,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device ftiic010_0_device = {
	.name		= "ftiic010",
	.id		    = 0,
	.num_resources	= ARRAY_SIZE(ftiic010_0_resources),
	.resource	= ftiic010_0_resources,
};

/*GPIO 0*/
static struct resource ftgpio010_0_resource[] = {
	{
		.start	= GPIO_FTGPIO010_PA_BASE,
		.end 	= GPIO_FTGPIO010_PA_LIMIT,
		.flags  = IORESOURCE_MEM
	},
	{
		.start	= GPIO_FTGPIO010_0_IRQ,
		.end	= GPIO_FTGPIO010_0_IRQ,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device ftgpio010_0_device = {
	.name	= "ftgpio010",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(ftgpio010_0_resource),
	.resource	= ftgpio010_0_resource
};

/*GPIO 1*/
static struct resource ftgpio010_1_resource[] = {
	{
		.start	= GPIO_FTGPIO010_1_PA_BASE,
		.end 	= GPIO_FTGPIO010_1_PA_LIMIT,
		.flags  = IORESOURCE_MEM
	},
	{
		.start	= GPIO_FTGPIO010_1_IRQ,
		.end	= GPIO_FTGPIO010_1_IRQ,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device ftgpio010_1_device = {
	.name	= "ftgpio010",
	.id		= 1,
	.num_resources	= ARRAY_SIZE(ftgpio010_1_resource),
	.resource	= ftgpio010_1_resource
};

/*GPIO 2 */
static struct resource ftgpio010_2_resource[] = {
	{
		.start	= GPIO_FTGPIO010_2_PA_BASE,
		.end 	= GPIO_FTGPIO010_2_PA_LIMIT,
		.flags  = IORESOURCE_MEM
	},
	{
		.start	= GPIO_FTGPIO010_2_IRQ,
		.end	= GPIO_FTGPIO010_2_IRQ,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device ftgpio010_2_device = {
	.name	= "ftgpio010",
	.id		= 2,
	.num_resources	= ARRAY_SIZE(ftgpio010_2_resource),
	.resource	= ftgpio010_2_resource
};

/* SPI 0 */
static struct resource ftssp010_spi_0_resources[] = {
    {
        .start  = SSP_FTSSP010_0_PA_BASE,
        .end    = SSP_FTSSP010_0_PA_LIMIT,
        .flags  = IORESOURCE_MEM,
    }, {
        .start  = SSP_FTSSP010_0_IRQ,
        .end    = SSP_FTSSP010_0_IRQ,
        .flags  = IORESOURCE_IRQ,
    },
};

static struct platform_device ftssp010_spi_0_device = {
    .name       = "ftssp010_spi",
    .id     = 0,
    .num_resources  = ARRAY_SIZE(ftssp010_spi_0_resources),
    .resource   = ftssp010_spi_0_resources,
};

/* SPI 1 */
static struct resource ftssp010_spi_1_resources[] = {
    {
        .start  = SSP_FTSSP010_1_PA_BASE,
        .end    = SSP_FTSSP010_1_PA_LIMIT,
        .flags  = IORESOURCE_MEM,
    }, {
        .start  = SSP_FTSSP010_1_IRQ,
        .end    = SSP_FTSSP010_1_IRQ,
        .flags  = IORESOURCE_IRQ,
    },
};

static struct platform_device ftssp010_spi_1_device = {
    .name       = "ftssp010_spi",
    .id     = 1,
    .num_resources  = ARRAY_SIZE(ftssp010_spi_1_resources),
    .resource   = ftssp010_spi_1_resources,
};

/* ****************************************************************************
 * array contains all platform devices
 * ****************************************************************************/ 
static struct platform_device *gm_devices[] __initdata = 
{
	/* OTG */
	&fotg210_0_device,
	/* I2C */
	&ftiic010_0_device,
	/* GPIO */
	&ftgpio010_0_device,
	&ftgpio010_1_device,
	&ftgpio010_2_device,
    /* SPI */
#ifdef CONFIG_SPI_FTSSP010    
    &ftssp010_spi_0_device,
#endif
#ifdef CONFIG_SPI_SECOND_IP
    &ftssp010_spi_1_device,
#endif
};

/******************************************************************************
 * timer - clockevent and clocksource
 *****************************************************************************/
static struct fttmr010_clockevent fttmr010_0_clockevent = {
	.clockevent = {
		.name	= "fttmr010:0",
		.irq	= TIMER_FTTMR010_0_IRQ0,
	},
	.base	= (void __iomem *)TIMER_FTTMR010_0_VA_BASE,
	.id	= 0,
};

static struct fttmr010_clocksource fttmr010_1_clocksource = {
	.clocksource = {	
		.name	= "fttmr010:1",
	},
	.base	= (void __iomem *)TIMER_FTTMR010_0_VA_BASE,
	.id	= 1,
};

static void __init gm_sys_timer_init(void)
{
	unsigned int pclk = pmu_get_apb_clk();
    
	fttmr010_0_clockevent.freq = pclk;
	fttmr010_clockevent_init(&fttmr010_0_clockevent);

	fttmr010_1_clocksource.freq = pclk;
	fttmr010_clocksource_init(&fttmr010_1_clocksource);
}

struct sys_timer gm_sys_timer = {
	.init	= gm_sys_timer_init,
};

/******************************************************************************
 * platform dependent functions
 *****************************************************************************/
extern const struct map_desc platform_io_desc[PLATFORM_IO_DESC_NR];

static void __init platform_map_io(void)
{        
    iotable_init((struct map_desc*)platform_io_desc, ARRAY_SIZE(platform_io_desc));
}

void __iomem *ftintc010_base_addr;

static void __init gm_init_irq(void)
{
	struct ftintc010_trigger_type master_trigger_type = {
		.irqmode	= PLATFORM_IRQ_TRIGGER_MODE2,
		.irqlevel	= ~PLATFORM_IRQ_TRIGGER_LEVEL2,
		.fiqmode	= PLATFORM_FIQ_TRIGGER_MODE2,
		.fiqlevel	= ~PLATFORM_FIQ_TRIGGER_LEVEL2,
#ifdef CONFIG_FTINTC010EX
        .irqmodeex	= PLATFORM_IRQ_TRIGGER_MODEEX2,
		.irqlevelex	= ~PLATFORM_IRQ_TRIGGER_LEVELEX2,
		.fiqmodeex	= PLATFORM_FIQ_TRIGGER_MODEEX2,
		.fiqlevelex	= ~PLATFORM_FIQ_TRIGGER_LEVELEX2,
#endif
	};

	/*
	 * initialize primary interrupt controller
	 */
	ftintc010_base_addr = __io(INTC_FTINTC010_0_VA_BASE);
	ftintc010_init(0, ftintc010_base_addr, 0, &master_trigger_type);
}

static struct mtd_partition partitions[] = {
    {
     .name = "Linux Section",           // for system usage
#ifdef CONFIG_VARFS_SUPPORT
	.offset = CONFIG_LINUX_PARTITION_ADDR,
	.size = CONFIG_LINUX_PARTTION_SIZE
#else
     .offset = 0xD6000,
     .size = 0xE28000					// 14.1 MB
#endif
     },
    {
     .name = "User Section",            // free for use
     .offset = 0xF80000,
     .size = MTDPART_SIZ_FULL},
    {
     .name = "Loader Section",
     .offset = 0x6000,
     .size = 0xA0000},
    {
     .name = "BurnIn Section",          // for system usage
     .offset = 0x6000,          // Burn-in ,start addr MTD_PA_BASE
     .size = 0xA0000},
    {
     .name = "UBoot Section",           // for system usage
     .offset = 0xA6000,         // start addr MTD_PA_BASE + 0x200000
     .size = 0x30000},
    {
     .name = "PIB Section",           
     .offset = 0xEFE000,
     .size = 0x2000},
    {
     .name = "DB Section",           
     .offset = 0xF00000,
     .size = 0x20000},
    {
     .name = "Log Section",           
     .offset = 0xF20000,
     .size = 0x20000},
    {
     .name = "DB_bak Section",           
     .offset = 0xF40000,
     .size = 0x20000},
    {
     .name = "Log_bak Section",           
     .offset = 0xF60000,
     .size = 0x20000},
	{
     .name = "full",
     .offset = 0,
     .size = MTDPART_SIZ_FULL},
#ifdef CONFIG_VARFS_SUPPORT
	 {
	 .name = "varfs Section",           
	 .offset = CONFIG_VARFS_PARTITION_ADDR,
	 .size = CONFIG_VARFS_PARTITION_SIZE},
#endif
};

static struct flash_platform_data spi_flash_platform_data = {
    .name = "wb_spi_flash",
    .parts = partitions,
    .nr_parts = ARRAY_SIZE(partitions)
};

#ifdef CONFIG_SECOND_SPI_FLASH
static struct mtd_partition extra_partitions[] = {
    {
     .name = "Extra Code Section",           
     .offset = 0x000000,
     .size = 0x800000           // 8MB
     },
    {
     .name = "Extra User Section",            // free for use
     .offset = 0x800000,
     .size = 0x800000},
};

static struct flash_platform_data extra_spi_flash_platform_data = {
    .name = "wb_spi_flash",
    .parts = extra_partitions,
    .nr_parts = ARRAY_SIZE(extra_partitions)
};
#endif//end of CONFIG_SECOND_SPI_FLASH

/**
 * Note: due to SSP's limitation, max_speed_hz can not be ove 40MHz
 */
static struct spi_board_info spi_devs_info[] __initdata = {
    {
        .modalias = "WINBOND_SPI_FLASH",
        .platform_data = &spi_flash_platform_data,
        .max_speed_hz = 15 * 1000 * 1000,   //for MX25L256
        .bus_num = 0,               //on bus 0
        .chip_select = 0,           //first chip select
        .mode = SPI_MODE_0,
    },
#ifdef CONFIG_SECOND_SPI_FLASH
    {
        .modalias = "WINBOND_SPI_FLASH",
        .platform_data = &extra_spi_flash_platform_data,
        .max_speed_hz = 15 * 1000 * 1000,   //for MX25L256
        .bus_num = 0,               //on bus 0
        .chip_select = 1,           //first chip select
        .mode = SPI_MODE_0,
    }
#endif
};

static void __init gm_board_init(void)
{    
    /* will invoke platform_device_register() to register all platform devices 
     */
	platform_add_devices(gm_devices, ARRAY_SIZE(gm_devices));
	
    /* add spi device here, when add corresponding spi driver, they will bind together
    */
    spi_register_board_info(spi_devs_info, ARRAY_SIZE(spi_devs_info));
}

/******************************************************************************
 * store memory information in parsing command line stage
 *****************************************************************************/
static inline void parse_early_mem(char **p)
{
	unsigned long size, start, order;
    int nr_bank;
    
	start = PHYS_OFFSET;
	size  = memparse(*p, p);
	/* This prevent some users reserved some memory space and cause the size is not 2^X */
	order = get_order(size);
	size = PAGE_SIZE << order;
	
	if (**p == '@')
		start = memparse(*p + 1, p);
    
    nr_bank = mem_info.nr_banks;
    mem_info.bank[nr_bank].start = start;
    mem_info.bank[nr_bank].size  = size;
    mem_info.bank[nr_bank].node = nr_bank;
    mem_info.nr_banks ++;
}

/*
 * UBOOT or Bootcmd line provides GM own memory parameter to parse frmmap memory size
 */
static inline void parse_early_gmmem(char **p)
{
	unsigned int size, nr_bank;
        
	size  = memparse(*p, p);
    nr_bank = gmmem_info.nr_banks;
        
    gmmem_info.bank[nr_bank].size  = size;
    gmmem_info.bank[nr_bank].node = nr_bank;

    gmmem_info.nr_banks ++;
}

static void __init platform_fixup_memory(struct machine_desc *desc, struct tag *tag,
                                         char **cmdline, struct meminfo *mi)
{
    char key[] = "mem=", *from, *command_line = NULL;
    struct tag  *t = tag;
    char gm_key[] = "gmmem=";
    int  i, tag_found = 0;
            
    memset(&mem_info, 0, sizeof(mem_info));    
    memset(&gmmem_info, 0, sizeof(gmmem_info));
    
    if (tag->hdr.tag == ATAG_CORE) {
        for (; t->hdr.size; t = tag_next(t)) {
            if (t->hdr.tag == ATAG_CMDLINE) {
                command_line = &t->u.cmdline.cmdline[0];
                from = command_line;
                tag_found = 1;
                break;
            }
        }
    }
    
    /* not found the boot argument parameters from UBOOT */
    if (!tag_found) {
        command_line = *cmdline;
        from = *cmdline;
    }
            
    for (i = 0; i < strlen(command_line) - strlen(key); i ++)
	{
		if (memcmp(from, key, strlen(key)) == 0)
		{
			from += strlen(key);
			parse_early_mem(&from);
		}
		else if (memcmp(from, gm_key, strlen(gm_key)) == 0)
		{
		    from += strlen(gm_key);
			parse_early_gmmem(&from);
		}
		else
			from ++;
	}
}

void platform_get_memory(struct meminfo **p_memory)
{
    *p_memory = &mem_info;
    return;
}

/*
 * UBOOT or Bootcmd line provides GM own memory parameter to parse frmmap memory size
 */
void platform_get_gmmemory(struct meminfo **p_memory)
{
    *p_memory = &gmmem_info;
    return;
}

MACHINE_START(GM, PLATFORM_NAME)
	.phys_io	    = PLATFORM_BOOTTIME_IO_PA_BASE, //uart io
	.io_pg_offst	= ((PLATFORM_BOOTTIME_IO_VA_BASE) >> 20) << 2,
	.boot_params	= BOOT_PARAMETER_PA_BASE,   //boot command line
	.map_io		    = platform_map_io,
	.init_irq	    = gm_init_irq,
	.timer		    = &gm_sys_timer,
	.fixup = platform_fixup_memory,
	.init_machine	= gm_board_init,
MACHINE_END
