/*
 * Buffered ssls io for ffmpeg system
 * Copyright (c) 2001 Fabrice Bellard
 *
 * This ssls is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include "libavutil/avstring.h"
#include "avformat.h"
#include <fcntl.h>
#if HAVE_SETMODE
#include <io.h>
#endif
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <stdlib.h>
#include "os_support.h"

#include <openssl/bio.h>
#include <openssl/ssl.h>
#include <openssl/err.h>

/* standard ssls protocol */

static int ssls_open(URLContext *h, const char *filename, int flags)
{
    av_strstart(filename, "ssls:", &filename);
    void *ssl = h->priv_data;
    if (ssl == NULL){
        return AVERROR(errno);
	}
    return 0;
}

static int ssls_read(URLContext *h, unsigned char *buf, int size)
{
	SSL *ssl = (SSL *) h->priv_data;
	int fd = SSL_get_fd(ssl);
	if(fd < 0){
		return -1;
	}
	int ret = 0;
	int retry = 0;
	do{
		fd_set rset;
        FD_ZERO(&rset);
        FD_SET(fd, &rset);
        struct timeval tval;
        tval.tv_sec = 1;
        tval.tv_usec = 0;
        ret = select(fd + 1,&rset, NULL, NULL, &tval);
        if(ret == 0){
            if(retry < 2){
                ++retry;
                continue;
            }else{
                return -1;
            }
        }else if(ret < 0){
            return -1;
        }else{
            if(!FD_ISSET(fd, &rset)){
                continue;
            }
        }
        ret = SSL_read(ssl, buf, size);
        if(ret < 0 && (errno == EAGAIN  || errno == EWOULDBLOCK || errno == EINTR)){
            continue;
        }
        break;
	}while(1);
	return ret;
}

static int ssls_write(URLContext *h, unsigned char *buf, int size)
{
	SSL *ssl = (SSL *) h->priv_data;
	int fd = SSL_get_fd(ssl);
	if(fd < 0){
		return -1;
	}
	int ret = 0;
	int hasSend = 0;
	int retry = 0;
	do{
	    fd_set wset;
		FD_ZERO(&wset);
		FD_SET(fd, &wset);
		struct timeval tval = {1, 0};
		ret = select(fd + 1, NULL, &wset, NULL, &tval);
		if(ret == 0){
            if(retry < 2){
                ++retry;
                continue;
            }else{
                return -1;
			}
		}else if(ret < 0){
			return -1;
		}else{
		    if(!FD_ISSET(fd, &wset)){
				continue;
			}
		}
		ret = SSL_write(ssl, buf, size);
		if(ret == size){
			break;
		}else if(ret >= 0 && ret < size){
			hasSend += ret;
			buf += ret;
			size -= ret;
			continue;
		}else{
			if(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR){
				continue;
			}
			return -1;
		}
	}while(1);
	return ret + hasSend;
}

static int ssls_get_handle(URLContext *h)
{
	SSL *ssl = (SSL *) h->priv_data;
    return SSL_get_fd(ssl);
}

URLProtocol ssls_protocol = {
    "ssls",
	ssls_open,
    ssls_read,
    ssls_write,
    .url_get_file_handle = ssls_get_handle,
};

