/*
  The oRTP library is an RTP (Realtime Transport Protocol - rfc3550) stack.
  Copyright (C) 2001  Simon MORLAT simon.morlat@linphone.org

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#if defined(_MSC_VER)  && (defined(WIN32) || defined(_WIN32_WCE))
#include "ortp-config-win32.h"
#else
#include "ortp-config.h"
#endif
#include "ortp/ortp.h"

#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#include "ortp/hrtp.h"

static int hrtp_sendto(RtpTransport * t, mblk_t * m, int flags, const struct sockaddr *to, socklen_t tolen)
{
	char *buf = (char *)t->data;
	int len = m->b_wptr - m->b_rptr;
	int total = len + 4;
	buf[2] = len >> 8;
	buf[3] = len & 0x00FF;
	memcpy(m->b_rptr - 4, buf, 4);

	buf = m->b_rptr - 4;
	int ret = 0;
	int retry = 0;
	int hasSpend = 0;
	int fd = t->session->rtp.outSocket;
	do{
		fd_set wset;
		FD_ZERO(&wset);
		FD_SET(fd, &wset);
		struct timeval tval;
		tval.tv_sec = 1;
		tval.tv_usec = 0;
		ret = select(fd + 1, NULL, &wset, NULL, &tval);
		if(ret == 0){
			if(retry < 2){
				++retry;
				continue;
			}else{
				return -1;
			}
		}else if(ret < 0){
			return -1;
		}else{
			if(!FD_ISSET(fd, &wset)){
				continue;
			}
		}
		ret = send(fd, buf, total, flags);
		if(ret == total){
			break;
		}else if(ret >= 0 && ret < total){
			hasSpend += ret;
			buf += ret;
			total -= ret;
		}else{
			if(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR){
				continue;
			}
			return -1;
		}
	}while(TRUE);
	return ret + hasSpend;
}

static int hrtp_recvfrom(RtpTransport * t, mblk_t * m, int flags, struct sockaddr *from, socklen_t * fromlen)
{
	int ret = 0;
	int retry = 0;
	int fd = t->session->rtp.socket;
	do{
		fd_set rset;
		FD_ZERO(&rset);
		FD_SET(fd, &rset);
		struct timeval tval;
		tval.tv_sec = 1;
		tval.tv_usec = 0;
		ret = select(fd + 1,&rset, NULL, NULL, &tval);
		if(ret == 0){
			if(retry < 2){
				++retry;
				continue;
			}else{
				return -1;
			}
		}else if(ret < 0){
			return -1;
		}else{
			if(!FD_ISSET(fd, &rset)){
				continue;
			}
		}
		ret = recv(fd, m->b_wptr, m->b_datap->db_lim - m->b_datap->db_base, flags);
		if(ret < 0 && (errno == EAGAIN  || errno == EWOULDBLOCK || errno == EINTR)){
			continue;
		}
		break;
	}while(TRUE);
	return ret;
}

static int hrtcp_sendto(RtpTransport * t, mblk_t * m, int flags, const struct sockaddr *to, socklen_t tolen)
{
	char *buf = (char *)t->data;
	int len = m->b_wptr - m->b_rptr;
	int total = len + 4;
	buf[2] = len >> 8;
	buf[3] = len & 0x00FF;
	memcpy(&buf[4], m->b_rptr, len);

	int ret = 0;
	int retry = 0;
	int hasSpend = 0;
	int fd = t->session->rtcp.outSocket;
	do{
		fd_set wset;
		FD_ZERO(&wset);
		FD_SET(fd, &wset);
		struct timeval tval;
		tval.tv_sec = 1;
		tval.tv_usec = 0;
		ret = select(fd + 1, NULL, &wset, NULL, &tval);
		if(ret == 0){
			if(retry < 2){
				++retry;
				continue;
			}else{
				return -1;
			}
		}else if(ret < 0){
			return -1;
		}else{
			if(!FD_ISSET(fd, &wset)){
				continue;
			}
		}
		ret = send(fd, buf, total, flags);
		if(ret == total){
			break;
		}else if(ret >= 0 && ret < total){
			hasSpend += ret;
			buf += ret;
			total -= ret;
		}else{
			if(errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR){
				continue;
			}
			return -1;
		}
	}while(TRUE);
	return ret + hasSpend;
}

static int hrtcp_recvfrom(RtpTransport * t, mblk_t * m, int flags, struct sockaddr *from, socklen_t * fromlen)
{
	int ret = 0;
	int retry = 0;
	int fd = t->session->rtcp.socket;
	do{
		fd_set rset;
		FD_ZERO(&rset);
		FD_SET(fd, &rset);
		struct timeval tval;
		tval.tv_sec = 1;
		tval.tv_usec = 0;
		ret = select(fd + 1,&rset, NULL, NULL, &tval);
		if(ret == 0){
			if(retry < 2){
				++retry;
				continue;
			}else{
				return -1;
			}
		}else if(ret < 0){
			return -1;
		}else{
			if(!FD_ISSET(fd, &rset)){
				continue;
			}
		}
		ret = recv(fd, m->b_wptr, m->b_datap->db_lim - m->b_datap->db_base, flags);
		if(ret < 0 && (errno == EAGAIN || errno == EWOULDBLOCK || errno == EINTR)){
			continue;
		}
		break;
	}while(TRUE);
	return ret;
}

ortp_socket_t hrtp_getsocket(RtpTransport * t)
{
	return t->session->rtp.socket;
}

ortp_socket_t hrtcp_getsocket(RtpTransport * t)
{
	return t->session->rtcp.socket;
}

/**
 * Creates a pair of Secure-RTP/Secure-RTCP RtpTransport's.
 * oRTP relies on libsrtp (see http://srtp.sf.net ) for secure RTP encryption.
 * This function creates a RtpTransport object to be used to the RtpSession using
 * rtp_session_set_transport().
 * @srtp: the srtp_t session to be used
 * 
**/
int hrtp_transport_rtp(char *buffer, RtpTransport *rtpt)
{
	if(rtpt){
	//(rtpt) = (RtpTransport *)malloc(sizeof(RtpTransport));//ortp_new(RtpTransport, 1);
		(rtpt)->data = buffer;
		(rtpt)->t_getsocket = hrtp_getsocket;
		(rtpt)->t_sendto = hrtp_sendto;
		(rtpt)->t_recvfrom = hrtp_recvfrom;
	}
	return 0;
}

int hrtp_transport_rtcp(char *buffer, RtpTransport *rtcpt)
{
	if(rtcpt){
		//(rtcpt) = (RtpTransport *)malloc(sizeof(RtpTransport));//ortp_new(RtpTransport, 1);
		(rtcpt)->data = buffer;
		(rtcpt)->t_getsocket = hrtcp_getsocket;
		(rtcpt)->t_sendto = hrtcp_sendto;
		(rtcpt)->t_recvfrom = hrtcp_recvfrom;
	}
	return 0;
}
