#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/string.h>
#include "../gpio/gpio.h"
#include <linux/delay.h>	//udelay(),mdelay()

#define PIN_DEV_DEF	// comment this definition in release version 
#define DEMO_MODE	// comment this definition in release version 
#define FONT_FUNC	// provide function to draw text 

//#define CONNECTOR_ON_TOP	// Connector on top or bottom 

// We try to normalize layout as: 
// (0,0)
//   +----------------------------+
//   |                            |
//   :                            :
//   +----------------------------+ (131,31)
//
// And each byte DB0~7 represent 8 top-down pixels in y-aix. 
// 

// GPIO PIN definition 
#ifdef PIN_DEV_DEF
	#define NT_PIN_RES		GPIO_ID(GPIO_PORT_A,3)	//output
	#define NT_PIN_A0		GPIO_ID(GPIO_PORT_A,2)	//output
	#define NT_PIN_WR		GPIO_ID(GPIO_PORT_A,4)	//output
	#define NT_PIN_DB0		GPIO_ID(GPIO_PORT_E,0)	//output
	#define NT_PIN_DB1		GPIO_ID(GPIO_PORT_E,1)	//output
	#define NT_PIN_DB2		GPIO_ID(GPIO_PORT_E,2)	//output
	#define NT_PIN_DB3		GPIO_ID(GPIO_PORT_E,3)	//output
	#define NT_PIN_DB4		GPIO_ID(GPIO_PORT_E,4)	//output
	#define NT_PIN_DB5		GPIO_ID(GPIO_PORT_E,6)	//output
	#define NT_PIN_DB6		GPIO_ID(GPIO_PORT_A,5)	//output
	#define NT_PIN_DB7		GPIO_ID(GPIO_PORT_D,7)	//output
#else
	#define NT_PIN_RES		GPIO_ID(GPIO_PORT_A,3)	//output
	#define NT_PIN_A0		GPIO_ID(GPIO_PORT_D,8)	//output
	#define NT_PIN_WR		GPIO_ID(GPIO_PORT_D,13)	//output
	#define NT_PIN_DB0		GPIO_ID(GPIO_PORT_E,0)	//output
	#define NT_PIN_DB1		GPIO_ID(GPIO_PORT_E,1)	//output
	#define NT_PIN_DB2		GPIO_ID(GPIO_PORT_E,2)	//output
	#define NT_PIN_DB3		GPIO_ID(GPIO_PORT_E,3)	//output
	#define NT_PIN_DB4		GPIO_ID(GPIO_PORT_E,4)	//output
	#define NT_PIN_DB5		GPIO_ID(GPIO_PORT_E,6)	//output
	#define NT_PIN_DB6		GPIO_ID(GPIO_PORT_D,0)	//output
	#define NT_PIN_DB7		GPIO_ID(GPIO_PORT_D,7)	//output
#endif

// NT7534 Command definition 
#define LCM_CMD_DISPLAY_ON			0xAF	// Display ON  
#define LCM_CMD_DISPLAY_OFF			0xAE	// Display OFF  
#define LCM_CMD_START_LINE			0x40	// Display start line set 
#define LCM_CMD_START_LINE_MASK		0x1F
#define LCM_CMD_SET_PAGE_ADDR		0xB0	// Page address set 
#define LCM_CMD_SET_PAGE_ADDR_MASK	0x0F
#define LCM_CMD_SET_COL_ADDR_HI		0x10	// Column address set up (upper bit)
#define LCM_CMD_SET_COL_ADDR_HI_MASK	0x0F
#define LCM_CMD_SET_COL_ADDR_LO		0x00	// Column address set up (lower bit)
#define LCM_CMD_SET_COL_ADDR_LO_MASK	0x0F
#define LCM_CMD_ADC_INC				0xA0	// ADC select (0->83h)
#define LCM_CMD_ADC_DEC				0xA1	// ADC select (83h->0)
#define LCM_CMD_REVERSE_DISPLAY_OFF	0xA6	// Normal/reverse display 
#define LCM_CMD_REVERSE_DISPLAY_ON	0xA7
#define LCM_CMD_ENTIRE_DISPLAY_ON	0xA4	// Entire display on/off 
#define LCM_CMD_ENTIRE_DISPLAY_OFF	0xA5 
#define LCM_CMD_BIAS_1_6			0xA2	// LCD bias set 
#define LCM_CMD_BIAS_1_5			0xA3
#define LCM_CMD_COMMON_MODE_NORMAL	0xC0	// Common output mode select 
#define LCM_CMD_COMMON_MODE_REVERSE	0xC8
#define LCM_CMD_POWER_CONTROL		0x28	// Power control set 
#define LCM_CMD_POWER_CONTROL_MASK	0x0F
#define LCM_CMD_REGULATOR_RATIO		0x20	// V0 voltage regulator internal resistor ration set 
#define LCM_CMD_REGULATOR_RATIO_MASK	0x07
#define LCM_CMD_ELEC_VOL_1ST_CMD	0x81	// Electronic volume mode set 
#define LCM_CMD_ELEC_VOL_2ND_MASK	0x3F


// NT7534 application definition 
#define LCM_PAGE_NUM	4		// 32 / 8 = 4 (But NT7534 can handle 65 bits)
//#define LCM_PAGE_NUM	9		// upper( 65 / 8 ) = 9 (the last page occupies D0 only)  
#define LCM_COLS_NUM	132

#ifdef FONT_FUNC
static const unsigned char eng_fonts[] = {
	/* This font is a part of Pcf2bdf, so make sure commerical usage is allowed. */
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x38, 0xFC, 0xFC, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0D, 0x0D, 0x00, 0x00, 0x00, 
	0x00, 0x0E, 0x1E, 0x00, 0x00, 0x1E, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x20, 0xF8, 0xF8, 0x20, 0xF8, 0xF8, 0x20, 0x00, 0x02, 0x0F, 0x0F, 0x02, 0x0F, 0x0F, 0x02, 0x00, 
	0x38, 0x7C, 0x44, 0x47, 0x47, 0xCC, 0x98, 0x00, 0x06, 0x0C, 0x08, 0x38, 0x38, 0x0F, 0x07, 0x00, 
	0x30, 0x30, 0x00, 0x80, 0xC0, 0x60, 0x30, 0x00, 0x0C, 0x06, 0x03, 0x01, 0x00, 0x0C, 0x0C, 0x00, 
	0x80, 0xD8, 0x7C, 0xE4, 0xBC, 0xD8, 0x40, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x07, 0x0F, 0x08, 0x00, 
	0x00, 0x10, 0x1E, 0x0E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0xF0, 0xF8, 0x0C, 0x04, 0x00, 0x00, 0x00, 0x00, 0x03, 0x07, 0x0C, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x04, 0x0C, 0xF8, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0C, 0x07, 0x03, 0x00, 0x00, 
	0x80, 0xA0, 0xE0, 0xC0, 0xC0, 0xE0, 0xA0, 0x80, 0x00, 0x02, 0x03, 0x01, 0x01, 0x03, 0x02, 0x00, 
	0x00, 0x80, 0x80, 0xE0, 0xE0, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x03, 0x03, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x1E, 0x0E, 0x00, 0x00, 0x00, 
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0C, 0x0C, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x80, 0xC0, 0x60, 0x30, 0x00, 0x0C, 0x06, 0x03, 0x01, 0x00, 0x00, 0x00, 0x00, 
	0xF0, 0xF8, 0x0C, 0xC4, 0x0C, 0xF8, 0xF0, 0x00, 0x03, 0x07, 0x0C, 0x08, 0x0C, 0x07, 0x03, 0x00, 
	0x00, 0x10, 0x18, 0xFC, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x0F, 0x0F, 0x08, 0x08, 0x00, 
	0x08, 0x0C, 0x84, 0xC4, 0x64, 0x3C, 0x18, 0x00, 0x0E, 0x0F, 0x09, 0x08, 0x08, 0x0C, 0x0C, 0x00, 
	0x08, 0x0C, 0x44, 0x44, 0x44, 0xFC, 0xB8, 0x00, 0x04, 0x0C, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0xC0, 0xE0, 0xB0, 0x98, 0xFC, 0xFC, 0x80, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 
	0x7C, 0x7C, 0x44, 0x44, 0x44, 0xC4, 0x84, 0x00, 0x04, 0x0C, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0xF0, 0xF8, 0x4C, 0x44, 0x44, 0xC0, 0x80, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0x0C, 0x0C, 0x04, 0x84, 0xC4, 0x7C, 0x3C, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x00, 
	0xB8, 0xFC, 0x44, 0x44, 0x44, 0xFC, 0xB8, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0x38, 0x7C, 0x44, 0x44, 0x44, 0xFC, 0xF8, 0x00, 0x00, 0x08, 0x08, 0x08, 0x0C, 0x07, 0x03, 0x00, 
	0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0E, 0x06, 0x00, 0x00, 0x00, 
	0x00, 0x80, 0xC0, 0x60, 0x30, 0x18, 0x08, 0x00, 0x00, 0x00, 0x01, 0x03, 0x06, 0x0C, 0x08, 0x00, 
	0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x00, 
	0x00, 0x08, 0x18, 0x30, 0x60, 0xC0, 0x80, 0x00, 0x00, 0x08, 0x0C, 0x06, 0x03, 0x01, 0x00, 0x00, 
	0x18, 0x1C, 0x04, 0xC4, 0xE4, 0x3C, 0x18, 0x00, 0x00, 0x00, 0x00, 0x0D, 0x0D, 0x00, 0x00, 0x00, 
	0xF0, 0xF8, 0x08, 0xC8, 0xC8, 0xF8, 0xF0, 0x00, 0x07, 0x0F, 0x08, 0x0B, 0x0B, 0x0B, 0x01, 0x00, 
	0xE0, 0xF0, 0x98, 0x8C, 0x98, 0xF0, 0xE0, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0x04, 0xFC, 0xFC, 0x44, 0x44, 0xFC, 0xB8, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0xF0, 0xF8, 0x0C, 0x04, 0x04, 0x0C, 0x18, 0x00, 0x03, 0x07, 0x0C, 0x08, 0x08, 0x0C, 0x06, 0x00, 
	0x04, 0xFC, 0xFC, 0x04, 0x0C, 0xF8, 0xF0, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x0C, 0x07, 0x03, 0x00, 
	0x04, 0xFC, 0xFC, 0x44, 0xE4, 0x0C, 0x1C, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x08, 0x0C, 0x0E, 0x00, 
	0x04, 0xFC, 0xFC, 0x44, 0xE4, 0x0C, 0x1C, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 0x00, 0x00, 
	0xF0, 0xF8, 0x0C, 0x84, 0x84, 0x8C, 0x98, 0x00, 0x03, 0x07, 0x0C, 0x08, 0x08, 0x07, 0x0F, 0x00, 
	0xFC, 0xFC, 0x40, 0x40, 0x40, 0xFC, 0xFC, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0x00, 0x00, 0x04, 0xFC, 0xFC, 0x04, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x04, 0xFC, 0xFC, 0x04, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x0F, 0x07, 0x00, 0x00, 
	0x04, 0xFC, 0xFC, 0xC0, 0xE0, 0x3C, 0x1C, 0x00, 0x08, 0x0F, 0x0F, 0x00, 0x01, 0x0F, 0x0E, 0x00, 
	0x04, 0xFC, 0xFC, 0x04, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x08, 0x0C, 0x0E, 0x00, 
	0xFC, 0xFC, 0x38, 0x70, 0x38, 0xFC, 0xFC, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0xFC, 0xFC, 0x38, 0x70, 0xE0, 0xFC, 0xFC, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0xF8, 0xFC, 0x04, 0x04, 0x04, 0xFC, 0xF8, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0x04, 0xFC, 0xFC, 0x44, 0x44, 0x7C, 0x38, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 0x00, 0x00, 
	0xF8, 0xFC, 0x04, 0x04, 0x04, 0xFC, 0xF8, 0x00, 0x07, 0x0F, 0x08, 0x0E, 0x3C, 0x3F, 0x27, 0x00, 
	0x04, 0xFC, 0xFC, 0x44, 0xC4, 0xFC, 0x38, 0x00, 0x08, 0x0F, 0x0F, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0x18, 0x3C, 0x64, 0x44, 0xC4, 0x9C, 0x18, 0x00, 0x06, 0x0E, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0x00, 0x1C, 0x0C, 0xFC, 0xFC, 0x0C, 0x1C, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 
	0xFC, 0xFC, 0x00, 0x00, 0x00, 0xFC, 0xFC, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0xFC, 0xFC, 0x00, 0x00, 0x00, 0xFC, 0xFC, 0x00, 0x01, 0x03, 0x06, 0x0C, 0x06, 0x03, 0x01, 0x00, 
	0xFC, 0xFC, 0x00, 0xC0, 0x00, 0xFC, 0xFC, 0x00, 0x07, 0x0F, 0x0E, 0x03, 0x0E, 0x0F, 0x07, 0x00, 
	0x0C, 0x3C, 0xF0, 0xE0, 0xF0, 0x3C, 0x0C, 0x00, 0x0C, 0x0F, 0x03, 0x01, 0x03, 0x0F, 0x0C, 0x00, 
	0x00, 0x3C, 0x7C, 0xC0, 0xC0, 0x7C, 0x3C, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 
	0x1C, 0x0C, 0x84, 0xC4, 0x64, 0x3C, 0x1C, 0x00, 0x0E, 0x0F, 0x09, 0x08, 0x08, 0x0C, 0x0E, 0x00, 
	0x00, 0x00, 0xFC, 0xFC, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x08, 0x08, 0x00, 0x00, 
	0x38, 0x70, 0xE0, 0xC0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x03, 0x07, 0x0E, 0x00, 
	0x00, 0x00, 0x04, 0x04, 0xFC, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x0F, 0x0F, 0x00, 0x00, 
	0x08, 0x0C, 0x06, 0x03, 0x06, 0x0C, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 
	0x00, 0x00, 0x02, 0x06, 0x0C, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0xA0, 0xA0, 0xA0, 0xE0, 0xC0, 0x00, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x07, 0x0F, 0x08, 0x00, 
	0x04, 0xFC, 0xFC, 0x20, 0x60, 0xC0, 0x80, 0x00, 0x00, 0x0F, 0x0F, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0xC0, 0xE0, 0x20, 0x20, 0x20, 0x60, 0x40, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0C, 0x04, 0x00, 
	0x80, 0xC0, 0x60, 0x24, 0xFC, 0xFC, 0x00, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x07, 0x0F, 0x08, 0x00, 
	0xC0, 0xE0, 0xA0, 0xA0, 0xA0, 0xE0, 0xC0, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0C, 0x04, 0x00, 
	0x00, 0x40, 0xF8, 0xFC, 0x44, 0x0C, 0x18, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 0x00, 
	0xC0, 0xE0, 0x20, 0x20, 0xC0, 0xE0, 0x20, 0x00, 0x27, 0x6F, 0x48, 0x48, 0x7F, 0x3F, 0x00, 0x00, 
	0x04, 0xFC, 0xFC, 0x40, 0x20, 0xE0, 0xC0, 0x00, 0x08, 0x0F, 0x0F, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0x00, 0x00, 0x20, 0xEC, 0xEC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x20, 0xEC, 0xEC, 0x00, 0x00, 0x30, 0x70, 0x40, 0x40, 0x7F, 0x3F, 0x00, 
	0x04, 0xFC, 0xFC, 0x80, 0xC0, 0x60, 0x20, 0x00, 0x08, 0x0F, 0x0F, 0x01, 0x03, 0x0E, 0x0C, 0x00, 
	0x00, 0x00, 0x04, 0xFC, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 
	0xE0, 0xE0, 0x60, 0xC0, 0x60, 0xE0, 0xC0, 0x00, 0x0F, 0x0F, 0x00, 0x07, 0x00, 0x0F, 0x0F, 0x00, 
	0x20, 0xE0, 0xC0, 0x20, 0x20, 0xE0, 0xC0, 0x00, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x0F, 0x0F, 0x00, 
	0xC0, 0xE0, 0x20, 0x20, 0x20, 0xE0, 0xC0, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x08, 0x0F, 0x07, 0x00, 
	0x20, 0xE0, 0xC0, 0x20, 0x20, 0xE0, 0xC0, 0x00, 0x40, 0x7F, 0x7F, 0x48, 0x08, 0x0F, 0x07, 0x00, 
	0xC0, 0xE0, 0x20, 0x20, 0xC0, 0xE0, 0x20, 0x00, 0x07, 0x0F, 0x08, 0x48, 0x7F, 0x7F, 0x40, 0x00, 
	0x20, 0xE0, 0xC0, 0x60, 0x20, 0xE0, 0xC0, 0x00, 0x08, 0x0F, 0x0F, 0x08, 0x00, 0x00, 0x00, 0x00, 
	0x40, 0xE0, 0xA0, 0x20, 0x20, 0x60, 0x40, 0x00, 0x04, 0x0C, 0x09, 0x09, 0x0B, 0x0E, 0x04, 0x00, 
	0x20, 0x20, 0xF8, 0xFC, 0x20, 0x20, 0x00, 0x00, 0x00, 0x00, 0x07, 0x0F, 0x08, 0x0C, 0x04, 0x00, 
	0xE0, 0xE0, 0x00, 0x00, 0xE0, 0xE0, 0x00, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x07, 0x0F, 0x08, 0x00, 
	0xE0, 0xE0, 0x00, 0x00, 0x00, 0xE0, 0xE0, 0x00, 0x03, 0x07, 0x0C, 0x08, 0x0C, 0x07, 0x03, 0x00, 
	0xE0, 0xE0, 0x00, 0x80, 0x00, 0xE0, 0xE0, 0x00, 0x07, 0x0F, 0x0C, 0x07, 0x0C, 0x0F, 0x07, 0x00, 
	0x20, 0x60, 0xC0, 0x80, 0xC0, 0x60, 0x20, 0x00, 0x08, 0x0C, 0x07, 0x03, 0x07, 0x0C, 0x08, 0x00, 
	0xE0, 0xE0, 0x00, 0x00, 0x00, 0xE0, 0xE0, 0x00, 0x47, 0x4F, 0x48, 0x48, 0x68, 0x3F, 0x1F, 0x00, 
	0x60, 0x60, 0x20, 0xA0, 0xE0, 0x60, 0x20, 0x00, 0x0C, 0x0E, 0x0B, 0x09, 0x08, 0x0C, 0x0C, 0x00, 
	0x00, 0x40, 0x40, 0xF8, 0xBC, 0x04, 0x04, 0x00, 0x00, 0x00, 0x00, 0x07, 0x0F, 0x08, 0x08, 0x00, 
	0x00, 0x00, 0x00, 0xFC, 0xFC, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x0F, 0x00, 0x00, 0x00, 
	0x00, 0x04, 0x04, 0xBC, 0xF8, 0x40, 0x40, 0x00, 0x00, 0x08, 0x08, 0x0F, 0x07, 0x00, 0x00, 0x00, 
	0x04, 0x06, 0x02, 0x06, 0x04, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 	
};
#endif 

// internal functions 
static inline void nt7534_udelay( unsigned long delay )
{
	// In our experimental
	// 1  ... ok 
	// 2  ... ok
	// 3  ... ok (long time) 
	// 4  ... ok
	// 5  ...  
	// 10 ... ok
	// 20 ... 
	
	if( delay < 1 )
		udelay( 1 );
	else
		udelay( delay );
}

static inline void nt7534_mdelay( unsigned long delay )
{
	mdelay( delay );
}

static inline void nt7534_nsdelay( unsigned long delay )
{
	nt7534_udelay( delay / 1000 );	// convert ns to us 
}

static void nt7534_init_lcm_gpio( void )
{
	_rtl8186_initGpioPin(NT_PIN_RES, GPIO_DIR_OUT, GPIO_INT_DISABLE);
	_rtl8186_initGpioPin(NT_PIN_A0, GPIO_DIR_OUT, GPIO_INT_DISABLE);
	_rtl8186_initGpioPin(NT_PIN_WR, GPIO_DIR_OUT, GPIO_INT_DISABLE);
	_rtl8186_initGpioPin(NT_PIN_DB0, GPIO_DIR_OUT, GPIO_INT_DISABLE);
	_rtl8186_initGpioPin(NT_PIN_DB1, GPIO_DIR_OUT, GPIO_INT_DISABLE);
	_rtl8186_initGpioPin(NT_PIN_DB2, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	_rtl8186_initGpioPin(NT_PIN_DB3, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	_rtl8186_initGpioPin(NT_PIN_DB4, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	_rtl8186_initGpioPin(NT_PIN_DB5, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	_rtl8186_initGpioPin(NT_PIN_DB6, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	_rtl8186_initGpioPin(NT_PIN_DB7, GPIO_DIR_OUT, GPIO_INT_DISABLE);	
	
	_rtl8186_setGpioDataBit( NT_PIN_WR, 1 );	// pull high WR 
}

static void nt7534_reset( void )
{
	_rtl8186_setGpioDataBit( NT_PIN_RES, 0 );	// pull low RES 
	
	//nt7534_udelay( 1 );	// delay 1us  
	nt7534_mdelay( 20 );	// delay 20ms 
	
	_rtl8186_setGpioDataBit( NT_PIN_RES, 1 );	// pull high RES 
	
	nt7534_udelay( 1 );	// delay 1us  
}

static inline void nt7534_write_DB0_to_DB7( unsigned char DB )
{
	_rtl8186_setGpioDataBit( NT_PIN_DB0, ( ( DB & 0x01 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB1, ( ( DB & 0x02 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB2, ( ( DB & 0x04 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB3, ( ( DB & 0x08 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB4, ( ( DB & 0x10 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB5, ( ( DB & 0x20 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB6, ( ( DB & 0x40 ) ? 1 : 0 ) );
	_rtl8186_setGpioDataBit( NT_PIN_DB7, ( ( DB & 0x80 ) ? 1 : 0 ) );
}

static void nt7534_write_data_core( int DB_type, unsigned char DB )
{
	// Now, its bus timing is for 8080 series MPU (parallel mode) 
	if( DB_type )
		_rtl8186_setGpioDataBit( NT_PIN_A0, 1 );	// display data 
	else
		_rtl8186_setGpioDataBit( NT_PIN_A0, 0 );	// control data 

	_rtl8186_setGpioDataBit( NT_PIN_WR, 0 );		// pull low WR 

	nt7534_write_DB0_to_DB7( DB );					// write DB0 ~ DB7 

	nt7534_nsdelay( 40 );	// tDS8 = 40ns 
	
	_rtl8186_setGpioDataBit( NT_PIN_WR, 1 );		// pull high WR 
	
	nt7534_nsdelay( 10 );	// tDH8 = 10ns 
}

static inline void nt7534_write_control_data( unsigned char cmd )
{
	nt7534_write_data_core( 0 /* control data */, cmd );
}

static inline void nt7534_write_display_data( unsigned char data )
{
	// When writing a display data, col_addr is increasing automatically 
	// and stops with 83h. 
	
	nt7534_write_data_core( 1 /* display data */, data );
}

static void nt7534_set_address( unsigned char page_addr, unsigned char col_addr )
{
	// page_addr = 0 ~ 8 
	// col_addr = 0 ~ 131 (83h)
	unsigned char cmd;
	
	// page address 
	cmd = LCM_CMD_SET_PAGE_ADDR | ( page_addr & LCM_CMD_SET_PAGE_ADDR_MASK );
	
	nt7534_write_control_data( cmd );	// set page address 
	
	// column address (hi)
	cmd = LCM_CMD_SET_COL_ADDR_HI | ( ( col_addr >> 4 ) & LCM_CMD_SET_COL_ADDR_HI_MASK );
	
	nt7534_write_control_data( cmd );

	// column address (lo)
	cmd = LCM_CMD_SET_COL_ADDR_LO | ( col_addr & LCM_CMD_SET_COL_ADDR_LO_MASK );
	
	nt7534_write_control_data( cmd );
}

static void nt7534_init_parameters( void )
{
	// Set start line 
	nt7534_write_control_data( LCM_CMD_START_LINE | 
							   ( 0 & LCM_CMD_START_LINE_MASK ) );

	// Set ADC
#ifdef CONNECTOR_ON_TOP
	nt7534_write_control_data( LCM_CMD_ADC_INC );
#else	
	nt7534_write_control_data( LCM_CMD_ADC_DEC );
#endif

	// Set Normal
	nt7534_write_control_data( LCM_CMD_REVERSE_DISPLAY_OFF );
	
	// Set normal
	nt7534_write_control_data( LCM_CMD_ENTIRE_DISPLAY_ON );

	// Set 1/6 Bias
	nt7534_write_control_data( LCM_CMD_BIAS_1_6 );

	// Set COM direction
#ifdef CONNECTOR_ON_TOP
	nt7534_write_control_data( LCM_CMD_COMMON_MODE_NORMAL );
#else	
	nt7534_write_control_data( LCM_CMD_COMMON_MODE_REVERSE );
#endif
	
	// Set power VC,VR,VF
	nt7534_write_control_data( LCM_CMD_POWER_CONTROL |
							   ( 7 & LCM_CMD_POWER_CONTROL_MASK ) );
	
	// Set Ra/Rb
	nt7534_write_control_data( LCM_CMD_REGULATOR_RATIO | 
							   ( 4 & LCM_CMD_REGULATOR_RATIO_MASK ) );
	
	// Set V0 (1st command)
	nt7534_write_control_data( LCM_CMD_ELEC_VOL_1ST_CMD );
	
	// 9.0V (2nd command) 
	nt7534_write_control_data( 0x2B & LCM_CMD_ELEC_VOL_2ND_MASK );
}

// API 
void nt7534_ClearScreen( int color )
{
	int i, j;
	const unsigned char data = ( color ? 0xFF : 0x00 );
	
	for( i = 0; i < LCM_PAGE_NUM; i ++ ) {
		// set address 
		nt7534_set_address( i, 0 );
		
		for( j = 0; j < LCM_COLS_NUM; j ++ )
			nt7534_write_display_data( data );
	}
}

void nt7534_WriteData( unsigned char page, unsigned char col, const unsigned char *pdata, int len )
{
	nt7534_set_address( page, col );
	
	while( len -- ) {
		nt7534_write_display_data( *pdata ++ );
		
		// wrap-around writing 
		if( ++ col >= LCM_COLS_NUM ) {
			col = 0;
			if( ++page >= LCM_PAGE_NUM )
				page = 0;
				
			nt7534_set_address( page, col );
		}
	}
}

void nt7534_DisplayOnOff( int onOff )
{
	if( onOff )
		nt7534_write_control_data( LCM_CMD_DISPLAY_ON );
	else
		nt7534_write_control_data( LCM_CMD_DISPLAY_OFF );
}

#ifdef FONT_FUNC
static void nt7534_DrawText_Core( unsigned char page, unsigned char col, unsigned char ch )
{
	const unsigned char *pBitmap;
		
	if( ch < 0x20 || ch >= 0x20 + 95 )	// there 95 words, and first one is 0x20 
		return;
	
	pBitmap = eng_fonts + ( ch - 0x20 ) * 16;	// 16 bytes per word 
	
	nt7534_WriteData( page, col, pBitmap, 8 );
	nt7534_WriteData( page + 1, col, pBitmap + 8, 8 );	
}

void nt7534_DrawText( unsigned char page, unsigned char col, unsigned char *pszText )
{
	if( page >= LCM_PAGE_NUM )
		return;

	while( *pszText && col < LCM_COLS_NUM ) {
		// write a text 
		nt7534_DrawText_Core( page, col, *pszText );
		
		// increase variables 
		col += 8;
		pszText ++;
	}
}
#endif

#ifdef DEMO_MODE
static void do_demo( void )
{
	static unsigned char logo[] = {
		0xFF,0x01,0x01,0x01,0x01,0xF9,0xF9,0xE1,0xC1,0xE1,0xF9,0xF9,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0xF9,0xF9,0xF1,0xE1,0xE1,0xF9,
		0xF9,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
		0x01,0x01,0x01,0xFF,0xFF,0x00,0x00,0x00,0x00,0x01,0x00,0x3F,0x7F,0x7F,0xF8,0xF0,
		0xE0,0x00,0x00,0x1C,0xFC,0x9C,0x80,0x80,0x98,0xFC,0x5C,0x08,0x00,0xE0,0xF0,0xF8,
		0xFF,0x7F,0x3F,0x0F,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF,
		0xFF,0xFF,0xFF,0xFF,0x03,0x03,0x83,0xFF,0xFF,0xFF,0xFF,0x3C,0x00,0x00,0x04,0xFC,
		0xFC,0xFC,0xFC,0x0C,0x0C,0x1C,0x1C,0x1C,0x00,0x00,0x00,0x00,0x80,0xE0,0xF8,0xFC,
		0xFC,0xF0,0x80,0x00,0x00,0x00,0x00,0x00,0x0C,0xFC,0xFC,0xFC,0xFC,0x00,0x00,0x00,
		0x00,0x1C,0x1C,0x0C,0x1C,0xFC,0xFC,0xFC,0xFC,0x0C,0x1C,0x1C,0x00,0x00,0xFC,0xFC,
		0xFC,0xFC,0x0C,0x1C,0x1C,0x1C,0x0C,0x00,0x00,0xFC,0xFC,0xFC,0xFC,0x00,0x80,0xE0,
		0xF0,0x38,0x1C,0x0C,0x00,0x00,0x00,0xFF,0xFF,0x00,0xE0,0xF0,0xD8,0xEC,0x7C,0xFC,
		0xFC,0xFC,0xFF,0xFF,0x7F,0xFC,0xFE,0xFE,0xFE,0x78,0xF0,0xF0,0x78,0xFC,0xFE,0xFE,
		0x1E,0x1F,0x1F,0x03,0xFC,0xFC,0xFC,0x6C,0xDC,0xB8,0xE0,0xC0,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0xFF,0x07,0x0F,0x3F,0xFF,0xFE,0xFC,0xF0,0xC0,
		0x80,0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x83,0x83,0x83,0x83,0x83,0x00,0xC0,0xF0,0xFC,
		0x1F,0x1F,0x18,0x19,0x1B,0x1F,0xFF,0xFE,0xF8,0xC0,0x80,0x00,0x80,0xFF,0xFF,0xFF,
		0xFF,0x80,0x80,0x80,0x80,0x00,0x00,0x00,0xC0,0xFF,0xFF,0xFF,0xFF,0x00,0x00,0x00,
		0x00,0x00,0xFF,0xFF,0xFF,0xFF,0x83,0x83,0x83,0x83,0x83,0x00,0x00,0xFF,0xFF,0xFF,
		0xFF,0x0F,0x3F,0x7E,0xFC,0xF8,0xF0,0xC0,0x00,0x00,0x00,0xFF,0xFF,0x80,0x83,0x81,
		0x87,0x8F,0x8F,0x9D,0x9F,0x83,0x87,0x87,0x80,0x81,0x87,0x87,0x87,0x80,0x80,0x80,
		0x80,0x87,0x87,0x87,0x86,0x86,0x86,0x81,0x9F,0x9E,0x81,0x8F,0x87,0x87,0x83,0x83,
		0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x81,0x81,0x81,0x81,0x81,0x80,0x80,0x80,0x80,
		0x81,0x81,0x81,0x81,0x81,0x80,0x80,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,
		0x80,0x81,0x81,0x80,0x80,0x80,0x80,0x80,0x80,0x80,0x81,0x81,0x81,0x81,0x81,0x80,
		0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x80,0x80,0x80,0x81,0x81,0x81,0x81,
		0x81,0x80,0x80,0x80,0x80,0x80,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x81,0x80,
		0x80,0x81,0x81,0x81,0x81,0x80,0x80,0x80,0x80,0x81,0x81,0x81,0x81,0x80,0x80,0xFF,
	};
	

	
	int i, j, x, y, z;
	unsigned long t;
	unsigned char *pdata;
	
	while( 1 ) {
		// DEMO 1: show fonts (there are 95 words)
		for( j = 0; j < 3; j ++ ) {
			// screen can show 16 * 2 words 
			for( y = 0; y < 2 * 2; y ++ ) {
				for( x = 0; x < 16; x ++ ) {
					
					z = j * 16 * 2 + y / 2 * 16 + x;
					
					if( z >= 95 )
						continue;
				
					nt7534_WriteData( y, x * 8,  
						eng_fonts + z * 16 + ( y & 0x01 ) * 8, 8 );
				}
			}
			
			nt7534_mdelay( 5000 );
		}
		
		// DEMO 2: draw a string 
		nt7534_DrawText( 0, 0, "Realtek Ltd. Co." );
		
		nt7534_DrawText( 2, 0, "This is a demo.." );
		
		nt7534_mdelay( 5000 );
		
		// DEMO 3: show Realtek logo 
  #if 1
		nt7534_WriteData( 0, 0, logo, sizeof( logo ) );
		nt7534_mdelay( 5000 );
  #else
		for( j = 0; j < LCM_COLS_NUM / 4; j ++ ) {	// shift 4 pixels per time 
			for( i = 0; i < LCM_PAGE_NUM; i ++ ) {
				pdata = logo + i * LCM_COLS_NUM;
	
				// shift left 4 pixels 
				t = *( ( unsigned long * )( pdata ) );
				memcpy( pdata, pdata + 4, LCM_COLS_NUM - 4 );
				*( ( unsigned long * )( pdata + LCM_COLS_NUM - 4 ) ) = t;
			}
		
			nt7534_WriteData( 0, 0, logo, sizeof( logo ) );
			nt7534_mdelay( 200 );
		}
  #endif
	}
}
#endif // DEMO_MODE

void nt7534_LCM_init( void )
{
	// init GPIO 
	nt7534_init_lcm_gpio();
	
	// reset nt7534 
	nt7534_reset();

	// clean screen 
	nt7534_ClearScreen( 0 );
	
	// init display parameters before display on 
	nt7534_init_parameters();

	// display on 
	nt7534_DisplayOnOff( 1 );

#ifdef DEMO_MODE
	do_demo();
#endif

#if 1
	while( 1 ) {
		nt7534_ClearScreen( 1 );
		printk( "black\n" );
		nt7534_mdelay( 1000 );

		nt7534_ClearScreen( 0 );
		printk( "white\n" );
		nt7534_mdelay( 1000 );
	}
	
	printk( "Block\n" );
	
	while( 1 );

#endif	
}

