#include <netinet/ether.h>

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include <syslog.h>
#include <fcntl.h>

#include <dbus/dbus.h>

#include <libmbtutl.h>
#include <lib_local_comm.h>

static volatile sig_atomic_t __io_canceled = 0;
static volatile sig_atomic_t __io_terminated = 0;
static volatile sig_atomic_t exit_on_release = 1;

#ifdef __unused
#undef __unused
#endif

static int debug = 0;

#define __unused __attribute__((unused))

static void sig_term(int sig __unused)
{
	__io_canceled = 1;
}

void
bluez_agent_syslog(int pri, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if (debug) {
		vprintf(fmt, ap);
		printf("\n");
	} else
		vsyslog(pri, fmt, ap);
	va_end(ap);
}

static DBusHandlerResult agent_filter(DBusConnection *conn __unused,
				      DBusMessage *msg,
				      void *data __unused)
{
	const char *name, *old, *new;
	dbus_bool_t ret;

	ret = dbus_message_is_signal(msg, DBUS_INTERFACE_DBUS,
				     "NameOwnerChanged");
	if (!ret) {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	ret = dbus_message_get_args(msg, NULL,
				    DBUS_TYPE_STRING, &name,
				    DBUS_TYPE_STRING, &old,
				    DBUS_TYPE_STRING, &new,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for NameOwnerChanged\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	if (strncmp(name, "org.bluez", sizeof("org.bluez")) == 0
	    && *new == '\0') {
		fprintf(stderr, "Agent has been terminated\n");
		__io_terminated = 1;
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

/*
 * Supports only default adapter.
 */
static char *get_adapter_path(DBusConnection *conn)
{
	DBusMessage *msg, *reply;
	DBusError err;
	dbus_bool_t ret;
	const char *reply_path;
	char *path;

	msg = dbus_message_new_method_call("org.bluez", "/",
					   "org.bluez.Manager",
					   "DefaultAdapter");
	if (msg == NULL) {
		fprintf(stderr, "Can't allocate new method call\n");
		return NULL;
	}

	dbus_error_init(&err);

	reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);
	dbus_message_unref(msg);

	if (reply == NULL) {
		fprintf(stderr, "Can't get default adapter\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		return NULL;
	}

	ret = dbus_message_get_args(reply, &err,
				    DBUS_TYPE_OBJECT_PATH, &reply_path,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Can't get reply arguments\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		dbus_message_unref(reply);
		return NULL;
	}

	path = strdup(reply_path);

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return path;
}

static int
server_confirm_request(int type, char *name, char *bdaddr_str, uint32_t passkey)
{
	int fd = -1, ret, confirmed = 0;
	struct bluez_agent_confirm_req req;
	struct local_comm_hdr rcv_hdr;
	int rcv_size;
	void *buf = NULL;
	struct ether_addr *addr;

	if (debug == 1) {
		char yes_or_no[4];
		bluez_agent_syslog(LOG_DEBUG, "Confirm passkey (yes/no): ");
		scanf("%3s", yes_or_no);

		if (strncmp(yes_or_no, "yes", sizeof("yes")) == 0)
			return 1;
		return 0;
	} else if (debug > 1) {
		return 1;
	}

	memset(&req, 0, sizeof(req));

	addr = ether_aton(bdaddr_str);
	if (addr == NULL) {
		syslog(LOG_DEBUG, "%s: ether_aton failed", __func__);
		goto end;
	}

	if (type == BLUEZ_AGENT_MSG_CONFIRM_REQ) {
		strlcpy(req.name, name, sizeof(req.name));
		memcpy(req.bdaddr, addr, sizeof(req.bdaddr));
		req.passkey = passkey;
	} else if (type == BLUEZ_AGENT_MSG_AUTHORIZE_REQ) {
		strlcpy(req.name, name, sizeof(req.name));
		memcpy(req.bdaddr, addr, sizeof(req.bdaddr));
	} else {
		syslog(LOG_DEBUG, "%s: unknown type %d", __func__, type);
		goto end;
	}

	fd = lib_local_comm_client_open(BLUEZ_AGENT_SOCK_PATH);
	if (fd < 0) {
		syslog(LOG_DEBUG, "%s: lib_local_comm_client_open(%s) error",
		       __func__, BLUEZ_AGENT_SOCK_PATH);
		goto end;
	}

	ret = lib_local_comm_send(fd, type, 0, &req, sizeof(req));
	if (ret) {
		syslog(LOG_DEBUG, "%s:lib_local_comm_send error", __func__);
		goto end;
	}

	ret = lib_local_comm_recv(fd, &rcv_hdr, (void **)&buf, &rcv_size);
	if (ret) {
		syslog(LOG_DEBUG, "%s: lib_local_comm_recv error", __func__);
		goto end;
	}

	if (rcv_hdr.type == type + 1) {
		struct bluez_agent_confirm_resp *resp =
			(struct bluez_agent_confirm_resp *)buf;
		if (resp->confirmed)
			confirmed = 1;
	} else {
		syslog(LOG_DEBUG, "%s: recv unknown message type=%d",
		       __func__, rcv_hdr.type);
	}

	bluez_agent_syslog(LOG_DEBUG, "%s: %s confirmed=%d",
			   __func__, name, confirmed);
 end:
	if (buf != NULL)
		free(buf);
	if (fd >= 0) {
		lib_local_comm_client_close(fd);
	}

	if (confirmed) {
		if (type == BLUEZ_AGENT_MSG_CONFIRM_REQ)
			libmbtutl_trust_device(LIBMBTUTL_DEFAULT_IFNAME,
					       req.bdaddr);
	}

	return confirmed;
}

/*
 * @@@@ strdup ѤƤ뤿 name, bdaddr ϻѸ free 뤳
 */
static int get_device_name_btaddr(DBusConnection *conn, const char *path,
				  char **name, char **bdaddr)
{
	DBusMessage *msg, *reply;
	DBusMessageIter reply_iter;
	DBusMessageIter reply_iter_entry;
	uint32_t class;

	msg = dbus_message_new_method_call("org.bluez", path,
					       "org.bluez.Device",
					       "GetProperties");
	reply = dbus_connection_send_with_reply_and_block(conn,
							msg, -1, NULL);
	dbus_message_unref(msg);
	if (!reply)
		return -1;
	dbus_message_iter_init(reply, &reply_iter);

	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY)
		return -1;

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) ==
	       DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		/* Key == Class ? */
		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (strcmp(key, "Class") != 0 &&
				strcmp(key, "Alias") != 0 &&
				strcmp(key, "Address") != 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		/* Try to get the value */
		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}
		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		if (strcmp(key, "Class") == 0) {
			dbus_message_iter_get_basic(&iter_dict_val, &class);
		} else {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);
			if (strcmp(key, "Alias") == 0) {
				*name = strdup(value);
			} else if (bdaddr) {
				*bdaddr = strdup(value);
			}
		}
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return 0;
}

static DBusHandlerResult RequestPinCode(DBusConnection *conn,
					DBusMessage *msg,
					void *data __unused)
{
	DBusMessage *reply;
	const char *path;
	dbus_bool_t ret;
	/*
	char pincode[16];
	*/

	ret = dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH,
				    &path, DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for RequestPinCode\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	reply = dbus_message_new_method_return(msg);
	if (!reply) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	bluez_agent_syslog(LOG_DEBUG, "Pincode request for device %s\n", path);

	/* @@@@ reject PinCode authentication */
	fprintf(stderr, "RequestPinCode is not supported\n");
	reply = dbus_message_new_error(msg, "org.bluez.Error.Rejected",
				       "");

	/*
	memset(pincode, 0, sizeof(pincode));

	bluez_agent_syslog(LOG_DEBUG, "Enter PinCode: ");
	ret = scanf("%15s", pincode);

	dbus_message_append_args(reply, DBUS_TYPE_STRING,
				 &pincode, DBUS_TYPE_INVALID);
	*/

	dbus_connection_send(conn, reply, NULL);

	dbus_connection_flush(conn);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult RequestPassKey(DBusConnection *conn,
					DBusMessage *msg,
					void *data __unused)
{
	DBusMessage *reply;
	const char *path;
	dbus_bool_t ret;
	/*
	uint32_t passkey;
	*/

	ret = dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH,
				    &path, DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for RequestPassKey\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	reply = dbus_message_new_method_return(msg);
	if (reply == NULL) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	bluez_agent_syslog(LOG_DEBUG, "Passkey request for device %s\n", path);

	/* @@@@ reject PassKey authentication */
	fprintf(stderr, "RequestPassKey is not supported\n");
	reply = dbus_message_new_error(msg, "org.bluez.Error.Rejected",
				       "");

	/*
	bluez_agent_syslog(LOG_DEBUG, "Enter passkey: ");
	ret = scanf("%" PRIu32, &passkey);

	dbus_message_append_args(reply, DBUS_TYPE_UINT32, &passkey,
				 DBUS_TYPE_INVALID);
	*/

	dbus_connection_send(conn, reply, NULL);

	dbus_connection_flush(conn);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult RequestConfirmation(DBusConnection *conn,
					     DBusMessage *msg,
					     void *data __unused)
{
	DBusMessage *reply;
	const char *path;
	dbus_bool_t ret;
	uint32_t passkey;
	char *name = NULL, *bdaddr = NULL;
	int confirmed = 0;

	ret = dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &path,
				    DBUS_TYPE_UINT32, &passkey,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for RequestPasskey\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	reply = dbus_message_new_method_return(msg);
	if (reply == NULL) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	get_device_name_btaddr(conn, path, &name, &bdaddr);

	bluez_agent_syslog(LOG_DEBUG, "RequestConfirmation (%s(%s), %06u)\n",
			   name, bdaddr, passkey);

	confirmed = server_confirm_request(BLUEZ_AGENT_MSG_CONFIRM_REQ,
					 name, bdaddr, passkey);
	if (!confirmed)
		reply = dbus_message_new_error(msg, "org.bluez.Error.Rejected",
					       "");

	if (name)
		free(name);
	if (bdaddr)
		free(bdaddr);

	dbus_connection_send(conn, reply, NULL);

	dbus_connection_flush(conn);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult Authorize(DBusConnection *conn,
				   DBusMessage *msg,
				   void *data __unused)
{
	DBusMessage *reply;
	const char *path, *uuid;
	dbus_bool_t ret;
	char *name = NULL, *bdaddr = NULL;
	int confirmed = 0;
	uint32_t flags;

	ret = dbus_message_get_args(msg, NULL,
				    DBUS_TYPE_OBJECT_PATH, &path,
				    DBUS_TYPE_STRING, &uuid,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for Authorize\n");
	}

	reply = dbus_message_new_method_return(msg);
	if (reply == NULL) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	get_device_name_btaddr(conn, path, &name, &bdaddr);

	bluez_agent_syslog(LOG_DEBUG, "Authorizing request for %s(%s)\n",
			   name, bdaddr);

	libmbtutl_get_hci_flags(LIBMBTUTL_DEFAULT_IFNAME, &flags);

	if (!(flags & LIBMBTUTL_HCI_ISCAN)) {
		confirmed = libmbtutl_check_authorized_device(
				LIBMBTUTL_DEFAULT_IFNAME,
				bdaddr);
	} else {
		confirmed = server_confirm_request(
				BLUEZ_AGENT_MSG_AUTHORIZE_REQ,
				name, bdaddr, 0);
		if (confirmed)
			libmbtutl_add_authorized_device(
				LIBMBTUTL_DEFAULT_IFNAME,
				bdaddr);
		else
			libmbtutl_del_authorized_device(
				LIBMBTUTL_DEFAULT_IFNAME,
				bdaddr);
	}

	if (!confirmed)
		reply = dbus_message_new_error(msg, "org.bluez.Error.Rejected",
					       "");

 skip_confirm:
	if (name)
		free(name);
	if (bdaddr)
		free(bdaddr);

	dbus_connection_send(conn, reply, NULL);

	dbus_connection_flush(conn);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult Cancel(DBusConnection *conn,
				DBusMessage *msg,
				void *data __unused)
{
	DBusMessage *reply;
	dbus_bool_t ret;

	ret = dbus_message_get_args(msg, NULL, DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for Cancel\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	bluez_agent_syslog(LOG_DEBUG, "Cancel\n");

	reply = dbus_message_new_method_return(msg);
	if (!reply) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	dbus_connection_send(conn, reply, NULL);

	dbus_connection_flush(conn);
	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult Release(DBusConnection *conn,
				 DBusMessage *msg,
				 void *data __unused)
{
	DBusMessage *reply;
	dbus_bool_t ret;

	ret = dbus_message_get_args(msg, NULL, DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Invalid arguments for Release\n");
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	if (!__io_canceled) {
		fprintf(stderr, "Agent has been released\n");
	}

	if (exit_on_release) {
		__io_terminated = 1;
	}

	bluez_agent_syslog(LOG_DEBUG, "Release\n");

	reply = dbus_message_new_method_return(msg);
	if (!reply) {
		fprintf(stderr, "Can't create reply message\n");
		return DBUS_HANDLER_RESULT_NEED_MEMORY;
	}

	dbus_connection_send(conn, reply, NULL);
	dbus_connection_flush(conn);

	dbus_message_unref(reply);

	return DBUS_HANDLER_RESULT_HANDLED;
}

static DBusHandlerResult agent_message(DBusConnection *conn,
				       DBusMessage *msg, void *data)
{
	if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					"RequestPinCode")) {
		return RequestPinCode(conn, msg, data);
	} else if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					       "RequestPassKey")) {
		return RequestPassKey(conn, msg, data);
	} else if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					       "RequestConfirmation")) {
		return RequestConfirmation(conn, msg, data);
	} else if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					       "Authorize")) {
		return Authorize(conn, msg, data);
	} else if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					       "Cancel")) {
		return Cancel(conn, msg, data);
	} else if (dbus_message_is_method_call(msg, "org.bluez.Agent",
					       "Release")) {
		return Release(conn, msg, data);
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static const DBusObjectPathVTable agent_table = {
	.message_function = agent_message,
};

static int register_agent(DBusConnection *conn, const char *adapter_path,
			  const char *agent_path, const char *capability)
{
	DBusMessage *msg, *reply;
	DBusError err;

	msg = dbus_message_new_method_call("org.bluez", adapter_path,
					   "org.bluez.Adapter",
					   "RegisterAgent");
	if (msg == NULL) {
		fprintf(stderr, "Can't allocate new method");
		return -1;
	}

	bluez_agent_syslog(LOG_DEBUG, "Register Agent\n");

	dbus_message_append_args(msg, DBUS_TYPE_OBJECT_PATH, &agent_path,
				 DBUS_TYPE_STRING, &capability,
				 DBUS_TYPE_INVALID);

	dbus_error_init(&err);

	reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);
	dbus_message_unref(msg);

	if (!reply) {
		fprintf(stderr, "Can't register agent\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		return -1;
	}

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return 0;
}

static int unregister_agent(DBusConnection *conn, const char *adapter_path,
			    const char *agent_path)
{
	DBusMessage *msg, *reply;
	DBusError err;

	msg = dbus_message_new_method_call("org.bluez", adapter_path,
					   "org.bluez.Adapter",
					   "UnregisterAgent");
	if (msg == NULL) {
		fprintf(stderr, "Can't allocate new method call\n");
		return -1;
	}

	bluez_agent_syslog(LOG_DEBUG, "Unregister Agent\n");

	dbus_message_append_args(msg,
				 DBUS_TYPE_OBJECT_PATH, &agent_path,
				 DBUS_TYPE_INVALID);

	dbus_error_init(&err);

	reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);

	dbus_message_unref(msg);

	if (reply == NULL) {
		fprintf(stderr, "Can't unregister agent\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		return -1;
	}

	dbus_message_unref(reply);

	dbus_connection_flush(conn);
	dbus_connection_unregister_object_path(conn, agent_path);

	return 0;
}

static struct option options[] = {
	{ "capability", 1, NULL, 'c' },
	{ "help", 0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

static void usage()
{
	bluez_agent_syslog(LOG_DEBUG, "NECAT Bluetooth agent\n\n");
	bluez_agent_syslog(LOG_DEBUG, "Usage: necat-bluez-agent [options]\n\n"
	       "Options:\n"
	       "\t-c, --capability : Set capability (Default: \"DisplayYesNo\")\n"
	       "\t-h, --help       : Show this message\n"
	       "\n");
}

int main(int argc, char *argv[])
{
	const char *capability = "DisplayYesNo";
	char agent_path[128];
	char *adapter_path = NULL;
	int opt;
	DBusConnection *conn;
	dbus_bool_t ret;
	struct sigaction sa;
	void *handle;
	int fd;

	snprintf(agent_path, sizeof(agent_path),
		 "/org/bluez/agent_%d", getpid());

	while ((opt = getopt_long(argc, argv, "c:hd", options, NULL)) != EOF) {
		switch (opt) {
		case 'c':
			capability = optarg;
			break;
		case 'd':
			debug++;
			break;
		case 'h':
			usage();
			exit(0);
		default:
			usage();
			exit(1);
		}
	}

	if (!debug) {
		if (daemon(1, 0)) {
			fprintf(stderr, "start fail\n");
			exit(EXIT_FAILURE);
		}
	}

	argc -= optind;
	argv += optind;
	optind = 0;

	libmbtutl_set_device_encrypt(LIBMBTUTL_DEFAULT_IFNAME, 1);

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Can't get on dbus system bus");
		exit(1);
	}

	adapter_path = get_adapter_path(conn);
	if (adapter_path == NULL) {
		exit(1);
	}

	ret = dbus_connection_register_object_path(conn, agent_path,
						   &agent_table, NULL);
	if (!ret) {
		fprintf(stderr, "Can't register object path for agent\n");
		exit(1);
	}

	if (register_agent(conn, adapter_path, agent_path, capability) < 0) {
		dbus_connection_unref(conn);
		exit(1);
	}

	ret = dbus_connection_add_filter(conn, agent_filter, NULL, NULL);
	if (!ret) {
		fprintf(stderr, "Can't add signal filter\n");
	}

	memset(&sa, 0, sizeof(sa));
	sa.sa_flags = SA_NOCLDSTOP;
	sa.sa_handler = sig_term;
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);

	while (!__io_canceled && !__io_terminated) {
		if (dbus_connection_read_write_dispatch(conn, 100) != TRUE)
			break;
	}

	if (!__io_terminated) {
		unregister_agent(conn, adapter_path, agent_path);
	}

	free(adapter_path);

	dbus_connection_unref(conn);

	return 0;
}
