#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>

#include <glib.h>
#include <glib/gprintf.h>

#include <dbus/dbus.h>

static int
adapter_key_to_type(const char *key)
{
	if (strcmp(key, "Address") == 0 || strcmp(key, "Name") == 0) {
		return DBUS_TYPE_STRING;
	} else if (strcmp(key, "Powered") == 0 || strcmp(key, "Discoverable") == 0
		|| strcmp(key, "Pairable") == 0 || strcmp(key, "Discovering") == 0) {
		return DBUS_TYPE_BOOLEAN;
	} else if (strcmp(key, "Class") == 0 || strcmp(key, "PairableTimeout") == 0
		   || strcmp(key, "DiscoverableTimeout") == 0) {
		return DBUS_TYPE_UINT32;
	} else if (strcmp(key, "Devices") == 0 || strcmp(key, "UUIDs") == 0) {
		return DBUS_TYPE_ARRAY;
	} else {
		fprintf(stderr, "Error: Unknown Adapter Peroperty: %s\n", key);
		exit(1);
	}
}

static int
device_key_to_type(const char *key)
{
	if (strcmp(key, "Address") == 0 || strcmp(key, "Name") == 0
	    || strcmp(key, "Icon") == 0 || strcmp(key, "Alias") == 0) {
		return DBUS_TYPE_STRING;
	} else if (strcmp(key, "Paired") == 0 || strcmp(key, "Connected") == 0
		|| strcmp(key, "Trusted") == 0 || strcmp(key, "LegachPairing") == 0
		|| strcmp(key, "Blocked") == 0) {
		return DBUS_TYPE_BOOLEAN;
	} else if (strcmp(key, "Class") == 0) {
		return DBUS_TYPE_UINT32;
	} else if (strcmp(key, "Vendor") == 0 || strcmp(key, "Product") == 0
		   || strcmp(key, "Version") == 0) {
		return DBUS_TYPE_UINT16;
	} else if (strcmp(key, "Nodes") == 0 || strcmp(key, "UUIDs") == 0
		   || strcmp(key, "Services") == 0) {
		return DBUS_TYPE_ARRAY;
	} else if (strcmp(key, "Adapter") == 0) {
		return DBUS_TYPE_OBJECT_PATH;
	} else {
		fprintf(stderr, "Error: Unknown Adapter Peroperty: %s\n", key);
		exit(1);
	}
}

static int
node_key_to_type(const char *key)
{
	if (strcmp(key, "Address") == 0 || strcmp(key, "Name") == 0
	    || strcmp(key, "Icon") == 0 || strcmp(key, "Alias") == 0) {
		return DBUS_TYPE_STRING;
	} else if (strcmp(key, "Paired") == 0 || strcmp(key, "Connected") == 0
		|| strcmp(key, "Trusted") == 0 || strcmp(key, "LegachPairing") == 0) {
		return DBUS_TYPE_BOOLEAN;
	} else if (strcmp(key, "Class") == 0) {
		return DBUS_TYPE_UINT32;
	} else if (strcmp(key, "Devices") == 0 || strcmp(key, "Nodes") == 0) {
		return DBUS_TYPE_ARRAY;
	} else if (strcmp(key, "Adapter") == 0) {
		return DBUS_TYPE_OBJECT_PATH;
	} else {
		fprintf(stderr, "Error: Unknown Adapter Peroperty: %s\n", key);
		exit(1);
	}
}

static void
print_string_range(const char *str, int start, int end)
{
	int i;

	for (i = start; i < end; i++) {
		putchar(str[i]);
	}
}

static int
ends_with(const char *str, const char *suffix)
{
	size_t len_str, len_suffix;

	if (str == NULL || suffix == NULL) {
		return 0;
	}

	len_str = strlen(str);
	len_suffix = strlen(suffix);

	if (len_suffix > len_str) {
		return 0;
	}

	return strncmp(str + len_str - len_suffix, suffix, len_suffix) == 0;
}

static int
show_object_paths(DBusMessageIter *devices_iter, char *paths[])
{
	DBusMessageIter iter_array;
	int i = 0;

	dbus_message_iter_recurse(devices_iter, &iter_array);
	while (dbus_message_iter_get_arg_type(&iter_array) == DBUS_TYPE_OBJECT_PATH) {
		const char *path;

		dbus_message_iter_get_basic(&iter_array, &path);
		if (!path) {
			dbus_message_iter_next(&iter_array);
			continue;
		}

		if (paths != NULL) {
			paths[i] = strdup(path);
		}

		printf("    Devices = %s\n", path);
		dbus_message_iter_next(&iter_array);

		i++;
	}

	return i;
}

static void
show_uuids_properties(DBusMessageIter *uuids_iter)
{
	DBusMessageIter iter_array;

	dbus_message_iter_recurse(uuids_iter, &iter_array);
	while (dbus_message_iter_get_arg_type(&iter_array) == DBUS_TYPE_STRING) {
		const char *uuid;

		dbus_message_iter_get_basic(&iter_array, &uuid);

		if (!uuid) {
			dbus_message_iter_next(&iter_array);
			continue;
		}

		if (ends_with(uuid, "-0000-1000-8000-00805f9b34fb")) {
			printf("0x");
			if (strncmp(uuid, "0000", 4) == 0) {
				print_string_range(uuid, 4, 8);
			} else {
				print_string_range(uuid, 0, 8);
			}
		} else {
			printf("%s", uuid);
		}

		printf(" ");
		dbus_message_iter_next(&iter_array);
	}
}

static void
show_node_properties(DBusConnection *conn, const char *node)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry;
	DBusError error;

	printf("        [ %s ]\n", node);

	message = dbus_message_new_method_call("org.bluez", node,
					       "org.bluez.Node",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Node#GetProperties)\n");
		return;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Node#GetProperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;
		int type;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		/* Try to get the value */
		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);

		type = node_key_to_type(key);

		if (type == DBUS_TYPE_STRING) {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("            %s = %s\n", key, value);
		} else if (type == DBUS_TYPE_OBJECT_PATH) {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("            %s = %s\n", key, value);
		} else {
			fprintf(stderr, "Can't handle key=%s\n", key);
			exit(1);
		}

		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return;
}

static void
show_device_properties(DBusConnection *conn, const char *device)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry;
	DBusError error;
	char *node_paths[256];
	int i, node_count;

	printf("    [ %s ]\n", device);

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#GetProperties)\n");
		return;
	}


	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: org.bluez.Device#Getproperties\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return;
	}

	memset(node_paths, sizeof(const char*), 256);

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;
		int type;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		/* Try to get the value */
		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);

		type = device_key_to_type(key);

		if (type == DBUS_TYPE_STRING) {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("        %s = %s\n", key, value);
		} else if (type == DBUS_TYPE_OBJECT_PATH) {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("        %s = %s\n", key, value);
		} else if (type == DBUS_TYPE_BOOLEAN) {
			gboolean value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("        %s = %d\n", key, value);
		} else if (type == DBUS_TYPE_UINT32) {
			uint32_t value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("        %s = 0x%06x\n", key, value);
		} else if (type == DBUS_TYPE_UINT16) {
			uint16_t value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("        %s = 0x%04x\n", key, value);
		} else if (type == DBUS_TYPE_ARRAY) {
			if (strcmp(key, "Nodes") == 0) {
				node_count = show_object_paths(&iter_dict_val, node_paths);
			} else if (strcmp(key, "Services") == 0) {
				show_object_paths(&iter_dict_val, NULL);
			} else if (strcmp(key, "UUIDs") == 0) {
				printf("        UUIDs = ");
				show_uuids_properties(&iter_dict_val);
				printf("\n");
			} else {
				fprintf(stderr, "Unknown key %s\n", key);
				exit(1);
			}
		} else {
			fprintf(stderr, "Error Can't handle key=%s\n", key);
			exit(1);
		}

		dbus_message_iter_next(&reply_iter_entry);
	}

	for (i = 0; i < node_count; i++) {
		show_node_properties(conn, node_paths[i]);
		free(node_paths[i]);
	}

	dbus_message_unref(reply);

	return;
}

static void
show_adapter_information(DBusConnection *conn, const char *adapter)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry;
	DBusError error;
	char *device_paths[256];
	int i, device_count;

	message = dbus_message_new_method_call("org.bluez", adapter,
					       "org.bluez.Adapter",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Adapter#GetProperties)\n");
		return;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Adapter#GetProperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return;
	}

	memset(device_paths, sizeof(char*), 256);

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;
		int type;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);

		type = adapter_key_to_type(key);
		if (type == DBUS_TYPE_STRING) {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("    %s = %s\n", key, value);
		} else if (type == DBUS_TYPE_BOOLEAN) {
			gboolean value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("    %s = %d\n", key, value);
		} else if (type == DBUS_TYPE_UINT32) {
			uint32_t value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("    %s = %06x\n", key, value);
		} else if (type == DBUS_TYPE_ARRAY) {
			if (strcmp(key, "Devices") == 0) {
				device_count = show_object_paths(&iter_dict_val, device_paths);
			} else if (strcmp(key, "UUIDs") == 0) {
				printf("    UUIDs = ");
				show_uuids_properties(&iter_dict_val);
				printf("\n");
			}
		} else {
			fprintf(stderr, "Error Can't handle key=%s\n", key);
			exit(1);
		}

		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	for (i = 0; i < device_count; i++) {
		show_device_properties(conn, device_paths[i]);
		free(device_paths[i]);
	}

	return;
}

static void
adapter_foreach(DBusConnection *conn, DBusMessageIter *adapter_iter)
{
	DBusMessageIter iter_adapters;

	if (dbus_message_iter_get_arg_type(adapter_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		exit(1);
	}

	dbus_message_iter_recurse(adapter_iter, &iter_adapters);
	while (dbus_message_iter_get_arg_type(&iter_adapters) == DBUS_TYPE_OBJECT_PATH) {
		const char *path;

		dbus_message_iter_get_basic(&iter_adapters, &path);
		if (!path) {
			dbus_message_iter_next(&iter_adapters);
			continue;
		}

		printf("[ %s ]\n", path);
		show_adapter_information(conn, path);

		dbus_message_iter_next(&iter_adapters);
	}
}

static void
list_devices(DBusConnection *conn)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter reply_iter, reply_iter_entry;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.bluez.Manager",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Manager#GetProperties)\n");
		return;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Manager#Getproperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (strcmp(key, "Adapters") != 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		adapter_foreach(conn, &iter_dict_val);
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return;
}

int main(void)
{
	DBusConnection *conn;

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Can't get on dbus system bus");
		exit(1);
	}

	list_devices(conn);
	dbus_connection_flush(conn);

	return 0;
}
