/* Copyright (c) 2011, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#define pr_fmt(fmt) "%s: " fmt, __func__

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/leds.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#include <linux/module.h>

#ifdef CONFIG_HAS_EARLYSUSPEND
#include <linux/earlysuspend.h>

/* Early-suspend level */
#define LED_SUSPEND_LEVEL 1
#endif

#define PDM_DEBUG (0)
#if PDM_DEBUG
#define PDM_DPRINTK(fmt, ...) printk(fmt, ## __VA_ARGS__)
#else
#define PDM_DPRINTK(fmt, ...)
#endif

#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
#include <linux/gpio.h>
/* Enable control */
#define LED_EN_GPIO (19)
static int led_en_gpio;
static bool led_enabled = false;

/* PDM1 (Brightness) Control */
#define PDM_DUTY_MAXVAL BIT(8)
#define PDM_DUTY_REFVAL BIT(7)

#define PDM_MIN_BRIGHTNESS (40)
#define PDM_MAX_BRIGHTNESS (172)
#define PDM_CLAMP(val, min, max) \
do { \
	if (val < min) val = min; \
	if (val > max) val = max; \
} while (0)

#else
#define PDM_DUTY_MAXVAL BIT(16)
#define PDM_DUTY_REFVAL BIT(15)
#endif

struct pdm_led_data {
	struct led_classdev cdev;
	void __iomem *perph_base;
	int pdm_offset;
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend early_suspend;
#endif
};

static struct mutex pdm_mutex = __MUTEX_INITIALIZER(pdm_mutex);

#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
static int msm_led_gpio_varidation(void)
{
	mutex_lock(&pdm_mutex);
	led_en_gpio = LED_EN_GPIO;
	led_enabled = false;

	if (!gpio_is_valid(led_en_gpio)) {
		mutex_unlock(&pdm_mutex);
		pr_err("%s: enable gpio not specified\n", __func__);
		return -EINVAL;
	}
	mutex_unlock(&pdm_mutex);
	return 0;
}

extern void msm9625_pdm_enable(int);

int msm_pdm_led_on(void)
{
	mutex_lock(&pdm_mutex);
	if (led_enabled) {
		mutex_unlock(&pdm_mutex);
		return 0;
	}

	if (gpio_request(led_en_gpio, "led_en")) {
		mutex_unlock(&pdm_mutex);
		pr_err("%s request enable gpio failed\n", __func__);
		return -EINVAL;
	}

	msm9625_pdm_enable(1);

	gpio_set_value(led_en_gpio, 1);
	msleep(10);

	PDM_DPRINTK("%s: LED %s\n", __func__,
		(1 == gpio_get_value(led_en_gpio)) ? "turn on": "off");

	led_enabled = true;
	mutex_unlock(&pdm_mutex);
	return 0;
}
EXPORT_SYMBOL(msm_pdm_led_on);

void msm_pdm_led_off(void)
{
	mutex_lock(&pdm_mutex);
	if (!led_enabled) {
		mutex_unlock(&pdm_mutex);
		return;
	}

	gpio_set_value(led_en_gpio, 0);
	msleep(10);

	PDM_DPRINTK("%s: LED %s\n", __func__,
		(0 == gpio_get_value(led_en_gpio)) ? "turn off": "on");

	gpio_free(led_en_gpio);

	msm9625_pdm_enable(0);

	led_enabled = false;
	mutex_unlock(&pdm_mutex);
	return;
}
EXPORT_SYMBOL(msm_pdm_led_off);

bool msm_pdm_get_led_status(void)
{
	PDM_DPRINTK("%s: LED status: %s\n", __func__, led_enabled ? "ON": "off");
	return led_enabled;
}
EXPORT_SYMBOL(msm_pdm_get_led_status);
#endif

static void msm_led_brightness_set_percent(struct pdm_led_data *led,
						int duty_per)
{
	u16 duty_val;

	duty_val = PDM_DUTY_REFVAL - ((PDM_DUTY_MAXVAL * duty_per) / 100);
#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	duty_val &= 0x00FF;
#endif

	if (!duty_per)
		duty_val--;

	PDM_DPRINTK("%s: duty_val=0x%02x\n", __func__, duty_val);
	writel_relaxed(duty_val, led->perph_base + led->pdm_offset);
}

static void msm_led_brightness_set(struct led_classdev *led_cdev,
				enum led_brightness value)
{
	int percent;
	u32 brightness = value;
	struct pdm_led_data *led =
		container_of(led_cdev, struct pdm_led_data, cdev);

#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	PDM_CLAMP(brightness, PDM_MIN_BRIGHTNESS, PDM_MAX_BRIGHTNESS);
#endif
	percent = (brightness * 100) / LED_FULL;
	PDM_DPRINTK("%s; brightness=%d/percent=%d\n",
			__func__, brightness, percent);

	msm_led_brightness_set_percent(led, percent);
}

#ifdef CONFIG_PM_SLEEP
static int msm_led_pdm_suspend(struct device *dev)
{
#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	msm_pdm_led_off();
#else
	struct pdm_led_data *led = dev_get_drvdata(dev);

	msm_led_brightness_set_percent(led, 0);
#endif

	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void msm_led_pdm_early_suspend(struct early_suspend *h)
{
	struct pdm_led_data *led = container_of(h,
			struct pdm_led_data, early_suspend);

	msm_led_pdm_suspend(led->cdev.dev->parent);
}

#endif

static const struct dev_pm_ops msm_led_pdm_pm_ops = {
#ifndef CONFIG_HAS_EARLYSUSPEND
	.suspend	= msm_led_pdm_suspend,
#endif
};
#endif

static int __devinit msm_pdm_led_probe(struct platform_device *pdev)
{
	const struct led_info *pdata = pdev->dev.platform_data;
	struct pdm_led_data *led;
	struct resource *res, *ioregion;
	u32 tcxo_pdm_ctl;
	int rc;
#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	int initial_percent;
#endif

	if (!pdata) {
		pr_err("platform data is invalid\n");
		return -EINVAL;
	}

	if (pdev->id > 2) {
		pr_err("pdm id is invalid\n");
		return -EINVAL;
	}

	led = kzalloc(sizeof(struct pdm_led_data), GFP_KERNEL);
	if (!led)
		return -ENOMEM;

	/* Enable runtime PM ops, start in ACTIVE mode */
	rc = pm_runtime_set_active(&pdev->dev);
	if (rc < 0)
		dev_dbg(&pdev->dev, "unable to set runtime pm state\n");
	pm_runtime_enable(&pdev->dev);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		pr_err("get resource failed\n");
		rc = -EINVAL;
		goto err_get_res;
	}

	ioregion = request_mem_region(res->start, resource_size(res),
						pdev->name);
	if (!ioregion) {
		pr_err("request for mem region failed\n");
		rc = -ENOMEM;
		goto err_get_res;
	}

	led->perph_base = ioremap(res->start, resource_size(res));
	if (!led->perph_base) {
		pr_err("ioremap failed\n");
		rc = -ENOMEM;
		goto err_ioremap;
	}

#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	rc = msm_led_gpio_varidation();
	if (rc < 0)
		goto err_led_reg;
#endif

	/* Pulse Density Modulation(PDM) ids start with 0 and
	 * every PDM register takes 4 bytes
	 */
	led->pdm_offset = ((pdev->id) + 1) * 4;

	/* program tcxo_pdm_ctl register to enable pdm*/
	tcxo_pdm_ctl = readl_relaxed(led->perph_base);
	tcxo_pdm_ctl |= (1 << pdev->id);
	writel_relaxed(tcxo_pdm_ctl, led->perph_base);

	/* Start with LED in off state */
#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	initial_percent = (PDM_MIN_BRIGHTNESS * 100) / LED_FULL;
	msm_led_brightness_set_percent(led, initial_percent);
#else
	msm_led_brightness_set_percent(led, 0);
#endif

	led->cdev.brightness_set = msm_led_brightness_set;
	led->cdev.name = pdata->name ? : "leds-msm-pdm";

	rc = led_classdev_register(&pdev->dev, &led->cdev);
	if (rc) {
		pr_err("led class registration failed\n");
		goto err_led_reg;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
	led->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN +
						LED_SUSPEND_LEVEL;
	led->early_suspend.suspend = msm_led_pdm_early_suspend;
	register_early_suspend(&led->early_suspend);
#endif

	platform_set_drvdata(pdev, led);
	return 0;

err_led_reg:
	iounmap(led->perph_base);
err_ioremap:
	release_mem_region(res->start, resource_size(res));
err_get_res:
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	kfree(led);
	return rc;
}

static int __devexit msm_pdm_led_remove(struct platform_device *pdev)
{
	struct pdm_led_data *led = platform_get_drvdata(pdev);
	struct resource *res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&led->early_suspend);
#endif
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);

	led_classdev_unregister(&led->cdev);
	msm_led_brightness_set_percent(led, 0);
	iounmap(led->perph_base);
	release_mem_region(res->start, resource_size(res));
	kfree(led);

	return 0;
}

static struct platform_driver msm_pdm_led_driver = {
	.probe		= msm_pdm_led_probe,
	.remove		= __devexit_p(msm_pdm_led_remove),
	.driver		= {
		.name	= "leds-msm-pdm",
		.owner	= THIS_MODULE,
#ifdef CONFIG_PM_SLEEP
		.pm	= &msm_led_pdm_pm_ops,
#endif
	},
};

static int __init msm_pdm_led_init(void)
{
	return platform_driver_register(&msm_pdm_led_driver);
}
module_init(msm_pdm_led_init);

static void __exit msm_pdm_led_exit(void)
{
	platform_driver_unregister(&msm_pdm_led_driver);
}
module_exit(msm_pdm_led_exit);

MODULE_DESCRIPTION("MSM PDM LEDs driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:leds-msm-pdm");
