#define BQ24251DEBUGMES 1
/*
 * bq24251 charger driver
 *
 * Copyright (C) 2011-2012  Pali Rohr <pali.rohar@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/param.h>
#include <linux/err.h>
#include <linux/workqueue.h>
#include <linux/sysfs.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/idr.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <mach/socinfo.h>

#define BQ24251_REG_WSF		0x00	/* WD/STAT/FAULT */
#define BQ24251_REG_RIECH	0x01	/* RESET/IIN/EN/CE/HZ */
#define BQ24251_REG_VU		0x02	/* VBAT/USB */
#define BQ24251_REG_II		0x03	/* ICHG/ITERM */
#define BQ24251_REG_LDCV	0x04	/* LOOP/LOW/DPDM/CE/VIN */
#define BQ24251_REG_TST		0x05	/* TIMER/SYSOFF/TS */
#define BQ24251_REG_VCF		0x06	/* VOVP/CLR/FORCE */

#define N_REGS	7	/* Number of Registers */
#define N_GPIOS	3	/* Number of use GPIOs */
#define BQ24251_POLLING_INTERVAL	500	/* ms */

enum SRC_TYPE {ACADP, DCP, SDP, CDP, NONSTD };
enum FAULT_TYPE {Normal, InOVP, InUVLO, Sleep, TSOpen, TSHot, TSCold, TSWarm, TSCool,
		 BatOVP, TShutdown, Timer, NoBat, ISET, IFandLDO, Unknown};

static const unsigned int DEF_REG_VAL[N_REGS] = {0x00, 0x2c, 0xa8, 0x02, 0x06, 0xc8, 0xf0};
static const unsigned int ACADP_REG_VAL[N_REGS] = {0x00, 0x4c, 0xa8, 0xa2, 0x06, 0xc8, 0xf0};
static int GPIONUM_PG, GPIONUM_USBID, GPIONUM_USBSWCONT;
static enum FAULT_TYPE FAULT_DETAIL;
static int CHG_STATUS;

struct bq24251_device {
	struct device *dev;
	struct power_supply charger;
	int id;
	int pre_pg;
	int pre_faultdetail;
};

static struct task_struct *bq24251_polling = NULL;
static int POLLING_ENABLE = 1;
static int change_usbsw_enable = 1;

/* each registered chip must have unique id */
static DEFINE_IDR(bq24251_id);

static DEFINE_MUTEX(bq24251_id_mutex);
static DEFINE_MUTEX(bq24251_i2c_mutex);

/**** i2c read functions ****/

/* read value from register */
static int bq24251_i2c_read(struct bq24251_device *bq, u8 reg)
{
	struct i2c_client *client = to_i2c_client(bq->dev);
	struct i2c_msg msg[2];
	u8 val;
	int ret;

	if (!client->adapter)
		return -ENODEV;

	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].buf = &reg;
	msg[0].len = sizeof(reg);
	msg[1].addr = client->addr;
	msg[1].flags = I2C_M_RD;
	msg[1].buf = &val;
	msg[1].len = sizeof(val);

	mutex_lock(&bq24251_i2c_mutex);
	ret = i2c_transfer(client->adapter, msg, ARRAY_SIZE(msg));
	mutex_unlock(&bq24251_i2c_mutex);

	if (ret < 0)
		return ret;

	return val;
}

/* read value from register, apply mask and right shift it */
static int bq24251_i2c_read_mask(struct bq24251_device *bq, u8 reg,
				 u8 mask, u8 shift)
{
	int ret;

	if (shift > 8)
		return -EINVAL;

	ret = bq24251_i2c_read(bq, reg);
	if (ret < 0)
		return ret;
	return (ret & mask) >> shift;
}

/* read value from register and return one specified bit */
static int bq24251_i2c_read_bit(struct bq24251_device *bq, u8 reg, u8 bit)
{
	if (bit > 8)
		return -EINVAL;
	return bq24251_i2c_read_mask(bq, reg, BIT(bit), bit);
}

/**** i2c write functions ****/

/* write value to register */
static int bq24251_i2c_write(struct bq24251_device *bq, u8 reg, u8 val)
{
	struct i2c_client *client = to_i2c_client(bq->dev);
	struct i2c_msg msg[1];
	u8 data[2];
	int ret;

	data[0] = reg;
	data[1] = val;

	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].buf = data;
	msg[0].len = ARRAY_SIZE(data);

	mutex_lock(&bq24251_i2c_mutex);
	ret = i2c_transfer(client->adapter, msg, ARRAY_SIZE(msg));
	mutex_unlock(&bq24251_i2c_mutex);

	/* i2c_transfer returns number of messages transferred */
	if (ret < 0)
		return ret;
	else if (ret != 1)
		return -EIO;

	return 0;
}

/* read value from register, change it with mask left shifted and write back */
static int bq24251_i2c_write_mask(struct bq24251_device *bq, u8 reg, u8 val,
				  u8 mask, u8 shift)
{
	int ret;

	if (shift > 8)
		return -EINVAL;

	ret = bq24251_i2c_read(bq, reg);
	if (ret < 0)
		return ret;

	ret &= ~mask;
	ret |= val << shift;

	return bq24251_i2c_write(bq, reg, ret);
}

/* change only one bit in register */
static int bq24251_i2c_write_bit(struct bq24251_device *bq, u8 reg,
				 bool val, u8 bit)
{
	if (bit > 8)
		return -EINVAL;
	return bq24251_i2c_write_mask(bq, reg, val, BIT(bit), bit);
}

/**** power supply interface code ****/

static enum power_supply_property bq24251_power_supply_props[] = {
	/* TODO: maybe add more power supply properties */
	POWER_SUPPLY_PROP_STATUS,
};

static int bq24251_power_supply_get_property(struct power_supply *psy,
					     enum power_supply_property psp,
					     union power_supply_propval *val)
{
#if 0
	struct bq24251_device *bq = container_of(psy, struct bq24251_device,
						 charger);
	int ret;
#endif
	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = CHG_STATUS;
#if 0 /* ǥ쥸˥ȥݡ󥰼֤ǤPG low->high->low 
	 ȯˡFAULTꥢƤޤΤǸФǤʤʤ */
		ret = bq24251_i2c_read_mask(bq, 0, 0x30, 4);
		if (ret < 0)
			return ret;
		else if (ret == 0) /* Ready */
			val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
		else if (ret == 1) /* Charge in progress */
			val->intval = POWER_SUPPLY_STATUS_CHARGING;
		else if (ret == 2) /* Charge done */
			val->intval = POWER_SUPPLY_STATUS_FULL;
		else
			val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
#endif
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int bq24251_set_all_register(struct bq24251_device *bq, const unsigned int *val)
{
	int ret = -1;
	unsigned int reg;

	for (reg = 0; reg < 7 ; reg++)
		if ((ret = bq24251_i2c_write(bq, reg, val[reg])) < 0)
			return ret;

	return ret;
}

static int getUsbSrcType(struct bq24251_device *bq)
{
	enum SRC_TYPE type;
	int usbid;
	int reg02;

	usbid = gpio_get_value(GPIONUM_USBID);

	if (usbid == 0)
		return ACADP;

	if (change_usbsw_enable) {
		if (socinfo_get_platform_type() == 4)
			gpio_set_value(GPIONUM_USBSWCONT, 0);
		else
			gpio_set_value(GPIONUM_USBSWCONT, 1);
	}

	bq24251_i2c_write_bit(bq, 0x04, 1, 4);

	msleep(1000); /* 700ms ˥ȽǤʤ*/

	reg02 = bq24251_i2c_read_mask(bq, 0x02, 0x03, 0);
#ifdef BQ24251DEBUGMES
	printk("bq24251: usbtype(reg02):%d\n", reg02);
#endif

	switch(reg02) {
	case 0:	type = DCP;	break;
	case 1:	type = CDP;	break;
	case 2:	type = SDP;	break;
	case 3:	type = NONSTD;	break;
	default:
		type = NONSTD;
		break;
	}

	if (change_usbsw_enable) {
		if (socinfo_get_platform_type() == 4)
			gpio_set_value(GPIONUM_USBSWCONT, 1);
		else
			gpio_set_value(GPIONUM_USBSWCONT, 0);
	}

	return type;
}

static int monitoring(struct bq24251_device *bq, int force)
{
	int pg;
	int err;
	int reg00, reg05;

	pg = gpio_get_value(GPIONUM_PG);
	reg00 = bq24251_i2c_read(bq, 0x00);
	reg05 = bq24251_i2c_read(bq, 0x05);

	/* ְ۾å */
	if ((reg00 & 0x30) == 0x30) {
		switch (reg00 & 0x0F) {
		case 0x0:	FAULT_DETAIL = Normal;	break;
		case 0x1:	FAULT_DETAIL = InOVP;	break;
		case 0x2:	FAULT_DETAIL = InUVLO;	break;
		case 0x3:	FAULT_DETAIL = Sleep;	break;
		case 0x4:
			switch (reg05 & 0x7) {
			case 0x1:	FAULT_DETAIL = TSHot;	break;
			case 0x2:	FAULT_DETAIL = TSWarm;	break;
			case 0x3:	FAULT_DETAIL = TSCool;	break;
			case 0x4:	FAULT_DETAIL = TSCold;	break;
			case 0x7:	FAULT_DETAIL = TSOpen;	break;
			default:	FAULT_DETAIL = Unknown;	break;
			}
			break;
		case 0x5:	FAULT_DETAIL = BatOVP;	break;
		case 0x6:	FAULT_DETAIL = TShutdown;	break;
		case 0x7:	FAULT_DETAIL = Timer;	break;
		case 0x8:	FAULT_DETAIL = NoBat;	break;
		case 0x9:	FAULT_DETAIL = ISET;	break;
		case 0xa:	FAULT_DETAIL = IFandLDO;	break;
		default:	FAULT_DETAIL = Unknown;	break;
		}
	}
	else
		FAULT_DETAIL = Normal;

#ifdef BQ24251DEBUGMES
	if (bq->pre_pg != pg)
		printk("bq24251: PG status changed. %d -> %d\n", bq->pre_pg, pg);
	if (bq->pre_faultdetail != FAULT_DETAIL)
		printk("bq24251: fault detail changed. %d -> %d\n", bq->pre_faultdetail, FAULT_DETAIL);
#endif

	/* IC */
	if (force ||
	    bq->pre_pg != pg ||
	    (reg00 & 0xFF) == 0x13 || (reg00 & 0xFF) == 0x23) {  /* ݡ󥰴֤³Ǣ³ä10x130x23 */
		if (pg == 0) {
			err = bq24251_set_all_register(bq, DEF_REG_VAL);
			if (err < 0)
				return -1;
			switch (getUsbSrcType(bq)) {
			case ACADP:
			case DCP:
				err = bq24251_set_all_register(bq, ACADP_REG_VAL);
#ifdef BQ24251DEBUGMES
				printk("bq24251: SET ACADAP_REG_VAL\n");
#endif
				if (err < 0)
					return -1;
				break;
			case SDP:
			case CDP:
			case NONSTD:
				err = bq24251_set_all_register(bq, DEF_REG_VAL);
#ifdef BQ24251DEBUGMES
				printk("bq24251: SET DEF_REG_VAL\n");
#endif
				if (err < 0)
					return -1;
				break;
			}
		}
		else { /* pg != 0 */
			err = bq24251_set_all_register(bq, DEF_REG_VAL);
#ifdef BQ24251DEBUGMES
			printk("bq24251: SET DEF_REG_VAL\n");
#endif
			if (err < 0)
				return -1;
		}
	}

	bq->pre_pg = pg;
	bq->pre_faultdetail = FAULT_DETAIL;

        switch (reg00 & 0x30) {
	case 0x00:	CHG_STATUS = POWER_SUPPLY_STATUS_NOT_CHARGING;	break;
	case 0x10:	CHG_STATUS = POWER_SUPPLY_STATUS_CHARGING;	break;
	case 0x20:	CHG_STATUS = POWER_SUPPLY_STATUS_FULL;		break;
	default:	CHG_STATUS = POWER_SUPPLY_STATUS_UNKNOWN;	break;
	}

	return 0;
}

static int bq24251_polling_thread(void *arg)
{
	struct i2c_client *client = arg;
	struct bq24251_device *bq = i2c_get_clientdata(client);
	int err;
	
	while (!kthread_should_stop()) {
		msleep(BQ24251_POLLING_INTERVAL);
		if (POLLING_ENABLE)
			err = monitoring(bq, 0);
	}
	return 0;
}

static int bq24251_power_supply_init(struct bq24251_device *bq)
{
	int ret;

	bq->charger.name = "bq24251-charger";
	bq->charger.type = POWER_SUPPLY_TYPE_USB;
	bq->charger.properties = bq24251_power_supply_props;
	bq->charger.num_properties = ARRAY_SIZE(bq24251_power_supply_props);
	bq->charger.get_property = bq24251_power_supply_get_property;

	ret = power_supply_register(bq->dev, &bq->charger);
	if (ret) {
		return ret;
	}

	return 0;
}

static void bq24251_power_supply_exit(struct bq24251_device *bq)
{
	power_supply_unregister(&bq->charger);
}

/* directly set raw value to chip register, format: 'register value' */
static ssize_t bq24251_sysfs_set_registers(struct device *dev,
					   struct device_attribute *attr,
					   const char *buf,
					   size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq24251_device *bq = container_of(psy, struct bq24251_device,
						 charger);
	ssize_t ret = 0;
	unsigned int reg;
	unsigned int val;

	if (sscanf(buf, "%x %x", &reg, &val) != 2)
		return -EINVAL;

	if (reg > 4 || val > 255)
		return -EINVAL;

	ret = bq24251_i2c_write(bq, reg, val);
	if (ret < 0)
		return ret;
	return count;
}

/* print value of chip register, format: 'register=value' */
static ssize_t bq24251_sysfs_print_reg(struct bq24251_device *bq,
				       u8 reg,
				       char *buf)
{
	int ret = bq24251_i2c_read(bq, reg);

	if (ret < 0)
		return sprintf(buf, "%#.2x=error %d\n", reg, ret);
	return sprintf(buf, "%#.2x=%#.2x\n", reg, ret);
}

/* show all raw values of chip register, format per line: 'register=value' */
static ssize_t bq24251_sysfs_show_registers(struct device *dev,
					    struct device_attribute *attr,
					    char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq24251_device *bq = container_of(psy, struct bq24251_device,
						 charger);
	ssize_t ret = 0;
	int i;

	for (i = 0; i <= 6; i++)
		ret += bq24251_sysfs_print_reg(bq, i, buf + ret);

	return ret;
}

static ssize_t bq24251_sysfs_show_gpios(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	int val;
	ssize_t ret = 0;

	val = gpio_get_value(GPIONUM_PG);
	ret += sprintf(buf, "PG(%d):%d\n", GPIONUM_PG, val);
	val = gpio_get_value(GPIONUM_USBID);
	ret += sprintf(buf+ret, "USBID(%d):%d\n", GPIONUM_USBID, val);
	val = gpio_get_value(GPIONUM_USBSWCONT);
	ret += sprintf(buf+ret, "USBSWCONT(%d):%d\n", GPIONUM_USBSWCONT, val);

	return ret;
}

static ssize_t bq24251_sysfs_show_pollsetting(struct device *dev,
					      struct device_attribute *attr,
					      char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", POLLING_ENABLE);
}

static ssize_t bq24251_sysfs_set_pollsetting(struct device *dev,
					     struct device_attribute *attr,
					     const char *buf,
					     size_t count)
{
	int enable;
	if (sscanf(buf, "%d", &enable) != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	if (enable != 0 && enable != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	POLLING_ENABLE = enable;

	return strnlen(buf, PAGE_SIZE);
}

static ssize_t bq24251_sysfs_show_faultdetail(struct device *dev,
					      struct device_attribute *attr,
					      char *buf)
{
	switch(FAULT_DETAIL) {
	case Normal:	return snprintf(buf, PAGE_SIZE, "Normal\n");
	case InOVP:	return snprintf(buf, PAGE_SIZE, "Input OVP\n");
	case InUVLO:	return snprintf(buf, PAGE_SIZE, "Input UVLO\n");
	case Sleep:	return snprintf(buf, PAGE_SIZE, "Sleep\n");
	case TSOpen:	return snprintf(buf, PAGE_SIZE, "TS open\n");
	case TSHot:	return snprintf(buf, PAGE_SIZE, "TS hot\n");
	case TSCold:	return snprintf(buf, PAGE_SIZE, "TS cold\n");
	case TSWarm:	return snprintf(buf, PAGE_SIZE, "TS warm\n");
	case TSCool:	return snprintf(buf, PAGE_SIZE, "TS cool\n");
	case BatOVP:	return snprintf(buf, PAGE_SIZE, "Battery OVP\n");
	case TShutdown:	return snprintf(buf, PAGE_SIZE, "Thermal Shutdown\n");
	case Timer:	return snprintf(buf, PAGE_SIZE, "Timer Fault\n");
	case NoBat:	return snprintf(buf, PAGE_SIZE, "No Battery connected\n");
	case ISET:	return snprintf(buf, PAGE_SIZE, "ISET short\n");
	case IFandLDO:	return snprintf(buf, PAGE_SIZE, "Input Fault and LDO low\n");
	case Unknown:	return snprintf(buf, PAGE_SIZE, "Unknown\n");
	default:	return -EINVAL;
	}

	return -EINVAL;
}

static ssize_t bq24251_sysfs_kick_monitoring(struct device *dev,
					     struct device_attribute *attr,
					     const char *buf,
					     size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq24251_device *bq = container_of(psy, struct bq24251_device,
						 charger);
	int enable, ret;

	if (sscanf(buf, "%d", &enable) != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;
	}

	if (enable != 0 && enable != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;
	}

	ret = monitoring(bq, 1);

	return count;
}

static ssize_t bq24251_sysfs_change_usbsw_enable(struct device *dev,
					     struct device_attribute *attr,
					     const char *buf,
					     size_t count)
{
	int enable;

	if (sscanf(buf, "%d", &enable) != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;
	}

	if (enable != 0 && enable != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;
	}

	change_usbsw_enable = enable;

	return count;
}

static DEVICE_ATTR(registers, S_IWUSR | S_IRUGO,
		bq24251_sysfs_show_registers, bq24251_sysfs_set_registers);
static DEVICE_ATTR(gpios, S_IRUGO,
		bq24251_sysfs_show_gpios, NULL);
static DEVICE_ATTR(polling, S_IWUSR | S_IRUGO,
		bq24251_sysfs_show_pollsetting, bq24251_sysfs_set_pollsetting);
static DEVICE_ATTR(faultdetail, S_IRUGO,
		bq24251_sysfs_show_faultdetail, NULL);
static DEVICE_ATTR(kick_monitoring, S_IWUSR,
		NULL, bq24251_sysfs_kick_monitoring);
static DEVICE_ATTR(change_usbsw_enable, S_IWUSR,
		NULL, bq24251_sysfs_change_usbsw_enable);

static struct attribute *bq24251_sysfs_attributes[] = {
	&dev_attr_registers.attr,
	&dev_attr_gpios.attr,
	&dev_attr_polling.attr,
	&dev_attr_faultdetail.attr,
	&dev_attr_kick_monitoring.attr,
	&dev_attr_change_usbsw_enable.attr,
	NULL,
};

static const struct attribute_group bq24251_sysfs_attr_group = {
	.attrs = bq24251_sysfs_attributes,
};

static int bq24251_sysfs_init(struct bq24251_device *bq)
{
	return sysfs_create_group(&bq->charger.dev->kobj,
			&bq24251_sysfs_attr_group);
}

static void bq24251_sysfs_exit(struct bq24251_device *bq)
{
	sysfs_remove_group(&bq->charger.dev->kobj, &bq24251_sysfs_attr_group);
}

static int bq24251_gpio_init(struct bq24251_device *bq)
{
#define ERRMSG(X)							\
	dev_err(bq->dev, "failed to request GPIO pin %d, error %d\n", X, err)
#define ERRMSG_DIR(X, D)						\
	dev_err(bq->dev, "failed to config GPIO %d as " D " pin, err %d\n", X, err)
	
	int err;
	struct device_node *node = bq->dev->of_node;
	int usbsw_defval;

#ifdef BQ24251DEBUGMES
	printk("bq24251: platform type:%d\n", socinfo_get_platform_type());
#endif

	if (socinfo_get_platform_type() == 4)  /* ES1Τͤ */
		usbsw_defval = 1;
	else
		usbsw_defval = 0;

	of_property_read_u32(node, "ti,chg_pg", &GPIONUM_PG);
#if 0 /* pgbq27421Ǽ */
	if ((err = gpio_request(GPIONUM_PG, "PG_connection")) < 0) {
		ERRMSG(GPIONUM_PG);
		goto PG_fail;
	}
#endif
	of_property_read_u32(node, "ti,usbid", &GPIONUM_USBID);
	if ((err = gpio_request(GPIONUM_USBID, "Cradle_NOT_connection")) < 0) {
		ERRMSG(GPIONUM_USBID);
		goto USB_ID_fail;
	}
	of_property_read_u32(node, "ti,usbsw_cont", &GPIONUM_USBSWCONT);
	if ((err = gpio_request(GPIONUM_USBSWCONT, "USB to PWRIC or to MDM9225")) < 0) {
		ERRMSG(GPIONUM_USBSWCONT);
		goto USBSW_CONT_fail;
	}

#if 0
	if ((err = gpio_direction_input(GPIONUM_PG)) < 0) {
		ERRMSG_DIR(GPIONUM_PG, "input");
		goto error;		
	}
#endif
	if ((err = gpio_direction_input(GPIONUM_USBID)) < 0) {
		ERRMSG_DIR(GPIONUM_USBID, "input");
		goto error;		
	}
	gpio_export(GPIONUM_USBID, false);
	if ((err = gpio_direction_output(GPIONUM_USBSWCONT, usbsw_defval)) < 0) { /* ʤMDM9225³ */
		ERRMSG_DIR(GPIONUM_USBSWCONT, "output");
		goto error;		
	}
	gpio_export(GPIONUM_USBSWCONT, false);

	return 0;

error:
	gpio_free(GPIONUM_USBSWCONT);
USBSW_CONT_fail:
	gpio_free(GPIONUM_USBID);
USB_ID_fail:
#if 0
	gpio_free(GPIONUM_PG);
PG_fail:
#endif
	return err;
#undef ERRMSG
#undef ERRMSG_DIR
}

/* main bq24251 probe function */
static int bq24251_probe(struct i2c_client *client,
			 const struct i2c_device_id *id)
{
	int ret;
	int num;
	struct bq24251_device *bq;

	(void)bq24251_i2c_read_bit;
	(void)bq24251_i2c_write_bit;

	/* Get new ID for the new battery device */
	ret = idr_pre_get(&bq24251_id, GFP_KERNEL);
	if (ret == 0)
		return -ENOMEM;
	mutex_lock(&bq24251_id_mutex);
	ret = idr_get_new(&bq24251_id, client, &num);
	mutex_unlock(&bq24251_id_mutex);
	if (ret < 0)
		return ret;

	bq = devm_kzalloc(&client->dev, sizeof(*bq), GFP_KERNEL);
	if (!bq) {
		dev_err(&client->dev, "failed to allocate device data\n");
		ret = -ENOMEM;
		goto error_2;
	}

	i2c_set_clientdata(client, bq);

	bq->id = num;
	bq->dev = &client->dev;

	ret = bq24251_power_supply_init(bq);
	if (ret) {
		dev_err(bq->dev, "failed to register power supply: %d\n", ret);
		goto error_2;
	}

	ret = bq24251_sysfs_init(bq);
	if (ret) {
		dev_err(bq->dev, "failed to create sysfs entries: %d\n", ret);
		goto error_3;
	}

	dev_info(bq->dev, "driver registered\n");

	if (ret < 0) {
		dev_err(bq->dev, "failed to initialize registers: %d\n", ret);
		goto error_3;
	}

	ret = bq24251_gpio_init(bq);
	if (ret < 0)
		goto error_3;

	ret = monitoring(bq, 1);	/* Initialize charge-IC settings */
	if (ret < 0)
		goto error_3;

	bq24251_polling = kthread_run(bq24251_polling_thread, client, "bq24251");
	if (IS_ERR(bq24251_polling)) {
		dev_err(bq->dev, "failed to create polling thread: %d\n", ret);
		goto error_3;
	}

	return 0;
	
error_3:
	bq24251_power_supply_exit(bq);
error_2:
	mutex_lock(&bq24251_id_mutex);
	idr_remove(&bq24251_id, num);
	mutex_unlock(&bq24251_id_mutex);

	return ret;
}

void bq24251_gpio_free(struct bq24251_device *bq)
{
	gpio_free(GPIONUM_PG);
	gpio_free(GPIONUM_USBID);
	gpio_free(GPIONUM_USBSWCONT);
}

/* main bq24251 remove function */
static int bq24251_remove(struct i2c_client *client)
{
	struct bq24251_device *bq = i2c_get_clientdata(client);

	bq24251_sysfs_exit(bq);
	bq24251_power_supply_exit(bq);

	mutex_lock(&bq24251_id_mutex);
	idr_remove(&bq24251_id, bq->id);
	mutex_unlock(&bq24251_id_mutex);

	dev_info(bq->dev, "driver unregistered\n");

	if (bq24251_polling) {
		kthread_stop(bq24251_polling);
		bq24251_polling = NULL;
	}
	bq24251_gpio_free(bq);

	return 0;
}

static const struct i2c_device_id bq24251_i2c_id_table[] = {
	{ "bq24251", 0 },
	{},
};
MODULE_DEVICE_TABLE(i2c, bq24251_i2c_id_table);

static const struct of_device_id bq24251_match[] = {
	{ .compatible = "ti,bq24251", },
	{ },
};

static struct i2c_driver bq24251_driver = {
	.driver = {
		.name = "bq24251-charger",
		.owner		= THIS_MODULE,
		.of_match_table	= bq24251_match,
	},
	.probe = bq24251_probe,
	.remove = bq24251_remove,
	.id_table = bq24251_i2c_id_table,
};

static int __init bq24251_init(void)
{
	int ret;

	ret = i2c_add_driver(&bq24251_driver);
	if (ret)
		printk(KERN_ERR "Unable to register BQ24251 driver\n");

	return ret;
}
module_init(bq24251_init);

static void __exit bq24251_exit(void)
{
	i2c_del_driver(&bq24251_driver);
}
module_exit(bq24251_exit);

MODULE_AUTHOR("NEC AccessTechinica, Ltd.");
MODULE_DESCRIPTION("bq24251 charger driver");
MODULE_LICENSE("GPL");
