/* Copyright (c) 2013, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013 by NEC AccessTechinica, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>
#include <linux/qpnp/pin.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/leds.h>
#include <linux/regulator/consumer.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>
#include <linux/kthread.h>
#include <linux/dma-mapping.h>

#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/init.h>
#include <linux/sched.h>

#include <mach/sps.h>
#include <mach/socinfo.h>
#include <mach/board.h>

#include "mdss.h"
#include "mdss_panel.h"
#include "mdss_qpic.h"
#include "mdss_qpic_panel.h"

enum writemode {
	WRITEMODE_DIRECT,
	WRITEMODE_BUFFER
};

static enum writemode writemode = WRITEMODE_DIRECT;
static u16 display_buffer[240*320];
static u32 panel_is_on = false;
static u32 display_is_on = false;
static u32 panel_refresh_rate;
static int (*qpic_panel_on)(void);
static void (*qpic_panel_off)(void);
static int (*qpic_display_on)(void);
static void (*qpic_display_off)(void);
static int (*qpic_panel_init)(struct platform_device *pdev,
			struct device_node *np);

#include "bites_image.h"

static volatile u32 panel_disp_flag = 0;

#define PANEL_DISP_ON_BIT		(0)
#define PANEL_DISP_ON_MASK		(0x1)
  #define PANEL_STATE_DISP_ON		(0x1)

#define PANEL_DISP_BOOT_BIT		(1)
#define PANEL_DISP_BOOT_MASK		(0x1)
  #define PANEL_STATE_BOOTING		(0x1)

#define PANEL_DISP_VERTICAL_BIT		(2)
#define PANEL_DISP_VERTICAL_MASK	(0x1)
  #define PANEL_STATE_HORIZON		(0x0)
  #define PANEL_STATE_VERTICAL		(0x1)

#define PANEL_DISP_SLEEP_BIT		(4)
#define PANEL_DISP_SLEEP_MASK		(0x1)
  #define PANEL_STATE_RESUME		(0x0)
  #define PANEL_STATE_SUSPEND		(0x1)
#define PANEL_DISP_WAKEONBT_BIT		(5)
#define PANEL_DISP_WAKEONBT_MASK	(0x1)
  #define PANEL_STATE_WAKEONBT_OFF	(0x0)
  #define PANEL_STATE_WAKEONBT_ON	(0x1)
#define PANEL_DISP_BATTERY_BIT		(8)
#define PANEL_DISP_BATTERY_MASK		(0x7f)
#define PANEL_DISP_CHARGE_BIT		(15)
#define PANEL_DISP_CHARGE_MASK		(0x1)
  #define PANEL_STATE_CHARGE_DISABLE	(0x0)
  #define PANEL_STATE_CHARGE_ENABLE	(0x1)

#define PANEL_DISP_CARRIER_SW_BIT	(16)
#define PANEL_DISP_CARRIER_SW_MASK	(0x1)
  #define PANEL_STATE_CARRIER_SW_EXEC	(0x1)

#define GET_PANEL_DISP_INFO(name)	\
	((panel_disp_flag >> PANEL_DISP_##name##_BIT) & PANEL_DISP_##name##_MASK)
#define SET_PANEL_DISP_INFO(name, val)					\
	panel_disp_flag |= ((val & PANEL_DISP_##name##_MASK) << PANEL_DISP_##name##_BIT)

static struct panel_softc {
	wait_queue_head_t wait_q;
	wait_queue_head_t sleep_wait_q;
	int wait_condition;
	struct mutex display_mutex;
	atomic_t sleeped;
} panel_sc;

static void bites_display_init(struct platform_device *pdev, struct mdss_panel_data *pdata);
struct platform_device *bb_pdev;

static struct task_struct *bites_display_polling = NULL;
static void *bb_virt;
static u32 bb_phys;

u32 qpic_panel_get_framerate(void)
{
	return panel_refresh_rate;
}

u32 qpic_send_panel_cmd(u32 cmd, u32 *val, u32 length)
{
	u32 ret;
	u32 cmd_index;
	u32 size;
	u32 buffer[LCDC_INTERNAL_BUFFER_SIZE];
	u32 i;

	cmd_index = LCDC_EXTRACT_OP_CMD(cmd);
	size = LCDC_EXTRACT_OP_SIZE(cmd);
	if (size == INV_SIZE)
		size = length;
	/* short or pixel data commands need not conversion */
	if ((cmd == OP_WRITE_MEMORY_CONTINUE) ||
		(cmd == OP_WRITE_MEMORY_START)) {
		ret = qpic_flush_buffer(cmd_index, size, val, false);
	} else {
		if (size > LCDC_INTERNAL_BUFFER_SIZE)
			size = LCDC_INTERNAL_BUFFER_SIZE;
		/* correcting for encoding issues */
		for (i = 0; i < size; i += sizeof(u32)) {
			buffer[i] = (val[(i>>2)] >> 0) & 0xff;
			buffer[i+1] = (val[(i>>2)] >> 8) & 0xff;
			buffer[i+2] = (val[(i>>2)] >> 16) & 0xff;
			buffer[i+3] = (val[(i>>2)] >> 24) & 0xff;
		}
		ret = qpic_flush_buffer(cmd_index,
				size * sizeof(u32), buffer, true);
	}
	return ret;
}

u32 qpic_panel_set_cmd_only(u32 command)
{
	u32 param;
	return qpic_send_panel_cmd(command, &param, 0);
}

/* write a frame of pixels to a MIPI screen */
u32 qpic_send_frame(u32 x_start,
				u32 y_start,
				u32 x_end,
				u32 y_end,
				u32 *data,
				u32 total_bytes)
{
	u32 param;
	u32 status;
	u32 start_0_7;
	u32 end_0_7;
	u32 start_8_15;
	u32 end_8_15;

	/* convert to 16 bit representation */
	x_start = x_start & 0xffff;
	y_start = y_start & 0xffff;
	x_end = x_end & 0xffff;
	y_end = y_end & 0xffff;

	/* set column/page */
	start_0_7 = x_start & 0xff;
	end_0_7 = x_end & 0xff;
	start_8_15 = (x_start >> 8) & 0xff;
	end_8_15 = (x_end >> 8) & 0xff;
	param = (start_8_15 << 0) | (start_0_7 << 8) |
		(end_8_15 << 16) | (end_0_7 << 24U);
	status = qpic_send_panel_cmd(OP_SET_COLUMN_ADDRESS, &param, 0);
	if (status) {
		pr_err("Failed to set column address");
		return status;
	}

	start_0_7 = y_start & 0xff;
	end_0_7 = y_end & 0xff;
	start_8_15 = (y_start >> 8) & 0xff;
	end_8_15 = (y_end >> 8) & 0xff;
	param = (start_8_15 << 0) | (start_0_7 << 8) |
		(end_8_15 << 16) | (end_0_7 << 24U);
	status = qpic_send_panel_cmd(OP_SET_PAGE_ADDRESS, &param, 0);
	if (status) {
		pr_err("Failed to set page address");
		return status;
	}

	status = qpic_send_panel_cmd(OP_WRITE_MEMORY_START,
		&(data[0]), total_bytes);
	if (status) {
		pr_err("Failed to start memory write");
		return status;
	}
	return 0;
}

int mdss_qpic_panel_on(struct mdss_panel_data *pdata)
{
	int rc = 0;

	mutex_lock(&panel_sc.display_mutex);
	if (panel_is_on) {
		mutex_unlock(&panel_sc.display_mutex);
		return 0;
	}
	mdss_qpic_init();

	if (qpic_panel_on)
		rc = qpic_panel_on();
	if (rc) {
		mutex_unlock(&panel_sc.display_mutex);
		return rc;
	}
	panel_is_on = true;
	mutex_unlock(&panel_sc.display_mutex);
	return 0;
}

int mdss_qpic_panel_off(struct mdss_panel_data *pdata)
{
	mutex_lock(&panel_sc.display_mutex);
	if (!panel_is_on) {
		mutex_unlock(&panel_sc.display_mutex);
		return 0;
	}
	if (display_is_on)
		qpic_display_off();
	if (qpic_panel_off)
		qpic_panel_off();
	panel_is_on = false;
	mutex_unlock(&panel_sc.display_mutex);
	return 0;
}

int mdss_qpic_display_on(struct mdss_panel_data *pdata)
{
	mutex_lock(&panel_sc.display_mutex);
	if (display_is_on) {
		mutex_unlock(&panel_sc.display_mutex);
		return 0;
	}
	if (qpic_display_on) {
		if (qpic_display_on() < 0) {
			mutex_unlock(&panel_sc.display_mutex);
			return -1;
		}

		display_is_on = true;
	}
	mutex_unlock(&panel_sc.display_mutex);
	return 0;
}

void mdss_qpic_display_off(struct mdss_panel_data *pdata)
{
	mutex_lock(&panel_sc.display_mutex);
	if (!display_is_on) {
		mutex_unlock(&panel_sc.display_mutex);
		return;
	}
	if (qpic_display_off)
		qpic_display_off();
	display_is_on = false;
	mutex_unlock(&panel_sc.display_mutex);
}

static int mdss_panel_parse_dt(struct platform_device *pdev,
			       struct mdss_panel_data *panel_data)
{
	struct device_node *np = pdev->dev.of_node;
	u32 res[6], tmp;
	int rc;

	rc = of_property_read_u32_array(np, "qcom,mdss-pan-res", res, 2);
	if (rc) {
		pr_err("%s:%d, panel resolution not specified\n",
						__func__, __LINE__);
		return -EINVAL;
	}
	panel_data->panel_info.xres = (!rc ? res[0] : 240);
	panel_data->panel_info.yres = (!rc ? res[1] : 320);
	rc = of_property_read_u32(np, "qcom,mdss-pan-bpp", &tmp);
	if (rc) {
		pr_err("%s:%d, panel bpp not specified\n",
						__func__, __LINE__);
		return -EINVAL;
	}
	panel_data->panel_info.bpp = (!rc ? tmp : 24);
	of_property_read_u32(np, "qcom,refresh_rate", &panel_refresh_rate);

	panel_data->panel_info.type = EBI2_PANEL;
	panel_data->panel_info.pdest = DISPLAY_1;

	if (qpic_panel_init)
		rc = qpic_panel_init(pdev, np);

	return rc;
}

static ssize_t qpic_panel_mode_info_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	uint32_t *mode_info;
	mode_info = msm_mode_info_get();
	return sprintf(buf, "0x%x\n", *mode_info);
}

static ssize_t qpic_panel_mode_info_store(struct device *dev,
	 struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	uint32_t *mode_info;

	mode_info = msm_mode_info_get();
	*mode_info = val;

	return count;
}

static DEVICE_ATTR(mode_info, S_IRUGO|S_IWUGO, qpic_panel_mode_info_show,
	qpic_panel_mode_info_store);


static ssize_t qpic_panel_sysfs_show_flag(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0x%x\n", panel_disp_flag);
}

static ssize_t qpic_panel_sysfs_set_flag(struct device *dev,
	 struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	panel_disp_flag = val;
	sc->wait_condition = 1;
	mutex_unlock(&panel_sc.display_mutex);
	wake_up(&sc->wait_q);
	if ((val >> PANEL_DISP_ON_BIT) & PANEL_DISP_ON_MASK)
		return count;	/* 륱 */

	/* 륱 */
	pr_info("%s pre wait_event\n", __func__);
	wait_event(sc->sleep_wait_q, atomic_read(&sc->sleeped) == 1);
	pr_info("%s after wait_event\n", __func__);

	return count;
}

static DEVICE_ATTR(ctl_flag, S_IRUGO|S_IWUGO, qpic_panel_sysfs_show_flag,
	qpic_panel_sysfs_set_flag);

static struct attribute *fs_attrs[] = {
	&dev_attr_mode_info.attr,
	&dev_attr_ctl_flag.attr,
	NULL,
};

static struct attribute_group fs_attr_group = {
	.attrs = fs_attrs,
};

static int qpic_panel_sysfs_add(struct device *dev)
{
	int retval;
	u32 boot_flag = get_msm_boot_flag();

	retval = sysfs_create_group(&dev->kobj, &fs_attr_group);

	mutex_lock(&panel_sc.display_mutex);
	if ((GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) ||
	    ((GET_BOOT_FLAG_INFO(boot_flag, TEST_MODE)
	      == BOOT_FLAG_TEST_MODE_ENABLE))
	    ) {
		panel_disp_flag = 0;
	} else {
		SET_PANEL_DISP_INFO(ON, PANEL_STATE_DISP_ON);
		SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	}
	mutex_unlock(&panel_sc.display_mutex);

	return retval;
}

static int __devinit mdss_qpic_panel_probe(struct platform_device *pdev)
{
	int rc = 0;
	static struct mdss_panel_data vendor_pdata;
	static const char *panel_name;
	struct panel_softc *sc = &panel_sc;

	pr_debug("%s:%d, debug info id=%d", __func__, __LINE__, pdev->id);
	if (!pdev->dev.of_node)
		return -ENODEV;

	panel_name = of_get_property(pdev->dev.of_node, "label", NULL);
	if (!panel_name)
		pr_info("%s:%d, panel name not specified\n",
						__func__, __LINE__);
	else
		pr_info("%s: Panel Name = %s\n", __func__, panel_name);

	/* select panel according to label */
#ifdef CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL
	qpic_panel_init = otm3221b_init;
	qpic_panel_on = otm3221b_power_on;
	qpic_panel_off = otm3221b_power_off;
	qpic_display_on = otm3221b_display_on;
	qpic_display_off = otm3221b_display_off;
#else /* FB_MSM_QPIC_ILI_QVGA_PANEL */
	qpic_panel_init = ili9341_init;
	qpic_panel_on = ili9341_on;
	qpic_panel_off = ili9341_off;
	qpic_display_on = NULL;
	qpic_display_off = NULL;
#endif /* CONFIG_FB_MSM_QPIC_ORISE_QVGA_PANEL */

	rc = mdss_panel_parse_dt(pdev, &vendor_pdata);
	if (rc)
		return rc;

	rc = qpic_register_panel(&vendor_pdata);
	if (rc)
		return rc;

	mutex_init(&panel_sc.display_mutex);

	rc = qpic_panel_sysfs_add(&pdev->dev);
	if (rc)
		return rc;

	init_waitqueue_head(&sc->wait_q);
	init_waitqueue_head(&sc->sleep_wait_q);

	bb_pdev = pdev;
	bites_display_init(pdev, &vendor_pdata);

	return 0;
}

extern void otm3221b_sleep_out(void);

static void
bites_convert_frame_data(const u16 *from, u32 *to,
    u32 pix_len)
{
	int i;

	for (i = 0; i < pix_len; i++, from++, to++) {
		*to = ((u32)(*from & 0xF800) << 7) |
		    ((u32)(*from & 0x0700) << 5) |
		    ((u32)(*from & 0x00C0) << 4) |
		    ((u32)(*from & 0x003F) << 2);
	}
}

static void set_display_writemode(enum writemode mode)
{
	writemode = mode;
}

static enum writemode get_display_writemode(void)
{
	return writemode;
}

static void flush_displaybuffer(void)
{
	bites_convert_frame_data(display_buffer, bb_virt, 320*240);
	
	qpic_send_frame(0, 0, 319, 239,	(u32 *)bb_phys, 320*240*4);
	msleep(20);
}

static void display_reset(struct display_info_t *info)
{
	int bit32_len;

	if (info == NULL)
		return;
	if ((info->image == NULL) || (info->coord == NULL))
		return;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->image->height; y++)
			for (x = 0; x < info->image->width; x++)
				display_buffer[320*(info->coord->y_start + y) + info->coord->x_start + x] = 0;
	}
	else {
		bit32_len = info->image->width * info->image->height * 4;
		memset(bb_virt, 0, bit32_len);
		qpic_send_frame(info->coord->x_start, info->coord->y_start,
				info->coord->x_end, info->coord->y_end,
				(u32 *)bb_phys, bit32_len);
		msleep(20);
	}
}

static void display_all_reset(void)
{
	struct display_info_t disp_info;
	struct image_info_t image;
	struct coordinate_info_t coord_info;

	image.width = 320;
	image.height = 240;
	coord_info.x_start = 0;
	coord_info.y_start = 0;
	coord_info.x_end = 319;
	coord_info.y_end = 239;

	disp_info.image = &image;
	disp_info.coord = &coord_info;

	display_reset(&disp_info);
}


static void display_picto(struct image_info_t *info,
			  struct coordinate_info_t *coord_info, const u16 *data)
{
	int pixel_len = 0;
	int bit32_len = 0;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->height; y++) {
			for (x = 0; x < info->width; x++) {
				display_buffer[320*(coord_info->y_start + y) + coord_info->x_start + x]
					= data[info->width * y + x];
			}
		}
	}
	else {
		pixel_len = info->width * info->height;
		bit32_len = pixel_len * 4;

		bites_convert_frame_data(data, bb_virt, pixel_len);
		qpic_send_frame(coord_info->x_start, coord_info->y_start,
				coord_info->x_end, coord_info->y_end,
				(u32 *)bb_phys, bit32_len);
		msleep(20);
	}
}

static void display_animation_picto(int index, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	if (!vertical) {
		/* horizontal */
		coord_info = &coord_info_horizontal[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
	} else {
		/* vertical */
		coord_info = &coord_info_vertical[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
	}

	data = (u16 *)anime_data[index].data;
	display_picto(info, coord_info, data);
}

static void display_boot_animation(int vertical, int msec)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	while(GET_PANEL_DISP_INFO(ON) &&
	      GET_PANEL_DISP_INFO(BOOT)) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		display_animation_picto(i%2, vertical);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		wait_event_timeout(sc->wait_q, sc->wait_condition,
				   msecs_to_jiffies(msec));
		i++;
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
	display_all_reset();
}

static void display_carrier_switch(int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();

	if (!vertical) {
		info = (struct image_info_t *)&carrier_str_h;
		coord_info = &coord_info_horizontal[CARRIER_STR];
		data = (u16 *)carrier_str_h.data;
	} else {
		info = (struct image_info_t *)&carrier_str_v;
		coord_info = &coord_info_vertical[CARRIER_STR];
		data = (u16 *)carrier_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (!vertical) {
		info = (struct image_info_t *)&switch_str_h;
		coord_info = &coord_info_horizontal[SWITCH_STR];
		data = (u16 *)switch_str_h.data;
	} else {
		info = (struct image_info_t *)&switch_str_v;
		coord_info = &coord_info_vertical[SWITCH_STR];
		data = (u16 *)switch_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (!vertical) {
		info = (struct image_info_t *)&reboot_str_h;
		coord_info = &coord_info_horizontal[REBOOT_STR];
		data = (u16 *)reboot_str_h.data;
	} else {
		info = (struct image_info_t *)&reboot_str_v;
		coord_info = &coord_info_vertical[REBOOT_STR];
		data = (u16 *)reboot_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		flush_displaybuffer();
		set_display_writemode(WRITEMODE_DIRECT);
	}
}

static void display_resume(void)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	int vertical;

	mutex_lock(&panel_sc.display_mutex);
	vertical = GET_PANEL_DISP_INFO(VERTICAL);
	mutex_unlock(&panel_sc.display_mutex);

	if (!vertical) {
		info = (struct image_info_t *)&aterm_logo_h;
		coord_info = &coord_info_horizontal[ATERM_LOGO];
		data = (u16 *)aterm_logo_h.data;
	} else {
		info = (struct image_info_t *)&aterm_logo_v;
		coord_info = &coord_info_vertical[ATERM_LOGO];
		data = (u16 *)aterm_logo_v.data;
	}

	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();
	display_picto(info, coord_info, data);

	mutex_lock(&panel_sc.display_mutex);
	SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	mutex_unlock(&panel_sc.display_mutex);
	display_boot_animation(vertical, 500);
	if (get_display_writemode() == WRITEMODE_BUFFER) {
		flush_displaybuffer();
		set_display_writemode(WRITEMODE_DIRECT);
	}
}

static void display_battery_picto(int remain, int charging, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	int batt_lv = BATT_CHG_LV1;

	if (remain <= 20)
		batt_lv = BATT_CHG_LV1;
	else if (remain <= 40)
		batt_lv = BATT_CHG_LV2;
	else if (remain <= 60)
		batt_lv = BATT_CHG_LV3;
	else if (remain <= 80)
		batt_lv = BATT_CHG_LV4;
	else
		batt_lv = BATT_CHG_LV5;

	if (!charging) {
		batt_lv += BATT_LV1;
		if (remain <= 5)
			batt_lv = BATT_LV0;
	}

	if (remain > 100 || remain < 0) {
		if (!charging)
			batt_lv = BATT_LV0;
		else
			batt_lv = BATT_CHG_LV1;
	}

	if (!vertical) {
		info = (struct image_info_t *)&batt_data[batt_lv];
		coord_info = &coord_info_horizontal[BATTERY_PICTO];
		data = (u16 *)batt_data[batt_lv].data;
	} else {
		info = (struct image_info_t *)&batt_v_data[batt_lv];
		coord_info = &coord_info_vertical[BATTERY_PICTO];
		data = (u16 *)batt_v_data[batt_lv].data;
	}

	display_picto(info, coord_info, data);
}

static void display_battery_remain(int remain, int vertical)
{
	u16 *data;
	struct image_info_t *image;
	struct coordinate_info_t *coord_info;
	struct display_info_t disp_info;
	int i;
	unsigned int num, tmp;
	int divider = 100;
	int num_disp = 0;

	/* % */
	if (!vertical) {
		image = (struct image_info_t *)&num_h_data[PERCENT];
		coord_info = &coord_info_horizontal[REMAIN_PER_PICTO];
		data = (u16 *)num_h_data[PERCENT].data;
	} else {
		image = (struct image_info_t *)&num_v_data[PERCENT];
		coord_info = &coord_info_vertical[REMAIN_PER_PICTO];
		data = (u16 *)num_v_data[PERCENT].data;
	}
	display_picto(image, coord_info, data);

	if (remain > 100 || remain < 0) {
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[MINUS];
			coord_info = &coord_info_horizontal[REMAIN_NUM2_PICTO];
			data = (u16 *)num_h_data[MINUS].data;
		} else {
			image = (struct image_info_t *)&num_v_data[MINUS];
			coord_info = &coord_info_vertical[REMAIN_NUM2_PICTO];
			data = (u16 *)num_v_data[MINUS].data;
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[REMAIN_NUM1_PICTO];
		} else {
			coord_info = &coord_info_vertical[REMAIN_NUM1_PICTO];
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[REMAIN_NUM0_PICTO];
		} else {
			coord_info = &coord_info_vertical[REMAIN_NUM0_PICTO];
		}
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);

		return;
	}

	num_disp = 0;
	tmp = remain;
	for (i = REMAIN_NUM0_PICTO; i <= REMAIN_NUM2_PICTO; i++) {
		if (divider <= 0)
			break;
		num = tmp / divider;
		tmp = tmp % divider;
		divider = divider / 10;
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[num];
			coord_info = &coord_info_horizontal[i];
			data = (u16 *)num_h_data[num].data;
		} else {
			image = (struct image_info_t *)&num_v_data[num];
			coord_info = &coord_info_vertical[i];
			data = (u16 *)num_v_data[num].data;
		}

		if ((num == 0) && (i != REMAIN_NUM2_PICTO) && !num_disp) {
			disp_info.image = image;
			disp_info.coord = coord_info;
			display_reset(&disp_info);
			continue;
		}

		display_picto(image, coord_info, data);
		num_disp = 1;
	}
}

static void display_bt_wave(int enable, int vertical)
{
	u16 *data;
	struct image_info_t *image;
	struct coordinate_info_t *coord_info;

	if (!vertical) {
		image = (struct image_info_t *)&bt_wave_h_data;
		coord_info = &coord_info_horizontal[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_h_data.data;
	} else {
		image = (struct image_info_t *)&bt_wave_v_data;
		coord_info = &coord_info_vertical[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_v_data.data;
	}

	if (!enable) {
		struct display_info_t disp_info;
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);
	} else {
		display_picto(image, coord_info, data);
	}
}

static void display_sleep_picto(int index, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	if (index > 1 || index < 0)
		index = 0;

	if (!vertical) {
		/* horizontal */
		coord_info = &coord_info_horizontal[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_h_data[index];
		data = (u16 *)sleep_h_data[index].data;
	} else {
		/* vertical */
		coord_info = &coord_info_vertical[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_v_data[index];
		data = (u16 *)sleep_v_data[index].data;
	}

	display_picto(info, coord_info, data);
}

static void display_suspend(void)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;
	int batt_remain;
	int bt_enable;
	int charging;
	int vertical;
	u32 prev_panel_disp_flag = panel_disp_flag;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {

		batt_remain = GET_PANEL_DISP_INFO(BATTERY);
		bt_enable = GET_PANEL_DISP_INFO(WAKEONBT);
		charging = GET_PANEL_DISP_INFO(CHARGE);
		vertical = GET_PANEL_DISP_INFO(VERTICAL);

		mutex_unlock(&panel_sc.display_mutex);
		set_display_writemode(WRITEMODE_BUFFER);
		display_all_reset();


		if (batt_remain > 100 || batt_remain < 0)
			batt_remain = -2;

		if (charging && batt_remain == 100) {
			batt_remain--;
		}

		display_battery_picto(batt_remain, charging, vertical);

		display_battery_remain(batt_remain, vertical);

		display_bt_wave(bt_enable, vertical);

		mutex_lock(&panel_sc.display_mutex);
		while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {
			if (prev_panel_disp_flag != panel_disp_flag) {
				prev_panel_disp_flag = panel_disp_flag;
				break;
			}
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			display_sleep_picto(i%2, vertical);
			if (get_display_writemode() == WRITEMODE_BUFFER) {
				flush_displaybuffer();
				set_display_writemode(WRITEMODE_DIRECT);
			}
			wait_event_timeout(sc->wait_q, sc->wait_condition,
			    msecs_to_jiffies(1000));
			i++;
			mutex_lock(&panel_sc.display_mutex);
		}

		mutex_unlock(&panel_sc.display_mutex);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static int bites_display_polling_thread(void *arg)
{
	int rc = 0;
	struct mdss_panel_data *pdata = (struct mdss_panel_data *)arg;
	int disp_len = 320 * 240 * 4;
	struct panel_softc *sc = &panel_sc;
	int vertical = 1;
	u32 boot_flag = get_msm_boot_flag();
	int goto_off = 0;

	bb_virt = dma_alloc_coherent(&bb_pdev->dev, disp_len,
				     &bb_phys, GFP_KERNEL | GFP_DMA);
	
	if (!bb_virt) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ Could not allocate external memory\n");
		return 0;
	}

	rc = mdss_qpic_panel_on(pdata);
	rc = mdss_qpic_display_on(pdata);

	if (GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ LCD disable\n");
	} else {
		if (GET_BOOT_FLAG_INFO(boot_flag, DISP_WAY)
		    == BOOT_FLAG_DISP_HORIZONTAL) {
			/* horizontal */
			vertical = 0;
		} else {
			/* vertical */
			vertical = 1;
		}
		display_boot_animation(vertical, 500);
	}

	while (!kthread_should_stop()) {
		mutex_lock(&panel_sc.display_mutex);
		if (!GET_PANEL_DISP_INFO(ON)) {
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			dev_err(&bb_pdev->dev,
			    "*************** sleep!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);
			if (goto_off) {
				display_all_reset();
				mdss_qpic_display_off(pdata);
				rc = mdss_qpic_panel_off(pdata);
				goto_off = 0;
			}
			atomic_set(&sc->sleeped, 1);
			wake_up(&sc->sleep_wait_q);
			wait_event(sc->wait_q, sc->wait_condition);
			atomic_set(&sc->sleeped, 0);
			dev_err(&bb_pdev->dev,
			    "*************** wake up!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);

			/* NB: ̤ΥĤĴΰ١̤椹 */
			rc = mdss_qpic_panel_on(pdata);
			display_all_reset();
			rc = mdss_qpic_display_on(pdata);
			msleep(30);

			continue;
		}
		if (GET_PANEL_DISP_INFO(BOOT)) {
			mutex_unlock(&panel_sc.display_mutex);
			display_boot_animation(vertical, 500);
		} else if (GET_PANEL_DISP_INFO(CARRIER_SW)) {
			mutex_unlock(&panel_sc.display_mutex);
			display_carrier_switch(vertical);
		} else {
			if (GET_PANEL_DISP_INFO(SLEEP)) {
				mutex_unlock(&panel_sc.display_mutex);
				display_suspend();
				goto_off = 1;
			} else {
				mutex_unlock(&panel_sc.display_mutex);
				display_resume();
			}
		}
	}

	return 0;
}

static void bites_display_init(struct platform_device *pdev, struct mdss_panel_data *pdata)
{
	bites_display_polling = kthread_run(bites_display_polling_thread, pdata, "bites-display");
	if (IS_ERR(bites_display_polling)) {
		pr_err("failed to create polling thread: %ld\n", PTR_ERR(bites_display_polling));
		return;
	}
}

static const struct of_device_id mdss_qpic_panel_match[] = {
	{.compatible = "qcom,mdss-qpic-panel"},
	{}
};

static struct platform_driver this_driver = {
	.probe  = mdss_qpic_panel_probe,
	.driver = {
		.name = "qpic_panel",
		.of_match_table = mdss_qpic_panel_match,
	},
};

static int __init mdss_qpic_panel_init(void)
{
	return platform_driver_register(&this_driver);
}
MODULE_DEVICE_TABLE(of, mdss_qpic_panel_match);
module_init(mdss_qpic_panel_init);
