/*
 * Copyright (c) 2013, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013 by NEC AccessTechinica, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/***************************************************************
 *  Includes
 **************************************************************/
#include <linux/memory.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/regulator/consumer.h>
#include <linux/regulator/msm-gpio-regulator.h>
#include <linux/io.h>
#include <linux/syscalls.h>
#include <linux/leds.h>

#include <mach/hardware.h>
#include <asm/system.h>
#include <asm/mach-types.h>

#include "mdss_qpic.h"
#include "mdss_qpic_panel.h"


/***************************************************************
 *  Definitions
 **************************************************************/
/* OQPQ_XXXX means Orise QVGA Panel with QPIC. */
/* for debug */
#define OQPQ_DEBUG (0)
#if OQPQ_DEBUG
#define OQPQ_DPRINT(fmt, ...) pr_err("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define OQPQ_DPRINT(fmt, ...)
#endif /* OQP_DEBUG */

/* Panel operations */
enum {
    OP_OTM3221B_TEON   = OP_SIZE_PAIR(0x35, 1),
    OP_OTM3221B_MADCTR = OP_SIZE_PAIR(0x36, 1),
    OP_OTM3221B_COLMOD = OP_SIZE_PAIR(0x3a, 1),
};

/* TEON(35h) design */
#define OQPQ_TEON_TELOM_ON_BIT   (1 << 0)
#define OQPQ_TEON_TELOM_OFF_BIT  (0 << 0)

#define OQPQ_TEON_PARAM01 (OQPQ_TEON_TELOM_OFF_BIT)

/* MADCTR(36h) design */
#define OQPQ_MADCTR_MY_BIT  (1 << 7)
#define OQPQ_MADCTR_MX_BIT  (1 << 6)
#define OQPQ_MADCTR_MV_BIT  (1 << 5)
#define OQPQ_MADCTR_ML_TB   (0 << 4)
#define OQPQ_MADCTR_ML_BT   (1 << 4)
#define OQPQ_MADCTR_RGB_RGB (0 << 3)
#define OQPQ_MADCTR_RGB_BGR (1 << 3)

#define OQPQ_MADCTR_PARAM01 \
     (OQPQ_MADCTR_ML_TB | OQPQ_MADCTR_RGB_RGB)

/* MADCTR(3Ah) design */
#define OQPQ_COLMOD_VIPF3_BIT (1 << 7)
#define OQPQ_COLMOD_VIPF2_BIT (1 << 6)
#define OQPQ_COLMOD_VIPF1_BIT (1 << 5)
#define OQPQ_COLMOD_VIPF0_BIT (1 << 4)
#define OQPQ_COLMOD_IFPF2_BIT (1 << 2)
#define OQPQ_COLMOD_IFPF1_BIT (1 << 1)
#define OQPQ_COLMOD_IFPF0_BIT (1 << 0)

#define OQPQ_COLMOD_VIPF_16BPP_2TIMES \
    (OQPQ_COLMOD_VIPF3_BIT | OQPQ_COLMOD_VIPF2_BIT | \
     OQPQ_COLMOD_VIPF1_BIT | OQPQ_COLMOD_VIPF0_BIT)
#define OQPQ_COLMOD_VIPF_18BPP_3TIMES \
    (OQPQ_COLMOD_VIPF3_BIT | OQPQ_COLMOD_VIPF2_BIT | \
     OQPQ_COLMOD_VIPF1_BIT)
#define OQPQ_COLMOD_VIPF_18BPP_1TIMES \
    (OQPQ_COLMOD_VIPF2_BIT | OQPQ_COLMOD_VIPF1_BIT)
#define OQPQ_COLMOD_VIPF_16BPP_1TIMES \
    (OQPQ_COLMOD_VIPF2_BIT | OQPQ_COLMOD_VIPF0_BIT)
#define OQPQ_COLMOD_IFPF_12BPP \
    (OQPQ_COLMOD_IFPF1_BIT | OQPQ_COLMOD_IFPF0_BIT)
#define OQPQ_COLMOD_IFPF_16BPP \
    (OQPQ_COLMOD_IFPF2_BIT | OQPQ_COLMOD_IFPF0_BIT)
#define OQPQ_COLMOD_IFPF_18BPP \
    (OQPQ_COLMOD_IFPF2_BIT | OQPQ_COLMOD_IFPF1_BIT)

#define OQPQ_COLMOD_PARAM01 \
    (OQPQ_COLMOD_VIPF_16BPP_2TIMES | OQPQ_COLMOD_IFPF_16BPP)

/* Max command argument size.
 *   (e.g. max argument = OQP_MAX_PARAM_SZ * 4) */
#define OQPQ_MAX_PARAM_SZ  (20)

/* Command argument */
#define OQPQ_NONE (0x00U)
#define CREATE_PARAM(a, b, c, d) \
    (((a)<<0) | ((b)<<8) | ((c)<<16) | ((d)<<24U))

/***************************************************************
 *  Globals
 **************************************************************/
static int rst_gpio;
static struct regulator *vdd_regulator = NULL;


/***************************************************************
 *  Function declairs
 **************************************************************/
/* Internal functions */
static int otm3221b_check_gpio_varidation(struct platform_device *pdev,
						struct device_node *np);
static int otm3221b_enable_gpio(void);
static void otm3221b_disable_gpio(void);
static void otm3221b_set_custom_param(void);


/***************************************************************
 *  Function definitions
 **************************************************************/
int otm3221b_init(struct platform_device *pdev,
                        struct device_node *np)
{
    int ret;

    /* Confirm GPIOs varidation */
    ret = otm3221b_check_gpio_varidation(pdev, np);
    if (ret)
        return ret;

    return 0;
}

int otm3221b_power_on(void)
{
    int ret;

    /* Panel power on */
    ret = otm3221b_enable_gpio();
    if (ret)
        return ret;

    /* Set custom parameters */
    otm3221b_set_custom_param();

    OQPQ_DPRINT("Panel power on\n");

    return 0;
}

void otm3221b_exit_sleep_mode(void)
{
   /* Set SLPOUT(11h): Sleep Out
     *   Wait for min 120 ms after sleep out. */
    OQPQ_DPRINT("OP_EXIT_SLEEP_MODE(11h)\n");
    qpic_panel_set_cmd_only(OP_EXIT_SLEEP_MODE);
    msleep(120);
}

void otm3221b_set_display_on(void)
{
    /* Set DISPON(29h): Display On
     *   Wait for min 30 ms after display on.
     *   10ms wait - led on (10ms wait) - 10ms wait. */
    OQPQ_DPRINT("OP_SET_DISPLAY_ON(29h)\n");
    qpic_panel_set_cmd_only(OP_SET_DISPLAY_ON);
    msleep(10);
}

void otm3221b_backlight_on(void)
{
    msm_pdm_led_on();
}

int otm3221b_display_on(void)
{
    /* Set SLPOUT(11h): Sleep Out
     *   Wait for min 120 ms after sleep out. */
    OQPQ_DPRINT("OP_EXIT_SLEEP_MODE(11h)\n");
    qpic_panel_set_cmd_only(OP_EXIT_SLEEP_MODE);
    msleep(120);

    /* Set DISPON(29h): Display On
     *   Wait for min 30 ms after display on.
     *   10ms wait - led on (10ms wait) - 10ms wait. */
    OQPQ_DPRINT("OP_SET_DISPLAY_ON(29h)\n");
    qpic_panel_set_cmd_only(OP_SET_DISPLAY_ON);
    msleep(10);

    /* LED On */
    if (msm_pdm_led_on() < 0)
        return -1;
    msleep(10);

    OQPQ_DPRINT("Panel display on\n");

    return 0;
}

void otm3221b_display_off(void)
{
    /* LED Off */
    msm_pdm_led_off();

    /* Set DISPOFF(28h): Display Off *
     *   Wait for min 30 ms after display off. */
    OQPQ_DPRINT("OP_SET_DISPLAY_OFF(28h)\n");
    qpic_panel_set_cmd_only(OP_SET_DISPLAY_OFF);
    msleep(30);

    /* Set SLPOUT(10h): Sleep In
     *   Wait for min 120 ms after sleep in. */
    OQPQ_DPRINT("OP_ENTER_SLEEP_MODE(10h)\n");
    qpic_panel_set_cmd_only(OP_ENTER_SLEEP_MODE);
    msleep(120);

    OQPQ_DPRINT("Panel display off\n");

    return;
}

void otm3221b_power_off(void)
{
    /* Panel power off */
    otm3221b_disable_gpio();

    OQPQ_DPRINT("Panel power off\n");

    return;
}

static int otm3221b_check_gpio_varidation(struct platform_device *pdev,
						struct device_node *np)
{
    /* Get GPIO numbers */
    /*   NOTE: cs/te gpio is not care in this module. */
    if (!vdd_regulator) 
        vdd_regulator = regulator_get(&pdev->dev, GPIO_REGULATOR_DEV_NAME);
    rst_gpio = of_get_named_gpio(np, "qcom,rst-gpio", 0);

    /* Confirm GPIOs varidation */
    if (!gpio_is_valid(rst_gpio)) {
        pr_err("%s: reset gpio is not specified\n" , __func__);
        return -EINVAL;
    }
    OQPQ_DPRINT("Completed to initialize panel driver.\n");

    return 0;
}

static int otm3221b_enable_gpio(void)
{
    /* Request GPIOs */
    if (gpio_request(rst_gpio, "disp_rst_n")) {
        pr_err("%s: Failed to request reset gpio.\n", __func__);
        return -EINVAL;
    }

    /* Enable GPIOs */
    /* Power on:
     *   Wait for min 10 us after vdd on,
     *   and wait for min 5ms (OTP loading). */
    if (regulator_enable(vdd_regulator)) {
        return -EINVAL;
    }
    msleep(10);

    /* SW Reset:
     *   Wait for min 10 us after reset,
     *   and wait for min 120ms (OTP loading). */
    gpio_set_value(rst_gpio, 1);
    msleep(130);

    return 0;
}

static void otm3221b_disable_gpio(void)
{
    /* SW Reset:
     *   Wait formin 120ms. */
    gpio_set_value(rst_gpio, 0);
    msleep(120);

    /* Power off:
     *   HW no requires. But wait for min 10 ms here. */
    regulator_disable(vdd_regulator);
    msleep(10);

    /* Release GPIOs */
    gpio_free(rst_gpio);

    return;
}

static void otm3221b_set_custom_param(void)
{
    u32 param[OQPQ_MAX_PARAM_SZ];

    OQPQ_DPRINT("Set custom parameters\n");

    /* Set panel initial parameters */
    /* Set INVOFF(20h): Display Inversion Off */
    OQPQ_DPRINT("OP_EXIT_INVERT_MODE(20h)\n");
    qpic_panel_set_cmd_only(OP_EXIT_INVERT_MODE);
    msleep(10);

    /* Set TEON(35h): Tearing Effect Line ON */
    param[0] = CREATE_PARAM(OQPQ_TEON_PARAM01,
                            OQPQ_NONE, OQPQ_NONE, OQPQ_NONE);
    OQPQ_DPRINT("TEON(35h) <-- 0x%08x\n" , param[0]);
    qpic_send_panel_cmd(OP_OTM3221B_TEON, param, 0);
    msleep(10);

    /* Set MADCTR(36h): Memory Data Access Control */
    param[0] = CREATE_PARAM(OQPQ_MADCTR_PARAM01,
                            OQPQ_NONE, OQPQ_NONE, OQPQ_NONE);
    OQPQ_DPRINT("MADCTR(36h) <-- 0x%08x\n" , param[0]);
    qpic_send_panel_cmd(OP_OTM3221B_MADCTR, param, 0);
    msleep(10);

    /* Set COLMOD(3Ah): Interface Pixel Format */
    param[0] = CREATE_PARAM(OQPQ_COLMOD_PARAM01,
                            OQPQ_NONE, OQPQ_NONE, OQPQ_NONE);
    OQPQ_DPRINT("COLMOD(3Ah) <-- 0x%08x\n" , param[0]);
    qpic_send_panel_cmd(OP_OTM3221B_COLMOD, param, 0);
    msleep(10);

    /* Wait for 200 ms after completed to set custom parameters. */
    msleep(200);

    return;
}
