/*
 * HITUJI BUTTON driver
 *
 * Copyright (C) 2014 NEC Platforms, Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307USA
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/gpio.h>
#include <asm/atomic.h>
#include <linux/init.h>
#include <linux/genhd.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/wait.h>
#include <linux/sched.h>

#include <linux/htbtn.h>
#include <linux/htbtnvar.h>
#include <linux/pfmgio.h>

#define DRVNAME		"htbtn"
#define DEVNAME		"btnu"

//#define HTBTN_DEBUG

#ifdef HTBTN_DEBUG
#define DEBUG_OUT(x) printk x
#else
#define DEBUG_OUT(x)
#endif  /* HTBTN_DEBUG */

static int dev_major;
static struct class *htbtn_class;

#define HTBTN_TICK 10  /* x 10msec */

static struct timer_list htbtn_timer;
static int timerstop = 0;

static void htbtn_timer_handler( unsigned long data );

struct btnu_event_info {
	#define BTNU_MAX_EVENTS 10
	unsigned sc_event_in, sc_event_out;
	int sc_event[BTNU_MAX_EVENTS];
};

struct htbtn_button {
	struct htbtn_button_unit unit;
#ifdef HTBTN_DEBUG
	int stat;
	int count;
#endif  /* HTBTN_DEBUG */

	int open_count;
	int sc_current;
	int sc_detect;
	unsigned sc_threshold;
	unsigned sc_timeout;
	unsigned sc_count;
	unsigned sc_detect_count;	

	struct btnu_event_info event;
	wait_queue_head_t event_q;
};

static struct htbtn_button *htbtn_buttons;
static int htbtn_num = 0;
#define BTN_COUNT 2


static void
btnu_enqueue_event(struct htbtn_button *sc, int event)
{
	unsigned next;
	struct btnu_event_info *info = &sc->event;

	next = (info->sc_event_in + 1) % BTNU_MAX_EVENTS;
	if (next != info->sc_event_out) {
		info->sc_event[info->sc_event_in] = event;
		info->sc_event_in = next;
	}
	wake_up_interruptible(&sc->event_q);
}

static int
btnu_exist_event(struct htbtn_button *sc)
{
	struct btnu_event_info *info = &sc->event;

	return info->sc_event_in != info->sc_event_out;
}

static int
btnu_dequeue_event(struct htbtn_button *sc)
{
	int event;
	struct btnu_event_info *info = &sc->event;

	if (info->sc_event_in == info->sc_event_out) {
		return BTNU_EVENT_NONE; /* current status */
	}
	event = info->sc_event[info->sc_event_out];
	info->sc_event_out = (info->sc_event_out + 1) % BTNU_MAX_EVENTS;

	return event;
}

static void
btnu_tick(struct htbtn_button *sc)
{
	int cur, detect;

	cur = (gpio_get_value(sc->unit.gpio) == sc->unit.btn_det) ? 1 : 0;
	detect = sc->sc_detect;
	sc->sc_count++;
	if (cur != sc->sc_current) {
		sc->sc_count = 0;
		sc->sc_current = cur;
	} else if (sc->sc_count >= sc->sc_threshold) {
		detect = cur;
		sc->sc_count = sc->sc_threshold;
	}

	if (detect != sc->sc_detect) {
		if (!detect) {/* on->off */
			if (sc->sc_detect_count < sc->sc_timeout) {
				btnu_enqueue_event(sc, BTNU_EVENT_PUSH);
			} else { /* timeout -> off */
				btnu_enqueue_event(sc, BTNU_EVENT_OFF);
			}
		} else {/* off->on */
			btnu_enqueue_event(sc, BTNU_EVENT_ON);
		}
		sc->sc_detect_count = 0;
	} else {/* on->on OR off->off */
		sc->sc_detect_count++;
	}
	if (detect && sc->sc_detect_count == sc->sc_timeout) { /* on->on */
		btnu_enqueue_event(sc, BTNU_EVENT_TIMEOUT);
		sc->sc_detect_count = sc->sc_timeout + 1;
	}
	sc->sc_detect = detect;
}

static void
htbtn_timer_register(struct htbtn_gpio_platform_data *pdata)
{
	init_timer(&htbtn_timer);

	htbtn_timer.expires  = jiffies + HTBTN_TICK;
	htbtn_timer.data     = (unsigned long)pdata;
	htbtn_timer.function = htbtn_timer_handler;

	add_timer(&htbtn_timer);
}

static void
htbtn_timer_handler(unsigned long data)
{
	int i;
	struct htbtn_gpio_platform_data *pdata = (struct htbtn_gpio_platform_data *)data;

	if (timerstop)
		return;

	for (i = 0; i < htbtn_num; i++) {
		struct htbtn_button *p = &htbtn_buttons[i];
#ifdef HTBTN_DEBUG
		int cur;
		int gpio = p->unit.gpio;

		cur = gpio_get_value(gpio);
		if (cur != p->stat) {
			if (p->count++ > BTN_COUNT) {
				printk("@@@ GPIO(%d) %d->%d\n", gpio, p->stat, cur);
				p->stat = cur;
				p->count = 0;
			}
		} else {
			p->count = 0;
		}
#endif  /* HTBTN_DEBUG */

		btnu_tick(p);
	}

	if (!timerstop) {
		htbtn_timer_register(pdata);
	}
}

static long
htbtn_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct htbtn_button *sc;
	unsigned int minor = iminor(file->f_path.dentry->d_inode);
	long retval = 0;

	if (minor >= htbtn_num) {
		return -ENODEV;
	}
	sc = &htbtn_buttons[minor];

	switch (cmd) {
	case BTNU_IOC_READ: {
		struct btnu_read_info *info =
			(struct btnu_read_info *) arg;
		info->on = sc->sc_detect;
		break;
	}
	case BTNU_IOC_EVENT: {
		struct btnu_read_event_info *info =
			(struct btnu_read_event_info *) arg;
		info->event = btnu_dequeue_event(sc);
		break;
	}
	default:
		retval = -EINVAL;
		break;
	}
	return retval;
}

static int
htbtn_open(struct inode *inode, struct file *file)
{
	struct htbtn_button *sc;
	int result = 0;
	unsigned int dev_minor = MINOR(inode->i_rdev);
	struct btnu_event_info *info;

	if (dev_minor >= htbtn_num) {
		return -ENODEV;
	}

	sc = &htbtn_buttons[dev_minor];

	if (sc->open_count > 0) {
		return -EBUSY;
	}

	info = &sc->event;
	memset(info, 0, sizeof(*info));

	sc->open_count++;
	
	return result;
}

static int
htbtn_close(struct inode * inode, struct file * file)
{
	struct htbtn_button *sc;
	unsigned int dev_minor = MINOR(inode->i_rdev);

	if (dev_minor >= htbtn_num) {
		return -ENODEV;
	}

	sc = &htbtn_buttons[dev_minor];

	if (sc->open_count > 0) {
		sc->open_count--;
	}

	return 0;
}

static unsigned int
htbtn_poll(struct file* filp, poll_table* wait)
{
	struct htbtn_button *sc;
	unsigned int minor = iminor(filp->f_path.dentry->d_inode);
	unsigned int retmask = 0;

	if (minor >= htbtn_num) {
		return 0;
	}
	sc = &htbtn_buttons[minor];
	poll_wait(filp, &sc->event_q, wait);

	if (btnu_exist_event(sc)) {
		retmask |= (POLLIN | POLLRDNORM);
	}

	return retmask;
}

static ssize_t
htbtn_read(struct file *filp, char __user *buf, size_t count, loff_t *pos)
{
	struct htbtn_button *sc;
	unsigned int minor = iminor(filp->f_path.dentry->d_inode);
	char strbuf[256];
	int len;

	if (minor >= htbtn_num) {
		return 0;
	}
	sc = &htbtn_buttons[minor];

	snprintf(strbuf, sizeof(strbuf),
		 "current: %s detect: %s\n",
		 sc->sc_current ? "on" : "off",
		 sc->sc_detect ? "on" : "off");

	len = strlen(strbuf);
	if (count < len)
		len = count;

	if (copy_to_user(buf, strbuf, len)) {
		printk(KERN_INFO "%s: copy_to_user failed\n", __func__);
		return -EFAULT;
	}
	*pos += len;

	return len;
}

struct file_operations htbtn_fops = {
	unlocked_ioctl:	htbtn_ioctl,
	open:		htbtn_open,
	read:		htbtn_read,
	poll:		htbtn_poll,
	release:	htbtn_close
};

static int
setup_button(struct htbtn_button_unit *p)
{
	int ret;
	int gpio = p->gpio;

	DEBUG_OUT(("@@@ %s: gpio=%d\n", __func__, gpio));
	if (!gpio_is_valid(gpio)) {
		printk(KERN_INFO "Skipping unavailable BUTTON gpio %d\n", gpio);
		return 0;
	}

	ret = gpio_request(gpio, DRVNAME);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_request() error (%d)\n", __func__, ret);
		return ret;
	}

	ret = gpio_direction_input(gpio);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_direction_input() error (%d)\n", __func__, ret);
		gpio_free(gpio);
		return ret;
	}

	return 0;
}

static int
htbtn_sc_init(struct htbtn_button *sc)
{
	sc->open_count = 0;
	sc->sc_current = sc->sc_detect = 0;
	sc->sc_count = sc->sc_detect_count = 0;
	sc->sc_threshold = sc->unit.threshold / (HTBTN_TICK * 10);
	DEBUG_OUT(("@@@ %s: pfmg_force_sta_mode() = %d\n", __func__, pfmg_force_sta_mode()));
	if (pfmg_force_sta_mode())
		sc->sc_timeout = sc->unit.timeout_sta / (HTBTN_TICK * 10);
	else
		sc->sc_timeout = sc->unit.timeout / (HTBTN_TICK * 10);

	init_waitqueue_head(&sc->event_q);

	return 0;
}

static int
htbtn_probe(struct platform_device *dev)
{
	int result = 0;
	struct htbtn_gpio_platform_data *pdata;
	int i;
	char dev_name[16];
	size_t len;

	dev_major = register_chrdev(0, DEVNAME, &htbtn_fops);
	if (!dev_major) {
		printk(KERN_ERR DRVNAME ": Error whilst opening %s \n", DEVNAME);
		result = -ENODEV;
		goto out;
	}
	htbtn_class = class_create(THIS_MODULE, DRVNAME);
	printk(KERN_INFO DRVNAME ": button registered with major %d\n", dev_major);

	pdata = dev->dev.platform_data;
	if (!pdata->tblnum) {
		result = -EINVAL;
		goto out;
	}

	len = sizeof(struct htbtn_button) * pdata->tblnum;
	htbtn_buttons = kmalloc(len, GFP_KERNEL);
	memset(htbtn_buttons, 0, len);

	htbtn_num = pdata->tblnum;
	for (i = 0; i < htbtn_num; i++) {
		struct htbtn_button *sc;

		sc = &htbtn_buttons[i];
		sc->unit = *(&pdata->unit_tbl[i]);
		setup_button(&sc->unit);

		htbtn_sc_init(sc);

		snprintf(dev_name, sizeof(dev_name)-1, DEVNAME "%d", sc->unit.number);
		device_create(htbtn_class, NULL, MKDEV(dev_major, i), dev, dev_name);
		printk("%s: create /dev/%s gpio=%d\n", DRVNAME, dev_name, sc->unit.gpio);
	}

	htbtn_timer_register(pdata);
out:
	return result;
}

static int
htbtn_remove(struct platform_device *dev)
{
	int i;

	timerstop = 1;
	del_timer_sync(&htbtn_timer);

	for (i = 0; i < htbtn_num; i++) {
		struct htbtn_button *sc;

		sc = &htbtn_buttons[i];
		device_destroy(htbtn_class, MKDEV(dev_major, i));
		gpio_free(sc->unit.gpio);
	}
	class_destroy(htbtn_class);
	unregister_chrdev(dev_major, DEVNAME);
	kfree(htbtn_buttons);
	return 0;
}

static struct
platform_driver htbtn_driver = {
	.probe = htbtn_probe,
	.remove = htbtn_remove,
	.driver = {
		.name = "htbtn-gpio",
		.owner = THIS_MODULE,
	},
};

static int __init
htbtn_mod_init(void)
{
	int ret = platform_driver_register(&htbtn_driver);
	if (ret)
		printk(KERN_INFO DRVNAME ": Error registering platfom driver!\n");

	return ret;
}

static void __exit
htbtn_mod_exit(void)
{
	platform_driver_unregister(&htbtn_driver);
}

module_init (htbtn_mod_init);
module_exit (htbtn_mod_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("Character device for HITUJI LED w/ gpio");
