/*
 * HITUJI LED driver
 *
 * Copyright (C) 2014 NEC Platforms, Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307USA
 */

/*
 * 新規にLED点滅表示(act_blink)を追加する際には、
 * LED_MAX_INTERVALの調整が必要 (BUG#9321)
 */
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/gpio.h>
#include <asm/atomic.h>
#include <linux/init.h>
#include <linux/genhd.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/timer.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/htledvar.h>

#define DRVNAME "htled"
#define DEVNAME "led"

static int dev_major;
static struct class *htled_class;

struct led_softc *led_sc;

/* 500 tick = 5 sec. */
#define HTLED_TICK 5  /* x 10msec */
#define LEDINTERVAL ((HTLED_TICK) * 10) /* 50ms */

static struct timer_list htled_timer;
static int timerstop = 0;

static void htled_timer_handler(unsigned long data);

static const status_name_list_t status_name_list[] = {
	{"none", LEDST_NONE},
	{"off", LEDST_OFF},
	{"green", LEDST_GREEN},
	{"red", LEDST_RED},
	{"orange", LEDST_ORANGE},
	{"blink_green", LEDST_BLINK_GREEN},
	{"blink_red", LEDST_BLINK_RED},
	{"blink_orange", LEDST_BLINK_ORANGE},
	{"blue", LEDST_BLUE},
	{"lightblue", LEDST_LIGHTBLUE},
	{"pink", LEDST_PINK},
	{"white", LEDST_WHITE},
	{"blink_blue", LEDST_BLINK_BLUE},
	{"blink_lightblue", LEDST_BLINK_LIGHTBLUE},
	{"blink_pink", LEDST_BLINK_PINK},
	{"blink_white", LEDST_BLINK_WHITE},
	{NULL, LEDST_MAX}
};

static const blinkmode_name_list_t blinkmode_name_list[] = {
	{"normal", LEDBM_NORMAL},
	{"ignore_eco", LEDBM_IGNORE_ECO},
	{NULL, LEDBM_MAX}
};

static int (* const ledfunc_list[])(ledact_t *) = {
	[LEDFUNC_ACT_ONESHOT] = act_oneshot,
	[LEDFUNC_ACT_ONESHOT_RED] = act_oneshot_red,
	[LEDFUNC_ACT_ONESHOT_ORANGE] = act_oneshot_orange,
	[LEDFUNC_ACT_1SEC_ORANGE] = act_1sec_orange,
	[LEDFUNC_ACT_2SEC_GREEN] = act_2sec_green,
	[LEDFUNC_ACT_BLINK_NORMAL1] = act_blink_normal1,
	[LEDFUNC_ACT_FLUSH] = act_flush,
	[LEDFUNC_ACT_FLUSH_GREEN] = act_flush_green,
	[LEDFUNC_ACT_FLUSH2_RED] = act_flush2_red,
	[LEDFUNC_ACT_FLUSH3_LIGHTBLUE] = act_flush3_lightblue,
	[LEDFUNC_ACT_ON] = act_on,
	[LEDFUNC_ACT_OFF] = act_off,
	[LEDFUNC_ACT_BLINK1] = act_blink1,
	[LEDFUNC_ACT_BLINK2] = act_blink2,
	[LEDFUNC_ACT_BLINK3] = act_blink3,
	[LEDFUNC_ACT_BLINK4] = act_blink4,
	[LEDFUNC_ACT_BLINK5] = act_blink5,
	[LEDFUNC_ACT_BLINK6] = act_blink6,
	[LEDFUNC_ACT_BLINK7] = act_blink7,
	[LEDFUNC_ACT_BLINK8] = act_blink8,
	[LEDFUNC_ACT_BLINK9] = act_blink9,
	[LEDFUNC_ACT_BLINK10] = act_blink10,
	[LEDFUNC_ACT_MAX] = NULL
};

static void
htled_init(struct htled_platform_data *pdata)
{
	struct led_softc *sc = pdata->sc;

	memset(&sc->led, 0, sizeof(sc->led));

	if (sc->type_alias == NULL)
		panic("led: type_alias is null");
	if (sc->act_alias == NULL)
		panic("led: act_alias is null");
	if (sc->action_list == NULL)
		panic("led: action_list is null");
	if (sc->type_name_list == NULL)
		panic("led: type_name_list is null");
	if (sc->act_name_list == NULL)
		panic("led: act_name_list is null");
	if (sc->bitmap == NULL)
		panic("led: bitmap is null");
	if (sc->ledwrite == NULL)
		panic("led: ledwrite function is null");

	led_sc = sc;

	sc->ledwrite(sc->bitmap[LEDT_ALL].mask, 0); /* ALL OFF */
	ledwrite(LEDT_POWER, LEDACT_NORMAL, 1); /* POWER ON */
}

/*
 * タイマー登録処理
 */
static void
htled_timer_register(struct htled_platform_data *pdata)
{
	init_timer(&htled_timer);

	htled_timer.expires  = jiffies + HTLED_TICK;
	htled_timer.data     = (unsigned long)pdata;
	htled_timer.function = htled_timer_handler;

	add_timer(&htled_timer);
}

/***
 * ledbit[].mask が 0 ならば該当LEDは存在しないと判断する
 ***/
int
ledactive(ledtype_t type)
{
	struct led_softc *sc = led_sc;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return 0;
	}
	return sc->bitmap[type].mask == 0 ? 0 : 1;
}
EXPORT_SYMBOL(ledactive);

ledact_status_t
ledstatus(ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return LEDST_NONE;
	}
	action = sc->led.action[type].action;
	for (idx = 0; sc->action_list[idx].action != 0; idx++) {
		if (sc->action_list[idx].status == LEDST_NONE)
			continue;
		if (action & sc->action_list[idx].action)
			break;
	}
	return sc->action_list[idx].status;
}
EXPORT_SYMBOL(ledstatus);

/* LEDACT_INACTIVATEが解除された場合のLED状態を返す BUG#7155 */
ledact_status_t
ledstatus_inactive(ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return LEDST_NONE;
	}
	action = sc->led.action[type].action;
	action &= ~((ledact_bitmap_t) 1 <<
		    (sc->act_alias[LEDACT_INACTIVATE] - 1));
	for (idx = 0; sc->action_list[idx].action != 0; idx++) {
		if (sc->action_list[idx].status == LEDST_NONE)
			continue;
		if (action & sc->action_list[idx].action)
			break;
	}
	return sc->action_list[idx].status;
}
EXPORT_SYMBOL(ledstatus_inactive);

ledact_blinkspeed_t
ledstatus_blink(ledact_blinkmode_t mode, ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	if (sc->blinkspeed_list == NULL)
		return LEDBS_NONE;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX)
		return LEDBS_NONE;

	action = sc->led.action[type].action;

	/* LEDBM_IGNORE_ECO の場合は、点灯状態から INACTIVATE を */
	/* 取り除く BUG#13924 */
	if (mode == LEDBM_IGNORE_ECO)
		action &= ~((ledact_bitmap_t) 1 <<
			    (sc->act_alias[LEDACT_INACTIVATE] - 1));
	for (idx = 0; sc->blinkspeed_list[idx].action != 0; idx++) {
		if (action & sc->blinkspeed_list[idx].action)
			break;
	}
	return sc->blinkspeed_list[idx].blinkspeed;
}
EXPORT_SYMBOL(ledstatus_blink);

/*
 * LED_MAX_INTERVAL定義時には、各act_blinkの点滅周期(on + off)の
 * 公倍数を用いること (BUG#9321)
 *  ・act_blink1   1000ms
 *  ・act_blink2   2000ms
 *  ・act_blink3    400ms
 *  ・act_blink4    800ms
 *  ・act_blink5   1000ms
 *  ・act_blink6    200ms
 *  ・act_blink7    300ms
 *  ・act_blink8   1500ms
 *  ・act_blink9    600ms
 *  ・act_blink10   500ms 
 *    最小公倍数  12000ms
 */
#define LED_MAX_INTERVAL (12000 / LEDINTERVAL) /* 12sec */
/*
 * タイムアウト処理
 */
static void
htled_timer_handler(unsigned long data)
{
	struct htled_platform_data *pdata = (struct htled_platform_data *)data;
	struct led_softc *sc = led_sc;
	ledact_t *act;
	int type;
	int refresh = 0;

	sc->led.count = (sc->led.count + 1) % LED_MAX_INTERVAL;
	for (type = 0, act = sc->led.action;
	     type < LEDT_REAL_MAX; type++, act++) {
		ledact_bitmap_t action = act->action;
		int on;
		int idx;
		
		if (type == LEDT_ALL && !action) {
			if (act->exec_action) {
				refresh = 1;
				act->exec_action = action;
			}
			continue;
		}
		for (idx = 0; sc->action_list[idx].action != 0; idx++) {
			if (action & sc->action_list[idx].action) {
				break;
			}
		}
		if (sc->action_list[idx].sync) {
			act->count = sc->led.count;
		}
		on = ledfunc_list[sc->action_list[idx].func](act);
		/*
		 * LEDACT_INACTIVATE の場合には
		 * この優先度の次の状態も遷移させておかないと、
		 * ledstatus_inactive() の状態が正しく変化しない
		 */
		if (action & ((ledact_bitmap_t)1 << 
			      (sc->act_alias[LEDACT_INACTIVATE] - 1))) {
			int i;

			for (i = idx + 1;
			     sc->action_list[i].action != 0; i++) {
				if (!(action & sc->action_list[i].action)) {
					continue;
				}
				(void)ledfunc_list[sc->action_list[idx].func](act);
				break;
			}
		}
		if (refresh || (act->exec_action != action) ||
		    (on != act->on)) {
			int color = 0;
			switch (on) {
			case 0:
				color = sc->action_list[idx].off_color;
				break;
			case 1:
				color = sc->action_list[idx].on_color;
				break;
			case 2:
				color = sc->action_list[idx].color_3rd;
				break;
			default:
				/* Nothing */
				break;
			}

			sc->ledwrite(sc->bitmap[type].mask, 0);
			if (color != LEDCOLOR_NONE) {
				sc->ledwrite(sc->bitmap[type].color[color], 1);
			}
			act->on = on;
			act->exec_action = action;
		}
		if (type == LEDT_ALL && action) {
			break;
		}
	}

	if (!timerstop) {
		htled_timer_register(pdata);
	}
}

#ifndef LEDACT_FLUSH_1
#define LEDACT_FLUSH_1 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_1 */
int
act_oneshot(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_1] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_10
#define LEDACT_FLUSH_10 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_10 */
int
act_oneshot_red(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &=  ~((ledact_bitmap_t) 1 <<
				  (sc->act_alias[LEDACT_FLUSH_10] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_2
#define LEDACT_FLUSH_2 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_2 */
int
act_oneshot_orange(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_2] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_3
#define LEDACT_FLUSH_3 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_3 */
#define LED1SEC_INTERVAL (1000 / LEDINTERVAL) /* 1sec */
int
act_1sec_orange(ledact_t *act)
{
	struct led_softc *sc = led_sc;

	if (++act->count <= LED1SEC_INTERVAL)
		return 1;
	act->action &= ~((ledact_bitmap_t) 1 <<
			 (sc->act_alias[LEDACT_FLUSH_3] - 1));
	act->count = 0;
	return 0;
}

#ifndef LEDACT_FLUSH_4
#define LEDACT_FLUSH_4 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_4 */
#define LED2SEC_INTERVAL (2000 / LEDINTERVAL) /* 2s */
int
act_2sec_green(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = 1;

	act->count++;
	if (act->count >= LED2SEC_INTERVAL) {/* for no-sync */
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_4] - 1));
		act->count = 0;
		on = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_5
#define LEDACT_FLUSH_5 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_5 */
#define LEDBLINK_NORMAL1_BLINK_INTERVAL (1000 / LEDINTERVAL) /* 1000ms */
#define LEDBLINK_NORMAL1_BLINK_COUNT 3
#define LEDBLINK_NORMAL1_ON_INTERVAL (2000 / LEDINTERVAL) /* 2000ms */
#define LEDBLINK_NORMAL1_INTERVAL (LEDBLINK_NORMAL1_BLINK_INTERVAL * \
				   LEDBLINK_NORMAL1_BLINK_COUNT + \
				   LEDBLINK_NORMAL1_ON_INTERVAL)
int
act_blink_normal1(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = 0;
	unsigned count = act->count % LEDBLINK_NORMAL1_BLINK_INTERVAL;

	act->count++;
	if (act->count < LEDBLINK_NORMAL1_BLINK_INTERVAL *
	    LEDBLINK_NORMAL1_BLINK_COUNT)
		on = (count < (LEDBLINK_NORMAL1_BLINK_INTERVAL / 2)) ? 1 : 0;
	else if (act->count >= LEDBLINK_NORMAL1_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_5] - 1));
		act->count = 0;
		on = 0;
	}
	else
		on = 1;

	return on;
}

#ifndef LEDACT_FLUSH_6
#define LEDACT_FLUSH_6 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_6 */
#define LEDFLUSH_INTERVAL (1000 / LEDINTERVAL) /* 1sec */
#define LEDFLUSH_ONESHOT_COUNT 3
#define LEDFLUSH_TOTAL_INTERVAL (LEDFLUSH_INTERVAL * LEDFLUSH_ONESHOT_COUNT)
int
act_flush(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH_TOTAL_INTERVAL) {	/* for no-sync */
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_6] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_7
#define LEDACT_FLUSH_7 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_7 */
int
act_flush_green(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH_TOTAL_INTERVAL) {	/* for no-sync */
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_7] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_8
#define LEDACT_FLUSH_8 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_8 */
#define LEDFLUSH2_ONESHOT_COUNT 10
#define LEDFLUSH2_TOTAL_INTERVAL (LEDFLUSH_INTERVAL * LEDFLUSH2_ONESHOT_COUNT)
int
act_flush2_red(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH2_TOTAL_INTERVAL) {	/* for no-sync */
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_8] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_9
#define LEDACT_FLUSH_9 LEDACT_DUMMY
#endif /* LEDACT_FLUSH_9 */
#define LEDFLUSH3_INTERVAL (500 / LEDINTERVAL) /* 500ms */
#define LEDFLUSH3_ONESHOT_COUNT 4
#define LEDFLUSH3_TOTAL_INTERVAL (LEDFLUSH3_INTERVAL * LEDFLUSH3_ONESHOT_COUNT)
int
act_flush3_lightblue(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH3_INTERVAL;
	int on = (count < (LEDFLUSH3_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH3_TOTAL_INTERVAL) {	/* for no-sync */
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_9] - 1));
		act->count = 0;
	}
	return on;
}

int
act_on(ledact_t *act)
{
	return 1;
}

int
act_off(ledact_t *act)
{
	return 0;
}

#define LEDBLINK1_ON_INTERVAL (500 / LEDINTERVAL) /* 500ms */
#define LEDBLINK1_OFF_INTERVAL (500 / LEDINTERVAL) /* 500ms */
#define LEDBLINK1_INTERVAL (LEDBLINK1_ON_INTERVAL + LEDBLINK1_OFF_INTERVAL)
int
act_blink1(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK1_INTERVAL;
	int on = (count < LEDBLINK1_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK1_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK2_ON_INTERVAL (1000 / LEDINTERVAL) /* 1sec */
#define LEDBLINK2_OFF_INTERVAL (1000 / LEDINTERVAL) /* 1sec */
#define LEDBLINK2_INTERVAL (LEDBLINK2_ON_INTERVAL + LEDBLINK2_OFF_INTERVAL)
int
act_blink2(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK2_INTERVAL;
	int on = (count < LEDBLINK2_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK2_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK3_ON_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK3_OFF_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK3_INTERVAL (LEDBLINK3_ON_INTERVAL + LEDBLINK3_OFF_INTERVAL)
int
act_blink3(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK3_INTERVAL;
	int on = (count < LEDBLINK3_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK3_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK4_ON_INTERVAL (400 / LEDINTERVAL) /* 400ms */
#define LEDBLINK4_OFF_INTERVAL (400 / LEDINTERVAL) /* 400ms */
#define LEDBLINK4_INTERVAL (LEDBLINK4_ON_INTERVAL + LEDBLINK4_OFF_INTERVAL)
int
act_blink4(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK4_INTERVAL;
	int on = (count < LEDBLINK4_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK4_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK5_ON_INTERVAL (100 / LEDINTERVAL) /* 100ms */
#define LEDBLINK5_OFF_INTERVAL (900 / LEDINTERVAL) /* 900ms */
#define LEDBLINK5_INTERVAL (LEDBLINK5_ON_INTERVAL + LEDBLINK5_OFF_INTERVAL)
int
act_blink5(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK5_INTERVAL;
	int on = (count < LEDBLINK5_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK5_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK6_ON_INTERVAL (100 / LEDINTERVAL) /* 100ms */
#define LEDBLINK6_OFF_INTERVAL (100 / LEDINTERVAL) /* 100ms */
#define LEDBLINK6_INTERVAL (LEDBLINK6_ON_INTERVAL + LEDBLINK6_OFF_INTERVAL)
int
act_blink6(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK6_INTERVAL;
	int on = (count < LEDBLINK6_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK6_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK7_ON_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK7_OFF_INTERVAL (100 / LEDINTERVAL) /* 100ms */
#define LEDBLINK7_INTERVAL (LEDBLINK7_ON_INTERVAL + LEDBLINK7_OFF_INTERVAL)
int
act_blink7(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK7_INTERVAL;
	int on = (count < LEDBLINK7_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK7_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK8_ON_INTERVAL (1000 / LEDINTERVAL) /* 1000ms */
#define LEDBLINK8_OFF_INTERVAL (500 / LEDINTERVAL) /* 500ms */
#define LEDBLINK8_INTERVAL (LEDBLINK8_ON_INTERVAL + LEDBLINK8_OFF_INTERVAL)
int
act_blink8(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK8_INTERVAL;
	unsigned count2 = act->count % LEDBLINK6_INTERVAL;
	int on;

	if (count < LEDBLINK8_ON_INTERVAL && count2 < LEDBLINK6_ON_INTERVAL) {
		on = 1;
	} else {
		on = 0;
	}

	act->count++;
	if (act->count >= LEDBLINK8_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK9_1ST_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK9_2ND_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK9_3RD_INTERVAL (200 / LEDINTERVAL) /* 200ms */
#define LEDBLINK9_INTERVAL (LEDBLINK9_1ST_INTERVAL + \
				LEDBLINK9_2ND_INTERVAL + \
				LEDBLINK9_3RD_INTERVAL)
int
act_blink9(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK9_INTERVAL;
	int on;

	if (count < LEDBLINK9_1ST_INTERVAL) {
		on = 1;
	} else if (count >= LEDBLINK9_1ST_INTERVAL &&
		   count < (LEDBLINK9_1ST_INTERVAL +
			    LEDBLINK9_2ND_INTERVAL)) {
		on = 0;
	} else {
		on = 2;
	}

	act->count++;
	if (act->count >= LEDBLINK9_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

#define LEDBLINK10_ON_INTERVAL (250 / LEDINTERVAL) /* 250ms */
#define LEDBLINK10_OFF_INTERVAL (250 / LEDINTERVAL) /* 250ms */
#define LEDBLINK10_INTERVAL (LEDBLINK10_ON_INTERVAL + LEDBLINK10_OFF_INTERVAL)
int
act_blink10(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK10_INTERVAL;
	int on = (count < LEDBLINK10_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK10_INTERVAL) { /* for no-sync */
		act->count = 0;
	}
	return on;
}

static int
type_name_lookup(struct led_softc *sc, struct led_type_name_info *info)
{
	int i;
	const type_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = sc->type_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->type = list->type;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	/* GET IDX NAME */
	for (i = 0, list = sc->type_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->type = list->type;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static int
act_name_lookup(struct led_softc *sc, struct led_act_name_info *info)
{
	int i;
	const act_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = sc->act_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->act = list->act;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	/* GET IDX NAME */
	for (i = 0, list = sc->act_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->act = list->act;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static int
status_name_lookup(struct led_softc *sc, struct led_status_name_info *info)
{
	int i;
	const status_name_list_t *list;

	if (info->idx == LED_IDX_MATCHSTATUS) {
		for (i = 0, list = status_name_list;
		     list->name != NULL; list++, i++) {
			if (info->status == list->status)
				break;
		}
		if (list->name != NULL) {
			strlcpy(info->name, list->name, sizeof(info->name));
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	/* GET IDX NAME */
	for (i = 0, list = status_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->status = list->status;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static int
blinkmode_name_lookup(struct led_softc *sc, struct led_blinkmode_name_info *info)
{
	int i;
	const blinkmode_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = blinkmode_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->mode = list->mode;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	/* GET IDX NAME */
	for (i = 0, list = blinkmode_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->mode = list->mode;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

/* third argument of user space ioctl ('arg' here) contains the <pin> */
static long
htled_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct led_softc *sc = led_sc;
	int retval = 0;

	switch (cmd)
	{
	case LED_IOC_WRITE: {
		struct led_write_info *info = (struct led_write_info *) arg;

		if ((info->type >= LEDT_MAX) || (info->action >= LEDACT_MAX)) {
			return 0; /* iginore */
		}
		ledwrite(info->type, info->action, info->enable);
		return 0;
	}
	case LED_IOC_SCAN: {
		struct led_scan_info *info = (struct led_scan_info *) arg;

		info->active = ledactive(info->type) ? 1 : 0;
		return 0;
	}
	case LED_IOC_STATUS: {
		struct led_status_info *info = (struct led_status_info *) arg;

		info->status = ledstatus(info->type);
		return 0;
	}
	case LED_IOC_STATUS_INACTIVE: {
		struct led_status_info *info = (struct led_status_info *) arg;

		info->status = ledstatus_inactive(info->type);
		return 0;
	}
	case LED_IOC_TYPENAME: {
		struct led_type_name_info *info = (void *)arg;

		return type_name_lookup(sc, info);
	}
	case LED_IOC_ACTNAME: {
		struct led_act_name_info *info = (void *)arg;

		return act_name_lookup(sc, info);
	}
	case LED_IOC_STATUSNAME: {
		struct led_status_name_info *info = (void *)arg;

		return status_name_lookup(sc, info);
	}
	case LED_IOC_STATUS_BLINKSPEED: {
		struct led_blinkspeed_info *info =
			(struct led_blinkspeed_info *) arg;
		
		info->blinkspeed = ledstatus_blink(info->mode, info->type);
		return 0;
	}
	case LED_IOC_BLINKMODENAME: {
		struct led_blinkmode_name_info *info = (void *)arg;

		return blinkmode_name_lookup(sc, info);
	}
	default:
		retval = -EINVAL;
		break;
	}
	return retval;
}

/* Allow co-incident opens */
static int
htled_open(struct inode *inode, struct file *file)
{
	int result = 0;
	unsigned int dev_minor = MINOR(inode->i_rdev);
	
	if (dev_minor != 0)
	{
		printk(KERN_ERR DRVNAME ": trying to access unknown minor device -> %d\n", dev_minor);
		result = -ENODEV;
		goto out;
	}
out:
	return result;
}

static int
htled_close(struct inode * inode, struct file * file)
{
	/* could track all <pin>s requested by this fd and gpio_free()
	 * them here
	 */
	return 0;
}

struct file_operations htled_fops = {
	.unlocked_ioctl = htled_ioctl,
	.open = htled_open,
	.release = htled_close
};

static int
htled_probe(struct platform_device *dev)
{
	int result = 0;
	struct htled_platform_data *pdata;

	dev_major = register_chrdev(0, DEVNAME, &htled_fops);
	if (!dev_major)
	{
		printk(KERN_ERR DRVNAME ": Error whilst opening %s \n",
		       DEVNAME);
		result = -ENODEV;
		goto out;
	}
	htled_class = class_create(THIS_MODULE, DRVNAME);
	device_create(htled_class, NULL, MKDEV(dev_major, 0), dev, DEVNAME);
	printk(KERN_INFO DRVNAME ": device registered with major %d\n",
	       dev_major);

	pdata = dev->dev.platform_data;
	htled_init(pdata);
	htled_timer_register(pdata);
out:
	return result;
}

static int
htled_remove(struct platform_device *dev)
{
	device_destroy(htled_class, MKDEV(dev_major, 0));
	class_destroy(htled_class);
	unregister_chrdev(dev_major, DEVNAME);
	return 0;
}

static struct
platform_driver htled_driver = {
	.probe = htled_probe,
	.remove = htled_remove,
	.driver = {
		.name = "htled",
		.owner = THIS_MODULE,
	},
};

static int __init
htled_mod_init(void)
{
	int ret = platform_driver_register(&htled_driver);
	if (ret)
		printk(KERN_INFO DRVNAME ": Error registering platfom driver!\n");

	return ret;
}

static void __exit
htled_mod_exit(void)
{
	remove_proc_entry(DRVNAME, NULL);
	platform_driver_unregister(&htled_driver);
}

module_init (htled_mod_init);
module_exit (htled_mod_exit);

EXPORT_SYMBOL(led_sc);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("Character device for HITUJI LED");
