/*
 * HITUJI pfmg-io driver
 *
 * Copyright (C) 2014 NEC Platforms, Ltd. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307USA
 */

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/gpio.h>
#include <asm/atomic.h>
#include <linux/init.h>
#include <linux/genhd.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/pfmgio.h>
#include <linux/string.h>
#include <linux/macutil.h>
#include <linux/mtd/mtd.h>
#include <linux/err.h>

#define DRVNAME		"pfmgio"
#define DEVNAME		"pfmg"

static struct pfmgio_macutil_data {
	u_int8_t additional_info[ADDITIONAL_INFO_LENGTH];
	u_int8_t serial_num[SERIAL_NUM_LENGTH];
	u_int8_t preset_id[PRESET_ID_LENGTH];
	int config_id;
	int random_key;
} mdata = {
	.additional_info = {'\0',},
	.serial_num = {'\0',},
	.preset_id = {'\0',},
	.config_id = -1,
	.random_key = -1,
};

static int dev_major;
static struct class *pfmgio_class;
struct pfmgio_platform_data *pdata = NULL;

static const int mac_type[PFMG_MACADDR_MAX] = {
	-1,                     /* PFMG_MACADDR_UNKNOWN */
	MAC_ADDR_ETHER,         /* PFMG_MACADDR_ETHER0 */
	MAC_ADDR_WAN,           /* PFMG_MACADDR_ETHER1 */
	MAC_ADDR_WLAN,          /* PFMG_MACADDR_WIRELESS0 */
	MAC_ADDR_WLAN_2ND,      /* PFMG_MACADDR_WIRELESS1 */
};

//#define PFMGIO_DEBUG

#ifdef PFMGIO_DEBUG
#define DEBUG_OUT(x) printk x
#else
#define DEBUG_OUT(x)
#endif

int
pfmg_read_macaddr(pfmg_macaddr_id_t id, u_int8_t *addr)
{
	static const mac_addr local_mac =
		{ {0x02, 0x00, 0x00, 0x00, 0x00, 0x00} };
	int i;

	if (id >= PFMG_MACADDR_MAX || id == PFMG_MACADDR_UNKNOWN) {
		return -1;
	}
	for (i = 0; i < pdata->mac_count; i++) {
		if (pdata->mac[i].type == id) {
			memcpy(addr, pdata->mac[i].pmac.mac_addr, PFMG_ETHER_ADDR_LEN);
			return 0;
		}
	}
	memcpy(addr, &local_mac, PFMG_ETHER_ADDR_LEN);
	return -1;
}

EXPORT_SYMBOL(pfmg_read_macaddr);

int
pfmg_force_ap_mode(void)
{
	if (pdata == NULL) {
		printk("%s: not initialized.\n", __func__);
		return 0;
	}

	return (pdata->mode == PFMG_MODE_AP) ? 1 : 0;
}
EXPORT_SYMBOL(pfmg_force_ap_mode);

int
pfmg_force_sta_mode(void)
{
	if (pdata == NULL) {
		printk("%s: not initialized.\n", __func__);
		return 0;
	}

	return (pdata->mode == PFMG_MODE_STA) ? 1 : 0;
}
EXPORT_SYMBOL(pfmg_force_sta_mode);


static void
get_macutil_value(void)
{
	int i = 0;
	for (; i < pdata->mac_count; i++) {
		if (pdata->mac[i].type == PFMG_MACADDR_UNKNOWN)
			continue;
		get_mac_addr(mac_type[pdata->mac[i].type], &pdata->mac[i].pmac);
	}
	mdata.config_id = get_config_id();
	mdata.random_key = get_random_key();
	strlcpy(mdata.serial_num, get_serial_num(),
		sizeof(mdata.serial_num));
	strlcpy(mdata.additional_info, get_additional_info(),
		sizeof(mdata.additional_info));
	strlcpy(mdata.preset_id, get_preset_id(),
		sizeof(mdata.preset_id));
}

static int
pfmgio_open(struct inode *inode, struct file *file)
{
	unsigned int dev_minor = MINOR(inode->i_rdev);

	if (dev_minor != 0) {
		printk(KERN_ERR DRVNAME ": trying to access unknown minor device -> %d\n",
		       dev_minor);
		return -ENODEV;
	}

	return 0;
}

static int
pfmgio_close(struct inode * inode, struct file * file)
{
	return 0;
}

struct file_operations pfmgio_fops = {
	.open = pfmgio_open,
	.release = pfmgio_close,
};

static char *
pfmgio_mac_str(char *buf, size_t len, struct mac_addr *mac)
{
	snprintf(buf, len, "%02x-%02x-%02x-%02x-%02x-%02x",
		 mac->mac_addr[0], mac->mac_addr[1], mac->mac_addr[2],
		 mac->mac_addr[3], mac->mac_addr[4], mac->mac_addr[5]);
	return buf;
}


static int
pfmgio_read_proc(char *page, char **start, off_t off,
		int count, int *eof, void *data)
{
	int i;
	int len;
	char macstr[18];

	snprintf(page, PAGE_SIZE,
		 "rk=%x:sn=%s:pi=%s:aim_cid=%x:",
		 (mdata.random_key) & 0xffff,
		 mdata.serial_num,
		 mdata.preset_id,
		 (mdata.config_id) & 0xff);
	len = strlen(page);

	if (PAGE_SIZE - 1 <= len)
		goto end;

	pfmgio_mac_str(macstr, sizeof(macstr),
		       &pdata->mac[pdata->main_mac_index].pmac);
	snprintf(page + len, PAGE_SIZE - len, "mm=%s:", macstr);
	len = strlen(page);

	for (i = 0; i < pdata->mac_count; i++) {
		if (PAGE_SIZE - 1 <= len)
			goto end;
		if (pdata->mac[i].type == PFMG_MACADDR_UNKNOWN)
			continue;

		pfmgio_mac_str(macstr, sizeof(macstr), &pdata->mac[i].pmac);
		snprintf(page + len, PAGE_SIZE - len, "m%d=%s:", i, macstr);
		len = strlen(page);
	}

	if (PAGE_SIZE - 1 <= len)
		goto end;

	snprintf(page + len, PAGE_SIZE - len,
		 "fap=%d:fsta=%d:rm=%d:dbg=%d:prt=%d:",
		 (pdata->mode == PFMG_MODE_AP) ? 1 : 0,
		 (pdata->mode == PFMG_MODE_STA) ? 1 : 0,
		 (pdata->mode == PFMG_MODE_RT) ? 1 : 0,
		 pdata->debug, pdata->debug);
	len = strlen(page);

end:
	*eof = 1;

	return len;
}

static int
setup_mode_gpio(int gpio)
{
	int ret;

	DEBUG_OUT(("%s: %s port=%d, gpio=%d\n", DRVNAME, __func__, gpio));

	ret = gpio_request(gpio, DRVNAME);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_request() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		return ret;
	}

	ret = gpio_direction_input(gpio);
	if (ret < 0) {
		printk(KERN_INFO "%s: gpio_direction_input() error (%d) gpio=%d\n",
		       __func__, ret, gpio);
		gpio_free(gpio);
		return ret;
	}

	return 0;
}

static int
update_mode_data(struct pfmgio_platform_data *pdata)
{
	int mode0 = pdata->modesw.mode0;
	int mode1 = pdata->modesw.mode1;
	int mode;

	if (setup_mode_gpio(mode0) || setup_mode_gpio(mode1)) {
		return -1;
	}

	mode = (gpio_get_value(mode1) << 1) | gpio_get_value(mode0);
	pdata->mode = pdata->modeval[mode];

	return 0;
}

static int
pfmgio_probe(struct platform_device *dev)
{
	struct mtd_info *mtd;
	struct proc_dir_entry *entry;

	dev_major = register_chrdev(0, DEVNAME, &pfmgio_fops);
	if (!dev_major) {
		printk(KERN_ERR DRVNAME ": Error whilst opening %s \n", DEVNAME);
		return -ENODEV;
	}

	pfmgio_class = class_create(THIS_MODULE, DRVNAME);
	device_create(pfmgio_class, NULL, MKDEV(dev_major, 0), dev, DEVNAME);
	printk(KERN_INFO DRVNAME ": device registered with major %d\n", dev_major);

	pdata = dev->dev.platform_data;

	if (update_mode_data(pdata)) {
		printk("%s: update_mode_data() fail\n", __func__);
		return -EIO;
	}

	entry = create_proc_entry(DEVNAME, S_IRUGO, NULL);
	if (entry == NULL) {
		printk("%s: create_proc_read_entry() fail\n", __func__);
	} else {
		printk("%s: create /proc/%s\n", DRVNAME, DEVNAME);
		entry->data = pdata;
		entry->read_proc = pfmgio_read_proc;
	}

	mtd = get_mtd_device_nm(pdata->mtd_name);
	if (IS_ERR_OR_NULL(mtd)) {
		printk("%s: partition not found\n", __func__);
		return -ENXIO;
	}

	macutil_init(mtd);
	get_macutil_value();

	return 0;
}

static int
pfmgio_remove(struct platform_device *dev)
{
	pdata = dev->dev.platform_data;
	remove_proc_entry(DEVNAME, NULL);
	device_destroy(pfmgio_class, MKDEV(dev_major, 0));
	class_destroy(pfmgio_class);
	unregister_chrdev(dev_major, DEVNAME);

	return 0;
}

static struct
platform_driver pfmgio_driver = {
	.probe = pfmgio_probe,
	.remove = pfmgio_remove,
	.driver = {
		.name = "pfmgio",
		.owner = THIS_MODULE,
	},
};

static int __init
pfmgio_mod_init(void)
{
	int ret = platform_driver_register(&pfmgio_driver);
	if (ret)
		printk(KERN_INFO DRVNAME ": Error registering platfom driver!\n");

	return ret;
}

static void __exit
pfmgio_mod_exit(void)
{
	platform_driver_unregister(&pfmgio_driver);
}

module_init (pfmgio_mod_init);
module_exit (pfmgio_mod_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("Character device for HITUJI pfmg-io");
