/* Copyright(c) NEC Platforms, Ltd. 2001-2015 */
/*
 * $Id: vmgr_utils.c,v 1.1.44.1 2010/06/09 04:44:29 minari Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <syslog.h>
#include <errno.h>
#include <time.h>
#include <linux/videodev2.h>
#include <jpeglib.h>
#include <jerror.h>
#include "videomgr.h"
#include "vmgr_utils.h"

#define IS_MARKER(buf) ((buf[0] == 0xFF) && (buf[1] != 0x00))

#define MARKER_SOF0		0xFFC0
#define MARKER_DHT		0xFFC4
#define MARKER_SOI		0xFFD8
#define MARKER_EOI		0xFFD9
#define MARKER_SOS		0xFFDA

#define MARKER_SIZE		2
#define LENGTH_SIZE		2

#ifdef DEBUG_NO_OVERWRITE
static unsigned int capture_idx = 0;
#endif

static int is_huffman(unsigned char *);
static void yuv_to_jpeg(FILE *, unsigned char *);

#define DHT_SIZE 432

static const unsigned char dht_data[DHT_SIZE] = {
    0xff, 0xc4, 0x00, 0x1f, 0x00, 0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a,
    0x0b, 0xff, 0xc4, 0x00, 0xb5, 0x10, 0x00, 0x02,
    0x01, 0x03, 0x03, 0x02, 0x04, 0x03, 0x05, 0x05, 0x04, 0x04, 0x00, 0x00,
    0x01, 0x7d, 0x01, 0x02, 0x03, 0x00, 0x04, 0x11,
    0x05, 0x12, 0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07, 0x22, 0x71,
    0x14, 0x32, 0x81, 0x91, 0xa1, 0x08, 0x23, 0x42,
    0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0, 0x24, 0x33, 0x62, 0x72, 0x82, 0x09,
    0x0a, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x25, 0x26,
    0x27, 0x28, 0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43,
    0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x53,
    0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67,
    0x68, 0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92,
    0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a,
    0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
    0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
    0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
    0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2, 0xe3, 0xe4,
    0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6,
    0xf7, 0xf8, 0xf9, 0xfa, 0xff, 0xc4, 0x00, 0x1f,
    0x01, 0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02,
    0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0xff, 0xc4, 0x00,
    0xb5, 0x11, 0x00, 0x02, 0x01, 0x02, 0x04, 0x04,
    0x03, 0x04, 0x07, 0x05, 0x04, 0x04, 0x00, 0x01, 0x02, 0x77, 0x00, 0x01,
    0x02, 0x03, 0x11, 0x04, 0x05, 0x21, 0x31, 0x06,
    0x12, 0x41, 0x51, 0x07, 0x61, 0x71, 0x13, 0x22, 0x32, 0x81, 0x08, 0x14,
    0x42, 0x91, 0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33,
    0x52, 0xf0, 0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34, 0xe1, 0x25,
    0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26, 0x27, 0x28,
    0x29, 0x2a, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x43, 0x44, 0x45, 0x46,
    0x47, 0x48, 0x49, 0x4a, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a,
    0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x92, 0x93, 0x94,
    0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3,
    0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
    0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
    0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8,
    0xd9, 0xda, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
    0xe8, 0xe9, 0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa
};

static int
is_huffman(unsigned char *buf)
{
	unsigned char *ptbuf;
	int i = 0;
	ptbuf = buf;
	while (((ptbuf[0] << 8) | ptbuf[1]) != 0xffda) {
		if (i++ > 2048)
			return 0;
		if (((ptbuf[0] << 8) | ptbuf[1]) == 0xffc4)
			return 1;
		ptbuf++;
	}
	return 0;
}

static unsigned char *
search_marker(unsigned int marker, unsigned char *buf, unsigned int bufsize)
{
	unsigned char *ptbuf;
	unsigned char *pos = NULL;
	unsigned int length;

	for (ptbuf = buf; ptbuf <= (buf + bufsize - MARKER_SIZE); ) {
		if (((ptbuf[0] << 8) | ptbuf[1]) == marker) {
			pos = ptbuf;
			break;
		}

		if (((ptbuf[0] << 8) | ptbuf[1]) == MARKER_SOI) {
			ptbuf += MARKER_SIZE;
		} else if (((ptbuf[0] << 8) | ptbuf[1]) == MARKER_EOI) {
			break;
		} else if (((ptbuf[0] << 8) | ptbuf[1]) == MARKER_SOS) {
			break;
		} else if (IS_MARKER(ptbuf)) {
			if (ptbuf > (buf + bufsize - MARKER_SIZE - LENGTH_SIZE))
				break;

			length = ((ptbuf[2] << 8) | ptbuf[3]);
			ptbuf += length + MARKER_SIZE;
		} else {
			break;
		}
	}
	return pos;
}

int
save_picture(int mode)
{
	FILE *file = NULL;
	unsigned char *ptdeb, *ptcur = vd_in->vbuf[vd_in->vbuf_index].mem;
	int sizein, ret = 0;
	unsigned int i;
	
	if (vd_in->vbuf[vd_in->vbuf_index].mem == NULL) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid pointer\n");
		ret = -1;
		goto end;
	}

	file = fopen(vd_in->filename[mode], "wb");
	if (file == NULL) {
		VMGR_LOG_SYSERR("failed to fopen(wb)");
		ret = -1;
		goto end;
	}
	if (!is_huffman(vd_in->vbuf[vd_in->vbuf_index].mem)) {
		ptdeb = ptcur = vd_in->vbuf->mem;
		i = 0;
		while (((ptcur[0] << 8) | ptcur[1]) != 0xffc0) {
			ptcur++;
			i++;
			if (vd_in->newlen <= i) {
				VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
					"Invalied data format.\n");
				ret = -1;

				goto end;
			}
		}
		sizein = ptcur - ptdeb;
		fwrite(vd_in->vbuf[vd_in->vbuf_index].mem, sizein, 1, file);
		fwrite(dht_data, DHT_SIZE, 1, file);
		fwrite(ptcur, vd_in->newlen - sizein, 1, file);
	} else {
		fwrite(ptcur, vd_in->newlen, 1, file);
	}

	fflush(file);	
	fclose(file);
	file = NULL;

end:
	if (file != NULL) {
		fclose(file);
	}

	return ret;
}

int
get_picture(int mode)
{
	int fd;
	char tmp_fname[VMGR_PATH_LEN];
	char fname[VMGR_PATH_LEN];
	int ret = 0;
	unsigned int length;
	FILE *file;
	unsigned char *pttop, *ptframe, *ptwrite;
	unsigned int i;

	if ((vd_in->vbuf == NULL) ||
	    (vd_in->vbuf[vd_in->vbuf_index].mem == NULL)) {
		VMGR_LOG(LOG_ERR, VMGR_LOCATION, "invalid pointer\n");
		ret = -1;
		goto end;
	}

#ifdef DEBUG_NO_OVERWRITE
	sprintf(tmp_fname, "%s_%d.jpg", STILL_FNAME, capture_idx);
#else
	sprintf(tmp_fname, "%s.jpg", STILL_FNAME_TMP);
	sprintf(fname, "%s.jpg", STILL_FNAME);
#endif

	if (vd_in->info[mode].format == V4L2_PIX_FMT_MJPEG) {
		fd = open(tmp_fname, O_CREAT|O_TRUNC|O_RDWR, 0644);
		if (fd < 0) {
			VMGR_LOG_SYSERR("failed to open");
			ret = -1;
			goto end;
		}

		if (!search_marker(MARKER_DHT, vd_in->vbuf[vd_in->vbuf_index].mem,
							vd_in->newlen)) {
			pttop = vd_in->vbuf[vd_in->vbuf_index].mem;
			ptframe = search_marker(MARKER_SOF0,
									vd_in->vbuf[vd_in->vbuf_index].mem,
									vd_in->newlen);
			if (!ptframe) {
				VMGR_LOG(LOG_ERR, VMGR_LOCATION, 
					"Invalied data format.\n");
				ret = -1;
				close(fd);
				goto end;
			}

			length = ptframe - pttop;
			ptwrite = pttop;
			while (length > 0) {
				ret = write(fd, ptwrite, length);
				if (ret < 0)
					break;
				length -= ret;
				ptwrite += ret;
			}
			length = DHT_SIZE;
			ptwrite = dht_data;
			while (length > 0) {
				ret = write(fd, ptwrite, length);
				if (ret < 0)
					break;
				length -= ret;
				ptwrite += ret;
			}
			length = vd_in->newlen - (ptframe - pttop);
			ptwrite = ptframe;
			while (length > 0) {
				ret = write(fd, ptwrite, length);
				if (ret < 0)
					break;
				length -= ret;
				ptwrite += ret;
			}
		} else {
			/* 画像ファイル書き込み */
			length = vd_in->newlen;
			ptwrite = vd_in->vbuf[vd_in->vbuf_index].mem;
			while (length > 0) {
				ret = write(fd, ptwrite, length);
				if (ret < 0)
					break;
				length -= ret;
				ptwrite += ret;
			}
		}

		close(fd);
	}
	else if (vd_in->info[mode].format == V4L2_PIX_FMT_YUYV) {
		file = fopen(tmp_fname, "wb");
		if (file == NULL) {
			VMGR_LOG_SYSERR("failed to fopen(wb)");
			ret = -1;
			goto end;
		}

		/* YUV->JPEG画像変換 */
		yuv_to_jpeg(file, vd_in->vbuf[vd_in->vbuf_index].mem);

		fclose(file);
	}

#ifdef DEBUG_NO_OVERWRITE
	capture_idx++;
#else
	rename(tmp_fname, fname);
#endif

end:
	return ret;
}

static void yuv_to_jpeg(FILE *fp, unsigned char *image)
{
	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;
	int mode;
	unsigned char *newimage = NULL;
	unsigned int i, j, offset;
	JSAMPROW row_pointer[1];

	/* jpegオブジェクトの初期化 */
	cinfo.err = jpeg_std_error( &jerr );
	jpeg_create_compress(&cinfo);

	/* 出力画像ファイル設定 */
	jpeg_stdio_dest(&cinfo, fp);

	/* パラメータ設定 */
	mode = get_current_mode();
	cinfo.image_width = vd_in->info[mode].width;
	cinfo.image_height = vd_in->info[mode].height;
	cinfo.input_components = 3; /* 1ピクセルあたりの色数 */
	cinfo.in_color_space = JCS_YCbCr;
	jpeg_set_defaults(&cinfo);
	jpeg_set_quality(&cinfo, 90, TRUE); /* 画質設定 */

	/* 圧縮開始 */
	jpeg_start_compress(&cinfo, TRUE);

	/* 圧縮 */
	newimage = vd_in->jpg_buf;
	memset(newimage, 0x00,
		vd_in->info[mode].height * vd_in->info[mode].width * 3);
	row_pointer[0] = newimage;
	while (cinfo.next_scanline < cinfo.image_height) {
		offset = cinfo.next_scanline * cinfo.image_width * 2;
		for (i = 0, j = 0; i < cinfo.image_width*2; i += 4, j += 6) {
			newimage[j + 0] = image[offset + i + 0]; // Y
			newimage[j + 1] = image[offset + i + 1]; // U
			newimage[j + 2] = image[offset + i + 3]; // V
			newimage[j + 3] = image[offset + i + 2]; // Y
			newimage[j + 4] = image[offset + i + 1]; // U
			newimage[j + 5] = image[offset + i + 3]; // V
		}
		jpeg_write_scanlines(&cinfo, row_pointer, 1);
	}

	/* 圧縮終了 */
	jpeg_finish_compress(&cinfo);

	/* jpegオブジェクトの破棄 */
	jpeg_destroy_compress(&cinfo);

	return;
}
