/*
 * Copyright (C) 2004-2006 Kay Sievers <kay@vrfy.org>
 * Copyright (C) 2006 Hannes Reinecke <hare@suse.de>
 *
 *	This program is free software; you can redistribute it and/or modify it
 *	under the terms of the GNU General Public License as published by the
 *	Free Software Foundation version 2 of the License.
 * 
 *	This program is distributed in the hope that it will be useful, but
 *	WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	General Public License for more details.
 * 
 *	You should have received a copy of the GNU General Public License along
 *	with this program; if not, write to the Free Software Foundation, Inc.,
 *	51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

/* Include header files */
#include "btnctrl_proc.h"

/* /proc entry data storage buffers */
static char btnctrl_buf[BUFFER_SIZE], status_buf[BUFFER_SIZE];


/* Proc handlers for "/proc/btnctrl" */

/* This function shows the contents of "/proc/btnctrl" */
static int btnctrl_proc_show(struct seq_file *seq, void *v)
{
	seq_printf(seq, "%s", btnctrl_buf);
	return 0;
}

/* This function opens "/proc/btnctrl" */
static int btnctrl_proc_open(struct inode * inode, struct file *file)
{
	if (!MODULE_GET)
		return 0;

	return single_open(file, btnctrl_proc_show, NULL);
}

/* This function reads in data from the user into the kernel for "/proc/btnctrl"
 * Validate User Buffer for the allowable strings ("start\n" and "stop\n") and report error if supplied otherwise */
static int btnctrl_proc_write(struct file *file, const char __user *buf, size_t size, loff_t *offset)
{
	char test_buf[BUFFER_SIZE] = {'\0'};

	if (size >= BUFFER_SIZE)
	{
		printk(KERN_ERR "btnctrl_proc: Data size %zu from User exceeded limit %d!\n", size, BUFFER_SIZE - 1);
		return -EFAULT;
	}

	if (copy_from_user(test_buf, buf, size))
		return -EIO;

	if (strcmp(test_buf, "start\n") && strcmp(test_buf, "stop\n"))
	{
		printk(KERN_ERR "btnctrl_proc: Invalid btnctrl Status - %s!\n", test_buf);
		return -EINVAL;
	}

	memset(btnctrl_buf, '\0', BUFFER_SIZE);
	strcpy(btnctrl_buf, test_buf);

	return (int) size;
}

/* This function closes "/proc/btnctrl" */
static int btnctrl_proc_release(struct inode *inode, struct file *file)
{
	MODULE_PUT;
	return 0;
}

/* File operations for "/proc/btnctrl" */
struct file_operations btnctrl_fops = {
	.owner = THIS_MODULE,
	.open = btnctrl_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.write = btnctrl_proc_write,
	.release = btnctrl_proc_release
};


/* Proc handlers for "/proc/status" */

/* This function shows the contents of "/proc/status" */
static int status_proc_show(struct seq_file *seq, void *v)
{
	seq_printf(seq, "%s", status_buf);
	return 0;
}

/* This function opens "/proc/status" */
static int status_proc_open(struct inode * inode, struct file *file)
{
	if (!MODULE_GET)
		return 0;

	return single_open(file, status_proc_show, NULL);
}

/* This function reads in data from the user into the kernel for "/proc/status"
 * Validate User Buffer for the allowable strings ("normal\n" and "readytoinitialization\n") and report error if supplied otherwise */
static int status_proc_write(struct file *file, const char __user *buf, size_t size, loff_t *offset)
{
	char test_buf[BUFFER_SIZE] = {'\0'};

	if (size >= BUFFER_SIZE)
	{
		printk(KERN_ERR "btnctrl_proc: Data size %zu from User exceeded limit %d!\n", size, BUFFER_SIZE - 1);
		return -EFAULT;
	}

	if (copy_from_user(test_buf, buf, size))
		return -EIO;

	if (strcmp(test_buf, "normal\n") && strcmp(test_buf, "readytoinitialization\n"))
	{
		printk(KERN_ERR "btnctrl_proc: Invalid system Status - %s!\n", test_buf);
		return -EINVAL;
	}

	memset(status_buf, '\0', BUFFER_SIZE);
	strcpy(status_buf, test_buf);

	return (int) size;
}

/* This function closes "/proc/status" */
static int status_proc_release(struct inode *inode, struct file *file)
{
	MODULE_PUT;
	return 0;
}

/* File operations for "/proc/status" */
struct file_operations status_fops = {
	.owner = THIS_MODULE,
	.open = status_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.write = status_proc_write,
	.release = status_proc_release
};


/* Remove /proc file entries */
static int remove_btnctrl_proc(void)
{
	remove_proc_entry(PROC_ENTRY_BTNCTRL_FILENAME, NULL);
	remove_proc_entry(PROC_ENTRY_STATUS_FILENAME, NULL);
	printk(KERN_INFO "btnctrl_proc: exit btnctrl_proc\n");

	return 0;
}

/* Create /proc file entries */
static int create_btnctrl_proc(void)
{
	if (!proc_create(PROC_ENTRY_BTNCTRL_FILENAME, 0666, NULL, &btnctrl_fops))
	{
		printk(KERN_INFO "btnctrl_proc: ERROR - proc_create() failed to create entry: %s!\n", PROC_ENTRY_BTNCTRL_FILENAME);
		return -ENOMEM;
	}
	if (!proc_create(PROC_ENTRY_STATUS_FILENAME, 0666, NULL, &status_fops))
	{
		printk(KERN_INFO "btnctrl_proc: ERROR - proc_create() failed to create entry: %s!\n", PROC_ENTRY_STATUS_FILENAME);
		return -ENOMEM;
	}
	printk(KERN_INFO "btnctrl_proc: init btnctrl_proc\n");

	return 0;
}

static int __init init_btnctrl_proc (void)
{
	create_btnctrl_proc();
	return 0;
}

static void __exit exit_btnctrl_proc(void)
{
	remove_btnctrl_proc();
}

module_init(init_btnctrl_proc);
module_exit(exit_btnctrl_proc);
MODULE_AUTHOR("Alumnus Software Ltd. <info@alumnux.com>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("A btnctrl module to input/output using proc filesystem");
