/*
 * IO-expander control driver
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/atomic.h>
#include <linux/init.h>
//#include <linux/genhd.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/kthread.h>
#include <linux/jiffies.h>
#include <linux/types.h>
#include <linux/mutex.h>

#include <linux/seq_file.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/of_gpio.h>
#include <linux/i2c.h>

#include <linux/uaccess.h>

#include <linux/ioexp.h>
#include <linux/ioexp-dev.h>

//#define DEBUG_IOEXP
#if defined(DEBUG_IOEXP)
#define IOEXP_DPRINT printk
#else  /* DEBUG_IOEXP */
#define IOEXP_DPRINT(...)
#endif	/* DEBUG_IOEXP */

#define DRVNAME "IO-expnader"
#define DEVNAME "IO-expander"

/* 500 tick = 5 sec. */
#define IOEXP_TICK 2  /* x 10msec */

#define I2C_BUS 0

static int dev_major;
static struct class *ioexp_class;

static struct ioexp_param ioexp_bus[IOEXP_BUS_MAX] = {
	[IOEXP_BUS0] = {"ioexp_bus0", IOEXP1, 0xff, {0}, {0}, {0}, {0}, {{0}}, IOEXP_INACTIVE,},
	[IOEXP_BUS1] = {"ioexp_bus1", IOEXP2, 0xff, {0}, {0}, {0}, {0}, {{0}}, IOEXP_INACTIVE,},
};

static struct task_struct *thread;

static DEFINE_MUTEX(ioexp_i2c_lock);

static int ioexp_write_sync(int bus, int offset);
static void ioexp_config(void);

/*
 * ioexp_set_value: 外部モジュールからの要求により使用される関数
 *                  変更要求のあった bit の値を変更する
 *                  この関数は非同期処理で、 HW 制御を直接行うわけではなく、
 *                  スレッドでの定期処理の中で IO-expander の値が変わる
 *                  同期処理で HW 制御したい場合は ioexp_set_value_sync を使用する
 */
int
ioexp_set_value(ioexp_device_t device, int port, int value)
{
	int bus, offset;

	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		if (device == ioexp_bus[bus].device)
			break;
	}
	if (bus >= IOEXP_BUS_MAX) {
		pr_err("@@@ [%s:%d] device: %d is invalid parameter\n", __func__, __LINE__, device);
		goto err;
	}

	for (offset = 0; offset < OFFSET_NUM; offset++) {
		if (port >= (offset * 10) && port <= ((offset * 10) + PORT_NUM - 1)) {
			port = port - (offset * 10);
			break;
		}
	}
	if (offset >= OFFSET_NUM) {
		pr_err("@@@ [%s:%d] port: %d is invalid parameter\n", __func__, __LINE__, port);
		goto err;
	}

	if (value != 0 && value != 1) {
		pr_err("@@@ [%s:%d] value: %d is invalid parameter\n", __func__, __LINE__, value);
		goto err;
	}

	ioexp_bus[bus].next_output_value[offset][port] = value;

	return 0;
err:
	return -EINVAL;
}
EXPORT_SYMBOL(ioexp_set_value);

int
ioexp_set_value_sync(ioexp_device_t device, int port, int value)
{
	int ret, bus, offset, result;

	ret = ioexp_set_value(device, port, value);
	if (ret != 0)
		goto err;

	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		if (device == ioexp_bus[bus].device)
			break;
	}

	for (offset = 0; offset < OFFSET_NUM; offset++) {
		if (port >= (offset * 10) && port < ((offset * 10) + PORT_NUM)) {
			port = port - (offset * 10);
			break;
		}
	}

	result = ioexp_write_sync(bus, offset);

	return result;
err:
	return ret;
}
EXPORT_SYMBOL(ioexp_set_value_sync);

/*
 * i2c_read: 制御に必要なパラメータを I2C へ転送して IO-expander の値を読み出す
 */
static u8
i2c_read(u8 devaddr, u8 regoffset)
{
	struct i2c_adapter *adap;
	int err;
	struct i2c_msg msg[2];
	unsigned char data[1];
	u8 value = 0;

	adap = i2c_get_adapter(I2C_BUS);
	if (adap == NULL) {
		pr_err("@@@ [%s:%d] i2c_get_adapter NULL\n",
		       __func__, __LINE__);
		return -ENODEV;
	}
	msg[0].addr = devaddr;   /* I2C address of chip */
	msg[0].flags = 0;
	msg[0].len = 1;
	msg[0].buf = data;
	data[0] = regoffset;     /* register num */
	msg[1].addr = devaddr;   /* I2C address of chip */
	msg[1].flags = I2C_M_RD; /* Read operation */
	msg[1].len = 1;
	msg[1].buf = &value;     /* Read value */
	err = i2c_transfer(adap, msg, ARRAY_SIZE(msg));
	i2c_put_adapter(adap);
	if (err < 0) {
		pr_err("@@@ [%s:%d] err:%d\n", __func__, __LINE__, err);
		return 0;
	}
	return value;
}

/*
 * i2c_write: 制御に必要なパラメータを I2C へ転送して IO-expander を制御する
 */
static int
i2c_write(u8 devaddr, u8 regoffset, u8 value)
{
	struct i2c_adapter *adap;
	int err;
	struct i2c_msg msg[1];
	unsigned char data[2];

	adap = i2c_get_adapter(I2C_BUS);
	if (adap == NULL) {
		pr_err("@@@ [%s:%d] i2c_get_adapter NULL\n",
			   __func__, __LINE__);
		return -ENODEV;
	}

	msg->addr = devaddr;    /* I2C address of chip */
	msg->flags = 0;
	msg->len = 2;
	msg->buf = data;
	data[0] = regoffset;    /* register num */
	data[1] = value;        /* register data */

	err = i2c_transfer(adap, msg, 1);
	i2c_put_adapter(adap);
	if (err < 0) {
		pr_err("@@@ [%s:%d] err:%d\n", __func__, __LINE__, err);
		return err;
	}
	return 0;
}

/*
 * next_value_set: 各 bit の値を1つの変数にまとめて、その値を返す
 */
static u8
next_value_set(int bus, int offset)
{
	u8 next_value = 0;
	int port;
	struct ioexp_param *ioexp;

	ioexp = &ioexp_bus[bus];
	for (port = 0; port < PORT_NUM; port++)
		next_value |= (ioexp->next_output_value[offset][port]) << port;

	return next_value;
}

/*
 * ioexp_write     : スレッドで定期的に制御有無を確認し、I2C 転送に必要なパラメータを送信する
 *                   IO-expander の値が変更された場合、set_value を更新する
 * ioexp_write_sync: スレッドの定期処理とは別に制御要求があったタイミングでパラメータを送信する
 */
static int
ioexp_write_sync(int bus, int offset)
{
	int result;
	u8 devaddr, regoffset, next_set;
	struct ioexp_param *ioexp;

	ioexp = &ioexp_bus[bus];
	next_set = next_value_set(bus, offset);
	devaddr = ioexp->bus_addr;
	regoffset = ioexp->output_pin[offset];

	mutex_lock(&ioexp_i2c_lock);
	result = i2c_write(devaddr, regoffset, next_set);
	mutex_unlock(&ioexp_i2c_lock);

	if (result >= 0)
		ioexp->output_value[offset] = next_set;

	return result;
}

static void
ioexp_write(unsigned long data)
{
	int result, bus, offset;
	u8 devaddr, regoffset, set, next_set;
	struct ioexp_param *ioexp;

	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		ioexp = &ioexp_bus[bus];

		if (ioexp->active != IOEXP_ACTIVE)
			continue;

		for (offset = 0; offset < OFFSET_NUM; offset++) {
			set = ioexp->output_value[offset];
			next_set = next_value_set(bus, offset);

			if (next_set != set) {
				devaddr = ioexp->bus_addr;
				regoffset = ioexp->output_pin[offset];

				mutex_lock(&ioexp_i2c_lock);
				result = i2c_write(devaddr, regoffset, next_set);
				mutex_unlock(&ioexp_i2c_lock);

				if (result >= 0)
					ioexp->output_value[offset] = next_set;
			}
		}
	}
}

static int
ioexp_thread(void *data)
{
	unsigned long jif = jiffies;
	long j;

	while (!kthread_should_stop()) {
		j = jif - jiffies;
		if (j <= 0)
			j = 1;
		set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(j);
		jif += IOEXP_TICK;

		ioexp_write((unsigned long)data);
	}
	return 0;
}

static void
ioexp_config()
{
	int result, bus, offset;
	u8 devaddr, set_offset, set_val;
	struct ioexp_param *ioexp;

	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		ioexp = &ioexp_bus[bus];

		if (ioexp->active != IOEXP_ACTIVE)
			continue;

		devaddr = ioexp->bus_addr;
                for (offset = 0; offset < OFFSET_NUM; offset++) {
			set_offset = ioexp->config_pin[offset];
			set_val = ioexp->config_value[offset];

			result = i2c_write(devaddr, set_offset, set_val);
			IOEXP_DPRINT("[%s] i2c_write devaddr=0x%02x set_offset=0x%02x set_val=0x%02x\n", __func__,
				     devaddr, set_offset, set_val);
			if (result < 0)
				pr_err("@@@ [%s:%d] error devaddr:0x%02x offset:%d\n", __func__, __LINE__, devaddr, set_offset);
		}
	}
}

static noinline int
ioexp_ioctl_wr(unsigned long arg)
{
	struct ioexp_ioctl_wr_data wdata;
	int ret = -1;

	if (copy_from_user(&wdata, (struct ioexp_ioctl_wr_data *)arg,
			   sizeof(wdata))) {
		IOEXP_DPRINT("[%s:%d]\n", __func__, __LINE__);
		return -EFAULT;
	}
	IOEXP_DPRINT("[%s] port:%d value:%d sync:%d\n",
		     __func__, wdata.port, wdata.value, wdata.sync);

	if (wdata.sync) {
		ret = ioexp_set_value_sync(
			wdata.device, wdata.port, wdata.value);
	} else {
		ret = ioexp_set_value(wdata.device, wdata.port, wdata.value);
	}

	return ret;
}

static long
ioexp_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int ret = -1;

	IOEXP_DPRINT("[%s] cmd:%d\n", __func__, cmd);
	switch (cmd) {
	case IOEXP_WR:
		ret = ioexp_ioctl_wr(arg);
		break;
	default:
		ret = -EINVAL;
		break;
	}

	return ret;
}

static int
ioexp_open(struct inode *inode, struct file *file)
{
	unsigned int dev_minor = MINOR(inode->i_rdev);

	if (dev_minor != 0) {
		printk(KERN_ERR DRVNAME ": trying to access unknown minor device -> %d\n",
			   dev_minor);
		return -ENODEV;
	}

	return 0;
}

static int
ioexp_close(struct inode * inode, struct file * file)
{
  return 0;
}

struct file_operations ioexp_fops = {
	.open = ioexp_open,
	.release = ioexp_close,
	.unlocked_ioctl = ioexp_ioctl,
};

static void
ioexp_start(struct ioexp_platform_data *pdata)
{
	int bus, offset, port;
	u8 match_bit;
	struct ioexp_param *ioexp;

	/* 初期値設定 */
	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		ioexp = &ioexp_bus[bus];

		if (ioexp->active != IOEXP_ACTIVE)
			continue;

		for (offset = 0; offset < OFFSET_NUM; offset++) {
			match_bit = 1;
			ioexp->output_value[offset] = i2c_read(ioexp->bus_addr, ioexp->output_pin[offset]);
			IOEXP_DPRINT("[%s] i2c_read bus=%d offset=%d reg=0x%02x\n", __func__,
				     bus, offset, ioexp->output_value[offset]);
			for (port = 0; port < PORT_NUM; port++, match_bit <<= 1)
				ioexp->next_output_value[offset][port] =
						   (ioexp->output_value[offset] & match_bit) >> port;
		}
	}

	/* loop start */
	thread = kthread_run(ioexp_thread, pdata, "ioexp_thread");
}

static void
ioexp_init(void)
{
	struct device_node *ioexp_np, *bus_np, *config_np, *output_np;
	struct ioexp_param *ioexp;
	int bus;

	ioexp_np = of_find_node_by_name(NULL, "ioexp");

	for (bus = 0; bus < IOEXP_BUS_MAX; bus++) {
		ioexp = &ioexp_bus[bus];
		bus_np = of_find_node_by_name(ioexp_np, ioexp->name);

		if (bus_np == NULL) {
			continue;
		}
		ioexp->active = IOEXP_ACTIVE;
		of_property_read_u8(bus_np, "bus_addr", &ioexp->bus_addr);

		config_np = of_find_node_by_name(bus_np, "config");
		output_np = of_find_node_by_name(bus_np, "output");

		if (config_np != NULL) {
			of_property_read_u8_array(config_np, "pin", ioexp->config_pin, OFFSET_NUM);
			of_property_read_u8_array(config_np, "value", ioexp->config_value, OFFSET_NUM);
		}

		if (output_np != NULL) {
			of_property_read_u8_array(output_np, "pin", ioexp->output_pin, OFFSET_NUM);
		}
		IOEXP_DPRINT("[%s] bus=%d bus_addr=%d\n", __func__, bus, ioexp->bus_addr);
	}
}

int
ioexp_probe(struct platform_device *dev)
{
	int result = 0;
	struct ioexp_platform_data *pdata;

	dev_major = register_chrdev(0, DEVNAME, &ioexp_fops);
	if (!dev_major) {
		printk(KERN_ERR DRVNAME ": Error whilst opening %s \n", DEVNAME);
		result = -ENODEV;
		goto out;
	}
#if 1  /* @@@ ILQ13.0 */
	ioexp_class = class_create(DRVNAME);
#else
	ioexp_class = class_create(THIS_MODULE, DRVNAME);
#endif /* @@@ ILQ13.0 */
	device_create(ioexp_class, NULL, MKDEV(dev_major, 0), dev, DEVNAME);
	printk(KERN_INFO DRVNAME ": device registered with major %d\n",
		   dev_major);

	pdata = dev->dev.platform_data;
	ioexp_init();
	ioexp_config();
	ioexp_start(pdata);
out:
	return result;
}

int
ioexp_remove(struct platform_device *dev)
{
	device_destroy(ioexp_class, MKDEV(dev_major, 0));
	class_destroy(ioexp_class);
	unregister_chrdev(dev_major, DEVNAME);
	return 0;
}

static const struct of_device_id of_ioexp_match[] = {
	{ .compatible = "ioexp", },
	{},
};

static struct platform_driver ioexp_driver = {
	.probe = ioexp_probe,
	.remove = ioexp_remove,
	.driver = {
		.name = "ioexp",
		.owner= THIS_MODULE,
		.of_match_table = of_match_ptr(of_ioexp_match),
	},
};

module_platform_driver(ioexp_driver);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC Platforms, Ltd.");
MODULE_DESCRIPTION("IO-expander control driver");
