// SPDX-License-Identifier: GPL-2.0
// Copyright (c) 2017-2018, The Linux Foundation. All rights reserved.

/* Disable MMIO tracing to prevent excessive logging of unwanted MMIO traces */
#define __DISABLE_TRACE_MMIO__

#include <linux/acpi.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_platform.h>
#include <linux/pinctrl/consumer.h>
#include <linux/platform_device.h>
#include <linux/soc/qcom/geni-se.h>

/**
 * DOC: Overview
 *
 * Generic Interface (GENI) Serial Engine (SE) Wrapper driver is introduced
 * to manage GENI firmware based Qualcomm Universal Peripheral (QUP) Wrapper
 * controller. QUP Wrapper is designed to support various serial bus protocols
 * like UART, SPI, I2C, I3C, etc.
 */

/**
 * DOC: Hardware description
 *
 * GENI based QUP is a highly-flexible and programmable module for supporting
 * a wide range of serial interfaces like UART, SPI, I2C, I3C, etc. A single
 * QUP module can provide upto 8 serial interfaces, using its internal
 * serial engines. The actual configuration is determined by the target
 * platform configuration. The protocol supported by each interface is
 * determined by the firmware loaded to the serial engine. Each SE consists
 * of a DMA Engine and GENI sub modules which enable serial engines to
 * support FIFO and DMA modes of operation.
 *
 *
 *                      +-----------------------------------------+
 *                      |QUP Wrapper                              |
 *                      |         +----------------------------+  |
 *   --QUP & SE Clocks-->         | Serial Engine N            |  +-IO------>
 *                      |         | ...                        |  | Interface
 *   <---Clock Perf.----+    +----+-----------------------+    |  |
 *     State Interface  |    | Serial Engine 1            |    |  |
 *                      |    |                            |    |  |
 *                      |    |                            |    |  |
 *   <--------AHB------->    |                            |    |  |
 *                      |    |                            +----+  |
 *                      |    |                            |       |
 *                      |    |                            |       |
 *   <------SE IRQ------+    +----------------------------+       |
 *                      |                                         |
 *                      +-----------------------------------------+
 *
 *                         Figure 1: GENI based QUP Wrapper
 *
 * The GENI submodules include primary and secondary sequencers which are
 * used to drive TX & RX operations. On serial interfaces that operate using
 * master-slave model, primary sequencer drives both TX & RX operations. On
 * serial interfaces that operate using peer-to-peer model, primary sequencer
 * drives TX operation and secondary sequencer drives RX operation.
 */

/**
 * DOC: Software description
 *
 * GENI SE Wrapper driver is structured into 2 parts:
 *
 * geni_wrapper represents QUP Wrapper controller. This part of the driver
 * manages QUP Wrapper information such as hardware version, clock
 * performance table that is common to all the internal serial engines.
 *
 * geni_se represents serial engine. This part of the driver manages serial
 * engine information such as clocks, containing QUP Wrapper, etc. This part
 * of driver also supports operations (eg. initialize the concerned serial
 * engine, select between FIFO and DMA mode of operation etc.) that are
 * common to all the serial engines and are independent of serial interfaces.
 */

#define MAX_CLK_PERF_LEVEL 32
#define MAX_CLKS 2

/**
 * struct geni_wrapper - Data structure to represent the QUP Wrapper Core
 * @dev:		Device pointer of the QUP wrapper core
 * @base:		Base address of this instance of QUP wrapper core
 * @clks:		Handle to the primary & optional secondary AHB clocks
 * @num_clks:		Count of clocks
 * @to_core:		Core ICC path
 */
struct geni_wrapper {
	struct device *dev;
	void __iomem *base;
	struct clk_bulk_data clks[MAX_CLKS];
	unsigned int num_clks;
};

/**
 * struct geni_se_desc - Data structure to represent the QUP Wrapper resources
 * @clks:		Name of the primary & optional secondary AHB clocks
 * @num_clks:		Count of clock names
 */
struct geni_se_desc {
	unsigned int num_clks;
	const char * const *clks;
};

static const char * const icc_path_names[] = {"qup-core", "qup-config",
						"qup-memory"};

#define QUP_HW_VER_REG			0x4

/* Common SE registers */
#define GENI_INIT_CFG_REVISION		0x0
#define GENI_S_INIT_CFG_REVISION	0x4
#define GENI_OUTPUT_CTRL		0x24
#define GENI_CGC_CTRL			0x28
#define GENI_CLK_CTRL_RO		0x60
#define GENI_FW_S_REVISION_RO		0x6c
#define SE_GENI_BYTE_GRAN		0x254
#define SE_GENI_TX_PACKING_CFG0		0x260
#define SE_GENI_TX_PACKING_CFG1		0x264
#define SE_GENI_RX_PACKING_CFG0		0x284
#define SE_GENI_RX_PACKING_CFG1		0x288
#define SE_GENI_M_GP_LENGTH		0x910
#define SE_GENI_S_GP_LENGTH		0x914
#define SE_DMA_TX_PTR_L			0xc30
#define SE_DMA_TX_PTR_H			0xc34
#define SE_DMA_TX_ATTR			0xc38
#define SE_DMA_TX_LEN			0xc3c
#define SE_DMA_TX_IRQ_EN		0xc48
#define SE_DMA_TX_IRQ_EN_SET		0xc4c
#define SE_DMA_TX_IRQ_EN_CLR		0xc50
#define SE_DMA_TX_LEN_IN		0xc54
#define SE_DMA_TX_MAX_BURST		0xc5c
#define SE_DMA_RX_PTR_L			0xd30
#define SE_DMA_RX_PTR_H			0xd34
#define SE_DMA_RX_ATTR			0xd38
#define SE_DMA_RX_LEN			0xd3c
#define SE_DMA_RX_IRQ_EN		0xd48
#define SE_DMA_RX_IRQ_EN_SET		0xd4c
#define SE_DMA_RX_IRQ_EN_CLR		0xd50
#define SE_DMA_RX_LEN_IN		0xd54
#define SE_DMA_RX_MAX_BURST		0xd5c
#define SE_DMA_RX_FLUSH			0xd60
#define SE_GSI_EVENT_EN			0xe18
#define SE_IRQ_EN			0xe1c
#define SE_DMA_GENERAL_CFG		0xe30

/* GENI_OUTPUT_CTRL fields */
#define DEFAULT_IO_OUTPUT_CTRL_MSK	GENMASK(6, 0)

/* GENI_CGC_CTRL fields */
#define CFG_AHB_CLK_CGC_ON		BIT(0)
#define CFG_AHB_WR_ACLK_CGC_ON		BIT(1)
#define DATA_AHB_CLK_CGC_ON		BIT(2)
#define SCLK_CGC_ON			BIT(3)
#define TX_CLK_CGC_ON			BIT(4)
#define RX_CLK_CGC_ON			BIT(5)
#define EXT_CLK_CGC_ON			BIT(6)
#define PROG_RAM_HCLK_OFF		BIT(8)
#define PROG_RAM_SCLK_OFF		BIT(9)
#define DEFAULT_CGC_EN			GENMASK(6, 0)

/* SE_GSI_EVENT_EN fields */
#define DMA_RX_EVENT_EN			BIT(0)
#define DMA_TX_EVENT_EN			BIT(1)
#define GENI_M_EVENT_EN			BIT(2)
#define GENI_S_EVENT_EN			BIT(3)

/* SE_IRQ_EN fields */
#define DMA_RX_IRQ_EN			BIT(0)
#define DMA_TX_IRQ_EN			BIT(1)
#define GENI_M_IRQ_EN			BIT(2)
#define GENI_S_IRQ_EN			BIT(3)

/* SE_DMA_GENERAL_CFG */
#define DMA_RX_CLK_CGC_ON		BIT(0)
#define DMA_TX_CLK_CGC_ON		BIT(1)
#define DMA_AHB_SLV_CFG_ON		BIT(2)
#define AHB_SEC_SLV_CLK_CGC_ON		BIT(3)
#define DUMMY_RX_NON_BUFFERABLE		BIT(4)
#define RX_DMA_ZERO_PADDING_EN		BIT(5)
#define RX_DMA_IRQ_DELAY_MSK		GENMASK(8, 6)
#define RX_DMA_IRQ_DELAY_SHFT		6

/**
 * geni_se_get_qup_hw_version() - Read the QUP wrapper Hardware version
 * @se:	Pointer to the corresponding serial engine.
 *
 * Return: Hardware Version of the wrapper.
 */
u32 geni_se_get_qup_hw_version(struct geni_se *se)
{
	struct geni_wrapper *wrapper = se->wrapper;

	return readl_relaxed(wrapper->base + QUP_HW_VER_REG);
}
EXPORT_SYMBOL(geni_se_get_qup_hw_version);

static void geni_se_io_set_mode(void __iomem *base)
{
	u32 val;

	val = readl_relaxed(base + SE_IRQ_EN);
	val |= GENI_M_IRQ_EN | GENI_S_IRQ_EN;
	val |= DMA_TX_IRQ_EN | DMA_RX_IRQ_EN;
	writel_relaxed(val, base + SE_IRQ_EN);

	val = readl_relaxed(base + SE_GENI_DMA_MODE_EN);
	val &= ~GENI_DMA_MODE_EN;
	writel_relaxed(val, base + SE_GENI_DMA_MODE_EN);

	writel_relaxed(0, base + SE_GSI_EVENT_EN);
}

static void geni_se_io_init(void __iomem *base)
{
	u32 val;

	val = readl_relaxed(base + GENI_CGC_CTRL);
	val |= DEFAULT_CGC_EN;
	writel_relaxed(val, base + GENI_CGC_CTRL);

	val = readl_relaxed(base + SE_DMA_GENERAL_CFG);
	val |= AHB_SEC_SLV_CLK_CGC_ON | DMA_AHB_SLV_CFG_ON;
	val |= DMA_TX_CLK_CGC_ON | DMA_RX_CLK_CGC_ON;
	writel_relaxed(val, base + SE_DMA_GENERAL_CFG);

	writel_relaxed(DEFAULT_IO_OUTPUT_CTRL_MSK, base + GENI_OUTPUT_CTRL);
	writel_relaxed(FORCE_DEFAULT, base + GENI_FORCE_DEFAULT_REG);
}

static void geni_se_irq_clear(struct geni_se *se)
{
	writel_relaxed(0, se->base + SE_GSI_EVENT_EN);
	writel_relaxed(0xffffffff, se->base + SE_GENI_M_IRQ_CLEAR);
	writel_relaxed(0xffffffff, se->base + SE_GENI_S_IRQ_CLEAR);
	writel_relaxed(0xffffffff, se->base + SE_DMA_TX_IRQ_CLR);
	writel_relaxed(0xffffffff, se->base + SE_DMA_RX_IRQ_CLR);
	writel_relaxed(0xffffffff, se->base + SE_IRQ_EN);
}

/**
 * geni_se_init() - Initialize the GENI serial engine
 * @se:		Pointer to the concerned serial engine.
 * @rx_wm:	Receive watermark, in units of FIFO words.
 * @rx_rfr:	Ready-for-receive watermark, in units of FIFO words.
 *
 * This function is used to initialize the GENI serial engine, configure
 * receive watermark and ready-for-receive watermarks.
 */
void geni_se_init(struct geni_se *se, u32 rx_wm, u32 rx_rfr)
{
	u32 val;

	geni_se_irq_clear(se);
	geni_se_io_init(se->base);
	geni_se_io_set_mode(se->base);

	writel_relaxed(rx_wm, se->base + SE_GENI_RX_WATERMARK_REG);
	writel_relaxed(rx_rfr, se->base + SE_GENI_RX_RFR_WATERMARK_REG);

	val = readl_relaxed(se->base + SE_GENI_M_IRQ_EN);
	val |= M_COMMON_GENI_M_IRQ_EN;
	writel_relaxed(val, se->base + SE_GENI_M_IRQ_EN);

	val = readl_relaxed(se->base + SE_GENI_S_IRQ_EN);
	val |= S_COMMON_GENI_S_IRQ_EN;
	writel_relaxed(val, se->base + SE_GENI_S_IRQ_EN);
}
EXPORT_SYMBOL(geni_se_init);

static void geni_se_select_fifo_mode(struct geni_se *se)
{
	u32 proto = geni_se_read_proto(se);
	u32 val, val_old;

	geni_se_irq_clear(se);

	/* UART driver manages enabling / disabling interrupts internally */
	if (proto != GENI_SE_UART) {
		/* Non-UART use only primary sequencer so dont bother about S_IRQ */
		val_old = val = readl_relaxed(se->base + SE_GENI_M_IRQ_EN);
		val |= M_CMD_DONE_EN | M_TX_FIFO_WATERMARK_EN;
		val |= M_RX_FIFO_WATERMARK_EN | M_RX_FIFO_LAST_EN;
		if (val != val_old)
			writel_relaxed(val, se->base + SE_GENI_M_IRQ_EN);
	}

	val_old = val = readl_relaxed(se->base + SE_GENI_DMA_MODE_EN);
	val &= ~GENI_DMA_MODE_EN;
	if (val != val_old)
		writel_relaxed(val, se->base + SE_GENI_DMA_MODE_EN);
}

static void geni_se_select_dma_mode(struct geni_se *se)
{
	u32 proto = geni_se_read_proto(se);
	u32 val, val_old;

	geni_se_irq_clear(se);

	/* UART driver manages enabling / disabling interrupts internally */
	if (proto != GENI_SE_UART) {
		/* Non-UART use only primary sequencer so dont bother about S_IRQ */
		val_old = val = readl_relaxed(se->base + SE_GENI_M_IRQ_EN);
		val &= ~(M_CMD_DONE_EN | M_TX_FIFO_WATERMARK_EN);
		val &= ~(M_RX_FIFO_WATERMARK_EN | M_RX_FIFO_LAST_EN);
		if (val != val_old)
			writel_relaxed(val, se->base + SE_GENI_M_IRQ_EN);
	}

	val_old = val = readl_relaxed(se->base + SE_GENI_DMA_MODE_EN);
	val |= GENI_DMA_MODE_EN;
	if (val != val_old)
		writel_relaxed(val, se->base + SE_GENI_DMA_MODE_EN);
}

static void geni_se_select_gpi_mode(struct geni_se *se)
{
	u32 val;

	geni_se_irq_clear(se);

	writel(0, se->base + SE_IRQ_EN);

	val = readl(se->base + SE_GENI_M_IRQ_EN);
	val &= ~(M_CMD_DONE_EN | M_TX_FIFO_WATERMARK_EN |
		 M_RX_FIFO_WATERMARK_EN | M_RX_FIFO_LAST_EN);
	writel(val, se->base + SE_GENI_M_IRQ_EN);

	writel(GENI_DMA_MODE_EN, se->base + SE_GENI_DMA_MODE_EN);

	val = readl(se->base + SE_GSI_EVENT_EN);
	val |= (DMA_RX_EVENT_EN | DMA_TX_EVENT_EN | GENI_M_EVENT_EN | GENI_S_EVENT_EN);
	writel(val, se->base + SE_GSI_EVENT_EN);
}

/**
 * geni_se_select_mode() - Select the serial engine transfer mode
 * @se:		Pointer to the concerned serial engine.
 * @mode:	Transfer mode to be selected.
 */
void geni_se_select_mode(struct geni_se *se, enum geni_se_xfer_mode mode)
{
	WARN_ON(mode != GENI_SE_FIFO && mode != GENI_SE_DMA && mode != GENI_GPI_DMA);

	switch (mode) {
	case GENI_SE_FIFO:
		geni_se_select_fifo_mode(se);
		break;
	case GENI_SE_DMA:
		geni_se_select_dma_mode(se);
		break;
	case GENI_GPI_DMA:
		geni_se_select_gpi_mode(se);
		break;
	case GENI_SE_INVALID:
	default:
		break;
	}
}
EXPORT_SYMBOL(geni_se_select_mode);

/**
 * DOC: Overview
 *
 * GENI FIFO packing is highly configurable. TX/RX packing/unpacking consist
 * of up to 4 operations, each operation represented by 4 configuration vectors
 * of 10 bits programmed in GENI_TX_PACKING_CFG0 and GENI_TX_PACKING_CFG1 for
 * TX FIFO and in GENI_RX_PACKING_CFG0 and GENI_RX_PACKING_CFG1 for RX FIFO.
 * Refer to below examples for detailed bit-field description.
 *
 * Example 1: word_size = 7, packing_mode = 4 x 8, msb_to_lsb = 1
 *
 *        +-----------+-------+-------+-------+-------+
 *        |           | vec_0 | vec_1 | vec_2 | vec_3 |
 *        +-----------+-------+-------+-------+-------+
 *        | start     | 0x6   | 0xe   | 0x16  | 0x1e  |
 *        | direction | 1     | 1     | 1     | 1     |
 *        | length    | 6     | 6     | 6     | 6     |
 *        | stop      | 0     | 0     | 0     | 1     |
 *        +-----------+-------+-------+-------+-------+
 *
 * Example 2: word_size = 15, packing_mode = 2 x 16, msb_to_lsb = 0
 *
 *        +-----------+-------+-------+-------+-------+
 *        |           | vec_0 | vec_1 | vec_2 | vec_3 |
 *        +-----------+-------+-------+-------+-------+
 *        | start     | 0x0   | 0x8   | 0x10  | 0x18  |
 *        | direction | 0     | 0     | 0     | 0     |
 *        | length    | 7     | 6     | 7     | 6     |
 *        | stop      | 0     | 0     | 0     | 1     |
 *        +-----------+-------+-------+-------+-------+
 *
 * Example 3: word_size = 23, packing_mode = 1 x 32, msb_to_lsb = 1
 *
 *        +-----------+-------+-------+-------+-------+
 *        |           | vec_0 | vec_1 | vec_2 | vec_3 |
 *        +-----------+-------+-------+-------+-------+
 *        | start     | 0x16  | 0xe   | 0x6   | 0x0   |
 *        | direction | 1     | 1     | 1     | 1     |
 *        | length    | 7     | 7     | 6     | 0     |
 *        | stop      | 0     | 0     | 1     | 0     |
 *        +-----------+-------+-------+-------+-------+
 *
 */

#define NUM_PACKING_VECTORS 4
#define PACKING_START_SHIFT 5
#define PACKING_DIR_SHIFT 4
#define PACKING_LEN_SHIFT 1
#define PACKING_STOP_BIT BIT(0)
#define PACKING_VECTOR_SHIFT 10
/**
 * geni_se_config_packing() - Packing configuration of the serial engine
 * @se:		Pointer to the concerned serial engine
 * @bpw:	Bits of data per transfer word.
 * @pack_words:	Number of words per fifo element.
 * @msb_to_lsb:	Transfer from MSB to LSB or vice-versa.
 * @tx_cfg:	Flag to configure the TX Packing.
 * @rx_cfg:	Flag to configure the RX Packing.
 *
 * This function is used to configure the packing rules for the current
 * transfer.
 */
void geni_se_config_packing(struct geni_se *se, int bpw, int pack_words,
			    bool msb_to_lsb, bool tx_cfg, bool rx_cfg)
{
	u32 cfg0, cfg1, cfg[NUM_PACKING_VECTORS] = {0};
	int len;
	int temp_bpw = bpw;
	int idx_start = msb_to_lsb ? bpw - 1 : 0;
	int idx = idx_start;
	int idx_delta = msb_to_lsb ? -BITS_PER_BYTE : BITS_PER_BYTE;
	int ceil_bpw = ALIGN(bpw, BITS_PER_BYTE);
	int iter = (ceil_bpw * pack_words) / BITS_PER_BYTE;
	int i;

	if (iter <= 0 || iter > NUM_PACKING_VECTORS)
		return;

	for (i = 0; i < iter; i++) {
		len = min_t(int, temp_bpw, BITS_PER_BYTE) - 1;
		cfg[i] = idx << PACKING_START_SHIFT;
		cfg[i] |= msb_to_lsb << PACKING_DIR_SHIFT;
		cfg[i] |= len << PACKING_LEN_SHIFT;

		if (temp_bpw <= BITS_PER_BYTE) {
			idx = ((i + 1) * BITS_PER_BYTE) + idx_start;
			temp_bpw = bpw;
		} else {
			idx = idx + idx_delta;
			temp_bpw = temp_bpw - BITS_PER_BYTE;
		}
	}
	cfg[iter - 1] |= PACKING_STOP_BIT;
	cfg0 = cfg[0] | (cfg[1] << PACKING_VECTOR_SHIFT);
	cfg1 = cfg[2] | (cfg[3] << PACKING_VECTOR_SHIFT);

	if (tx_cfg) {
		writel_relaxed(cfg0, se->base + SE_GENI_TX_PACKING_CFG0);
		writel_relaxed(cfg1, se->base + SE_GENI_TX_PACKING_CFG1);
	}
	if (rx_cfg) {
		writel_relaxed(cfg0, se->base + SE_GENI_RX_PACKING_CFG0);
		writel_relaxed(cfg1, se->base + SE_GENI_RX_PACKING_CFG1);
	}

	/*
	 * Number of protocol words in each FIFO entry
	 * 0 - 4x8, four words in each entry, max word size of 8 bits
	 * 1 - 2x16, two words in each entry, max word size of 16 bits
	 * 2 - 1x32, one word in each entry, max word size of 32 bits
	 * 3 - undefined
	 */
	if (pack_words || bpw == 32)
		writel_relaxed(bpw / 16, se->base + SE_GENI_BYTE_GRAN);
}
EXPORT_SYMBOL(geni_se_config_packing);

static void geni_se_clks_off(struct geni_se *se)
{
	struct geni_wrapper *wrapper = se->wrapper;

	clk_disable_unprepare(se->clk);
	clk_bulk_disable_unprepare(wrapper->num_clks, wrapper->clks);
}

/**
 * geni_se_resources_off() - Turn off resources associated with the serial
 *                           engine
 * @se:	Pointer to the concerned serial engine.
 *
 * Return: 0 on success, standard Linux error codes on failure/error.
 */
int geni_se_resources_off(struct geni_se *se)
{
	int ret;

	if (has_acpi_companion(se->dev))
		return 0;

	ret = pinctrl_pm_select_sleep_state(se->dev);
	if (ret)
		return ret;

	geni_se_clks_off(se);
	return 0;
}
EXPORT_SYMBOL(geni_se_resources_off);

static int geni_se_clks_on(struct geni_se *se)
{
	int ret;
	struct geni_wrapper *wrapper = se->wrapper;

	ret = clk_bulk_prepare_enable(wrapper->num_clks, wrapper->clks);
	if (ret)
		return ret;

	ret = clk_prepare_enable(se->clk);
	if (ret)
		clk_bulk_disable_unprepare(wrapper->num_clks, wrapper->clks);
	return ret;
}

/**
 * geni_se_resources_on() - Turn on resources associated with the serial
 *                          engine
 * @se:	Pointer to the concerned serial engine.
 *
 * Return: 0 on success, standard Linux error codes on failure/error.
 */
int geni_se_resources_on(struct geni_se *se)
{
	int ret;

	if (has_acpi_companion(se->dev))
		return 0;

	ret = geni_se_clks_on(se);
	if (ret)
		return ret;

	ret = pinctrl_pm_select_default_state(se->dev);
	if (ret)
		geni_se_clks_off(se);

	return ret;
}
EXPORT_SYMBOL(geni_se_resources_on);

/**
 * geni_se_clk_tbl_get() - Get the clock table to program DFS
 * @se:		Pointer to the concerned serial engine.
 * @tbl:	Table in which the output is returned.
 *
 * This function is called by the protocol drivers to determine the different
 * clock frequencies supported by serial engine core clock. The protocol
 * drivers use the output to determine the clock frequency index to be
 * programmed into DFS.
 *
 * Return: number of valid performance levels in the table on success,
 *	   standard Linux error codes on failure.
 */
int geni_se_clk_tbl_get(struct geni_se *se, unsigned long **tbl)
{
	long freq = 0;
	int i;

	if (se->clk_perf_tbl) {
		*tbl = se->clk_perf_tbl;
		return se->num_clk_levels;
	}

	se->clk_perf_tbl = devm_kcalloc(se->dev, MAX_CLK_PERF_LEVEL,
					sizeof(*se->clk_perf_tbl),
					GFP_KERNEL);
	if (!se->clk_perf_tbl)
		return -ENOMEM;

	for (i = 0; i < MAX_CLK_PERF_LEVEL; i++) {
		freq = clk_round_rate(se->clk, freq + 1);
		if (freq <= 0 || freq == se->clk_perf_tbl[i - 1])
			break;
		se->clk_perf_tbl[i] = freq;
	}
	se->num_clk_levels = i;
	*tbl = se->clk_perf_tbl;
	return se->num_clk_levels;
}
EXPORT_SYMBOL(geni_se_clk_tbl_get);

/**
 * geni_se_clk_freq_match() - Get the matching or closest SE clock frequency
 * @se:		Pointer to the concerned serial engine.
 * @req_freq:	Requested clock frequency.
 * @index:	Index of the resultant frequency in the table.
 * @res_freq:	Resultant frequency of the source clock.
 * @exact:	Flag to indicate exact multiple requirement of the requested
 *		frequency.
 *
 * This function is called by the protocol drivers to determine the best match
 * of the requested frequency as provided by the serial engine clock in order
 * to meet the performance requirements.
 *
 * If we return success:
 * - if @exact is true  then @res_freq / <an_integer> == @req_freq
 * - if @exact is false then @res_freq / <an_integer> <= @req_freq
 *
 * Return: 0 on success, standard Linux error codes on failure.
 */
int geni_se_clk_freq_match(struct geni_se *se, unsigned long req_freq,
			   unsigned int *index, unsigned long *res_freq,
			   bool exact)
{
	unsigned long *tbl;
	int num_clk_levels;
	int i;
	unsigned long best_delta;
	unsigned long new_delta;
	unsigned int divider;

	num_clk_levels = geni_se_clk_tbl_get(se, &tbl);
	if (num_clk_levels < 0)
		return num_clk_levels;

	if (num_clk_levels == 0)
		return -EINVAL;

	best_delta = ULONG_MAX;
	for (i = 0; i < num_clk_levels; i++) {
		divider = DIV_ROUND_UP(tbl[i], req_freq);
		new_delta = req_freq - tbl[i] / divider;
		if (new_delta < best_delta) {
			/* We have a new best! */
			*index = i;
			*res_freq = tbl[i];

			/* If the new best is exact then we're done */
			if (new_delta == 0)
				return 0;

			/* Record how close we got */
			best_delta = new_delta;
		}
	}

	if (exact)
		return -EINVAL;

	return 0;
}
EXPORT_SYMBOL(geni_se_clk_freq_match);

#define GENI_SE_DMA_DONE_EN BIT(0)
#define GENI_SE_DMA_EOT_EN BIT(1)
#define GENI_SE_DMA_AHB_ERR_EN BIT(2)
#define GENI_SE_DMA_EOT_BUF BIT(0)

/**
 * geni_se_tx_init_dma() - Initiate TX DMA transfer on the serial engine
 * @se:			Pointer to the concerned serial engine.
 * @iova:		Mapped DMA address.
 * @len:		Length of the TX buffer.
 *
 * This function is used to initiate DMA TX transfer.
 */
void geni_se_tx_init_dma(struct geni_se *se, dma_addr_t iova, size_t len)
{
	u32 val;

	val = GENI_SE_DMA_DONE_EN;
	val |= GENI_SE_DMA_EOT_EN;
	val |= GENI_SE_DMA_AHB_ERR_EN;
	writel_relaxed(val, se->base + SE_DMA_TX_IRQ_EN_SET);
	writel_relaxed(lower_32_bits(iova), se->base + SE_DMA_TX_PTR_L);
	writel_relaxed(upper_32_bits(iova), se->base + SE_DMA_TX_PTR_H);
	writel_relaxed(GENI_SE_DMA_EOT_BUF, se->base + SE_DMA_TX_ATTR);
	writel(len, se->base + SE_DMA_TX_LEN);
}
EXPORT_SYMBOL(geni_se_tx_init_dma);

/**
 * geni_se_tx_dma_prep() - Prepare the serial engine for TX DMA transfer
 * @se:			Pointer to the concerned serial engine.
 * @buf:		Pointer to the TX buffer.
 * @len:		Length of the TX buffer.
 * @iova:		Pointer to store the mapped DMA address.
 *
 * This function is used to prepare the buffers for DMA TX.
 *
 * Return: 0 on success, standard Linux error codes on failure.
 */
int geni_se_tx_dma_prep(struct geni_se *se, void *buf, size_t len,
			dma_addr_t *iova)
{
	struct geni_wrapper *wrapper = se->wrapper;

	if (!wrapper)
		return -EINVAL;

	*iova = dma_map_single(wrapper->dev, buf, len, DMA_TO_DEVICE);
	if (dma_mapping_error(wrapper->dev, *iova))
		return -EIO;

	geni_se_tx_init_dma(se, *iova, len);
	return 0;
}
EXPORT_SYMBOL(geni_se_tx_dma_prep);

/**
 * geni_se_rx_init_dma() - Initiate RX DMA transfer on the serial engine
 * @se:			Pointer to the concerned serial engine.
 * @iova:		Mapped DMA address.
 * @len:		Length of the RX buffer.
 *
 * This function is used to initiate DMA RX transfer.
 */
void geni_se_rx_init_dma(struct geni_se *se, dma_addr_t iova, size_t len)
{
	u32 val;

	val = GENI_SE_DMA_DONE_EN;
	val |= GENI_SE_DMA_EOT_EN;
	val |= GENI_SE_DMA_AHB_ERR_EN;
	writel_relaxed(val, se->base + SE_DMA_RX_IRQ_EN_SET);
	writel_relaxed(lower_32_bits(iova), se->base + SE_DMA_RX_PTR_L);
	writel_relaxed(upper_32_bits(iova), se->base + SE_DMA_RX_PTR_H);
	/* RX does not have EOT buffer type bit. So just reset RX_ATTR */
	writel_relaxed(0, se->base + SE_DMA_RX_ATTR);
	writel(len, se->base + SE_DMA_RX_LEN);
}
EXPORT_SYMBOL(geni_se_rx_init_dma);

/**
 * geni_se_rx_dma_prep() - Prepare the serial engine for RX DMA transfer
 * @se:			Pointer to the concerned serial engine.
 * @buf:		Pointer to the RX buffer.
 * @len:		Length of the RX buffer.
 * @iova:		Pointer to store the mapped DMA address.
 *
 * This function is used to prepare the buffers for DMA RX.
 *
 * Return: 0 on success, standard Linux error codes on failure.
 */
int geni_se_rx_dma_prep(struct geni_se *se, void *buf, size_t len,
			dma_addr_t *iova)
{
	struct geni_wrapper *wrapper = se->wrapper;

	if (!wrapper)
		return -EINVAL;

	*iova = dma_map_single(wrapper->dev, buf, len, DMA_FROM_DEVICE);
	if (dma_mapping_error(wrapper->dev, *iova))
		return -EIO;

	geni_se_rx_init_dma(se, *iova, len);
	return 0;
}
EXPORT_SYMBOL(geni_se_rx_dma_prep);

/**
 * geni_se_tx_dma_unprep() - Unprepare the serial engine after TX DMA transfer
 * @se:			Pointer to the concerned serial engine.
 * @iova:		DMA address of the TX buffer.
 * @len:		Length of the TX buffer.
 *
 * This function is used to unprepare the DMA buffers after DMA TX.
 */
void geni_se_tx_dma_unprep(struct geni_se *se, dma_addr_t iova, size_t len)
{
	struct geni_wrapper *wrapper = se->wrapper;

	if (!dma_mapping_error(wrapper->dev, iova))
		dma_unmap_single(wrapper->dev, iova, len, DMA_TO_DEVICE);
}
EXPORT_SYMBOL(geni_se_tx_dma_unprep);

/**
 * geni_se_rx_dma_unprep() - Unprepare the serial engine after RX DMA transfer
 * @se:			Pointer to the concerned serial engine.
 * @iova:		DMA address of the RX buffer.
 * @len:		Length of the RX buffer.
 *
 * This function is used to unprepare the DMA buffers after DMA RX.
 */
void geni_se_rx_dma_unprep(struct geni_se *se, dma_addr_t iova, size_t len)
{
	struct geni_wrapper *wrapper = se->wrapper;

	if (!dma_mapping_error(wrapper->dev, iova))
		dma_unmap_single(wrapper->dev, iova, len, DMA_FROM_DEVICE);
}
EXPORT_SYMBOL(geni_se_rx_dma_unprep);

int geni_icc_get(struct geni_se *se, const char *icc_ddr)
{
	int i, err;
	const char *icc_names[] = {"qup-core", "qup-config", icc_ddr};

	if (has_acpi_companion(se->dev))
		return 0;

	for (i = 0; i < ARRAY_SIZE(se->icc_paths); i++) {
		if (!icc_names[i])
			continue;

		se->icc_paths[i].path = devm_of_icc_get(se->dev, icc_names[i]);
		if (IS_ERR(se->icc_paths[i].path))
			goto err;
	}

	return 0;

err:
	err = PTR_ERR(se->icc_paths[i].path);
	if (err != -EPROBE_DEFER)
		dev_err_ratelimited(se->dev, "Failed to get ICC path '%s': %d\n",
					icc_names[i], err);
	return err;

}
EXPORT_SYMBOL(geni_icc_get);

int geni_icc_set_bw(struct geni_se *se)
{
	int i, ret;

	for (i = 0; i < ARRAY_SIZE(se->icc_paths); i++) {
		ret = icc_set_bw(se->icc_paths[i].path,
			se->icc_paths[i].avg_bw, se->icc_paths[i].avg_bw);
		if (ret) {
			dev_err_ratelimited(se->dev, "ICC BW voting failed on path '%s': %d\n",
					icc_path_names[i], ret);
			return ret;
		}
	}

	return 0;
}
EXPORT_SYMBOL(geni_icc_set_bw);

void geni_icc_set_tag(struct geni_se *se, u32 tag)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(se->icc_paths); i++)
		icc_set_tag(se->icc_paths[i].path, tag);
}
EXPORT_SYMBOL(geni_icc_set_tag);

/* To do: Replace this by icc_bulk_enable once it's implemented in ICC core */
int geni_icc_enable(struct geni_se *se)
{
	int i, ret;

	for (i = 0; i < ARRAY_SIZE(se->icc_paths); i++) {
		ret = icc_enable(se->icc_paths[i].path);
		if (ret) {
			dev_err_ratelimited(se->dev, "ICC enable failed on path '%s': %d\n",
					icc_path_names[i], ret);
			return ret;
		}
	}

	return 0;
}
EXPORT_SYMBOL(geni_icc_enable);

int geni_icc_disable(struct geni_se *se)
{
	int i, ret;

	for (i = 0; i < ARRAY_SIZE(se->icc_paths); i++) {
		ret = icc_disable(se->icc_paths[i].path);
		if (ret) {
			dev_err_ratelimited(se->dev, "ICC disable failed on path '%s': %d\n",
					icc_path_names[i], ret);
			return ret;
		}
	}

	return 0;
}
EXPORT_SYMBOL(geni_icc_disable);

#ifdef CONFIG_QCOM_GENI_SE_FW_LOAD

#define SE_GENI_INIT_CFG_REVISION_OFFSET	0x0000
#define SE_GENI_S_INIT_CFG_REVISION_OFFSET	0x0004
#define SE_GENI_FORCE_DEFAULT_REG_OFFSET	0x0020
#define SE_GENI_OUTPUT_CTRL_OFFSET		0x0024
#define SE_GENI_CGC_CTRL_OFFSET			0x0028
#define SE_GENI4_IMAGE_REG_OFFSET		0x0100
#define SE_HW_PARAM_2_OFFSET			0x0E2C
#define SE_GENI_FW_REVISION_OFFSET		0x1000
#define SE_GENI_S_FW_REVISION_OFFSET		0x1004
#define SE_GENI_CFG_RAMn_OFFSET			0x1010
#define SE_GENI_CLK_CTRL_OFFSET			0x2000
#define SE_DMA_IF_EN_OFFSET			0x2004
#define SE_GENI_USE_MINICORES_MASK		BIT(12)

typedef struct __attribute__ ((packed)) se_fw_hdr
{
    uint32_t firmware_magic;
    /* Structure version number */
    uint32_t se_fw_hdr_version;
    /* QUPV3_HW_VERSION */
    uint32_t qup_core_version;
    /* Programmed into GENI_FW_REVISION */
    uint16_t serial_protocol;
    /* Programmed into GENI_FW_REVISION */
    uint16_t firmware_version;
    /* Programmed into GENI_INIT_CFG_REVISION */
    uint16_t config_version;
    /* Number of (uint32) GENI_FW_RAM words */
    uint16_t firmware_sz_in_items;
    /* Byte offset of GENI_FW_RAM array */
    uint16_t firmware_offset;
    /* Number of GENI_FW_CFG index/value pairs */
    uint16_t config_sz_in_items;
    /* Byte offset of GENI_FW_CFG indexes array */
    uint16_t config_idx_offset;
    /* Byte offset of GENI_FW_CFG values array */
    uint16_t config_val_offset;
} se_fw_hdr;

static uint8_t uart_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x02, 0x00, 0x05, 0x00,
    0x0a, 0x00, 0xc4, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x40, 0x08, 0x2c, 0x07, 0x32, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x0f, 0x00,
    0x20, 0x00, 0x78, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x42, 0x00, 0xc8, 0xc6, 0x7f, 0x00, 0x00, 0x00, 0x3f, 0x00,
    0xc8, 0xc6, 0x7f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x21, 0x38, 0x7f, 0x00, 0xa1, 0x0b, 0x7f, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x49, 0xd0, 0x9f, 0x00, 0x0d, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x4f, 0xd0, 0x9f, 0x00,
    0x15, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x57, 0xd0, 0x9f, 0x00, 0x45, 0x10, 0x99, 0x00,
    0x1d, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x5f, 0x50, 0x9f, 0x00, 0x4b, 0x10, 0x99, 0x00,
    0x25, 0x2e, 0x7f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0x53, 0x10, 0x99, 0x00, 0x2d, 0x2e, 0x7f, 0x00,
    0x6d, 0x10, 0x99, 0x00, 0x7f, 0x48, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0x67, 0x18, 0x9f, 0x00,
    0x7f, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0x6d, 0x60, 0x9f, 0x00, 0x03, 0x3a, 0x7f, 0x00,
    0x79, 0x48, 0x9f, 0x00, 0x6c, 0x18, 0x9f, 0x00, 0x7f, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0x7f, 0x18, 0x9f, 0x00, 0x79, 0x68, 0x9f, 0x00, 0xa1, 0x0b, 0x7f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0x83, 0x60, 0x9f, 0x00, 0x28, 0x34, 0x7f, 0x00, 0x00, 0x08, 0x1f, 0x00, 0x04, 0x34, 0x7f, 0x00,
    0x07, 0x39, 0x7f, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x8f, 0x60, 0x9f, 0x00, 0x00, 0x80, 0x04, 0x00,
    0x07, 0x39, 0x7f, 0x00, 0x8f, 0x50, 0x9f, 0x00, 0xed, 0x08, 0x9f, 0x00, 0x99, 0x60, 0x9f, 0x00,
    0x08, 0x34, 0x7f, 0x00, 0xf5, 0x10, 0x99, 0x00, 0x08, 0x39, 0x7f, 0x00, 0x80, 0x1c, 0x7f, 0x00,
    0xb7, 0x70, 0x9f, 0x00, 0xad, 0x00, 0x9a, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0xd9, 0x58, 0x9f, 0x00,
    0xc5, 0x78, 0x9f, 0x00, 0xd9, 0x10, 0x9f, 0x00, 0x07, 0x20, 0x7f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0xb1, 0x98, 0x98, 0x00, 0xd9, 0x78, 0x9f, 0x00, 0xd8, 0x30, 0x9f, 0x00, 0xc4, 0x10, 0x9f, 0x00,
    0xd9, 0x78, 0x9f, 0x00, 0xe7, 0x00, 0x9f, 0x00, 0xd1, 0x18, 0x9f, 0x00, 0xbd, 0x68, 0x9f, 0x00,
    0xd1, 0x78, 0x9f, 0x00, 0xc5, 0x60, 0x9f, 0x00, 0xd0, 0x10, 0x99, 0x00, 0x03, 0x3a, 0x7f, 0x00,
    0xe7, 0x00, 0x9f, 0x00, 0xbd, 0x48, 0x9f, 0x00, 0xca, 0x18, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00,
    0x28, 0x34, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0xe7, 0x00, 0x9f, 0x00,
    0xe0, 0x60, 0x9f, 0x00, 0xd8, 0x10, 0x9f, 0x00, 0x63, 0x78, 0x9f, 0x00, 0xdc, 0x18, 0x9f, 0x00,
    0x28, 0x34, 0x7f, 0x00, 0xdd, 0x78, 0x9f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0xe9, 0x60, 0x9f, 0x00,
    0xd3, 0x78, 0x9f, 0x00, 0xed, 0x60, 0x9f, 0x00, 0x00, 0x00, 0x05, 0x00, 0x07, 0x39, 0x7f, 0x00,
    0x99, 0x78, 0x9f, 0x00, 0x07, 0x39, 0x7f, 0x00, 0xf7, 0x60, 0x9f, 0x00, 0x18, 0x34, 0x7f, 0x00,
    0x9f, 0x78, 0x9f, 0x00, 0x00, 0x00, 0x0f, 0x00, 0xe1, 0x0b, 0x7f, 0x00, 0x05, 0x01, 0x9f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x03, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x0f, 0x00,
    0xa0, 0x04, 0x7f, 0x00, 0x0d, 0xd1, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x22, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc8, 0x0c, 0x00,
    0x10, 0x00, 0x5e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x92, 0x00,
    0x08, 0x80, 0x94, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x7a, 0x7f, 0x00, 0x08, 0x4c, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0xf7, 0x79, 0x9d, 0x00,
    0xe9, 0x79, 0x9d, 0x00, 0x41, 0x88, 0x98, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x01, 0x50, 0x7f, 0x00,
    0x05, 0x20, 0x7f, 0x00, 0x03, 0x20, 0x7f, 0x00, 0x04, 0x19, 0x9f, 0x00, 0x31, 0x8a, 0x98, 0x00,
    0xd3, 0x49, 0x9f, 0x00, 0x01, 0x44, 0x6a, 0x00, 0x00, 0x80, 0x0b, 0x00, 0x66, 0xc0, 0x98, 0x00,
    0x58, 0x40, 0x98, 0x00, 0x43, 0x38, 0x7f, 0x00, 0x03, 0x20, 0x7f, 0x00, 0x67, 0x30, 0x9f, 0x00,
    0x62, 0x40, 0x99, 0x00, 0x08, 0x1c, 0x7f, 0x00, 0x10, 0x1c, 0x7f, 0x00, 0x67, 0x78, 0x9f, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x6a, 0x38, 0x99, 0x00, 0x04, 0x1c, 0x7f, 0x00,
    0x6b, 0x68, 0x9f, 0x00, 0x07, 0x39, 0x68, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x82, 0xb8, 0x7f, 0x00,
    0x91, 0x50, 0x9f, 0x00, 0x1d, 0x41, 0x9f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x57, 0x11, 0x9f, 0x00,
    0x7b, 0x60, 0x9f, 0x00, 0x01, 0x50, 0x7f, 0x00, 0x9d, 0x48, 0x8b, 0x00, 0x07, 0x39, 0x7f, 0x00,
    0x87, 0x18, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x02, 0x24, 0x7f, 0x00,
    0x8e, 0x29, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00, 0x8f, 0x79, 0x1f, 0x01, 0x95, 0x88, 0x98, 0x00,
    0xdf, 0x79, 0x9d, 0x00, 0x95, 0x60, 0x9f, 0x00, 0x07, 0xb9, 0x68, 0x00, 0x09, 0x20, 0x7f, 0x00,
    0x73, 0x78, 0x9f, 0x00, 0x01, 0x8a, 0x98, 0x00, 0xaa, 0x30, 0x99, 0x00, 0x88, 0x4c, 0x7f, 0x00,
    0x02, 0x60, 0x7f, 0x00, 0xbc, 0x10, 0x9f, 0x00, 0x02, 0x60, 0x7f, 0x00, 0x34, 0x4a, 0x9f, 0x00,
    0xbd, 0x90, 0x98, 0x00, 0xb1, 0x38, 0x9f, 0x00, 0x80, 0x1c, 0x7f, 0x00, 0xbc, 0x00, 0x9a, 0x00,
    0xbc, 0x28, 0x9f, 0x00, 0xb9, 0x18, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x10, 0x24, 0x7f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0xc2, 0x38, 0x99, 0x00, 0x04, 0x1c, 0x7f, 0x00,
    0x01, 0x40, 0x7f, 0x00, 0xcb, 0x58, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0x47, 0x78, 0x9f, 0x00,
    0x0a, 0x04, 0x7f, 0x00, 0x4c, 0xc8, 0x9f, 0x00, 0xd2, 0xb8, 0x99, 0x00, 0x04, 0x9c, 0x7f, 0x00,
    0xf4, 0x98, 0x9f, 0x00, 0xd9, 0x88, 0x98, 0x00, 0x13, 0xf9, 0x9f, 0x00, 0x4c, 0xc8, 0x9f, 0x00,
    0xd9, 0xd0, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0xcb, 0x58, 0x9f, 0x00,
    0xe8, 0x18, 0x9f, 0x00, 0x4c, 0x48, 0x9f, 0x00, 0xcb, 0x78, 0x9f, 0x00, 0x01, 0x1c, 0x7f, 0x00,
    0x07, 0x20, 0x7f, 0x00, 0x10, 0x60, 0x7f, 0x00, 0x02, 0x20, 0x7f, 0x00, 0x4c, 0x48, 0x9f, 0x00,
    0x47, 0x78, 0x9f, 0x00, 0x00, 0xa1, 0x7f, 0x00, 0x00, 0x80, 0x1f, 0x00, 0x00, 0xa2, 0x7f, 0x00,
    0xd4, 0x98, 0x9f, 0x00, 0x10, 0xe0, 0x7f, 0x00, 0x02, 0xa0, 0x7f, 0x00, 0x4c, 0xc8, 0x9f, 0x00,
    0x47, 0xf8, 0x9f, 0x00, 0x00, 0x21, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x22, 0x7f, 0x00,
    0x48, 0x18, 0x9f, 0x00, 0x10, 0x60, 0x7f, 0x00, 0x02, 0x20, 0x7f, 0x00, 0x49, 0x78, 0x9f, 0x00,
    0x18, 0x89, 0x9f, 0x00, 0x4c, 0xc8, 0x8e, 0x00, 0xd9, 0xf8, 0x9f, 0x00, 0x4c, 0xc8, 0x8d, 0x00,
    0xd9, 0xf8, 0x9f, 0x00, 0x39, 0x11, 0x9f, 0x00, 0x1f, 0x61, 0x9f, 0x00, 0x26, 0x21, 0x8b, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x7b, 0x78, 0x9f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x2c, 0x21, 0x9a, 0x00,
    0x48, 0x4c, 0x7f, 0x00, 0x31, 0x19, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x08, 0x20, 0x7f, 0x00, 0x01, 0x24, 0x7f, 0x00, 0x7b, 0x78, 0x9f, 0x00, 0x39, 0x61, 0x9f, 0x00,
    0x42, 0x21, 0x8b, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x59, 0x79, 0x9f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x49, 0x19, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00, 0x01, 0x24, 0x7f, 0x00,
    0x08, 0x20, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0x54, 0x21, 0x9a, 0x00,
    0x48, 0x4c, 0x7f, 0x00, 0x59, 0x79, 0x9f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x59, 0x61, 0x9f, 0x00,
    0x01, 0x50, 0x7f, 0x00, 0x71, 0x49, 0x8b, 0x00, 0x07, 0x39, 0x7f, 0x00, 0x65, 0x19, 0x9f, 0x00,
    0x08, 0x60, 0x7f, 0x00, 0x02, 0x24, 0x7f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00,
    0x8e, 0x29, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00, 0x8f, 0x79, 0x1f, 0x01, 0x23, 0x8a, 0x98, 0x00,
    0x01, 0x34, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x7d, 0x89, 0x98, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x16, 0x09, 0x7f, 0x00, 0x69, 0x11, 0x7f, 0x00, 0x81, 0x19, 0x9f, 0x00,
    0x86, 0x39, 0x9a, 0x00, 0x00, 0x18, 0x0e, 0x00, 0x07, 0x20, 0x7f, 0x00, 0x10, 0x60, 0x7f, 0x00,
    0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x00, 0x30, 0x7f, 0x00, 0x00, 0x28, 0x7f, 0x00,
    0xc0, 0x68, 0x7f, 0x00, 0x01, 0x20, 0x7f, 0x00, 0xaf, 0x29, 0x9f, 0x00, 0xab, 0x11, 0x9f, 0x00,
    0x9b, 0x61, 0x9f, 0x00, 0x07, 0x39, 0x6b, 0x00, 0x01, 0x50, 0x7f, 0x00, 0xa7, 0x29, 0x9f, 0x00,
    0xc9, 0x49, 0x9f, 0x00, 0x9b, 0x79, 0x9f, 0x00, 0x04, 0x24, 0x7f, 0x00, 0xb7, 0x79, 0x9f, 0x00,
    0x01, 0x34, 0x7f, 0x00, 0x9b, 0x79, 0x9f, 0x00, 0xb2, 0x31, 0x9a, 0x00, 0x48, 0x4c, 0x7f, 0x00,
    0x04, 0x24, 0x7f, 0x00, 0xcf, 0x11, 0x9f, 0x00, 0xc0, 0x31, 0x99, 0x00, 0x88, 0x4c, 0x7f, 0x00,
    0x02, 0x60, 0x7f, 0x00, 0xc0, 0x11, 0x9f, 0x00, 0x02, 0x60, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0xc7, 0x49, 0x9f, 0x00, 0xc3, 0x79, 0x9f, 0x00, 0x08, 0x24, 0x7f, 0x00, 0x01, 0x50, 0x7f, 0x00,
    0xe9, 0x79, 0x9d, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x02, 0x34, 0x7f, 0x00, 0xb7, 0x79, 0x9f, 0x00,
    0x01, 0x50, 0x7f, 0x00, 0xda, 0x09, 0x9f, 0x00, 0x47, 0x48, 0x8e, 0x00, 0x4d, 0x78, 0x9f, 0x00,
    0x47, 0x48, 0x8d, 0x00, 0x4d, 0x78, 0x9f, 0x00, 0xe4, 0x09, 0x9f, 0x00, 0x00, 0x00, 0x0e, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x8b, 0x68, 0x7f, 0x00,
    0x40, 0x68, 0x7f, 0x00, 0xf3, 0x41, 0x9f, 0x00, 0xc1, 0x68, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0xc2, 0x68, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x2c, 0x7f, 0x00, 0xfd, 0x41, 0x9f, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0x03, 0x62, 0x9f, 0x00, 0x1b, 0x0a, 0x9f, 0x00, 0x40, 0x36, 0x7f, 0x00, 0x40, 0x35, 0x7f, 0x00,
    0x01, 0x39, 0x7f, 0x00, 0x11, 0x4a, 0x99, 0x00, 0x2b, 0x4a, 0x9f, 0x00, 0x0d, 0x62, 0x9f, 0x00,
    0x17, 0x4a, 0x99, 0x00, 0x2b, 0x4a, 0x9f, 0x00, 0x1d, 0x0a, 0x9f, 0x00, 0x13, 0x7a, 0x9f, 0x00,
    0x2b, 0x4a, 0x9f, 0x00, 0x80, 0x36, 0x7f, 0x00, 0x80, 0x35, 0x7f, 0x00, 0x2b, 0x7a, 0x9f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x01, 0x34, 0x7f, 0x00, 0x03, 0x7a, 0x9f, 0x00,
    0x2b, 0x6a, 0x9f, 0x00, 0x01, 0x50, 0x7f, 0x00, 0x9f, 0x78, 0x9f, 0x00, 0x47, 0x48, 0x9f, 0x00,
    0x4d, 0x78, 0x9f, 0x00, 0x4d, 0x90, 0x98, 0x00, 0x3b, 0x3a, 0x9f, 0x00, 0x80, 0x1c, 0x7f, 0x00,
    0x4c, 0x00, 0x9a, 0x00, 0x4c, 0x28, 0x9f, 0x00, 0x43, 0x1a, 0x9f, 0x00, 0x08, 0x60, 0x7f, 0x00,
    0x01, 0x20, 0x7f, 0x00, 0x10, 0x24, 0x7f, 0x00, 0x4d, 0x78, 0x9f, 0x00, 0x4d, 0x8a, 0x98, 0x00,
    0x00, 0x00, 0x0d, 0x00, 0x57, 0x08, 0x7f, 0x00, 0x29, 0x10, 0x7f, 0x00, 0x55, 0x02, 0x9f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x53, 0x7a, 0x9f, 0x00, 0x24, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x9a, 0x01, 0x00,
    0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xe0, 0x00, 0x20, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x24, 0x00, 0xc0, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x31, 0x02, 0x02, 0x00, 0x05, 0xce, 0x00, 0x00,
    0xe7, 0x60, 0x03, 0x00, 0xa8, 0xe6, 0x41, 0x09, 0x10, 0x05, 0x10, 0x00, 0x51, 0x1e, 0xc0, 0x42,
    0x01, 0x04, 0x00, 0x00, 0x00, 0x84, 0x2e, 0x00, 0x1a, 0x58, 0x94, 0x16, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x1c, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x1c, 0x08, 0x00, 0x10, 0x40, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x1c, 0x08, 0x00,
    0x10, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x15, 0x24, 0xc0, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
    0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c,
    0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c,
    0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c,
    0x6d, 0x6e, 0x6f, 0x70, 0x71
};

static uint8_t spi_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x01, 0x00, 0x02, 0x0b,
    0x09, 0x00, 0xba, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x18, 0x08, 0x04, 0x07, 0xce, 0x07, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xa8, 0x11, 0x00,
    0x20, 0xc0, 0x34, 0x00, 0x00, 0xc8, 0x3d, 0x00, 0x10, 0x40, 0x62, 0x00, 0x00, 0xe8, 0x67, 0x00,
    0x30, 0xc0, 0x83, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xe8, 0xa8, 0x00, 0x30, 0x40, 0xd7, 0x00, 0x00, 0xe8, 0x11, 0x00,
    0x10, 0xc0, 0x33, 0x00, 0x00, 0x88, 0x8a, 0x00, 0x08, 0x40, 0x00, 0x00, 0x00, 0x08, 0x8b, 0x00,
    0x08, 0x40, 0x00, 0x00, 0x00, 0x88, 0x8c, 0x00, 0x08, 0x40, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6b, 0x0b, 0x61, 0x00, 0x40, 0x11, 0x7f, 0x00,
    0x81, 0x38, 0x7f, 0x00, 0x83, 0x7a, 0xff, 0x00, 0x5f, 0x6f, 0x7f, 0x00, 0x07, 0x78, 0x7f, 0x00,
    0x53, 0x79, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0x9b, 0x30, 0x9f, 0x00, 0x58, 0x10, 0x9f, 0x00,
    0x61, 0x48, 0x86, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x90, 0x68, 0x9f, 0x00, 0x6f, 0x6f, 0x7f, 0x00,
    0x69, 0x48, 0x9f, 0x00, 0x1f, 0x78, 0x7f, 0x00, 0x83, 0x78, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00,
    0x0b, 0x78, 0x7f, 0x00, 0x77, 0x78, 0x9f, 0x00, 0x6f, 0x60, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00,
    0x72, 0x28, 0x3f, 0x01, 0x75, 0x60, 0xcc, 0x00, 0x28, 0xb8, 0x67, 0x00, 0x00, 0xba, 0x67, 0x00,
    0x7b, 0xe8, 0x87, 0x00, 0x6e, 0xb0, 0x25, 0x01, 0x9b, 0x78, 0x9f, 0x00, 0x80, 0x28, 0xc6, 0x00,
    0x28, 0xb8, 0x67, 0x00, 0x00, 0xb9, 0x67, 0x00, 0x87, 0xe0, 0x27, 0x01, 0x80, 0xb0, 0x85, 0x00,
    0x9b, 0x78, 0x9f, 0x00, 0x8c, 0x28, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x28, 0xb8, 0x67, 0x00,
    0x00, 0xba, 0x67, 0x00, 0x95, 0xe8, 0x87, 0x00, 0x91, 0xa8, 0xc5, 0x00, 0x8c, 0x30, 0x9f, 0x00,
    0xbb, 0x38, 0x3b, 0x01, 0x0b, 0x0b, 0x63, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0xa3, 0x60, 0x9f, 0x00, 0xb8, 0x28, 0x83, 0x00, 0xaa, 0x28, 0x83, 0x00, 0x00, 0x1e, 0x7f, 0x00,
    0xb0, 0x28, 0x83, 0x00, 0x03, 0x39, 0x7f, 0x00, 0xaf, 0x60, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00,
    0xb6, 0x30, 0x9b, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0xa7, 0x78, 0x84, 0x00,
    0xc2, 0x48, 0x9f, 0x00, 0xc2, 0x30, 0x9a, 0x00, 0xbf, 0x60, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0xc9, 0x68, 0x9f, 0x00,
    0xca, 0x08, 0x9f, 0x00, 0xff, 0x78, 0x9f, 0x00, 0xfb, 0x0b, 0x7f, 0x00, 0x89, 0x11, 0x9f, 0x00,
    0xef, 0x50, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0xdb, 0x60, 0x9f, 0x00, 0x4b, 0x0b, 0x72, 0x00, 0x00, 0x10, 0x7f, 0x00, 0xeb, 0x18, 0x9a, 0x00,
    0xb7, 0x08, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0xb7, 0x78, 0x9f, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0xe3, 0x78, 0x9f, 0x00, 0x50, 0xf9, 0x7f, 0x00, 0xf4, 0xd0, 0x9f, 0x00,
    0xf3, 0xd0, 0x87, 0x00, 0xd5, 0x78, 0x85, 0x00, 0x00, 0x00, 0x01, 0x00, 0x83, 0x01, 0x9a, 0x00,
    0x7f, 0x09, 0x9a, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x05, 0x5c, 0x61, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x09, 0x49, 0x9f, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x28, 0x69, 0x9f, 0x00, 0x08, 0x78, 0x7f, 0x00,
    0x56, 0x79, 0x7f, 0x00, 0xef, 0x6e, 0x7f, 0x00, 0x20, 0x38, 0x7f, 0x00, 0x01, 0x2c, 0x7f, 0x00,
    0x05, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x00, 0x04, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x50, 0x78, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0x1f, 0x61, 0x9f, 0x00, 0x25, 0x49, 0x9f, 0x00,
    0x45, 0x79, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x39, 0x79, 0x9f, 0x00, 0x52, 0x79, 0x7f, 0x00,
    0xbf, 0x6e, 0x7f, 0x00, 0x2d, 0x61, 0x9f, 0x00, 0x57, 0x79, 0x9f, 0x00, 0x31, 0x61, 0x9f, 0x00,
    0x04, 0x3a, 0x64, 0x00, 0x34, 0x29, 0x3f, 0x01, 0x37, 0x61, 0x8b, 0x00, 0x28, 0x38, 0x68, 0x00,
    0x00, 0xba, 0x68, 0x00, 0x3d, 0xe9, 0x88, 0x00, 0x31, 0xd9, 0x28, 0x01, 0x5f, 0x79, 0x9f, 0x00,
    0x42, 0x29, 0x9f, 0x00, 0x28, 0x38, 0x68, 0x00, 0x00, 0xb9, 0x68, 0x00, 0x49, 0xe1, 0x28, 0x01,
    0x43, 0xd9, 0x88, 0x00, 0x5f, 0x79, 0x9f, 0x00, 0x57, 0x29, 0x88, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x52, 0x29, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x20, 0x38, 0x68, 0x00, 0x59, 0xd1, 0x88, 0x00,
    0x4f, 0x59, 0x88, 0x00, 0x5f, 0x79, 0x88, 0x00, 0x0b, 0x0b, 0x7f, 0x00, 0x01, 0x44, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x65, 0x61, 0x9f, 0x00, 0x7c, 0x29, 0x83, 0x00, 0x6c, 0x29, 0x83, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x72, 0x29, 0x83, 0x00, 0x03, 0x39, 0x7f, 0x00, 0x71, 0x61, 0x9f, 0x00,
    0x76, 0x31, 0x9b, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x09, 0x20, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x69, 0x79, 0x84, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0xfd, 0x78, 0x9f, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x85, 0x61, 0x9f, 0x00, 0xfb, 0x78, 0x9f, 0x00, 0x89, 0xd1, 0xdf, 0x00,
    0x8f, 0x11, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x95, 0x61, 0x9f, 0x00, 0x9b, 0x19, 0x9a, 0x00, 0x73, 0x79, 0x92, 0x00,
    0x00, 0x1e, 0x72, 0x00, 0x73, 0x79, 0x9f, 0x00, 0x5f, 0x6e, 0x61, 0x00, 0x6b, 0x0b, 0x7f, 0x00,
    0x40, 0x11, 0x7f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x81, 0x38, 0x7f, 0x00, 0x83, 0x7a, 0xff, 0x00,
    0x53, 0x79, 0x7f, 0x00, 0x09, 0x78, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0xb7, 0x49, 0x86, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0xe6, 0x69, 0x9f, 0x00, 0x6f, 0x6e, 0x7f, 0x00, 0xbf, 0x49, 0x9f, 0x00,
    0x1f, 0x78, 0x7f, 0x00, 0xd9, 0x79, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x0b, 0x78, 0x7f, 0x00,
    0xcd, 0x79, 0x9f, 0x00, 0xc5, 0x61, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00, 0xc8, 0x29, 0x3f, 0x01,
    0xcb, 0x61, 0xcc, 0x00, 0x28, 0xb8, 0x69, 0x00, 0x00, 0xba, 0x69, 0x00, 0xd1, 0xe9, 0x89, 0x00,
    0xc4, 0xb1, 0x28, 0x01, 0xf1, 0x79, 0x9f, 0x00, 0xd6, 0x29, 0xc6, 0x00, 0x28, 0xb8, 0x69, 0x00,
    0x00, 0xb9, 0x69, 0x00, 0xdd, 0xe1, 0x29, 0x01, 0xd6, 0xb1, 0x88, 0x00, 0xf1, 0x79, 0x9f, 0x00,
    0xe2, 0x29, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x28, 0xb8, 0x69, 0x00, 0x00, 0xba, 0x69, 0x00,
    0xeb, 0xe9, 0x89, 0x00, 0xe7, 0xa9, 0xc8, 0x00, 0xe2, 0x31, 0x9f, 0x00, 0x0b, 0x0b, 0x63, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0xf7, 0x61, 0x9f, 0x00, 0x0c, 0x2a, 0x83, 0x00,
    0x00, 0x00, 0x1f, 0x00, 0x00, 0x2a, 0x83, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x06, 0x2a, 0x83, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x05, 0x62, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0xfd, 0x79, 0x84, 0x00, 0x23, 0x52, 0x9f, 0x00, 0x01, 0x44, 0x7f, 0x00,
    0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x17, 0x62, 0x9f, 0x00, 0x08, 0x5c, 0x72, 0x00,
    0x1e, 0x1a, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x50, 0xf9, 0x7f, 0x00, 0x28, 0xd2, 0x9f, 0x00, 0x27, 0xd2, 0x89, 0x00, 0x11, 0x7a, 0x88, 0x00,
    0x00, 0x40, 0x0b, 0x00, 0x30, 0x02, 0x9f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x04, 0x00,
    0x38, 0x3a, 0x9a, 0x00, 0x01, 0x40, 0x6f, 0x00, 0x37, 0x7a, 0x8f, 0x00, 0x20, 0x38, 0x6f, 0x00,
    0x00, 0x00, 0x0f, 0x00, 0x3b, 0xd2, 0x8f, 0x00, 0x00, 0x20, 0x0f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x43, 0x62, 0x9f, 0x00, 0x7f, 0x6d, 0x7f, 0x00, 0x6b, 0x09, 0x7f, 0x00, 0xc4, 0x78, 0x7f, 0x00,
    0x07, 0x78, 0x7f, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x56, 0x6a, 0x9f, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x57, 0x7a, 0x9f, 0x00, 0x61, 0x2a, 0xc5, 0x00, 0xd5, 0x00, 0x9f, 0x00, 0x56, 0x12, 0x9f, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x5d, 0x6a, 0x9f, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0xb9, 0x67, 0x00,
    0x82, 0x1c, 0x67, 0x00, 0x65, 0x62, 0x87, 0x00, 0x55, 0x52, 0x87, 0x00, 0x00, 0x00, 0x05, 0x00,
    0xd5, 0x00, 0x9f, 0x00, 0x00, 0x10, 0x1f, 0x00, 0x79, 0x4a, 0x9f, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x73, 0x6a, 0x9f, 0x00, 0x00, 0x30, 0x05, 0x00, 0x6b, 0x7a, 0x85, 0x00, 0x6b, 0x0b, 0x7f, 0x00,
    0x20, 0x38, 0x7f, 0x00, 0xcc, 0x78, 0x7f, 0x00, 0x9b, 0x38, 0x9b, 0x00, 0x95, 0x7a, 0x9f, 0x00,
    0x88, 0x02, 0x9a, 0x00, 0x03, 0x39, 0x7f, 0x00, 0x87, 0x62, 0x9f, 0x00, 0x8c, 0x0a, 0x9a, 0x00,
    0x00, 0x1e, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x93, 0x3a, 0x9b, 0x00, 0x94, 0x0a, 0x9b, 0x00,
    0x43, 0x2a, 0x9a, 0x00, 0x01, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00, 0x05, 0x2c, 0x7f, 0x00,
    0x00, 0x04, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x9f, 0x62, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x6b, 0x0b, 0x61, 0x00, 0x40, 0x11, 0x7f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x82, 0x38, 0x7f, 0x00,
    0x83, 0x7a, 0xff, 0x00, 0x01, 0x44, 0x7f, 0x00, 0x53, 0x79, 0x7f, 0x00, 0x5f, 0x6e, 0x7f, 0x00,
    0x09, 0x78, 0x7f, 0x00, 0x50, 0x78, 0x7f, 0x00, 0xbd, 0x4a, 0x86, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x24, 0x6b, 0x9f, 0x00, 0x6f, 0x6e, 0x7f, 0x00, 0xc5, 0x4a, 0x9f, 0x00, 0x1f, 0x78, 0x7f, 0x00,
    0xfb, 0x7a, 0x9f, 0x00, 0x48, 0x78, 0x7f, 0x00, 0x0b, 0x78, 0x7f, 0x00, 0xd5, 0x7a, 0x9f, 0x00,
    0xcb, 0x62, 0x9f, 0x00, 0x04, 0x3a, 0x64, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0xce, 0x2a, 0x3f, 0x01,
    0xd3, 0x62, 0xcc, 0x00, 0x20, 0x38, 0x69, 0x00, 0xd7, 0xd2, 0x89, 0x00, 0xea, 0x5a, 0x89, 0x00,
    0xca, 0x32, 0x28, 0x01, 0xef, 0x6e, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x08, 0x78, 0x7f, 0x00,
    0x02, 0x1c, 0x7f, 0x00, 0x56, 0x79, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x31, 0x79, 0x83, 0x00,
    0x6f, 0x6f, 0x68, 0x00, 0x0b, 0x78, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x6e, 0x30, 0x3f, 0x01, 0x43, 0x7b, 0x9f, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0xf6, 0x2a, 0xc6, 0x00,
    0x20, 0x38, 0x69, 0x00, 0xfd, 0xd2, 0x29, 0x01, 0x12, 0x5b, 0x89, 0x00, 0xf8, 0x32, 0x88, 0x00,
    0xef, 0x6e, 0x7f, 0x00, 0x08, 0x20, 0x7f, 0x00, 0x08, 0x78, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00,
    0x56, 0x79, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x45, 0x29, 0x83, 0x00, 0x43, 0x79, 0x9f, 0x00,
    0x6f, 0x6f, 0x68, 0x00, 0x1f, 0x78, 0x7f, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00,
    0x80, 0x30, 0x9f, 0x00, 0x43, 0x7b, 0x9f, 0x00, 0x5d, 0x03, 0x9f, 0x00, 0x1e, 0x2b, 0x9f, 0x00,
    0x00, 0x00, 0x06, 0x00, 0x20, 0x38, 0x69, 0x00, 0x27, 0xd3, 0x89, 0x00, 0x36, 0x5b, 0x89, 0x00,
    0x25, 0x2b, 0xc8, 0x00, 0x1e, 0x33, 0x9f, 0x00, 0x52, 0x79, 0x7f, 0x00, 0xbf, 0x6e, 0x63, 0x00,
    0x57, 0x29, 0x9f, 0x00, 0x51, 0x79, 0x9f, 0x00, 0x5f, 0x6f, 0x68, 0x00, 0x07, 0x78, 0x7f, 0x00,
    0x04, 0x20, 0x7f, 0x00, 0x01, 0x40, 0x7f, 0x00, 0x8f, 0x28, 0x9f, 0x00, 0x8c, 0x30, 0x9f, 0x00,
    0x0b, 0x0b, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x47, 0x63, 0x9f, 0x00, 0x4d, 0x13, 0x9a, 0x00,
    0x00, 0x00, 0x04, 0x00, 0x50, 0x1b, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x56, 0x23, 0x9a, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0x55, 0x63, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x62, 0x53, 0x9f, 0x00, 0x89, 0x91, 0x98, 0x00, 0xef, 0xb0, 0x98, 0x00,
    0x01, 0x44, 0x7f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x69, 0x63, 0x9f, 0x00,
    0x08, 0x5c, 0x72, 0x00, 0x70, 0x1b, 0x9a, 0x00, 0x00, 0x1e, 0x7f, 0x00, 0x40, 0x1c, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x78, 0x0e, 0x24, 0x00, 0x88, 0x10, 0x01, 0x00, 0x07, 0x00, 0x24, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x01, 0x00, 0x00, 0x03, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x44, 0x15, 0x00,
    0xa0, 0x83, 0x14, 0x00, 0x28, 0x81, 0xaa, 0x00, 0x02, 0x10, 0x64, 0x00, 0x00, 0x40, 0x00, 0x00,
    0x01, 0x02, 0x00, 0x00, 0x05, 0xfe, 0x01, 0x00, 0xe7, 0xc2, 0x02, 0x00, 0x00, 0x5c, 0x43, 0x0a,
    0x1a, 0x01, 0x10, 0x00, 0x00, 0x00, 0x80, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x0c, 0x10,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xe4, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x07, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00,
    0x09, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3f, 0x03, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44,
    0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54,
    0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64,
    0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71
};

static uint8_t i2c_fw[] __attribute__ ((aligned (4))) = {
    0x53, 0x45, 0x46, 0x57, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x30, 0x03, 0x00, 0x04, 0x02,
    0x09, 0x00, 0x57, 0x01, 0x1c, 0x00, 0x45, 0x00, 0x8c, 0x06, 0x78, 0x05, 0xde, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x0a, 0x00,
    0x20, 0x6d, 0x59, 0x00, 0x00, 0x48, 0x0a, 0x00, 0x10, 0x6d, 0x59, 0x00, 0x00, 0x68, 0x0a, 0x00,
    0x10, 0x6d, 0x59, 0x00, 0x00, 0x08, 0x0a, 0x00, 0x28, 0x6d, 0x59, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x88, 0xa4, 0x00, 0x08, 0x6d, 0x59, 0x00, 0x00, 0x08, 0x9b, 0x00,
    0x08, 0x42, 0x59, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x78, 0x7f, 0x00,
    0x1f, 0x78, 0x7f, 0x00, 0xe7, 0x6f, 0x7f, 0x00, 0x20, 0x08, 0x7f, 0x00, 0x10, 0x4c, 0x7f, 0x00,
    0xfb, 0x11, 0x9b, 0x00, 0x73, 0x41, 0x9a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x3d, 0x48, 0x99, 0x00,
    0x04, 0x20, 0x7f, 0x00, 0x45, 0x20, 0x9b, 0x00, 0x45, 0x08, 0x9b, 0x00, 0x44, 0x38, 0x9f, 0x00,
    0xdd, 0x79, 0x9f, 0x00, 0xa3, 0x01, 0x9c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x81, 0x38, 0x7f, 0x00,
    0xa9, 0x09, 0x9c, 0x00, 0x00, 0x00, 0x09, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x00, 0x10, 0x7f, 0x00,
    0x5b, 0x38, 0x98, 0x00, 0x5b, 0xb0, 0x98, 0x00, 0x5a, 0x38, 0x99, 0x00, 0x4f, 0x78, 0x9f, 0x00,
    0x5a, 0x18, 0x9f, 0x00, 0xe8, 0x33, 0x3f, 0x00, 0x65, 0x38, 0x98, 0x00, 0x65, 0xb0, 0x98, 0x00,
    0x4f, 0x38, 0x99, 0x00, 0xfe, 0x19, 0x98, 0x00, 0x07, 0x04, 0x65, 0x00, 0xb0, 0x19, 0x98, 0x00,
    0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00,
    0xb0, 0x19, 0x98, 0x00, 0xb0, 0x19, 0x98, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x78, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x81, 0x60, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x82, 0x18, 0x9f, 0x00, 0x64, 0x89, 0x9f, 0x00, 0x79, 0x50, 0x9f, 0x00,
    0x03, 0x12, 0x9b, 0x00, 0x90, 0x38, 0x9f, 0x00, 0x03, 0x1a, 0x9b, 0x00, 0x90, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x69, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x01, 0x20, 0x7f, 0x00,
    0x9b, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x9c, 0x18, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00,
    0xa2, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0xb1, 0x30, 0x9a, 0x00,
    0xaf, 0x48, 0x9f, 0x00, 0xb1, 0x78, 0x9f, 0x00, 0x02, 0x24, 0x79, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x8d, 0x21, 0x9b, 0x00, 0xb5, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xb6, 0x18, 0x9f, 0x00,
    0xbf, 0x30, 0x9a, 0x00, 0x79, 0x49, 0x82, 0x00, 0x08, 0x04, 0x7f, 0x00, 0xc0, 0x28, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x80, 0x13, 0x00, 0xc9, 0x60, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0xca, 0x18, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0xc1, 0x50, 0x9f, 0x00,
    0xd2, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x08, 0x04, 0x72, 0x00,
    0x80, 0x1c, 0x7f, 0x00, 0x28, 0x08, 0x7f, 0x00, 0xd7, 0x71, 0x9f, 0x00, 0x02, 0x1c, 0x7f, 0x00,
    0x87, 0x59, 0x9f, 0x00, 0xed, 0x78, 0x83, 0x00, 0x8c, 0x21, 0x9f, 0x00, 0xec, 0x10, 0x9f, 0x00,
    0x89, 0x79, 0x9f, 0x00, 0xed, 0x60, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xee, 0x18, 0x9f, 0x00,
    0xf7, 0x30, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00, 0xf6, 0x28, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x20, 0x08, 0x7f, 0x00, 0xc1, 0x01, 0x99, 0x00, 0x09, 0x39, 0x9a, 0x00,
    0x1a, 0x39, 0x9f, 0x00, 0x1b, 0x09, 0x9b, 0x00, 0xdd, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x1f, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x0a, 0x19, 0x9f, 0x00, 0x12, 0x19, 0x98, 0x00, 0x0e, 0x29, 0x9f, 0x00,
    0x04, 0x3a, 0x67, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x35, 0x11, 0x9a, 0x00, 0xec, 0x03, 0x25, 0x00, 0xad, 0x79, 0x9d, 0x00, 0xf3, 0x79, 0x9d, 0x00,
    0x07, 0x2e, 0x7f, 0x00, 0x08, 0x39, 0x7f, 0x00, 0x62, 0x19, 0x98, 0x00, 0x27, 0x61, 0x9f, 0x00,
    0x62, 0x19, 0x98, 0x00, 0x00, 0x00, 0x12, 0x00, 0x64, 0x19, 0x98, 0x00, 0x64, 0x19, 0x98, 0x00,
    0x39, 0x79, 0x9f, 0x00, 0xe4, 0x03, 0x29, 0x00, 0x40, 0x32, 0x3f, 0x00, 0xa9, 0x19, 0x9c, 0x00,
    0xa3, 0x21, 0x9c, 0x00, 0x4c, 0x39, 0x9f, 0x00, 0x49, 0x41, 0x9f, 0x00, 0x4f, 0x21, 0x9b, 0x00,
    0x21, 0x08, 0x7f, 0x00, 0x55, 0x01, 0x9f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0x4d, 0x19, 0x9b, 0x00,
    0x4f, 0x79, 0x9f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x5d, 0x11, 0x9a, 0x00, 0x05, 0x20, 0x7f, 0x00,
    0x00, 0x40, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x4f, 0x09, 0x9b, 0x00, 0x01, 0x5c, 0x7f, 0x00,
    0x4d, 0x79, 0x9f, 0x00, 0x51, 0xc1, 0x99, 0x00, 0x04, 0x20, 0x7f, 0x00, 0x53, 0x11, 0x9a, 0x00,
    0x00, 0x00, 0x12, 0x00, 0x10, 0x24, 0x79, 0x00, 0x6f, 0x11, 0x9b, 0x00, 0x6f, 0x19, 0x9b, 0x00,
    0x05, 0x20, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00, 0x40, 0x1c, 0x7f, 0x00, 0x6b, 0x79, 0x9f, 0x00,
    0x82, 0x38, 0x7f, 0x00, 0x07, 0x04, 0x7f, 0x00, 0x2f, 0x7a, 0x9f, 0x00, 0xf7, 0x38, 0x9a, 0x00,
    0x9f, 0x21, 0x9b, 0x00, 0x9f, 0x29, 0x9a, 0x00, 0x50, 0x4c, 0x7f, 0x00, 0x02, 0x24, 0x79, 0x00,
    0x03, 0x20, 0x7f, 0x00, 0xc7, 0x79, 0x9f, 0x00, 0xd1, 0x31, 0x9f, 0x00, 0x8d, 0x01, 0x99, 0x00,
    0x88, 0x11, 0x98, 0x00, 0x8d, 0x61, 0xc7, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0x90, 0x19, 0xd2, 0x00, 0xf7, 0x00, 0x99, 0x00, 0x9b, 0x31, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00,
    0xbe, 0x20, 0x9b, 0x00, 0xc7, 0x79, 0x9f, 0x00, 0x02, 0x24, 0x7f, 0x00, 0x83, 0x79, 0x9f, 0x00,
    0x03, 0x39, 0x7f, 0x00, 0xa5, 0x61, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x00, 0x1e, 0x7f, 0x00,
    0x01, 0x00, 0x1f, 0x00, 0xad, 0x61, 0xc7, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x04, 0x3a, 0x67, 0x00,
    0x00, 0x00, 0x1f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x7f, 0x78, 0x9f, 0x00, 0xb8, 0x29, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00,
    0x29, 0x08, 0x7f, 0x00, 0x1d, 0x79, 0x89, 0x00, 0xc6, 0x29, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x29, 0x08, 0x7f, 0x00, 0x1d, 0x79, 0x89, 0x00, 0x88, 0x21, 0x9f, 0x00,
    0xec, 0x10, 0x9f, 0x00, 0x89, 0x79, 0x9f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x87, 0x79, 0x9f, 0x00,
    0xdb, 0x01, 0x9f, 0x00, 0x05, 0x5c, 0x7f, 0x00, 0x20, 0x1c, 0x7f, 0x00, 0xb4, 0x3c, 0x72, 0x00,
    0x00, 0x08, 0x1f, 0x00, 0xe5, 0x61, 0x9f, 0x00, 0xe4, 0x03, 0x27, 0x00, 0x00, 0x08, 0x09, 0x00,
    0x4f, 0x78, 0x9f, 0x00, 0x06, 0x39, 0x7f, 0x00, 0xec, 0x19, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0xf2, 0x19, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0x01, 0x00, 0x1f, 0x00,
    0x05, 0x5c, 0x7f, 0x00, 0x35, 0x78, 0x9f, 0x00, 0xfe, 0x39, 0x99, 0x00, 0x5b, 0x78, 0x9f, 0x00,
    0x02, 0x2a, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x69, 0x00, 0x00, 0x00, 0x12, 0x00,
    0xad, 0x79, 0x9d, 0x00, 0xed, 0x79, 0x9d, 0x00, 0x13, 0x4a, 0x89, 0x00, 0x65, 0x79, 0x9f, 0x00,
    0x82, 0x38, 0x7f, 0x00, 0x08, 0x04, 0x7f, 0x00, 0xb9, 0x79, 0x9d, 0x00, 0xad, 0x79, 0x9d, 0x00,
    0xed, 0x79, 0x9d, 0x00, 0x21, 0x32, 0x9a, 0x00, 0x79, 0x49, 0x9f, 0x00, 0x27, 0x5a, 0x9f, 0x00,
    0x03, 0x20, 0x7f, 0x00, 0x09, 0x79, 0x9f, 0x00, 0x26, 0x2a, 0xc7, 0x00, 0x04, 0x3a, 0x7f, 0x00,
    0x05, 0x39, 0x7f, 0x00, 0x00, 0x80, 0x12, 0x00, 0x2f, 0x62, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0x30, 0x1a, 0x9f, 0x00, 0x27, 0x52, 0x88, 0x00, 0x08, 0x04, 0x7f, 0x00, 0x38, 0x2a, 0xc7, 0x00,
    0x04, 0x3a, 0x7f, 0x00, 0x05, 0x39, 0x7f, 0x00, 0x02, 0x1c, 0x7f, 0x00, 0x47, 0x02, 0x99, 0x00,
    0x5d, 0x5a, 0x9f, 0x00, 0x4d, 0x3a, 0x9a, 0x00, 0x28, 0x08, 0x7f, 0x00, 0x00, 0x00, 0x12, 0x00,
    0x51, 0xfa, 0x9f, 0x00, 0x00, 0x00, 0x05, 0x00, 0xf7, 0xf8, 0x9f, 0x00, 0x51, 0x62, 0xc7, 0x00,
    0x06, 0x39, 0x7f, 0x00, 0x52, 0x1a, 0x9f, 0x00, 0x64, 0x09, 0x9f, 0x00, 0x02, 0x20, 0x7f, 0x00,
    0xf7, 0x48, 0x82, 0x00, 0x23, 0x08, 0x65, 0x00, 0x5e, 0x0a, 0x9f, 0x00, 0x00, 0x08, 0x1f, 0x00,
    0x28, 0x08, 0x7f, 0x00, 0x65, 0x62, 0xc7, 0x00, 0x06, 0x39, 0x7f, 0x00, 0x66, 0x1a, 0x9f, 0x00,
    0x27, 0x7a, 0x9f, 0x00, 0x21, 0x08, 0x7f, 0x00, 0x72, 0x1a, 0x9f, 0x00, 0x7d, 0x7a, 0x9f, 0x00,
    0xec, 0x23, 0x25, 0x00, 0xad, 0x79, 0x9d, 0x00, 0x76, 0x1a, 0x9f, 0x00, 0xe8, 0x33, 0x3f, 0x00,
    0xf8, 0x03, 0x32, 0x00, 0x00, 0x18, 0x1f, 0x00, 0x83, 0x4a, 0x9f, 0x00, 0x8b, 0x7a, 0x9d, 0x00,
    0x00, 0x10, 0x1f, 0x00, 0x05, 0x20, 0x7f, 0x00, 0x8f, 0x02, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x08, 0x24, 0x79, 0x00, 0x01, 0x00, 0x1f, 0x00, 0x01, 0x60, 0x7f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x20, 0x08, 0x7f, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x0a, 0x04, 0x7f, 0x00, 0xaa, 0x1a, 0x9f, 0x00,
    0xad, 0x79, 0x9d, 0x00, 0x00, 0x00, 0x09, 0x00, 0x9e, 0x1a, 0x9f, 0x00, 0x06, 0x39, 0x7f, 0x00,
    0xb9, 0x79, 0x9d, 0x00, 0x00, 0x80, 0x02, 0x00, 0x9b, 0x52, 0x9f, 0x00, 0x00, 0x40, 0x1f, 0x00,
    0x09, 0x04, 0x7f, 0x00, 0x9b, 0x7a, 0x9f, 0x00, 0x90, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x80, 0x03, 0x00,
    0x80, 0x40, 0x08, 0x00, 0x43, 0x03, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x1a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x08, 0x80, 0x80, 0x00, 0x20, 0x00, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x00, 0x00, 0x01, 0xfc, 0x01, 0x00,
    0x22, 0x62, 0x03, 0x00, 0xfc, 0x1f, 0xc0, 0x09, 0x20, 0x01, 0x10, 0x00, 0x00, 0x00, 0xc0, 0x02,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x04, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00,
    0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xf8, 0x07, 0x00, 0xfe, 0xfe, 0x0f, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x07, 0xf8, 0x07, 0x00,
    0xfe, 0xfe, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    0x10, 0x11, 0x12, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c,
    0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c,
    0x5d, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c,
    0x6d, 0x6e, 0x6f, 0x70, 0x71
};

void geni_se_fw_load(struct geni_se *se, uint8_t se_mode)
{
	se_fw_hdr *hdr;
	uint8_t *fw;
	uint32_t *fw32;
	uint32_t i, val, cfg_val_off, cfg_indx_off, cfg_limit;

	if (se_mode == QUPV3_SE_UART) {
		hdr = (se_fw_hdr*)&uart_fw;
		fw = (uint8_t*)&uart_fw;
	} else if (se_mode == QUPV3_SE_SPI) {
		hdr = (se_fw_hdr*)&spi_fw;
		fw = (uint8_t*)&spi_fw;
	} else if (se_mode == QUPV3_SE_I2C) {
		hdr = (se_fw_hdr*)&i2c_fw;
		fw = (uint8_t*)&i2c_fw;
	}

	/* Disable Peripheral clock and Output control */
	writel(0x0, se->base + SE_GENI_OUTPUT_CTRL_OFFSET);
	writel(0x378, se->base + SE_GENI_CGC_CTRL_OFFSET);
	writel(0x0, se->base + SE_GENI_CLK_CTRL_OFFSET);
	writel(0x78, se->base + SE_GENI_CGC_CTRL_OFFSET);

	/* Configure INIT_CFG revision */
	val = *((uint16_t*)&hdr->config_version);
	writel(val, se->base + SE_GENI_INIT_CFG_REVISION_OFFSET);

	/* CFG_REG Programming */
	cfg_indx_off = hdr->config_idx_offset;
	cfg_val_off = *((uint16_t*)&hdr->config_val_offset);
	fw32 = (uint32_t*)&fw[cfg_val_off];
	cfg_limit = *((uint16_t*)&hdr->config_sz_in_items);
	for (i=0; i<cfg_limit; i++) {
		writel(fw32[i], (se->base + SE_GENI4_IMAGE_REG_OFFSET +
				(fw[cfg_indx_off+i] * 4)));
	}

	/* If SE is not using mini-core, program CFG_RAMn region as well */
	val = readl(se->base + SE_HW_PARAM_2_OFFSET);
	if (!(val & SE_GENI_USE_MINICORES_MASK)) {
		cfg_val_off = *((uint16_t*)&hdr->firmware_offset);
		fw32 = (uint32_t*)&fw[cfg_val_off];
		cfg_limit = *((uint16_t*)&hdr->firmware_sz_in_items);
		memcpy_toio((void*)(se->base + SE_GENI_CFG_RAMn_OFFSET),
				fw32, (cfg_limit * sizeof(uint32_t)));
	}

	/* Enable Peripheral clock */
	writel(0x1, se->base + SE_GENI_FORCE_DEFAULT_REG_OFFSET);
	writel(0x378, se->base + SE_GENI_CGC_CTRL_OFFSET);
	writel(0x1, se->base + SE_GENI_CLK_CTRL_OFFSET);
	writel(0x78, se->base + SE_GENI_CGC_CTRL_OFFSET);

	/* Configure FW Revision and Protocol */
	val = ((*((uint16_t*)&hdr->serial_protocol) << 8) |
			(*((uint16_t*)&hdr->firmware_version) >> 8));
	writel(val, se->base + SE_GENI_FW_REVISION_OFFSET);
	val = (*((uint16_t*)&hdr->firmware_version) & 0xFF);
	writel(val, se->base + SE_GENI_S_FW_REVISION_OFFSET);

	/* Enable Output control */
	writel(0x7F, se->base + SE_GENI_OUTPUT_CTRL_OFFSET);

	/* Enable DMA mode */
	writel(0x1, se->base + SE_DMA_IF_EN_OFFSET);
}
#endif /* CONFIG_QCOM_GENI_SE_FW_LOAD */

static int geni_se_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct geni_wrapper *wrapper;
	int ret;

	wrapper = devm_kzalloc(dev, sizeof(*wrapper), GFP_KERNEL);
	if (!wrapper)
		return -ENOMEM;

	wrapper->dev = dev;
	wrapper->base = devm_platform_ioremap_resource(pdev, 0);
	if (IS_ERR(wrapper->base))
		return PTR_ERR(wrapper->base);

	if (!has_acpi_companion(&pdev->dev)) {
		const struct geni_se_desc *desc;
		int i;

		desc = device_get_match_data(&pdev->dev);
		if (!desc)
			return -EINVAL;

		wrapper->num_clks = min_t(unsigned int, desc->num_clks, MAX_CLKS);

		for (i = 0; i < wrapper->num_clks; ++i)
			wrapper->clks[i].id = desc->clks[i];

		ret = of_count_phandle_with_args(dev->of_node, "clocks", "#clock-cells");
		if (ret < 0) {
			dev_err(dev, "invalid clocks property at %pOF\n", dev->of_node);
			return ret;
		}

		if (ret < wrapper->num_clks) {
			dev_err(dev, "invalid clocks count at %pOF, expected %d entries\n",
				dev->of_node, wrapper->num_clks);
			return -EINVAL;
		}

		ret = devm_clk_bulk_get(dev, wrapper->num_clks, wrapper->clks);
		if (ret) {
			dev_err(dev, "Err getting clks %d\n", ret);
			return ret;
		}
	}

	dev_set_drvdata(dev, wrapper);
	dev_dbg(dev, "GENI SE Driver probed\n");
	return devm_of_platform_populate(dev);
}

static const char * const qup_clks[] = {
	"m-ahb",
	"s-ahb",
};

static const struct geni_se_desc qup_desc = {
	.clks = qup_clks,
	.num_clks = ARRAY_SIZE(qup_clks),
};

static const char * const i2c_master_hub_clks[] = {
	"s-ahb",
};

static const struct geni_se_desc i2c_master_hub_desc = {
	.clks = i2c_master_hub_clks,
	.num_clks = ARRAY_SIZE(i2c_master_hub_clks),
};

static const struct of_device_id geni_se_dt_match[] = {
	{ .compatible = "qcom,geni-se-qup", .data = &qup_desc },
	{ .compatible = "qcom,geni-se-i2c-master-hub", .data = &i2c_master_hub_desc },
	{}
};
MODULE_DEVICE_TABLE(of, geni_se_dt_match);

static struct platform_driver geni_se_driver = {
	.driver = {
		.name = "geni_se_qup",
		.of_match_table = geni_se_dt_match,
	},
	.probe = geni_se_probe,
};
module_platform_driver(geni_se_driver);

MODULE_DESCRIPTION("GENI Serial Engine Driver");
MODULE_LICENSE("GPL v2");
