#if defined(CONFIG_RTL_ENABLE_CP0_WATCH_PROC) || defined(CONFIG_RTL_ENABLE_CP0_WATCH_PROC_MODULE)
#include <linux/version.h>
#include <linux/module.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <asm/watch.h>

MODULE_LICENSE("GPL");

static const char watch_proc_root_name[] = "mips_watch";
static struct proc_dir_entry *ent_root;

static const char watch_proc_wp_name[] = "watchpoints";
static struct proc_dir_entry *ent_wpoints;

static int watchpoints_open(struct inode *inode, struct file *file);
static int watchpoints_write(struct file *file, const char __user *buf, size_t count,loff_t *data);
void dump_watchpoints(void *s, void *v);
extern void (*show_mips_watchpoint_registers)(void*, void *);

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5,10,0)
static struct proc_ops watchpoints_fops = {
	.proc_open			= watchpoints_open,
	.proc_write			= watchpoints_write,
	.proc_read			= seq_read,
	.proc_lseek 		= seq_lseek,
	.proc_release		= single_release,
};
#else
static struct file_operations watchpoints_fops = {
	.open			= watchpoints_open,
	.write			= watchpoints_write,
	.read			= seq_read,
	.llseek 		= seq_lseek,
	.release		= single_release,
};
#endif

static void rtl_watch_proc_fini(void)
{
	if(ent_wpoints)
		remove_proc_entry(watch_proc_wp_name, ent_root);

	if(ent_root)
		proc_remove(ent_root);

	show_mips_watchpoint_registers = NULL;
	
	pr_info("[%s] done\n", __func__);
}

static int __init rtl_watch_proc_init(void)
{
	ent_root = proc_mkdir(watch_proc_root_name, NULL);
	if( ent_root == NULL ) {
		pr_err("Failed to register /proc/%s\n", watch_proc_root_name);
		goto err_exit;
	}

	ent_wpoints = proc_create_data(watch_proc_wp_name, 0600, ent_root, &watchpoints_fops, NULL);
	if( ent_wpoints == NULL ) {
		pr_err("Failed to register /proc/%s/%s\n", watch_proc_root_name, watch_proc_wp_name);
		goto err_exit;
	}

	show_mips_watchpoint_registers = dump_watchpoints;

	pr_info("[%s] done\n", __func__);
	return 0;

err_exit:
	pr_err("[%s] fail\n", __func__);
	rtl_watch_proc_fini();
	return -1;
}

/* watchpoints get */

struct watchpoint_reg_get_arg {
	int cpu;
	struct seq_file *seq;

	int regIdx;
};

#define RTW_WP_PRINT(s,fmt,args...) do{ if(s) seq_printf(s,fmt,##args); else pr_info(fmt,##args); }while(0);

static void _watchpoint_reg_get_single(void *arg)
{
	unsigned int hi = 0;
	unsigned long lo = 0;
	struct watchpoint_reg_get_arg *req = (struct watchpoint_reg_get_arg *)arg;

#ifdef CONFIG_SMP
	if(req->cpu!=smp_processor_id()){
		pr_err("[%s] Error: req->cpu=%d, current cpu:%d\n", __func__, req->cpu, smp_processor_id());
		return;
	}
#endif

	switch (req->regIdx) {
	case 3:
		hi = read_c0_watchhi3();
		lo = read_c0_watchlo3();
		break;
	case 2:
		hi = read_c0_watchhi2();
		lo = read_c0_watchlo2();
		break;
	case 1:
		hi = read_c0_watchhi1();
		lo = read_c0_watchlo1();
		break;
	case 0:
		hi = read_c0_watchhi0();
		lo = read_c0_watchlo0();
		break;
	default:
		pr_err("[%s] Error: cpu=%d, invalid register index %d\n", __func__, req->cpu, req->regIdx);
		return ;
	}

	RTW_WP_PRINT(req->seq, "\tWatchLo[%d]:0x%08lx, WatchHi[%d]:0x%08x\n", req->regIdx, lo, req->regIdx, hi);
}

static int watchpoints_show(struct seq_file *s, void *v)
{
	int cpu, regIdx;
	struct watchpoint_reg_get_arg req;
	
	RTW_WP_PRINT(s, "Dump watch registers:\n");
	for_each_online_cpu(cpu) {
		memset(&req, 0x0, sizeof(req));

		req.cpu = cpu;
		req.seq = s;
		RTW_WP_PRINT(s, "CPU %d\n", req.cpu);
		
		for(regIdx = 0; regIdx<NUM_WATCH_REGS; regIdx++) {
			req.regIdx = regIdx;
			
#ifdef CONFIG_SMP
			smp_call_function_single(cpu, _watchpoint_reg_get_single, &req, 1);
#else
			_watchpoint_reg_get_single(&req);
#endif
		}
	}

	return 0;
}

static int watchpoints_open(struct inode *inode, struct file *file)
{
	return (single_open(file, watchpoints_show, NULL));
}

/* watchpoints set */

struct watchpoint_reg_set_arg {
	int cpu;
	int regIdx;
	
	unsigned int new_hi;
	unsigned long new_lo;
};

static void _watchpoint_reg_set_single(void *arg)
{
	struct watchpoint_reg_set_arg *req = (struct watchpoint_reg_set_arg *)arg;
	
#ifdef CONFIG_SMP
	if(req->cpu!=smp_processor_id()){
		pr_err("[%s] Error: req->cpu=%d, current cpu:%d\n", __func__, req->cpu, smp_processor_id());
		return;
	}
#endif

	switch (req->regIdx) {
	case 3:
		write_c0_watchhi3(req->new_hi);
		write_c0_watchlo3(req->new_lo);
		break;
	case 2:
		write_c0_watchhi2(req->new_hi);
		write_c0_watchlo2(req->new_lo);
		break;
	case 1:
		write_c0_watchhi1(req->new_hi);
		write_c0_watchlo1(req->new_lo);
		break;
	case 0:
		write_c0_watchhi0(req->new_hi);
		write_c0_watchlo0(req->new_lo);
		break;
	default:
		pr_err("[%s] Error: cpu=%d, invalid register index %d\n", __func__, req->cpu, req->regIdx);
		return ;
	}
}

static int watchpoints_write(struct file *file, const char __user *buf, size_t count,loff_t *data)
{
	char tmp[512] = {0};

	if (copy_from_user(tmp, buf, count>=sizeof(tmp)?sizeof(tmp):count))
		return -EFAULT;

	if (strncmp(tmp, "help", 4)==0) {
		printk("Set the watchHi and watchLo registers.\n");
		printk("\nCommands:\n");
		printk("\techo 'clearall' > /proc/%s/%s\n", watch_proc_root_name, watch_proc_wp_name);
		printk("\techo 'clear=$index' > /proc/%s/%s\n", watch_proc_root_name, watch_proc_wp_name);
		printk("\techo 'index=$index addr=$vaddr type=$type' > /proc/%s/%s\n", watch_proc_root_name, watch_proc_wp_name);
		printk("\techo 'index=$index addr=$vaddr ctrl=$hival type=$type' > /proc/%s/%s\n", watch_proc_root_name, watch_proc_wp_name);
		printk("\nArguments:\n");
		printk("\t$index: a 1-byte watch register index, valid=[0/1/.../%d]\n", NUM_WATCH_REGS);
		printk("\t$addr:  a 4-byte heximal virtual address in double word, e.g: 0x80001000, or 0xb8003000\n");
		printk("\t$hival: watchHi register value. If not set, default use 0xC0000000\n");
		printk("\t$type:  bitwise OR watch type (Any bitwise OR value is valid).\n\t\tbit0 = watch WRITE\n\t\tbit1 = watch READ\n\t\tbit2 = watch LOAD\n");
	}
	else if (strncmp(tmp, "clearall", 8)==0) {
		int cpu, regIdx;
		struct watchpoint_reg_set_arg req;

		for_each_online_cpu(cpu) {
			memset(&req, 0x0, sizeof(req));

			req.cpu = cpu;
			req.new_hi = 0;
			req.new_lo = 0;
			
			for(regIdx = 0; regIdx<NUM_WATCH_REGS; regIdx++) {
				req.regIdx = regIdx;
				
#ifdef CONFIG_SMP
				smp_call_function_single(cpu, _watchpoint_reg_set_single, &req, 1);
#else
				_watchpoint_reg_set_single(&req);
#endif
			}
		}
		printk("clearall done.\n");
	}
	else if (strncmp(tmp, "clear=", 6)==0) {
		int ret, idx, cpu;
		struct watchpoint_reg_set_arg req;

		ret = sscanf(tmp, "clear=%d", &idx);
		if(ret!=1 || idx>=NUM_WATCH_REGS)
			return -EFAULT;

		for_each_online_cpu(cpu) {
			memset(&req, 0x0, sizeof(req));

			req.cpu = cpu;
			req.regIdx = idx;
			req.new_hi = 0;
			req.new_lo = 0;
			
#ifdef CONFIG_SMP
			smp_call_function_single(cpu, _watchpoint_reg_set_single, &req, 1);
#else
			_watchpoint_reg_set_single(&req);
#endif
		}
		printk("clear reg[%d] done.\n", idx);
	}
	else{
		int ret, cpu;
		unsigned int idx, type, vaddr;
		unsigned int hi_val=0;
		unsigned long lo_val=0;
		struct watchpoint_reg_set_arg req;

		if(strstr(tmp, "ctrl=")) {
			ret = sscanf(tmp, "index=%d addr=%x ctrl=%x type=%d", &idx, &vaddr, &hi_val, &type);
			if(ret==4)
				printk("watch lo: 0x%08x, hi: 0x%08x, type:%04x => watch[%d]\n", vaddr, hi_val, type, idx);
			else
				return -EFAULT;
		}else {
			ret = sscanf(tmp, "index=%d addr=%x type=%d", &idx, &vaddr, &type);
			hi_val = 0xC0000000;
			if(ret==3)
				printk("watch lo: 0x%08x, hi: 0x%08x, type:%04x => watch[%d]\n", vaddr, hi_val, type, idx);
			else
				return -EFAULT;
		}

		if(idx>=NUM_WATCH_REGS)
			return -EFAULT;

		lo_val = (vaddr & (~0x7)); /* msak low 3 bit */
		lo_val |= ( type & 0x7 ); /* set low 3 bit as I/R/W */

		for_each_online_cpu(cpu) {
			memset(&req, 0x0, sizeof(req));

			req.cpu = cpu;
			req.regIdx = idx;
			req.new_hi = hi_val;
			req.new_lo = lo_val;
			
#ifdef CONFIG_SMP
			smp_call_function_single(cpu, _watchpoint_reg_set_single, &req, 1);
#else
			_watchpoint_reg_set_single(&req);
#endif
		}
		printk("set watchHi[%d]: 0x%08x, watchLo[%d]: 0x%08lx\n", idx, hi_val, idx, lo_val);
	}

	return count;
}

void dump_watchpoints(void *s, void *v)
{
	watchpoints_show((struct seq_file *)s, v);
}

module_init(rtl_watch_proc_init);
module_exit(rtl_watch_proc_fini);

#endif

