/*
 * Copyright (c) Cortina-Access Limited 2015.  All rights reserved.
 *
 */

#include <stdlib.h>
#include <rtk_ipc_priv.h>
#include <rtk_ipc_dbg_string.h>
#include <cyg/infra/diag.h>
#include <cyg/infra/cyg_type.h>
#include <cyg/io/devtab.h>
#include "rtl8198d/hal_ipc.h"
#include "pkgconf/io_serial_rtl819x.h"

#define _KERNEL
#include <asm-generic/barrier.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/mutex.h>
#include <linux/kthread.h>

#if (CONFIG_LINUX_COMPAT_SKB_METHOD==2)
#include <linux/interrupt.h>
#endif /* CONFIG_LINUX_COMPAT_SKB_METHOD==2 */

#if defined(DBG_WFO_CMD)
#include "../../../dbg_wfo_cmd.c"
#endif /* DBG_WFO_CMD */

int sp_feature = -1;

/* IPC global data */
ipc_module_t g_func_data;

uint8_t isInit;
uint32_t pe_todo_offset;

/* eCos interrup */
static cyg_interrupt ipc_intr;
static cyg_handle_t ipc_intr_handle;
static cyg_sem_t wait_for_ack;

/* IPC MSG thread */
cyg_flag_t ipc_thread_flag;
unsigned char ipc_thread_stack[24 * 1024];
cyg_handle_t ipc_thread_handle;
cyg_thread ipc_thread_obj;

cyg_flag_t ipc_domsg_thread_flag;
unsigned char ipc_domsg_thread_stack[24 * 1024];
cyg_handle_t ipc_domsg_thread_handle;
cyg_thread ipc_domsg_thread_obj;
unsigned long todo_list_flag;
unsigned long send_idx_flag;
spinlock_t todo_list_lock;
spinlock_t send_idx_lock;
struct list_head todo_list = LIST_HEAD_INIT(todo_list);

struct mutex sync_mutex;

static unsigned int ipc_send_msg(rtk_ipc_pkt_t *p_ipc_pkt, uint8_t ipc_flag);
static void ipc_send_ack(ipc_module_t *ipc_module, msg_header_t *sync_hdr);
void dump_thread_info(cyg_uint16 pid);

/* Utility API: dump memory */
void _rtk_ipc_dumpmem(int *ptr, int size)
{
	int i;

	for (i = 0; i < size; i += 4) {
		if (i%16 == 0)
			diag_printf("\n%p: ", ptr);
		diag_printf("0x%x ", *ptr);
		ptr++;
	}
	diag_printf("\n");
}

/* Initialize all the members in the IPC list */
static void initial_list_ctrl(list_t *ipc_list)
{
	list_ctrl_t *list_ctrl;

	/* Initialize all the offset value */
	list_ctrl = &(ipc_list->list_ctrl);
	list_ctrl->version = IPC_VERSION;
	list_ctrl->done_offset = LAST_ITEM_NO;
	list_ctrl->enqueue_offset = 0xFFFFFFFF;
	list_ctrl->current_send_offset = LAST_ITEM_NO;
}

static ipc_session_t *find_session(ipc_module_t *ipc_module, uint8_t session_id)
{
	struct list_head *ptr;
	ipc_session_t *session;

	list_for_each(ptr, &ipc_module->session_list) {
		session = list_entry(ptr, ipc_session_t, session_node);
		IPC_DBG("Iterating session list, now id %u", session->id);
		if (session->id == session_id)
			return session;
	}

	return NULL;
}

/*
 * Register message procedures for a given session
 * Need to pass the reference to the function store
 * inputs	: session_id < IPC_SESSION_MAX
 * msg_proc	: message number and their callbacks
 * msg_count	: number of messages session is interested < RE_IPC_MAX_PROCS
 */
rtk_ipc_status_t _rtk_ipc_msg_handle_register(uint8_t session_id,
		const ipc_msg_t *msg_procs, uint16_t msg_count)
{
	ipc_module_t *ipc_module = &g_func_data;
	ipc_session_t *session;

	/* session id should in range 1 --> IPC_SESSION_MAX */
	if (session_id > IPC_SESSION_MAX || msg_count == 0) {
		IPC_ERR("%s: parameter error", __func__);
		return IPC_EINVAL;
	}

	session = find_session(ipc_module, session_id);
	if (session != NULL) {
		IPC_ERR("session_id %u had been registered.", session_id);
		return IPC_EINVAL;
	}
	session = (ipc_session_t *)kmalloc(sizeof(ipc_session_t), GFP_KERNEL);
	if (session == NULL) {
		IPC_ERR("session id %u allocate failed.", session_id);
		return IPC_ENOMEM;
	}

	session->id = session_id;
	session->msg_cnt = (uint8_t)msg_count;
	session->msg_procs = (ipc_msg_t *)kmalloc(sizeof(ipc_msg_t) * msg_count, GFP_KERNEL);
	if (session->msg_procs == NULL) {
		IPC_ERR("Allocate msg_procs failed.");
		kfree(session);
		return IPC_ENOMEM;
	}

	memcpy(session->msg_procs, msg_procs, sizeof(ipc_msg_t) * msg_count);
	list_add(&session->session_node, &ipc_module->session_list);
	diag_printf("IPC: Register session id = %u, msg_count = %u\n", session_id, msg_count);

	return IPC_OK;
}

/*
 * Deregisters all msg_procs for the session.
 * Need to pass the reference to the function store
 * inputs	: session_id < IPC_SESSION_MAX
 */
rtk_ipc_status_t _rtk_ipc_msg_handle_unregister(uint8_t session_id)
{
	ipc_module_t *ipc_module = &g_func_data;
	ipc_session_t *session;
	struct list_head *ptr;

	/* The session id should in range 0 --> IPC_SESSION_MAX */
	if (session_id > IPC_SESSION_MAX)
		return IPC_EINVAL;

	list_for_each(ptr, &ipc_module->session_list) {
		session = list_entry(ptr, ipc_session_t, session_node);
		if (session_id == session->id) {
			diag_printf("IPC: Unregister session %u, msg_count = %d\n",
					session->id, session->msg_cnt);
			list_del(ptr);
			kfree(session->msg_procs);
			kfree(session);
			return IPC_OK;
		}
	}

	return IPC_ENOCLIENT;
}

static void print_msg_header(msg_header_t *header)
{
	int src_session = header->src_addr.session_id;
	int dst_session = header->dst_addr.session_id;
	int src_cpu = header->src_addr.cpu_id;
	int dst_cpu = header->dst_addr.cpu_id;
	int type = header->ipc_flag;

	if (src_cpu < 0 || dst_cpu < 0) {
		src_cpu = dst_cpu = 3;
		src_session = dst_session = type = 0;
	}

	diag_printf("===== IPC msg_header =====\n");
	diag_printf("src CPU = %s, session = %s\n", cpu_name[src_cpu], session_name[src_session]);
	diag_printf("dst CPU = %s, session = %s\n", cpu_name[dst_cpu], session_name[dst_session]);
	diag_printf("ipc_type = %s, msg_no = %u\n", ipc_type[type], header->msg_no);
	diag_printf("payload_size = 0x%x, trans_id = %d\n\n", header->payload_size, header->trans_id);
}

/*
 * Get the available item slot
 * inputs	: session_id < IPC_SESSION_MAX
 */
static msg_header_t *get_list_item(uint8_t cpu_id, ipc_module_t *ipc_module, uint32_t *offset)
{
	msg_header_t *msg_header = NULL;
	list_ctrl_t *pe_list_ctrl = &(ipc_module->pe_to_cpu->list_ctrl);
	list_ctrl_t *cpu_list_ctrl = &(ipc_module->cpu_to_pe->list_ctrl);
	uint32_t tmp_offset;

	tmp_offset = pe_list_ctrl->current_send_offset + 1U;

	if (tmp_offset >= MAX_ITEM_NO)
		tmp_offset = 0U;

	if (tmp_offset == cpu_list_ctrl->done_offset) {
		IPC_DBG("%s list offset full.", __func__);
		return NULL;
	}

	*offset = tmp_offset;
	msg_header = &(ipc_module->pe_to_cpu->list_item[tmp_offset].msg_header);

	IPC_DBG("get msg_header offset = %u", tmp_offset);

	return msg_header;
}

/*
 * Fill all the members in this header
 * ipc_data	: the information for communication
 * ipc_flag	: the IPC flags
 * header	: the IPC mesenger header
 */
static unsigned int fill_header(rtk_ipc_pkt_t *ipc_data,
		uint8_t ipc_flag,
		msg_header_t *header,
		ipc_module_t *ipc_module)
{
	/* Assign all the information for this communication */

	header->src_addr.session_id = ipc_data->session_id;
	header->src_addr.cpu_id = (uint8_t)ipc_module->cpu_id;
	header->dst_addr.cpu_id = (uint8_t)ipc_data->dst_cpu_id;
	header->dst_addr.session_id = ipc_data->session_id;
	header->ipc_flag = ipc_flag;
	header->priority = ipc_data->priority;
	header->msg_no = ipc_data->msg_no;
	header->payload_size = ipc_data->msg_size;
	ipc_module->trans_id++;
	header->trans_id = (uint16_t)ipc_module->trans_id;

	if (header->ipc_flag == IPC_SYNC_MSG)
		ipc_module->wait_trans_id = header->trans_id;

	return IPC_OK;
}

/*
 * This function returns the address of the payload which can be
 * directly used to put it data before requesting a ipc send
 * Input: Target CPU id: arget CPU id 0 --> ARM, 1 --> TAROKO0, 2 --> TAROKO1
 * Ouput: Returns the buf address
 */
msg_header_t *_rtk_ipc_msg_context(uint8_t target_cpu, uint32_t *offset)
{
	ipc_module_t *ipc_module = &g_func_data;
	msg_header_t *msg_header = NULL;
	int count = 0;

	/* No error checking. Target CPU must not be the source */
	do {
		msg_header = get_list_item(target_cpu, ipc_module, offset);

		if (msg_header != NULL)
			break;
		count++;
		HAL_DELAY_US(1);
	} while (count < 5000);

	if (msg_header == NULL) {
		IPC_ERR("%s thread %u IPC list is full, can't get IPC item", __func__, CURRENT_THREAD_ID);
		return NULL;
	}

	return msg_header;
}

static void print_list_offset(void)
{
	list_ctrl_t *list_ctrl =  &(g_func_data.pe_to_cpu->list_ctrl);

	IPC_LOG("done offset = %u, enqueue offset = %u, send offset = %u",
			list_ctrl->done_offset, list_ctrl->enqueue_offset,
			list_ctrl->current_send_offset);
}

static void update_send_offset(uint32_t offset)
{
	list_ctrl_t *list_ctrl;

	list_ctrl = &(g_func_data.pe_to_cpu->list_ctrl);
	list_ctrl->current_send_offset = offset;
}

static void update_done_offset(uint32_t offset)
{
	list_ctrl_t *list_ctrl;

	list_ctrl = &(g_func_data.pe_to_cpu->list_ctrl);
	list_ctrl->done_offset = offset;
}

static uint32_t get_next_todo_offset(void)
{
	list_ctrl_t *list_ctrl;
	uint32_t update_offset;

	list_ctrl = &(g_func_data.pe_to_cpu->list_ctrl);

	if (list_ctrl->done_offset < LAST_ITEM_NO)
		update_offset = list_ctrl->done_offset + 1U;
	else
		update_offset = 0U;

	return update_offset;
}

/*
 * Find the corresponding registered callback function according to the "msg_number"
 * session_id	: < IPC_SESSION_MAX
 * msg_number	: the ID of message
 * Ouput	: Returns the address of the callback function
 */
static uint32_t find_callback(uint8_t sessionid, uint16_t msg_number, ipc_module_t *ipc_module)
{
	unsigned int i;
	ipc_session_t *session = find_session(ipc_module, sessionid);
	
	if (session == NULL) {
		return 0U;
	}

	for(i = 0U; i < session->msg_cnt; i++) {
		if (session->msg_procs[i].msg_no == msg_number)
			return (uint32_t)session->msg_procs[i].proc;
	}

	return 0U;
}

/*
 * Execute the callback function withought syncronizing signal
 * session_id	: < IPC_SESSION_MAX
 * msg_number	: the message header
 */
static void do_asyn_message(uint8_t sessionid, msg_header_t *header,
		ipc_module_t *ipc_module)
{
	uint32_t addr;
	rtk_ipc_msg_proc_t callback;

	/* Obtain the callback function according to "msg_header->msg_no" */
	addr = find_callback(sessionid, header->msg_no, ipc_module);

	/* Execute the callback function */
	if (addr == 0U) {
		IPC_ERR("%s, no callback found session[%d] msg_no[%d]\n",
			__func__, sessionid, header->msg_no);
		print_msg_header(header);
	} else {
		callback = (rtk_ipc_msg_proc_t) addr;
		callback(header->src_addr, header->msg_no, header->trans_id,
				header + 1, &(header->payload_size));
	}
}

static void do_sync_message(uint8_t sessionid, msg_header_t *header,
		ipc_module_t *ipc_module)
{
	uint32_t addr;
	rtk_ipc_msg_proc_t callback;

	IPC_DBG("%s, sessionid %d ,msg_no %d", __func__, sessionid, header->msg_no);

	/* Obtain the callback function according to "msg_header->msg_no" */
	addr = find_callback(sessionid, header->msg_no, ipc_module);

	/* Execute the callback function */
	if (addr == 0U) {
		IPC_ERR("%s, no callback found session[%d] msg_no[%d]",
			__func__, sessionid, header->msg_no);
		print_msg_header(header);
	} else {
		callback = (rtk_ipc_msg_proc_t) addr;
		callback(header->src_addr, header->msg_no, header->trans_id,
				header + 1, &(header->payload_size));
	}

	ipc_send_ack(ipc_module, header);
}

static void ipc_send_ack(ipc_module_t *ipc_module, msg_header_t *sync_hdr)
{
	msg_header_t *list_ack_header;
	uint8_t *list_ack_data;

	/* Copy to sync callback return data to list ack */
	list_ack_header = &(ipc_module->pe_to_cpu->sync_ack.msg_header);
	list_ack_data = ipc_module->pe_to_cpu->sync_ack.payload;
	memcpy(list_ack_header, sync_hdr, sizeof(msg_header_t));

	if (unlikely(sync_hdr->payload_size > PAYLOAD_SIZE)) {
		IPC_LOG("%s, callback return payload size exceed "
			"session[%d] msg_no[%d]", __func__,
			sync_hdr->src_addr.session_id, sync_hdr->msg_no);
		memcpy(list_ack_data, sync_hdr + 1, PAYLOAD_SIZE);
		list_ack_header->payload_size = (uint16_t)PAYLOAD_SIZE;
	} else {
		memcpy(list_ack_data, sync_hdr + 1, (uint32_t)sync_hdr->payload_size);
		list_ack_header->payload_size = sync_hdr->payload_size;
	}

	list_ack_header->ipc_flag = IPC_ACK_MSG;
	IPC_DBG("ack buffer address %p, payload size = %u", list_ack_data,
			sync_hdr->payload_size);
	hal_ipc_raise_int(CPU_RCPU0, sync_hdr->src_addr.cpu_id);
}

static void do_ack_message(uint8_t sessionid, msg_header_t *header, ipc_module_t *ipc_module)
{
	IPC_DBG("%s: ACK id 0x%x, wait trans id 0x%x",
			__func__, header->trans_id, ipc_module->wait_trans_id);

	if (ipc_module->wait_trans_id == header->trans_id) {
		g_func_data.ack_msg = header;
		cyg_semaphore_post(&wait_for_ack); 
	} else {
		IPC_ERR("ACK trans id 0x%x not match module wait trans id 0x%x",
				header->trans_id, ipc_module->wait_trans_id);
	}
}

#if (STATIC_IPC_MSG_NODE == 1)
ipc_msg_node_t ipc_msg_array[MAX_ITEM_NO];
inline ipc_msg_node_t *allc_ipc_msg_node(uint32_t offset)
{
	ipc_msg_node_t *ipc_msg = &ipc_msg_array[offset];
	if (ipc_msg) memset(ipc_msg, 0, sizeof(ipc_msg_node_t));
	return ipc_msg;
}
inline void free_ipc_msg_node(ipc_msg_node_t *ipc_msg)
{
	//nothing
}
#else /* !STATIC_IPC_MSG_NODE */
inline ipc_msg_node_t *allc_ipc_msg_node(uint32_t offset)
{
	ipc_msg_node_t *ipc_msg = (ipc_msg_node_t *)kmalloc(sizeof( ipc_msg_node_t), GFP_KERNEL);
	if (ipc_msg) memset(ipc_msg, 0, sizeof(ipc_msg_node_t));
	return ipc_msg;
}
inline void free_ipc_msg_node(ipc_msg_node_t *ipc_msg)
{
	if (ipc_msg) kfree(ipc_msg);
}
#endif /* STATIC_IPC_MSG_NODE */

void ipc_msg_list_add(uint8_t sessionid, msg_header_t *msg_header, uint32_t offset)
{
	ipc_msg_node_t *ipc_msg;
	unsigned long flags;

	ipc_msg = allc_ipc_msg_node(offset);

	if (ipc_msg == NULL) {
		IPC_ERR("%s allocate msg node failed", __func__);
		return;
	}

	ipc_msg->msg_header = msg_header;
	ipc_msg->session_id = sessionid;
	ipc_msg->offset = offset;

	spin_lock_irqsave(todo_list_lock, flags);
	list_add_tail(&ipc_msg->msg_list, &todo_list);
	spin_unlock_irqrestore(todo_list_lock, flags);
}

/*
 * Handle the queued packet in the waiting list
 * Note that IPC_SYNC_MSG and IPC_ACK_MSG are not support
 * No more need hwsem when handle the item in the waiting list
 */
static void do_message(ipc_module_t *ipc_module)
{
	uint8_t session_id;
	msg_header_t *item;
	list_ctrl_t *cpu_list_ctrl = &(g_func_data.cpu_to_pe->list_ctrl);
	list_ctrl_t *pe_list_ctrl = &(g_func_data.pe_to_cpu->list_ctrl);
	uint32_t offset, num = 0U;

	if (unlikely(pe_list_ctrl->enqueue_offset == 0xFFFFFFFF))
		offset = pe_list_ctrl->done_offset;
	else
		offset = pe_list_ctrl->enqueue_offset;
	IPC_DBG("%s, todo offset = %u, done_offset = %u, offset = %u", __func__,
			pe_todo_offset, pe_list_ctrl->done_offset, offset);

	while ((pe_todo_offset != offset) && (++num < MAX_ITEM_NO)) {
		offset = (offset + 1U) % MAX_ITEM_NO;
		item = (msg_header_t *)&(ipc_module->cpu_to_pe->list_item[offset]);

		//print_msg_header(item);
		//_rtk_ipc_dumpmem((int *)item, 64);
		session_id = item->dst_addr.session_id;

		if (session_id > IPC_SESSION_MAX)
			continue;

		switch (item->ipc_flag) {
		case IPC_ASYN_MSG:
		case IPC_SYNC_MSG:
			ipc_msg_list_add(session_id, item, offset);
			pe_list_ctrl->enqueue_offset = offset;
			cyg_flag_setbits(&ipc_thread_flag, 0x01U);
			break;
		default:
			IPC_ERR("IPC flag error, offset=%d", offset);
		}

		pe_todo_offset = cpu_list_ctrl->current_send_offset;
		IPC_DBG("%s current offset = %u, todo offset = %u, session id = %u", __func__,
				offset, pe_todo_offset, session_id);
	}
}

static unsigned int ipc_send_msg(rtk_ipc_pkt_t *p_ipc_pkt, uint8_t ipc_flag)
{
	msg_header_t *header = NULL;
	uint32_t offset = 0U;
	unsigned int rc;
	unsigned long flags;

	spin_lock_irqsave(send_idx_lock, flags);
	header = _rtk_ipc_msg_context(p_ipc_pkt->dst_cpu_id, &offset);
	if (header == NULL) {
		spin_unlock_irqrestore(send_idx_lock, flags);
		IPC_ERR("%s get msg header failed.", __func__);
		return IPC_ENOMEM;
	}

	rc = fill_header(p_ipc_pkt, ipc_flag, header, &g_func_data);

	//print_msg_header(header);
	if (rc != IPC_OK) {
		spin_unlock_irqrestore(send_idx_lock, flags);
		IPC_ERR("%s fill msg header failed.", __func__);
		return rc;
	}

	IPC_DBG("msg_header address = %p", header);

	if (p_ipc_pkt->msg_size > 0)
		memcpy(header + 1, p_ipc_pkt->msg_data, (uint32_t)p_ipc_pkt->msg_size);
	mb();
	update_send_offset(offset);
	mb();
	spin_unlock_irqrestore(send_idx_lock, flags);

	rc = hal_ipc_raise_int(CPU_RCPU0, p_ipc_pkt->dst_cpu_id);

	return rc;
}

typedef void (*rtl819x_sysrq_handler_t)(cyg_uint8  letter, cyg_addrword_t regs);
extern rtl819x_sysrq_handler_t rtl819x_sysrq_handler;

static inline unsigned __get_a2(void)
{
	unsigned long retval;
	asm volatile (" move  %0, $6" : "=r" (retval) : /* no inputs */	);
	return retval;
}

cyg_uint32 ipc_interrupt_isr(cyg_vector_t vector, cyg_addrword_t data)
{
	list_ctrl_t *cpu_list_ctrl = &(g_func_data.cpu_to_pe->list_ctrl);
	list_ctrl_t *pe_list_ctrl = &(g_func_data.pe_to_cpu->list_ctrl);
	msg_header_t *ack_item = &(g_func_data.cpu_to_pe->sync_ack.msg_header);

	rtk_interrupt_count[vector]++;

	hal_ipc_clear_int(CPU_RCPU0);
	cyg_interrupt_mask(vector);

	IPC_DBG("in %s vector = %d", __func__, vector);

#ifdef CYGNUM_IO_SERIAL_RTL819X_SERIAL_SYSRQ
	if (PE_STATUS->sysrq_trigger) {
		cyg_addrword_t regs;
		/*a2 is prepare by default_interrup_vsr as saved regs*/
		regs = __get_a2();
		rtl819x_sysrq_handler((cyg_uint8)PE_STATUS->sysrq_trigger, regs);
		PE_STATUS->sysrq_trigger = 0;
		//cyg_interrupt_unmask(vector);
		//return CYG_ISR_HANDLED;
	}
#endif

	if (pe_list_ctrl->done_offset != cpu_list_ctrl->current_send_offset ||
			ack_item->ipc_flag == IPC_ACK_MSG) {
		IPC_DBG("Calling DSR, done offset = %d, send offset = %d, ack flag = 0x%x",
				pe_list_ctrl->done_offset,
				cpu_list_ctrl->current_send_offset,
				ack_item->ipc_flag);
		pe_todo_offset = cpu_list_ctrl->current_send_offset;
		return (cyg_uint32)(CYG_ISR_HANDLED | CYG_ISR_CALL_DSR);
	}

	cyg_interrupt_unmask(vector);
	return (cyg_uint32)CYG_ISR_HANDLED;
}

void ipc_interrupt_dsr(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
	msg_header_t *ack_item = &(g_func_data.cpu_to_pe->sync_ack.msg_header);
	IPC_DBG("in %s", __func__);

	if (ack_item->ipc_flag == IPC_ACK_MSG) {
		IPC_DBG("%s do ack", __func__);
		do_ack_message(ack_item->dst_addr.session_id, ack_item, &g_func_data);
	}

	cyg_flag_setbits(&ipc_domsg_thread_flag, 0x01U);
	cyg_interrupt_unmask(vector);
}

void ipc_domsg_thread(cyg_addrword_t context)
{
	while (1) {
		cyg_flag_wait(&ipc_domsg_thread_flag, 0x01U, CYG_FLAG_WAITMODE_OR | CYG_FLAG_WAITMODE_CLR);
		do_message(&g_func_data);
	}
}

ipc_msg_node_t *dequeue_from_todo_list(void)
{
	unsigned long flags;
	ipc_msg_node_t *msg_node = NULL;

	spin_lock_irqsave(todo_list_lock, flags);
	if (!list_empty(&todo_list)) {
		msg_node = list_first_entry(&todo_list, ipc_msg_node_t, msg_list);
		list_del(&msg_node->msg_list);
	}
	spin_unlock_irqrestore(todo_list_lock, flags);
	return msg_node;
}

void dump_ipc_thread_stack(void)
{
	dump_thread_info(2U);
}

void ipc_thread_wakeup(cyg_addrword_t context)
{
	ipc_msg_node_t *ipc_msg;
	ipc_module_t *ipc_module = &g_func_data;
	msg_header_t *item;

	while (1) {
		cyg_flag_wait(&ipc_thread_flag, 0x01U, CYG_FLAG_WAITMODE_OR | CYG_FLAG_WAITMODE_CLR);
		while ((ipc_msg = dequeue_from_todo_list()) != NULL) {
			item = ipc_msg->msg_header;

			switch (item->ipc_flag) {
			case IPC_ASYN_MSG:
				do_asyn_message(ipc_msg->session_id, item, ipc_module);
				break;
			case IPC_SYNC_MSG:
				do_sync_message(ipc_msg->session_id, item, ipc_module);
				break;
			}
			IPC_DBG("thread[%d] Update done offset to %d, IPC type = %d",
					CURRENT_THREAD_ID, ipc_msg->offset, item->ipc_flag);
			mb();
			update_done_offset(ipc_msg->offset);
			mb();

			free_ipc_msg_node(ipc_msg);
		}
		cyg_flag_setbits(&ipc_domsg_thread_flag, 0x01U);
	}
}

void create_ipc_msg_thread(void)
{
	cyg_flag_init(&ipc_thread_flag);
	/* Create the thread */
	cyg_thread_create((cyg_addrword_t)PRIO_IPC,
			  ipc_thread_wakeup,
			  (cyg_addrword_t)0,
			  "IPC MSG2",
			  &ipc_thread_stack,
			  sizeof(ipc_thread_stack),
			  &ipc_thread_handle,
			  &ipc_thread_obj);
	/* Let the thread run when the scheduler starts */
	cyg_thread_resume(ipc_thread_handle);

	cyg_flag_init(&ipc_domsg_thread_flag);
	/* Create the thread */
	cyg_thread_create((cyg_addrword_t)PRIO_IPC,
			  ipc_domsg_thread,
			  (cyg_addrword_t)0,
			  "IPC MSG1",
			  &ipc_domsg_thread_stack,
			  sizeof(ipc_domsg_thread_stack),
			  &ipc_domsg_thread_handle,
			  &ipc_domsg_thread_obj);
	/* Let the thread run when the scheduler starts */
	cyg_thread_resume(ipc_domsg_thread_handle);
}

void _ipc_enable_interrupt(void)
{
	IPC_DBG("in %s", __func__);

	cyg_vector_t ipc_vec = (cyg_vector_t)CYGNUM_HAL_INTERRUPT_IPC_CPU2DSP;

	cyg_interrupt_create(ipc_vec, 0U, 0U, &ipc_interrupt_isr,
			&ipc_interrupt_dsr, &ipc_intr_handle, &ipc_intr);
	cyg_interrupt_attach(ipc_intr_handle);
	cyg_interrupt_unmask(ipc_vec);
}

void share_memory_init(void)
{
	PE_STATUS->enable_pe_rx = 1;
	PE_STATUS->enable_pe_tx = 1;
	PE_STATUS->sysrq_trigger = 0;
	PE_STATUS->nic_tx_hang = 0;
	PE_STATUS->wifi_hang = 0;
	PE_STATUS->ipc_recover = 0;
}

/* Initialize IPC.*/
uint32_t _rtk_ipc_init(uint8_t cpuid)
{
#if defined(CONFIG_SP_FEATURE)
	sp_feature = 1;
#else
	sp_feature = 0;
#endif

	ipc_module_t *ipc_module = &g_func_data;

	/*
	 * IPC cpu id 0 : ARM core 0/1
	 * IPC cpu id 1 : Taroko core 0
	 * IPC cpu id 2 : Taroko core 1
	 * Initialize the cpu id. To be used for accessing the read queues
	 */
	ipc_module->cpu_id = cpuid;

	/* The base of the shared memory region */
	ipc_module->shm_addr = SHM_ADDR;

	/* The starting of the list for this CPU */
	ipc_module->root_list = (list_t *) (ipc_module->shm_addr);
	ipc_module->cpu_to_pe = &(ipc_module->root_list[0]);
	ipc_module->pe_to_cpu = &(ipc_module->root_list[1]);
	//ipc_module->cpu_to_pe = &(ipc_module->root_list[cpuid - 1]);
	//ipc_module->pe_to_cpu = &(ipc_module->root_list[cpuid + 1]);
	ipc_module->trans_id = 0U;

	IPC_LOG("root list %p", ipc_module->root_list);
	IPC_LOG("cpu list addr = %p", ipc_module->cpu_to_pe);
	IPC_LOG("pe list addr = %p", ipc_module->pe_to_cpu);
	IPC_LOG("IPC list size = 0x%x", IPC_LIST_SIZE);
	IPC_DBG("list size = 0x%x", sizeof(list_t));
	IPC_DBG("list ctrl size = 0x%x", sizeof(list_ctrl_t));
	IPC_DBG("list item size = 0x%x", sizeof(list_item_t));
	IPC_DBG("msg_header size = 0x%x", sizeof(msg_header_t));
	IPC_DBG("PAYLOAD_SIZE = 0x%x", PAYLOAD_SIZE);
	IPC_DBG("MAX_ITEM_NO = 0x%x", MAX_ITEM_NO);

	/* Initialize the packet list buffer */
	memset((uint8_t *) ipc_module->cpu_to_pe, 0, (size_t)IPC_LIST_SIZE);
	memset((uint8_t *) ipc_module->pe_to_cpu, 0, (size_t)IPC_LIST_SIZE);

	/* Initialize the packet list index */
	initial_list_ctrl(ipc_module->cpu_to_pe);
	initial_list_ctrl(ipc_module->pe_to_cpu);

	INIT_LIST_HEAD(&ipc_module->session_list);
	mutex_init(&sync_mutex);

	create_ipc_msg_thread();

	_ipc_enable_interrupt();

	share_memory_init();

	return 0U;
}

#ifdef CONFIG_RTK_TAROKO_IPC_TEST
int asyn_callback(rtk_ipc_addr_t peer, uint16_t msg_no,
		uint16_t trans_id, const void *msg_data, uint16_t *msg_size)
{
	diag_printf("%s msg_no[%d] message[%s]\n", __func__, msg_no, (const char *)msg_data);
	return 1;
}

int sync_callback(rtk_ipc_addr_t peer, uint16_t msg_no,
		uint16_t trans_id, const void *msg_data, uint16_t *msg_size)
{
	diag_printf("%s msg_no[%d] message[%s]\n", __func__, msg_no, (const char *)msg_data);

	char sync_str[] = {"Bye from Taroko, nice to meet you"};
	int sync_str_size = sizeof(sync_str);

	*msg_size = sync_str_size;
	memcpy(msg_data, sync_str, sync_str_size);

	return 1;
}

void init_rtk_ipc_test(void)
{
	rtk_ipc_msg_handle_t test_msg[] = {
		{ .msg_no = 1, .proc = asyn_callback },
	};

	rtk_ipc_msg_handle_t sync_test_msg[] = {
		{ .msg_no = 1, .proc = sync_callback },
	};

	rtk_ipc_msg_handle_register(5, test_msg, 1);
	rtk_ipc_msg_handle_register(6, test_msg, 1);
	rtk_ipc_msg_handle_register(3, sync_test_msg, 1);
	rtk_ipc_msg_handle_unregister(6);
	rtk_ipc_msg_handle_register(6, test_msg, 1);
}
#endif /* CONFIG_RTK_TAROKO_IPC_TEST */

static bool rtk_ipc_init(struct cyg_devtab_entry *tab)
{
	unsigned int ret;
	unsigned int gimr0, gimr1;
	rtk_ipc_pkt_t pkt;

	if (isInit == 1)
		return (bool)true;

	ret = _rtk_ipc_init(CPU_RCPU0);

	if (ret != IPC_OK)
		return (bool)false;

	isInit = 1U;

	gimr0 = REG32(BSP_GIMR0_0);
	gimr1 = REG32(BSP_GIMR1_0);

	pkt.dst_cpu_id = IPC_CPU_HOST;
	pkt.session_id = IPC_SESSION_SYSTEM;
	pkt.msg_no = IPC_SYSTEM_BOOT;
	pkt.msg_data = NULL;
	pkt.msg_size = 0;

	REG32(BSP_GIMR0_0) = 0;
	REG32(BSP_GIMR1_0) = 0;
	REG32(BSP_GIMR0_0) |= BSP_CPI_P0_IE;

	rtk_ipc_msg_async_send(&pkt);

	IPC_LOG("Taroko start to sleep");
	__asm_sleep();
	IPC_LOG("Taroko wake up!");

	REG32(BSP_GIMR0_0) = gimr0;
	REG32(BSP_GIMR1_0) = gimr1;

	return (bool)true;
}

rtk_ipc_status_t rtk_ipc_msg_handle_register(rtk_ipc_session_id_t session_id,
		const rtk_ipc_msg_handle_t *msg_handle_array,
		uint16_t msg_handle_count)
{
	ipc_msg_t *ipc_msg = (ipc_msg_t *)msg_handle_array;

	return _rtk_ipc_msg_handle_register((uint8_t)session_id, ipc_msg,
			msg_handle_count);
}

rtk_ipc_status_t rtk_ipc_msg_handle_unregister(rtk_ipc_session_id_t session_id)
{
	return _rtk_ipc_msg_handle_unregister((uint8_t)session_id);
}

rtk_ipc_status_t rtk_ipc_msg_async_send(rtk_ipc_pkt_t *p_ipc_pkt)
{
	if (p_ipc_pkt->msg_size > PAYLOAD_SIZE)	{
		IPC_ERR("IPC MSG Size over %d", PAYLOAD_SIZE);
		return IPC_EINVAL;
	}

	return ipc_send_msg(p_ipc_pkt, IPC_ASYN_MSG);
}

rtk_ipc_status_t rtk_ipc_msg_sync_send(rtk_ipc_pkt_t *p_ipc_pkt,
		void *result_data, uint16_t *result_size)
{
	rtk_ipc_status_t rc;
	ipc_module_t *ipc_module = &g_func_data;
	msg_header_t *ack_header = &(ipc_module->cpu_to_pe->sync_ack.msg_header);
	uint8_t *ack_data;

	IPC_DBG("in %s, result data addr %p, result_size = %u", __func__, result_data, *result_size);

	if (p_ipc_pkt->msg_size > PAYLOAD_SIZE)	{
		IPC_ERR("IPC MSG Size over %d", PAYLOAD_SIZE);
		return IPC_EINVAL;
	}

	mutex_lock(&sync_mutex);

	ack_header->ipc_flag = IPC_USED_MSG;
	rc = ipc_send_msg(p_ipc_pkt, IPC_SYNC_MSG);
	if (rc != IPC_OK) {
		mutex_unlock(&sync_mutex);
		IPC_ERR("thread %d send sync msg failed. error code = %u",
				CURRENT_THREAD_ID, rc);
		return rc;
	}

	cyg_semaphore_init(&wait_for_ack, 0);
	rc = 0U;
	IPC_DBG("waiting for ack...");
	rc = (rtk_ipc_status_t)cyg_semaphore_timed_wait(&wait_for_ack,
			cyg_current_time() + (cyg_tick_count_t)IPC_DEFAULT_TIMEOUT);
	if (!rc) {
		ack_header->ipc_flag = IPC_USED_MSG;
		mutex_unlock(&sync_mutex);
		IPC_ERR("IPC send sync msg timeout thread %u, session_id = %u, msg_no=%u.",
				CURRENT_THREAD_ID, p_ipc_pkt->session_id, p_ipc_pkt->msg_no);
		print_list_offset();
		dump_ipc_thread_stack();
		return IPC_EINTERNAL;
	}
	IPC_DBG("wait_for_ack semaphore triggered rc = %d.", rc);

	ack_data = ipc_module->cpu_to_pe->sync_ack.payload;
	if (ack_header->payload_size > *result_size) {
		ack_header->ipc_flag = IPC_USED_MSG;
		mutex_unlock(&sync_mutex);
		IPC_ERR("%s result_size too small, payload_size = %u, result_size = %u",
				__func__, ack_header->payload_size, *result_size);
		return IPC_ENOMEM;
	}

	*result_size = ack_header->payload_size;
	IPC_DBG("%s result_size = %u", __func__, *result_size);

	memcpy(result_data, ack_data, (uint32_t)ack_header->payload_size);
	memset(ack_data, 0, PAYLOAD_SIZE);
	ack_header->ipc_flag = IPC_USED_MSG;

	mutex_unlock(&sync_mutex);

	return IPC_OK;
}

DEVTAB_ENTRY(realtek_ipc,
	"Realtek IPC",
	0,      /* Does not depend on a lower level interface */
	0,
	rtk_ipc_init,
	0,
	0
);
