#ifndef __RTK_IPC_PRIV__
#define __RTK_IPC_PRIV__

/*
 * Copyright (c) Realtek Semiconductor 2015.  All rights reserved.
 *
 */
#include <linux/list.h>
#include <linux/types.h>
#include <include/rtk_ipc.h>
#include <soc/realtek/rtk_shm.h>

#define IPC_VERSION		0x0200UL
#define IPC_DEFAULT_TIMEOUT	(5*HZ) //in ticks

/* Message Type */
#define IPC_USED_MSG		0x00U
#define	IPC_ASYN_MSG		0x01U
#define IPC_SYNC_MSG		0x02U
#define IPC_ACK_MSG		0x03U

/* Message Priority */
#define IPC_LPRIO 		0x0
#define IPC_HPRIO		0x1

/* 
 * IPC cpu id 0 : ARM core 0/1/2/3
 * IPC cpu id 1 : RCPU core 0
 * IPC cpu id 2 : RCPU core 1 
 */

#define RCPU_0

#ifdef RCPU_0
#define RCPU_NUM 		0
#define PE_IPC_CPUID_RCPU 	2
#endif

#ifdef RCPU_1
#define RCPU_NUM 		1
#define PE_IPC_CPUID_RCPU 	1
#endif

// CPU_ID
#define CPU_HOST		0x0U
#define CPU_RCPU0		0x1U
#define CPU_RCPU1		0x2U

/* Maximum number of message procedure per client */
#define RE_IPC_MAX_PROCS	0x14

#define IPC_ITEM_SIZE 		0x4000

#if (IPC_LIST_SIZE % IPC_ITEM_SIZE != 0)
#error "IPC_LIST_SIZE can not divid by IPC_ITEM_SIZE"
#endif

#define SHM_ADDR 		(RTK_SHM_DRAM_BASE + sizeof(wfo_dram_t))	/* Take this from the linker script */

#define ALIGN_CNT	(IPC_ITEM_SIZE - (4 * sizeof(uint32_t)))
#define PAYLOAD_SIZE	(IPC_ITEM_SIZE - sizeof(msg_header_t))
#define MAX_ITEM_NO	(((IPC_LIST_SIZE - sizeof(list_ctrl_t)) / IPC_ITEM_SIZE) - 1)
#define LAST_ITEM_NO	(MAX_ITEM_NO - 1)

//#define RTK_IPC_DEBUG
#ifdef RTK_IPC_DEBUG
#define IPC_DBG(fmt, args...) diag_printf("IPC_DBG: "fmt"\n", ##args)
#else
#define IPC_DBG(fmt, args...)
#endif

#define IPC_ERR(fmt, args...) diag_printf("IPC_ERR: "fmt"\n", ##args)
#define IPC_LOG(fmt, args...) diag_printf("IPC: "fmt"\n", ##args)

/**
 * struct list_ctrl_t - list index for send/receive item.
 * @done_offset: The tail index for consuming IPC item
 * @current_send_offset: The head index for sending IPC item
 * @version: IPC internal version number
 * @dummy: Used to align to IPC_ITEM_SIZE
 */
typedef struct
{
	uint32_t done_offset;
	uint32_t enqueue_offset;
	uint32_t current_send_offset;
	uint32_t version;
	uint8_t dummy[ALIGN_CNT];
} __attribute__ ((__packed__)) list_ctrl_t;

/**
 * struct msg_header_t - Information of a list_item.
 * @src_addr: Source address of list item
 * @dst_addr: Destination address of list item
 * @priority: Priority of list item
 * @ipc_flag: IPC message type of list item, ASYNC, SYNC or ACK
 * @msg_no: Target callback function id for this list item
 * @payload_size: Actual size of IPC data, can not exceed PAYLOAD_SIZE
 * @trans_id: Transmission number of this list_item
 */
typedef struct 
{
	rtk_ipc_addr_t src_addr;
	rtk_ipc_addr_t dst_addr;

	uint8_t priority;
	uint8_t ipc_flag;
	uint16_t msg_no;

	uint16_t payload_size;
	uint16_t trans_id;
} __attribute__ ((__packed__)) msg_header_t;

/**
 * struct list_item_t - Basic unit in IPC list.
 * @msg_header: Basic description about list_item
 * @payload: The data which passed to callback function
 */
typedef struct 
{
	msg_header_t msg_header;
	uint8_t payload[PAYLOAD_SIZE];
} __attribute__ ((__packed__)) list_item_t;

/**
 * struct list_t - A shared memory used to store IPC message.
 * @list_ctrl: Set of index to read/write to the list
 * @list_item: Basic unit of IPC list
 * @sync_ack: Temporary storage space for synchronous callback result data
 */
typedef struct 
{
	list_ctrl_t list_ctrl;
	list_item_t list_item[MAX_ITEM_NO];
	list_item_t sync_ack;
} __attribute__ ((__packed__)) list_t;

/**
 * struct ipc_msg_t - For user to register callbacks to IPC module.
 * @msg_no: A unique ID in each IPC session to identify this handle
 * @rsrvd: For alignment
 * @proc: A function pointer to callback for this handle
 */
typedef struct 
{
	uint16_t msg_no;
	uint16_t rsrvd;
	unsigned long proc;
} __attribute__ ((__packed__)) ipc_msg_t;

typedef struct
{
	struct list_head session_node;
	ipc_msg_t *msg_procs;
	uint8_t id;
	uint8_t msg_cnt;
} __attribute__ ((__packed__)) ipc_session_t;

/**
 * struct ipc_module_t - Information of IPC driver mixture with ipc session and ipc module.
 * @cpu_id: Unique id of CPU in the system
 * @trans_id: Cumulative transmision number
 * @rsrvd: Used for alignedment
 * @shm_addr: Base of shared memory region
 * @root_list: Base address of all IPC list
 * @cpu_to_pe: IPC list for host CPU sends message to PE0
 * @pe_to_cpu: IPC list for PE0 sends message to CPU
 * @ack_msg: msg_header pointer to target list sync_ack, used for copy back
 * 		to result_data and result_size
 * @msg_procs: Callback functions of this session
 * @wait_trans_id: Current SYNC IPC trans id waiting for ACK
 */
typedef struct 
{
	struct list_head session_list;
	uint32_t cpu_id:8;
	uint32_t trans_id:16;
	uint32_t rsrvd:8;
	uint32_t shm_addr;
	list_t *root_list;
	list_t *cpu_to_pe;
	list_t *pe_to_cpu;
	msg_header_t *ack_msg;
	uint16_t wait_trans_id;
} __attribute__ ((__packed__)) ipc_module_t;

/**
 * struct ipc_send_data_t - metadata between IPC package and IPC header.
 * @session_id: Source session id
 * @cpu_id: Source cpu ID
 * @dst_session_id: Target session id in CPU side
 * @priority: Priority of IPC message
 * @msg_no: Target callback id for this IPC message
 * @msg_data: Data for callback function
 * @msg_size: Actual size of msg_data
 */
typedef struct 
{
	uint8_t session_id;
	uint8_t cpu_id;
	uint8_t dst_session_id;
	uint8_t priority;
	uint16_t msg_no;
	const void *msg_data;
	uint16_t msg_size;
} __attribute__ ((__packed__)) ipc_send_data_t;

typedef struct
{
	msg_header_t *msg_header;
	struct list_head msg_list;
	uint32_t offset;
	uint8_t session_id;
} ipc_msg_node_t;

#endif /* __RTK_IPC_PRIV__ */
