#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>	/* printk() */
#include <linux/slab.h>		/* kmalloc() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>	/* O_ACCMODE */
#include <linux/cdev.h>
#include <linux/major.h>
#include <linux/device.h>
#include <linux/version.h>
#include <linux/kthread.h>
#include <linux/delay.h>

#ifdef CONFIG_VOIP_LINUX_PLATFORM_DRIVER
#include <linux/io.h>
#include <linux/platform_device.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_address.h>
#include <linux/of_platform.h>
#include <linux/of_irq.h>
#include <linux/reset.h>
#endif

#ifdef CONFIG_ARCH_CORTINA
#if defined(CONFIG_ARM64) || defined(CONFIG_ARM)
#define UNCACHED_RAM(a)	((((phys_addr_t)a) & 0xffffffff) | BIT_ULL(32))
#elif defined(CONFIG_MIPS)
#include <asm/addrspace.h>
#define UNCACHED_RAM(a)	((((phys_addr_t)a) & 0x1fffffff) | KSEG1)
#endif
#endif

//#include <asm/system.h>		/* cli(), *_flags */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,12,0)
#include <linux/uaccess.h>
#else
#include <asm/uaccess.h>	/* copy_*_user */
#endif

#ifdef CONFIG_RTK_SOC_RTL8198D
#include "rtk/switch.h"
unsigned int aipcBoardChipId;
unsigned int aipcBoardChipRev;
unsigned int aipcBoardChipSubType;
#endif

#include "bspchip.h"

#include "./include/dram_share.h"
#include "./include/aipc_dev.h"		/* local definitions */
#include "./include/aipc_reg.h"
#include "./include/soc_type.h"
#include "./include/aipc_mem.h"
#include "./include/aipc_debug.h"

//#define AIPC_BUF_MALLOC

/*
 *	Need to be modified as apollo setting
 */
static ul32_t *dst_addr = NULL;		//setup by ioctl
static ul32_t *src_addr = NULL;		//setup by ioctl
static ul32_t length = 0;			//setup by ioctl

#ifdef AIPC_BUF_MALLOC
static ul32_t *aipc_buf = NULL;
#else
static ul32_t  aipc_buf[(AIPC_BUF_SIZE)/(sizeof(ul32_t))] = {0};
#endif

static const size_t ul_size = sizeof(ul32_t);
static struct class *charmodule_class;

static int aipc_dev_major =   AIPC_DEV_MAJOR;
static int aipc_dev_minor =   0;
static int aipc_dev_nr_devs = AIPC_DEV_NR_DEVS;	/* number of bare aipc_dev devices */

module_param(aipc_dev_major, int, S_IRUGO);
module_param(aipc_dev_minor, int, S_IRUGO);
module_param(aipc_dev_nr_devs, int, S_IRUGO);

static aipc_dev_t aipc_dev;
static aipc_ioc_t aipc_ioc;

extern int  aipc_dsp_boot(void);
extern unsigned int SOC_ID, SOC_BOND_ID;

MODULE_AUTHOR("Darren Shiue <darren_shiue@realtek.com>");
MODULE_LICENSE("GPL");

/*
 * Open and close
 */
int aipc_dev_open(struct inode *inode, struct file *filp)
{
	aipc_dev_t *dev; /* device information */

	dev = container_of(inode->i_cdev, aipc_dev_t, cdev);
	filp->private_data = dev; /* for other methods */
	return 0;          /* success */
}

int aipc_dev_release(struct inode *inode, struct file *filp)
{
	return 0;
}


ssize_t aipc_dev_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	aipc_dev_t *dev = filp->private_data;
	ssize_t retval = 0;

	if (down_interruptible(&dev->sem))
		return -ERESTARTSYS;

	if (copy_to_user(buf , dst_addr , count)) {
		printk("%s: copy_to_user fail\n", __func__);
		return -EFAULT;
	}

	*f_pos += count;
	retval = (ssize_t)count;

	//	SDEBUG("buf=%p count=%u f_pos=%p *f_pos=%lu\n",
	//			buf , count , f_pos , (ul32_t)*f_pos);

	up(&dev->sem);
	return retval;
}


//add padding byte in the end
ul32_t * aipc_word_copy(ul32_t *dst , ul32_t *src , size_t count)
{
	ul32_t * plDst = dst;
	ul32_t const * plSrc = src;
	size_t len = count;

	if (count%ul_size != 0U){
		SDEBUG("wrong count indication. count=%zu sizeof(ul32_t)=%zu\n", count, ul_size);
		return NULL;
	}

	while (len>=ul_size){
		*plDst++ = *plSrc++;
		len -= ul_size;
	}

	return (dst);
}

ul32_t * aipc_align_copy(ul32_t *dst , ul32_t *src , size_t count)
{
	ul32_t remainder = count%ul_size;
	u8_t   padding = 0U;

	switch (remainder)
	{
		case 1:
			padding=3U;
			break;
		case 2:
			padding=2U;
			break;
		case 3:
			padding=1U;
			break;
		default:
			padding=0U;
			break;
	}
	return aipc_word_copy(dst , src , count+padding);
}

void aipc_memcpy_padding(void *dst_addr , void *src_addr , ul32_t length)

{
	ul32_t *dp = dst_addr;			//dst pointer. should be change to final destination
	char   *bp = src_addr;
	ul32_t wc = 0U;
	ul32_t rest = length;

#ifdef AIPC_BUF_MALLOC
	aipc_buf = (ul32_t *)kmalloc(AIPC_BUF_SIZE , GFP_KERNEL);
	if (aipc_buf==NULL){
		SDEBUG("malloc failed\n");
		return;
	}
#endif

	for (wc = 0U ; rest>=AIPC_BUF_SIZE ; wc+=AIPC_BUF_SIZE, rest-=AIPC_BUF_SIZE){
#ifdef AIPC_BUF_MALLOC
		memcpy(aipc_buf  , bp , AIPC_BUF_SIZE);
		aipc_word_copy(dp , aipc_buf  , AIPC_BUF_SIZE);
#else
		memcpy(&aipc_buf , bp , AIPC_BUF_SIZE);
		aipc_word_copy(dp , (ul32_t*)&aipc_buf , AIPC_BUF_SIZE);
#endif

		bp += AIPC_BUF_SIZE;			//char * pointer
		dp += AIPC_BUF_SIZE/ul_size;	//ul32_t * pointer
	}

	if (rest>0U && rest<AIPC_BUF_SIZE){
#ifdef AIPC_BUF_MALLOC
		memset(aipc_buf, 0 , AIPC_BUF_SIZE);	//make sure all 0 in buffer before copy
		memcpy(aipc_buf , bp , rest);
		aipc_align_copy(dp , aipc_buf , rest);	//for the 4 byte padding
#else
		memset(&aipc_buf, 0 , AIPC_BUF_SIZE);	//make sure all 0 in buffer before copy
		memcpy(&aipc_buf , bp , rest);
		aipc_align_copy(dp , (ul32_t*)&aipc_buf , rest);	//for the 4 byte padding
#endif
		bp += rest;
	}else{
		SDEBUG("rest=%lu\n", rest);
	}

#ifdef AIPC_BUF_MALLOC
	if (aipc_buf){
		kfree(aipc_buf);
	}

	aipc_buf=NULL;
#endif

	//SDEBUG("dst_addr=%p src_addr=%p length=%lu rest=%lu\n",
	//		dst_addr , src_addr ,   length , rest);
}

ssize_t aipc_dev_write(struct file *filp, const char __user *buf, size_t count, loff_t *f_pos)
{
	aipc_dev_t *dev = filp->private_data;
	ssize_t retval = -ENOMEM;

	ul32_t *dp = dst_addr;			//dst pointer. should be change to final destination
	const char __user *bp = buf;	//user space pointer
	ul32_t wc;
	ul32_t rest = count;
	ul32_t tmp = 0U;

	if (down_interruptible(&dev->sem))
		return -ERESTARTSYS;

#ifdef AIPC_BUF_MALLOC
	aipc_buf = (ul32_t *)kmalloc(AIPC_BUF_SIZE , GFP_KERNEL);
	if (aipc_buf==NULL){
		SDEBUG("malloc failed\n");
		return -ERESTARTSYS;
	}
#endif

#if defined(CONFIG_VOIP_LINUX_PLATFORM_DRIVER) && defined(CONFIG_ARCH_CORTINA)
	dp = (ul32_t *)ioremap(UNCACHED_RAM(dst_addr), count);
#endif

	if (aipc_ioc.write_file){
		for (wc=0U ; rest>=AIPC_BUF_SIZE ; wc+=AIPC_BUF_SIZE, rest-=AIPC_BUF_SIZE){
#ifdef AIPC_BUF_MALLOC
			if (copy_from_user(aipc_buf  , bp , AIPC_BUF_SIZE)) {
				printk("%s aipc_ioc.write_file: copy_from_user fail\n", __func__);

				if (aipc_buf){
					kfree(aipc_buf);
				}
				aipc_buf = NULL

					return -EFAULT;
			}
			aipc_word_copy(dp , aipc_buf  , AIPC_BUF_SIZE);
#else
			if (copy_from_user(&aipc_buf , bp , AIPC_BUF_SIZE)) {
				printk("%s aipc_ioc.write_file: copy_from_user fail\n", __func__);
				return -EFAULT;
			}
			aipc_word_copy(dp , (ul32_t*)&aipc_buf , AIPC_BUF_SIZE);
#endif

			bp += AIPC_BUF_SIZE;			//char * pointer
			dp += AIPC_BUF_SIZE/ul_size;	//ul32_t * pointer
		}

		if (rest>0U && rest<AIPC_BUF_SIZE){
#ifdef AIPC_BUF_MALLOC
			memset(aipc_buf, 0 , AIPC_BUF_SIZE);	//make sure all 0 in buffer before copy
			if (copy_from_user(aipc_buf , bp , rest)) {
				printk("%s aipc_ioc.write_file: copy_from_user fail\n", __func__);

				if (aipc_buf){
					kfree(aipc_buf);
				}
				aipc_buf = NULL

					return -EFAULT;
			}
			aipc_align_copy(dp , aipc_buf , rest);	//for the 4 byte padding
#else
			memset(&aipc_buf, 0 , AIPC_BUF_SIZE);	//make sure all 0 in buffer before copy
			if (copy_from_user(&aipc_buf , bp , rest)) {
				printk("%s aipc_ioc.write_file: copy_from_user fail\n", __func__);

				return -EFAULT;
			}
			aipc_align_copy(dp , (ul32_t*)&aipc_buf , rest);	//for the 4 byte padding
#endif
			bp += rest;
		} else if (rest==0U) {
			//SDEBUG("rest=%lu\n", rest);
		} else {
			SDEBUG("rest=%lu\n", rest);
		}

	}
	else if (aipc_ioc.write_word){
		if (copy_from_user(&tmp , bp , ul_size)) {
			printk("%s aipc_ioc.write_word: copy_from_user fail\n", __func__);
#ifdef AIPC_BUF_MALLOC
			if (aipc_buf){
				kfree(aipc_buf);
			}
			aipc_buf = NULL;
#endif
			return -EFAULT;
		}
		if(dst_addr){
			*dst_addr = tmp;
			SDEBUG("write word=0x%lx tmp=%lx\n", *dst_addr , tmp);
		}
		else{
			SDEBUG("wrong dst_address\n");
		}
	}
	else if (aipc_ioc.bitop_and){
		if (copy_from_user(&tmp , bp , ul_size)) {
			printk("%s aipc_ioc.bitop_and: copy_from_user fail\n", __func__);
#ifdef AIPC_BUF_MALLOC
			if (aipc_buf){
				kfree(aipc_buf);
			}
			aipc_buf = NULL;
#endif
			return -EFAULT;
		}
		if(dst_addr){
			*dst_addr &= tmp;
			SDEBUG("and result 0x%lx tmp=%lx\n", *dst_addr , tmp);
		}
		else{
			SDEBUG("wrong dst_address\n");
		}
	}
	else if (aipc_ioc.bitop_or){
		if (copy_from_user(&tmp , bp , ul_size)) {
			printk("%s aipc_ioc.bitop_or: copy_from_user fail\n", __func__);
#ifdef AIPC_BUF_MALLOC
			if (aipc_buf){
				kfree(aipc_buf);
			}
			aipc_buf = NULL;
#endif
			return -EFAULT;
		}
		if(dst_addr){
			*dst_addr |= tmp;
			SDEBUG("or result 0x%lx tmp=%lx\n", *dst_addr , tmp);
		}
		else{
			SDEBUG("wrong dst_address\n");
		}
	}
	else if (aipc_ioc.bitop_xor){
		if (copy_from_user(&tmp , bp , ul_size)) {
			printk("%s aipc_ioc.bitop_xor: copy_from_user fail\n", __func__);
#ifdef AIPC_BUF_MALLOC
			if (aipc_buf){
				kfree(aipc_buf);
			}
			aipc_buf = NULL;
#endif
			return -EFAULT;
		}
		if(dst_addr){
			*dst_addr ^= tmp;
			SDEBUG("xor result 0x%lx tmp=%lx\n", *dst_addr , tmp);
		}
		else{
			SDEBUG("wrong dst_address\n");
		}
	}
	else if (aipc_ioc.bitop_not){
		if (copy_from_user(&tmp , bp , ul_size)) {
			printk("%s aipc_ioc.bitop_not: copy_from_user fail\n", __func__);
#ifdef AIPC_BUF_MALLOC
			if (aipc_buf){
				kfree(aipc_buf);
			}
			aipc_buf = NULL;
#endif
			return -EFAULT;
		}
		if(dst_addr){
			*dst_addr &= ~tmp;
			SDEBUG("not result 0x%lx tmp=%lx\n", *dst_addr , tmp);
		}
		else{
			SDEBUG("wrong dst_address\n");
		}
	}else{
		SDEBUG("undefined write action\n");
	}


#ifdef AIPC_BUF_MALLOC
	if (aipc_buf){
		kfree(aipc_buf);
	}

	aipc_buf=NULL;
#endif

	*f_pos += count;
	retval = (ssize_t)count;

	/* update the size */
	if (dev->size < *f_pos)
		dev->size = (unsigned long)(*f_pos);

	//	SDEBUG("buf=%p bp=%p count=%u rest=%lu f_pos=%p *f_pos=%lu\n",
	//			buf , bp , count , rest , f_pos , (ul32_t)*f_pos);

#if defined(CONFIG_VOIP_LINUX_PLATFORM_DRIVER) && defined(CONFIG_ARCH_CORTINA)
	iounmap((void __iomem *)dp);
#endif

	up(&dev->sem);

	return retval;
}

void aipc_dev_memcpy( void *dst_addr , void *src_addr , ul32_t length , int add_padding )
{
	if (!add_padding)
		memcpy(dst_addr , src_addr , length);
	else{
		aipc_memcpy_padding(dst_addr , src_addr , length);
	}
}

/*
 * The ioctl() implementation
 */
long aipc_dev_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int retval = 0;
	ul32_t rom_addr = 0U;
	ul32_t sram_addr = 0U;

	switch(cmd) {
		case IOCTL_DST_ADDR: /* Set: arg points to the value */
			dst_addr	= (ul32_t*)arg;
			break;

		case IOCTL_SRC_ADDR:
			src_addr	= (ul32_t*)arg;
			break;

		case IOCTL_LENGTH:
			length		= (ul32_t)arg;
			break;

		case IOCTL_COPY:
			aipc_dev_memcpy(dst_addr , src_addr , length , 0);
			break;

		case IOCTL_COPY_PADDING:
			aipc_dev_memcpy(dst_addr , src_addr , length , 1);
			break;

		case IOCTL_WF:	/* Set: Write File*/
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.write_file = 1;
			break;

		case IOCTL_WW:	/* Set: Write Word*/
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.write_word = 1;
			break;

		case IOCTL_BITOP_AND:
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.bitop_and  = 1;
			break;

		case IOCTL_BITOP_OR:
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.bitop_or   = 1;
			break;

		case IOCTL_BITOP_XOR:
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.bitop_xor  = 1;
			break;

		case IOCTL_BITOP_NOT:
			memset(&aipc_ioc , 0 , sizeof(aipc_ioc));
			aipc_ioc.bitop_not  = 1;
			break;

		case IOCTL_BOOT_PE_PHASE1:
#ifdef BOOT_DSP_DELAY
			//msleep(BOOT_DSP_DELAY_TIME);
			mdelay(BOOT_DSP_DELAY_TIME);
#endif
#ifndef CONFIG_VOIP_LINUX_PLATFORM_DRIVER
			aipc_zone_set(zp_dsp_boot);
#endif
			break;

		case IOCTL_BOOT_PE_PHASE2:
			REG32( 0xb8000200 ) = 0x80171600;
			REG32( 0xb8000208 ) = 0xd805c224;
			mdelay(5);
			SDEBUG("boot dsp\n");
			aipc_dsp_boot();
			SDEBUG("boot dsp done\n");
			break;

		case IOCTL_SET_SHARE_MEM_VAR:
			dst_addr = aipc_share_mem_var_map_addr((char *)arg);
			break;

		case IOCTL_CPU_DRAM_UNMAP:
			break;

		case IOCTL_CPU_SRAM_MAP:
			break;

		case IOCTL_DSP_ENTRY:
			aipc_dsp_entry();
			break;

		case IOCTL_SOC_SRAM_BACKUP:
			aipc_soc_sram_backup();
			break;

		case IOCTL_ZONE_SET:
#ifndef CONFIG_VOIP_LINUX_PLATFORM_DRIVER
			aipc_zone_set(zp_dsp_init);
#endif
			break;

		case IOCTL_TRIGGER:
			break;
		case IOCTL_ROM_SET:
			rom_addr = (ul32_t)arg;
			aipc_rom_set(rom_addr);
			SDEBUG("set ROM. rom_addr=0x%lx\n" , rom_addr);
			break;

		case IOCTL_SOC_SRAM_SET:
			sram_addr = (ul32_t)arg;
			aipc_soc_sram_set(sram_addr);
			break;

		case IOCTL_DBG_DUMP:
			break;

		default:
			retval = -ENOTTY;
	}
	return retval;
}

loff_t aipc_dev_llseek(struct file *filp, loff_t off, int whence)
{
	aipc_dev_t *dev = filp->private_data;
	loff_t newpos;

	switch(whence) {
		case 0: /* SEEK_SET */
			newpos = off;
			break;

		case 1: /* SEEK_CUR */
			newpos = filp->f_pos + off;
			break;

		case 2: /* SEEK_END */
			newpos = dev->size + off;
			break;

		default: /* can't happen */
			return -EINVAL;
	}
	if (newpos < 0) return -EINVAL;
	filp->f_pos = newpos;
	return newpos;
}

struct file_operations aipc_dev_fops = {
	.owner =    THIS_MODULE,
	.llseek =   aipc_dev_llseek,
	.read =     aipc_dev_read,
	.write =    aipc_dev_write,
	.unlocked_ioctl = aipc_dev_ioctl,
	.open =     aipc_dev_open,
	.release =  aipc_dev_release,
};


/*
 * Set up the char_dev structure for this device.
 */
static void aipc_dev_setup_cdev(aipc_dev_t *dev, int index)
{
	int err;
	dev_t devno = MKDEV(aipc_dev_major, aipc_dev_minor + index);

	cdev_init(&dev->cdev, &aipc_dev_fops);
	dev->cdev.owner = THIS_MODULE;
	dev->cdev.ops = &aipc_dev_fops;
	err = cdev_add (&dev->cdev, devno, 1U);
	/* Fail gracefully if need be */
	if (err)
		printk(KERN_NOTICE "Error %d adding aipc_dev%d\n", err, index);
}

void aipc_dev_cleanup_module(void)
{
	dev_t devno = MKDEV(aipc_dev_major, aipc_dev_minor);

	/* cleanup_module is never called if registering failed */
	unregister_chrdev_region(devno, (unsigned)aipc_dev_nr_devs);

	device_destroy(charmodule_class, devno);
	class_destroy(charmodule_class);
}

#ifdef AIPC_MODULE_INIT_ZONE_ENTRY
static void aipc_module_init_zone_entry(void)
{
	aipc_zone_set(zp_dsp_init);
}
#endif

#ifdef AIPC_MODULE_VOIP_BOARD_INFO
static void aipc_module_dump_soc_bond_info(void)
{
#if defined(CONFIG_RTL9607C) || defined(CONFIG_RTL9603CVD) || defined(CONFIG_RTL9603CVD_SERIES)
    rtk_switch_version_get(&aipcBoardChipId, &aipcBoardChipRev, &aipcBoardChipSubType);

	printk( "aipcBoardChipId=0x%08x aipcBoardChipRev=0x%08x aipcBoardChipSubType=0x%08x\n" ,
		aipcBoardChipId , aipcBoardChipRev , aipcBoardChipSubType );

	switch(aipcBoardChipId)
	{
		case APOLLO_CHIP_ID  : printk( "APOLLO_CHIP_ID   0x%08x\n" , APOLLO_CHIP_ID  ); break;
		case APOLLOMP_CHIP_ID: printk( "APOLLOMP_CHIP_ID 0x%08x\n" , APOLLOMP_CHIP_ID); break;
		case RTL9601B_CHIP_ID: printk( "RTL9601B_CHIP_ID 0x%08x\n" , RTL9601B_CHIP_ID); break;
		case RTL9602C_CHIP_ID: printk( "RTL9602C_CHIP_ID 0x%08x\n" , RTL9602C_CHIP_ID); break;
		case RTL9603CVD_CHIP_ID: printk( "RTL9603CVD_CHIP_ID 0x%08x\n" , RTL9603CVD_CHIP_ID); break;
		case RTL9607C_CHIP_ID: printk( "RTL9607C_CHIP_ID 0x%08x\n" , RTL9607C_CHIP_ID); break;
		default:               printk( "Unknown Board Chip Id\n" );
	}

	switch(aipcBoardChipRev)
	{
		case CHIP_REV_ID_0   : printk( "CHIP_REV_ID_0   0x%08x\n" , CHIP_REV_ID_0   ); break;
		case CHIP_REV_ID_A   : printk( "CHIP_REV_ID_A   0x%08x\n" , CHIP_REV_ID_A   ); break;
		case CHIP_REV_ID_B   : printk( "CHIP_REV_ID_B   0x%08x\n" , CHIP_REV_ID_B   ); break;
		case CHIP_REV_ID_C   : printk( "CHIP_REV_ID_C   0x%08x\n" , CHIP_REV_ID_C   ); break;
		case CHIP_REV_ID_D   : printk( "CHIP_REV_ID_D   0x%08x\n" , CHIP_REV_ID_D   ); break;
		case CHIP_REV_ID_E   : printk( "CHIP_REV_ID_E   0x%08x\n" , CHIP_REV_ID_E   ); break;
		case CHIP_REV_ID_F   : printk( "CHIP_REV_ID_F   0x%08x\n" , CHIP_REV_ID_F   ); break;
		case CHIP_REV_ID_G   : printk( "CHIP_REV_ID_G   0x%08x\n" , CHIP_REV_ID_G   ); break;
		case CHIP_REV_ID_MAX : printk( "CHIP_REV_ID_MAX 0x%08x\n" , CHIP_REV_ID_MAX ); break;
		default:               printk( "Unknown Board Chip Rev\n" );
	}
	
#else
	printk( "SOC_ID 0x%04x\n" , SOC_ID );

	switch(SOC_BOND_ID)	{
	    case CHIP_901   : printk( "CHIP_901   0x%04x\n" , CHIP_901 );   break;
		case CHIP_906_1 : printk( "CHIP_906_1 0x%04x\n" , CHIP_906_1 ); break;
		case CHIP_906_2 : printk( "CHIP_906_2 0x%04x\n" , CHIP_906_2 ); break;
		case CHIP_907   : printk( "CHIP_907   0x%04x\n" , CHIP_907 );   break;
		case CHIP_902   : printk( "CHIP_902   0x%04x\n" , CHIP_902 );   break;
		case CHIP_903   : printk( "CHIP_903   0x%04x\n" , CHIP_903 );   break;
		case CHIP_96    : printk( "CHIP_96    0x%04x\n" , CHIP_96 );    break;
		case CHIP_98B   : printk( "CHIP_98B   0x%04x\n" , CHIP_98B );   break;
		case CHIP_DBG   : printk( "CHIP_DBG   0x%04x\n" , CHIP_DBG );   break;
		default: printk( "unknown bond id\n");
	}
#endif
}
#endif

#ifdef CONFIG_RTL9607C
static void aipc_enable_9607c_uart1(void)
{
	#define IO_MODE_EN  0xbb023014
	#define IO_UART_SEL 0xbb000034

	REG32( IO_MODE_EN )  |= BIT(3);
	REG32( IO_UART_SEL ) |= ( BIT(10) | BIT(2) );
}
#endif

static void aipc_module_pre_init(void)
{
#ifdef CONFIG_RTL9607C
	aipc_enable_9607c_uart1();
#endif

#ifdef AIPC_MODULE_VOIP_BOARD_INFO
	aipc_module_dump_soc_bond_info();
#endif

#ifdef  AIPC_MODULE_INIT_ZONE_ENTRY
    aipc_module_init_zone_entry();
#endif
}

#ifdef CONFIG_VOIP_LINUX_PLATFORM_DRIVER
u32_t ca_voip_ipc_phy_base = 0;
void __iomem *ca_voip_ipc_mem_base = NULL;
void __iomem *ca_voip_ipc_per_int_base = NULL;
struct reset_control *ca_voip_pe_reset = NULL;
struct reset_control *ca_voip_rcpu_reset = NULL;
void __iomem *ca_voip_dsp_boot_ins_base = NULL;

static const struct of_device_id cortina_voip_ipc_of_match[] = {
	{ .compatible = "cortina,voip_ipc", },
	{ .compatible = "realtek,wfo_pe", },
	{},
};

static int aipc_module_mem_init(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	const struct of_device_id *match;
	struct resource mem_resource;
	struct device_node *memory;
	phys_addr_t addr;
	u64 size;
	int ret;

	match = of_match_device(cortina_voip_ipc_of_match, &pdev->dev);
	if (!match){
		pr_err("%s: Not found matched device\n", __func__);
		return -ENODEV;
	}

	ret = of_address_to_resource(np, 0, &mem_resource);
	if (ret){
		pr_err("%s: of_address_to_resource(0) returns %d\n", __func__, ret);
		return ret;
	}
	printk("%s: mem_resource for voip ipc=%pR\n", __func__, &mem_resource);

	ca_voip_ipc_per_int_base = devm_ioremap(&pdev->dev, mem_resource.start, resource_size(&mem_resource));
	if (!ca_voip_ipc_per_int_base)
		pr_err("%s: ioremap of voip ipc per int failed\n", __func__);
	printk("%s: ca_voip_ipc_per_int_base=%p\n", __func__, ca_voip_ipc_per_int_base);

	ret = of_address_to_resource(np, 1, &mem_resource);
	if (ret){
		pr_err("%s: of_address_to_resource(1) returns %d\n", __func__, ret);
		return ret;
	}
	printk("%s: mem_resource for voip dsp boot ins=%pR\n", __func__, &mem_resource);

	ca_voip_dsp_boot_ins_base = devm_ioremap(&pdev->dev, mem_resource.start, resource_size(&mem_resource));
	if (!ca_voip_dsp_boot_ins_base)
		pr_err("%s: ioremap of voip dsp boot ins failed\n", __func__);
	printk("%s: ca_voip_dsp_boot_ins_base=%p\n", __func__, ca_voip_dsp_boot_ins_base);

	memory = of_parse_phandle(np, "memory-region", 0);
	if (!memory)
		return -EINVAL;

	addr = of_translate_address(memory, of_get_address(memory, 0, &size, NULL));

	ca_voip_ipc_phy_base = addr & 0xFFFFFFFF;
	printk("%s: ca_voip_ipc_phy_base=0x%08x\n", __func__, ca_voip_ipc_phy_base);

	ca_voip_ipc_mem_base = devm_ioremap_wc(&pdev->dev, UNCACHED_RAM(ca_voip_ipc_phy_base), size);
	if (!ca_voip_ipc_mem_base)
		pr_err("%s: ioremap of voip ipc failed\n", __func__);
	printk("%s: ca_voip_ipc_mem_base=0x%p\n", __func__, ca_voip_ipc_mem_base);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,7,0))
	ca_voip_pe_reset = of_reset_control_get_shared(np, "pe_reset");
#else
	ca_voip_pe_reset = of_reset_control_get(np, "pe_reset");
#endif
	if (IS_ERR(ca_voip_pe_reset))
		pr_err("Error: can't get PE reset controller!\n");
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4,7,0))
	ca_voip_rcpu_reset = of_reset_control_get_shared(np, "rcpu_reset");
#else
	ca_voip_rcpu_reset = of_reset_control_get(np, "rcpu_reset");
#endif
	if (IS_ERR(ca_voip_rcpu_reset))
		pr_err("Error: can't get RCPU reset controller!\n");

	return 0;
}

static int ca_voip_ipc_probe(struct platform_device *pdev)
{
	aipc_module_mem_init(pdev);

	return 0;
}

static int ca_voip_ipc_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver ca_voip_ipc_driver = {
	.probe = ca_voip_ipc_probe,
	.remove = ca_voip_ipc_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "ca_voip_ipc",
		.of_match_table = of_match_ptr(cortina_voip_ipc_of_match),
	},
};

MODULE_DEVICE_TABLE(of, cortina_voip_ipc_of_match);

static int __init aipc_dev_init_module(void)
#else
int aipc_dev_init_module(void)
#endif /* CONFIG_VOIP_LINUX_PLATFORM_DRIVER */
{
	int result;
	dev_t dev = 0U;
	struct device *dp;

#ifdef CONFIG_RTK_WFO
	extern int wfo_enable;
	if (!wfo_enable) {
		return 0;
	}
#endif /* CONFIG_RTK_WFO */

	aipc_module_pre_init();

	memset(&aipc_dev , 0 , sizeof(aipc_dev));
	memset(&aipc_ioc , 0 , sizeof(aipc_ioc));

	result = alloc_chrdev_region(&dev, (unsigned)aipc_dev_minor, (unsigned)aipc_dev_nr_devs, DEVICE_NAME);
	aipc_dev_major = (int)MAJOR(dev);
	aipc_dev_minor = (int)MINOR(dev);

	printk("wfo_pe: register chrdev(%d,%d)\n", aipc_dev_major, aipc_dev_minor);

	if (result) {
		printk("wfo_pe: can't get major %d\n", aipc_dev_major);
		goto fail;
	}

	charmodule_class = class_create(THIS_MODULE, DEVICE_NAME);
	if (IS_ERR(charmodule_class))
		return -EFAULT;

	dp = device_create(charmodule_class, NULL , MKDEV(aipc_dev_major, aipc_dev_minor) , NULL , DEVICE_NAME);

	if (IS_ERR(dp))
		printk( "wfo_pe: create device failed\n" );
	else
		printk( "wfo_pe: create device successed\n" );

	sema_init(&aipc_dev.sem , 1);

	aipc_dev_setup_cdev(&aipc_dev, 0);

	printk( "wfo_pe: init done\n" );

#ifdef CONFIG_VOIP_LINUX_PLATFORM_DRIVER
       return platform_driver_register(&ca_voip_ipc_driver);
#else
	return 0; /* succeed */
#endif


fail:
	aipc_dev_cleanup_module();
	return result;
}

module_init(aipc_dev_init_module);
module_exit(aipc_dev_cleanup_module);
