#include <linux/module.h>
#include <linux/slab.h>
#include <linux/mutex.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>

#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_address.h>
#include <linux/of_irq.h>
#include <linux/of_platform.h>
#include <linux/io.h>

#include <soc/realtek/rtk_shm.h>

#include "rtk_soft_ipc_priv.h"
#include "rtk_pelog.h"

MODULE_LICENSE("GPL");

DEFINE_MUTEX(ipc_sync_send);
static uint32_t pe0_publisher_todo_offset;
static uint16_t max_list_use[IPC_CPU_NUMBER];
static struct IPC_Module *module_context;

#if defined(DBG_WFO_CMD)
#include "../dbg_wfo_cmd.c"
#endif /* DBG_WFO_CMD */

#if (STATIC_IPC_MSG_NODE == 1)
ipc_msg_node_t ipc_msg_array[MAX_ITEM_NO] = {0};
inline ipc_msg_node_t *alloc_ipc_msg_node(uint32_t offset)
{
	ipc_msg_node_t *ipc_msg = &ipc_msg_array[offset];

	if (ipc_msg)
		memset(ipc_msg, 0, sizeof(ipc_msg_node_t));
	return ipc_msg;
}

inline void free_ipc_msg_node(ipc_msg_node_t *ipc_msg)
{
	/* No need to free */
}
#else
inline ipc_msg_node_t *alloc_ipc_msg_node(uint32_t offset)
{
	ipc_msg_node_t *ipc_msg = (ipc_msg_node_t *)kzalloc(sizeof( ipc_msg_node_t), GFP_KERNEL);

	return ipc_msg;
}

inline void free_ipc_msg_node(ipc_msg_node_t *ipc_msg)
{
	if (ipc_msg)
		kfree(ipc_msg);
}
#endif /* STATIC_IPC_MSG_NODE */

/**
 * ipc_msg_list_add - Add incoming message for CALLBACK_WORKER to execute.
 * @session: Session of IPC message.
 * @item: Information of the IPC message.
 * @offset: Index located in IPC list of the IPC message
 *
 * There might be new message received while we doing callback function and
 * yet the done_offset is not updated, so we need to check if the message
 * had been added before.
 */
void ipc_msg_list_add(struct ipc_context *session, struct msg_header *item,
		uint32_t offset)
{
	ipc_msg_node_t *ipc_msg;

	ipc_msg = alloc_ipc_msg_node(offset);
	if (ipc_msg == NULL) {
		IPC_ERR("%s allocate msg node failed", __func__);
		return;
	}

	ipc_msg->session = session;
	ipc_msg->item = item;
	ipc_msg->offset = offset;

	IPC_SPIN_LOCK(&module_context->todo_list_lock);
	list_add_tail(&ipc_msg->msg_list, &module_context->todo_list);
	IPC_SPIN_UNLOCK(&module_context->todo_list_lock);

	IPC_DBG("%s@cpu[%d], ipc_msg addr: %p ipc_flag = %u", __func__,
			smp_processor_id(), ipc_msg, item->ipc_flag);
}

/**
 * find_session - Find a target session before send/receive message.
 * @context: IPC module of this driver
 * @session_id: Target session ID
 *
 * Before send/recive a IPC message, should check whether the session had
 * registered or not.
 *
 * Return:
 * Pointer of a session - if success
 * NULL - if session_id has no register session
 */
static struct ipc_context *find_session(struct IPC_Module *context,
				      uint8_t session_id)
{
	struct list_head *ptr;
	struct ipc_context *session;
	unsigned long flags;

	IPC_SPIN_LOCK_IRQ(&module_context->session_lock, flags);
	list_for_each(ptr, &context->session_list) {
		session = list_entry(ptr, struct ipc_context, list);
		if (session->addr.session_id == session_id) {
			IPC_SPIN_UNLOCK_IRQ(&module_context->session_lock, flags);
			return session;
		}
	}
	IPC_SPIN_UNLOCK_IRQ(&module_context->session_lock, flags);

	return NULL;
}

void dumpmem(int *ptr, int size)
{
	int i;

	for (i = 0; i < size; i += 4) {
		if (i%16 == 0) {
			printk("\n0x%lx: ", (unsigned long)ptr);
		}
		printk("0x%x ", *ptr);
		ptr++;
	}
	printk("\n");
}

static void initial_list_ctrl(struct list *ipc_list)
{
	struct list_ctrl *listCtrl;

	listCtrl = &(ipc_list->list_ctrl);
	listCtrl->version = IPC_VERSION;
	listCtrl->done_offset = LAST_ITEM_NO;
	listCtrl->enqueue_offset = 0xFFFFFFFF;
	listCtrl->current_send_offset = LAST_ITEM_NO;
}

/**
 * _ipc_msg_handle_register - Private function to register IPC message callback.
 * @session_id: Id of target sessoin these msg callback registers to
 * @msg_procs: Pointer of callback functions will be registered
 * @msg_count: Amount of callback functions will be registered
 * @invoke_count: Unused variable for now, could be removed
 * @private_data: Unused variable for now, for future extension
 *
 * This function is only called by rtk_ipc_msg_handle_register with invoke_count = 0
 * and private_data NULL for now, private data maybe useful in the future.
 *
 * This function might be called simultaneously, so session_list shall take care
 * with lock.
 *
 * Return:
 * IPC_EINVAL - Parameter is invalid.
 * IPC_EEXIST - session had been registered before, each session can not register
 *		more than once.
 * IPC_ENOMEM - Could not allocate resource needed.
 * IPC_OK - OK
 */
rtk_ipc_status_t _ipc_msg_handle_register(uint8_t session_id, const struct rtk_ipc_msg *msg_procs,
		    uint16_t msg_count, uint16_t invoke_count,
		    void *private_data)
{
	struct ipc_context *session;
	uint16_t total_procs_no;
	unsigned long flags;

	if (module_context == NULL)
		return IPC_EINVAL;

	total_procs_no = msg_count + invoke_count;
	if (session_id == 0 || (total_procs_no != 0 && msg_procs == NULL) ||
	    (total_procs_no == 0 && msg_procs != NULL)) {
		IPC_ERR("register parameters error:session[%d]", session_id);
		return IPC_EINVAL;
	}

	session = find_session(module_context, session_id);
	if (session != NULL) {
		IPC_ERR("has the session:id[%d]", session_id);
		return IPC_EEXIST;
	}

	session = kmalloc(sizeof(struct ipc_context), GFP_KERNEL);
	if (session == NULL) {
		IPC_ERR("malloc failed:session[%d]", session_id);
		return IPC_ENOMEM;
	}

	if (total_procs_no == 0) {
		session->msg_procs = NULL;
	} else {
		session->msg_procs =
		    (struct rtk_ipc_msg *)kmalloc(sizeof(struct rtk_ipc_msg) *
						 total_procs_no, GFP_KERNEL);

		if (session->msg_procs == NULL) {
			kfree(session->msg_procs);
			kfree(session);
			IPC_ERR("malloc failed:session[%d]", session_id);
			return IPC_ENOMEM;
		}

		memcpy(session->msg_procs, msg_procs,
		       sizeof(struct rtk_ipc_msg) * total_procs_no);
	}

	session->msg_number = msg_count;
	session->invoke_number = invoke_count;
	session->addr.cpu_id = module_context->addr.cpu_id;
	session->addr.session_id = session_id;
	session->trans_id = 0U;
	session->private_data = private_data;
	session->wait_trans_id = 0U;
	session->ack_item = NULL;
	session->ack_offset = 0U;

	init_completion(&session->complete);
	INIT_LIST_HEAD(&session->list);

	IPC_SPIN_LOCK_IRQ(&module_context->session_lock, flags);
	list_add(&session->list, &(module_context->session_list));
	IPC_SPIN_UNLOCK_IRQ(&module_context->session_lock, flags);
	IPC_LOG("register successful:session_id[%d][%s] msg_count %d",
		session_id, session_name[session_id], msg_count);

	return IPC_OK;
}

/**
 * rtk_ipc_msg_handle_register - To register callback functions to an session
 * @session_id: Target session id to register callback function
 * @msg_handle_array: An array of callback functions to be registered
 * @msg_handle_count: Length of msg_handle_array.
 *
 * This is an public API and could be called simultaneously, but critical
 * data is protected in _ipc_msg_handle_register.
 *
 * Return:
 *	Check _ipc_msg_handle_register return values.
 */
rtk_ipc_status_t rtk_ipc_msg_handle_register(rtk_ipc_session_id_t session_id,
	const rtk_ipc_msg_handle_t *msg_handle_array, uint16_t msg_handle_count)
{
	struct rtk_ipc_msg msg_handle[MAX_CALLBACK_NUM];
	rtk_ipc_status_t result;
	uint16_t i;

	if (msg_handle_count > MAX_CALLBACK_NUM) {
		IPC_ERR("register handler more than session callback limit: %lu", MAX_CALLBACK_NUM);
		return IPC_EINVAL;
	}

	for (i = 0U; i < msg_handle_count; i++) {
		msg_handle[i].msg_no = (uint8_t) msg_handle_array[i].msg_no;
		msg_handle[i].proc = (unsigned long)msg_handle_array[i].proc;
	}
	result = _ipc_msg_handle_register((uint8_t)session_id,
			msg_handle, msg_handle_count, 0U, NULL);

	return result;
}
EXPORT_SYMBOL(rtk_ipc_msg_handle_register);

/**
 * _ipc_msg_handle_unregister - Private function to unregister IPC message callback.
 * @context: IPC session returned by find_session to be unregistered.
 *
 * This function is only called by rtk_ipc_msg_handle_unregister, and could be called
 * simultaneously, so session_list shall be protected by lock.
 *
 * Return:
 * IPC_EINVAL - Parameter error
 * IPC_ENOCLIENT - Target session is not found
 * IPC_OK - OK
 */
rtk_ipc_status_t _ipc_msg_handle_unregister(struct ipc_context *context)
{
	struct list_head *ptr;
	struct ipc_context *session;
	uint8_t session_id;
	unsigned long flags;

	if (NULL == context) {
		IPC_ERR("%s parameters invalid", __func__);
		return IPC_EINVAL;
	}

	session_id = context->addr.session_id;

	IPC_SPIN_LOCK_IRQ(&module_context->session_lock, flags);
	list_for_each(ptr, &module_context->session_list) {
		session = list_entry(ptr, struct ipc_context, list);
		if (session == context) {
			list_del(ptr);
			kfree(session->msg_procs);
			kfree(session);
			IPC_SPIN_UNLOCK_IRQ(&module_context->session_lock, flags);
			IPC_DBG("session deregister:session[%d]", session_id);
			return IPC_OK;
		}
	}
	IPC_SPIN_UNLOCK_IRQ(&module_context->session_lock, flags);
	IPC_ERR("No such session:session[%d]", session_id);

	return IPC_ENOCLIENT;
}

/**
 * rtk_ipc_msg_handle_unregister - To unregister ALL callback functions of an session
 * @session_id: Session ID to be unregistered
 *
 * This is an public API and could be called simultaneously, but critical data
 * is protected in _ipc_msg_handle_unregister.
 */
rtk_ipc_status_t rtk_ipc_msg_handle_unregister(rtk_ipc_session_id_t session_id)
{
	struct ipc_context *session;
	rtk_ipc_status_t result;

	session = find_session(module_context, session_id);
	if (session == NULL) {
		IPC_ERR("IPC has no the session:id[%d]", session_id);
		return IPC_EEXIST;
	}

	result = _ipc_msg_handle_unregister(session);

	return result;
}
EXPORT_SYMBOL(rtk_ipc_msg_handle_unregister);

/**
 * check_client_ready - Check Taroko is ready or not.
 * @cpu_id: Processor id to be check status
 *
 * Once IPC client is ready, the list version will set to an magical number.
 * We shall check client status before sending IPC message.
 */
static rtk_ipc_status_t check_client_ready(uint8_t cpu_id)
{
	struct list_ctrl *list;
	rtk_ipc_status_t result = IPC_OK;

	if (cpu_id == CPU_RCPU0)
		list = &(module_context->pe0_arm->list_ctrl);
	else if (cpu_id == CPU_RCPU1)
		list = &(module_context->pe1_arm->list_ctrl);
	else/* FIXME This is for test only */
		list = &(module_context->arm_pe0->list_ctrl);

	if (list->version == 0xFFFFFFFF) {
		IPC_ERR("please check co-processer is ready");
		result = IPC_ENOCLIENT;
	}
	return result;
}

/**
 * update_send_offset - Update current_send_offset in list_ctrl
 * @target_cpu_id: Target list send_offset to be update.
 * @send_offset: New value of send_offset
 *
 * There might be multiple threads sending IPC message in system at the
 * same time so caller should guarantee current_send_offset is protected by
 * lock.
 */
static void update_send_offset(uint8_t target_cpu_id, uint32_t send_offset)
{
	struct list_ctrl *this_list_ctrl;

	if (target_cpu_id == CPU_RCPU0) {
		this_list_ctrl = &(module_context->arm_pe0->list_ctrl);
	} else if (target_cpu_id == CPU_RCPU1) {
		this_list_ctrl = &(module_context->arm_pe1->list_ctrl);
	} else {
		this_list_ctrl = &(module_context->pe0_arm->list_ctrl);
		IPC_DBG("%s : Sending IPC to host CPU.", __func__);
	}
	IPC_DBG("%s this_list_ctrl = 0x%p, send_offset = 0x%x", __func__,
			this_list_ctrl, send_offset);

	this_list_ctrl->current_send_offset = send_offset;
}

void _update_list_use_num(uint32_t send_offset, uint32_t done_offset, uint8_t cpu_id)
{
	int32_t offset_diff = (int32_t)send_offset - (int32_t)done_offset;

	IPC_DBG("%s: cpu[%u] send %u, done %u", __func__, cpu_id,
			send_offset, done_offset);

	/* Send side, 2 offsets is the same means list full */
	if (send_offset == done_offset && cpu_id == CPU_MAIN)
		offset_diff = (int32_t)MAX_ITEM_NO;

	if (offset_diff < 0)
		offset_diff += MAX_ITEM_NO;

	if (offset_diff > max_list_use[cpu_id])
		max_list_use[cpu_id] = (uint16_t)offset_diff;
}

/**
 * _rtk_ipc_msg_context - Get a free IPC header to prepare for sending IPC message.
 * @target_cpu_id: Target list to get msg_header
 * @offset: Return msg_header index value would be stored here
 *
 * This function is used when sending a message to get a free message slot to fill
 * IPC data and might called by multiple threads, use lock to protect send_offset.
 */
static struct msg_header *_rtk_ipc_msg_context(uint8_t target_cpu_id, uint32_t *offset)
{
	struct msg_header *msg_header = NULL;
	struct list_ctrl *this_list_ctrl, *target_list_ctrl;
	struct list *this_list;
	int cnt = 0;
	uint32_t tmp_current_send_offset;

	if (target_cpu_id == CPU_RCPU0)	{
		this_list_ctrl = &(module_context->arm_pe0->list_ctrl);
		target_list_ctrl = &(module_context->pe0_arm->list_ctrl);
		this_list = module_context->arm_pe0;
	} else if (target_cpu_id == CPU_RCPU1) {
		this_list_ctrl = &(module_context->arm_pe1->list_ctrl);
		target_list_ctrl = &(module_context->pe1_arm->list_ctrl);
		this_list = module_context->arm_pe1;
	} else {
		this_list_ctrl = &(module_context->pe0_arm->list_ctrl);
		target_list_ctrl = &(module_context->arm_pe0->list_ctrl);
		this_list = module_context->pe0_arm;
	}

	tmp_current_send_offset = this_list_ctrl->current_send_offset + 1UL;
	if (tmp_current_send_offset >= MAX_ITEM_NO)
		tmp_current_send_offset = 0UL;
	_update_list_use_num(tmp_current_send_offset, target_list_ctrl->done_offset, CPU_MAIN);

	do {
		if (tmp_current_send_offset != target_list_ctrl->done_offset)
			break;
		cnt++;
		udelay(10UL);
	} while (cnt < 5000);

	if (tmp_current_send_offset == (target_list_ctrl->done_offset)) {
		IPC_ERR("%s IPC list is full", __func__);
		goto done;
	}

	/* update to this_list_ctrl after data prepared */
	*offset = tmp_current_send_offset;
	msg_header = &(this_list->list_item[tmp_current_send_offset].msg_header);
	IPC_DBG("offset = %u, addr of msg_header = %p", *offset, msg_header);
done:
	return msg_header;
}

/**
 * fill_header - Fill message header content from IPC packet.
 * @context: Source session for this msg_header
 * @cpu_id: Target cpu id for this msg_header
 * @seesion_id: Target session for this msg_header
 * @ipc_flag: IPC message type
 * @priority: IPC message priority
 * @msg_no: Target callback function id for this IPC message
 * @msg_size: Size of IPC data
 * @header: Target header to be filled with, usually returnd by _rtk_ipc_msg_context
 *
 * This function is used before sending IPC message translate IPC packet to IPC message
 * header and data.
 */
static unsigned int fill_header(struct ipc_context *context, uint8_t cpu_id,
		       uint8_t session_id, uint8_t ipc_flag, uint8_t priority,
		       uint16_t msg_no, uint16_t msg_size,
		       struct msg_header *header)
{
	if (NULL == context || IPC_CPU_NUMBER <= cpu_id ||
	    IPC_ITEM_SIZE < (msg_size + sizeof(struct msg_header))) {

		IPC_ERR("Invalid Parameter");
		return IPC_EINVAL;
	}

	header->src_addr = context->addr;
	header->dst_addr.cpu_id = cpu_id;
	header->dst_addr.session_id = session_id;
	header->ipc_flag = ipc_flag;
	header->priority = priority;
	header->msg_no = msg_no;
	header->payload_size = msg_size;

	context->trans_id += 1;
	header->trans_id = context->trans_id;

	if (ipc_flag == IPC_SYNC_MSG)
		context->wait_trans_id = header->trans_id;

	return IPC_OK;
}

/**
 * find_callback - Search target callback function.
 * @messages: A set of callback functions in session
 * @msg_number: Amount of callback functions in session
 * @search_target: Target callback function id
 *
 * This function is called when receiving IPC message.
 *
 * Return:
 * Valid callback address - Target callback found
 * 0 - Target callback not found
 */
static unsigned long find_callback(struct rtk_ipc_msg *messages,
				   uint16_t msg_number,
				   uint16_t search_target)
{
	int i;

	for (i = 0; i < msg_number; ++i) {
		if (messages[i].msg_no == search_target) {
			return messages[i].proc;
		}
	}
	return 0UL;
}

/**
 * do_asyn_message - Process an asynchronous IPC message
 * @session: Specified session, used to find target callback
 * @header: IPS msg_header to be processed
 */
static int do_asyn_message(struct ipc_context *session,
			   struct msg_header *header)
{
	unsigned long addr;
	rtk_ipc_msg_proc_t callback;

	IPC_DBG("in %s", __func__);

	addr = find_callback(session->msg_procs, session->msg_number,
			     header->msg_no);
	if (addr == 0UL) {
		IPC_ERR("do_asyn_message : No intestesd message:"
			 "sender[%d:%d] msg_no[%d] receiver[%d:%d]",
			 header->src_addr.cpu_id,
			 header->src_addr.session_id, header->msg_no,
			 session->addr.cpu_id, session->addr.session_id);
	} else {
		callback = (rtk_ipc_msg_proc_t) addr;
		callback(header->src_addr, header->msg_no, header->trans_id, header + 1,
				&(header->payload_size));
	}

	return 0;
}

/**
 * do_sync_message - Process a synchronous IPC message
 * @session: Specified session, used to find target callback
 * @header: IPS msg_header to be processed
 *
 * Find the corresponding SYNC callback to call and send an ACK message back.
 */
static int do_sync_message(struct ipc_context *session,
			   struct msg_header *header)
{
	unsigned long addr;
	rtk_ipc_msg_proc_t callback;

	IPC_DBG("in %s", __func__);
	addr = find_callback(session->msg_procs, session->msg_number,
			     header->msg_no);
	if (0UL == addr) {
		IPC_ERR("do_sync_message : No intestesd message:"
			 "sender[%d:%d] msg_no[%d] receiver[%d:%d]",
			 header->src_addr.cpu_id,
			 header->src_addr.session_id, header->msg_no,
			 session->addr.cpu_id, session->addr.session_id);
	} else {
		callback = (rtk_ipc_msg_proc_t) addr;
		callback(header->src_addr, header->msg_no, header->trans_id, header + 1,
				&(header->payload_size));
	}

	ipc_send_ack(header);
	return 0;
}

/**
 * ipc_send_ack - Send back ACK message for specific SYNC message
 * @sync_hdr: Target SYNC message to send ACK message
 *
 * Sending ACK message to release the SYNC wait.
 * Callback return data is actually store in list sync_ack slot.
 */
static void ipc_send_ack(struct msg_header *sync_hdr)
{
	struct msg_header *list_ack_header;
	uint8_t *list_ack_data;

	list_ack_header = &(module_context->arm_pe0->sync_ack.msg_header);
	list_ack_data = module_context->arm_pe0->sync_ack.payload;
	memcpy(list_ack_header, sync_hdr, sizeof(struct msg_header));

	if (unlikely(sync_hdr->payload_size > PAYLOAD_SIZE)) {
		IPC_LOG("%s, callback return payload size exceed:"
			 "sender[%d:%d] msg_no[%d]",
			 __func__, sync_hdr->src_addr.cpu_id,
			 sync_hdr->src_addr.session_id, sync_hdr->msg_no);
		memcpy(list_ack_data, sync_hdr + 1, PAYLOAD_SIZE);
		list_ack_header->payload_size = (uint16_t)PAYLOAD_SIZE;
	} else {
		memcpy(list_ack_data, sync_hdr + 1, (uint32_t)sync_hdr->payload_size);
		list_ack_header->payload_size = sync_hdr->payload_size;
	}
	list_ack_header->ipc_flag = IPC_ACK_MSG;
	_ipc_raise_int(sync_hdr->src_addr.cpu_id);
}

/**
 * do_ack_message - Process a ack IPC message
 * @session: Specified session, used to find target callback
 * @header: IPS msg_header to be processed
 */
static int do_ack_message(struct ipc_context *session, struct msg_header *header)
{
	IPC_DBG("%s: ACK id 0x%x, session wait id 0x%x",
			__func__, header->trans_id, session->wait_trans_id);

	if (NULL == session || NULL == header) {
		IPC_ERR("session or msg header NULL");
		return 0;
	}

	if (session->wait_trans_id == header->trans_id) {
		session->ack_item = header;
		complete(&session->complete);
	} else {
		IPC_ERR("ACK trans id 0x%x not match session wait id 0x%x",
				header->trans_id, session->wait_trans_id);
	}

	return 1;
}

/**
 * update_done_offset - Update done_offset after processing an IPC message
 * @peID: Target list's done_offset to be updated.
 * @done_offset: New value of done_offset.
 *
 * This function shall only called by CALLBACK_WORKER in do_msg_callback
 */
static void update_done_offset(uint8_t peID, uint32_t done_offset)
{
	struct list_ctrl *this_list;

	if (peID == 0)
		this_list = &(module_context->arm_pe0->list_ctrl);
	else
		this_list = &(module_context->arm_pe1->list_ctrl);

	if (done_offset > MAX_ITEM_NO) {
		IPC_ERR("%s done_offset over MAX_ITEM_NO", __func__);
		return;
	}

	this_list->done_offset = done_offset;
	IPC_DBG("done_offset now = %u", this_list->done_offset);
}

/**
 * NOT USED.
 * get_next_todo_offset - Update CPU list done offset and return to caller.
 * This function operation list offset, lock should be used to prevent race condition.
 */

static uint32_t get_next_todo_offset(void)
{
	struct list_ctrl *this_list = &(module_context->arm_pe0->list_ctrl);
	uint32_t update_offset;

	if (this_list->done_offset < LAST_ITEM_NO) {
		IPC_DBG("%s done offset less than LAST_ITEM_NO.", __func__);
		update_offset = this_list->done_offset + 1UL;
	} else {
		IPC_DBG("%s done offse greater than LAST_ITEM_NO.", __func__);
		update_offset = 0UL;
	}

	return update_offset;
}

void rtk_ipc_print_module_info(void)
{
	wfo_dram_t *shm = module_context->shm_base;
	printk("============ IPC module  ============\n");
	printk("CPU id : %d\n", module_context->addr.cpu_id);
	printk("CPU to PE list address: %lx\n", (unsigned long)(module_context->arm_pe0));
	printk("CPU to PE ack item address: %p\n", &(module_context->arm_pe0->sync_ack));
	printk("PE to CPU list address: %lx\n", (unsigned long)(module_context->pe0_arm));
	printk("PE to CPU ack item address: %p\n", &(module_context->pe0_arm->sync_ack));
	printk("IPC list size : 0x%x\n", IPC_LIST_SIZE);
	printk("IPC list packet size : 0x%x\n", IPC_ITEM_SIZE);
	printk("IPC list packet header size : 0x%x\n", sizeof(struct msg_header));
	printk("IPC list packet data size : 0x%x\n", PAYLOAD_SIZE);
	printk("IPC list item num : %d\n", MAX_ITEM_NO);
	printk("IPC mailbox irq num : %u\n", module_context->mbx_irq);
	printk("HOST IPC to do offset : %u\n", pe0_publisher_todo_offset);
	printk("PE log address : %p size : 0x%x\n", module_context->pe0_arm + IPC_LIST_SIZE, PE_LOG_SIZE);
	printk("Shared Memory addr : %p, sizeof dram_size: 0x%x sizeof heart_beat: 0x%x\n",
			shm, sizeof(shm->dram_size), sizeof(shm->hb));
#if (CONFIG_WFO_VIRT_MODULE || CONFIG_WFO_VIRT)
	printk("sizeof middleware: 0x%x\n", sizeof(shm->md));
#endif
}

void rtk_ipc_print_list_info(uint8_t cpu_id)
{
	struct list *list = &module_context->root_list[cpu_id];

	printk("done offset : %u\n", list->list_ctrl.done_offset);
	printk("send offset : %u\n", list->list_ctrl.current_send_offset);
	printk("max list use num : %u\n", max_list_use[cpu_id]);
}

void rtk_ipc_print_list_info_all(void)
{
	printk("===== CPU to PE0 list info  =====\n");
	rtk_ipc_print_list_info(CPU_MAIN);
	/* FIXME remove PE1 */
	printk("===== PE0 to CPU list info  =====\n");
	rtk_ipc_print_list_info(CPU_RCPU0);
}

int rtk_ipc_list_out_of_bound(uint32_t idx)
{
	if (idx > LAST_ITEM_NO)
		return true;
	return false;
}

static void print_msg_header(struct msg_header *header)
{
	int src_session = header->src_addr.session_id;
	int dst_session = header->dst_addr.session_id;
	int src_cpu = header->src_addr.cpu_id;
	int dst_cpu = header->dst_addr.cpu_id;
	int type = header->ipc_flag;

	if (src_cpu < 0 || dst_cpu < 0) {
		src_cpu = dst_cpu = 3;
		src_session = dst_session = type = 0;
	}

	printk("===== IPC msg_header =====\n");
	printk("src CPU = %s, session = %s\n", cpu_name[src_cpu], session_name[src_session]);
	printk("dst CPU = %s, session = %s\n", cpu_name[dst_cpu], session_name[dst_session]);
	printk("ipc_type = %s, msg_no = %u\n", ipc_type[type], header->msg_no);
	printk("payload_size = 0x%x, trans_id = %d\n", header->payload_size, header->trans_id);
}

void rtk_ipc_print_pe_list_item(uint32_t idx)
{
	struct msg_header *header = &(module_context->arm_pe0->list_item[idx].msg_header);

	IPC_LOG("Dump PE list with index = %u", idx);
	print_msg_header(header);
}

void rtk_ipc_print_cpu_list_item(uint32_t idx)
{
	struct msg_header *header = &(module_context->pe0_arm->list_item[idx].msg_header);

	printk("Dump CPU list with index = %u\n", idx);
	print_msg_header(header);
}

void print_todo_list(void)
{
	struct list_head *msg_node;
	ipc_msg_node_t *ipc_msg;

	IPC_LOG("Current IPC todo list:");
	list_for_each(msg_node, &module_context->todo_list) {
		ipc_msg = list_entry(msg_node, ipc_msg_node_t, msg_list);
		IPC_LOG("\tipc_msg:%p offset = %u", ipc_msg, ipc_msg->offset);
	}
}

ipc_msg_node_t *dequeue_from_todo_list(void)
{
	ipc_msg_node_t *msg_node = NULL;

	IPC_SPIN_LOCK(&module_context->todo_list_lock);
	if (!list_empty(&module_context->todo_list)) {
		msg_node = list_first_entry(&module_context->todo_list, ipc_msg_node_t, msg_list);
		list_del(&msg_node->msg_list);
	}
	IPC_SPIN_UNLOCK(&module_context->todo_list_lock);
	return msg_node;
}

/**
 * do_msg_callback - Do ASYN/SYNC callback functions and update done_offset.
 *
 * Executed by CALLBACK_WORKER, response for executing callback function
 * of ASYN/SYNC message and update done_offset for all type of message.
 */
static void do_msg_callback(struct work_struct *work)
{
	struct msg_header *item;
	ipc_msg_node_t *ipc_msg;

	IPC_DBG("%s @ cpu[%d]", __func__, smp_processor_id());
	//print_todo_list();

	while ((ipc_msg = dequeue_from_todo_list()) != NULL) {
		IPC_DBG("%s ipc_msg:%p offset = %u", __func__, ipc_msg, ipc_msg->offset);
		item = ipc_msg->item;
		switch (item->ipc_flag) {
		case IPC_ASYN_MSG:
			do_asyn_message(ipc_msg->session, item);
			break;
		case IPC_SYNC_MSG:
			do_sync_message(ipc_msg->session, item);
			break;
		}
		smp_mb();
		update_done_offset(CPU_MAIN, ipc_msg->offset);
		smp_mb();

		free_ipc_msg_node(ipc_msg);
	}
	schedule_work_on(IPC_WORKER, &module_context->work_msg);
}

/**
 * do_message_pe0 - Start to process IPC message.
 *
 * This is the defered work after IPC interrupt is triggered.
 * We only process ACK message immediatly in this function, leave ASYN/SYNC
 * to CALLBACK_WORKER to prevent deadlock of nested IPC SYNC message.
 */
static void do_message_pe0(struct work_struct *work)
{
	uint32_t offset, num = 0UL;
	struct ipc_context *session;
	struct msg_header *item;
	struct list_ctrl *publisher_list, *this_list;

	IPC_DBG("%s @ cpu[%d]", __func__, smp_processor_id());
	publisher_list = &(module_context->pe0_arm->list_ctrl);
	this_list = &(module_context->arm_pe0->list_ctrl);

	if (unlikely(this_list->enqueue_offset == 0xFFFFFFFF))
		offset = this_list->done_offset;
	else
		offset = this_list->enqueue_offset;
	_update_list_use_num(publisher_list->current_send_offset, this_list->done_offset,
			CPU_RCPU0);

	while ((pe0_publisher_todo_offset != offset) && (++num < MAX_ITEM_NO)) {
		offset = (offset + 1UL) % MAX_ITEM_NO;
		IPC_DBG("%s: Getting offset %d, pe0_publisher_todo_offset = %u",
				__func__, offset, pe0_publisher_todo_offset);
		item = (struct msg_header *)&(module_context->pe0_arm->list_item[offset]);

		session = find_session(module_context, item->dst_addr.session_id);
		if (session == NULL) {
			IPC_ERR("do_message: can't find the session id = %d", item->dst_addr.session_id);
			IPC_ERR("do_message: Getting offset = %u", offset);
			continue;
		}

		IPC_DBG("do_message session id = %d", item->dst_addr.session_id);
		IPC_DBG("Recev msg_type[%d] session %p", item->ipc_flag, session);
		IPC_DBG("message: sender[%d:%d] msg_no[%d] receiver[%d:%d] len %d",
					 item->src_addr.cpu_id,
					 item->src_addr.session_id, item->msg_no,
					 session->addr.cpu_id, session->addr.session_id,
					 item->payload_size);

		switch (item->ipc_flag) {
		case IPC_ASYN_MSG:
		case IPC_SYNC_MSG:
			ipc_msg_list_add(session, item, offset);
			this_list->enqueue_offset = offset;
			schedule_work(&module_context->work_cb);
			break;
		default:
			IPC_ERR("%s: IPC flag is wrong, offset = %u", __func__, offset);
			break;
		};

		pe0_publisher_todo_offset = publisher_list->current_send_offset;
	}
}

static irqreturn_t list_proc(int irq, void *device)
{
	struct list_ctrl *publisher_list, *this_list;
	struct ipc_context *session;
	struct msg_header *ack_item;

	IPC_DBG("interrupt handler: %s, cpu[%d]", __func__, smp_processor_id());
	_ipc_clear_int();

	publisher_list = &(module_context->pe0_arm->list_ctrl);
	this_list = &(module_context->arm_pe0->list_ctrl);
	ack_item = (struct msg_header *)&(module_context->pe0_arm->sync_ack);

	if (ack_item->ipc_flag == IPC_ACK_MSG) {
		IPC_DBG("%s do_ack, ack_item addr %p", __func__, ack_item);
		session = find_session(module_context, ack_item->dst_addr.session_id);
		do_ack_message(session, ack_item);
	}

	if (this_list->done_offset != publisher_list->current_send_offset) {
		IPC_DBG("IPC interrupt handler : recivie from pe0");
		pe0_publisher_todo_offset = publisher_list->current_send_offset;
		schedule_work_on(IPC_WORKER, &module_context->work_msg);
	}

	return IRQ_HANDLED;
}

/**
 * ipc_send_msg - Helper function for sending IPC message.
 * @context: Source session to send IPC message
 * @cpu_id: Target CPU id of IPC message
 * @session_id: Target session id of IPC message
 * @ipc_flag: IPC message type
 * @priority: IPC message priority
 * @msg_no: Target callback id of IPC message
 * @msg_data: Data to be used in callback function
 * @msg_size: Size of msg_data
 */
static unsigned int ipc_send_msg(struct ipc_context *context, uint8_t cpu_id,
			uint8_t session_id, uint8_t ipc_flag,
			uint8_t priority, uint16_t msg_no,
			const void *msg_data, uint16_t msg_size)
{
	struct msg_header *header = NULL;
	unsigned int rc;
	uint32_t offset = 0UL;

	if (check_client_ready(cpu_id) != IPC_OK)
		return IPC_ENOCLIENT;

	IPC_SPIN_LOCK(&module_context->send_idx_lock);
	header = _rtk_ipc_msg_context(cpu_id, &offset);
	IPC_DBG(" target msg header address:%p\n", header);

	if (header == NULL) {
		IPC_SPIN_UNLOCK(&module_context->send_idx_lock);
		return IPC_ENOMEM;
	}

	rc = fill_header(context, cpu_id, session_id, ipc_flag, priority, msg_no, msg_size, header);
	if (rc != IPC_OK) {
		IPC_SPIN_UNLOCK(&module_context->send_idx_lock);
		return rc;
	}

	if (msg_size > 0)
		memcpy_toio(header + 1, msg_data, (int)msg_size);
	smp_mb();
	update_send_offset(cpu_id, offset);
	smp_mb();
	IPC_SPIN_UNLOCK(&module_context->send_idx_lock);

	_ipc_raise_int(cpu_id);
	IPC_DBG("Send Message to [%d:%d] OK", cpu_id, session_id);

	return IPC_OK;
}

/**
 * rtk_ipc_msg_async_send - Public API to send an asynchronous IPC message.
 * @p_ipc_pkt: Pointer of IPC packet to be sended
 *
 * Return:
 * IPC_EINVAL - IPC data_size exceed maximum data size supported.
 * IPC_ENOCLIENT - Session is not registered.
 * IPC_ENOMEM - Can not get an IPC memory slot.
 * IPC_OK - OK.
 */
rtk_ipc_status_t rtk_ipc_msg_async_send(rtk_ipc_pkt_t *p_ipc_pkt)
{
	struct ipc_context *sender;

	sender = find_session(module_context, p_ipc_pkt->session_id);

	if (p_ipc_pkt->msg_size > PAYLOAD_SIZE) {
		IPC_ERR("%s message size[%d] more than PAYLOAD_SIZE[%u]",
				__func__, p_ipc_pkt->msg_size, PAYLOAD_SIZE);
		return IPC_EINVAL;
	}

	if (NULL == sender) {
		IPC_ERR("%s No Such session_id %d", __func__, p_ipc_pkt->session_id);
		return IPC_ENOCLIENT;
	}

	return ipc_send_msg(sender, p_ipc_pkt->dst_cpu_id, p_ipc_pkt->session_id, IPC_ASYN_MSG,
				p_ipc_pkt->priority, p_ipc_pkt->msg_no,
				p_ipc_pkt->msg_data, p_ipc_pkt->msg_size);
}
EXPORT_SYMBOL(rtk_ipc_msg_async_send);

/**
 * rtk_ipc_msg_sync_send - Public API to send a synchronous IPC message.
 * @p_ipc_pkt: Pointer of IPC packet to be sended
 * @result_data: User pre-allocated memory space for callback result data
 * @result_size: Size of User pre-allocated memory space
 *
 * User should aware is there any result data passed from callback function.
 * If there is result_data, user should allocate memory space before calling this API.
 */
rtk_ipc_status_t rtk_ipc_msg_sync_send(rtk_ipc_pkt_t *p_ipc_pkt, void *result_data, uint16_t *result_size)
{
	#define IPC_RECOVER_CNT 3
	static unsigned int ipc_timeout = 0;

	rtk_ipc_status_t rc;
	struct ipc_context *sender;
	struct msg_header *ack_header = (struct msg_header *)&(module_context->pe0_arm->sync_ack);;

	sender = find_session(module_context, p_ipc_pkt->session_id);

	if (p_ipc_pkt->msg_size > PAYLOAD_SIZE) {
		IPC_ERR("%s message size[%d] more than PAYLOAD_SIZE[%u]",
				__func__, p_ipc_pkt->msg_size, PAYLOAD_SIZE);
		return IPC_EINVAL;
	}

	if (sender == NULL) {
		IPC_ERR("%s No Such session_id %d", __func__, p_ipc_pkt->session_id);
		return IPC_ENOCLIENT;
	}

	IPC_MUTEX_LOCK(&ipc_sync_send);

	reinit_completion(&sender->complete);
	ack_header->ipc_flag = IPC_USED_MSG;

	rc = ipc_send_msg(sender, p_ipc_pkt->dst_cpu_id,
				p_ipc_pkt->session_id, IPC_SYNC_MSG,
				p_ipc_pkt->priority, p_ipc_pkt->msg_no,
				p_ipc_pkt->msg_data, p_ipc_pkt->msg_size);

	if (IPC_OK != rc) {
		IPC_MUTEX_UNLOCK(&ipc_sync_send);
		return rc;
	}
	IPC_DBG("%s session %p\n", __func__, sender);

	rc = wait_for_completion_timeout(&sender->complete,
					(unsigned long)IPC_DEFAULT_TIMEOUT);
	if (0UL == rc) {
		if (++ipc_timeout >= IPC_RECOVER_CNT) {
			PE_STATUS->ipc_recover = 1;
		}
		IPC_ERR("rtk_ipc_msg_sync_send timeout\n");
		IPC_MUTEX_UNLOCK(&ipc_sync_send);
		return IPC_EINTERNAL;
	}
	rc = IPC_EINTERNAL;
	PE_STATUS->ipc_recover = ipc_timeout = 0;

	if (sender->ack_item != NULL) {
		if (IPC_ACK_MSG == sender->ack_item->ipc_flag) {
			ack_header = sender->ack_item;

			if (ack_header->payload_size > 0) {
				if (ack_header->payload_size <= *result_size) {
					*result_size = ack_header->payload_size;
					memcpy_fromio(result_data, ack_header + 1,
							(int)ack_header->payload_size);
					rc = IPC_OK;
				} else {
					IPC_ERR("Buffer is too small, payload_size = %d, result_size = %d", ack_header->payload_size, *result_size);
					rc = IPC_ENOMEM;
					*result_size = ack_header->payload_size;
					goto err;
				}
			} else {
				*result_size = 0U;
				rc = IPC_OK;
			}
		} else {
			IPC_ERR("ack item %p ipc flag error(0x%x), sender = %p",
				sender->ack_item, sender->ack_item->ipc_flag, sender);
			rc = IPC_EINVAL;
			goto err;
		}
	} else {
		IPC_ERR("%s error send ack_item NULL", __func__);
		goto err;
	}

	memset_io(ack_header + 1, 0U, (int)PAYLOAD_SIZE);
err:
	ack_header->ipc_flag = IPC_USED_MSG;
	sender->ack_item = NULL;
	IPC_MUTEX_UNLOCK(&ipc_sync_send);
	return rc;
}
EXPORT_SYMBOL(rtk_ipc_msg_sync_send);

#ifdef CONFIG_OF
/* Match table for of_platform binding */
static const struct of_device_id rtk_ipc_of_match[] = {
	{ .compatible = "cortina,soft_ipc", },
	{ .compatible = "rtk,soft_ipc", },
	{},
};
MODULE_DEVICE_TABLE(of, rtk_ipc_of_match);
#endif

static int rtk_ipc_probe(struct platform_device *pdev)
{
	int rc;
	struct resource mem_resource;
	const struct of_device_id *match;
	struct device_node *np;
	unsigned long shm_size;

	module_context = kmalloc(sizeof(struct IPC_Module), GFP_KERNEL);
	module_context->addr.cpu_id = CPU_MAIN;

	/* assign DT node pointer */
	np = pdev->dev.of_node;

	/* search DT for a match */
	match = of_match_device(rtk_ipc_of_match, &pdev->dev);
	if (!match) {
		IPC_ERR("No match device node found in device tree");
		return -EINVAL;
	}

	module_context->irq_reg_to_pe = of_iomap(np, 0);
	WARN_ON(!module_context->irq_reg_to_pe);

	module_context->irq_reg_from_pe = of_iomap(np, 1);
	WARN_ON(!module_context->irq_reg_from_pe);

	if ((!module_context->irq_reg_to_pe) || (!module_context->irq_reg_from_pe))
		return -ENOMEM;

	IPC_DBG("irq_reg_to_pe map reg 0x%lx", module_context->irq_reg_to_pe);

	/* get "shared memory" from DT and convert to platform mem address resource */
	np = of_parse_phandle(pdev->dev.of_node, "memory-region", 0);
	if (!np) {
		IPC_ERR("No %s specified", "memory-region");
		return -EINVAL;
	}

	rc = of_address_to_resource(np, 0, &mem_resource);
	if (rc) {
		IPC_ERR("%s: of_address_to_resource(RTK IPC shared memory) return %d", __func__, rc);
		return rc;
	}

	shm_size = resource_size(&mem_resource) - sizeof(wfo_dram_t);
	module_context->shm_base = devm_ioremap(&pdev->dev, mem_resource.start, shm_size);
	module_context->root_list = module_context->shm_base + sizeof(wfo_dram_t);
	IPC_LOG("shared memory physical base: 0x%lx", (unsigned long)mem_resource.start);
	IPC_LOG("shared memory base: 0x%lx", (unsigned long)module_context->shm_base);
	IPC_LOG("list virtual start from: 0x%lx", (unsigned long)module_context->root_list);
	IPC_LOG("list shared memory size: 0x%lx", shm_size);

	if (!module_context->root_list)
		return -ENOMEM;

	/* let't shm set to un-init states */
	memset_io((void *)&(module_context->root_list[0]), 0xFFU, (int)shm_size);

	module_context->arm_pe0 = &(module_context->root_list[0]);

	IPC_DBG("...Clearing arm_pe0 memory.... from 0x%lx", (unsigned long)(module_context->arm_pe0));
	memset_io((void *)module_context->arm_pe0, 0x0U, IPC_LIST_SIZE);

	//module_context->arm_pe1 = &(module_context->root_list[1]);

	//IPC_DBG("...Clearing arm_pe1 memory.... from 0x%lx", (unsigned long)(module_context->arm_pe1));
	//memset_io((void *)module_context->arm_pe1, 0x0, IPC_LIST_SIZE);

	module_context->pe0_arm = &(module_context->root_list[1]);
	//module_context->pe1_arm = &(module_context->root_list[3]);

	spin_lock_init(&module_context->session_lock);
	spin_lock_init(&module_context->send_idx_lock);
	spin_lock_init(&module_context->todo_list_lock);

	INIT_WORK(&module_context->work_msg, do_message_pe0);
	INIT_WORK(&module_context->work_cb, do_msg_callback);

	IPC_DBG("cpu_id[%d] memory address 0x%lx", module_context->addr.cpu_id, (unsigned long)(module_context->arm_pe0));

	initial_list_ctrl(module_context->arm_pe0);
	//initial_list_ctrl(module_context->arm_pe1);

	INIT_LIST_HEAD(&module_context->session_list);
	INIT_LIST_HEAD(&module_context->todo_list);

	IPC_LOG("version is %x", module_context->arm_pe0->list_ctrl.version);
	np = pdev->dev.of_node;
	/* get "interrupts" property from DT */
	module_context->mbx_irq = irq_of_parse_and_map(np, 0);
	IPC_LOG("irq = %d", module_context->mbx_irq);

	rc = request_irq(module_context->mbx_irq, list_proc, 0UL, "rtk-soft-ipc", NULL);
	if (rc > 0) {
		IPC_ERR("request irq failed[%d]", rc);
		kfree(module_context);
		return rc;
	}

	rtk_pelog_init((void *)module_context->pe0_arm + IPC_LIST_SIZE);

	return 0;
}

static int rtk_ipc_remove(struct platform_device *op)
{
	if (module_context->addr.cpu_id == 0) {
		iounmap(module_context->root_list);
	}

	free_irq(module_context->mbx_irq, NULL);
	kfree(module_context);

	return 0;
}

static struct platform_driver rtk_ipc_driver = {
	.probe = rtk_ipc_probe,
	.remove = rtk_ipc_remove,
	.driver = {
		.owner = THIS_MODULE,
		.name = "rtk_ipc",
		.of_match_table = of_match_ptr(rtk_ipc_of_match),
	},
};

static int __init rtk_ipc_init(void)
{
#ifdef CONFIG_RTK_WFO
	if (!wfo_enable) {
		IPC_LOG("WFO is disable");
		return 0;
	}
#endif /* CONFIG_RTK_WFO */

	return platform_driver_register(&rtk_ipc_driver);
}

static void __exit rtk_ipc_exit(void)
{
	platform_driver_unregister(&rtk_ipc_driver);
}

module_init(rtk_ipc_init);
module_exit(rtk_ipc_exit);
