#ifndef __RTK_IPC_PRI_H__
#define __RTK_IPC_PRI_H__

#include <linux/types.h>
#include <linux/spinlock_types.h>
#include <linux/workqueue.h>
#include <linux/completion.h>
#include <linux/delay.h>
#include <rtk_ipc.h>
#include "rtk_soft_ipc_err.h"

#define IPC_VERSION	0x200UL

/* Message Type */
#define IPC_USED_MSG		0x00U
#define IPC_ASYN_MSG		0x01U
#define IPC_SYNC_MSG		0x02U
#define IPC_ACK_MSG		0x03U

/* Message Priority */
#define IPC_LPRIO		0x0
#define IPC_HPRIO		0x1

#define IPC_WORKER		0
#define CALLBACK_WORKER		2
#define IPC_CPU_NUMBER		2U
#define IPC_ITEM_SIZE		0x4000U
#define IPC_DEFAULT_TIMEOUT	(10*HZ)

#ifndef IPC_CPU_NUMBER
#error "CA IPC CPU number is not defined"
#endif

#if (IPC_LIST_SIZE % IPC_ITEM_SIZE != 0)
#error "IPC_LIST_SIZE can not divid by IPC_ITEM_SIZE"
#endif

#ifndef IPC_LIST_SIZE
#error "CA IPC list size is not defined"
#endif

#ifndef IPC_ITEM_SIZE
#error "CA IPC item size in list is not defined"
#endif

#define DUMMY_SIZE	(IPC_ITEM_SIZE  - (4 * sizeof(uint32_t)))
#define PAYLOAD_SIZE	(IPC_ITEM_SIZE - sizeof(struct msg_header))
#define MAX_ITEM_NO	(((IPC_LIST_SIZE - sizeof(struct list_ctrl)) / IPC_ITEM_SIZE) - 1)
#define LAST_ITEM_NO	(MAX_ITEM_NO - 1)

#define IPC_SPIN_LOCK(l) \
	do { \
		IPC_DBG("%s, Lock IPC, %p", __func__, l); \
		spin_lock(l); \
	} while (0)

#define IPC_SPIN_UNLOCK(l) \
	do { \
		IPC_DBG("%s, Unlock IPC, %p", __func__, l); \
		spin_unlock(l); \
	} while (0)

#define IPC_SPIN_LOCK_IRQ(l, f) \
	do { \
		IPC_DBG("%s, Lock IRQ, %p", __func__, l); \
		spin_lock_irqsave(l, f); \
	} while (0)

#define IPC_SPIN_UNLOCK_IRQ(l, f) \
	do { \
		IPC_DBG("%s, Unlock IRQ, %p", __func__, l); \
		spin_unlock_irqrestore(l, f); \
	} while (0)

#define IPC_MUTEX_LOCK(m) \
	do { \
		IPC_DBG("Lock IPC sync mutex, %p", m); \
		mutex_lock(m); \
	} while (0)

#define IPC_MUTEX_UNLOCK(m) \
	do { \
		IPC_DBG("Unlock IPC sync mutex, %p", m); \
		mutex_unlock(m); \
	} while (0)

/**
 * struct list_ctrl - list index for send/receive item.
 * @done_offset: The tail index for consuming IPC item
 * @current_send_offset: The head index for sending IPC item
 * @version: IPC internal version number
 * @dummy: Used to align to IPC_ITEM_SIZE
 */
struct list_ctrl
{
	uint32_t done_offset;
	uint32_t enqueue_offset;
	uint32_t current_send_offset;
	uint32_t version;
	uint8_t dummy[DUMMY_SIZE];
} __attribute__ ((__packed__));

/**
 * struct msg_header - Information of a list_item.
 * @src_addr: Source address of list item
 * @dst_addr: Destination address of list item
 * @priority: Priority of list item
 * @ipc_flag: IPC message type of list item, ASYNC, SYNC or ACK
 * @msg_no: Target callback function id for this list item
 * @payload_size: Actual size of IPC data, can not exceed PAYLOAD_SIZE
 * @trans_id: Transmission number of this list_item
 */
struct msg_header
{
	rtk_ipc_addr_t src_addr;
	rtk_ipc_addr_t dst_addr;
	uint8_t priority;
	uint8_t ipc_flag;
	uint16_t msg_no;	//16 bit
	uint16_t payload_size;	//16 bit
	uint16_t trans_id;	//16 bit
} __attribute__ ((__packed__));

/**
 * struct list_item - Basic unit in IPC list.
 * @msg_header: Basic description about list_item
 * @payload: The data which passed to callback function
 */
struct list_item
{
    struct msg_header msg_header;
    uint8_t payload[PAYLOAD_SIZE];
} __attribute__ ((__packed__));

/**
 * struct list - A shared memory used to store IPC message.
 * @list_ctrl: Set of index to read/write to the list
 * @list_item: Basic unit of IPC list
 * @sync_ack: Temporary storage space for synchronous callback result data
 */
struct list
{
	struct list_ctrl list_ctrl;
	struct list_item list_item[MAX_ITEM_NO];
	struct list_item sync_ack;
} __attribute__ ((__packed__)) list;

/**
 * struct ipc_context - Information used by session.
 * @list: A session link which will add to session_list
 * @addr: An unique id for this session
 * @trans_id: Cumulative transmision number
 * @private_data: Private data for this session, maybe extend for future
 * @msg_procs: Callback functions of this session
 * @msg_number: Amount of callbacks in this session
 * @invoke_number: Unused variable
 * @wait_trans_id: Current SYNC IPC trans id waiting for ACK
 * @complete: For synchronous send to wait for ACK message
 * @ack_item: msg_header pointer to target list sync_ack, used for copy back
 *		to result_data and result_size
 * @ack_offset: Unused variable
 */
struct ipc_context {
	struct list_head list;
	rtk_ipc_addr_t addr;
	uint16_t trans_id;
	void *private_data;
	struct rtk_ipc_msg *msg_procs;
	uint16_t msg_number;
	uint16_t invoke_number;
	uint16_t wait_trans_id;
	struct completion complete;
	struct msg_header *ack_item;
	uint16_t ack_offset;
};

/**
 * struct IPC_Module - Information of IPC driver.
 * @addr: CPU id of this module
 * @root_list: Address of IPC shared memory
 * @arm_pe0: IPC list for host CPU sends message to PE0
 * @arm_pe1: IPC list for host CPU sends message to PE1
 * @pe0_arm: IPC list for PE0 sends message to CPU
 * @pe1_arm: IPC list for PE1 sends message to CPU
 * @irq_reg_to_pe: Interrupt address to tirgger PE0
 * @irq_reg_from_pe: Interrupt address to trigger CPU
 * @mbx_irq: IRQ number to trigger CPU processing incoming IPC message
 * @work_msg: Defered work of interrupt handler to interpret IPC message
 * @work_cb: worker to execute ASYN/SYNC callback
 * @session_lock: lock for access session_list, used in ISR and thread context
 * @send_idx_lock: lock for sending IPC message
 * @todo_list_lock: lock for access todo_list
 * @session_list: A list of registered IPC session
 * @todo_list: A list of IPC msg for processing
 */
struct IPC_Module {
	rtk_ipc_addr_t addr;
	struct list *root_list;
	struct list *arm_pe0;
	struct list *arm_pe1;
	struct list *pe0_arm;
	struct list *pe1_arm;
	void __iomem *shm_base;
	void __iomem *irq_reg_to_pe;
	void __iomem *irq_reg_from_pe;
	unsigned int mbx_irq;
	struct work_struct work_msg;
	struct work_struct work_cb;
	spinlock_t session_lock;
	spinlock_t send_idx_lock;
	spinlock_t todo_list_lock;
	struct list_head session_list;
	struct list_head todo_list;
};

/**
 * struct rtk_ipc_msg - Handle for IPC message.
 * @msg_no: A unique ID in each IPC session to identify this handle
 * @proc: A function pointer to callback for this handle
 */
struct rtk_ipc_msg {
	uint8_t msg_no;
	unsigned long proc;
};

#define MAX_CALLBACK_NUM	32UL

/**
 * ipc_msg_node_t - Add to todo_list for CALLBACK thread.
 * @msg_list: Linux list support.
 * @session: Use to find corresponding callback to execute.
 * @item: Information of this message.
 * @offset: Index of this message in IPC list,
 * 		used to update done_offset after callback executed.
 */
typedef struct {
	struct list_head msg_list;
	struct ipc_context *session;
	struct msg_header *item;
	uint32_t offset;
} ipc_msg_node_t;

const char *session_name[] = {
	"N/A",
	"VPN",
	"WiFi Offload",
	"PKT Shaper",
	"SW Rate limiter",
	"MACSEC",
	"WMC Flooding",
	"FDB",
	"System",
};

const char *cpu_name[] = {
	"CPU_MAIN",
	"PE0",
	"PE1",
	"Invalid",
};

const char *ipc_type[] = {
	"N/A",
	"ASYNC",
	"SYNC",
	"ACK",
};

void rtk_ipc_print_status(uint8_t cpu_id);
void rtk_ipc_reset_list_info(uint8_t cpu_id);
void _ipc_raise_int(uint8_t target_cpu);
void _ipc_clear_int(void);

static void print_msg_header(struct msg_header *header);
static void ipc_send_ack(struct msg_header *sync_hdr);
static unsigned int ipc_send_msg(struct ipc_context *context, uint8_t cpu_id,
			uint8_t session_id, uint8_t ipc_flag,
			uint8_t priority, uint16_t msg_no,
			const void *msg_data, uint16_t msg_size);

#endif /* __RTK_IPC_PRI_H__ */
