#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include "rtk_soft_ipc_pe.h"
#include "rtk_soft_ipc_proc.h"
#include "rtk_pelog.h"

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 6, 0))
#define rtw_proc_ops proc_ops

#define rtw_proc_ops_open proc_open
#define rtw_proc_ops_read proc_read
#define rtw_proc_ops_write proc_write
#define rtw_proc_ops_lseek proc_lseek
#define rtw_proc_ops_release proc_release
#define rtw_proc_release single_release
#else
#define rtw_proc_ops file_operations

#define rtw_proc_ops_open open
#define rtw_proc_ops_read read
#define rtw_proc_ops_write write
#define rtw_proc_ops_lseek llseek
#define rtw_proc_ops_release release
#define rtw_proc_release single_release
#endif

uint32_t enable_print = 1;
int32_t pe_log_timer = HZ/2;
uint32_t pe_timeout_reset = 1;

static int rtk_pe_proc_name_r(struct seq_file *s, void *v)
{
	char *name = get_pe0_name();
	seq_printf(s, "%s\n", name);

	return 0;
}

static int rtk_pe_proc_open_name(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pe_proc_name_r, NULL);
}


static const struct rtw_proc_ops rtk_pe_proc_name_fops = {
	.rtw_proc_ops_open = rtk_pe_proc_open_name,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_pe_proc_version_r(struct seq_file *s, void *v)
{
	char *version = get_pe0_version();

	seq_printf(s, "%s\n", version);
	return 0;
}

static int rtk_pe_proc_open_version(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pe_proc_version_r, NULL);
}

static const struct rtw_proc_ops rtk_pe_proc_version_fops = {
	.rtw_proc_ops_open = rtk_pe_proc_open_version,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_pe_proc_status_r(struct seq_file *s, void *v)
{
	if (is_pe0_ready())
		seq_puts(s, "Boot Complete\n");
	else
		seq_puts(s, "Resetting\n");

	return 0;
}

static int rtk_pe_proc_open_status(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pe_proc_status_r, NULL);
}

static const struct rtw_proc_ops rtk_pe_proc_status_fops = {
	.rtw_proc_ops_open = rtk_pe_proc_open_status,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

#define BUF_SIZE 1024UL
struct pe_seq_context {
	char *log_ptr;
	int32_t log_size;
	int32_t data_len;
	int32_t show_state;
	int32_t done;
};


static void * pe_seq_start(struct seq_file *s, loff_t *pos)
{
	struct pe_seq_context *p = (struct pe_seq_context *)s->private;

	if (p->done)
		return NULL;

	if (p->show_state == 1)
		return p->log_ptr;

	if ((*pos) == 0) {
		p->log_ptr = rtk_pelog_get_log();
		if (p->log_ptr == NULL)
			return NULL;

		p->log_size = strlen(p->log_ptr) + 1U;
	} else
		return p->log_ptr;

	if (p->log_size > (BUF_SIZE - 1))
		p->data_len = BUF_SIZE - 1;
	else
		p->data_len = p->log_size;

	p->show_state = 1;
	return p->log_ptr;

}

static void * pe_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
	struct pe_seq_context *p = (struct pe_seq_context *)s->private;

	(*pos)++;
	if (p->done)
		return NULL;

	if (p->show_state == 1)
		return p->log_ptr;

	if (p->log_size > (BUF_SIZE - 1))
		p->data_len = BUF_SIZE - 1;
	else
		p->data_len = p->log_size;

	p->show_state = 1;
	return p->log_ptr;
}

static void pe_seq_stop(struct seq_file *s, void *v)
{
}

static int rtk_pelog_dump(struct seq_file *s, void *v)
{
	struct pe_seq_context *p = (struct pe_seq_context *)s->private;

	//seq_printf(s, (char *)v);
	if (seq_write(s, p->log_ptr, p->data_len) == 0) {
		p->log_ptr = p->log_ptr + p->data_len;
		p->log_size = p->log_size - p->data_len;
		p->show_state = 0;
		p->data_len = 0;
		if (p->log_size <= 0)
			p->done = 1;
	}

	return 0;
}

static struct seq_operations pe_seq_ops = {
	.start = pe_seq_start,
	.next = pe_seq_next,
	.stop = pe_seq_stop,
	.show = rtk_pelog_dump
};

static int rtk_pelog_open(struct inode *inode, struct file *file)
{
	struct pe_seq_context *p = __seq_open_private(file, &pe_seq_ops, sizeof(*p));

	if (!p)
		return -ENOMEM;
	memset(p, 0, sizeof(*p));
	return 0;
}

static const struct rtw_proc_ops rtk_pe_proc_log_fops = {
	.rtw_proc_ops_open = rtk_pelog_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = seq_release_private,
};

static int rtk_pelog_cmd_help(struct seq_file *s, void *v)
{
	seq_printf(s, "In rtk_pelog_cmd_help\n");

	return 0;
}

static int rtk_pelog_cmd_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pelog_cmd_help, NULL);
}

#if defined(CONFIG_WFO_VIRT_MODULE)
#define CMD_STR "show debug_cnt wfo"

void (*show_wfo_trace_count_fp)(void);
EXPORT_SYMBOL(show_wfo_trace_count_fp);

static void show_wfo_trace_cnt(char *str, size_t size)
{
	if (strncmp(str, CMD_STR, size /*strlen(CMD_STR)*/)==0) {
		if (show_wfo_trace_count_fp) {
			show_wfo_trace_count_fp();
		}
	}
}
#endif /* CONFIG_WFO_VIRT_MODULE */

static ssize_t rtk_pelog_cmd_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	int ret;
	char *str = kmalloc(sizeof(char) * size, GFP_KERNEL);

	if (!str)
		return -ENOMEM;
	if (copy_from_user(str, buf, size)) {
		kfree(str);
		return -EFAULT;
	}

	str[size-1UL] = '\0';
	RTK_PROC_DBG("buf = %s, size = %u", str, size);

#if defined(CONFIG_WFO_VIRT_MODULE)
	show_wfo_trace_cnt(str, size);
#endif /* CONFIG_WFO_VIRT_MODULE */

	ret = rtk_pelog_cmd_send(str, (uint16_t)size);
	if (ret)
		RTK_PROC_ERR("PE log cmd send error = %d", ret);

	kfree(str);
	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_cmd_fops = {
	.rtw_proc_ops_open = rtk_pelog_cmd_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_pelog_cmd_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_pelog_print_log_r(struct seq_file *s, void *v)
{
	seq_printf(s, "%d\n", enable_print);

	return 0;
}

static int rtk_pelog_print_log_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pelog_print_log_r, NULL);
}

static ssize_t rtk_pelog_print_log_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];

	if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		int value;
		int num = sscanf(tmp, "%d", &value);

		if (num) {
			if (value == 1)
				enable_print = 1UL;
			else if (value == 0)
				enable_print = 0UL;
			else {
				printk("[Wrong input] 1:enable print pe log/0: disable print pe log\n");
				return -EFAULT;
			}
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;

	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_print_log_fops = {
	.rtw_proc_ops_open = rtk_pelog_print_log_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_pelog_print_log_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_pelog_print_timer_r(struct seq_file *s, void *v)
{
	seq_printf(s, "Dump PE log for every %dms\n", pe_log_timer*10);

	return 0;
}

static int rtk_pelog_print_timer_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pelog_print_timer_r, NULL);
}

static ssize_t rtk_pelog_print_timer_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];

	if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		int value;
		int num = sscanf(tmp, "%d", &value);

		if (num) {
			if (value >= 10 && value <= 10000)
				pe_log_timer = (value * HZ/1000);
			else if (value > 10000) {
				printk("Max value is 10000ms, so set timer to 10000ms.\b");
			}
			else {
				printk("[Wrong input] input value must be larger than 10 ms & less then 10000 ms.\n");
				return -EFAULT;
			}
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;

	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_print_timer_fops = {
	.rtw_proc_ops_open = rtk_pelog_print_timer_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_pelog_print_timer_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_pe_timeout_reset_r(struct seq_file *s, void *v)
{
	seq_printf(s, "%d\n", pe_timeout_reset);

	return 0;
}

static int rtk_pe_timeout_reset_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_pe_timeout_reset_r, NULL);
}

static ssize_t rtk_pe_timeout_reset_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];

		if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		int value;
		int num = sscanf(tmp, "%d", &value);

		if (num) {
			if (value == 1)
				pe_timeout_reset = 1UL;
			else if (value == 0)
				pe_timeout_reset = 0UL;
			else {
				printk("[Wrong input] 1:enable pe reset when wdt timeout\n"
						"0: disable pe reset when wdt timeout\n");
				return -EFAULT;
			}
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;

	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_timeout_reset_fops = {
	.rtw_proc_ops_open = rtk_pe_timeout_reset_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_pe_timeout_reset_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_enable_pe_rx_r(struct seq_file *s, void *v)
{
	seq_printf(s, "%d\n", PE_STATUS->enable_pe_rx);

	return 0;
}

static int rtk_enable_pe_rx_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_enable_pe_rx_r, NULL);
}

static ssize_t rtk_enable_pe_rx_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];

	if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		int value;
		int num = sscanf(tmp, "%d", &value);

		if (num) {
			if (value == 1)
				PE_STATUS->enable_pe_rx = 1;
			else if (value == 0)
				PE_STATUS->enable_pe_rx = 0;
			else {
				printk("[Wrong input] 1:enable pe rx/0: disable pe rx\n");
				return -EFAULT;
			}
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;


	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_enable_pe_rx_fops = {
	.rtw_proc_ops_open = rtk_enable_pe_rx_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_enable_pe_rx_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_enable_pe_tx_r(struct seq_file *s, void *v)
{
	seq_printf(s, "%d\n", PE_STATUS->enable_pe_tx);

	return 0;
}

static int rtk_enable_pe_tx_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_enable_pe_tx_r, NULL);
}

static ssize_t rtk_enable_pe_tx_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];

	if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		int value;
		int num = sscanf(tmp, "%d", &value);

		if (num) {
			if (value == 1)
				PE_STATUS->enable_pe_tx = 1;
			else if (value == 0)
				PE_STATUS->enable_pe_tx = 0;
			else {
				printk("[Wrong input] 1:enable pe tx/0: disable pe tx\n");
				return -EFAULT;
			}
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;

	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_enable_pe_tx_fops = {
	.rtw_proc_ops_open = rtk_enable_pe_tx_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_enable_pe_tx_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_sysrq_trigger_r(struct seq_file *s, void *v)
{
	if (PE_STATUS->sysrq_trigger == 0)
		seq_printf(s, "0\n");
	else
		seq_printf(s, "%c\n", PE_STATUS->sysrq_trigger);

	return 0;
}

static int rtk_sysrq_trigger_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_sysrq_trigger_r, NULL);
}

static ssize_t rtk_sysrq_trigger_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	char tmp[32];
	extern void _ipc_raise_int(uint8_t target_cpu);

	if (size < 1UL)
		return -EINVAL;

	if (size > sizeof(tmp))
		return -EFAULT;

	if (buf && !copy_from_user(tmp, buf, size)) {
		char value;
		int num = sscanf(tmp, "%c", &value);

		if ((num) && (('a'<=value) && (value<='z'))) {
			PE_STATUS->sysrq_trigger = value;
			_ipc_raise_int((uint8_t)IPC_CPU_PE0);
		}
		else
			return -EFAULT;
	}
	else
		return -EFAULT;

	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_pe_proc_sysrq_trigger_fops = {
	.rtw_proc_ops_open = rtk_sysrq_trigger_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write	= rtk_sysrq_trigger_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int proc_pestatus_init(void)
{
	struct proc_dir_entry *proc_pedsp_dev_dir;
	struct proc_dir_entry *entry;

	proc_pedsp_dev_dir = proc_mkdir("rtk_pe_info", NULL);
	if (proc_pedsp_dev_dir == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_info failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "name", 0U, NULL, &rtk_pe_proc_name_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_info name failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "version", 0U, NULL, &rtk_pe_proc_version_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_info version failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "status", 0U, NULL, &rtk_pe_proc_status_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_info status failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "pe_log", 0U, NULL, &rtk_pe_proc_log_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_log failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "pe_log_cmd", 0U, NULL, &rtk_pe_proc_cmd_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_log_cmd failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "enable_print_pe_log", 0U, NULL, &rtk_pe_proc_print_log_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_enable_print_pe_log failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "print_pe_log_timer", 0U, NULL, &rtk_pe_proc_print_timer_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc rtk_pe_log_timer failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "pe_timeout_reset", 0U, NULL, &rtk_pe_proc_timeout_reset_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc pe_timeout_reset failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "enable_pe_rx", 0U, NULL, &rtk_pe_proc_enable_pe_rx_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc enable_pe_rx failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "enable_pe_tx", 0U, NULL, &rtk_pe_proc_enable_pe_tx_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc enable_pe_tx failed!");
		return 1;
	}

	entry = proc_create("rtk_pe_info" "/" "sysrq_trigger", 0U, NULL, &rtk_pe_proc_sysrq_trigger_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("create proc sysrq_trigger failed!");
		return 1;
	}

	return 0;
}

static int rtk_ipc_module_info_func(struct seq_file *s, void *v)
{
	rtk_ipc_print_module_info();

	return 0;
}

static int rtk_ipc_module_info_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_ipc_module_info_func, NULL);
}

static const struct rtw_proc_ops rtk_ipc_module_info_fops = {
	.rtw_proc_ops_open = rtk_ipc_module_info_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_ipc_list_info_func(struct seq_file *s, void *v)
{
	rtk_ipc_print_list_info_all();

	return 0;
}

static int rtk_ipc_list_info_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_ipc_list_info_func, NULL);
}

static const struct rtw_proc_ops rtk_ipc_list_info_fops = {
	.rtw_proc_ops_open = rtk_ipc_list_info_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int rtk_ipc_list_item_func(struct seq_file *s, void *v)
{
	seq_puts(s, "Use Guide: echo an index of list item\n");
	seq_puts(s, "ex. echo 2 > cpu_list_item\n");
	return 0;
}

static int rtk_ipc_list_item_open(struct inode *inode, struct file *file)
{
	return single_open(file, rtk_ipc_list_item_func, NULL);
}

static ssize_t rtk_ipc_list_item_write(struct file *file,
					const char __user *buf,
					size_t size, loff_t *pos)
{
	uint32_t idx = 0U;
	int ret;
	unsigned int proc_len;
	const char *proc_name = file->f_path.dentry->d_name.name;

	RTK_PROC_DBG("proc name = %s", proc_name);

	ret = kstrtou32_from_user(buf, size, 0UL, &idx);
	if (ret) {
		RTK_PROC_ERR("Input format error = %d.", ret);
		goto err;
	}

	if (rtk_ipc_list_out_of_bound(idx)) {
		RTK_PROC_ERR("Index value out of bound.");
		ret = -EFAULT;
		goto err;
	}

	proc_len = strlen(proc_name);
	RTK_PROC_DBG("proc name length = %d", proc_len);
	if (proc_len == CPU_PROC_LEN)
		rtk_ipc_print_cpu_list_item(idx);
	else
		rtk_ipc_print_pe_list_item(idx);
err:
	return (ssize_t)size;
}

static const struct rtw_proc_ops rtk_ipc_cpu_list_item_fops = {
	.rtw_proc_ops_open = rtk_ipc_list_item_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write = rtk_ipc_list_item_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static const struct rtw_proc_ops rtk_ipc_pe_list_item_fops = {
	.rtw_proc_ops_open = rtk_ipc_list_item_open,
	.rtw_proc_ops_read = seq_read,
	.rtw_proc_ops_write = rtk_ipc_list_item_write,
	.rtw_proc_ops_lseek = seq_lseek,
	.rtw_proc_ops_release = rtw_proc_release,
};

static int proc_rtk_ipc_init(void)
{
	struct proc_dir_entry *proc_rtk_ipc_dir;
	struct proc_dir_entry *entry;

	proc_rtk_ipc_dir = proc_mkdir("rtk_ipc_info", NULL);
	if (proc_rtk_ipc_dir == NULL) {
		RTK_PROC_ERR("Create proc rtk_ipc_info failed!");
		return 1;
	}

	entry = proc_create("rtk_ipc_info" "/" "module_info", 0U, NULL,
			&rtk_ipc_module_info_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("Create proc module_info failed!");
		return 1;
	}

	entry = proc_create("rtk_ipc_info" "/" "list_info", 0U, NULL,
			&rtk_ipc_list_info_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("Create proc list_info failed!");
		return 1;
	}

	entry = proc_create("rtk_ipc_info" "/" "cpu_list_item", 0U, NULL,
			&rtk_ipc_cpu_list_item_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("Create proc cpu_list_item failed!");
		return 1;
	}

	entry = proc_create("rtk_ipc_info" "/" "pe_list_item", 0U, NULL,
			&rtk_ipc_pe_list_item_fops);
	if (entry == NULL) {
		RTK_PROC_ERR("Create proc pe_list_item failed!");
		return 1;
	}

	return 0;
}

static int __init rtk_soft_ipc_proc_init(void)
{
	int ret;

#ifdef CONFIG_RTK_WFO
	if (!wfo_enable) {
		return 0;
	}
#endif /* CONFIG_RTK_WFO */

	ret = proc_pestatus_init();
	if (ret)
		RTK_PROC_ERR("PE status proc init failed.");

	ret = proc_rtk_ipc_init();
	if (ret)
		RTK_PROC_ERR("IPC module status init failed.");

	return 0;
}

fs_initcall(rtk_soft_ipc_proc_init);
