// SPDX-License-Identifier: GPL-2.0-only
/*
 * LEDs driver for GPIOs
 *
 * Copyright (C) 2007 8D Technologies inc.
 * Raphael Assenat <raph@8d.com>
 * Copyright (C) 2008 Freescale Semiconductor, Inc.
 */
 /******************************************************************************
*
* Copyright(c) 2024 Wistron NeWeb Corp.
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of version 2 of the GNU General Public License as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful, but WITHOUT
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
* more details.
*
*****************************************************************************/

#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/gpio/consumer.h>
#include <linux/kernel.h>
#include <linux/leds.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>
#include <linux/property.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/uaccess.h>
//#include "leds.h"
#include <linux/spinlock.h>

#define SELF_TEST_TIMEOUT   6000
#define DFS_LED_BLINK_INTERVAL   500
#define FWUpdate_LED_BLINK_INTERVAL   500
#define DFS_LED_ON 1
#define DFS_LED_OFF 0
#define FWUpdate_LED_ON 1
#define FWUpdate_LED_OFF 0

static int led_state = DFS_LED_OFF;// Assuming '0' means LED is off and '1' means LED is on
static int fwUpdate_led_state = FWUpdate_LED_OFF;// Assuming '0' means LED is off and '1' means LED is on

static int dfsled_proc_state = -1;
spinlock_t DFS_LED_lock;
spinlock_t FWUpdate_LED_lock;

#define PROC_FILENAME       "dfsled"
#define PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS       "fwUpdate_led"

#define RED_5G              3
#define GREEN_5G            4
#define RED_Option              5
#define GREEN_Option            9

struct self_test_stru {
	struct gpio_leds_priv *priv;
	struct timer_list self_test_timer;
	struct timer_list dfs_led_timer;
	struct timer_list fwUpdate_led_timer;
} self_test;
struct gpio_led_data {
	struct led_classdev cdev;
	struct gpio_desc *gpiod;
	u8 can_sleep;
	u8 blinking;
	gpio_blink_set_t platform_gpio_blink_set;
};

struct gpio_leds_priv {
	int num_leds;
	struct gpio_led_data leds[];
};

struct gpio_leds_priv *g_priv;
extern struct list_head trigger_list;

void gpio_set_dfsled(struct gpio_leds_priv *priv, int led_state)
{
	//struct gpio_leds_priv *priv = (void *)data;
	struct gpio_led_data *led1 = NULL;
	struct gpio_led_data *led2 = NULL;

	if (priv == NULL) {
		printk("gpio_set_dfsled data error\n");
		return;
	}

	led1 = &priv->leds[GREEN_5G];
	led2 = &priv->leds[RED_5G];

	if ((led1 == NULL) || (led2 == NULL)) {
		printk("gpio_set_dfsled LED error\n");
		return;
	}

	if (led_state == DFS_LED_ON) {
		//printk("Turn ON DFS LED\n");
		spin_lock_bh(&DFS_LED_lock);
		gpiod_set_value(led1->gpiod, 1);
		gpiod_set_value(led2->gpiod, 1);
		spin_unlock_bh(&DFS_LED_lock);
	} else if (led_state == DFS_LED_OFF) {
		//printk("Turn OFF DFS LED\n");
		spin_lock_bh(&DFS_LED_lock);
		gpiod_set_value(led1->gpiod, 0);
		gpiod_set_value(led2->gpiod, 0);
		spin_unlock_bh(&DFS_LED_lock);
	} else {
		//???
	}
}

void gpio_dfsled_timer_act(struct timer_list *dfstimer)
{
	struct self_test_stru *st = from_timer(st, dfstimer, dfs_led_timer);
	struct gpio_leds_priv *priv = st->priv;

	if (priv == NULL) {
		printk("gpio_dfsled_timer_act data error\n");
		return;
	}

    // Switching led state
    led_state = (led_state == DFS_LED_ON) ? DFS_LED_OFF : DFS_LED_ON;

    gpio_set_dfsled(priv, led_state);

    mod_timer(&self_test.dfs_led_timer, jiffies + msecs_to_jiffies(DFS_LED_BLINK_INTERVAL));
}

void gpio_dfsled_timer_start(struct gpio_leds_priv *priv)
{
	//struct gpio_leds_priv *priv = (void *)data;

	printk("gpio_dfsled_timer_start del_timer_sync\n");

	del_timer_sync(&self_test.dfs_led_timer);

	if (priv == NULL) {
		printk("gpio_dfsled_timer_start data error\n");
		return;
	}

	gpio_set_dfsled(priv, DFS_LED_OFF);
	led_state = DFS_LED_OFF;

	printk("gpio_dfsled_timer_start timer_setup\n");
	timer_setup(&self_test.dfs_led_timer, gpio_dfsled_timer_act, 0);
	mod_timer(&self_test.dfs_led_timer, jiffies + msecs_to_jiffies(DFS_LED_BLINK_INTERVAL));

    return;
}

void gpio_dfsled_timer_end(struct gpio_leds_priv *priv)
{
	//struct gpio_leds_priv *priv = (void *)data;

	printk("gpio_dfsled_timer_end\n");

	if (priv == NULL) {
		printk("gpio_dfsled_timer_end data error\n");
		return;
	}

	gpio_set_dfsled(priv, DFS_LED_OFF);
	led_state = DFS_LED_OFF;

	printk("gpio_dfsled_timer_end del_timer_sync\n");
    del_timer_sync(&self_test.dfs_led_timer);
}

#if 0
static ssize_t dfsled_read(struct file *file, char __user *buf, size_t count, loff_t *pos)
{
    ssize_t ret;
    char data_with_newline[2]; // Buffer to hold the data and newline character
   
    if (*pos != 0) // If we've reached the end of the data
        return 0; // Indicate EOF

    if (count < sizeof(data_with_newline)) // Check if there's enough space in the user buffer
        return -EINVAL; // Return error if insufficient buffer space

    data_with_newline[0] = dfsled_proc_state; // Copy LED state to the buffer
    data_with_newline[1] = '\n'; // Add newline character

    ret = simple_read_from_buffer(buf, count, pos, data_with_newline, sizeof(data_with_newline));
    
    // Increment the position by the number of bytes read
    if (ret > 0)
        *pos += ret;
    
    return ret;
}
#endif

static ssize_t dfsled_write(struct file *file, const char __user *buf, size_t count, loff_t *pos)
{
//    ssize_t ret = 0;
    char input_value;
    int state;

    if (count > 0) {
        if (copy_from_user(&input_value, buf, sizeof(input_value)))
            return -EFAULT;

		state = input_value - '0';

		//printk("dfsled_write state = [%d], dfsled_proc_state=[%d]\n", state, dfsled_proc_state);

		if (state == dfsled_proc_state) {
			//Ignore this case
			return count;
		} else {
			dfsled_proc_state = state;
		}

		 if (g_priv == NULL) {
		 	printk("g_priv = NULL\n");
		 	return -EFAULT;
		 }

		if (state == 0) {
			printk("dfsled_write 0\n");
			gpio_dfsled_timer_end(g_priv);
		} else if (state == 1) {
			printk("dfsled_write 1\n");
			gpio_dfsled_timer_start(g_priv);
		} else {
			//???
		}
    }
    return count;
}

static const struct proc_ops dfsled_fops = {
    //.owner = THIS_MODULE,
	//.proc_open		= dfsled_proc_open,
	.proc_read		= seq_read,
    //.proc_read = dfsled_read,
    .proc_write = dfsled_write,
};


void gpio_set_fwUpdate(struct gpio_leds_priv *priv, int led_state)
{
	struct gpio_led_data *led1 = NULL;
	struct gpio_led_data *led2 = NULL;

	if (priv == NULL) {
		printk("gpio_set_fwUpdate data error\n");
		return;
	}

	led1 = &priv->leds[GREEN_Option];
	led2 = &priv->leds[RED_Option];

	if ((led1 == NULL) || (led2 == NULL)) {
		printk("gpio_set_fwUpdate LED error\n");
		return;
	}

	if (led_state == FWUpdate_LED_ON) {
		//printk("Turn ON LED\n");
		spin_lock_bh(&FWUpdate_LED_lock);
		gpiod_set_value(led1->gpiod, 1);
		gpiod_set_value(led2->gpiod, 1);
		spin_unlock_bh(&FWUpdate_LED_lock);
	} else if (led_state == FWUpdate_LED_OFF) {
		//printk("Turn OFF LED\n");
		spin_lock_bh(&FWUpdate_LED_lock);
		gpiod_set_value(led1->gpiod, 0);
		gpiod_set_value(led2->gpiod, 0);
		spin_unlock_bh(&FWUpdate_LED_lock);
	}
}

void gpio_fwUpdate_led_timer_act(struct timer_list *fwUpdatetimer)
{
	struct self_test_stru *st = from_timer(st, fwUpdatetimer, fwUpdate_led_timer);
	struct gpio_leds_priv *priv = st->priv;

	if (priv == NULL) {
		printk("gpio_fwUpdate_led_timer_act data error\n");
		return;
	}

	// Switching led state
	fwUpdate_led_state = (fwUpdate_led_state == FWUpdate_LED_ON) ? FWUpdate_LED_OFF : FWUpdate_LED_ON;

	gpio_set_fwUpdate(priv, fwUpdate_led_state);

	mod_timer(&self_test.fwUpdate_led_timer, jiffies + msecs_to_jiffies(FWUpdate_LED_BLINK_INTERVAL));
}


void gpio_fwUpdate_led_timer_start(struct gpio_leds_priv *priv)
{
	del_timer_sync(&self_test.fwUpdate_led_timer);

	if (priv == NULL) {
		printk("gpio_fwUpdate_led_timer_start data error\n");
		return;
	}

	//to initialize the option led as OFF
	gpio_set_fwUpdate(priv, FWUpdate_LED_OFF);
	fwUpdate_led_state = FWUpdate_LED_OFF;

	printk("gpio_fwUpdate_led_timer_start timer_setup\n");
	timer_setup(&self_test.fwUpdate_led_timer, gpio_fwUpdate_led_timer_act, 0);
	mod_timer(&self_test.fwUpdate_led_timer, jiffies + msecs_to_jiffies(FWUpdate_LED_BLINK_INTERVAL));

	return;
}

void gpio_fwUpdate_led_timer_end(struct gpio_leds_priv *priv)
{

	if (priv == NULL) {
		printk("gpio_fwUpdate_led_timer_end data error\n");
		return;
	}

	gpio_set_fwUpdate(priv, FWUpdate_LED_OFF);
	fwUpdate_led_state = FWUpdate_LED_OFF;

	printk("gpio_fwUpdate_led_timer_end del_timer_sync\n");
	del_timer_sync(&self_test.fwUpdate_led_timer);
}

static ssize_t fwUpdate_write(struct file *file, const char __user *buf, size_t count, loff_t *pos)
{
    char input_value;
    int state;

    if (count > 0) {
        if (copy_from_user(&input_value, buf, sizeof(input_value)))
            return -EFAULT;

		state = input_value - '0';

		if (g_priv == NULL) {
			printk("g_priv = NULL\n");
			return -EFAULT;
		}

		if (state == 0) {
			printk("fwUpdate_write 0\n");
			gpio_fwUpdate_led_timer_end(g_priv);
		} else if (state == 1) {
			printk("fwUpdate_write 1\n");
			gpio_fwUpdate_led_timer_start(g_priv);
		}
    }
    return count;
}

static const struct proc_ops fwUpdate_fops = {
	.proc_read	= seq_read,
	.proc_write	= fwUpdate_write,
};

static inline struct gpio_led_data *
			cdev_to_gpio_led_data(struct led_classdev *led_cdev)
{
	return container_of(led_cdev, struct gpio_led_data, cdev);
}

static void gpio_led_set(struct led_classdev *led_cdev,
	enum led_brightness value)
{
	struct gpio_led_data *led_dat = cdev_to_gpio_led_data(led_cdev);
	int level;

	if (value == LED_OFF)
		level = 0;
	else
		level = 1;

	if (led_dat->blinking) {
		led_dat->platform_gpio_blink_set(led_dat->gpiod, level,
						 NULL, NULL);
		led_dat->blinking = 0;
	} else {
		if (led_dat->can_sleep)
			gpiod_set_value_cansleep(led_dat->gpiod, level);
		else
			gpiod_set_value(led_dat->gpiod, level);
	}
}

static int gpio_led_set_blocking(struct led_classdev *led_cdev,
	enum led_brightness value)
{
	gpio_led_set(led_cdev, value);
	return 0;
}

static int gpio_blink_set(struct led_classdev *led_cdev,
	unsigned long *delay_on, unsigned long *delay_off)
{
	struct gpio_led_data *led_dat = cdev_to_gpio_led_data(led_cdev);

	led_dat->blinking = 1;
	return led_dat->platform_gpio_blink_set(led_dat->gpiod, GPIO_LED_BLINK,
						delay_on, delay_off);
}

static int create_gpio_led(const struct gpio_led *template,
	struct gpio_led_data *led_dat, struct device *parent,
	struct fwnode_handle *fwnode, gpio_blink_set_t blink_set)
{
	struct led_init_data init_data = {};
	int ret, state;

	led_dat->cdev.default_trigger = template->default_trigger;
	led_dat->can_sleep = gpiod_cansleep(led_dat->gpiod);
	if (!led_dat->can_sleep)
		led_dat->cdev.brightness_set = gpio_led_set;
	else
		led_dat->cdev.brightness_set_blocking = gpio_led_set_blocking;
	led_dat->blinking = 0;
	if (blink_set) {
		led_dat->platform_gpio_blink_set = blink_set;
		led_dat->cdev.blink_set = gpio_blink_set;
	}
	if (template->default_state == LEDS_GPIO_DEFSTATE_KEEP) {
		state = gpiod_get_value_cansleep(led_dat->gpiod);
		if (state < 0)
			return state;
	} else {
		state = (template->default_state == LEDS_GPIO_DEFSTATE_ON);
	}
	led_dat->cdev.brightness = state ? LED_FULL : LED_OFF;
	if (!template->retain_state_suspended)
		led_dat->cdev.flags |= LED_CORE_SUSPENDRESUME;
	if (template->panic_indicator)
		led_dat->cdev.flags |= LED_PANIC_INDICATOR;
	if (template->retain_state_shutdown)
		led_dat->cdev.flags |= LED_RETAIN_AT_SHUTDOWN;

	ret = gpiod_direction_output(led_dat->gpiod, state);
	if (ret < 0)
		return ret;

	if (template->name) {
		led_dat->cdev.name = template->name;
		ret = devm_led_classdev_register(parent, &led_dat->cdev);
	} else {
		init_data.fwnode = fwnode;
		ret = devm_led_classdev_register_ext(parent, &led_dat->cdev,
						     &init_data);
	}

	return ret;
}

static inline int sizeof_gpio_leds_priv(int num_leds)
{
	return sizeof(struct gpio_leds_priv) +
		(sizeof(struct gpio_led_data) * num_leds);
}

static struct gpio_leds_priv *gpio_leds_create(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct fwnode_handle *child;
	struct gpio_leds_priv *priv;
	int count, ret;

	count = device_get_child_node_count(dev);
	if (!count)
		return ERR_PTR(-ENODEV);

	priv = devm_kzalloc(dev, sizeof_gpio_leds_priv(count), GFP_KERNEL);
	if (!priv)
		return ERR_PTR(-ENOMEM);

	device_for_each_child_node(dev, child) {
		struct gpio_led_data *led_dat = &priv->leds[priv->num_leds];
		struct gpio_led led = {};
		const char *state = NULL;

		/*
		 * Acquire gpiod from DT with uninitialized label, which
		 * will be updated after LED class device is registered,
		 * Only then the final LED name is known.
		 */
		led.gpiod = devm_fwnode_get_gpiod_from_child(dev, NULL, child,
							     GPIOD_ASIS,
							     NULL);
		if (IS_ERR(led.gpiod)) {
			fwnode_handle_put(child);
			return ERR_CAST(led.gpiod);
		}

		led_dat->gpiod = led.gpiod;

		fwnode_property_read_string(child, "linux,default-trigger",
					    &led.default_trigger);

		if (!fwnode_property_read_string(child, "default-state",
						 &state)) {
			if (!strcmp(state, "keep"))
				led.default_state = LEDS_GPIO_DEFSTATE_KEEP;
			else if (!strcmp(state, "on"))
				led.default_state = LEDS_GPIO_DEFSTATE_ON;
			else
				led.default_state = LEDS_GPIO_DEFSTATE_OFF;
		}

		if (fwnode_property_present(child, "retain-state-suspended"))
			led.retain_state_suspended = 1;
		if (fwnode_property_present(child, "retain-state-shutdown"))
			led.retain_state_shutdown = 1;
		if (fwnode_property_present(child, "panic-indicator"))
			led.panic_indicator = 1;

		ret = create_gpio_led(&led, led_dat, dev, child, NULL);
		if (ret < 0) {
			fwnode_handle_put(child);
			return ERR_PTR(ret);
		}
		/* Set gpiod label to match the corresponding LED name. */
		gpiod_set_consumer_name(led_dat->gpiod,
					led_dat->cdev.dev->kobj.name);
		priv->num_leds++;
	}

	return priv;
}

static const struct of_device_id of_gpio_leds_match[] = {
	{ .compatible = "gpio-leds", },
	{},
};

MODULE_DEVICE_TABLE(of, of_gpio_leds_match);

static struct gpio_desc *gpio_led_get_gpiod(struct device *dev, int idx,
					    const struct gpio_led *template)
{
	struct gpio_desc *gpiod;
	unsigned long flags = GPIOF_OUT_INIT_LOW;
	int ret;

	/*
	 * This means the LED does not come from the device tree
	 * or ACPI, so let's try just getting it by index from the
	 * device, this will hit the board file, if any and get
	 * the GPIO from there.
	 */
	gpiod = devm_gpiod_get_index(dev, NULL, idx, flags);
	if (!IS_ERR(gpiod)) {
		gpiod_set_consumer_name(gpiod, template->name);
		return gpiod;
	}
	if (PTR_ERR(gpiod) != -ENOENT)
		return gpiod;

	/*
	 * This is the legacy code path for platform code that
	 * still uses GPIO numbers. Ultimately we would like to get
	 * rid of this block completely.
	 */

	/* skip leds that aren't available */
	if (!gpio_is_valid(template->gpio))
		return ERR_PTR(-ENOENT);

	if (template->active_low)
		flags |= GPIOF_ACTIVE_LOW;

	ret = devm_gpio_request_one(dev, template->gpio, flags,
				    template->name);
	if (ret < 0)
		return ERR_PTR(ret);

	gpiod = gpio_to_desc(template->gpio);
	if (!gpiod)
		return ERR_PTR(-EINVAL);

	return gpiod;
}

/********************************
** RAK3K<rtl8198d_engboard.dts> in node order from top to bottom.
** 2   = act green
** 4   = 5g green
** 5   = option red
** 6   = 2g green
** 9   = option green
** 10  = power green
** 11  = mesh green
*********************************/

void gpio_led_self_test_start(unsigned long data)
{
	struct platform_device *pdev = (void *)data;
	struct gpio_leds_priv *priv = NULL;
	int i = 0;

	if (pdev == NULL)
		return;

	priv = platform_get_drvdata(pdev);	

	printk("LED self test start\n");
	for (i = 0; i < priv->num_leds; i++) {
		switch (i) {
			case 2:
			case 4:
			//case 5:
			case 6:
			case 9:
			case 10:
			case 11:
				{
					struct gpio_led_data *led = &priv->leds[i];
					gpio_led_set(&led->cdev, LED_FULL);
					break;
				}
			default:
				break;
		}
	}
}

void gpio_led_self_test_end(struct timer_list *not_used)
{
	struct self_test_stru *st = from_timer(st, not_used, self_test_timer);
	struct gpio_leds_priv *priv = st->priv;
	int i = 0;

	for (i = 0; i < priv->num_leds; i++) {
		switch (i) {
			case 2:
			case 4:
			//case 5:
			case 6:
			case 9:
			case 11:

				{
					struct gpio_led_data *led = &priv->leds[i];
					//Turn OFF all LEDs, except power_green LED
					gpio_led_set(&led->cdev, LED_OFF);
					break;
				}
			default:
				break;
		}
	}
	printk("LED self test end\n");
}

static int gpio_led_probe(struct platform_device *pdev)
{
	struct gpio_led_platform_data *pdata = dev_get_platdata(&pdev->dev);
	struct gpio_leds_priv *priv;
	int i, ret = 0;

	if (pdata && pdata->num_leds) {
		priv = devm_kzalloc(&pdev->dev,
				sizeof_gpio_leds_priv(pdata->num_leds),
					GFP_KERNEL);
		if (!priv)
			return -ENOMEM;

		priv->num_leds = pdata->num_leds;
		for (i = 0; i < priv->num_leds; i++) {
			const struct gpio_led *template = &pdata->leds[i];
			struct gpio_led_data *led_dat = &priv->leds[i];

			if (template->gpiod)
				led_dat->gpiod = template->gpiod;
			else
				led_dat->gpiod =
					gpio_led_get_gpiod(&pdev->dev,
							   i, template);

			if (IS_ERR(led_dat->gpiod)) {
				dev_info(&pdev->dev, "Skipping unavailable LED gpio %d (%s)\n",
					 template->gpio, template->name);
				continue;
			}

			ret = create_gpio_led(template, led_dat,
					      &pdev->dev, NULL,
					      pdata->gpio_blink_set);
			if (ret < 0)
				return ret;
		}
	} else {
		priv = gpio_leds_create(pdev);
		if (IS_ERR(priv))
			return PTR_ERR(priv);
	}

	platform_set_drvdata(pdev, priv);

    g_priv = priv;

	gpio_led_self_test_start((unsigned long)pdev);
	self_test.priv = priv;
	timer_setup(&self_test.self_test_timer, gpio_led_self_test_end, 0);
	mod_timer(&self_test.self_test_timer, jiffies + msecs_to_jiffies(SELF_TEST_TIMEOUT));

	if (!proc_create(PROC_FILENAME, 0666, NULL, &dfsled_fops)) {
		printk(KERN_ERR "Failed to create /proc/%s\n", PROC_FILENAME);
		remove_proc_entry(PROC_FILENAME, NULL);
	}else{
		printk(KERN_INFO "/proc/%s created\n", PROC_FILENAME);
	}

	if (!proc_create(PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS, 0666, NULL, &fwUpdate_fops)) {
		printk(KERN_ERR "Failed to create /proc/%s\n", PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS);
		remove_proc_entry(PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS, NULL);
	}else{
		printk(KERN_INFO "/proc/%s created\n", PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS);
	}

	return 0;
}

static void gpio_led_shutdown(struct platform_device *pdev)
{
	struct gpio_leds_priv *priv = platform_get_drvdata(pdev);
	int i;

	for (i = 0; i < priv->num_leds; i++) {
		struct gpio_led_data *led = &priv->leds[i];

		if (!(led->cdev.flags & LED_RETAIN_AT_SHUTDOWN))
			gpio_led_set(&led->cdev, LED_OFF);
	}

    remove_proc_entry(PROC_FILENAME, NULL);
    printk(KERN_INFO "/proc/%s removed\n", PROC_FILENAME);
    remove_proc_entry(PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS, NULL);
    printk(KERN_INFO "/proc/%s removed\n", PROC_FILENAME_FOR_FW_UPDATE_BY_HW_BTNS);

}

static struct platform_driver gpio_led_driver = {
	.probe		= gpio_led_probe,
	.shutdown	= gpio_led_shutdown,
	.driver		= {
		.name	= "leds-gpio",
		.of_match_table = of_gpio_leds_match,
	},
};

module_platform_driver(gpio_led_driver);

MODULE_AUTHOR("Raphael Assenat <raph@8d.com>, Trent Piepho <tpiepho@freescale.com>");
MODULE_DESCRIPTION("GPIO LED driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:leds-gpio");
