/*
 * Copyright (C) 2017 Realtek Semiconductor Corp.
 * All Rights Reserved.
 *
 * This program is the proprietary software of Realtek Semiconductor
 * Corporation and/or its licensors, and only be used, duplicated,
 * modified or distributed under the authorized license from Realtek.
 *
 * ANY USE OF THE SOFTWARE OTHER THAN AS AUTHORIZED UNDER
 * THIS LICENSE OR COPYRIGHT LAW IS PROHIBITED.
*/


#include <linux/if_pppox.h>
#include <net/ip.h>

#include "rtk_fc_dualHeader.h"
#include "rtk_fc_internal.h"
#include "rtk_fc_callback.h"
#include "rtk_fc_debug.h"
#include "rtk_fc_mappingAPI.h"
#include "rtk_fc_external.h"
#include "rtk_fc_vxlan.h"
#include "rtk_fc_acl.h"

#define EXTRATAG_BUFFER_OFFSET 	64	// offset per each content extra tag action list
#define EXTRATAG_LEN				40	// outer header length for PPTP/L2TP/DSLite
#define V6RD_EXTRATAG_LEN			20	// outer header length for 6RD

static int _rtk_fc_fragIPID_save(int hwNetifIdx, rtk_fc_hwTableNetif_t *egressIntf, uint16 oriIPID, uint16 newIPID)
{
	int i = 0, freeEntIdx = -1, hitEntIdx = -1, lruEntIdx = -1;
	uint16 lruIPID = 65535;

	for(i = 0; i < MAX_SW_FRAGIPID_SIZE; i++){
		if((freeEntIdx==-1)&& (!egressIntf->psFragIPID[i].valid))
			freeEntIdx = i;

		if((egressIntf->psFragIPID[i].valid) && (egressIntf->psFragIPID[i].oriIPID == oriIPID)){
			hitEntIdx = i;
			break;
		}

		if((egressIntf->psFragIPID[i].valid) && (lruIPID>egressIntf->psFragIPID[i].oriIPID)){
			lruEntIdx = i;
			lruIPID = egressIntf->psFragIPID[i].oriIPID;
		}

	}

	if(hitEntIdx!=-1){
		TRACE("netif[%d] duplicate ipid was found, entryidx: %d, IPID: %d, new IPID %d", hwNetifIdx, hitEntIdx, egressIntf->psFragIPID[hitEntIdx].oriIPID, newIPID);
	}else if(freeEntIdx!=-1){
		hitEntIdx = freeEntIdx;
		TRACE("netif[%d] new frag packet, entryidx: %d, IPID: %d, new IPID %d", hwNetifIdx, hitEntIdx, egressIntf->psFragIPID[hitEntIdx].oriIPID, newIPID);
	}else{
		hitEntIdx = lruEntIdx;
		TRACE("netif[%d] ipid records are full, replace entryidx %d, IPID: %d, new IPID %d", hwNetifIdx, lruEntIdx, egressIntf->psFragIPID[hitEntIdx].oriIPID, newIPID);
	}

	egressIntf->psFragIPID[hitEntIdx].valid = TRUE;
	egressIntf->psFragIPID[hitEntIdx].oriIPID = oriIPID;
	egressIntf->psFragIPID[hitEntIdx].newIPID = newIPID;

	return SUCCESS;
}


static int _rtk_fc_fragIPID_delete(int hwNetifIdx, rtk_fc_hwTableNetif_t *egressIntf, uint16 oriIPID, uint16 *newIPID)
{
	int i = 0, hitEntIdx = -1;

	for(i = 0; i < MAX_SW_FRAGIPID_SIZE; i++){
		if((egressIntf->psFragIPID[i].valid) && (egressIntf->psFragIPID[i].oriIPID == oriIPID)){
			hitEntIdx = i;
			break;
		}
	}

	if(hitEntIdx!=-1){
		TRACE("netif[%d] duplicate ipid was found, idx: %d, IPID: %d", hwNetifIdx, hitEntIdx, egressIntf->psFragIPID[hitEntIdx].oriIPID);
	}else{
		TRACE("netif[%d] ipid records was not found, IPID: %d", hwNetifIdx, oriIPID);
		return FAILED;
	}

	*newIPID = egressIntf->psFragIPID[hitEntIdx].newIPID;

	egressIntf->psFragIPID[hitEntIdx].valid = FALSE;
	egressIntf->psFragIPID[hitEntIdx].oriIPID = 0;
	egressIntf->psFragIPID[hitEntIdx].newIPID = 0;

	return SUCCESS;

}

static int _rtk_fc_fragIPID_get(int hwNetifIdx, rtk_fc_hwTableNetif_t *egressIntf, uint16 oriIPID, uint16 *newIPID)
{
	int i = 0, hitEntIdx = -1;

	for(i = 0; i < MAX_SW_FRAGIPID_SIZE; i++){
		if((egressIntf->psFragIPID[i].valid) && (egressIntf->psFragIPID[i].oriIPID == oriIPID)){
			hitEntIdx = i;
			break;
		}
	}

	if(hitEntIdx!=-1){
		TRACE("netif[%d] duplicate ipid was found, idx: %d, IPID: %d", hwNetifIdx, hitEntIdx, egressIntf->psFragIPID[hitEntIdx].oriIPID);
	}else{
		TRACE("netif[%d] ipid records was not found, IPID: %d", hwNetifIdx, oriIPID);
		return FAILED;
	}

	*newIPID = egressIntf->psFragIPID[hitEntIdx].newIPID;

	return SUCCESS;
}

static int _rtk_fc_extraTagActionList_empty_find(void)
{
	/* return 1~7 for action list idx */
	int i = 0;
	for(i = 0; i < RTK_FC_TABLESIZE_EXTRATAG; i++)
	{
		if(fc_db.extraTagActionList_netifIdx[i] == -1)
			return i+1;
	}

	WARNING("NO empty action list could be used!");
	return 0;
}

static int _rtk_fc_extraTagActionList_apply(int actionListIdx, int intfIdx)
{
	FC_PARAM_CHK(((actionListIdx<RTK_FC_TABLESIZE_EXTRATAG_LISTMIN) || (actionListIdx>RTK_FC_TABLESIZE_EXTRATAG_LISTMAX)), FAIL);

	if(fc_db.extraTagActionList_netifIdx[actionListIdx-1] != -1)
		DEBUG("Try to replace action list, used by intf %d, new intf: %d",fc_db.extraTagActionList_netifIdx[actionListIdx-1], intfIdx);

	fc_db.extraTagActionList_netifIdx[actionListIdx-1] = intfIdx;

	return SUCCESS;
}

static int _rtk_fc_extraTagActionList_clear(int actionListIdx)
{
	if((actionListIdx<RTK_FC_TABLESIZE_EXTRATAG_LISTMIN) || (actionListIdx>RTK_FC_TABLESIZE_EXTRATAG_LISTMAX))
		return FAIL;

	fc_db.extraTagActionList_netifIdx[actionListIdx-1] = -1;

	return SUCCESS;
}

static int _rtk_fc_extraTag_setupPPTP(int wan_intf_idx, uint8 *outerHdr, rtk_fc_tunnelInfo_t *tunnelInfo, int *extraTagActListIdx)
{
	rtk_rg_asic_extraTagAction_t extraTagAction;
	int extraTagIdx = 0, actSeq = 0;
	int TUNN_tag_len=EXTRATAG_LEN;	//IP(20)+GRE(16)+PPP(4) for PPTP

	//correct GRE header length here
	if((tunnelInfo->tunnelTag&GRESEQ_TAGIF)==0)TUNN_tag_len-=4;
	if((tunnelInfo->tunnelTag&GREACK_TAGIF)==0)TUNN_tag_len-=4;
	TUNN_tag_len-=4;
	TUNN_tag_len+=tunnelInfo->pppMode;

	if(*extraTagActListIdx==0)
	{
		// need to get one available ActionList
		extraTagIdx = _rtk_fc_extraTagActionList_empty_find();
		if(extraTagIdx==0) return FAIL;
	}else{
		// use the force assigned ActionList
		extraTagIdx = *extraTagActListIdx;
	}

	_rtk_fc_extraTagActionList_apply(extraTagIdx, wan_intf_idx);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type1.act_bit = FB_EXTG_ACTBIT_1;
	extraTagAction.type1.length = TUNN_tag_len;
	extraTagAction.type1.src_addr_offset = 0 + (EXTRATAG_BUFFER_OFFSET * (extraTagIdx-1));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type2.act_bit = FB_EXTG_ACTBIT_2;
	extraTagAction.type2.ethertype = 0x0800;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 2;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = TUNN_tag_len;
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type4.act_bit = FB_EXTG_ACTBIT_4;
	extraTagAction.type4.pkt_buff_offset = 4;
	extraTagAction.type4.data_src_type = 1;						// 1:IP ID;
	extraTagAction.type4.seq_ack_reg_idx = wan_intf_idx;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type5.act_bit = FB_EXTG_ACTBIT_5;
	extraTagAction.type6.pkt_buff_offset = 10;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 24;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = tunnelInfo->pppMode;
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type4.act_bit = FB_EXTG_ACTBIT_4;
	extraTagAction.type4.pkt_buff_offset = 28;
	extraTagAction.type4.data_src_type = 0;
	extraTagAction.type4.reduce_seq = !((tunnelInfo->tunnelTag&GRESEQ_TAGIF)>0);
	extraTagAction.type4.reduce_ack = !((tunnelInfo->tunnelTag&GREACK_TAGIF)>0);
	extraTagAction.type4.seq_ack_reg_idx = wan_intf_idx;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - Before");
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGCONTENTBUFFER_SET(extraTagIdx, outerHdr), SUCCESS);
	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - After");

	*extraTagActListIdx = extraTagIdx;

	return SUCCESS;
}

static int _rtk_fc_extraTag_setupL2TP(int wan_intf_idx, uint8 *outerHdr, int *extraTagActListIdx)
{
	rtk_rg_asic_extraTagAction_t extraTagAction;
	int extraTagIdx = 0, actSeq = 0;
	int TUNN_tag_len=EXTRATAG_LEN;	//IP(20)+UDP(8)+L2TP(8)+PPP(4) for L2TP

	if(*extraTagActListIdx==0)
	{
		// need to get one available ActionList
		extraTagIdx = _rtk_fc_extraTagActionList_empty_find();
		if(extraTagIdx==0) return FAIL;
	}else{
		// use the force assigned ActionList
		extraTagIdx = *extraTagActListIdx;
	}

	_rtk_fc_extraTagActionList_apply(extraTagIdx, wan_intf_idx);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type1.act_bit = FB_EXTG_ACTBIT_1;
	extraTagAction.type1.length = TUNN_tag_len;
	extraTagAction.type1.src_addr_offset = 0 + (EXTRATAG_BUFFER_OFFSET * (extraTagIdx-1));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type2.act_bit = FB_EXTG_ACTBIT_2;
	extraTagAction.type2.ethertype = 0x0800;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 2;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = TUNN_tag_len;
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type4.act_bit = FB_EXTG_ACTBIT_4;
	extraTagAction.type4.pkt_buff_offset = 4;
	extraTagAction.type4.data_src_type = 1;						// 1:IP ID;
	extraTagAction.type4.seq_ack_reg_idx = wan_intf_idx;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type5.act_bit = FB_EXTG_ACTBIT_5;
	extraTagAction.type6.pkt_buff_offset = 10;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 24;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = TUNN_tag_len-20;		// 20: ipv4 header length
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 30;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = TUNN_tag_len-20-8;		//20: ipv4 header length; 8: udp header length
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type6.act_bit = FB_EXTG_ACTBIT_6;
	extraTagAction.type6.pkt_buff_offset = 26;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);


	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - Before");
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGCONTENTBUFFER_SET(extraTagIdx, outerHdr), SUCCESS);
	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - After");

	*extraTagActListIdx = extraTagIdx;

	return SUCCESS;
}

static int _rtk_fc_extraTag_setupDsLite(int wan_intf_idx, uint8 *outerHdr, int *extraTagActListIdx)
{
	rtk_rg_asic_extraTagAction_t extraTagAction;
	int extraTagIdx = 0, actSeq = 0;
	int TUNN_tag_len=EXTRATAG_LEN;	//IPv6(40)

	if(*extraTagActListIdx==0)
	{
		// need to get one available ActionList
		extraTagIdx = _rtk_fc_extraTagActionList_empty_find();
		if(extraTagIdx==0) return FAIL;
	}else{
		// use the force assigned ActionList
		extraTagIdx = *extraTagActListIdx;
	}

	_rtk_fc_extraTagActionList_apply(extraTagIdx, wan_intf_idx);

	//insert 40 length
	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type1.act_bit = FB_EXTG_ACTBIT_1;
	extraTagAction.type1.length = TUNN_tag_len;
	extraTagAction.type1.src_addr_offset = 0 + (EXTRATAG_BUFFER_OFFSET * (extraTagIdx-1));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	//change ethertype to 0x86dd
	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type2.act_bit = FB_EXTG_ACTBIT_2;
	extraTagAction.type2.ethertype = 0x86dd;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	//
	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 4;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = 0;
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - Before");
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGCONTENTBUFFER_SET(extraTagIdx, outerHdr), SUCCESS);
	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - After");

	*extraTagActListIdx = extraTagIdx;

	return SUCCESS;
}

static int _rtk_fc_extraTag_setup6RD(int wan_intf_idx, uint8 *outerHdr, int *extraTagActListIdx)
{
	rtk_rg_asic_extraTagAction_t extraTagAction;
	int extraTagIdx = 0, actSeq = 0;
	int TUNN_tag_len=V6RD_EXTRATAG_LEN;	//IPv4(20)

	if(*extraTagActListIdx==0)
	{
		// need to get one available ActionList
		extraTagIdx = _rtk_fc_extraTagActionList_empty_find();
		if(extraTagIdx==0) return FAIL;
	}else{
		// use the force assigned ActionList
		extraTagIdx = *extraTagActListIdx;
	}

	_rtk_fc_extraTagActionList_apply(extraTagIdx, wan_intf_idx);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type1.act_bit = FB_EXTG_ACTBIT_1;
	extraTagAction.type1.length = TUNN_tag_len;
	extraTagAction.type1.src_addr_offset = 0 + (EXTRATAG_BUFFER_OFFSET * (extraTagIdx-1));
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type2.act_bit = FB_EXTG_ACTBIT_2;
	extraTagAction.type2.ethertype = 0x0800;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type3.act_bit = FB_EXTG_ACTBIT_3;
	extraTagAction.type3.pkt_buff_offset = 2;
	extraTagAction.type3.length = 2;
	extraTagAction.type3.value = 20; //IPv4 header length
	extraTagAction.type3.operation = 0;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	bzero(&extraTagAction, sizeof(rtk_rg_asic_extraTagAction_t));
	extraTagAction.type4.act_bit = FB_EXTG_ACTBIT_4;
	extraTagAction.type4.pkt_buff_offset = 4;
	extraTagAction.type4.data_src_type = 1;						// 1:IP ID;
	extraTagAction.type4.seq_ack_reg_idx = wan_intf_idx;
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_ADD(extraTagIdx, actSeq++, &extraTagAction), SUCCESS);

	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - Before");
	ASSERT_EQ(RTK_RG_ASIC_EXTRATAGCONTENTBUFFER_SET(extraTagIdx, outerHdr), SUCCESS);
	//memDump(outerHdr, TUNN_tag_len, "Content Buffer - After");

	*extraTagActListIdx = extraTagIdx;

	return SUCCESS;
}

#if defined(CONFIG_FC_RTL8277C_SERIES)
/* Setup ingress ACL for 6RD downstream*/
int rtk_fc_8277C_v6rd_downStream_reservedACL_add(int igrIntfIdx, rtk_fc_ingress_data_t *pFcIngressData, int *result_idx)
{
	rtk_fc_aclAndCf_reserved_dual_hdr_chk_t dual_hdr_chk = {0};
	int ret;
	// set downstream cls for ingress interface
	dual_hdr_chk.packet_type = RTK_FC_CLS_DUAL_6RD;
	dual_hdr_chk.igr_intf_idx		= igrIntfIdx;
	dual_hdr_chk.hash_tuple_idx = RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;
	dual_hdr_chk.six_rd.src_ip = pFcIngressData->srcIp;
	dual_hdr_chk.six_rd.dst_ip = pFcIngressData->dstIp;

	ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
	if(ret != RTK_FC_RET_OK)
	{
		DEBUG("add special ACL DUAL_HEADER_DS_CHECK failed!!! ret = %x",ret);
		return RTK_FC_RET_ERR;
	}
	else
	{
		DEBUG("add special ACL DUAL_HEADER_DS_CHECK success, idx: 0x%x",dual_hdr_chk.rslt_idx);
		//_rtk_ca_cls_rule_dump(dual_hdr_chk.rslt_idx, 0, 0-RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, FALSE);
		*result_idx = dual_hdr_chk.rslt_idx;
	}

	return RTK_FC_RET_OK;
}

int _rtk_fc_dualHeader_ctrl_setupPPTP(int dualControlIdx, rtk_fc_pktHdr_t *pPktHdr)
{
	int32 ret=SUCCESS;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;
	uint32 gre_len=8, ppp_len=0;

	//set dual hdr control for PPTP
	memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));

	if(pPktHdr->tunnelInfo.tunnelTag&GRESEQ_TAGIF){
		gre_len+=4;
		dual_ctrl_entry.dual_ctrl_gre_mod_fld += 1;	// only seq
	}
	if(pPktHdr->tunnelInfo.tunnelTag&GREACK_TAGIF){
		gre_len+=4;
		dual_ctrl_entry.dual_ctrl_gre_mod_fld += 2;	// only ack || both seq/ack
	}

	ppp_len=pPktHdr->tunnelInfo.pppMode;

	dual_ctrl_entry.dual_ctrl_hdr_fmt = 0;
	if(pPktHdr->outer_iph){
		//v4
		dual_ctrl_entry.dual_ctrl_hdr_len 				= 20+gre_len+ppp_len; 	// IPv4(20)+GRE(8~16)+PPP(1~4)

		//FIXME: because egress netif always has small MTU already, therefore we do not use this diff value to adjust MTU comparison.
		//dual_ctrl_entry.dual_ctrl_mtu_diff 			= 20+gre_len+ppp_len; 	// IPv4(20)+GRE(8~16)+PPP(1~4)

		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost 		= 28;	// IPv4(20)+GRE seq offset(8)

		dual_ctrl_entry.dual_ctrl_rep_ip_len 			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff 		= 20+gre_len+ppp_len;	// IPv4(20)+GRE(8~16)+PPP(1~4)

		dual_ctrl_entry.dual_ctrl_rep_len_ost1 			= 24;	// IPv4(20)+GRE length field offset(4)
		dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= ppp_len;	// PPP(1~4)

		dual_ctrl_entry.dual_ctrl_rep_len_ost2 			= 127;	// X
		dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 0;	// X

		dual_ctrl_entry.dual_ctrl_ip4_ost 				= 0;
		dual_ctrl_entry.dual_ctrl_ip6_oft 				= 127;

		dual_ctrl_entry.dual_ctrl_rep_ip_en				= 1;	// replace IP version
		dual_ctrl_entry.dual_ctrl_rep_ip_ver 			= 0;	// IPv4
	}else{
		//v6
		dual_ctrl_entry.dual_ctrl_hdr_len 				= 40+gre_len+ppp_len; 	// IPv6(40)+GRE(8~16)+PPP(1~4)

		//FIXME: because egress netif always has small MTU already, therefore we do not use this diff value to adjust MTU comparison.
		//dual_ctrl_entry.dual_ctrl_mtu_diff 			= 40+gre_len+ppp_len; 	// IPv6(40)+GRE(8~16)+PPP(1~4)

		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost 		= 48;	// IPv6(40)+GRE seq offset(8)

		dual_ctrl_entry.dual_ctrl_rep_ip_len 			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff 		= gre_len+ppp_len;	// GRE(8~16)+PPP(1~4)

		dual_ctrl_entry.dual_ctrl_rep_len_ost1 			= 44;	// IPv6(40)+GRE length field offset(4)
		dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= ppp_len;	// PPP(1~4)

		dual_ctrl_entry.dual_ctrl_rep_len_ost2 			= 127;	// X
		dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 0;	// X

		dual_ctrl_entry.dual_ctrl_ip4_ost 				= 127;
		dual_ctrl_entry.dual_ctrl_ip6_oft 				= 0;

		dual_ctrl_entry.dual_ctrl_rep_ip_en				= 1;	// replace IP version
		dual_ctrl_entry.dual_ctrl_rep_ip_ver 			= 1;	// IPv6
	}
	//dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en 		= (TOS_hash_replace_ctrl&PPTP_TOS_REPLACE_DSCP_MASK)>0;
	//dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= (TOS_hash_replace_ctrl&PPTP_TOS_REPLACE_ECN_MASK)>0;
	//dual_ctrl_entry.dual_ctrl_rep_ip_ptc 			= 0;
	//dual_ctrl_entry.dual_ctrl_rep_ip_ttl 			= 0;
	//dual_ctrl_entry.dual_ctrl_mapt_pre_len 		= 0;
	//dual_ctrl_entry.dual_ctrl_map_drf_ver 		= 0;
	//dual_ctrl_entry.dual_ctrl_6rd_dip_pre_len 	= 0;
	//dual_ctrl_entry.dual_ctrl_6rd_dip_suf_str_bit = 0;
	//dual_ctrl_entry.dual_ctrl_dsl_uc_chk 			= 0;
	//dual_ctrl_entry.dual_ctrl_dsl_mc_chk 			= 0;
	//dual_ctrl_entry.dual_ctrl_l4sum_rec 			= 0;
	dual_ctrl_entry.dual_ctrl_vxlan_sp_ost 			= 127;
	dual_ctrl_entry.dual_ctrl_inr_to_out_fld 		= 0xb8;	// bit [3,4,5,7]
	dual_ctrl_entry.dual_ctrl_tos_hsh_sel 			= 0;	//hashed by outer TOS(1) or by inner TOS(0)
	dual_ctrl_entry.dual_ctrl_mc_mac_map 			= 1;
	dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= dual_ctrl_entry.dual_ctrl_hdr_len-1;		//modify ppp ip version based on inner packet ip version

	ret = aal_pe_dual_control_entry_add(dualControlIdx, &dual_ctrl_entry);

	return ret;
}

int _rtk_fc_dualHeader_ctrl_setupL2TP(int dualControlIdx, rtk_fc_pktHdr_t *pPktHdr)
{
	int32 ret=SUCCESS;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;
	uint32 l2tpHdrLen=8, pppLen=pPktHdr->tunnelInfo.pppMode;
	uint32 l2tp_length_bit=1, outer_isIpv6=(pPktHdr->outer_iph)?0:1;

	//set dual hdr control for L2TP
	{
		memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));
		dual_ctrl_entry.dual_ctrl_hdr_fmt 				= 0;
		dual_ctrl_entry.dual_ctrl_hdr_len 				= (outer_isIpv6)?(40+8+l2tpHdrLen+pppLen):(20+8+l2tpHdrLen+pppLen);	// IP header(20/40)+UDP(8)+L2TP(8)+PPP(4)
		dual_ctrl_entry.dual_ctrl_mtu_diff 				= 0; //(outer_isIpv6)?(40+8+l2tpHdrLen+pppLen):(20+8+l2tpHdrLen+pppLen);	// IP header(20/40)+UDP(8)+L2TP(8)+PPP(4)
		dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= dual_ctrl_entry.dual_ctrl_hdr_len - 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_en				= TRUE;
		dual_ctrl_entry.dual_ctrl_rep_ip_ver 			= outer_isIpv6;							// IPv4 or IPv6
		dual_ctrl_entry.dual_ctrl_rep_len_ost1 			= (outer_isIpv6)?(40+4):(20+4);			// IP header(20/40)+UDP length field offset(4)
		dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= 8+l2tpHdrLen+pppLen;					// UDP(8)+L2TP(8)+PPP(4)
		if(l2tp_length_bit)
		{
			dual_ctrl_entry.dual_ctrl_rep_len_ost2 		= (outer_isIpv6)?(40+8+2):(20+8+2);		// IP header(20/40)+UDP(8)+L2TP length field offset(2)
			dual_ctrl_entry.dual_ctrl_rep_len_diff2 	= l2tpHdrLen+pppLen;					// L2TP(8)+PPP(4)
		}
		else
		{
			dual_ctrl_entry.dual_ctrl_rep_len_ost2		= 127;
		}
		dual_ctrl_entry.dual_ctrl_ip4_ost 				= (outer_isIpv6)?127:0;
		dual_ctrl_entry.dual_ctrl_ip6_oft 				= (outer_isIpv6)?0:127;
		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost 		= 127;
		//dual_ctrl_entry.dual_ctrl_gre_mod_fld 		= 0;
		dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en 		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_len 			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff 		= (outer_isIpv6)?(8+l2tpHdrLen+pppLen):(20+8+l2tpHdrLen+pppLen);	// IP header(20/40)+UDP(8)+L2TP(8)+PPP(4)
		//dual_ctrl_entry.dual_ctrl_rep_ip_ptc 			= 0;
		//dual_ctrl_entry.dual_ctrl_rep_ip_ttl 			= 0;
		//dual_ctrl_entry.dual_ctrl_mapt_pre_len 		= 0;
		//dual_ctrl_entry.dual_ctrl_map_drf_ver 		= 0;
		//dual_ctrl_entry.dual_ctrl_6rd_dip_pre_len 	= 0;
		//dual_ctrl_entry.dual_ctrl_6rd_dip_suf_str_bit = 0;
		//dual_ctrl_entry.dual_ctrl_dsl_uc_chk 			= 0;
		//dual_ctrl_entry.dual_ctrl_dsl_mc_chk 			= 0;
		dual_ctrl_entry.dual_ctrl_l4sum_rec 			= 0;
		dual_ctrl_entry.dual_ctrl_vxlan_sp_ost 			= 127;
		dual_ctrl_entry.dual_ctrl_inr_to_out_fld 		= 0xf8;	// bit [3,4,5,6,7]
		dual_ctrl_entry.dual_ctrl_tos_hsh_sel 			= 0;	//hashed by outer TOS(1) or by inner TOS(0)
		dual_ctrl_entry.dual_ctrl_mc_mac_map 			= 1;

		ret = aal_pe_dual_control_entry_add(dualControlIdx, &dual_ctrl_entry);
		if(ret)
			WARNING("Add dual control failed, ret = %d", ret);
	}

	return ret;
}


int _rtk_fc_dualHeader_ctrl_setupMAPT(int dualControlIdx, uint32 mapt_draft_version, uint32 dmr_prefix_len)
{
	int32 ret=SUCCESS;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;
	
	//set dual hdr control for mapT
	memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));
	dual_ctrl_entry.dual_ctrl_hdr_fmt				= 2;
	dual_ctrl_entry.dual_ctrl_hdr_len				= 40;	// IPv6(40)
	dual_ctrl_entry.dual_ctrl_mtu_diff				= 40;	// IPv6(40)
	dual_ctrl_entry.dual_ctrl_rep_ip_en 			= 1;	// replace IP version
	dual_ctrl_entry.dual_ctrl_rep_ip_ver			= 1;	// IPv6
	dual_ctrl_entry.dual_ctrl_rep_len_ost1			= 127;	// X
	dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= 0;	// X
	dual_ctrl_entry.dual_ctrl_rep_len_ost2			= 127;	// X
	dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 0;	// X
	dual_ctrl_entry.dual_ctrl_ip4_ost				= 127;
	dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost		= 127;	// X
	dual_ctrl_entry.dual_ctrl_gre_mod_fld			= 0;	// X
	dual_ctrl_entry.dual_ctrl_ip6_oft				= 0;
	dual_ctrl_entry.dual_ctrl_rep_ip_len			= 1;
	dual_ctrl_entry.dual_ctrl_rep_ip_len_diff		= 0;
	//dual_ctrl_entry.dual_ctrl_rep_ip_len_diff_pad_ctrl=(ip_len_ctrl_disable==0);
	//dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en		= (TOS_hash_replace_ctrl&MAPT_TOS_REPLACE_DSCP_MASK)>0;
	//dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= (TOS_hash_replace_ctrl&MAPT_TOS_REPLACE_ECN_MASK)>0;
	dual_ctrl_entry.dual_ctrl_rep_ip_ptc			= 1;
	dual_ctrl_entry.dual_ctrl_rep_ip_ttl			= 1;
	switch(dmr_prefix_len){
		case 32:
		case 40:
		case 48:
		case 56:
		case 64:
			dual_ctrl_entry.dual_ctrl_mapt_pre_len	= (dmr_prefix_len>>3)-3;
			break;
		case 96:
			dual_ctrl_entry.dual_ctrl_mapt_pre_len	= (dmr_prefix_len>>3)-6;
			break;
		default:
			break;
	}
	dual_ctrl_entry.dual_ctrl_map_drf_ver			= mapt_draft_version;
	//dual_ctrl_entry.dual_ctrl_6rd_dip_pre_len 	= 0;
	//dual_ctrl_entry.dual_ctrl_6rd_dip_suf_str_bit = 0;
	//dual_ctrl_entry.dual_ctrl_dsl_uc_chk			= 0;
	//dual_ctrl_entry.dual_ctrl_dsl_mc_chk			= 0;
	dual_ctrl_entry.dual_ctrl_l4sum_rec 			= 1;	// pop IPv4, recalculate L4 checksum by IPV6 pseudo header
	dual_ctrl_entry.dual_ctrl_vxlan_sp_ost			= 127;
	dual_ctrl_entry.dual_ctrl_inr_to_out_fld		= 0x88; // bit [3,7]
	dual_ctrl_entry.dual_ctrl_tos_hsh_sel			= 0; 	//enable or disable won't make any difference
	dual_ctrl_entry.dual_ctrl_mc_mac_map			= 1;
	dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= 127;	//disable
	ret = aal_pe_dual_control_entry_add(dualControlIdx, &dual_ctrl_entry);
	if(ret)
		WARNING("Add dual control failed, ret = %d", ret);

	return ret;
}


int _rtk_fc_dualHeader_ctrl_setupDsliteMape(int dualControlIdx, rtk_fc_pktHdr_t *pPktHdr)
{
	int32 ret=SUCCESS;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;

	//set dual hdr control for Dslite/Mape
	{
		memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));
		dual_ctrl_entry.dual_ctrl_hdr_fmt				= 0;
		dual_ctrl_entry.dual_ctrl_hdr_len				= 40;	// IPv6 Hdr Len(40)
		dual_ctrl_entry.dual_ctrl_mtu_diff				= 40;	// IPv6 Hdr Len(40)
		dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= 127;
		dual_ctrl_entry.dual_ctrl_rep_ip_en 			= 1;	//chnage L2 to v6 0x86dd/0x0057
		dual_ctrl_entry.dual_ctrl_rep_ip_ver			= 1;	//chnage L2 to v6 0x86dd/0x0057
		dual_ctrl_entry.dual_ctrl_rep_len_ost1			= 127;	
		//dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= 0;
		dual_ctrl_entry.dual_ctrl_rep_len_ost2			= 127;	
		//dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 0;	
		dual_ctrl_entry.dual_ctrl_ip4_ost				= 127;
		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost		= 127;
		dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_len			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff		= 0;	// IPv6 Hdr Len(40)
		dual_ctrl_entry.dual_ctrl_rep_ip_ttl			= FALSE;
		dual_ctrl_entry.dual_ctrl_dsl_uc_chk			= FALSE;
		dual_ctrl_entry.dual_ctrl_dsl_mc_chk			= TRUE;
		dual_ctrl_entry.dual_ctrl_vxlan_sp_ost			= 127;
		/*0:L2 1:Vlan 2:802.1p 3:l2+pppoe 4 L3 5:DSCP 6:L4 7:ehtertype+pppoe_enc*/
		dual_ctrl_entry.dual_ctrl_inr_to_out_fld		= 0x30; // bit [3,4]
		dual_ctrl_entry.dual_ctrl_tos_hsh_sel			= 0;
		dual_ctrl_entry.dual_ctrl_mc_mac_map			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff_pad_ctrl = 0;

		ret = aal_pe_dual_control_entry_add(dualControlIdx, &dual_ctrl_entry);
		if(ret)
			WARNING("Add dual control failed, ret = %d", ret);
	}


	return ret;
}



#endif

static int _rtk_fc_flow_setupPPTP_by_egrPktHdr(rtk_fc_pktHdr_t *pPktHdr, int wan_intf_idx, int gmacL2Idx)
{
	rtk_rg_err_code_t ret;
	rtk_fc_tableFlowEntry_t flowPathEntry;
	rtk_rg_asic_path6_entry_t *flowPath6 = &flowPathEntry.path6;
	rtk_fc_hwTableNetif_t *pNetif = &(fc_db.netifHwTbl[wan_intf_idx]);
	uint32 flow_idx;
	int32 flowHashIdx = 0, extraTagActListIdx = 0;
	uint8 addSwOnly = FALSE;
	rtk_fc_pmap_t portmap;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;


	TRACE("[Dual] prepare PPTP tag");

	//1 1: Add flow extra tag
	{
		// Generate IPv4+GRE+PPP header
		u8 contenBuffer[64]={0};
		u8 *outerHdr = &contenBuffer[0];
		u8 off;
		u32 flag;
		TRACE("[Dual] Gen outer header and add to SRAM content buffer");

		// Prepare IP(20)+GRE(16)+PPP(4) for PPTP
		// L3 header --
		{
			*((uint16 *)(outerHdr + 0)) = htons(0x4500);	// ver, IHL
			//*((uint16 *)(outerHdr + 2)) = htons(0x0);	// total length	: updated by extra tag actions
			//*((uint16 *)(outerHdr + 4)) = htons(0x0);	// ipid		: updated by extra tag actions
			//*((uint16 *)(outerHdr + 6)) = htons(0x0);	// flags, frag offset
			*((uint16 *)(outerHdr + 8)) = htons(0x402f);	// ttl, protocol
			//*((uint16 *)(outerHdr + 10)) = htons(0x0);	// hdr checksum
			*((uint32 *)(outerHdr + 12)) = pPktHdr->outer_iph->saddr;	// src ip
			*((uint32 *)(outerHdr + 16)) = pPktHdr->outer_iph->daddr;	// dst ip
		}

		// GRE --
		flag=0x2001880b;
		//*(uint16 *)(outerHdr + 24) = htons(0x0);									//Payload length
		*(uint16 *)(outerHdr + 26) = htons(pPktHdr->tunnelInfo.pptp.grecallid);			//Peer CallID
		//*(uint32 *)(outerHdr + 28) = htonl(0x0);										//Sequence Num
		//*(uint32 *)(outerHdr + 32) = htonl(0x0);										//Acknowledgment Num
		off=28;
		if(pPktHdr->tunnelInfo.tunnelTag&GRESEQ_TAGIF){
			flag|=0x10000000;
			off+=4;
		}
		if(pPktHdr->tunnelInfo.tunnelTag&GREACK_TAGIF){
			flag|=0x00800000;
			off+=4;
		}
		*(uint32 *)(outerHdr + 20) = htonl(flag);									//Flags, version, Protocol Type=PPP

		// PPP --
		{
			switch(pPktHdr->tunnelInfo.pppMode){
				case RTK_FC_COMP_PPP_FF0300XX:
					*(uint32 *)(outerHdr + off) = htonl(0xff030021);							// Address, Control, Protocol=IPv4
					break;
				case RTK_FC_COMP_PPP_FF03XX:
					*(uint32 *)(outerHdr + off) = htonl(0xff032100);							// Address, Control, Protocol=IPv4
					break;
				case RTK_FC_COMP_PPP_00XX:
					*(uint32 *)(outerHdr + off) = htonl(0x00210000);							// Address, Control, Protocol=IPv4
					break;
				case RTK_FC_COMP_PPP_XX:
					*(uint32 *)(outerHdr + off) = htonl(0x21000000);							// Address, Control, Protocol=IPv4
					break;
				default:
					break;
			}
		}

		extraTagActListIdx = pNetif->outerHdrExtratagIdx;	// use the previous used list if readd.
		if((ret = _rtk_fc_extraTag_setupPPTP(wan_intf_idx, outerHdr, &(pPktHdr->tunnelInfo), &extraTagActListIdx)) == SUCCESS)
			pNetif->outerHdrExtratagIdx = extraTagActListIdx;
		else
			return FAIL;
#if defined(CONFIG_FC_RTL8277C_SERIES)
		if(aal_pe_dual_content_entry_add(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), (l3pe_dual_tbl_entry_t *)outerHdr))
			DEBUG("[Dual][PPTP]Dual content add failed.");
		if(_rtk_fc_dualHeader_ctrl_setupPPTP(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), pPktHdr))
			DEBUG("[Dual][PPTP]Dual contrl entry add failed.");
#endif
	}
	
	/*
	*  For PPTP, pptp interface should record call id to identify the correct interface.
	*/
	if( pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP){
		DEBUG("[Dual][PPTP]Need to add call id : %d to interface table.", pPktHdr->tunnelInfo.pptp.grecallid);
		if(rtk_fc_callId_to_interface_setting(pPktHdr)!=SUCCESS)
			return FAIL;
	}

	//1 2: Add flow path6
	bzero(flowPath6, sizeof(rtk_rg_asic_path6_entry_t));
	flowPath6->valid = FALSE;		// set as VAILD when ds flow was received.
	flowPath6->in_path = FB_PATH_6;
	flowPath6->in_src_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SIP];
	flowPath6->in_src_ipv4_addr = ntohl(pPktHdr->outer_iph->daddr);
	flowPath6->in_dst_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DIP];
	flowPath6->in_dst_ipv4_addr = ntohl(pPktHdr->outer_iph->saddr);
	flowPath6->in_l4_src_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SPORT];
	flowPath6->in_l4_src_port = 0;
	flowPath6->in_l4_dst_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DPORT];
	flowPath6->in_l4_dst_port = 0;
	flowPath6->in_intf_idx = wan_intf_idx;
	flowPath6->in_ctagif = pPktHdr->cvh ? TRUE : FALSE;
	flowPath6->in_stagif = pPktHdr->svh ? TRUE : FALSE;
	flowPath6->in_pppoeif= pPktHdr->ppph ? TRUE : FALSE;
	flowPath6->in_pppoe_sid = pPktHdr->ppph ? ntohs(pPktHdr->ppph->sid) : 0;
	flowPath6->in_src_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SMAC_IDX];
	flowPath6->in_smac_lut_idx = gmacL2Idx;
	flowPath6->in_dst_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DMAC_IDX];
	flowPath6->in_dmac_lut_idx = pNetif->lutIdx;
	flowPath6->in_pptpif = TRUE;
	flowPath6->in_gre_call_id_check = 0;			//update when receiving downstream data
	flowPath6->in_gre_call_id = 0;				//update when receiving downstream data
	flowPath6->in_protocol = FB_INPROTOCOL_ALL_ACCEPT;
	flowPath6->lock = TRUE;

#if defined(CONFIG_RTK_L34_G3_PLATFORM)
	addSwOnly = TRUE;
#endif
#if defined(CONFIG_RTK_SOC_RTL8198D)
	// addSwOnly may be changed here
	if (rtk_fc_flow_limit_check((bool *)&addSwOnly, flowPath6->in_path) != SUCCESS) {
		return FAIL;
	}
#endif

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;

#if defined(CONFIG_FC_RTL8277C_SERIES)
	{
		rtk_8277c_asic_flow_hash_crc_t flow_hash_crc;
		if(_rtk_fc_flow_hashIndex(flowPathEntry, pG3IgrExtraInfo, &flow_hash_crc) != RTK_FC_RET_OK)
		{
			WARNING("Get flow CRC by flowPathEntry failed");
			return RT_ERR_RG_FAILED;
		}
		flowHashIdx = flow_hash_crc.crc16;
	}
#else
	flowHashIdx = _rtk_fc_flow_hashIndex(flowPathEntry, pPktHdr->svlanid, pPktHdr->cvlanid, FALSE);
#endif

	portmap.macPortIdx = RTK_FC_MAC_PORT_PON;

	ret = rtk_fc_flow_add(&flow_idx, flowHashIdx, flowPath6, NULL,
							NULL, FALSE, portmap, addSwOnly, TRUE,
							pG3IgrExtraInfo, pPktHdr);
	if(ret == RT_ERR_RG_OK)
		pNetif->outerHdrFlowIdx = flow_idx;

	//1 3: GRE SEQ starts from 1
	if((ret = rtk_fc_dualHdrInfo_set(RTK_FC_FB_DUALHDR_OUTER_IPV4ID, wan_intf_idx, ntohs(pPktHdr->outer_iph->id)/*network endian*/)) != RT_ERR_RG_OK)
		WARNING("set ipid was fail, ret = %d", ret);

	//if(pPktHdr->tunnelInfo.pptp.greseqTagif){
	if(pPktHdr->tunnelInfo.tunnelTag & GRESEQ_TAGIF){
		if((ret = rtk_fc_dualHdrInfo_set(RTK_FC_FB_DUALHDR_GRESEQ, wan_intf_idx, pPktHdr->tunnelInfo.pptp.greseq/*host endian*/)) != RT_ERR_RG_OK)
			WARNING("set gre sequence was fail, ret = %d", ret);
	}

	//if(pPktHdr->tunnelInfo.pptp.greackTagif){
	if(pPktHdr->tunnelInfo.tunnelTag & GREACK_TAGIF){
		if((ret = rtk_fc_dualHdrInfo_set(RTK_FC_FB_DUALHDR_GREACK, wan_intf_idx, pPktHdr->tunnelInfo.pptp.greack/*host endian*/)) != RT_ERR_RG_OK)
			WARNING("set gre ack was fail, ret = %d", ret);
	}

	return ret;
}


static int _rtk_fc_flow_setupL2TP_by_egrPktHdr(rtk_fc_pktHdr_t *pPktHdr, int wan_intf_idx, int gmacL2Idx)
{
	rtk_rg_err_code_t ret;
	rtk_fc_tableFlowEntry_t flowPathEntry;
	rtk_rg_asic_path6_entry_t *flowPath6 = &flowPathEntry.path6;
	rtk_fc_hwTableNetif_t *pNetif = &(fc_db.netifHwTbl[wan_intf_idx]);
	uint32 flow_idx;
	int32 flowHashIdx = 0, extraTagActListIdx = 0;
	uint8 addSwOnly = FALSE;
	rtk_fc_pmap_t portmap;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	TRACE("[Dual] prepare L2TP tag");

	//1 1: Add flow extra tag
	{
		// Generate IPv4+UDP+L2TP header
		u32 bufLen = 0;
		u8 contenBuffer[64]={0};
		u8 *outerHdr = &contenBuffer[0];
		TRACE("[Dual] Gen outer header and add to SRAM content buffer");

		// Prepare IP(20/40)+UDP(8)+L2TP(8)+PPP(4) for L2TP
		// L3 header --
		if(pPktHdr->outer_iph)
		{
			*((uint16 *)(outerHdr + 0)) = htons(0x4500);	// ver, IHL
			//*((uint16 *)(outerHdr + 2)) = htons(0x0);	// total length	: updated by extra tag actions
			//*((uint16 *)(outerHdr + 4)) = htons(0x0);	// ipid		: updated by extra tag actions
			//*((uint16 *)(outerHdr + 6)) = htons(0x0);	// flags, frag offset
			*((uint16 *)(outerHdr + 8)) = htons(0x4011);	// ttl, protocol
			//*((uint16 *)(outerHdr + 10)) = htons(0x0);	// hdr checksum
			*((uint32 *)(outerHdr + 12)) = pPktHdr->outer_iph->saddr;	// src ip
			*((uint32 *)(outerHdr + 16)) = pPktHdr->outer_iph->daddr;	// dst ip
			bufLen += 20;
		}
		else
		{
			TRACE("[Dual][L2TP] Do not support outer non-ipv4 hdr");
			return FAIL;
		}
		// L4 UDP --
		{
			*(uint16 *)(outerHdr + bufLen) = pPktHdr->outer_udph->source;	// src port
			*(uint16 *)(outerHdr + bufLen + 2) = pPktHdr->outer_udph->dest;	// dst port
			//*(uint16 *)(outerHdr + bufLen + 4) = htons(0x0);	// len: updated by extra tag actions
			//*(uint16 *)(outerHdr + bufLen + 6) = htons(0x0);	// checksum: updated by extra tag actions
			bufLen += 8;
		}
		// L2TP --
		{
			*(uint16 *)(outerHdr + bufLen) = htons(0x4002);	// Type(0), Length(1), Sequence(0), Offset(0), Priority(0), Version(2)
			//*(uint16 *)(outerHdr + bufLen +2) = htons(0x0);	//Total length: updated by extra tag actions
			*(uint16 *)(outerHdr + bufLen +4) = htons(pPktHdr->tunnelInfo.l2tp.tunnelid);	//Peer's Tunnel ID
			*(uint16 *)(outerHdr + bufLen +6) = htons(pPktHdr->tunnelInfo.l2tp.sessionid);	//Peer's Session ID
			bufLen += 8;
		}
		// PPP --
		{
#if defined(CONFIG_FC_RTL8277C_SERIES)		
			switch(pPktHdr->tunnelInfo.pppMode) // Address, Control, Protocol=IPv4
			{
				case RTK_FC_COMP_PPP_FF0300XX:
					outerHdr[bufLen]	= 0xff ;
					outerHdr[bufLen+1]	= 0x3 ;
					outerHdr[bufLen+2]	= 0x0 ;
					outerHdr[bufLen+3]	= 0x21 ;					
					break;
				case RTK_FC_COMP_PPP_FF03XX:
					outerHdr[bufLen]	= 0xff ;
					outerHdr[bufLen+1]	= 0x3 ;
					outerHdr[bufLen+2]	= 0x21 ;							
					break;
				case RTK_FC_COMP_PPP_00XX:
					outerHdr[bufLen]	= 0x0 ;
					outerHdr[bufLen+1]	= 0x21 ;							
					break;
				case RTK_FC_COMP_PPP_XX:
					outerHdr[bufLen]	= 0x21 ;						
					break;
				default:
					break;
			}
			bufLen += pPktHdr->tunnelInfo.pppMode;
#else
			*(uint32 *)(outerHdr + bufLen) = htonl(0xff030021);							// Address, Control, Protocol=IPv4
			bufLen += 4;
#endif
		}
		extraTagActListIdx = pNetif->outerHdrExtratagIdx;	// use the previous used list if readd.
		if((ret = _rtk_fc_extraTag_setupL2TP(wan_intf_idx, outerHdr, &extraTagActListIdx)) == SUCCESS){
			pNetif->outerHdrExtratagIdx = extraTagActListIdx;
			TRACE("[Dual] extra tag index: %d", extraTagActListIdx);
		}
#if defined(CONFIG_FC_RTL8277C_SERIES)
		if(aal_pe_dual_content_entry_add(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), (l3pe_dual_tbl_entry_t *)outerHdr))
			DEBUG("[Dual][L2TP] Dual content add failed.");
		if(_rtk_fc_dualHeader_ctrl_setupL2TP(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), pPktHdr))
			DEBUG("[Dual][L2TP] Dual contrl entry add failed.");
#endif		

	}

	//1 2: Add flow path6
	bzero(flowPath6, sizeof(rtk_rg_asic_path6_entry_t));
	flowPath6->valid = FALSE;		// set as VAILD when ds flow was received.
	flowPath6->in_path = FB_PATH_6;
	flowPath6->in_src_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SIP];
	flowPath6->in_src_ipv4_addr = ntohl(pPktHdr->outer_iph->daddr);
	flowPath6->in_dst_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DIP];
	flowPath6->in_dst_ipv4_addr = ntohl(pPktHdr->outer_iph->saddr);
	flowPath6->in_l4_src_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SPORT];
	flowPath6->in_l4_src_port = pPktHdr->outer_udph->source;
	flowPath6->in_l4_dst_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DPORT];
	flowPath6->in_l4_dst_port = pPktHdr->outer_udph->dest;
	flowPath6->in_intf_idx = wan_intf_idx;
	flowPath6->in_ctagif = pPktHdr->cvh ? TRUE : FALSE;
	flowPath6->in_stagif = pPktHdr->svh ? TRUE : FALSE;
	flowPath6->in_pppoeif= pPktHdr->ppph ? TRUE : FALSE;
	flowPath6->in_pppoe_sid = pPktHdr->ppph ? ntohs(pPktHdr->ppph->sid) : 0;
	flowPath6->in_src_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SMAC_IDX];
	flowPath6->in_smac_lut_idx = gmacL2Idx;
	flowPath6->in_src_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DMAC_IDX];
	flowPath6->in_dmac_lut_idx = pNetif->lutIdx;
	flowPath6->in_l2tpif = TRUE;
	flowPath6->in_l2tp_tunnel_id_check = 0;		//update when receiving downstream data
	flowPath6->in_l2tp_tunnel_id = 0;			//update when receiving downstream data
	flowPath6->in_l2tp_session_id_check = 0;		//update when receiving downstream data
	flowPath6->in_l2tp_session_id = 0;			//update when receiving downstream data
	flowPath6->in_protocol = FB_INPROTOCOL_ALL_ACCEPT;
	flowPath6->lock = TRUE;

#if defined(CONFIG_RTK_L34_G3_PLATFORM)
		addSwOnly = TRUE;
#endif
#if defined(CONFIG_RTK_SOC_RTL8198D)
	// addSwOnly may be changed here
	if (rtk_fc_flow_limit_check((bool *)&addSwOnly, flowPath6->in_path) != SUCCESS) {
		return FAIL;
	}
#endif

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;

#if defined(CONFIG_FC_RTL8277C_SERIES)
	{
		rtk_8277c_asic_flow_hash_crc_t flow_hash_crc;
		if(_rtk_fc_flow_hashIndex(flowPathEntry, pG3IgrExtraInfo, &flow_hash_crc) != RTK_FC_RET_OK)
		{
			WARNING("Get flow CRC by flowPathEntry failed");
			return RT_ERR_RG_FAILED;
		}
		flowHashIdx = flow_hash_crc.crc16;
	}
#else
	flowHashIdx = _rtk_fc_flow_hashIndex(flowPathEntry, pPktHdr->svlanid, pPktHdr->cvlanid, FALSE);
#endif


	portmap.macPortIdx = RTK_FC_MAC_PORT_PON;

	ret = rtk_fc_flow_add(&flow_idx, flowHashIdx, flowPath6, NULL,
						NULL, FALSE, portmap, addSwOnly, TRUE,
						pG3IgrExtraInfo, pPktHdr);
	pNetif->outerHdrFlowIdx = flow_idx;

	TRACE("[Dual] intf idx[%d], path 6 idx[%d], extra tag idx[%d]", wan_intf_idx, flowHashIdx, extraTagActListIdx);

	return ret;
}

static int _rtk_fc_flow_setupDSLITE_by_egrPktHdr(rtk_fc_pktHdr_t *pPktHdr, int wan_intf_idx, int gmacL2Idx)
{
	rtk_rg_err_code_t ret;
	rtk_fc_tableFlowEntry_t flowPathEntry;
	rtk_rg_asic_path6_entry_t *flowPath6 = &flowPathEntry.path6;
	rtk_fc_hwTableNetif_t *pNetif = &(fc_db.netifHwTbl[wan_intf_idx]);
	uint32 flow_idx;
	int32 flowHashIdx = 0, extraTagActListIdx = 0;
	uint32 ihl_tc_flow = 0x60000000;	// version: 6
	uint8 addSwOnly = FALSE;
	rtk_fc_pmap_t portmap;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	TRACE("[Dual] prepare DsLite tag");

	//1 1: Add flow extra tag
	{
		// Generate IPv6 header
		u8 contenBuffer[64]={0};
		u8 *outerHdr = &contenBuffer[0];
		TRACE("[Dual] Gen outer header and add to SRAM content buffer");

		// Prepare IPv6(40) for DSLite
		// L3 header --
		{
			*(uint32 *)(outerHdr + 0) = htonl(ihl_tc_flow);					//ver, tc, flowlabel
			//*(uint16 *)(pData + 4) = htons(0x0);							//Payload length: updated by extra tag actions
			*(uint16 *)(outerHdr + 6) = htons(0x0400 | 0xff);					//Next header(IPIP:4), Hop limit
			memcpy(outerHdr + 8, &pPktHdr->ip6h->saddr, IPV6_ADDR_LEN);	// src ip
			memcpy(outerHdr + 24, &pPktHdr->ip6h->daddr, IPV6_ADDR_LEN);	// dst ip
		}

		extraTagActListIdx = pNetif->outerHdrExtratagIdx;	// use the previous used list if readd.
		if((ret = _rtk_fc_extraTag_setupDsLite(wan_intf_idx, outerHdr, &extraTagActListIdx)) == SUCCESS)
		{
			TRACE("Set up extraTag[%d]", extraTagActListIdx);
			pNetif->outerHdrExtratagIdx = extraTagActListIdx;
		}
		
#if defined(CONFIG_FC_RTL8277C_SERIES)
		if(aal_pe_dual_content_entry_add(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), (l3pe_dual_tbl_entry_t *)outerHdr))
			DEBUG("[Dual][DSlite/MAPE] Dual content add failed.");
		if(_rtk_fc_dualHeader_ctrl_setupDsliteMape(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), pPktHdr))
			DEBUG("[Dual][DSlite/MAPE] Dual contrl entry add failed.");
#endif	
	}

	/*
	*	Fix DSLITE Downstream hash unmatch
	*/
	pPktHdr->ipv6Sip_hash = _rtk_rg_flowHashIPv6DstAddr_get(pPktHdr->ip6h->saddr.s6_addr);
	pPktHdr->ipv6Dip_hash = _rtk_rg_flowHashIPv6SrcAddr_get(pPktHdr->ip6h->daddr.s6_addr);

	//1 2: Add flow path6
	bzero(flowPath6, sizeof(rtk_rg_asic_path6_entry_t));
	flowPath6->valid = FALSE;		// set as VAILD when ds flow was received.
	flowPath6->in_path = FB_PATH_6;
	flowPath6->in_src_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SIP];
	flowPath6->in_src_ipv6_addr_hash = pPktHdr->ipv6Dip_hash;
	flowPath6->in_dst_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DIP];
	flowPath6->in_dst_ipv6_addr_hash = pPktHdr->ipv6Sip_hash;
	flowPath6->in_l4_src_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SPORT];
	flowPath6->in_l4_src_port = 0;
	flowPath6->in_l4_dst_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DPORT];
	flowPath6->in_l4_dst_port = 0;
	flowPath6->in_intf_idx = wan_intf_idx;
	flowPath6->in_ctagif = pPktHdr->cvh ? TRUE : FALSE;
	flowPath6->in_stagif = pPktHdr->svh ? TRUE : FALSE;
	flowPath6->in_pppoeif= pPktHdr->ppph ? TRUE : FALSE;
	flowPath6->in_pppoe_sid = pPktHdr->ppph ? ntohs(pPktHdr->ppph->sid) : 0;
	flowPath6->in_src_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SMAC_IDX];
	flowPath6->in_smac_lut_idx = gmacL2Idx;
	flowPath6->in_dst_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DMAC_IDX];
	flowPath6->in_dmac_lut_idx = pNetif->lutIdx;
	flowPath6->in_dsliteif = TRUE;
	flowPath6->in_protocol = FB_INPROTOCOL_ALL_ACCEPT;
	flowPath6->lock = TRUE;

#if defined(CONFIG_RTK_L34_G3_PLATFORM)
	addSwOnly = TRUE;
#endif
#if defined(CONFIG_RTK_SOC_RTL8198D)
	// addSwOnly may be changed here
	if (rtk_fc_flow_limit_check((bool *)&addSwOnly, flowPath6->in_path) != SUCCESS) {
		return FAIL;
	}
#endif

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;

#if defined(CONFIG_FC_RTL8277C_SERIES)
	{
		rtk_8277c_asic_flow_hash_crc_t flow_hash_crc;
		if(_rtk_fc_flow_hashIndex(flowPathEntry, pG3IgrExtraInfo, &flow_hash_crc) != RTK_FC_RET_OK)
		{
			WARNING("Get flow CRC by flowPathEntry failed");
			return RT_ERR_RG_FAILED;
		}
		flowHashIdx = flow_hash_crc.crc16;
	}
#else
	flowHashIdx = _rtk_fc_flow_hashIndex(flowPathEntry, pPktHdr->svlanid, pPktHdr->cvlanid, FALSE);
#endif


	portmap.macPortIdx = RTK_FC_MAC_PORT_PON;

	ret = rtk_fc_flow_add(&flow_idx, flowHashIdx, flowPath6, NULL,
						NULL, FALSE, portmap, addSwOnly, TRUE,
						pG3IgrExtraInfo, pPktHdr);
	if(ret == RT_ERR_RG_OK){
		pNetif->outerHdrFlowIdx = flow_idx;
		// Use the pFlowTable->mapeInfo.mape_flow6 field to distinguish the downstream between Ds-Lite and MAP-E
		rtk_fc_mark_mape_flow6(pPktHdr, flow_idx);		
	}


	return ret;
}

extern 	rtk_fc_ret_t _rtk_fc_dualHeader_ctrl_setup6RD(int dualControlIdx, rtk_fc_pktHdr_t *pPktHdr);
static int _rtk_fc_flow_setup6RD_by_egrPktHdr(rtk_fc_pktHdr_t *pPktHdr, int wan_intf_idx, int gmacL2Idx)
{
	rtk_rg_err_code_t ret;
	rtk_fc_tableFlowEntry_t flowPathEntry;
	rtk_rg_asic_path6_entry_t *flowPath6 = &flowPathEntry.path6;
	rtk_fc_hwTableNetif_t *pNetif = &(fc_db.netifHwTbl[wan_intf_idx]);
	uint32 flow_idx;
	int32 flowHashIdx = 0, extraTagActListIdx = 0;
	uint32 ihvl = 0x45000000;	// version: 4, hlen: 5
	uint8 addSwOnly = FALSE;
	rtk_fc_pmap_t portmap;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	//1 1: Add flow extra tag
	{
		// Generate IPv4 header
		u8 contenBuffer[64]={0};
		u8 *outerHdr = &contenBuffer[0];

		// Prepare IPv4(20) for 6RD
		// L3 header --
		{
			*(uint32 *)(outerHdr + 0) = htonl(ihvl);					//ver, hlen
			*(uint16 *)(outerHdr + 8) = htons(0xff00 | 0x29);					//Hop limit,protocol(41)
			memcpy(outerHdr + 12, &pPktHdr->outer_iph->saddr, sizeof(uint32));	// src ip
			memcpy(outerHdr + 16, &pPktHdr->outer_iph->daddr, sizeof(uint32));	// dst ip
		}

		extraTagActListIdx = pNetif->outerHdrExtratagIdx;	// use the previous used list if readd.
		if((ret = _rtk_fc_extraTag_setup6RD(wan_intf_idx, outerHdr, &extraTagActListIdx)) == SUCCESS)
		{
			TRACE("Set up extraTag[%d]", extraTagActListIdx);
			pNetif->outerHdrExtratagIdx = extraTagActListIdx;
		}

#if defined(CONFIG_FC_RTL8277C_SERIES)
		if(aal_pe_dual_content_entry_add(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), (l3pe_dual_tbl_entry_t *)outerHdr))
			DEBUG("[Dual][6RD] Dual content add failed.");
		if(_rtk_fc_dualHeader_ctrl_setup6RD(RTK_FC_DUAL_CONTROL_IDX(wan_intf_idx), pPktHdr))
			DEBUG("[Dual][6RD] Dual contrl entry add failed.");
#endif		

	}

	//1 2: Add flow path6
	bzero(flowPath6, sizeof(rtk_rg_asic_path6_entry_t));
	flowPath6->valid = FALSE;		// set as VAILD when ds flow was received.
	flowPath6->in_path = FB_PATH_6;
	flowPath6->in_src_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SIP];
	flowPath6->in_src_ipv4_addr = ntohl(pPktHdr->outer_iph->daddr);
	flowPath6->in_dst_ip_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DIP];
	flowPath6->in_dst_ipv4_addr = ntohl(pPktHdr->outer_iph->saddr);
	flowPath6->in_l4_src_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SPORT];
	flowPath6->in_l4_dst_port_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DPORT];
	flowPath6->in_l4_src_port = 0; //pPktHdr->tcph->source;
	flowPath6->in_l4_dst_port = 0; //pPktHdr->tcph->source;
	flowPath6->in_intf_idx = wan_intf_idx;
	flowPath6->in_ctagif = pPktHdr->cvh ? TRUE : FALSE;
	flowPath6->in_stagif = pPktHdr->svh ? TRUE : FALSE;
	flowPath6->in_pppoeif= pPktHdr->ppph ? TRUE : FALSE;
	flowPath6->in_pppoe_sid = pPktHdr->ppph ? ntohs(pPktHdr->ppph->sid) : 0;
	flowPath6->in_src_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_SMAC_IDX];
	flowPath6->in_smac_lut_idx = gmacL2Idx;
	flowPath6->in_dst_mac_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_DMAC_IDX];
	flowPath6->in_dmac_lut_idx = pNetif->lutIdx;
	flowPath6->rsvd_in_6rdif = TRUE; //h/w not support 6rd, thus we use flowPath6->reserved1 as is6RDif here
	flowPath6->in_protocol = FB_INPROTOCOL_ALL_ACCEPT;
	flowPath6->lock = TRUE;

#if defined(CONFIG_RTK_L34_G3_PLATFORM)
	addSwOnly = TRUE;
#endif

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;

#if defined(CONFIG_FC_RTL8277C_SERIES)
	{
		rtk_8277c_asic_flow_hash_crc_t flow_hash_crc;
		if(_rtk_fc_flow_hashIndex(flowPathEntry, pG3IgrExtraInfo, &flow_hash_crc) != RTK_FC_RET_OK)
		{
			WARNING("Get flow CRC by flowPathEntry failed");
			return RT_ERR_RG_FAILED;
		}
		flowHashIdx = flow_hash_crc.crc16;
	}
#else
	flowHashIdx = _rtk_fc_flow_hashIndex(flowPathEntry, pPktHdr->svlanid, pPktHdr->cvlanid, FALSE);
#endif


	portmap.macPortIdx = RTK_FC_MAC_PORT_PON;

	ret = rtk_fc_flow_add(&flow_idx, flowHashIdx, flowPath6, NULL,
						NULL, FALSE, portmap, addSwOnly, TRUE,
						pG3IgrExtraInfo, pPktHdr);
	if(ret == RT_ERR_RG_OK)
		pNetif->outerHdrFlowIdx = flow_idx;

	//IPID 
	if((ret = rtk_fc_dualHdrInfo_set(RTK_FC_FB_DUALHDR_OUTER_IPV4ID, wan_intf_idx, ntohs(pPktHdr->outer_iph->id)/*network endian*/)) != RT_ERR_RG_OK)
		WARNING("set ipid was fail, ret = %d", ret);

	return ret;
}

static int _rtk_fc_flow_updatePPTP_by_igrPktHdr(rtk_fc_pktHdr_t *pPktHdr, uint32 path6Idx)
{
	rtk_rg_asic_path6_entry_t *flowPath6 = &fc_db.flowTbl[path6Idx].pFlowEntry->path6;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
#if defined(CONFIG_RTK_L34_G3_PLATFORM)	
	if(!FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY))
		WARNING("Please check! Path 6 flow entry should be SW entry.");
#endif	
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;


	flowPath6->valid = TRUE;
	flowPath6->in_pptpif = TRUE;
	flowPath6->in_gre_call_id_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_GRE_CALL_ID];
	flowPath6->in_gre_call_id = pPktHdr->tunnelInfo.pptp.grecallid;
	flowPath6->lock = TRUE;

	ASSERT_EQ(RTK_RG_ASIC_FLOWPATH_SET(&path6Idx, flowPath6, pG3IgrExtraInfo, FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY)), RTK_FC_RET_OK);

#if defined(CONFIG_FC_RTL8277C_SERIES)
	{
		rtk_fc_aclAndCf_reserved_dual_hdr_chk_t dual_hdr_chk = {0};
		int ret;

		dual_hdr_chk.packet_type			= RTK_FC_CLS_DUAL_PPTP_GRE;
		dual_hdr_chk.profile_idx			= 0; //WAN
		dual_hdr_chk.igr_intf_idx			= _rtk_fc_flow_intf_mapping_idx_get(flowPath6->in_intf_idx);
		dual_hdr_chk.hash_tuple_idx 		= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;
		dual_hdr_chk.pptp_gre.gre_call_id 	= pPktHdr->tunnelInfo.pptp.grecallid;
		//check ip header
		dual_hdr_chk.pptp_gre.is_ipv4 = (pPktHdr->outer_iph) ? 1 : 0;
		if(dual_hdr_chk.pptp_gre.is_ipv4)
		{
			dual_hdr_chk.pptp_gre.ipv4_dst_ip = ntohl(pPktHdr->outer_iph->daddr);
		}
		else
		{
			dual_hdr_chk.pptp_gre.ipv6_dst_ip[0] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[0]);
			dual_hdr_chk.pptp_gre.ipv6_dst_ip[1] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[1]);
			dual_hdr_chk.pptp_gre.ipv6_dst_ip[2] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[2]);
			dual_hdr_chk.pptp_gre.ipv6_dst_ip[3] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[3]);
		}
		//check vlan
		dual_hdr_chk.pptp_gre.vlan.vlan_chcek_en	= 1;
		dual_hdr_chk.pptp_gre.vlan.has_stag 		= (pPktHdr->svh) ? 1 : 0 ;
		dual_hdr_chk.pptp_gre.vlan.svlan_id 		= (pPktHdr->svh) ? pPktHdr->svlanid : 0 ;
		dual_hdr_chk.pptp_gre.vlan.has_ctag 		= (pPktHdr->cvh) ? 1 : 0 ;
		dual_hdr_chk.pptp_gre.vlan.cvlan_id 		= (pPktHdr->cvh) ? pPktHdr->cvlanid : 0 ;

		ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
		if(ret != RTK_FC_RET_OK)
		{
			WARNING("add special ACL DUAL_HEADER_DS_CHECK failed!!! ret = %x", ret);
			return RTK_FC_RET_ERR;
		}
		else
		{
			DEBUG("add special ACL DUAL_HEADER_DS_CHECK success, idx: %d, intf_idx=%d", dual_hdr_chk.rslt_idx, dual_hdr_chk.igr_intf_idx);
			fc_db.netifHwTbl[flowPath6->in_intf_idx].dualHdr_ds_clsIdx = dual_hdr_chk.rslt_idx;
		}
	}
#endif

	return SUCCESS;
}

static int _rtk_fc_flow_updateL2TP_by_igrPktHdr(rtk_fc_pktHdr_t *pPktHdr, uint32 path6Idx)
{
	rtk_rg_asic_path6_entry_t *flowPath6 = &fc_db.flowTbl[path6Idx].pFlowEntry->path6;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
#if defined(CONFIG_RTK_L34_G3_PLATFORM)	
	if(!FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY))
		WARNING("Please check! Path 6 flow entry should be SW entry.");
#endif	
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;


	flowPath6->valid = TRUE;
	flowPath6->in_l2tpif = TRUE;
	flowPath6->in_l2tp_tunnel_id_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_L2TP_TUNNEL_ID];
	flowPath6->in_l2tp_tunnel_id = pPktHdr->tunnelInfo.l2tp.tunnelid;
	flowPath6->in_l2tp_session_id_check = fc_db.systemGlobal.flowCheckState[FB_FLOW_CHECK_PATH6_L2TP_SID];
	flowPath6->in_l2tp_session_id = pPktHdr->tunnelInfo.l2tp.sessionid;
	flowPath6->lock = TRUE;

	ASSERT_EQ(RTK_RG_ASIC_FLOWPATH_SET(&path6Idx, flowPath6, pG3IgrExtraInfo, FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY)), RTK_FC_RET_OK);

	//Add downstream dual header CLS
#if defined(CONFIG_FC_RTL8277C_SERIES)
{
	rtk_fc_aclAndCf_reserved_dual_hdr_chk_t dual_hdr_chk = {0};
	int ret;

	dual_hdr_chk.packet_type		= RTK_FC_CLS_DUAL_L2TP;
	dual_hdr_chk.profile_idx		= 0; //WAN
	dual_hdr_chk.igr_intf_idx		= _rtk_fc_flow_intf_mapping_idx_get(flowPath6->in_intf_idx);
	dual_hdr_chk.hash_tuple_idx 	= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;
	dual_hdr_chk.l2tp.tunnel_id		= pPktHdr->tunnelInfo.l2tp.tunnelid;
	dual_hdr_chk.l2tp.session_id	= pPktHdr->tunnelInfo.l2tp.sessionid;
	//check ip header
	dual_hdr_chk.l2tp.is_ipv4 = (pPktHdr->outer_iph) ? 1 : 0 ;
	if(dual_hdr_chk.l2tp.is_ipv4)
	{
		dual_hdr_chk.l2tp.ipv4_dst_ip = ntohl(pPktHdr->outer_iph->daddr);
	}
	else
	{
		dual_hdr_chk.l2tp.ipv6_dst_ip[0] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[0]);
		dual_hdr_chk.l2tp.ipv6_dst_ip[1] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[1]);
		dual_hdr_chk.l2tp.ipv6_dst_ip[2] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[2]);
		dual_hdr_chk.l2tp.ipv6_dst_ip[3] = ntohl(pPktHdr->outer_ip6h->daddr.s6_addr32[3]);
	}
	//check vlan
	dual_hdr_chk.l2tp.vlan.vlan_chcek_en	= 1;
	dual_hdr_chk.l2tp.vlan.has_stag			= (pPktHdr->svh) ? 1 : 0 ;
	dual_hdr_chk.l2tp.vlan.svlan_id			= (pPktHdr->svh) ? pPktHdr->svlanid : 0 ;
	dual_hdr_chk.l2tp.vlan.has_ctag			= (pPktHdr->cvh) ? 1 : 0 ;
	dual_hdr_chk.l2tp.vlan.cvlan_id			= (pPktHdr->cvh) ? pPktHdr->cvlanid : 0 ;
	
	ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
	if(ret != RTK_FC_RET_OK)
	{
		WARNING("add special ACL DUAL_HEADER_DS_CHECK failed!!! ret = %x", ret);
		return RTK_FC_RET_ERR;
	}
	else
	{
		DEBUG("add special ACL DUAL_HEADER_DS_CHECK success, idx: %d, intf_idx=%d", dual_hdr_chk.rslt_idx, dual_hdr_chk.igr_intf_idx);
		fc_db.netifHwTbl[dual_hdr_chk.igr_intf_idx].dualHdr_ds_clsIdx = dual_hdr_chk.rslt_idx;
	}
}
#endif

	return SUCCESS;
}

static int _rtk_fc_flow_updateDSLITE_by_igrPktHdr(rtk_fc_pktHdr_t *pPktHdr, uint32 path6Idx)
{
	rtk_rg_asic_path6_entry_t *flowPath6 = &fc_db.flowTbl[path6Idx].pFlowEntry->path6;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	TRACE("dslite path6Idx=%d flowPath6=%p",path6Idx,flowPath6);
	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
#if defined(CONFIG_RTK_L34_G3_PLATFORM)	
#if defined(CONFIG_FC_RTL8277C_SERIES)
	//8277C support dslite/mape
{
	rtk_fc_aclAndCf_reserved_dual_hdr_chk_t dual_hdr_chk = {0};
	int ret;
	if(!pPktHdr->ip6h)
		return RTK_FC_RET_ERR_NULL_POINTER;

	if(!fc_db.flowTbl[path6Idx].mapeInfo.mape_flow6)
	{
		TRACE("ADD Dslite Reserved ACL");
		//dslite
		dual_hdr_chk.packet_type		= RTK_FC_CLS_DUAL_DSLITE;
		dual_hdr_chk.profile_idx		= 0; //WAN
		dual_hdr_chk.igr_intf_idx		= _rtk_fc_flow_intf_mapping_idx_get(flowPath6->in_intf_idx);
		dual_hdr_chk.hash_tuple_idx 	= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;

		dual_hdr_chk.dslite.src_ip[0] = ntohl(pPktHdr->ip6h->saddr.s6_addr32[0]);
		dual_hdr_chk.dslite.src_ip[1] = ntohl(pPktHdr->ip6h->saddr.s6_addr32[1]);
		dual_hdr_chk.dslite.src_ip[2] = ntohl(pPktHdr->ip6h->saddr.s6_addr32[2]);
		dual_hdr_chk.dslite.src_ip[3] = ntohl(pPktHdr->ip6h->saddr.s6_addr32[3]);
		//check vlan
		dual_hdr_chk.dslite.vlan.vlan_chcek_en	= 1;
		dual_hdr_chk.dslite.vlan.has_stag			= (pPktHdr->svh) ? 1 : 0 ;
		dual_hdr_chk.dslite.vlan.svlan_id			= (pPktHdr->svh) ? pPktHdr->svlanid : 0 ;
		dual_hdr_chk.dslite.vlan.has_ctag			= (pPktHdr->cvh) ? 1 : 0 ;
		dual_hdr_chk.dslite.vlan.cvlan_id			= (pPktHdr->cvh) ? pPktHdr->cvlanid : 0 ;
	}
	else
	{
		TRACE("ADD MAPE Reserved ACL");
		//mape
		dual_hdr_chk.packet_type		= RTK_FC_CLS_DUAL_MAP_E;
		dual_hdr_chk.profile_idx		= 0; //WAN
		dual_hdr_chk.igr_intf_idx		= _rtk_fc_flow_intf_mapping_idx_get(flowPath6->in_intf_idx);
		dual_hdr_chk.hash_tuple_idx 	= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;

		dual_hdr_chk.map_e.dst_ip[0] = ntohl(pPktHdr->ip6h->daddr.s6_addr32[0]);
		dual_hdr_chk.map_e.dst_ip[1] = ntohl(pPktHdr->ip6h->daddr.s6_addr32[1]);
		dual_hdr_chk.map_e.dst_ip[2] = ntohl(pPktHdr->ip6h->daddr.s6_addr32[2]);
		dual_hdr_chk.map_e.dst_ip[3] = ntohl(pPktHdr->ip6h->daddr.s6_addr32[3]);
		
		//check vlan
		dual_hdr_chk.map_e.vlan.vlan_chcek_en	= 1;
		dual_hdr_chk.map_e.vlan.has_stag			= (pPktHdr->svh) ? 1 : 0 ;
		dual_hdr_chk.map_e.vlan.svlan_id			= (pPktHdr->svh) ? pPktHdr->svlanid : 0 ;
		dual_hdr_chk.map_e.vlan.has_ctag			= (pPktHdr->cvh) ? 1 : 0 ;
		dual_hdr_chk.map_e.vlan.cvlan_id			= (pPktHdr->cvh) ? pPktHdr->cvlanid : 0 ;		
	}
	ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
	if(ret != RTK_FC_RET_OK)
	{
		WARNING("add special ACL DUAL_HEADER_DS_CHECK failed!!! ret = %x", ret);
		return RTK_FC_RET_ERR;
	}
	else
	{
		DEBUG("add special ACL DUAL_HEADER_DS_CHECK success, idx: %d, intf_idx=%d", dual_hdr_chk.rslt_idx, dual_hdr_chk.igr_intf_idx);
		fc_db.netifHwTbl[dual_hdr_chk.igr_intf_idx].dualHdr_ds_clsIdx = dual_hdr_chk.rslt_idx;
	}
}	
#endif

	if(!FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY))
		WARNING("Please check! Path 6 flow entry should be SW entry.");
#endif	

	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;

	flowPath6->valid = TRUE;
	flowPath6->in_dsliteif = TRUE;
	flowPath6->lock = TRUE;

	ASSERT_EQ(RTK_RG_ASIC_FLOWPATH_SET(&path6Idx, flowPath6, pG3IgrExtraInfo, FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY)), RTK_FC_RET_OK);

	return SUCCESS;
}

static int _rtk_fc_flow_update6RD_by_igrPktHdr(rtk_fc_pktHdr_t *pPktHdr, uint32 path6Idx)
{
	rtk_rg_asic_path6_entry_t *flowPath6 = &fc_db.flowTbl[path6Idx].pFlowEntry->path6;
	rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo = NULL;
	rtk_fc_g3IgrExtraInfo_t g3IgrExtraInfo;

	//prepare info that needed for adding G3 HW flow entry. Add pure sw flow here so don't care the content of g3IgrExtraInfo
#if defined(CONFIG_RTK_L34_G3_PLATFORM)	
	if(!FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY))
		WARNING("Please check! Path 6 flow entry should be SW entry.");
#endif	
	_rtk_fc_g3IgrExtraInfo_init(&g3IgrExtraInfo);
	pG3IgrExtraInfo = &g3IgrExtraInfo;


	flowPath6->valid = TRUE;
	flowPath6->rsvd_in_6rdif = TRUE;
	flowPath6->lock = TRUE;

	ASSERT_EQ(RTK_RG_ASIC_FLOWPATH_SET(&path6Idx, flowPath6, pG3IgrExtraInfo, FLOW_INFO_IS_SET(&fc_db.flowTbl[path6Idx], FLOW_INFO_SOFTWARE_ONLY)), RTK_FC_RET_OK);

	// 6RD: set l3 reserved cls
#if defined(CONFIG_FC_RTL8277C_SERIES)
	if(pPktHdr->outer_iph)
	{
		int ret;
		int result_idx;
		rtk_fc_ingress_data_t fcIngressData;
		int ingressIntfIdx;

		ingressIntfIdx = _rtk_fc_flow_intf_mapping_idx_get(flowPath6->in_intf_idx);
		fcIngressData.srcIp = pPktHdr->outer_iph->saddr;
		fcIngressData.dstIp = pPktHdr->outer_iph->daddr;
		ret = rtk_fc_8277C_v6rd_downStream_reservedACL_add(ingressIntfIdx,&fcIngressData, &result_idx);
		if(ret!=RTK_FC_RET_OK)
		{
			DEBUG("Set Reserved ACL failed!");
			return RTK_FC_RET_ERR;
		}
	}
#endif

	return SUCCESS;
}

static int _rtk_fc_applyExtraTagAction(int extraTagIdx, rtk_rg_asic_extraTagAction_t *pExtAct, struct rt_skbuff *rtskb, u32 pktBufOff, rtk_fc_pktHdr_t *pPktHdr)
{
	u32 value = 0;
	u8 actBit = pExtAct->type1.act_bit;
	int TUNN_tag_len = fc_db.extraTagList[extraTagIdx].contentLen;
	u8 *pData = RTSKB_DATA(rtskb);

	if(actBit == FB_EXTG_ACTBIT_1)
	{
		u8 *insertBuf = &fc_db.extraTagContentBuffer[pExtAct->type1.src_addr_offset];
		// insert_location: Insert outer header after PPP header(PPPoE+PPP).
				// if without PPP header, insert outer header after Ethertype(IPoE).

		RTK_FC_HOOK_PS_SKB_SKB_PUSH(RTSKB_SKB(rtskb), TUNN_tag_len, &pData);
		memcpy(pData, pData + TUNN_tag_len, pktBufOff);
		RTSKB_MAC_HEADER(rtskb) -= (TUNN_tag_len);
		// reset DMAC and SMAC pointer
		pPktHdr->eth=(struct ethhdr *)pData;
		// copy outer hdr buffer
		memcpy(pData+pktBufOff, insertBuf, TUNN_tag_len);

		pPktHdr->outer_iph = (struct iphdr *)(pData+pktBufOff);	//record outer header pointer

		DEBUG("[Dual] actionbit %d - insert outer hdr buffer", actBit);
		//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG){
		//	memDump(pData, pktBufOff+TUNN_tag_len+20, "insert outer hdr");
		//}
	}

	if(actBit == FB_EXTG_ACTBIT_2)
	{
		uint16 replaceType = pExtAct->type2.ethertype;
		u16 *pEthType =(u16 *)(pData + pktBufOff - 2);

		*pEthType = htons(replaceType);
		DEBUG("[Dual] actionbit %d - replace etherType to 0x%04x", actBit, pExtAct->type2.ethertype);
	}

	if(actBit == FB_EXTG_ACTBIT_3)
	{
		u16 *tmplen = NULL;
		value = RTSKB_LEN(rtskb) - TUNN_tag_len - pktBufOff;	// reset to original l2 payload length.
		if(pExtAct->type3.operation==0)	// +
			value += pExtAct->type3.value;
		else{
			WARNING("Not support!! FIXME");
		}

		if(pExtAct->type3.length == 2){
			 tmplen = (u16 *)(pData+pktBufOff+pExtAct->type3.pkt_buff_offset);
			*tmplen = htons((u16)value);
		}else{
			WARNING("Not support!! FIXME");
		}
		DEBUG("[Dual] actionbit %d - skb len: %d, tunnTagLen: %d, pktoffset: %d", actBit, RTSKB_LEN(rtskb), TUNN_tag_len, pktBufOff);
		if(tmplen) {
			DEBUG("[Dual] actionbit %d - update offset %d to value 0x%x", actBit, pExtAct->type3.pkt_buff_offset, htons(*tmplen));
		}
	}

	if(actBit == FB_EXTG_ACTBIT_4)
	{
		u32 counter = 0;

		if(pExtAct->type4.data_src_type == 0)
		{
			u32 *target = NULL;
			u32 seqackOffset = 0;
			if(pExtAct->type4.reduce_seq != 1)
			{
				target = (u32 *)(pData+pktBufOff+pExtAct->type4.pkt_buff_offset);
				rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_GRESEQ, pExtAct->type4.seq_ack_reg_idx, &counter);
				*target = htonl(counter);
				seqackOffset+=4;
				DEBUG("[Dual] actionbit %d - use gre SEQ# 0x%x", actBit, counter);
			}
			if(pExtAct->type4.reduce_ack != 1)
			{
				target = (u32 *)(pData+pktBufOff+pExtAct->type4.pkt_buff_offset+seqackOffset);
				rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_GREACK, pExtAct->type4.seq_ack_reg_idx, &counter);
				*target = htonl(counter);
				DEBUG("[Dual] actionbit %d - use gre ACK# 0x%x", actBit, counter);
			}
		}
		else{
			u16 *target = NULL;
			target = (u16 *)(pData+pktBufOff+pExtAct->type4.pkt_buff_offset);
			rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_OUTER_IPV4ID, pExtAct->type4.seq_ack_reg_idx, &counter);
			*target = htons(counter);
			DEBUG("[Dual] actionbit %d - use IPv4 ID# 0x%x", actBit, counter);
		}
	}

	if(actBit == FB_EXTG_ACTBIT_5)
	{
		struct iphdr *iph = (struct iphdr *)(pData+pktBufOff);
		//iph->check = ip_fast_csum((unsigned char *)iph, iph->ihl);	// done by nic checksum offload
		DEBUG("[Dual] actionbit %d - ip checksum 0x%x", actBit, ntohs(iph->check));
	}

	if(actBit == FB_EXTG_ACTBIT_6)
	{
		struct iphdr *iph = (struct iphdr *)(pData+pktBufOff);
		struct udphdr *udph = (struct udphdr *)(pData+pktBufOff+(iph->ihl<<2));
		if(iph->protocol == 0x11)
		{
			//udph->check = inet_chksum_pseudo((u8 *)udph, udph->len, iph->saddr, iph->daddr, iph->protocol);	// done by nic checksum offload
		}
		else
		{
			WARNING("l4 protocol is not supported");
		}

		DEBUG("[Dual] actionbit %d - l4 checksum 0x%x", actBit, ntohs(udph->check));
	}
	return SUCCESS;

}


/*
* For downstream dual header learning: update tunnel info to path6 config
*/
int rtk_fc_ingress_outerHdrPathUpdate(rtk_fc_pktHdr_t *pPktHdr, uint32 path6Idx)
{
	rtk_fc_ret_t ret = RTK_FC_RET_OK;

	if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP){
		ret = _rtk_fc_flow_updatePPTP_by_igrPktHdr(pPktHdr, path6Idx);
	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_L2TP){
		ret = _rtk_fc_flow_updateL2TP_by_igrPktHdr(pPktHdr, path6Idx);
	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_DSLITE){
		ret = _rtk_fc_flow_updateDSLITE_by_igrPktHdr(pPktHdr, path6Idx);
	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_6RD && !fc_db.controlFuc.v6RDSwAcceleration_disable){
		ret = _rtk_fc_flow_update6RD_by_igrPktHdr(pPktHdr, path6Idx);
	}

#if defined(CONFIG_RTK_L34_CANDIDATE_FLOW)
	if(ret==SUCCESS)
		fc_db.flowTbl[path6Idx].candidateState = CANDIDATE_STATE_READY;
#endif

	return ret;
}
int rtk_fc_callId_to_interface_setting(rtk_fc_pktHdr_t *pPktHdr)
{
	int ret = SUCCESS;
	int i;
		
	for(i = 0; i< RTK_FC_TABLESIZE_INTF; i++)
	{
	
		//DEBUG("i = %d, netif: %s gateway ip = %pI4, packet sip =  %pI4", i, fc_db.netifTbl[i].dev->name, &fc_db.netifTbl[i].intf.gateway_ipv4_addr, &pPktHdr->iph->saddr);
		if(fc_db.netifHwTbl[i].intf.valid==FALSE)
			continue;
		
		if(fc_db.netifHwTbl[i].dualType != RTK_FC_DUALTYPE_PPTP)
			continue;
		
		if(pPktHdr->iph &&
			(fc_db.netifHwTbl[i].intf.gateway_ipv4_addr!=0) && (fc_db.netifHwTbl[i].intf.gateway_ipv4_addr==ntohl(pPktHdr->iph->saddr))) {
			
			//DEBUG("[PPTP]fc_db.netifTbl[%d].psTunnelInfo.call_id = %d, pPktHdr->tunnelInfo.pptp.grecallid = %d", i , fc_db.netifTbl[i].psTunnelInfo.call_id, pPktHdr->tunnelInfo.pptp.grecallid);
			if(fc_db.netifHwTbl[i].psTunnelInfo.call_id != pPktHdr->tunnelInfo.pptp.grecallid){
				TRACE("[PPTP]Find pptp device! netif hw Tbl[%d], ready to set call id = %d", i, pPktHdr->tunnelInfo.pptp.grecallid);
				fc_db.netifHwTbl[i].psTunnelInfo.call_id = pPktHdr->tunnelInfo.pptp.grecallid;
			}
			break;
		}


	}

	return ret;
}

/*
* For upstream dual header learning: basic path6 config and extra tag action setup
*/
int rtk_fc_egress_extratagDecision(rtk_fc_pktHdr_t *pPktHdr, uint32 netifIdx, uint32 *extraTagIdx)
{
	int ret = SUCCESS;
	int16 gmacL2Idx = 0;

	FC_PARAM_CHK((fc_db.netifHwTbl[netifIdx].intf.valid == FALSE), RTK_FC_RET_ERR_ENTRY_NOT_FOUND);

	if(fc_db.netifHwTbl[netifIdx].outerHdrExtratagIdx > 0){
		// already exist
		*extraTagIdx = fc_db.netifHwTbl[netifIdx].outerHdrExtratagIdx;
		DEBUG("extra tag idx %d", *extraTagIdx);
	}else{
		// create new extratag and decide dual header type for egress interface
		TRACE("extra tag was not exist, netif[%d] extratag_idx[%d]", netifIdx, fc_db.netifHwTbl[netifIdx].outerHdrExtratagIdx);
		fc_db.netifHwTbl[netifIdx].dualType = pPktHdr->dualHdrType;
#if 0	// user configuration should minus 40 bytes by itself. so FC change to follow interfce original setting.
		fc_db.netifHwTbl[netifIdx]->intf.intf_mtu -= EXTRATAG_LEN;				// outer header length
#endif

		if(_rtk_fc_lut_find(pPktHdr->eth->h_dest, &gmacL2Idx) != RTK_FC_RET_OK)
			return FAIL;

		if(fc_db.netifHwTbl[netifIdx].dualType == RTK_FC_DUALTYPE_PPTP)
			ret = _rtk_fc_flow_setupPPTP_by_egrPktHdr(pPktHdr, netifIdx, gmacL2Idx);

		else if(fc_db.netifHwTbl[netifIdx].dualType == RTK_FC_DUALTYPE_L2TP)
			ret = _rtk_fc_flow_setupL2TP_by_egrPktHdr(pPktHdr, netifIdx, gmacL2Idx);

		else if(fc_db.netifHwTbl[netifIdx].dualType == RTK_FC_DUALTYPE_DSLITE)
			ret = _rtk_fc_flow_setupDSLITE_by_egrPktHdr(pPktHdr, netifIdx, gmacL2Idx);
		else if(fc_db.netifHwTbl[netifIdx].dualType == RTK_FC_DUALTYPE_6RD)
		{
			if(fc_db.controlFuc.v6RDSwAcceleration_disable)
			{
				fc_db.netifHwTbl[netifIdx].outerHdrFlowIdx = SIGNED_INVALID;
				fc_db.netifHwTbl[netifIdx].outerHdrExtratagIdx = 0;
				return FAIL;
			}
			else
			{
				ret = _rtk_fc_flow_setup6RD_by_egrPktHdr(pPktHdr, netifIdx, gmacL2Idx);
			}
		}
#if 0	// user configuration should minus 40 bytes by itself. so FC change to follow interfce original setting.
		// update MTU config for 9607C, and update hw intf for dualType.
		//ret = RTK_RG_ASIC_NETIFTABLE_ADD(netifIdx, NULL);
#endif

		*extraTagIdx = fc_db.netifHwTbl[netifIdx].outerHdrExtratagIdx;

		_rtk_fc_netif_dummy_packet_set(netifIdx); //initial for PPTP/L2TP dummy packet
	}


	// error check
	FC_PARAM_CHK((fc_db.netifHwTbl[netifIdx].dualType != pPktHdr->dualHdrType), RTK_FC_RET_ERR_INVALID_PARAM);


	return ret;
}

int rtk_fc_mapt_accelerate(rtk_fc_ingress_data_t *pFcIngressData, rtk_fc_pktHdr_t *pPktHdr, uint32 igrNetifIdx, uint32 egrNetifIdx)
{
	int ret = FAIL;	//default return FAIL to forward as sw only
	
#if defined(CONFIG_FC_RTL8277C_SERIES)
	if(pPktHdr->ip6h){
		//egress with ipv6 header, upstream
		int aal_ret;
		int i,first_invalid=-1;
		unsigned int ihl_tc_flow = 0x60000000;	// version: 6
		unsigned char contenBuffer[64]={0};
		unsigned char *outerHdr = &contenBuffer[0];
		
		if((fc_db.netifHwTbl[egrNetifIdx].dualHdr_HWState&FC_DUALHDR_HW_CONTENT_BUF_SETUP)==0){
			TRACE("[MAPT][up] Gen outer header and add to SRAM content buffer");
			// Prepare IPv6(40) for MAPT sip translation
			// L3 header --
			*(uint32 *)(outerHdr + 0) = htonl(ihl_tc_flow);					//ver, tc, flowlabel
			//*(uint16 *)(pData + 4) = htons(0x0);							//Payload length
			//*(uint16 *)(outerHdr + 6) = htons(0x0400 | 0xff);				//Next header, Hop limit
			memcpy(outerHdr + 8, &pPktHdr->ip6h->saddr, IPV6_ADDR_LEN);		// src ip
			memcpy(outerHdr + 24, &pPktHdr->ip6h->daddr, IPV6_ADDR_LEN);	// dst ip
			aal_ret=aal_pe_dual_content_entry_add(RTK_FC_DUAL_CONTROL_IDX(egrNetifIdx), (l3pe_dual_tbl_entry_t *)outerHdr);
			if(aal_ret){
				DEBUG("[MAPT][up]Dual content add failed.(ret=%d)",aal_ret);
				return ret;
			}
			//indicate this hw interface had already setup.
			fc_db.netifHwTbl[egrNetifIdx].dualHdr_HWState|=FC_DUALHDR_HW_CONTENT_BUF_SETUP;
		}

		if(pFcIngressData->mapt_info.dst_style==FC_MAPT_STYLE_MAP0 || pFcIngressData->mapt_info.dst_style==FC_MAPT_STYLE_MAP){
			//to CE, dip translation need fmr rule.
			//get invalid FMR and add it to hw
			l3pe_dual_FMR_tbl_entry_t fmr_entry,exist_entry;
			uint8_t psid_bits_len = pFcIngressData->mapt_info.src_ea_len - (32 - pFcIngressData->mapt_info.src_v4_pref_len);

			memset(&fmr_entry,0,sizeof(l3pe_dual_FMR_tbl_entry_t));
			memset(&exist_entry,0,sizeof(l3pe_dual_FMR_tbl_entry_t));
			for(i=0;i<L3PE_DUAL_FMR_TBL_BUFF_SIZE_MAX;i++)
				fmr_entry.fmr_pre_addr[i]	= pPktHdr->ip6h->daddr.s6_addr[i];
			fmr_entry.fmr_psid_str_ost		= pFcIngressData->mapt_info.dst_psid_offset;
			fmr_entry.fmr_psid_end_ost		= pFcIngressData->mapt_info.dst_psid_offset+psid_bits_len;
			fmr_entry.fmr_psid_refill		= 1;
			fmr_entry.fmr_pref_len			= pFcIngressData->mapt_info.dst_v6_pref_len+pFcIngressData->mapt_info.src_ea_len;
			fmr_entry.fmr_drf_ver			= pFcIngressData->mapt_info.dst_style==FC_MAPT_STYLE_MAP0?0:1;
				
			//get invalid FMR and add it to hw
			for(i=0;i<L3PE_DUAL_FMR_TBL_ENTRY_MAX;i++){
				aal_ret=rtk_8277c_l3_pe_dual_fmr_get(i, &exist_entry);
				if(((ret == ASIC_RET_ENTRYNOTRSVD) || (ret == ASIC_RET_NOT_FOUND) || (ret == ASIC_RET_NULL_POINTER)) && first_invalid<0){
					first_invalid=i;
					continue;
				}else if(!memcmp(&exist_entry,&fmr_entry,sizeof(l3pe_dual_FMR_tbl_entry_t))){
					//compare with exist FMR, if match perfectly, use it
					break;
				}
			}
			if(i!=L3PE_DUAL_FMR_TBL_ENTRY_MAX){ //find exist one
				pPktHdr->fmr_idx=i;
			}else if(first_invalid){ //find invalid one
				pPktHdr->fmr_idx=first_invalid;
			}else
				return ret;
			aal_ret = rtk_8277c_l3_pe_dual_fmr_add(pPktHdr->fmr_idx, &fmr_entry);
			if(aal_ret){
				WARNING("[MAPT][up] Add dual fmr failed, ret = %d", aal_ret);
				return ret;
			}
			ret=SUCCESS;	//accelerate by hw
			TRACE("[MAPT][up] to CE ready, fmr_idx=%d",pPktHdr->fmr_idx);
		}else{
			//to BR, dip translation need dual control setting only.
			pPktHdr->fmr_idx=L3PE_DUAL_FMR_TBL_ENTRY_MAX;
			if((fc_db.netifHwTbl[egrNetifIdx].dualHdr_HWState&FC_DUALHDR_HW_CTRL_REG_SETUP)==0){
				TRACE("[MAPT][up] setup control register");
				if(_rtk_fc_dualHeader_ctrl_setupMAPT(RTK_FC_DUAL_CONTROL_IDX(egrNetifIdx), pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP0?0:1, pFcIngressData->mapt_info.dst_v6_pref_len)){
					DEBUG("[MAPT][up]Dual contrl entry add failed.");
					return ret;
				}
				//indicate this hw interface had already setup.
				fc_db.netifHwTbl[egrNetifIdx].dualHdr_HWState|=FC_DUALHDR_HW_CTRL_REG_SETUP;
			}
			ret=SUCCESS;	//accelerate by hw
			TRACE("[MAPT][up] to BR ready, fmr_idx=%d",pPktHdr->fmr_idx);
		}
	}else{
		int aal_ret;
		int i,first_invalid=-1;
		//otherwise, downstream
		if(pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP0 || pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP){
			//from CE sip translation have to lookup fmr table for longest prefix match, dip translation by dual control setting.
			l3pe_dual_FMR_tbl_entry_t fmr_entry,exist_entry;
			uint8_t psid_bits_len = pFcIngressData->mapt_info.src_ea_len - (32 - pFcIngressData->mapt_info.src_v4_pref_len);

			memset(&fmr_entry,0,sizeof(l3pe_dual_FMR_tbl_entry_t));
			memset(&exist_entry,0,sizeof(l3pe_dual_FMR_tbl_entry_t));
			for(i=0;i<L3PE_DUAL_FMR_TBL_BUFF_SIZE_MAX;i++)
				fmr_entry.fmr_pre_addr[i]	= pFcIngressData->mapt_info.src_v6_addr[i];
			fmr_entry.fmr_psid_str_ost		= pFcIngressData->mapt_info.src_psid_offset;
			fmr_entry.fmr_psid_end_ost		= pFcIngressData->mapt_info.src_psid_offset+psid_bits_len;
			fmr_entry.fmr_psid_refill		= 1;
			fmr_entry.fmr_pref_len			= pFcIngressData->mapt_info.src_v6_pref_len+pFcIngressData->mapt_info.src_ea_len;
			fmr_entry.fmr_drf_ver			= pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP0?0:1;
				
			//get invalid FMR and add it to hw
			for(i=0;i<L3PE_DUAL_FMR_TBL_ENTRY_MAX;i++){
				aal_ret=rtk_8277c_l3_pe_dual_fmr_get(i, &exist_entry);
				if(((ret == ASIC_RET_ENTRYNOTRSVD) || (ret == ASIC_RET_NOT_FOUND) || (ret == ASIC_RET_NULL_POINTER)) && first_invalid<0){
					first_invalid=i;
					continue;
				}else if(!memcmp(&exist_entry,&fmr_entry,sizeof(l3pe_dual_FMR_tbl_entry_t))){
					//compare with exist FMR, if match perfectly, use it
					break;
				}
			}
			if(i!=L3PE_DUAL_FMR_TBL_ENTRY_MAX){ //find exist one
				pPktHdr->fmr_idx=i;
			}else if(first_invalid){ //find invalid one
				pPktHdr->fmr_idx=first_invalid;
			}else
				return ret;
			aal_ret = rtk_8277c_l3_pe_dual_fmr_add(pPktHdr->fmr_idx, &fmr_entry);
			if(aal_ret){
				WARNING("[MAPT]Add dual fmr failed, ret = %d", aal_ret);
				return ret;
			}
		}else{
			pPktHdr->fmr_idx=L3PE_DUAL_FMR_TBL_ENTRY_MAX;
			//from BR need dual control setting for both sip and dip translation.
			if((fc_db.netifHwTbl[igrNetifIdx].dualHdr_HWState&FC_DUALHDR_HW_CTRL_REG_SETUP)==0){
				TRACE("[MAPT][down] setup control register");
				if(_rtk_fc_dualHeader_ctrl_setupMAPT(RTK_FC_DUAL_CONTROL_IDX(igrNetifIdx), pFcIngressData->mapt_info.dst_style==FC_MAPT_STYLE_MAP0?0:1, pFcIngressData->mapt_info.src_v6_pref_len)){
					DEBUG("[MAPT][down]Dual contrl entry add failed.");
					return ret;
				}
				//indicate this hw interface had already setup.
				fc_db.netifHwTbl[igrNetifIdx].dualHdr_HWState|=FC_DUALHDR_HW_CTRL_REG_SETUP;
			}
		}

		//setup downstream CLS
		if(fc_db.netifHwTbl[igrNetifIdx].dualHdr_ds_clsIdx==CA_UINT32_INVALID){
			rtk_fc_aclAndCf_reserved_dual_hdr_chk_t dual_hdr_chk = {0};
			int cls_ret;

			dual_hdr_chk.packet_type			= RTK_FC_CLS_DUAL_MAP_T;
			dual_hdr_chk.profile_idx			= 0; //WAN
			dual_hdr_chk.igr_intf_idx			= _rtk_fc_flow_intf_mapping_idx_get(igrNetifIdx);
			dual_hdr_chk.hash_tuple_idx			= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE;
			dual_hdr_chk.map_t.dst_ip[0]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[0]);
			dual_hdr_chk.map_t.dst_ip[1]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[1]);
			dual_hdr_chk.map_t.dst_ip[2]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[2]);
			dual_hdr_chk.map_t.dst_ip[3]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[3]);
			//check vlan
			dual_hdr_chk.map_t.vlan.vlan_chcek_en	= 1;
			dual_hdr_chk.map_t.vlan.has_stag 		= (pFcIngressData->ingressTagif&SVLAN_TAGIF) ? 1 : 0 ;
			dual_hdr_chk.map_t.vlan.svlan_id 		= pFcIngressData->srcSVlanId;
			dual_hdr_chk.map_t.vlan.has_ctag 		= (pFcIngressData->ingressTagif&CVLAN_TAGIF) ? 1 : 0 ;
			dual_hdr_chk.map_t.vlan.cvlan_id 		= pFcIngressData->srcCVlanId;

			cls_ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
			if(cls_ret != RTK_FC_RET_OK)
			{
				WARNING("[MAPT][down] add special ACL DUAL_HEADER_DS_CHECK failed!!! acl_ret = %x", cls_ret);
				return ret;
			}
			else
			{
				DEBUG("[MAPT][down] add special ACL DUAL_HEADER_DS_CHECK success, idx: %x, intf_idx=%d", dual_hdr_chk.rslt_idx, dual_hdr_chk.igr_intf_idx);
				fc_db.netifHwTbl[igrNetifIdx].dualHdr_ds_clsIdx = dual_hdr_chk.rslt_idx;
			}
			//add another CLS for TCP flag packets using different hash profile
			dual_hdr_chk.packet_type			= RTK_FC_CLS_DUAL_MAP_T_TCP_FLAG0;
			dual_hdr_chk.profile_idx			= 0; //WAN
			dual_hdr_chk.igr_intf_idx			= _rtk_fc_flow_intf_mapping_idx_get(igrNetifIdx);
			dual_hdr_chk.hash_tuple_idx			= RTK_8277C_FLOW_PROFILE_FLOW_5TUPLE_TCP_FLAG0;
			dual_hdr_chk.map_t.dst_ip[0]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[0]);
			dual_hdr_chk.map_t.dst_ip[1]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[1]);
			dual_hdr_chk.map_t.dst_ip[2]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[2]);
			dual_hdr_chk.map_t.dst_ip[3]		= ntohl(pFcIngressData->ingress_v6ip_info.s6_addr32[3]);
			//check vlan
			dual_hdr_chk.map_t.vlan.vlan_chcek_en	= 1;
			dual_hdr_chk.map_t.vlan.has_stag 		= (pFcIngressData->ingressTagif&SVLAN_TAGIF) ? 1 : 0 ;
			dual_hdr_chk.map_t.vlan.svlan_id 		= pFcIngressData->srcSVlanId;
			dual_hdr_chk.map_t.vlan.has_ctag 		= (pFcIngressData->ingressTagif&CVLAN_TAGIF) ? 1 : 0 ;
			dual_hdr_chk.map_t.vlan.cvlan_id 		= pFcIngressData->srcCVlanId;

			cls_ret = _rtk_rg_aclAndCfReservedRuleAddSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &dual_hdr_chk);
			if(cls_ret != RTK_FC_RET_OK)
			{
				WARNING("[MAPT][down] add TCP flag0 special ACL DUAL_HEADER_DS_CHECK failed!!! acl_ret = %x", cls_ret);
				fc_db.netifHwTbl[igrNetifIdx].dualHdr_ds_clsIdx = dual_hdr_chk.rslt_idx;
				assert_ok(_rtk_rg_aclAndCfReservedRuleDelSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &fc_db.netifHwTbl[igrNetifIdx].dualHdr_ds_clsIdx));
				fc_db.netifHwTbl[igrNetifIdx].dualHdr_ds_clsIdx = RTK_FC_UINT32_INVALID;
				return ret;
			}
			else
			{
				DEBUG("[MAPT][down] add TCP flag0 special ACL DUAL_HEADER_DS_CHECK success, idx: %x, intf_idx=%d", dual_hdr_chk.rslt_idx, dual_hdr_chk.igr_intf_idx);
				fc_db.netifHwTbl[igrNetifIdx].hwMaptInfo.ds_clsIdx_TCP_flag0 = dual_hdr_chk.rslt_idx;
			}
		}
		ret=SUCCESS;
		TRACE("[MAPT][down] from %s ready, fmr_idx=%d",(pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP0 || pFcIngressData->mapt_info.src_style==FC_MAPT_STYLE_MAP)?"CE":"BR",pPktHdr->fmr_idx);
	}
#endif

	return ret;
}

int rtk_fc_mapt_translate(struct rt_skbuff *rtskb, rtk_fc_pktHdr_t *pPktHdr, rtk_fc_tableFlow_t *pFlowTable)		//enlarge skb from IPv4 to IPv6 or shrink skb from IPv6 to IPv4
{
	uint32 tmpOffset=0;
	uint32 moveLen = 0;
	u8 *origDataStart = NULL;
	struct iphdr *iph;
	struct ipv6hdr *ip6h;

	if(!rtskb || !pPktHdr || !pFlowTable)
		return RT_ERR_RG_FAILED;

	if(pPktHdr->isFragment)
		return RT_ERR_RG_FAILED;

	//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//dump_packet(RTSKB_DATA(rtskb),RTSKB_LEN(rtskb),"before translate packet");

	origDataStart = RTSKB_DATA(rtskb);
	if(pPktHdr->iph){
		iph=pPktHdr->iph;
		tmpOffset = (unsigned char *)iph - origDataStart;
		moveLen = sizeof(struct ipv6hdr) - (iph->ihl<<2);		//push
		skb_push(RTSKB_SKB(rtskb), moveLen);
		memmove(RTSKB_DATA(rtskb),origDataStart,tmpOffset);
		ip6h=(struct ipv6hdr *)(RTSKB_DATA(rtskb)+tmpOffset);

		//copy original v4 header field to v6 header
		ip6h->version = 6;
		ip6h->priority = ((iph->tos>>4)&0xf);
		ip6h->flow_lbl[0] &= 0xf;
		ip6h->flow_lbl[0] |= (iph->tos&0xf)<<4;
		ip6h->flow_lbl[1] = 0;
		ip6h->flow_lbl[2] = 0;
		ip6h->payload_len = htons(ntohs(iph->tot_len) - (iph->ihl<<2));
		ip6h->nexthdr = iph->protocol;
		ip6h->hop_limit = iph->ttl;
		memcpy(&ip6h->saddr.s6_addr32[0],&fc_db.ipv6_mapt_mappingTbl[pFlowTable->maptInfo.sip_indirect_mapping_index].addr.s6_addr32[0],sizeof(struct in6_addr));
		memcpy(&ip6h->daddr.s6_addr32[0],&fc_db.ipv6_mapt_mappingTbl[pFlowTable->maptInfo.dip_indirect_mapping_index].addr.s6_addr32[0],sizeof(struct in6_addr));

		// update eth_type/ppp_protocol
		if(pPktHdr->ppph)
			*(unsigned short *)((unsigned char *)ip6h-2) = htons(0x0057);
		else
			*(unsigned short *)((unsigned char *)ip6h-2) = htons(ETH_P_IPV6);

		pPktHdr->iph=NULL;
		pPktHdr->ip6h=ip6h;
	}else if(pPktHdr->ip6h){
		ip6h=pPktHdr->ip6h;
		tmpOffset = (unsigned char *)ip6h - origDataStart;
		moveLen = sizeof(struct ipv6hdr) - sizeof(struct iphdr);		//pull
		skb_pull(RTSKB_SKB(rtskb), moveLen);

		iph=(struct iphdr *)(RTSKB_DATA(rtskb)+tmpOffset);

		//copy original v6 header field to v4 header
		iph->version = 4;
		iph->ihl = 5;
		iph->tos = (((ip6h->priority<<4)&0xf0)|((ip6h->flow_lbl[0]>>4)&0xf));
		iph->tot_len = htons(ntohs(ip6h->payload_len) + 20);
		iph->id = htons(jiffies&0xffff);
		iph->frag_off = htons(IP_DF);
		iph->ttl = ip6h->hop_limit;
		iph->protocol = ip6h->nexthdr;
		iph->saddr = htonl(pFlowTable->pFlowEntry->path5.in_src_ipv4_addr);
		iph->daddr = htonl(pFlowTable->pFlowEntry->path5.out_dst_ipv4_addr);
		iph->check = 0;
		iph->check = htons(inet_chksum((u8 *)iph,20));

		memmove(RTSKB_DATA(rtskb),origDataStart,tmpOffset);

		// update eth_type/ppp_protocol
		if(pPktHdr->ppph)
			*(unsigned short *)((unsigned char *)iph-2) = htons(0x0021);
		else
			*(unsigned short *)((unsigned char *)iph-2) = htons(ETH_P_IP);

		pPktHdr->iph=iph;
		pPktHdr->ip6h=NULL;
	}else
		return RT_ERR_RG_FAILED;

	RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));

	//reset header pointer
 	if(pPktHdr->eth) pPktHdr->eth = (struct ethhdr *)RTSKB_DATA(rtskb);
	if(pPktHdr->svh) pPktHdr->svh = (struct vlan_hdr *)((uint8 *)pPktHdr->svh + tmpOffset);
	if(pPktHdr->cvh) pPktHdr->cvh = (struct vlan_hdr *)((uint8 *)pPktHdr->cvh + tmpOffset);
	if(pPktHdr->ppph) pPktHdr->ppph = (struct pppoe_hdr *)((uint8 *)pPktHdr->ppph + tmpOffset);
	//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//dump_packet(RTSKB_DATA(rtskb),RTSKB_LEN(rtskb),"after translate packet");

	return (RT_ERR_RG_OK);
}

int rtk_fc_outerHdr_insert(struct rt_skbuff *rtskb, rtk_fc_pktHdr_t *pPktHdr, rtk_fc_tableFlow_t *pFlowTblEnt)
{
	int TUNN_tag_len=0;				//IP(20)+GRE(16)+PPP(4) for PPTP, IP(20)+UDP(8)+L2TP(8)+PPP(4) for L2TP
	int i = 0, tmpOffset = 0;

	int extraTagIdx = (pFlowTblEnt->pFlowEntry->path5.out_extra_tag_index - 1);

	rtk_fc_dualHdrtype_t wan_type = fc_db.netifHwTbl[pFlowTblEnt->pFlowEntry->path5.out_intf_idx].dualType;

	TUNN_tag_len = fc_db.extraTagList[extraTagIdx].contentLen;

	if(pPktHdr->eth) tmpOffset+=ETH_HLEN;
	if(pPktHdr->svh) tmpOffset+=4;
	if(pPktHdr->cvh) tmpOffset+=4;
	if(pPktHdr->ppph) tmpOffset+=8;

	if(_rtk_fc_skb_cow_head_and_pktHdr_update(RTSKB_SKB(rtskb), TUNN_tag_len, rtskb, pPktHdr)<0)
	{
		WARNING("skb head room is not enough..return without insert %s tag",wan_type==RTK_FC_DUALTYPE_PPTP?"PPTP":"L2TP");
	}
	else
	{
		//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//	dump_packet(skb->data,skb->len,"before adding outer hdr");

		if(wan_type==RTK_FC_DUALTYPE_PPTP)
		{
			TRACE("add PPTP tag");
			pPktHdr->dualHdrType = RTK_FC_DUALTYPE_PPTP; //G3 need this info to fill nic tx info
		}
		else if(wan_type==RTK_FC_DUALTYPE_L2TP)
		{
			TRACE("add L2TP tag");
			pPktHdr->dualHdrType = RTK_FC_DUALTYPE_L2TP; //G3 need this info to fill nic tx info
		}
		else if(wan_type==RTK_FC_DUALTYPE_DSLITE)
		{
			TRACE("add DSLite tag");
			pPktHdr->dualHdrType = RTK_FC_DUALTYPE_DSLITE; //G3 need this info to fill nic tx info
		}
		else if(wan_type==RTK_FC_DUALTYPE_6RD)
		{
			TRACE("add 6RD tag");
			pPktHdr->dualHdrType = RTK_FC_DUALTYPE_6RD; //G3 need this info to fill nic tx info
		}

		for(i=0; i<RTK_FC_TABLESIZE_EXTRATAG_ACTIONS; i++){
			if(fc_db.extraTagList[extraTagIdx].actions[i].type1.act_bit== FB_EXTG_ACTBIT_NOACTION)
				break;
			else
			{
				_rtk_fc_applyExtraTagAction(extraTagIdx, &fc_db.extraTagList[extraTagIdx].actions[i], rtskb, tmpOffset, pPktHdr);
			}
		}


		if (wan_type == RTK_FC_DUALTYPE_DSLITE && fc_db.controlFuc.dsliteV6TosFromV4)
		{
			struct ipv6hdr *ip6h = (struct ipv6hdr *)pPktHdr->outer_iph;
			uint8 tos1, tos2;

			tos1 = (pPktHdr->iph->tos >> 4) & 0x0F;
			tos2 = (pPktHdr->iph->tos) & 0x0F;

			ip6h->priority = tos1;
			ip6h->flow_lbl[0] &= 0xF;
			ip6h->flow_lbl[0] |= (tos2<<4);

			DSLITE("%pI4 => %pI4, tos=0x%x\n", &pPktHdr->iph->saddr, &pPktHdr->iph->daddr, pPktHdr->iph->tos);
		}
	

		if ((wan_type == RTK_FC_DUALTYPE_DSLITE) && (pFlowTblEnt->protoCtrl == FLOW_PROTO_CTRL_MAPE_ACC)){//MAP-E upstream may have different outHdr for FMR
			uint32 outif_idx = pFlowTblEnt->pFlowEntry->path5.out_intf_idx;
			uint16 fmr_idx = pFlowTblEnt->mapeInfo.mape_out_dst6_idx;
			rtk_fc_mape_dst6_update(pPktHdr, outif_idx, fmr_idx);
		}		

		//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//	dump_packet(skb->data,skb->len,"after adding outer hdr");
	}
	return (RT_ERR_RG_OK);
}


int rtk_fc_outerHdr_remove(struct rt_skbuff *rtskb, rtk_fc_pktHdr_t *pPktHdr)
{
	uint32 tmpOffset=0;
	uint32 rmLen = 0;
	u8 *innerDataStart = NULL;
	char innerV4Hdr = FALSE;


	//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//dump_packet(skb->data,skb->len,"before remove packet");

	if(pPktHdr->eth) tmpOffset+=ETH_HLEN;
	if(pPktHdr->svh) tmpOffset+=4;
	if(pPktHdr->cvh) tmpOffset+=4;
	if(pPktHdr->ppph) tmpOffset+=8;

	//since the length we remove must longer than 2 MACs address, we use memcpy to replace memmove for speed.
	if((pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP) || (pPktHdr->dualHdrType == RTK_FC_DUALTYPE_L2TP)){

		if(pPktHdr->iph){
			innerV4Hdr = TRUE;
			innerDataStart = (char *)pPktHdr->iph;
		}else{
			innerDataStart = (char *)pPktHdr->ip6h;
		}
	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_DSLITE){
		innerV4Hdr = TRUE;
		innerDataStart = (char *)pPktHdr->iph;
		pPktHdr->ip6h = NULL;
	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_6RD){
		innerDataStart = (char *)pPktHdr->ip6h;
	}
	else {
		WARNING("innerDataStart NULL!\n");
		return RT_ERR_RG_FAILED;
	}

	rmLen = innerDataStart - RTSKB_DATA(rtskb)- tmpOffset;

	memcpy(innerDataStart - tmpOffset, RTSKB_DATA(rtskb), tmpOffset);		// remove outer headers

	RTSKB_DATA(rtskb)+=rmLen;
	RTSKB_LEN(rtskb)-=rmLen;
	RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));

	// update eth_type/ppp_protocol
	if(innerV4Hdr){
		if(pPktHdr->ppph)
			*(unsigned short *)(innerDataStart-2) = htons(0x0021);
		else
			*(unsigned short *)(innerDataStart-2) = htons(ETH_P_IP);
	}else {
		if(pPktHdr->ppph)
			*(unsigned short *)(innerDataStart-2) = htons(0x0057);
		else
			*(unsigned short *)(innerDataStart-2) = htons(ETH_P_IPV6);
	}

	//reset header pointer
 	if(pPktHdr->eth) pPktHdr->eth = (struct ethhdr *)RTSKB_DATA(rtskb);
	if(pPktHdr->svh) pPktHdr->svh = (struct vlan_hdr *)((uint8 *)pPktHdr->svh + rmLen);
	if(pPktHdr->cvh) pPktHdr->cvh = (struct vlan_hdr *)((uint8 *)pPktHdr->cvh + rmLen);
	if(pPktHdr->ppph) pPktHdr->ppph = (struct pppoe_hdr *)((uint8 *)pPktHdr->ppph + rmLen);
	//if(fc_db.debugLevel & FC_DEBUG_LEVEL_DEBUG)
		//dump_packet(skb->data,skb->len,"after remove packet");

	return (RT_ERR_RG_OK);
}
int rtk_fc_l2Dual_outerHdr_remove(struct rt_skbuff *rtskb, uint8 outer_tag_len, rtk_fc_pktHdr_t *pPktHdr, rtk_fc_tableFlow_t *pFlowTable)
{
	//uint8 *pData = RTSKB_DATA(rtskb);
	
	RTSKB_DATA(rtskb)+=outer_tag_len;
	RTSKB_LEN(rtskb)-=outer_tag_len;
	RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));
	//pData = RTSKB_DATA(rtskb);

	if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].type == RTK_FC_L2DUAL_TYPE_VXLAN)
	{
		_rtk_fc_swPktHdrModify(rtskb, pPktHdr, pFlowTable);

		pPktHdr->skbLen_egress = RTSKB_LEN(rtskb);			// egress length is updated already by _rtk_fc_swPktHdrModify
	}
	return (SUCCESS);

}
int rtk_fc_l2Dual_outerHdr_insert(struct rt_skbuff *rtskb, rtk_fc_pktHdr_t *pPktHdr, rtk_fc_tableFlow_t *pFlowTable)
{
	uint8 *pData = RTSKB_DATA(rtskb);

	if(atomic_read(&fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].refCount)==0)
	{
		WARNING("Invalid l2DualTbl index");
		return FAIL;
	}

	if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].action != RTK_FC_L2DUAL_ACT_ADD)
	{
		WARNING("Invalid l2DualTbl type %d", fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].action != RTK_FC_L2DUAL_ACT_ADD);
		return FAIL;
	}

	if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].type == RTK_FC_L2DUAL_TYPE_VXLAN)
	{
		if(_rtk_fc_skb_cow_head_and_pktHdr_update(RTSKB_SKB(rtskb), fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_tag_len, rtskb, pPktHdr)<0)
		{
			WARNING("skb head room is not enough");
			return FAIL;
		}

		RTK_FC_HOOK_PS_SKB_SKB_PUSH(RTSKB_SKB(rtskb), fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_tag_len, &pData);
		memcpy(pData, fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.contentBuffer, fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_tag_len);

		if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_ppph_tag_off)
		{
			DEBUG("[VXLAN]Outer pppoe offset = %d",fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_ppph_tag_off);
			pPktHdr->ppph = (struct pppoe_hdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_ppph_tag_off);
			pPktHdr->ppph->length = htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_ppph_tag_off - PPPOE_SES_HLEN +2); // payload length includes p2p protocol(2 bytes)
		}

		RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));
		pPktHdr->outer_eth = (struct ethhdr *)(pData);
		if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_is_ipv6)
		{
			pPktHdr->outer_ip6h = (struct ipv6hdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_iph_tag_off);
			//The ipv6 header is not included in payload_len
			pPktHdr->outer_ip6h->payload_len =  htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_udph_tag_off);
		}
		else
		{
			pPktHdr->outer_iph = (struct iphdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_iph_tag_off);
			pPktHdr->outer_iph->tot_len =  htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_iph_tag_off);

		}

		pPktHdr->outer_udph = (struct udphdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_udph_tag_off);
		pPktHdr->outer_udph->len =	htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.outer_udph_tag_off);
		
	
		if(pFlowTable->pFlowEntry->path5.out_svid_format_act == FALSE)
		{
			if(pPktHdr->svh)
			{
				rtk_fc_vxlan_inner_skbVlanTag_remove(rtskb, pPktHdr, TRUE, pFlowTable->l2Dual_table_info.l2Dual_table_index);
			}	
		}

		// For inner cvlan
		if(pFlowTable->pFlowEntry->path5.out_cvid_format_act && 
			fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.inner_cvlan_tagif ==1 )
		{
			uint16 vlanTCI = 0;
			vlanTCI = fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.inner_cvlan_cvid | ( fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.inner_cvlan_pri<<VLAN_PRIO_SHIFT | (fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].vxlan_us.inner_cvlan_cfi<<VLAN_CFI_SHIFT));
			if(pPktHdr->cvh)
			{
				pPktHdr->cvh->h_vlan_TCI = htons(vlanTCI);
			}
			else
			{
				rtk_fc_vxlan_inner_skbVlanTag_insert(rtskb, pPktHdr, vlanTCI, FALSE, pFlowTable->l2Dual_table_info.l2Dual_table_index);
			}
		}

		RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));
		
	}
	else if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].type == RTK_FC_L2DUAL_TYPE_GRE_ETH_BR)
	{
		if(_rtk_fc_skb_cow_head_and_pktHdr_update(RTSKB_SKB(rtskb), fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_tag_len, rtskb, pPktHdr)<0)
		{
			WARNING("skb head room is not enough");
			return FAIL;
		}

		RTK_FC_HOOK_PS_SKB_SKB_PUSH(RTSKB_SKB(rtskb), fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_tag_len, &pData);
		memcpy(pData, fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.contentBuffer, fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_tag_len);

		if(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_pppoe_tag_off)
		{
			DEBUG("[GRE_BR_ETH] Outer pppoe offset = %d",fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_pppoe_tag_off);
			pPktHdr->ppph = (struct pppoe_hdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_pppoe_tag_off);
			pPktHdr->ppph->length = htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_pppoe_tag_off - PPPOE_SES_HLEN +2); // payload length includes p2p protocol(2 bytes)
		}

		RTK_FC_HOOK_PS_SKB_SKB_RESET_MAC_HEADER(RTSKB_SKB(rtskb));
		pPktHdr->outer_eth = (struct ethhdr *)(pData);
		pPktHdr->outer_iph = (struct iphdr *)(pData+fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_ip_tag_off);
		pPktHdr->outer_iph->tot_len =  htons(RTSKB_LEN(rtskb) - fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.outer_ip_tag_off);
		pPktHdr->outer_iph->id = htons(fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.upstream_outerIpId);
		fc_db.l2DualTbl[pFlowTable->l2Dual_table_info.l2Dual_table_index].greEthBr_us.upstream_outerIpId++;
	}
	return (SUCCESS);
}

int rtk_fc_getPPTPEgressDevIndex(rtk_fc_pktHdr_t *pPktHdr)
{
	int i;
	int index=FAIL;

	if(pPktHdr->eth==NULL || pPktHdr->iph==NULL)
		return index;
	
	for(i = 0; i < RTK_FC_TABLESIZE_INTF; i++)
	{
		if(fc_db.netifHwTbl[i].intf.valid==FALSE)
			continue;
		
		if(fc_db.netifHwTbl[i].dualType != RTK_FC_DUALTYPE_PPTP)
			continue;
		
		if(!memcmp(&fc_db.netifHwTbl[i].intf.gateway_mac_addr, pPktHdr->eth->h_source, ETH_ALEN) &&
			(fc_db.netifHwTbl[i].outerHdrFlowIdx != SIGNED_INVALID) &&
			//(fc_db.flowTbl[fc_db.netifHwTbl[i]->outerHdrFlowIdx].pFlowEntry->path6.in_gre_call_id == pPktHdr->tunnelInfo.pptp.grecallid) &&
			(fc_db.flowTbl[fc_db.netifHwTbl[i].outerHdrFlowIdx].pFlowEntry->path6.in_src_ipv4_addr == ntohl(pPktHdr->iph->daddr) &&
			(pPktHdr->tunnelInfo.pptp.grecallid == fc_db.netifHwTbl[i].psTunnelInfo.call_id))
		)
		{
			index=i;
			break;
		}
	}
	return index;
}

/*
* Sync PPTP/L2TP IPID/GRESEQ/ACK
*/
int rtk_fc_egress_tunnelInfoSync(struct rt_skbuff *rtskb, rtk_fc_pktHdr_t *pPktHdr)
{
	uint32 tmpval = 0, i;
	uint8 frag = FALSE, frag_first = FALSE, frag_last = FALSE;
	struct net_device *dstDev=NULL;
	int baseIntfIdx = FAIL, egressIntfIdx = FAIL;
	uint16 fragIPID = 0;

	if(pPktHdr->outer_iph && (RTSKB_FCIGRDATA(rtskb)->isDualHeader == FALSE) && (pPktHdr->dualHdrType == RTK_FC_DUALTYPE_GRE_ETH_BR))
	{
		for(i = RTK_FC_L2DUAL_TABLE_START_INDEX ; i < RTK_FC_TABLESIZE_L2DUAL_TABLE; i++)
		{
			if((atomic_read(&fc_db.l2DualTbl[i].refCount)!=0) && (fc_db.l2DualTbl[i].type == RTK_FC_L2DUAL_TYPE_GRE_ETH_BR) && (fc_db.l2DualTbl[i].action == RTK_FC_L2DUAL_ACT_ADD))
			{
				if((fc_db.l2DualTbl[i].greEthBr_us.pUpstreamIpHdr->saddr == pPktHdr->outer_iph->saddr) && (fc_db.l2DualTbl[i].greEthBr_us.pUpstreamIpHdr->daddr ==  pPktHdr->outer_iph->daddr))
				{
					fc_db.l2DualTbl[i].greEthBr_us.upstream_outerIpId = (ntohs(pPktHdr->outer_iph->id) + 1);
				}
			}
		}
	}


	// common field: IPID, the packet may be dual hdr or single hdr.
	if(pPktHdr->outer_iph && ((pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP) || (pPktHdr->dualHdrType == RTK_FC_DUALTYPE_L2TP)) &&
		ip_is_fragment(pPktHdr->outer_iph) )
	{
		//2 Dual hdr: 1st frag pkt
		// force using outer ip to find egress intf (the fragment is made by outer hdr).
		struct iphdr *pIphTmp = pPktHdr->outer_iph;

		 frag = TRUE;
		 frag_first = FALSE; frag_last = FALSE;

		// find the inner interface
		dstDev = rtk_fc_getEgressDev(pPktHdr, rtskb, pPktHdr->fwdType, RTK_FC_FLOW_DIRECTION_UPSTREAM);
		egressIntfIdx = rtk_fc_decideNetifIndex(RTK_FC_DEV_TYPE_EGRESS, dstDev, rtskb, pPktHdr);

		if((dstDev==NULL) || (egressIntfIdx==FAIL) ||
			((fc_db.netifHwTbl[egressIntfIdx].dualType!=RTK_FC_DUALTYPE_PPTP)&&(fc_db.netifHwTbl[egressIntfIdx].dualType!=RTK_FC_DUALTYPE_L2TP))){
			DEBUG("incorrect/unnecessary egress interface %d", egressIntfIdx);
			return (FAILED);
		}

		// replace
		pPktHdr->iph = pPktHdr->outer_iph;
		// find the outer interface
		dstDev = rtk_fc_getEgressDev(pPktHdr, rtskb, pPktHdr->fwdType, RTK_FC_FLOW_DIRECTION_UPSTREAM);
		baseIntfIdx = rtk_fc_decideNetifIndex(RTK_FC_DEV_TYPE_EGRESS, dstDev, rtskb, pPktHdr);
		// restore
		pPktHdr->iph = pIphTmp;

		if((dstDev==NULL) || (baseIntfIdx==FAIL)){
			WARNING("incorrect egress interface %d", baseIntfIdx);
			return (FAILED);
		}
		if(!(pPktHdr->outer_iph->frag_off & htons(IP_OFFSET)) && (pPktHdr->outer_iph->frag_off & htons(IP_MF))){
			frag_first = TRUE;
		}else {
			// error case
			WARNING("dual hdr fmt but without ipmf flag");
			return (FAILED);
		}
		DEBUG("[FRAG] %s packet, ori ipid: %d, skb len: %d, egress to intf[%d] base on intf[%d]",
			(pPktHdr->dualHdrType== RTK_FC_DUALTYPE_PPTP)?"PPTP":"L2TP", ntohs(pPktHdr->outer_iph->id), RTSKB_LEN(rtskb), egressIntfIdx, baseIntfIdx);

		// Get new identifier
		fc_db.netifHwTbl[egressIntfIdx].psTunnelInfo.ipid = ntohs(pPktHdr->outer_iph->id);
		ASSERT_EQ(rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_OUTER_IPV4ID, egressIntfIdx, &tmpval), SUCCESS);
		pPktHdr->outer_iph->id = htons(tmpval);

		// maintain fragment ipid
		if(frag_first){
			// record the ipid because of first fragment packet
			_rtk_fc_fragIPID_save(baseIntfIdx, &(fc_db.netifHwTbl[baseIntfIdx]), fc_db.netifHwTbl[egressIntfIdx].psTunnelInfo.ipid, ntohs(pPktHdr->outer_iph->id));

		}

		DEBUG("[FRAG] %s frag packet, use ipid: %d, skb len: %d",
			(pPktHdr->dualHdrType== RTK_FC_DUALTYPE_PPTP)?"PPTP":"L2TP", ntohs(pPktHdr->outer_iph->id), RTSKB_LEN(rtskb));

	}
	else if((pPktHdr->outer_iph==NULL) && (pPktHdr->iph!=NULL) && ip_is_fragment(pPktHdr->iph)){
		//2 Dual hdr: 2nd~N frag pkt
		frag = TRUE;
		dstDev = rtk_fc_getEgressDev(pPktHdr, rtskb, pPktHdr->fwdType, RTK_FC_FLOW_DIRECTION_UPSTREAM);
		egressIntfIdx = rtk_fc_decideNetifIndex(RTK_FC_DEV_TYPE_EGRESS, dstDev, rtskb, pPktHdr);
		baseIntfIdx = egressIntfIdx;

		if((dstDev==NULL) || (egressIntfIdx==FAIL)){
			DEBUG("incorrect/unnecessary egress interface %d", egressIntfIdx);
			return (FAILED);
		}
		if(!(pPktHdr->iph->frag_off & htons(IP_OFFSET)) && (pPktHdr->iph->frag_off & htons(IP_MF))){
			frag_first = TRUE;
		}else if((pPktHdr->iph->frag_off & htons(IP_OFFSET)) && !(pPktHdr->iph->frag_off & htons(IP_MF))){
			frag_last = TRUE;
 		}

		if(frag_first){
			// from wan to lan, it may fragment packet and without outer header, ignore such case.
		}
		if(!frag_last){
			// get the ipid record
			if(_rtk_fc_fragIPID_get(baseIntfIdx, &(fc_db.netifHwTbl[baseIntfIdx]), ntohs(pPktHdr->iph->id), &fragIPID) == SUCCESS)
				pPktHdr->iph->id = htons(fragIPID);
		}else{
			// remove the ipid record because of last fragment packet
			if(_rtk_fc_fragIPID_delete(baseIntfIdx, &(fc_db.netifHwTbl[baseIntfIdx]), ntohs(pPktHdr->iph->id), &fragIPID) == SUCCESS)
				pPktHdr->iph->id = htons(fragIPID);
		}

		DEBUG("[FRAG] single hdr packet, last_frag: %d, use ipid: %d, skb len: %d", frag_last, ntohs(pPktHdr->iph->id), RTSKB_LEN(rtskb));

	}
	else if(pPktHdr->outer_iph && ((pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP) || (pPktHdr->dualHdrType == RTK_FC_DUALTYPE_L2TP))){
		//2 Dual hdr: not frag pkt
		// no fragment, consider normal dual hdr ipid sync.
		dstDev = rtk_fc_getEgressDev(pPktHdr, rtskb, pPktHdr->fwdType, RTK_FC_FLOW_DIRECTION_UPSTREAM);
		egressIntfIdx = rtk_fc_decideNetifIndex(RTK_FC_DEV_TYPE_EGRESS, dstDev, rtskb, pPktHdr);

		if((dstDev==NULL) || (egressIntfIdx==FAIL) ||
			((fc_db.netifHwTbl[egressIntfIdx].dualType!=RTK_FC_DUALTYPE_PPTP)&&(fc_db.netifHwTbl[egressIntfIdx].dualType!=RTK_FC_DUALTYPE_L2TP))){
			DEBUG("incorrect/unnecessary egress interface %d", egressIntfIdx);
			return (FAILED);
		}

		fc_db.netifHwTbl[egressIntfIdx].psTunnelInfo.ipid = ntohs(pPktHdr->outer_iph->id);
		ASSERT_EQ(rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_OUTER_IPV4ID, egressIntfIdx, &tmpval), SUCCESS);
		pPktHdr->outer_iph->id = htons(tmpval);

		DEBUG("[FRAG] %s not frag packet, use ipid: %d, skb len: %d", (pPktHdr->dualHdrType== RTK_FC_DUALTYPE_PPTP)?"PPTP":"L2TP", ntohs(pPktHdr->outer_iph->id), RTSKB_LEN(rtskb));
	}

	//if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP){
	if((pPktHdr->dualHdrType == RTK_FC_DUALTYPE_PPTP) || (pPktHdr->tunnelInfo.tunnelTag & (GRESEQ_TAGIF|GREACK_TAGIF))){

		if(egressIntfIdx == FAIL)
			egressIntfIdx=rtk_fc_getPPTPEgressDevIndex(pPktHdr);

		if(egressIntfIdx == FAIL)
			return (FAILED);	

		// gre seq/ack
		//if(pPktHdr->tunnelInfo.pptp.greseqTagif){
		if((pPktHdr->tunnelInfo.tunnelTag & GRESEQ_TAGIF) && (egressIntfIdx >= 0)){
			fc_db.netifHwTbl[egressIntfIdx].psTunnelInfo.tx_greseq = pPktHdr->tunnelInfo.pptp.greseq;

			rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_GRESEQ, egressIntfIdx, &tmpval);
			DEBUG("[Dual] PPTP GRE seq: hw(%d) ps(%d)", tmpval, pPktHdr->tunnelInfo.pptp.greseq);
			if(pPktHdr->tunnelInfo.pptp.greseq > tmpval){
				rtk_fc_dualHdrInfo_set(RTK_FC_FB_DUALHDR_GRESEQ, egressIntfIdx, pPktHdr->tunnelInfo.pptp.greseq+1);
			}
			else{
				*pPktHdr->tunnelInfo.pptp.pGRESeq = htonl(tmpval);
			}
		}
		//if(pPktHdr->tunnelInfo.pptp.greackTagif){
		if((pPktHdr->tunnelInfo.tunnelTag & GREACK_TAGIF) && (egressIntfIdx >= 0)){
			fc_db.netifHwTbl[egressIntfIdx].psTunnelInfo.tx_greack = pPktHdr->tunnelInfo.pptp.greack;
			rtk_fc_dualHdrInfo_get(RTK_FC_FB_DUALHDR_GREACK, egressIntfIdx, &tmpval);
			*pPktHdr->tunnelInfo.pptp.pGREAck = htonl(tmpval);
			DEBUG("[Dual] PPTP GRE ack: hw(%d) ps(%d)", tmpval, pPktHdr->tunnelInfo.pptp.greack);
		}

	}
	else if(pPktHdr->dualHdrType == RTK_FC_DUALTYPE_L2TP){
		// no requirement
	}
	return (SUCCESS);
}


/*
* To delete PPTP/L2TP/DSLite related configuration
* 1. Extra tag actions and content buffer
* 2. Path6 flow entry
*/
int rtk_fc_dualConfig_del(int hwIntfIdx)
{
	rtk_fc_hwTableNetif_t *pNetif = &(fc_db.netifHwTbl[hwIntfIdx]);

	FC_PARAM_CHK(pNetif->intf.valid==FALSE, RTK_FC_RET_ERR_ENTRY_NOT_FOUND);

	TRACE("clear intf config: dual type[%d] path6_idx[%d] extratag_idx[%d] ", pNetif->dualType, pNetif->outerHdrFlowIdx, pNetif->outerHdrExtratagIdx);

	// Del flow entra tag action
	if(pNetif->outerHdrExtratagIdx>0)
	{
		ASSERT_EQ(RTK_RG_ASIC_EXTRATAGACTION_DEL(pNetif->outerHdrExtratagIdx, 0), RT_ERR_RG_OK);
		ASSERT_EQ(_rtk_fc_extraTagActionList_clear(pNetif->outerHdrExtratagIdx), RT_ERR_RG_OK);
		//TODO: clear extra tag content buffer
	}

	// Del flow path6 entry (MAP-T do not have path6 flow)
	if(pNetif->dualType && pNetif->dualType!=RTK_FC_DUALTYPE_MAPT && pNetif->outerHdrFlowIdx!=-1)
		rtk_fc_flow_delete(pNetif->outerHdrFlowIdx);		// delete and ignore return fail (e.g. flow not found) because flow entry may be deleded already.

#if defined(CONFIG_FC_RTL8277C_SERIES)
	//clear DualHeader control and content register
	aal_pe_dual_control_entry_del(RTK_FC_DUAL_CONTROL_IDX(hwIntfIdx));

	do{
		int i;
		for(i=pNetif->hwEntryNum;i>0;i--)
			aal_pe_dual_content_entry_del(RTK_FC_DUAL_CONTROL_IDX(hwIntfIdx)+pNetif->hwEntryNum-i);
	}while(0);

	//Del downstream dual header CLS
	if(pNetif->dualHdr_ds_clsIdx != RTK_FC_UINT32_INVALID)
	{
		assert_ok(_rtk_rg_aclAndCfReservedRuleDelSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &pNetif->dualHdr_ds_clsIdx));
		pNetif->dualHdr_ds_clsIdx = RTK_FC_UINT32_INVALID;
	}
	if(pNetif->dualType && pNetif->dualType==RTK_FC_DUALTYPE_MAPT && pNetif->hwMaptInfo.ds_clsIdx_TCP_flag0 != RTK_FC_UINT32_INVALID)
	{
		assert_ok(_rtk_rg_aclAndCfReservedRuleDelSpecial(RTK_8277C_CLS_TYPE_DUAL_HEADER_DS_CHECK, &pNetif->hwMaptInfo.ds_clsIdx_TCP_flag0));
		pNetif->hwMaptInfo.ds_clsIdx_TCP_flag0 = RTK_FC_UINT32_INVALID;
	}
#endif

	return (RTK_FC_RET_OK);
}

/*
* init extra tag action list
*/
int rtk_fc_dualInfoInit(void)
{
	int i = 0;

	for(i=RTK_FC_TABLESIZE_EXTRATAG_LISTMIN; i<RTK_FC_TABLESIZE_EXTRATAG_LISTMAX; i++){
		_rtk_fc_extraTagActionList_clear(i);
	}

	return SUCCESS;
}


/*=====================================================
* MAP-E:
*	In addition to DMR, MAP-E can also support FMR at the same time. 
* 		1. (DMR) CE1 -> BR
*		2. (FMR) CE1 -> CE2
*			2.1 CE1 -> CE2(Support FMR) -> CE1
*			2.2 CE1 -> CE2(Not Support FMR) -> BR -> CE1
*	- For case 1 and 2, the IPv6 dst addresses of upstream packets forwarded through DMR/FMR are different.
*	So the IPv6 dstAddr needs to be updated after inserting thr outHdr for MAP-E upstream .
*	- For case 2.2, the IPv6 dstAddr of upstream packets is different from the IPv6 srcAddr of downstream
*	So the check of IPv6 srcAddr in downstream should be bypassed
*=====================================================
*/

/*
* Init the MAP-E IPv6 dst addresses
*/
int rtk_fc_mape_dst6Info_init(void){
	int i=0, j=0;
	for (i = 0; i < RTK_FC_TABLESIZE_INTF; i++){
		memset(&(fc_db.mapeDst6s[i]), 0x00, sizeof(rtk_fc_mape_dst6_t));		
		INIT_LIST_HEAD(&(fc_db.mapeDst6s[i].free_list));
		INIT_LIST_HEAD(&(fc_db.mapeDst6s[i].used_list));
		for (j = 0; j < RTK_FC_MAPE_FMR_DST6_MAX_NUMS; j++){
			fc_db.mapeDst6s[i].dst6_info[j].index = j;
			list_add_tail(&(fc_db.mapeDst6s[i].dst6_info[j].list), &(fc_db.mapeDst6s[i].free_list));
		}
	}
	return SUCCESS;
}

/*
* Use pFlowTable->mapeInfo.mape_flow6 field to distinguish the downstream between Ds-Lite and MAP-E
*/
void rtk_fc_mark_mape_flow6(rtk_fc_pktHdr_t *pPktHdr, uint32 flowIdx){
	rtk_fc_tableFlow_t *pFlowTable = &fc_db.flowTbl[flowIdx];
	pFlowTable->mapeInfo.mape_flow6 = 0;

	if (pPktHdr->fwdType == RTK_FC_FWDTYPE_NAPT){
		pFlowTable->mapeInfo.mape_flow6 = 1;
	}
	
}

int rtk_fc_is_mape_upstream_flow5(uint32 flowIdx){
	rtk_fc_tableFlow_t *pFlowTable = &fc_db.flowTbl[flowIdx];
	rtk_rg_asic_path5_entry_t *pFlowPath5 = (rtk_rg_asic_path5_entry_t *)pFlowTable->pFlowEntry;

	if (fc_db.netifHwTbl[pFlowPath5->out_intf_idx].dualType != RTK_FC_DUALTYPE_DSLITE)
	{
		//TRACE("dualType != RTK_FC_DUALTYPE_DSLITE");
		return FALSE;
	}
	if (pFlowPath5->in_path != FB_PATH_5)
	{
		//TRACE("pFlowPath5->in_path != FB_PATH_5");
		return FALSE;
	}
	if (pFlowPath5->out_l4_direction != 1) //upstream
	{
		//TRACE("pFlowPath5->out_l4_direction != 1");
		return FALSE;
	}
	if (pFlowPath5->out_l4_act != 1) //napt
	{
		//TRACE("pFlowPath5->out_l4_act != 1");
		return FALSE;
	}
	return (TRUE);
}

/*
* Record the IPv6 dst address carried in pPktHdr, when adding the MAP-E upstream flow5 in egress.
* And hook the index of dst6 address to the corresponding pFlowTable->mapeInfo.mape_out_dst6_idx.
*/
int rtk_fc_mape_dst6_decision(rtk_fc_pktHdr_t *pPktHdr, uint32 flowIdx){
	rtk_fc_mape_fmr_dst6_t *mape_fmr;
	rtk_fc_tableFlow_t *pFlowTable = &fc_db.flowTbl[flowIdx];
	rtk_rg_asic_path5_entry_t *pFlowPath5 = (rtk_rg_asic_path5_entry_t *)pFlowTable->pFlowEntry;		
	uint32 netif_idx = pFlowPath5->out_intf_idx;
	struct list_head *p=NULL, *n=NULL;

	if (!pPktHdr || !(pPktHdr->ip6h))
		return (SUCCESS);
	
	if (fc_db.mapeDst6s[netif_idx].in_used == 0)
		fc_db.mapeDst6s[netif_idx].in_used = 1;

	if (!list_empty(&(fc_db.mapeDst6s[netif_idx].used_list))){	
		list_for_each_safe(p, n, &(fc_db.mapeDst6s[netif_idx].used_list)) {
			mape_fmr = list_entry(p, rtk_fc_mape_fmr_dst6_t, list);
			if (memcmp(&(mape_fmr->d_addr6), &(pPktHdr->ip6h->daddr), sizeof(struct in6_addr)) == 0){
				mape_fmr->ref_cnt++;
	
				pFlowTable->mapeInfo.mape_out_dst6_idx = mape_fmr->index;
	
				DEBUG("[flowIdx=%d] netif_idx=%d, fmr_idx=%d, mape_fmr->ref_cnt=%d\n", flowIdx, netif_idx, mape_fmr->index, mape_fmr->ref_cnt);

				return (SUCCESS);
			}else{
				continue;
			}
		}		
	}
	
	if (!list_empty(&(fc_db.mapeDst6s[netif_idx].free_list))){
		mape_fmr = list_first_entry(&(fc_db.mapeDst6s[netif_idx].free_list), rtk_fc_mape_fmr_dst6_t, list);	
		list_del(&(mape_fmr->list));
		list_add_tail(&(mape_fmr->list), &(fc_db.mapeDst6s[netif_idx].used_list));
		
		memcpy(&(mape_fmr->d_addr6), &(pPktHdr->ip6h->daddr), sizeof(struct in6_addr));
		mape_fmr->ref_cnt = 1;

		pFlowTable->mapeInfo.mape_out_dst6_idx = mape_fmr->index;

		DEBUG("[flowIdx=%d] netif_idx=%d, fmr_idx=%d, mape_fmr->ref_cnt=%d\n", flowIdx, netif_idx, mape_fmr->index, mape_fmr->ref_cnt);

		return (SUCCESS);
	}	
	return (FAILED);	
}

/*
* After inserting the out header into the MAP-E upstream packet in shortcut, 
* update the IPv6 dst address based on the pFlowTable->mapeInfo.mape_out_dst6_idx.
*/
void rtk_fc_mape_dst6_update(rtk_fc_pktHdr_t *pPktHdr, uint32 outif_idx, uint16 fmr_idx){
	struct ipv6hdr *ip6h = (struct ipv6hdr *)pPktHdr->outer_iph;
	rtk_fc_mape_fmr_dst6_t *mape_fmr;
	
	if (outif_idx >= RTK_FC_TABLESIZE_INTF)
		return;
		
	if (fmr_idx >= RTK_FC_MAPE_FMR_DST6_MAX_NUMS)
		return;

	mape_fmr = &(fc_db.mapeDst6s[outif_idx].dst6_info[fmr_idx]);
	if (mape_fmr->ref_cnt > 0)
		memcpy(&ip6h->daddr, &mape_fmr->d_addr6, sizeof(struct in6_addr));	
}


static void _rtk_fc_mape_dst6_refcnt_put(uint32 netif_idx, uint16 fmr_idx){
	rtk_fc_mape_fmr_dst6_t *mape_fmr;
	struct list_head *list;

	mape_fmr = &(fc_db.mapeDst6s[netif_idx].dst6_info[fmr_idx]);
	mape_fmr->ref_cnt--;

	DEBUG("netif_idx=%d, fmr_idx=%d, mape_fmr->ref_cnt=%u\n", netif_idx, fmr_idx, mape_fmr->ref_cnt);	
	if (mape_fmr->ref_cnt <= 0)	{
		list = &(mape_fmr->list);
		list_del(list);
		list_add_tail(list, &(fc_db.mapeDst6s[netif_idx].free_list));
	}
}

/*
* When the flow5 which representing the MAP-E upstream timeout,
* the ref_cnt of IPv6 dst address referenced by it needs to be decreased
*/
int rtk_fc_mape_dst6_refcnt_put(uint32 flowIdx){
	rtk_fc_tableFlow_t *pFlowTable = &fc_db.flowTbl[flowIdx];
	rtk_rg_asic_path5_entry_t *pFlowPath5 = (rtk_rg_asic_path5_entry_t *)pFlowTable->pFlowEntry;

	//if (!(pFlowPath5->out_intf_idx>=0 && pFlowPath5->out_intf_idx<RTK_FC_TABLESIZE_INTF))
		//return FAILED;
	
	if (pFlowTable->mapeInfo.mape_out_dst6_idx >= RTK_FC_MAPE_FMR_DST6_MAX_NUMS)
		return FAILED;
	
	DEBUG("[flowIdx=%d] ", flowIdx);

	_rtk_fc_mape_dst6_refcnt_put(pFlowPath5->out_intf_idx, pFlowTable->mapeInfo.mape_out_dst6_idx);
	return (SUCCESS);
}

#if defined(CONFIG_FC_RTL8277C_SERIES)
rtk_fc_ret_t rtk_fc_dual_control_content_set6RD(uint32 dual_content_tbl_idx, rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo, u8 *outer_header_content, int outer_header_size, rtk_fc_pktHdr_t *pPktHdr)
{
	rtk_fc_ret_t ret=RTK_FC_RET_OK;
	//l3pe_dual_tbl_entry_t dual_buffer_entry;
	//int aal_ret, offset = 0;

	if(pG3IgrExtraInfo->dualHdrType == RTK_FC_DUALTYPE_6RD)
	{
		// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]+IP header(20/40)+UDP(8)+VxLAN(8)
		memset(outer_header_content, 0, outer_header_size);
		
		//6RD-TODO: get Outer content add set to content table
		
	}

	
	return ret;
}


rtk_fc_ret_t _rtk_fc_dualHeader_ctrl_setup6RD(int dualControlIdx, rtk_fc_pktHdr_t *pPktHdr)
{
	rtk_fc_ret_t ret=RTK_FC_RET_OK;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;
	int aal_ret;
	memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));
	
	{
		dual_ctrl_entry.dual_ctrl_hdr_fmt			= 0;
		dual_ctrl_entry.dual_ctrl_hdr_len			= 20;	//dual_buffer_size; IPv4 header 
		dual_ctrl_entry.dual_ctrl_mtu_diff			= 20; 	//IPv4 header(20)
		dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= 127;
		dual_ctrl_entry.dual_ctrl_rep_ip_en 			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_ver			= 0;	//IPv4
		dual_ctrl_entry.dual_ctrl_rep_len_ost1			= 2;
		dual_ctrl_entry.dual_ctrl_rep_len_diff1 		= 20;
		dual_ctrl_entry.dual_ctrl_rep_len_ost2			= 127;
		dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 16;
		dual_ctrl_entry.dual_ctrl_ip4_ost			= 0;
		dual_ctrl_entry.dual_ctrl_ip6_oft			= 127;
		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost		= 127;
		dual_ctrl_entry.dual_ctrl_gre_mod_fld 		= 0;
		dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en 		= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff		= 20;
		dual_ctrl_entry.dual_ctrl_rep_ip_ptc			= 0;
		dual_ctrl_entry.dual_ctrl_rep_ip_ttl			= 0;
		dual_ctrl_entry.dual_ctrl_mapt_pre_len			= 0;
		dual_ctrl_entry.dual_ctrl_map_drf_ver 			= 0;
		dual_ctrl_entry.dual_ctrl_6rd_dip_pre_len 		= 0;
		dual_ctrl_entry.dual_ctrl_6rd_dip_suf_str_bit 		= 32;
		dual_ctrl_entry.dual_ctrl_dsl_uc_chk			= 0;
		dual_ctrl_entry.dual_ctrl_dsl_mc_chk			= 0;
		dual_ctrl_entry.dual_ctrl_l4sum_rec 			= 0;
		dual_ctrl_entry.dual_ctrl_vxlan_sp_ost			= 127;
		dual_ctrl_entry.dual_ctrl_inr_to_out_fld		= 0xb8;
		dual_ctrl_entry.dual_ctrl_tos_hsh_sel 			= 0;
		dual_ctrl_entry.dual_ctrl_mc_mac_map			= 0;
	}
	aal_ret = aal_pe_dual_control_entry_add(dualControlIdx, &dual_ctrl_entry);  // VXLAN-TODO:Idx getting
	if(aal_ret)
	{
		WARNING("Add dual control failed, ret = %d", aal_ret);
		ret = RTK_FC_RET_ERR;
	}
	return ret;
}

rtk_fc_ret_t rtk_fc_dual_control_entry_setVXLAN(int idx, rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo)

{
	rtk_fc_ret_t ret=RTK_FC_RET_OK;
	l3pe_dual_control_tbl_entry_t dual_ctrl_entry;
	int aal_ret;
	memset(&dual_ctrl_entry, 0, sizeof(l3pe_dual_control_tbl_entry_t));
	
	if(pG3IgrExtraInfo->dualHdrType == RTK_FC_DUALTYPE_VXLAN)
	{

		dual_ctrl_entry.dual_ctrl_hdr_fmt				= 1;
		dual_ctrl_entry.dual_ctrl_hdr_len				= pG3IgrExtraInfo->vxlan_info.outer_tag_len;//dual_buffer_size;						// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]+IP header(20/40)+UDP(8)+VxLAN(8)
		dual_ctrl_entry.dual_ctrl_mtu_diff				= (pG3IgrExtraInfo->vxlan_info.outer_isV6)?(40+8+8):(20+8+8); 	// IP header(20/40)+UDP(8)+VxLAN(8)
		dual_ctrl_entry.dual_ctrl_ppp_ip_ver_ost		= 127;
		dual_ctrl_entry.dual_ctrl_rep_ip_en 			= FALSE;								// don't care
		dual_ctrl_entry.dual_ctrl_rep_ip_ver			= 0;									// don't care
		if(pG3IgrExtraInfo->vxlan_info.outer_pppoeTag)
		{
			// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+pppoe payload length field(4)
			dual_ctrl_entry.dual_ctrl_rep_len_ost1		= 14+((pG3IgrExtraInfo->vxlan_info.outer_stag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_ctag)?4:0)+4;
			// PPP(2)+IP header(20/40)+UDP(8)+VxLAN(8)
			dual_ctrl_entry.dual_ctrl_rep_len_diff1 	= (pG3IgrExtraInfo->vxlan_info.outer_isV6)?(2+40+8+8):(2+20+8+8);
		}
		else
		{
			dual_ctrl_entry.dual_ctrl_rep_len_ost1		= 127;
			dual_ctrl_entry.dual_ctrl_rep_len_diff1 	= 0;
		}
		// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]+IP header(20/40)+UDP length field offset(4)
		dual_ctrl_entry.dual_ctrl_rep_len_ost2			= 14+((pG3IgrExtraInfo->vxlan_info.outer_stag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_ctag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_pppoeTag)?8:0)+((pG3IgrExtraInfo->vxlan_info.outer_isV6)?40:20)+4;
		dual_ctrl_entry.dual_ctrl_rep_len_diff2 		= 16;										// UDP(8)+VxLAN(8)
		// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]
		dual_ctrl_entry.dual_ctrl_ip4_ost				= (pG3IgrExtraInfo->vxlan_info.outer_isV6)?127:(14+((pG3IgrExtraInfo->vxlan_info.outer_stag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_ctag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_pppoeTag)?8:0));
		dual_ctrl_entry.dual_ctrl_ip6_oft				= (pG3IgrExtraInfo->vxlan_info.outer_isV6)?(14+((pG3IgrExtraInfo->vxlan_info.outer_stag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_ctag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_pppoeTag)?8:0)):127;
		dual_ctrl_entry.dual_ctrl_gre_seq_ack_ost		= 127;
		//dual_ctrl_entry.dual_ctrl_gre_mod_fld 		= 0;
		dual_ctrl_entry.dual_ctrl_rep_ip_dscp_en 		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_ecn_en 		= FALSE;
		dual_ctrl_entry.dual_ctrl_rep_ip_len			= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff		= (pG3IgrExtraInfo->vxlan_info.outer_isV6)?(8+8):(20+8+8); 		// IP header(20/40)+UDP(8)+VxLAN(8)
		dual_ctrl_entry.dual_ctrl_rep_len_diff2_pad_ctrl	= 1;
		dual_ctrl_entry.dual_ctrl_rep_len_diff1_pad_ctrl	= 1;
		dual_ctrl_entry.dual_ctrl_rep_ip_len_diff_pad_ctrl	= 1;
		//dual_ctrl_entry.dual_ctrl_rep_ip_ptc			= 0;
		//dual_ctrl_entry.dual_ctrl_rep_ip_ttl			= 0;
		//dual_ctrl_entry.dual_ctrl_mapt_pre_len		= 0;
		//dual_ctrl_entry.dual_ctrl_map_drf_ver 		= 0;
		//dual_ctrl_entry.dual_ctrl_6rd_dip_pre_len 	= 0;
		//dual_ctrl_entry.dual_ctrl_6rd_dip_suf_str_bit = 0;
		//dual_ctrl_entry.dual_ctrl_dsl_uc_chk			= 0;
		//dual_ctrl_entry.dual_ctrl_dsl_mc_chk			= 0;
		dual_ctrl_entry.dual_ctrl_l4sum_rec 			= 0;
		// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]+IP header(20/40)
		dual_ctrl_entry.dual_ctrl_vxlan_sp_ost			= 14+((pG3IgrExtraInfo->vxlan_info.outer_stag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_ctag)?4:0)+((pG3IgrExtraInfo->vxlan_info.outer_pppoeTag)?8:0)+((pG3IgrExtraInfo->vxlan_info.outer_isV6)?40:20);
		dual_ctrl_entry.dual_ctrl_inr_to_out_fld		= 0x7f; // bit [0,1,2,3,4,5,6]
		dual_ctrl_entry.dual_ctrl_tos_hsh_sel 			= 0; //VXLAN-TODO: FIXME
		//dual_ctrl_entry.dual_ctrl_mc_mac_map			= 0;

		
	}
	aal_ret = aal_pe_dual_control_entry_add(idx, &dual_ctrl_entry);  // VXLAN-TODO:Idx getting
	if(aal_ret)
	{
		WARNING("Add dual control failed, ret = %d", aal_ret);
		ret = RTK_FC_RET_ERR;
	}
	return ret;
}


rtk_fc_ret_t rtk_fc_dual_control_content_setVXLAN(uint32 dual_content_tbl_idx, rtk_fc_g3IgrExtraInfo_t *pG3IgrExtraInfo, u8 *outer_header_content, int outer_header_size, rtk_fc_pktHdr_t *pPktHdr)
{
	rtk_fc_ret_t ret=RTK_FC_RET_OK;
	l3pe_dual_tbl_entry_t dual_buffer_entry;
	int aal_ret, offset = 0;

	if(pG3IgrExtraInfo->dualHdrType == RTK_FC_DUALTYPE_VXLAN)
	{
		// DMAC(6)+SMAC(6)+[SVLAN(4)+CVLAN(4)]+Ethertype(2)+[PPPoE(8)]+IP header(20/40)+UDP(8)+VxLAN(8)
		memset(outer_header_content, 0, outer_header_size);
		
		{
			//outer l2
			memcpy(outer_header_content+offset, pPktHdr->outer_eth, sizeof(struct ethhdr));
			offset += sizeof(struct ethhdr);

			//outer svlan
			if(pPktHdr->outer_svh)
			{
				memcpy(outer_header_content+offset, pPktHdr->outer_svh, sizeof(struct vlan_hdr));
				offset += sizeof(struct vlan_hdr);
			}
			

			//outer cvlan
			if(pPktHdr->outer_cvh)
			{
				memcpy(outer_header_content+offset, pPktHdr->outer_cvh, sizeof(struct vlan_hdr));
				offset += sizeof(struct vlan_hdr);
			}
			

			//outer pppoe
			if(pPktHdr->outer_ppph)
			{
				memcpy(outer_header_content+offset, pPktHdr->outer_ppph, sizeof(struct pppoe_hdr));
				
				if(pPktHdr->outer_iph)
				{
						outer_header_content[offset + sizeof(struct pppoe_hdr)] = 0x00;
						outer_header_content[offset + sizeof(struct pppoe_hdr)+1] = 0x21;
				}
				else if(pPktHdr->outer_ip6h)
				{
						outer_header_content[offset + sizeof(struct pppoe_hdr)] = 0x00;
						outer_header_content[offset + sizeof(struct pppoe_hdr)+1] = 0x57;
				
				}
				offset += PPPOE_SES_HLEN;
			}

			//outer ip
			if(pPktHdr->outer_iph)
			{
				memcpy(outer_header_content + offset, pPktHdr->outer_iph, sizeof(struct iphdr));

				offset += sizeof(struct iphdr);
			}
			else if(pPktHdr->outer_ip6h)
			{
				memcpy(outer_header_content + offset, pPktHdr->outer_ip6h, sizeof(struct ipv6hdr));
				offset += sizeof(struct ipv6hdr);
			}
			
			//outer udp
			memcpy(outer_header_content+offset, pPktHdr->outer_udph, sizeof(struct udphdr));
			offset += sizeof(struct udphdr);

			//outer vxlan
			memcpy(outer_header_content+offset, pPktHdr->vxlan_info.vxlanHdr, sizeof(rtk_fc_vxlanhdr_t));
			offset += sizeof(rtk_fc_vxlanhdr_t);
		}
		
		memset(dual_buffer_entry.buff, 0, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX);
		memcpy(dual_buffer_entry.buff, outer_header_content, (outer_header_size<=L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX) ? outer_header_size : L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX); 

		aal_ret = aal_pe_dual_content_entry_add(dual_content_tbl_idx, &dual_buffer_entry);
		if(aal_ret == (-4) ) //AAL_E_ENTRY_USED
			DEBUG("Outer is added before.");
		else if(aal_ret !=RTK_FC_RET_OK)
			WARNING("Add dual content failed, ret = %d", aal_ret);
		else{
			ret = aal_ret;
			return ret;
		}
		
		memset(dual_buffer_entry.buff, 0, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX);
		
		if(outer_header_size > L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX)
			memcpy(dual_buffer_entry.buff, outer_header_content+L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX, outer_header_size-L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX);
		
		aal_ret = aal_pe_dual_content_entry_add(dual_content_tbl_idx+1, &dual_buffer_entry);

		if(aal_ret == (-4) ) //AAL_E_ENTRY_USED
			DEBUG("Outer is added before.");
		else if(aal_ret !=RTK_FC_RET_OK)
			WARNING("Add dual content failed, ret = %d", aal_ret);
		else{
			ret = aal_ret;
			return ret;
		}
		/*
		memset(dual_buffer_entry.buff, 0, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX);
		
		aal_ret = aal_pe_dual_content_entry_get(dual_content_tbl_idx, &dual_buffer_entry);

		if(aal_ret)
			WARNING("Add dual content failed, ret = %d", aal_ret);
		else if(debug_level&RTK_FB_DEBUG_LEVEL_DEBUG)
			dump_packet(dual_buffer_entry.buff, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX, "vxlan HDR[0] GET");

		memset(dual_buffer_entry.buff, 0, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX);
		aal_ret = aal_pe_dual_content_entry_get(DOUBLE_DUAL_BUFFER_IDX+1, &dual_buffer_entry);
		if(aal_ret)
			WARNING("Add dual content failed, ret = %d", aal_ret);
		else if(debug_level&RTK_FB_DEBUG_LEVEL_DEBUG)
			dump_packet(dual_buffer_entry.buff, L3PE_DUAL_CONTENT_TBL_BUFF_SIZE_MAX, "vxlan HDR[1] GET");
		*/
	}

	
	return ret;
}


#endif

