/******************************************************************************
 *
 * Copyright(c) 2019 Realtek Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
 * more details.
 *
 ******************************************************************************/

#include "trx_desc_1115e.h"
#if MAC_AX_1115E_SUPPORT

#define RXD_RPKT_TYPE_INVALID	0xFF
#define TXD_AC_TYPE_MSK		0x3
#define TXD_TID_IND_SH		2
#define TID_MAX_NUM		8

#define TID_0_QSEL 0
#define TID_1_QSEL 1
#define TID_2_QSEL 1
#define TID_3_QSEL 0
#define TID_4_QSEL 2
#define TID_5_QSEL 2
#define TID_6_QSEL 3
#define TID_7_QSEL 3
#define TID_0_IND 0
#define TID_1_IND 0
#define TID_2_IND 1
#define TID_3_IND 1
#define TID_4_IND 0
#define TID_5_IND 1
#define TID_6_IND 0
#define TID_7_IND 1

#define WP_OFFSET_UNIT_1115E 0x8
#define SEC_HDR_SIZE_4B 0x4
#define SEC_HDR_SIZE_8B 0x8
#define HDR_SIZE_WLAN_MAX 0x24
#define HDR_SIZE_802P3 0xE
#define HDR_SIZE_LLC 0x8
#define HDR_SIZE_AMSDU 0xE

enum wd_info_pkt_type {
	WD_INFO_PKT_NORMAL,

	/* keep last */
	WD_INFO_PKT_LAST,
	WD_INFO_PKT_MAX = WD_INFO_PKT_LAST,
};

static u8 qsel_l[TID_MAX_NUM] = {
	TID_0_QSEL, TID_1_QSEL, TID_2_QSEL, TID_3_QSEL,
	TID_4_QSEL, TID_5_QSEL, TID_6_QSEL, TID_7_QSEL
};

static u8 tid_ind[TID_MAX_NUM] = {
	TID_0_IND, TID_1_IND, TID_2_IND, TID_3_IND,
	TID_4_IND, TID_5_IND, TID_6_IND, TID_7_IND
};

u32 mac_txdesc_len_1115e(struct mac_ax_adapter *adapter,
			 struct rtw_t_meta_data *info)
{
	u32 len;
	enum rtw_packet_type type = info->type;

	switch (type) {
	case RTW_PHL_PKT_TYPE_H2C:
	case RTW_PHL_PKT_TYPE_FWDL:
		len = RXD_SHORT_LEN;
		/* to prevent from USB 512-bytes alignment */
		if (adapter->hw_info->intf == MAC_AX_INTF_USB) {
			if (((info->pktlen + len) &
			     (0x200 - 1)) == 0)
				len = RXD_LONG_LEN;
		}
		break;
	default:
		len = WD_BODY_LEN_V1;
		if (info->wdinfo_en != 0)
			len += WD_INFO_LEN;
		break;
	}

	return len;
}

static u32 txdes_proc_h2c_fwdl_1115e(struct mac_ax_adapter *adapter,
				     struct rtw_t_meta_data *info,
				     u8 *buf, u32 len)
{
	struct rxd_short_t *s_rxd;
	struct rxd_long_t *l_rxd;
	struct mac_ax_ops *ops = adapter_to_mac_ops(adapter);

	if (len != ops->txdesc_len(adapter, info))
		return MACBUFSZ;

	s_rxd = (struct rxd_short_t *)buf;
	s_rxd->dword0 =
		cpu_to_le32(SET_WORD(info->pktlen, AX_RXD_RPKT_LEN) |
			    (info->type == RTW_PHL_PKT_TYPE_FWDL ?
			     SET_WORD(RXD_S_RPKT_TYPE_FWDL, AX_RXD_RPKT_TYPE) :
			     SET_WORD(RXD_S_RPKT_TYPE_H2C, AX_RXD_RPKT_TYPE)));
	s_rxd->dword1 = 0;
	s_rxd->dword2 = 0;
	s_rxd->dword3 = 0;

	if (len == sizeof(struct rxd_long_t)) {
		l_rxd = (struct rxd_long_t *)buf;
		l_rxd->dword0 |= AX_RXD_LONG_RXD;
		l_rxd->dword4 = 0;
		l_rxd->dword5 = 0;
		l_rxd->dword6 = 0;
		l_rxd->dword7 = 0;
	}

	return MACSUCCESS;
}

#if MAC_AX_FEATURE_HV
static u32 txdes_proc_hv_1115e(struct mac_ax_adapter *adapter,
			       struct rtw_t_meta_data *info,
			       struct wd_body_t_v1 *wdb,
			       struct wd_info_t *wdi)
{
	struct hv_txpkt_info *hv_info = (struct hv_txpkt_info *)info->mac_priv;

	wdb->dword0 |= cpu_to_le32((hv_info->chk_en ? AX_TXD_CHK_EN : 0));
	wdi->dword3 |=
		cpu_to_le32((hv_info->null_1 ? AX_TXD_NULL_1 : 0) |
			    (hv_info->null_0 ? AX_TXD_NULL_0 : 0) |
			    (hv_info->tri_frame ? AX_TXD_TRI_FRAME : 0) |
			    (hv_info->ht_data_snd ? AX_TXD_HT_DATA_SND : 0));
	wdi->dword5 |= cpu_to_le32(SET_WORD(hv_info->ndpa_dur, AX_TXD_NDPA_DURATION));

	return MACSUCCESS;
}
#endif

static u32 txdes_proc_data_1115e(struct mac_ax_adapter *adapter,
				 struct rtw_t_meta_data *info, u8 *buf, u32 len)
{
	struct wd_body_t_v1 *wdb;
	struct wd_info_t *wdi;
	u8 wd_info_tmpl[WD_INFO_PKT_MAX][24] = {{0}};
	u32 sec_iv_h;
	u16 sec_iv_l;
	u8 qsel, dbcc_wmm;

	if (len != mac_txdesc_len_1115e(adapter, info))
		return MACBUFSZ;

	if (info->dma_ch > MAC_AX_DATA_CH11) {
		PLTFM_MSG_ERR("[ERR]txd ch: %d\n", info->dma_ch);
		return MACNOITEM;
	}

	if (info->pktlen > BE_TXD_TXPKTSIZE_MSK || !info->pktlen) {
		PLTFM_MSG_ERR("[ERR] illegal txpktsize %d\n", info->pktlen);
		return MACFUNCINPUT;
	}

	wdb = (struct wd_body_t_v1 *)buf;
	wdb->dword0 = 0;
	wdb->dword1 = 0;
	wdb->dword2 = 0;
	wdb->dword3 = 0;
	wdb->dword4 = 0;
	wdb->dword5 = 0;
	wdb->dword6 = 0;
	wdb->dword7 = 0;

	if (adapter->hw_info->intf == MAC_AX_INTF_SDIO)
		wdb->dword0 =
			cpu_to_le32(BE_TXD_STF_MODE);
	else if (adapter->hw_info->intf == MAC_AX_INTF_USB)
		wdb->dword0 =
			cpu_to_le32(BE_TXD_STF_MODE |
				    (info->usb_pkt_ofst ?
				     BE_TXD_PKT_OFFSET : 0));
	else
		wdb->dword0 =
			cpu_to_le32((info->wd_page_size ? BE_TXD_WD_PAGE : 0) |
				    (adapter->dle_info.qta_mode ==
				     MAC_AX_QTA_SCC_STF ||
				     adapter->dle_info.qta_mode ==
				     MAC_AX_QTA_DBCC_STF ?
				     BE_TXD_STF_MODE : 0));

	wdb->dword0 |=
		cpu_to_le32(SET_WORD(info->hw_seq_mode, BE_TXD_EN_HWSEQ_MODE) |
			    SET_WORD(info->hw_ssn_sel, BE_TXD_HW_SSN_SEL) |
			    (info->hw_amsdu ? BE_TXD_HWAMSDU : 0) |
			    (info->hw_sec_iv ? BE_TXD_HW_SEC_IV : 0) |
			    /*WD page setting is above*/
			    /*(info->chk_en ? BE_TXD_CHK_EN : 0) |*/
			    /*(info->wp_int ? BE_TXD_WP_INT : 0) |*/
			    /*STF mode setting is above*/
			    SET_WORD(info->hdr_len, BE_TXD_HDR_LLC_LEN) |
			    SET_WORD(info->dma_ch, BE_TXD_CHANNEL_DMA) |
			    (info->smh_en ? BE_TXD_SMH_EN : 0) |
			    /*PKT_OFFSET setting is above*/
			    (info->wdinfo_en ? BE_TXD_WDINFO_EN : 0) |
			    /*(info->moredata ? BE_TXD_MOREDATA : 0) |*/
			    SET_WORD(info->wp_offset, BE_TXD_WP_OFFSET));
		/*SET_WORD(info->wdsource, BE_TXD_WD_SOURCE) |*/
		/*(info->hci_seq_mode ? BE_TXD_HCI_SEQNUM_MODE : 0) |*/

	wdb->dword1 =
		/*DMA_TXAGG_NUM in refill_txdesc*/
		/*SET_WORD(info->rensse_num, BE_TXD_REUSE_SIZE) |*/
		cpu_to_le32(SET_WORD(info->sec_type, BE_TXD_SEC_TYPE) |
			    SET_WORD(info->sec_keyid, BE_TXD_SEC_KEYID)
			    (info->sw_sec_iv ? BE_TXD_SW_SEC_IV : 0) |
			    SET_WORD(info->reuse_size, BE_TXD_REUSE_SIZE) |
			    SET_WORD(info->reuse_start_num, BE_TXD_REUSE_START_NUM) |
			    SET_WORD(info->addr_info_num, BE_TXD_ADDR_INFO_NUM));

	/* Get bb and qsel from qsel by according MAC ID */
	if (info->macid < DBCC_WMM_LIST_SIZE)
		dbcc_wmm = *(adapter->dbcc_info->dbcc_wmm_list + info->macid);
	else
		dbcc_wmm = MAC_AX_DBCC_WMM_INVALID;

	if (info->dma_ch == MAC_AX_DATA_CH9 || info->dma_ch == MAC_AX_DATA_CH11)
		qsel = info->band ? MAC_AX_HI1_SEL : MAC_AX_HI0_SEL;
	else if (dbcc_wmm != MAC_AX_DBCC_WMM_INVALID)
		qsel = (dbcc_wmm << 2) | qsel_l[info->tid];
	else
		qsel = (info->wmm << 2) | qsel_l[info->tid];
	wdb->dword2 =
		cpu_to_le32(SET_WORD(info->pktlen, BE_TXD_TXPKTSIZE) |
			    (info->ampdu_en ? BE_TXD_AGG_EN : 0) |
			    ((info->bk || info->ack_ch_info) ? BE_TXD_BK : 0) |
			    SET_WORD(qsel, BE_TXD_QSEL) |
			    (tid_ind[info->tid] ? BE_TXD_TID_IND : 0) |
			    SET_WORD(info->macid, BE_TXD_MACID));

	wdb->dword3 = cpu_to_le32(SET_WORD(info->sw_seq, BE_TXD_WIFI_SEQ));
			/*(info->mlo_flag ? BE_TXD_MLO_FLAG : 0) |*/
			/*(info->is_mlo_sw_en ? BE_TXD_IS_MLD_SW_EN : 0) |*/
	sec_iv_l = info->iv[0] | (info->iv[1] << 8);
	wdb->dword4 = cpu_to_le32(SET_WORD(sec_iv_l, BE_TXD_SEC_IV_L));
	sec_iv_h = info->iv[2] |
		(info->iv[3] << 8) |
		(info->iv[4] << 16) |
		(info->iv[5] << 24);
	wdb->dword5 = cpu_to_le32(SET_WORD(sec_iv_h, BE_TXD_SEC_IV_H));
	wdb->dword6 =
		/*info->chg_link_flag ? BE_TXD_CHG_LINK_FLAG : 0) |*/
		cpu_to_le32((info->bc || info->mc ? BE_TXD_BMC : 0) |
			    (info->no_ack ? BE_TXD_NO_ACK : 0)
			    (info->upd_wlan_hdr ? BE_TXD_UPD_WLAN_HDR : 0));
			    /*(info->a4_hdr ? BE_TXD_A4_HDR : 0) |*/
			    /*(info->esop_bit ? BE_TXD_EOSP_BIT : 0) |*/
	wdb->dword7 =
		/*SET_WORD(info->msdu_num, BE_TXD_MSDU_NUM) |*/
		cpu_to_le32((info->f_er ? BE_TXD_DATA_ER : 0) |
			     (info->data_bw_er ? BE_TXD_DATA_BW_ER : 0) |
			     (info->f_dcm ? BE_TXD_DATA_DCM : 0) |
			     SET_WORD(info->f_gi_ltf, BE_TXD_GI_LTF) |
			     SET_WORD(info->f_rate, BE_TXD_DATARATE) |
			     SET_WORD(info->f_bw, BE_TXD_DATA_BW) |
			     (info->userate_sel ? BE_TXD_USERATE_SEL : 0));

	wdi = (struct wd_info_t *)wd_info_tmpl[WD_INFO_PKT_NORMAL];
	wdi->dword0 =
		cpu_to_le32(SET_WORD(info->mbssid, BE_TXD_MBSSID) |
			    SET_WORD(info->hal_port, BE_TXD_MULTIPORT_ID) |
			    (info->dis_rts_rate_fb ? BE_TXD_DISRTSFB : 0) |
			    (info->dis_data_rate_fb ? BE_TXD_DISDATAFB : 0) |
			    (info->f_ldpc ? BE_TXD_DATA_LDPC : 0) |
			    (info->f_stbc ? BE_TXD_DATA_STBC : 0) |
			/* (info->bypass_punc ? BE_TXD_BYPASS_PUNC : 0) | */
			    SET_WORD(info->data_tx_cnt_lmt, BE_TXD_DATA_TXCNT_LMT) |
			    (info->data_tx_cnt_lmt_en ?
				BE_TXD_DATA_TXCNT_LMT_SEL : 0) |
			/* (info->rls_to_cpuio ? BE_TXD_RLS_TO_CPUIO : 0) | */
			    (info->ack_ch_info ? BE_TXD_ACK_CH_INFO : 0));

	if (info->max_agg_num > 0)
		info->max_agg_num -= 1;

	wdi->dword1 =
		cpu_to_le32(SET_WORD(info->max_agg_num, BE_TXD_MAX_AGG_NUM) |
		/*SET_WORD(info->bcn_srch_seq, BE_TXD_BCN_SRCH_SEQ)*/
			    (info->nav_use_hdr ? BE_TXD_NAVUSEHDR : 0) |
			    (info->a_ctrl_bqr ? BE_TXD_A_CTRL_BQR : 0) |
			    (info->a_ctrl_bsr ? BE_TXD_A_CTRL_BSR : 0) |
			    (info->a_ctrl_cas ? BE_TXD_A_CTRL_CAS : 0) |
			    SET_WORD(info->data_rty_lowest_rate,
				     BE_TXD_DATA_RTY_LOWEST_RATE) |
			    SET_WORD(info->sw_define, BE_TXD_SW_DEFINE));
	wdi->dword2 =
		cpu_to_le32(SET_WORD(info->sec_cam_idx, BE_TXD_SEC_CAM_IDX) |
			    (info->force_key_en ? BE_TXD_FORCE_KEY_EN : 0) |
			    SET_WORD(info->life_time_sel, BE_TXD_LIFETIME_SEL) |
			    /*(info->force_txop ? BE_TXD_FORCE_TXOP : 0) |*/
			    SET_WORD(info->ampdu_density, BE_TXD_AMPDU_DENSITY));
	/*(info->lsig_txop_en ? BE_TXD_LSIG_TXOP_EN : 0) |*/
	/*SET_WORD(info->obw_dup_type, BE_TXD_OBW_CTS2SELF_DUP_TYPE)*/

	wdi->dword3 =
		cpu_to_le32(SET_WORD(info->ndpa, BE_TXD_NDPA) |
			    SET_WORD(info->snd_pkt_sel, BE_TXD_SND_PKT_SEL) |
			    (info->sifs_tx ? BE_TXD_SIFS_TX : 0) |
			    /*(info->ht_data_snd ? BE_TXD_HT_DATA_SND : 0) |*/
			    /*(info->cqi_snd ? BE_TXD_CQI_SND : 0) |*/
			    (info->rtt_en ? BE_TXD_RTT_EN : 0) |
			    (info->spe_rpt ? BE_TXD_SPE_RPT : 0));
			    /*(info->bt_null ? BE_TXD_BT_NULL : 0) |*/
			    /*(info->tri_frame ? BE_TXD_TRI_FRAME : 0) |*/
			    /*(info->null_0 ? BE_TXD_NULL_0 : 0) |*/
			    /*(info->null_1 ? BE_TXD_NULL_1 : 0) |*/
			    /*(info->raw ? BE_TXD_RAW : 0) |*/

	wdi->dword4 =
		cpu_to_le32((info->rts_en ? BE_TXD_RTS_EN : 0) |
			    (info->cts2self ? BE_TXD_CTS2SELF : 0) |
			    SET_WORD(info->rts_cca_mode, BE_TXD_CCA_RTS) |
			    (info->hw_rts_en ? BE_TXD_HW_RTS_EN : 0));
	wdi->dword5 = 0;
		/*SET_WORD(info->ndpa_dur, BE_TXD_NDPA_DURATION) |*/

#if MAC_AX_FEATURE_HV
	txdes_proc_hv_1115e(adapter, info, wdb, wdi);
#endif
	if (info->wdinfo_en != 0)
		PLTFM_MEMCPY(buf + WD_BODY_LEN_V1, (u8 *)wdi, WD_INFO_LEN);

	if (adapter->hw_info->wd_checksum_en)
		mac_wd_checksum_1115e(adapter, info, buf);

	return MACSUCCESS;
}

static u32 txdes_proc_mgnt_1115e(struct mac_ax_adapter *adapter,
				 struct rtw_t_meta_data *info, u8 *buf, u32 len)
{
	struct wd_body_t_v1 *wdb;
	struct wd_info_t *wdi;
	u32 sec_iv_h;
	u16 sec_iv_l;
	u8 wd_info_tmpl[WD_INFO_PKT_MAX][24] = {{0}};

	if (len != mac_txdesc_len_1115e(adapter, info)) {
		PLTFM_MSG_ERR("[ERR] illegal len %d\n", len);
		return MACBUFSZ;
	}

	if (info->pktlen > AX_TXD_TXPKTSIZE_MSK || !info->pktlen) {
		PLTFM_MSG_ERR("[ERR] illegal txpktsize %d\n", info->pktlen);
		return MACFUNCINPUT;
	}

	/* only use ch0 in initial development phase, */
	/* and modify it for normal using later.*/
	/* wd_info is always appended in initial development phase */
	wdb = (struct wd_body_t_v1 *)buf;
	if (adapter->hw_info->intf == MAC_AX_INTF_SDIO)
		wdb->dword0 =
			cpu_to_le32(AX_TXD_STF_MODE);
	else if (adapter->hw_info->intf == MAC_AX_INTF_USB)
		wdb->dword0 =
			cpu_to_le32(AX_TXD_STF_MODE |
				    (info->usb_pkt_ofst ?
				     AX_TXD_PKT_OFFSET : 0));
	else
		wdb->dword0 =
			cpu_to_le32((info->wd_page_size ? AX_TXD_WD_PAGE : 0) |
				    (adapter->dle_info.qta_mode ==
				     MAC_AX_QTA_SCC_STF ||
				     adapter->dle_info.qta_mode ==
				     MAC_AX_QTA_DBCC_STF ?
				     AX_TXD_STF_MODE : 0));

	wdb->dword0 |=
		cpu_to_le32(SET_WORD(info->hw_seq_mode, BE_TXD_EN_HWSEQ_MODE) |
			    SET_WORD(info->hw_ssn_sel, BE_TXD_HW_SSN_SEL) |
			    (info->hw_amsdu ? BE_TXD_HWAMSDU : 0) |
			    (info->hw_sec_iv ? BE_TXD_HW_SEC_IV : 0) |
			    /*WD page setting is above*/
			    /*(info->chk_en ? BE_TXD_CHK_EN : 0) |*/
			    /*(info->wp_int ? BE_TXD_WP_INT : 0) |*/
			    /*STF mode setting is above*/
			    SET_WORD(info->hdr_len, BE_TXD_HDR_LLC_LEN) |
			    SET_WORD(ch, BE_TXD_CH_DMA) |
			    (info->smh_en ? BE_TXD_SMH_EN : 0) |
			    /*PKT_OFFSET setting is above*/
			    (info->wdinfo_en ? BE_TXD_WDINFO_EN : 0) |
			    /*(info->moredata ? BE_TXD_MOREDATA : 0) |*/
			    SET_WORD(info->wp_offset, BE_TXD_WP_OFFSET));
		/*SET_WORD(info->wdsource, BE_TXD_WD_SOURCE) |*/
		/*(info->hci_seq_mode ? BE_TXD_HCI_SEQNUM_MODE : 0) |*/

	wdb->dword1 =
		/*DMA_TXAGG_NUM in refill_txdesc*/
		/*SET_WORD(info->rensse_num, BE_TXD_REUSE_SIZE) |*/
		cpu_to_le32(SET_WORD(info->sec_type, BE_TXD_SEC_TYPE) |
			    SET_WORD(info->sec_keyid, BE_TXD_SEC_KEYID)
			    (info->sw_sec_iv ? BE_TXD_SW_SEC_IV : 0) |
			    SET_WORD(info->reuse_size, BE_TXD_REUSE_SIZE) |
			    SET_WORD(info->reuse_start_num, BE_TXD_REUSE_START_NUM) |
			    SET_WORD(info->addr_info_num, BE_TXD_ADDR_INFO_NUM));
	/* Get bb and qsel from qsel by according MAC ID */
	wdb->dword2 =
		cpu_to_le32(SET_WORD(info->pktlen, BE_TXD_TXPKTSIZE) |
			    (info->ampdu_en ? BE_TXD_AGG_EN : 0) |
			    ((info->bk || info->ack_ch_info) ? BE_TXD_BK : 0) |
			    SET_WORD(qsel, BE_TXD_QSEL) |
			    (tid_ind[info->tid] ? BE_TXD_TID_IND : 0) |
			    SET_WORD(info->macid, BE_TXD_MACID));

	wdb->dword3 = cpu_to_le32(SET_WORD(info->sw_seq, BE_TXD_WIFI_SEQ));
			/*(info->mlo_flag ? BE_TXD_MLO_FLAG : 0) |*/
			/*(info->is_mlo_sw_en ? BE_TXD_IS_MLD_SW_EN : 0) |*/
	sec_iv_l = info->iv[0] |
		(info->iv[1] << 8);
	wdb->dword4 = cpu_to_le32(SET_WORD(sec_iv_l, AX_TXD_SEC_IV_L));
	sec_iv_h = info->iv[2] |
		(info->iv[3] << 8) |
		(info->iv[4] << 16) |
		(info->iv[5] << 24);
	wdb->dword5 = cpu_to_le32(SET_WORD(sec_iv_h, BE_TXD_SEC_IV_H));
	wdb->dword6 =
		/*info->chg_link_flag ? BE_TXD_CHG_LINK_FLAG : 0) |*/
		cpu_to_le32((info->bc || info->mc ? BE_TXD_BMC : 0) |
			    (info->no_ack ? BE_TXD_NO_ACK : 0)
			    (info->upd_wlan_hdr ? BE_TXD_UPD_WLAN_HDR : 0));
			    /*(info->a4_hdr ? BE_TXD_A4_HDR : 0) |*/
			    /*(info->esop_bit ? BE_TXD_EOSP_BIT : 0) |*/
	wdb->dword7 =
		/*SET_WORD(info->msdu_num, BE_TXD_MSDU_NUM) |*/
		cpu_to_le32((info->f_er ? BE_TXD_DATA_ER : 0) |
			     (info->data_bw_er ? BE_TXD_DATA_BW_ER : 0) |
			     (info->f_dcm ? BE_TXD_DATA_DCM : 0) |
			     SET_WORD(info->f_gi_ltf, BE_TXD_GI_LTF) |
			     SET_WORD(info->f_rate, BE_TXD_DATARATE) |
			     SET_WORD(info->f_bw, BE_TXD_DATA_BW) |
			     (info->userate_sel ? BE_TXD_USERATE_SEL : 0));

	wdi = (struct wd_info_t *)wd_info_tmpl[WD_INFO_PKT_NORMAL];
	wdi->dword0 =
		cpu_to_le32(SET_WORD(info->mbssid, BE_TXD_MBSSID) |
			    SET_WORD(info->hal_port, BE_TXD_MULTIPORT_ID) |
			    (info->dis_rts_rate_fb ? BE_TXD_DISRTSFB : 0) |
			    (info->dis_data_rate_fb ? BE_TXD_DISDATAFB : 0) |
			    (info->f_ldpc ? BE_TXD_DATA_LDPC : 0) |
			    (info->f_stbc ? BE_TXD_DATA_STBC : 0) |
			/* (info->bypass_punc ? BE_TXD_BYPASS_PUNC : 0) | */
			    SET_WORD(info->data_tx_cnt_lmt, BE_TXD_DATA_TXCNT_LMT) |
			    (info->data_tx_cnt_lmt_en ?
				BE_TXD_DATA_TXCNT_LMT_SEL : 0) |
			/* (info->rls_to_cpuio ? BE_TXD_RLS_TO_CPUIO : 0) | */
			    (info->ack_ch_info ? BE_TXD_ACK_CH_INFO : 0));

	if (info->max_agg_num > 0)
		info->max_agg_num -= 1;

	wdi->dword1 =
		cpu_to_le32(SET_WORD(info->max_agg_num, BE_TXD_MAX_AGG_NUM) |
		/*SET_WORD(info->bcn_srch_seq, BE_TXD_BCN_SRCH_SEQ)*/
			    (info->nav_use_hdr ? BE_TXD_NAVUSEHDR : 0) |
			    (info->a_ctrl_bqr ? BE_TXD_A_CTRL_BQR : 0) |
			    (info->a_ctrl_bsr ? BE_TXD_A_CTRL_BSR : 0) |
			    (info->a_ctrl_cas ? BE_TXD_A_CTRL_CAS : 0) |
			    SET_WORD(info->data_rty_lowest_rate,
				     BE_TXD_DATA_RTY_LOWEST_RATE) |
			    SET_WORD(info->sw_define, BE_TXD_SW_DEFINE));
	wdi->dword2 =
		cpu_to_le32(SET_WORD(info->sec_cam_idx, BE_TXD_SEC_CAM_IDX) |
			    (info->force_key_en ? BE_TXD_FORCE_KEY_EN : 0) |
			    SET_WORD(info->life_time_sel, BE_TXD_LIFETIME_SEL) |
			    /*(info->force_txop ? BE_TXD_FORCE_TXOP : 0) |*/
			    SET_WORD(info->ampdu_density, BE_TXD_AMPDU_DENSITY));
	wdi->dword3 =
		cpu_to_le32(SET_WORD(info->ndpa, BE_TXD_NDPA) |
			    SET_WORD(info->snd_pkt_sel, BE_TXD_SND_PKT_SEL) |
			    (info->sifs_tx ? BE_TXD_SIFS_TX : 0) |
			    /*(info->ht_data_snd ? BE_TXD_HT_DATA_SND : 0) |*/
			    /*(info->cqi_snd ? BE_TXD_CQI_SND : 0) |*/
			    (info->rtt_en ? BE_TXD_RTT_EN : 0) |
			    (info->spe_rpt ? BE_TXD_SPE_RPT : 0));
			    /*(info->bt_null ? BE_TXD_BT_NULL : 0) |*/
			    /*(info->tri_frame ? BE_TXD_TRI_FRAME : 0) |*/
			    /*(info->null_0 ? BE_TXD_NULL_0 : 0) |*/
			    /*(info->null_1 ? BE_TXD_NULL_1 : 0) |*/
			    /*(info->raw ? BE_TXD_RAW : 0) |*/

	wdi->dword4 =
		cpu_to_le32((info->rts_en ? BE_TXD_RTS_EN : 0) |
			    (info->cts2self ? BE_TXD_CTS2SELF : 0) |
			    SET_WORD(info->rts_cca_mode, BE_TXD_CCA_RTS) |
			    (info->hw_rts_en ? BE_TXD_HW_RTS_EN : 0));
	wdi->dword5 = 0;
		/*SET_WORD(info->ndpa_dur, BE_TXD_NDPA_DURATION) |*/

#if MAC_AX_FEATURE_HV
	txdes_proc_hv_1115e(adapter, info, wdb, wdi);
#endif
	if (info->wdinfo_en != 0)
		PLTFM_MEMCPY(buf + WD_BODY_LEN_V1, (u8 *)wdi, WD_INFO_LEN);

	if (adapter->hw_info->wd_checksum_en)
		mac_wd_checksum_1115e(adapter, info, buf);

	return MACSUCCESS;
}

static struct txd_proc_type txdes_proc_mac_1115e[] = {
	{RTW_PHL_PKT_TYPE_H2C, txdes_proc_h2c_fwdl_1115e},
	{RTW_PHL_PKT_TYPE_FWDL, txdes_proc_h2c_fwdl_1115e},
	{RTW_PHL_PKT_TYPE_DATA, txdes_proc_data_1115e},
	{RTW_PHL_PKT_TYPE_MGNT, txdes_proc_mgnt_1115e},
	{RTW_PHL_PKT_TYPE_MAX, NULL},
};

u32 mac_build_txdesc_1115e(struct mac_ax_adapter *adapter,
			   struct rtw_t_meta_data *info, u8 *buf, u32 len)
{
	struct txd_proc_type *proc = txdes_proc_mac_1115e;
	enum rtw_packet_type pkt_type = info->type;
	u32 (*handler)(struct mac_ax_adapter *adapter,
		       struct rtw_t_meta_data *info, u8 *buf, u32 len) = NULL;

	for (; proc->type != RTW_PHL_PKT_TYPE_MAX; proc++) {
		if (pkt_type == proc->type) {
			handler = proc->handler;
			break;
		}
	}

	if (!handler) {
		PLTFM_MSG_ERR("[ERR]null type handler type: %X\n", proc->type);
		return MACNOITEM;
	}

	return handler(adapter, info, buf, len);
}

u32 mac_refill_txdesc_1115e(struct mac_ax_adapter *adapter,
			    struct rtw_t_meta_data *txpkt_info,
			    struct mac_ax_refill_info *mask,
			    struct mac_ax_refill_info *info)
{
	u32 dw0 = ((struct wd_body_t_v1 *)info->pkt)->dword0;
	u32 dw1 = ((struct wd_body_t_v1 *)info->pkt)->dword1;

	if (mask->packet_offset)
		((struct wd_body_t_v1 *)info->pkt)->dword0 =
			dw0 | (info->packet_offset ? AX_TXD_PKT_OFFSET : 0);

	if (mask->agg_num == AX_TXD_DMA_TXAGG_NUM_V1_MSK)
		((struct wd_body_t_v1 *)info->pkt)->dword1 =
			SET_CLR_WORD(dw1, info->agg_num, AX_TXD_DMA_TXAGG_NUM_V1);

	if (adapter->hw_info->wd_checksum_en)
		mac_wd_checksum_1115e(adapter, txpkt_info, info->pkt);

	return MACSUCCESS;
}

static u32 rxdes_parse_comm_1115e(struct mac_ax_adapter *adapter,
				  struct mac_ax_rxpkt_info *info, u8 *buf)
{
	u32 hdr_val = le32_to_cpu(((struct rxd_short_t *)buf)->dword0);

	info->rxdlen = hdr_val & AX_RXD_LONG_RXD ? RXD_LONG_LEN : RXD_SHORT_LEN;
	info->pktsize = GET_FIELD(hdr_val, AX_RXD_RPKT_LEN);
	info->shift = (u8)GET_FIELD(hdr_val, AX_RXD_SHIFT);
	info->drvsize = (u8)GET_FIELD(hdr_val, AX_RXD_DRV_INFO_SIZE);

	return MACSUCCESS;
}

static u32 rxdes_parse_wifi_1115e(struct mac_ax_adapter *adapter,
				  struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	u32 hdr_val;

	info->type = MAC_AX_PKT_DATA;

	hdr_val = le32_to_cpu(((struct rxd_short_t *)buf)->dword3);
	info->u.data.crc_err = !!(hdr_val & AX_RXD_CRC32_ERR);
	info->u.data.icv_err = !!(hdr_val & AX_RXD_ICV_ERR);

	return MACSUCCESS;
}

static u32 rxdes_parse_c2h_1115e(struct mac_ax_adapter *adapter,
				 struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	info->type = MAC_AX_PKT_C2H;

	return MACSUCCESS;
}

static u32 rxdes_parse_ch_info_1115e(struct mac_ax_adapter *adapter,
				     struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	info->type = MAC_AX_PKT_CH_INFO;

	return MACSUCCESS;
}

static u32 rxdes_parse_dfs_1115e(struct mac_ax_adapter *adapter,
				 struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	info->type = MAC_AX_PKT_DFS;

	return MACSUCCESS;
}

static u32 rxdes_parse_ppdu_1115e(struct mac_ax_adapter *adapter,
				  struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	u32 hdr_val = le32_to_cpu(((struct rxd_short_t *)buf)->dword0);

	info->type = MAC_AX_PKT_PPDU;
	info->u.ppdu.mac_info = !!(hdr_val & AX_RXD_MAC_INFO_VLD);

	return MACSUCCESS;
}

static struct rxd_parse_type rxdes_parse_mac_1115e[] = {
	{RXD_S_RPKT_TYPE_WIFI, rxdes_parse_wifi_1115e},
	{RXD_S_RPKT_TYPE_C2H, rxdes_parse_c2h_1115e},
	{RXD_S_RPKT_TYPE_PPDU, rxdes_parse_ppdu_1115e},
	{RXD_S_RPKT_TYPE_CH_INFO, rxdes_parse_ch_info_1115e},
	{RXD_S_RPKT_TYPE_DFS_RPT, rxdes_parse_dfs_1115e},
	{RXD_RPKT_TYPE_INVALID, NULL},
};

u32 mac_parse_rxdesc_1115e(struct mac_ax_adapter *adapter,
			   struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	struct rxd_parse_type *parse = rxdes_parse_mac_1115e;
	u8 rpkt_type;
	u32 hdr_val;
	u32 (*handler)(struct mac_ax_adapter *adapter,
		       struct mac_ax_rxpkt_info *info, u8 *buf, u32 len) = NULL;

	hdr_val = le32_to_cpu(((struct rxd_short_t *)buf)->dword0);
	rpkt_type = (u8)GET_FIELD(hdr_val, AX_RXD_RPKT_TYPE);

	//rxdes_parse_comm_1115e(adapter, info, buf);

	for (; parse->type != RXD_RPKT_TYPE_INVALID; parse++) {
		if (rpkt_type == parse->type) {
			handler = parse->handler;
			break;
		}
	}

	if (!handler) {
		PLTFM_MSG_ERR("[ERR]null type handler type: %X\n", parse->type);
		return MACNOITEM;
	}

	return handler(adapter, info, buf, len);
}

u32 mac_wd_checksum_1115e(struct mac_ax_adapter *adapter,
			  struct rtw_t_meta_data *info, u8 *wddesc)
{
	u16 chksum = 0;
	u32 wddesc_size;
	u16 *data;
	u32 i, dw4;

	if (!wddesc) {
		PLTFM_MSG_ERR("[ERR]null pointer\n");
		return MACNPTR;
	}

	if (adapter->hw_info->wd_checksum_en != 1)
		PLTFM_MSG_TRACE("[TRACE]chksum disable\n");

	dw4 = ((struct wd_body_t_v1 *)wddesc)->dword4;

	((struct wd_body_t_v1 *)wddesc)->dword4 =
		SET_CLR_WORD(dw4, 0x0, AX_TXD_TXDESC_CHECKSUM);

	data = (u16 *)(wddesc);
	/*unit : 4 bytes*/
	wddesc_size = mac_txdesc_len_1115e(adapter, info) >> 2;
	for (i = 0; i < wddesc_size; i++)
		chksum ^= (*(data + 2 * i) ^ *(data + (2 * i + 1)));

	/* *(data + 2 * i) & *(data + (2 * i + 1) have endain issue*/
	/* Process eniadn issue after checksum calculation */
	((struct wd_body_t_v1 *)wddesc)->dword4 =
		SET_CLR_WORD(dw4, (u16)(chksum), AX_TXD_TXDESC_CHECKSUM);
	return MACSUCCESS;
}

u32 mac_patch_rx_rate_1115e(struct mac_ax_adapter *adapter,
			    struct rtw_r_meta_data *info)
{
	return MACSUCCESS;
}

u32 mac_get_wp_offset_1115e(struct mac_ax_adapter *adapter,
			    struct mac_txd_ofld_wp_offset *ofld_conf, u16 *val)
{
	u16 ret_val = 0;

	if (ofld_conf->enc_type == MAC_TXD_OFLD_HW_ENC_CONF_MISSING) {
		PLTFM_MSG_ERR("missing configurations: ENC type\n");
		return MACFUNCINPUT;
	}
	if (ofld_conf->hw_amsdu_type == MAC_TXD_OFLD_HW_AMSDU_CONF_MISSING) {
		PLTFM_MSG_ERR("missing configurations: HW AMSDU type\n");
		return MACFUNCINPUT;
	}
	if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_CONF_MISSING) {
		PLTFM_MSG_ERR("missing configurations: HW HDR CONV type\n");
		return MACFUNCINPUT;
	}

	switch (ofld_conf->enc_type) {
	case MAC_TXD_OFLD_SW_ENC:
	case MAC_TXD_OFLD_HW_ENC_BIP128:
		ret_val = 0;
	break;
	case MAC_TXD_OFLD_HW_ENC_WAPI:
	case MAC_TXD_OFLD_HW_ENC_NONE:
		if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_ETHII_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX + HDR_SIZE_LLC - HDR_SIZE_802P3);
		else if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_SNAP_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX - HDR_SIZE_802P3);

		if (ofld_conf->hw_amsdu_type == MAC_TXD_OFLD_HW_AMSDU_ON)
			ret_val += HDR_SIZE_AMSDU;
	break;
	case MAC_TXD_OFLD_HW_ENC_WEP40:
	case MAC_TXD_OFLD_HW_ENC_WEP104:
		ret_val += SEC_HDR_SIZE_4B;
		if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_ETHII_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX + HDR_SIZE_LLC - HDR_SIZE_802P3);
		else if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_SNAP_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX - HDR_SIZE_802P3);

		if (ofld_conf->hw_amsdu_type == MAC_TXD_OFLD_HW_AMSDU_ON)
			ret_val += HDR_SIZE_AMSDU;
	break;
	case MAC_TXD_OFLD_HW_ENC_TKIP:
	case MAC_TXD_OFLD_HW_ENC_GCMSMS4:
	case MAC_TXD_OFLD_HW_ENC_CCMP128:
	case MAC_TXD_OFLD_HW_ENC_CCMP256:
	case MAC_TXD_OFLD_HW_ENC_GCMP128:
	case MAC_TXD_OFLD_HW_ENC_GCMP256:
		ret_val += SEC_HDR_SIZE_8B;
		if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_ETHII_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX + HDR_SIZE_LLC - HDR_SIZE_802P3);
		else if (ofld_conf->hw_hdr_conv_type == MAC_TXD_OFLD_HW_HDR_CONV_SNAP_TO_WLAN)
			ret_val += (HDR_SIZE_WLAN_MAX - HDR_SIZE_802P3);

		if (ofld_conf->hw_amsdu_type == MAC_TXD_OFLD_HW_AMSDU_ON)
			ret_val += HDR_SIZE_AMSDU;
	break;
	default:
		ret_val = 0;
	}

	*val = (ret_val / WP_OFFSET_UNIT_1115E) + (ret_val % WP_OFFSET_UNIT_1115E ? 0x1 : 0x0);

	return MACSUCCESS;
}

#else
u32 mac_build_txdesc_1115e(struct mac_ax_adapter *adapter,
			   struct rtw_t_meta_data *info, u8 *buf, u32 len)
{
	return 0;
}

u32 mac_refill_txdesc_1115e(struct mac_ax_adapter *adapter,
			    struct rtw_t_meta_data *txpkt_info,
			    struct mac_ax_refill_info *mask,
			    struct mac_ax_refill_info *info)
{
	return 0;
}

u32 mac_parse_rxdesc_1115e(struct mac_ax_adapter *adapter,
			   struct mac_ax_rxpkt_info *info, u8 *buf, u32 len)
{
	return 0;
}

u32 mac_txdesc_len_1115e(struct mac_ax_adapter *adapter,
			 struct rtw_t_meta_data *info)
{
	return 0;
}

#endif /* #if MAC_AX_1115E_SUPPORT */
