#include <net/rtl/rtk_transportable_proto.h>

//table of approved ethernet types
u16 transportable_eth_type[] = {
	0x0800, //IP
	0x0806, //ARP
	0x8137, //IPX
	0x86dd, //IPv6
	0x8863, //PPPoE Discovery Stage
	0x8864, //PPPoE Session Stage
	0x888e,  //802.1X
	0x893a  //Mesh
};

//table of approved IP protocols
//move TCP & UDP ahead to reduce matching time
u8 transportable_ip_proto[] = {
	6,   //TCP
	17,  //UDP
	1,   //ICMP
	2,   //IGMP
	4,   //IP
	41,  //IPv6
	47,  //GRE
	50,  //ESP
	115  //L2TP
};

//table of approved IPv6 protocols
//move TCP & UDP ahead to reduce matching time
u8 transportable_ipv6_proto[] = {
	6,   //TCP
	17,  //UDP
	4,   //IP
	41,  //IPv6
	44,  //IPv6 Frag
	47,  //GRE
	50,  //ESP
	58,  //ICMPv6
	115  //L2TP
};

static u16 get_eth_type(struct sk_buff *pskb)
{
	u16 eth_type = 0;

	if(pskb)
	{
		eth_type = (*(u16*)(pskb->data + (ETH_ALEN<<1)));

		// if C-VLAN tag exists, move 4 bytes forward
		if(eth_type == ETH_P_8021Q)
			eth_type = (*(u16*)(pskb->data + ((ETH_ALEN<<1) + VLAN_HLEN)));
	}

	return eth_type;
}

static u16 get_ppp_proto(struct sk_buff *pskb)
{
	u8 *ptr = NULL;
	u16 ppp_proto = 0;

	if(pskb)
	{
		ptr = pskb->data + (ETH_ALEN << 1);

		if(*(u16*)(ptr) == (u16)ETH_P_8021Q)
			ptr = ptr + VLAN_HLEN;

		if (*(u16*)(ptr) == (u16)ETH_P_PPP_SES)
		{
			ptr = ptr + ETH_TYPE_LEN + PPP_SESSION_LEN;
			ppp_proto = *(u16*)(ptr);
		}
	}

	return ppp_proto;
}

static struct iphdr* get_ipv4_header(struct sk_buff *pskb)
{
	u8 *ptr = NULL;
	struct iphdr *iph = NULL;

	if(pskb)
	{
		ptr = pskb->data + (ETH_ALEN << 1);

		if(*(u16*)(ptr) == (u16)ETH_P_8021Q)
			ptr = ptr + VLAN_HLEN;

		if(*(u16*)(ptr) == (u16)ETH_P_PPP_SES)
		{
			ptr = ptr + ETH_TYPE_LEN + PPP_SESSION_LEN;
			if(*(u16*)(ptr) != (u16)PPP_IPV4_PROTOCOL)
				return NULL;
		}
		else
		{
			if(*(u16*)(ptr) != (u16)ETH_P_IP)
				return NULL;
		}

		iph = (struct iphdr*)(ptr + ETH_TYPE_LEN);
	}

	return iph;
}

static struct ipv6hdr* get_ipv6_header(struct sk_buff *pskb)
{
	u8 *ptr;
	struct ipv6hdr *ipv6h = NULL;

	if(pskb)
	{
		ptr = pskb->data + (ETH_ALEN << 1);

		if(*(u16*)(ptr) == (u16)ETH_P_8021Q)
			ptr = ptr + VLAN_HLEN;

		if(*(u16*)(ptr) == (u16)ETH_P_PPP_SES)
		{
			ptr = ptr + ETH_TYPE_LEN + PPP_SESSION_LEN;
			if(*(u16*)(ptr) != (u16)PPP_IPV6_PROTOCOL)
				return NULL;
		}
		else
		{
			if(*(u16*)(ptr) != (u16)ETH_P_IPV6)
				return NULL;
		}

		ipv6h = (struct ipv6hdr*)(ptr + ETH_TYPE_LEN);
	}

	return ipv6h;
}

static u8 get_ip_proto(struct sk_buff *pskb, u8 is_ipv4)
{
	u8 ip_proto = 0;

	if(pskb)
	{
		if(is_ipv4)
		{
			struct iphdr *iph = get_ipv4_header(pskb);
			if(iph)
				ip_proto = iph->protocol;
		}
		else
		{
			struct ipv6hdr *ipv6h = get_ipv6_header(pskb);
			if(ipv6h)
			{
				if(ipv6h->nexthdr == IPV6_HOP_BY_HOP_OPTION)
				{
					//the offset of ipv6 hop-by-hop option's next headeris 40 bytes
					//(fixed length of ipv6 header), so (ipv6h+1) points to next header
					ip_proto = *(u8*)(ipv6h+1);
				}
				else
					ip_proto = ipv6h->nexthdr;
			}
		}
	}

	return ip_proto;
}

static int is_transportable_eth_type(u16 eth_type)
{
	int ret = 0, i = 0, eth_type_cnt = 0;

	eth_type_cnt = sizeof(transportable_eth_type)/sizeof(u16);
	for(i = 0; i < eth_type_cnt; ++i)
		if(eth_type == transportable_eth_type[i])
			ret = 1;

	return ret;
}

static int is_transpaortble_ip_proto(u8 ip_proto, u8 is_ipv4)
{
	int ret = 0, i = 0, ip_proto_cnt = 0;

	if(is_ipv4)
	{
		ip_proto_cnt = sizeof(transportable_ip_proto)/sizeof(u8);
		for(i = 0; i < ip_proto_cnt; ++i)
			if(ip_proto == transportable_ip_proto[i])
				ret = 1;
	}
	else
	{
		ip_proto_cnt = sizeof(transportable_ipv6_proto)/sizeof(u8);
		for(i = 0; i < ip_proto_cnt; ++i)
			if(ip_proto == transportable_ipv6_proto[i])
				ret = 1;
	}

	return ret;
}

int is_transportable_pkt(struct sk_buff *pskb)
{
	int ret = 0;
	u8 ip_proto = 0;
	u16 eth_type = 0;

	if(pskb)
	{
		eth_type  = get_eth_type(pskb);
		if(is_transportable_eth_type(eth_type) == 1)
		{
			if(eth_type == ETH_P_IP
			|| ((eth_type == ETH_P_PPP_SES) && (get_ppp_proto(pskb) == PPP_IPV4_PROTOCOL)))
			{
				ip_proto = get_ip_proto(pskb, 1);
				if(is_transpaortble_ip_proto(ip_proto, 1))
					ret = 1;
			}
			else if(eth_type == ETH_P_IPV6
			|| ((eth_type == ETH_P_PPP_SES) && (get_ppp_proto(pskb) == PPP_IPV6_PROTOCOL)))
			{
				ip_proto = get_ip_proto(pskb, 0);
				if(is_transpaortble_ip_proto(ip_proto, 0))
					ret = 1;
			}
			else
				ret = 1;
		}
	}

	/*
	if(ret == 0)
		printk("ethType: 0x%04x ipProto: 0x%02x Non-transportable protocol, DROP!\n", eth_type, ip_proto);
	*/

	return ret;
}
EXPORT_SYMBOL(is_transportable_pkt);
