#include <linux/bug.h>
#include <linux/kernel.h>

#include "osdep_service_linux.h"
#include "oui.h"

/* Rules of Adding OUIs:
 *  1. One OUI per line
 *  2. All characters should be upper-cases
 *  3. Sort OUIs in ascending order (0, ..., 9, A, ..., F)
 *  4. Use TAB as indention
 */

/* Reference of OUI database:
 *  -- Wireshark: https://gitlab.com/wireshark/wireshark/-/raw/master/manuf (Last update 2022.08.27, commit f3df91b)
 *  -- https://aruljohn.com/mac/vendor/Apple (Last update 2018.03.16)
 */

// Use bit 23:0 for OUI, bit 31:24 for IOT enum
#define OUI_TO_IOT(m1, m2, m3, iot) \
	((unsigned int)(iot) << 24) + OUI_TO_UINT32((m1), (m2), (m3))

typedef unsigned int oui_t;

static __always_inline HT_IOT_PEER_E get_iot(const oui_t *node)
{
#ifdef __LINUX_3_9__
	BUILD_BUG_ON_MSG(HT_IOT_PEER_MAX > 255, "The number of IOT types enumerated in HT_IOT_PEER_E should be less than 256.");
#else
	BUILD_BUG_ON(HT_IOT_PEER_MAX > 255);  // The number of IOT types enumerated in HT_IOT_PEER_E should be less than 256.
#endif

	return (HT_IOT_PEER_E)(((unsigned int)(*node) & 0xFF000000) >> 24);
}

static __always_inline int cmp_oui(const unsigned int oui, const oui_t *node)
{
	unsigned int node_oui = (((unsigned int)(*node)) & 0x00FFFFFF);

	if (oui > node_oui)
		return 1;
	else if (oui < node_oui)
		return -1;
	else
		return 0;
}

// OUI database for MAC address to IOT lookup
static const oui_t mac_oui_db[] = {
	OUI_TO_IOT(0x00, 0x01, 0x4A, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x02, 0xB3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x02, 0xC7, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x03, 0x47, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x04, 0x23, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x05, 0xB5, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x06, 0xF5, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x06, 0xF7, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x07, 0x04, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x07, 0xE9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x09, 0x2D, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x00, 0x09, 0x5B, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x0A, 0xF7, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x0C, 0x43, HT_IOT_PEER_RALINK),
	OUI_TO_IOT(0x00, 0x0C, 0xF1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x0E, 0x0C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x0E, 0x35, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x0F, 0xB5, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x10, 0x18, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x10, 0x94, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x11, 0x11, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x11, 0x75, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x11, 0x94, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x11, 0x95, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x11, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x11, 0xDD, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x12, 0xF0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x13, 0x02, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x13, 0x20, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x13, 0xA8, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x13, 0xA9, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x13, 0xCE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x13, 0xE8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x14, 0x6C, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x14, 0x94, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x14, 0xA4, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x15, 0x00, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x15, 0x17, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x16, 0x6F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x16, 0x76, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x16, 0xCE, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x16, 0xCF, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x16, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x16, 0xEB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x16, 0xFE, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x17, 0x35, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x17, 0xF2, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x00, 0x18, 0x4D, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x18, 0xDE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x19, 0x66, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x19, 0x7D, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x19, 0x7E, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x19, 0x7F, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x19, 0xD1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x19, 0xD2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1B, 0x21, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1B, 0x2F, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x1B, 0x77, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1B, 0xFB, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x1C, 0x26, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x1C, 0xBF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1C, 0xC0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1D, 0x40, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1D, 0xD9, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x1D, 0xE0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1D, 0xE1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1E, 0x2A, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x1E, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1E, 0x65, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1E, 0x67, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1F, 0x33, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x1F, 0x3A, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x1F, 0x3B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1F, 0x3C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x1F, 0xE1, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x1F, 0xE2, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x20, 0x23, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x20, 0x7B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x20, 0x98, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x21, 0x4F, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x21, 0x5C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x21, 0x5D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x21, 0x6A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x21, 0x6B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x21, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x21, 0xDD, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x22, 0x3F, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x22, 0x68, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x22, 0x69, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x22, 0xCF, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x22, 0xFA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x22, 0xFB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x23, 0x06, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x23, 0x14, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x23, 0x15, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x23, 0x4D, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x23, 0x4E, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x23, 0x76, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x00, 0x24, 0x2B, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x24, 0x2C, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x24, 0x33, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x24, 0xB2, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x24, 0xD6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x24, 0xD7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x25, 0x56, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x26, 0x43, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x26, 0x5C, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0x26, 0xC6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x26, 0xC7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x26, 0xF2, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x27, 0x0E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x27, 0x10, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x28, 0xF8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x31, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x31, 0xDD, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x41, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x41, 0xDD, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x42, 0x38, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x50, 0xF1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x51, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x60, 0xF3, HT_IOT_PEER_SPIRENT),
	OUI_TO_IOT(0x00, 0x71, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x81, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x8E, 0xF2, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x00, 0x90, 0x27, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x91, 0x9E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0x91, 0xC0, HT_IOT_PEER_VERIWAVE),
	OUI_TO_IOT(0x00, 0x93, 0x37, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xA0, 0xC9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xA5, 0x54, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xAA, 0x00, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xAA, 0x01, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xAA, 0x02, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xBB, 0x60, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xC2, 0xC6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xD0, 0xB7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xD4, 0x4B, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x00, 0xD4, 0x9E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xD7, 0x6D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xDB, 0xDF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x00, 0xE1, 0x8C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0x33, 0xC2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0x56, 0xE5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0x6C, 0x59, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0x76, 0x6E, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x04, 0xA1, 0x51, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x04, 0xCF, 0x4B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0xD3, 0xB0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0xE8, 0xB9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0xEA, 0x56, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0xEC, 0xD8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x04, 0xED, 0x33, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x02, 0x8E, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x08, 0x11, 0x96, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x5B, 0xD6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x6A, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x71, 0x90, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x8E, 0x90, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0x9D, 0xF4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0xBD, 0x43, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x08, 0xD2, 0x3E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x08, 0xD4, 0x0C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0x54, 0x15, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0x7A, 0x15, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0x8B, 0xFD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0x91, 0x92, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0x9A, 0x3C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0xD2, 0x92, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0xDD, 0x24, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x0C, 0xEE, 0xE6, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x10, 0x02, 0xB5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0x0B, 0xA9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0x0D, 0x7F, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x10, 0x2E, 0x00, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0x3D, 0x1C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0x4A, 0x7D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0x51, 0x07, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0xA5, 0x1D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0xDA, 0x43, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x10, 0xF0, 0x05, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x10, 0xF6, 0x0A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0x18, 0xC3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0x4F, 0x8A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0x75, 0x5B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0x85, 0x7F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0xAB, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0xF6, 0xD8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x14, 0xF8, 0x8A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x1D, 0x1A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x1D, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x26, 0x49, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x3D, 0xA2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x56, 0x80, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x5E, 0x0F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0x87, 0x96, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x18, 0xCC, 0x18, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x18, 0xFF, 0x0F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x1C, 0x1B, 0xB5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x1C, 0x36, 0xBB, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x1C, 0x4D, 0x70, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x1C, 0x99, 0x57, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x1C, 0xB0, 0x94, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x1C, 0xC1, 0x0C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0x0C, 0xC8, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x20, 0x16, 0xB9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0x1E, 0x88, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0x3A, 0x43, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0x4E, 0x7F, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x20, 0x79, 0x18, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0xC1, 0x9B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x20, 0xE5, 0x2A, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x24, 0x41, 0x8C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x24, 0x77, 0x03, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x24, 0xEE, 0x9A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0x11, 0xA8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0x16, 0xAD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0x6B, 0x35, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0x7F, 0xCF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0xB2, 0xBD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0xC5, 0xD2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0xC6, 0x3F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0xC6, 0x8E, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x28, 0xD0, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x28, 0xDF, 0xEB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0x0D, 0xA7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0x27, 0x9E, HT_IOT_PEER_OCTOSCOPE),
	OUI_TO_IOT(0x2C, 0x30, 0x33, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x2C, 0x33, 0x58, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0x6D, 0xC1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0x6E, 0x85, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0x81, 0x58, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x2C, 0x8D, 0xB1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x2C, 0xB0, 0x5D, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x2C, 0xDB, 0x07, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0x05, 0x05, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0x24, 0x32, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0x35, 0xAD, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x30, 0x3A, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0x3E, 0xA7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0x46, 0x9A, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x30, 0x89, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0xE3, 0x7A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x30, 0xF6, 0xEF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0x02, 0x86, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0x13, 0xE8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0x2E, 0xB7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0x41, 0x5D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0x7D, 0xF6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xC9, 0x3D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xCF, 0xF6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xDE, 0x1A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xE1, 0x2D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xE6, 0xAD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xF3, 0x9A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x34, 0xF6, 0x4B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0x00, 0x25, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0x68, 0x93, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0x7A, 0x0E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0x87, 0xD5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0xBA, 0xF8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0xDE, 0xAD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x38, 0xE7, 0xD8, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x38, 0xFC, 0x98, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0x21, 0x9C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0x2E, 0xFF, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x3C, 0x58, 0xC2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0x6A, 0xA7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0x9C, 0x0F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0xA9, 0xF4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0xE9, 0xF7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0xF0, 0x11, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0xF8, 0x62, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x3C, 0xFD, 0xFE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0x1C, 0x83, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0x25, 0xC2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0x5D, 0x82, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x40, 0x74, 0xE0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0x98, 0xAD, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x40, 0xA3, 0xCC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0xA6, 0xB7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x40, 0xCB, 0xC0, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x40, 0xEC, 0x99, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x44, 0x03, 0x2C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x44, 0x49, 0x88, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x44, 0x85, 0x00, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x44, 0x94, 0xFC, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x44, 0xAF, 0x28, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x44, 0xE5, 0x17, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x46, 0x8F, 0x25, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x48, 0x45, 0x20, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0x51, 0xB7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0x51, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0x68, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0x89, 0xE7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0xA1, 0x95, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x48, 0xA4, 0x72, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0xAD, 0x9A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x48, 0xF1, 0x7F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x03, 0x4F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x0F, 0x6E, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x4C, 0x1D, 0x96, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x32, 0x75, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x4C, 0x34, 0x88, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x44, 0x5B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x60, 0xDE, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x4C, 0x77, 0xCB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x79, 0x6E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x79, 0xBA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0x80, 0x93, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x4C, 0xEB, 0x42, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x28, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x2D, 0xA2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x2F, 0x9B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x32, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x50, 0x4A, 0x6E, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x50, 0x63, 0x13, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x50, 0x6A, 0x03, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x50, 0x76, 0xAF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x7C, 0x6F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0x84, 0x92, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0xE0, 0x85, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0xEB, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x50, 0xEB, 0x74, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x54, 0x14, 0xF3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x54, 0x6C, 0xEB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x54, 0x8D, 0x5A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x1C, 0xF8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x6C, 0x25, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x6D, 0x67, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x91, 0xCF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x94, 0x6B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0x96, 0x1D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0xA0, 0x23, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0xA8, 0x39, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0xCE, 0x2A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x58, 0xFB, 0x84, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0x51, 0x4F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0x5F, 0x67, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0x6D, 0x20, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x5C, 0x80, 0xB6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0x87, 0x97, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0x87, 0x9C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0xC5, 0xD4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0xCD, 0x5B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0xD2, 0xE4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0xE0, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x5C, 0xE4, 0x2A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0x36, 0xDD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0x57, 0x18, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0x67, 0x20, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0x6C, 0x66, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0xA5, 0xE2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0xDD, 0x8E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0xE3, 0x2B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0xF2, 0x62, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x60, 0xF4, 0x94, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x60, 0xF6, 0x77, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x32, 0xA8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x49, 0x7D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x4C, 0x36, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x5D, 0x86, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x6E, 0xE0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x79, 0xF0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0x80, 0x99, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0xA7, 0x69, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x64, 0xBC, 0x58, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0xD4, 0xDA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x64, 0xD6, 0x9A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x66, 0x60, 0xEC, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x68, 0x05, 0xCA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x07, 0x15, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x17, 0x29, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x3E, 0x26, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x54, 0x5A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x5D, 0x43, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0x7A, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x68, 0xEC, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0x29, 0x95, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0x4D, 0x73, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0x6A, 0x77, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0x88, 0x14, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0x94, 0x66, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0x96, 0xCF, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0xA1, 0x00, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x6C, 0xAB, 0x31, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0xB0, 0xCE, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x6C, 0xFE, 0x54, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0x1A, 0xB8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0x1C, 0xE7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0x32, 0x17, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0x9C, 0xD1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0xA2, 0xB3, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x70, 0xA6, 0xCC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0xA8, 0xD3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0xCD, 0x0D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0xCF, 0x49, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x70, 0xD8, 0x23, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0x04, 0xF1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0x13, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0x3A, 0xF4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0x44, 0x01, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x74, 0x70, 0xFD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0xD8, 0x3E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0xE5, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x74, 0xE5, 0xF9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x78, 0x0C, 0xB8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x78, 0x2B, 0x46, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x78, 0x92, 0x9C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x78, 0xAF, 0x08, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x78, 0xDD, 0x08, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x78, 0xFF, 0x57, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x79, 0xC9, 0x74, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x7C, 0x21, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x2A, 0x31, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x50, 0x49, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x7C, 0x50, 0x79, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x5C, 0xF8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x61, 0x93, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x7C, 0x67, 0xA2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x70, 0xDB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x76, 0x35, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0x7A, 0x91, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0xB0, 0xC2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0xB2, 0x7D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0xB5, 0x66, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x7C, 0xCC, 0xB8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x00, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x19, 0x34, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x32, 0x53, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x37, 0x73, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x80, 0x38, 0xFB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x45, 0xDD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x86, 0xF2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0x9B, 0x20, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x80, 0xB6, 0x55, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x14, 0x4D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x1B, 0x5E, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x84, 0x1B, 0x77, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x38, 0x35, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x84, 0x3A, 0x4B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x5C, 0xF3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x68, 0x3E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0x7A, 0x88, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x84, 0x7B, 0x57, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0xA6, 0xC8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0xC5, 0xA6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0xEF, 0x18, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x84, 0xFD, 0xD1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x88, 0x53, 0x2E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x88, 0x78, 0x73, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x88, 0xB1, 0x11, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x88, 0xD8, 0x2E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x17, 0x59, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x1D, 0x96, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x1F, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x29, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x8C, 0x3B, 0xAD, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x8C, 0x55, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x70, 0x5A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0x7C, 0xB5, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x8C, 0x85, 0x90, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x8C, 0x8D, 0x28, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0xA9, 0x82, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0xB8, 0x7E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0xC6, 0x81, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x8C, 0xF8, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x09, 0xDF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x21, 0x55, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x90, 0x2E, 0x1C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x34, 0xFC, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0x90, 0x49, 0xFA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x61, 0xAE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x65, 0x84, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0x78, 0x41, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0xCC, 0xDF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x90, 0xE2, 0xBA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x94, 0x65, 0x9C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x94, 0xB8, 0x6D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x94, 0xE2, 0x3C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x94, 0xE6, 0xF7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x94, 0xE7, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x01, 0xA7, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x98, 0x0D, 0x2E, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0x98, 0x2C, 0xBC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x3B, 0x8F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x43, 0xFA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x4F, 0xEE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x54, 0x1B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x59, 0x7A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x8D, 0x46, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x98, 0x9E, 0x63, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x98, 0xAF, 0x65, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x9C, 0x29, 0x76, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x9C, 0x3D, 0xCF, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x9C, 0x4E, 0x36, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x9C, 0xD3, 0x6D, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0x9C, 0xDA, 0x3E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0x9C, 0xFC, 0xE8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x04, 0x60, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xA0, 0x21, 0xB7, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xA0, 0x29, 0x42, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x36, 0x9F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x40, 0xA0, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xA0, 0x51, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x59, 0x50, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x63, 0x91, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xA0, 0x80, 0x69, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x88, 0x69, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0x88, 0xB4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xA4, 0xC5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xA8, 0xCD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xAF, 0xBD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xC5, 0x89, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xD3, 0x7A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xE7, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA0, 0xF4, 0x50, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xA4, 0x02, 0xB9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0x2B, 0x8C, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xA4, 0x34, 0xD9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0x42, 0x3B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0x4E, 0x31, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0x6B, 0xB6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0xB1, 0xC1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0xBF, 0x01, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0xC3, 0xF0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0xC4, 0x94, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA4, 0xF9, 0x33, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA8, 0x26, 0xD9, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xA8, 0x64, 0xF1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA8, 0x6D, 0xAA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA8, 0x7E, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xA8, 0xBE, 0x27, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xAC, 0x12, 0x03, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x19, 0x8E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x2B, 0x6E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x5A, 0xFC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x67, 0x5D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x72, 0x89, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x74, 0xB1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x7B, 0xA1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0x82, 0x47, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0xBC, 0x32, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xAC, 0xED, 0x5C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xAC, 0xFD, 0xCE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0x35, 0x9F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0x39, 0x56, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xB0, 0x3C, 0xDC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0x48, 0x1A, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xB0, 0x60, 0x88, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0x7D, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0x7F, 0xB9, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xB0, 0xA4, 0x60, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB0, 0xB9, 0x8A, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xB0, 0xDC, 0xEF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x0E, 0xDE, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x69, 0x21, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x6B, 0xFC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x6D, 0x83, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x83, 0x51, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0x96, 0x91, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0xB6, 0x76, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB4, 0xD5, 0xBD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0x03, 0x05, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0x08, 0xCF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0x63, 0x4D, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xB8, 0x81, 0x98, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0x8A, 0x60, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0x9A, 0x2A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0xB8, 0x1E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xB8, 0xBF, 0x83, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x03, 0x58, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x09, 0x1B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x0F, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x17, 0xB8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x54, 0x2F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x6E, 0xE2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0x77, 0x37, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0xA8, 0xA6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xBC, 0xCF, 0xCC, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xBC, 0xF1, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC0, 0x14, 0x3D, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0xC0, 0x3C, 0x59, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC0, 0x3F, 0x0E, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xC0, 0xA5, 0xE8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC0, 0xB6, 0xF9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC0, 0xB8, 0x83, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC0, 0xFF, 0xD4, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xC4, 0x03, 0xA8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0x04, 0x15, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xC4, 0x23, 0x60, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0x3D, 0x1A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0x3D, 0xC7, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xC4, 0x75, 0xAB, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0x84, 0x66, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xC4, 0x85, 0x08, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0xB3, 0x01, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xC4, 0xBD, 0xE5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0xD0, 0xE3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC4, 0xD9, 0x87, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0x09, 0xA8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0x21, 0x58, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0x34, 0x8E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0x58, 0xC0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0x5E, 0xA9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0xB2, 0x9B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0xCB, 0x9E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0xE2, 0x65, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xC8, 0xF7, 0x33, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xCC, 0x15, 0x31, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xCC, 0x2F, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xCC, 0x3D, 0x82, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xCC, 0x40, 0xD0, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xCC, 0xD9, 0xAC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xCC, 0xF9, 0xE4, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD0, 0x3C, 0x1F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD0, 0x57, 0x7B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD0, 0x7E, 0x35, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD0, 0xA6, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xD0, 0xAB, 0xD5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD0, 0xC6, 0x37, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0x20, 0x6D, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xD4, 0x25, 0x8B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0x3B, 0x04, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0x4B, 0x5E, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0xD4, 0x54, 0x8B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0x61, 0x9D, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xD4, 0x6D, 0x6D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0xD2, 0x52, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0xD8, 0x53, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD4, 0xE9, 0x8A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD8, 0x3B, 0xBF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD8, 0xB3, 0x77, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xD8, 0xF2, 0xCA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD8, 0xF8, 0x83, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xD8, 0xFC, 0x93, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x1B, 0xA1, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x21, 0x48, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x21, 0x5C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x41, 0xA9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x46, 0x28, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x53, 0x60, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x71, 0x96, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0x8B, 0x28, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0xA9, 0x04, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xDC, 0xA9, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xDC, 0xEF, 0x09, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xDC, 0xFB, 0x48, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0x2B, 0xE9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0x2E, 0x0B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0x46, 0x9A, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xE0, 0x91, 0xF5, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xE0, 0x94, 0x67, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0x9D, 0x31, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0xC2, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0xD0, 0x45, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0xD4, 0x64, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE0, 0xD4, 0xE8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x02, 0x9B, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x0D, 0x36, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x2B, 0x34, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xE4, 0x42, 0xA6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x5E, 0x37, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x60, 0x17, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0x70, 0xB8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xA4, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xA7, 0xA0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xB3, 0x18, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xF4, 0xC6, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xE4, 0xF8, 0x9C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xFA, 0xFD, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE4, 0xFD, 0x45, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE8, 0x2A, 0xEA, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE8, 0x84, 0xA5, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE8, 0x99, 0xC4, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xE8, 0xB1, 0xFC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE8, 0xF4, 0x08, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xE8, 0xFC, 0xAF, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xEC, 0x63, 0xD7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xEC, 0xE7, 0xA7, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0x42, 0x1C, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0x57, 0xA6, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0x76, 0x6F, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xF0, 0x77, 0xC3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0x9E, 0x4A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0xB2, 0xB9, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0xB6, 0x1E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0xD4, 0x15, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0xD5, 0xBF, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF0, 0xF0, 0x02, HT_IOT_PEER_PSP),
	OUI_TO_IOT(0xF4, 0x06, 0x69, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x26, 0x79, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x3B, 0xD8, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x46, 0x37, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x4E, 0xE3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x6D, 0x3F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x7B, 0x09, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x8C, 0x50, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0x96, 0x34, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0xA4, 0x75, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0xB3, 0x01, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0xC8, 0x8A, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0xCE, 0x23, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF4, 0xD1, 0x08, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x16, 0x54, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x34, 0x41, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x59, 0x71, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x5E, 0xA0, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x63, 0x3F, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x73, 0x94, HT_IOT_PEER_NETGEAR),
	OUI_TO_IOT(0xF8, 0x94, 0xC2, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0x9E, 0x94, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0xAC, 0x65, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0xB5, 0x4D, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0xDB, 0x7F, HT_IOT_PEER_HTC),
	OUI_TO_IOT(0xF8, 0xE4, 0xE3, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xF8, 0xF2, 0x1E, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xFC, 0x44, 0x82, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xFC, 0x77, 0x74, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xFC, 0xB3, 0xBC, HT_IOT_PEER_INTEL),
	OUI_TO_IOT(0xFC, 0xF8, 0xAE, HT_IOT_PEER_INTEL),
	// WARNING: DONOT append new OUIs to the end of array, add them in ascending order.
};
static const size_t MAC_OUI_DB_SIZE = ARRAY_SIZE(mac_oui_db);

// OUI database for vendor specific IE to IOT lookup
static const oui_t vsie_oui_db[] = {
	OUI_TO_IOT(0x00, 0x05, 0xB5, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x0A, 0xF7, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x0C, 0x43, HT_IOT_PEER_RALINK),
	OUI_TO_IOT(0x00, 0x10, 0x18, HT_IOT_PEER_BROADCOM),
	OUI_TO_IOT(0x00, 0x17, 0xF2, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x00, 0xE0, 0x4C, HT_IOT_PEER_REALTEK),
	OUI_TO_IOT(0x1C, 0x36, 0xBB, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x30, 0x35, 0xAD, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x3C, 0x2E, 0xFF, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x40, 0x98, 0xAD, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x40, 0xCB, 0xC0, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x48, 0xA1, 0x95, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x4C, 0x32, 0x75, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x50, 0x32, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0x4D, 0x73, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0x96, 0xCF, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x6C, 0xAB, 0x31, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x70, 0xA2, 0xB3, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x7C, 0x50, 0x49, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x84, 0x38, 0x35, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x8C, 0x29, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x8C, 0x85, 0x90, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x98, 0x01, 0xA7, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0x98, 0x9E, 0x63, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xA8, 0xBE, 0x27, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xAC, 0xBC, 0x32, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xB0, 0x48, 0x1A, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xB8, 0x63, 0x4D, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xC4, 0x84, 0x66, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xC4, 0xB3, 0x01, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xD0, 0xA6, 0x37, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xD4, 0x61, 0x9D, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xDC, 0xA9, 0x04, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xE4, 0x2B, 0x34, HT_IOT_PEER_APPLE),
	OUI_TO_IOT(0xF0, 0x76, 0x6F, HT_IOT_PEER_APPLE),
	// WARNING: DONOT append new OUIs to the end of array, add them in ascending order.
};
static const size_t VSIE_OUI_DB_SIZE = ARRAY_SIZE(vsie_oui_db);

/* Search IOT from OUI database */
static HT_IOT_PEER_E oui_2_iot(const unsigned int oui, const oui_t *oui_db, const int oui_db_size)
{
	int m = 0, l = 0, r = oui_db_size - 1;

	while(l <= r) {
		m = l + ((r - l) >> 1);

		switch (cmp_oui(oui, &(oui_db[m]))) {
			case 1:
				l = m + 1;
				break;
			case -1:
				r = m - 1;
				break;
			default:
				return get_iot(&(oui_db[m]));
		}
	}

	return HT_IOT_PEER_UNKNOWN;
}

/* Use STA's MAC OUI to search IOT */
HT_IOT_PEER_E mac_oui_2_iot(unsigned int oui)
{
	return oui_2_iot(oui, mac_oui_db, (int)MAC_OUI_DB_SIZE);
}

/* Use OUI in vendor specific IE to search IOT */
HT_IOT_PEER_E vsie_oui_2_iot(unsigned int oui)
{
	return oui_2_iot(oui, vsie_oui_db, (int)VSIE_OUI_DB_SIZE);
}
