//==============================================================================
//  global variable
//==============================================================================

#if defined(WFO_RADIO_SENDER)
void wfo_cfg80211_lock(struct net_device *ndev)
{
	_adapter *adapter = rtw_netdev_priv(ndev);

	if (adapter && adapter->cfg_owner_id != CURRENT_THREAD_ID) {
		WFO_DBGP("thread %u try lock netdev:%s, current owner %u",
				CURRENT_THREAD_ID, ndev->name, adapter->cfg_owner_id);
		mutex_lock(&adapter->cfg_mutex);
		WFO_DBGP("ty1hread %u get lock netdev:%s success",
				CURRENT_THREAD_ID, ndev->name);
		adapter->cfg_owner_id = CURRENT_THREAD_ID;
	}
}

void wfo_cfg80211_unlock(struct net_device *ndev)
{
	_adapter *adapter = rtw_netdev_priv(ndev);

	if (adapter) {
		if (adapter->cfg_owner_id == CURRENT_THREAD_ID) {
			WFO_DBGP("thread %u get wfo cfg mutex netdevice:%s success",
				CURRENT_THREAD_ID, ndev->name);
			adapter->cfg_owner_id = 0;
			mutex_unlock(&adapter->cfg_mutex);
		} else {
			WFO_PRINT("%s: ERROR thread %utry to unlock owner %u",
					__func__, CURRENT_THREAD_ID, adapter->cfg_owner_id);
		}
	}
}

inline wfo_cfg80211_t *get_ra_cfg_addr(struct net_device *ndev,
		unsigned int side,
		unsigned int cmd)
{
	_adapter *adapter = NULL;
	wfo_cfg80211_t *cfg = NULL;
	u8 dev_id = 0xf;

	if (!ndev) {
		goto exit;
	}

	wfo_cfg80211_lock(ndev);

	adapter = rtw_netdev_priv(ndev);
	if (adapter) {
	#ifdef CONFIG_RTW_MULTI_DEV_MULTI_BAND
		dev_id = adapter->dvobj->dev_id;
	#else
		dev_id = adapter->dev_id;
	#endif

		if (dev_id == WFO_SKIP_PCIE_SLOT) {
			cfg = &adapter->cfg;
			if (cfg) {
				memset(cfg, 0, sizeof(wfo_cfg80211_t));

				cfg->dev_idx = adapter->wfo_mapid;
				cfg->side = side;
				cfg->cmd = cmd;
				cfg->ndev = (void *)ndev;
				strncpy(cfg->dev_name, ndev->name, strlen(ndev->name));
			}
		}
	}

exit:
	if (!cfg) {
		WFO_WARN("cmd[%d][%s], ndev=%p, dev_id[%d], cfg[%p]\n",
			cmd, wfo_cmd_str[cmd],
			ndev, dev_id, cfg);
	}
	return cfg;
}
#endif /* WFO_RADIO_SENDER */

//==============================================================================
//	radio cmd
//==============================================================================
/*
# insmod /etc/conf/8852ae_98d.ko 
<4>8852ae_98d: Unknown symbol cfg80211_del_sta_sinfo (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_vendor_cmd_reply (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_ibss_joined (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_ch_switch_notify (err 0)
<4>8852ae_98d: Unknown symbol ieee80211_frequency_to_channel (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_ready_on_channel (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_inform_bss_frame_data (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_scan_done (err 0)
<4>8852ae_98d: Unknown symbol __cfg80211_alloc_reply_skb (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_new_sta (err 0)
<4>8852ae_98d: Unknown symbol wiphy_new_nm (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_roamed (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_mgmt_tx_status (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_rx_mgmt (err 0)
<4>8852ae_98d: Unknown symbol wiphy_unregister (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_remain_on_channel_expired (err 0)
<4>8852ae_98d: Unknown symbol backport_dependency_symbol (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_put_bss (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_unlink_bss (err 0)
<4>8852ae_98d: Unknown symbol ieee80211_get_channel (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_disconnected (err 0)
<4>8852ae_98d: Unknown symbol __cfg80211_send_event_skb (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_michael_mic_failure (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_external_auth_request (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_connect_done (err 0)
<4>8852ae_98d: Unknown symbol wiphy_register (err 0)
<4>8852ae_98d: Unknown symbol wiphy_free (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_get_bss (err 0)
<4>8852ae_98d: Unknown symbol cfg80211_ch_switch_started_notify (err 0)
<4>8852ae_98d: Unknown symbol __cfg80211_alloc_event_skb (err 0)
*/

#if defined(WFO_RADIO_SENDER) && defined(CPTCFG_WFO_VIRT_SAME_CPU)
#if !defined(CONFIG_RTK_WFO_NO_VIRT)
#undef cfg80211_new_sta
#undef cfg80211_del_sta
	/* cfg80211_del_sta_sinfo */
#undef cfg80211_scan_done
#undef cfg80211_inform_bss_frame
	/* cfg80211_inform_bss_frame_data */
#undef cfg80211_put_bss
#undef cfg80211_ch_switch_started_notify
#undef cfg80211_get_bss
#undef cfg80211_connect_result
#undef cfg80211_vendor_cmd_reply
#undef cfg80211_ibss_joined
#undef cfg80211_ch_switch_notify
#undef ieee80211_frequency_to_channel
#undef cfg80211_ready_on_channel
#undef cfg80211_vendor_cmd_alloc_reply_skb
	/* __cfg80211_alloc_reply_skb */
#undef cfg80211_roamed
#undef cfg80211_mgmt_tx_status
#undef cfg80211_rx_mgmt
#undef cfg80211_rx_unprot_mlme_mgmt
#undef cfg80211_remain_on_channel_expired
#undef cfg80211_unlink_bss
#undef cfg80211_disconnected
#undef cfg80211_vendor_event
	/* __cfg80211_send_event_skb */
#undef cfg80211_michael_mic_failure
#undef cfg80211_vendor_event_alloc
	/* __cfg80211_alloc_event_skb */
#undef cfg80211_external_auth_request
#undef ieee80211_get_channel
#undef cfg80211_connect_done
#endif /* !CONFIG_RTK_WFO_NO_VIRT */
#endif /* WFO_RADIO_SENDER && CPTCFG_WFO_VIRT_SAME_CPU */

//----------------------------------------------------------------------------
//ieee80211_get_channel

#if defined(WFO_RADIO_SENDER)
struct ieee80211_channel *wfo_ieee80211_get_channel(struct wiphy *wiphy,
		int freq)
{
	enum nl80211_band band;
	struct ieee80211_supported_band *sband;
	int i;

	for (band = 0; band < NUM_NL80211_BANDS; band++) {
		sband = wiphy->bands[band];

		if (!sband)
			continue;

		for (i = 0; i < sband->n_channels; i++) {
			if (sband->channels[i].center_freq == freq)
				return &sband->channels[i];
		}
	}

	return NULL;
}
#endif /* WFO_RADIO_SENDER */


//----------------------------------------------------------------------------
//ieee80211_frequency_to_channel

#if defined(WFO_RADIO_SENDER)
int wfo_ieee80211_frequency_to_channel(int freq)
{
	/* see 802.11 17.3.8.3.2 and Annex J */
	if (freq == 2484)
		return 14;
	else if (freq < 2484)
		return (freq - 2407) / 5;
	else if (freq >= 4910 && freq <= 4980)
		return (freq - 4000) / 5;
	else if (freq <= 45000) /* DMG band lower limit */
		return (freq - 5000) / 5;
	else if (freq >= 58320 && freq <= 70200)
		return (freq - 56160) / 2160;
	else
		return 0;
}
#endif /* WFO_RADIO_SENDER */


//----------------------------------------------------------------------------
//cfg80211_new_sta

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_new_sta(struct net_device *radio_ndev,
		const u8 *mac_addr,
		struct station_info *sinfo,
		gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_new_sta));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_new_sta(radio_ndev, mac_addr, sinfo, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_new_sta));

	if (mac_addr) {
		cfg->cfg80211_new_sta.mflags |= (1<<WFO_CFG80211_NEW_STA_HAS_MAC);
		memcpy(&cfg->cfg80211_new_sta.mac_addr, mac_addr, ETH_ALEN);
	}

	if (sinfo) {
		cfg->cfg80211_new_sta.mflags |= (1<<WFO_CFG80211_NEW_STA_HAS_SINFO);
		memcpy(&cfg->cfg80211_new_sta.sinfo.source, sinfo,
			sizeof(struct station_info));
		if (sinfo->assoc_req_ies)
			memcpy(&cfg->cfg80211_new_sta.sinfo.assoc_req_ies,
				sinfo->assoc_req_ies,
				MIN(sinfo->assoc_req_ies_len, IEEE80211_MAX_DATA_LEN));
		if (sinfo->pertid)
			memcpy(&cfg->cfg80211_new_sta.sinfo.pertid, sinfo->pertid,
				sizeof(struct cfg80211_tid_stats));
	}

	cfg->cfg80211_new_sta.gfp = gfp;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_new_sta));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_new_sta(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct station_info _sinfo, *sinfo = &_sinfo;
	u8 *mac_addr = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_new_sta));

	if (cfg->cfg80211_new_sta.mflags & (1<<WFO_CFG80211_NEW_STA_HAS_MAC))
		mac_addr = cfg->cfg80211_new_sta.mac_addr;

	if (cfg->cfg80211_new_sta.mflags & (1<<WFO_CFG80211_NEW_STA_HAS_SINFO)) {
		memset(sinfo, 0, sizeof(struct station_info));

		memcpy(sinfo, &cfg->cfg80211_new_sta.sinfo.source,
			sizeof(struct station_info));
		sinfo->assoc_req_ies = sinfo->assoc_req_ies_len? \
			cfg->cfg80211_new_sta.sinfo.assoc_req_ies:NULL;
		sinfo->pertid = sinfo->pertid? \
			&cfg->cfg80211_new_sta.sinfo.pertid:NULL;
	} else {
		sinfo = NULL;
	}

	cfg80211_new_sta(virt_ndev, mac_addr, sinfo, cfg->cfg80211_new_sta.gfp);

	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_new_sta));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
//cfg80211_del_sta

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_del_sta(struct net_device *radio_ndev,
		const u8 *mac_addr,
		gfp_t gfp)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_del_sta));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_del_sta(radio_ndev, mac_addr, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_del_sta));

	if (mac_addr) {
		cfg->cfg80211_del_sta.mflags |= (1<<WFO_CFG80211_DEL_STA_HAS_MAC);
		memcpy(&cfg->cfg80211_del_sta.mac_addr, mac_addr, ETH_ALEN);
	}

	cfg->cfg80211_del_sta.gfp = gfp;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_del_sta));
	return;
}

#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_del_sta(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	u8 *mac_addr = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_del_sta));

	if (cfg->cfg80211_del_sta.mflags & (1<<WFO_CFG80211_DEL_STA_HAS_MAC))
		mac_addr = cfg->cfg80211_del_sta.mac_addr;

	cfg80211_del_sta(virt_ndev, mac_addr, cfg->cfg80211_del_sta.gfp);

	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_del_sta));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
//cfg80211_scan_done

#if !defined(CONFIG_RTK_WFO_NO_VIRT)
#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_scan_done(
		struct cfg80211_scan_request *request,
		struct cfg80211_scan_info *info)
{
	int ret = 0;
	struct net_device *radio_ndev = wdev_to_ndev(request->wdev);
	int i, len = 0;
	wfo_cfg80211_t *cfg = NULL;

	if (!request) {
		printk("%s(): request=%p\n", __func__, request);
		return;
	}
	if (!info) {
		printk("%s(): info=%p\n", __func__, info);
		return;
	}

	cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_scan_done));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_scan_done(request, info);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_scan_done));

	if (request) {
		cfg->scan.mflags |= (1<<WFO_SCAN_HAS_REQUEST);

		/* copy request with *channel/ssids/ie */
		len = sizeof(*request) +
			sizeof(*request->ssids) * request->n_ssids +
			sizeof(*request->channels) * request->n_channels +
			request->ie_len;

	#if (WFO_SHRINK_CFG_SIZE==1)
	{
		wfo_dram_t *dram = RTK_SHM_DRAM;
		memcpy(&dram->md.cfg.scan.data.source, request, len);

		if (request->n_ssids) {
			for (i=0;i<request->n_ssids;i++) {
				memcpy(&dram->md.cfg.scan.data.ssids[i], &request->ssids[i],
					sizeof(struct cfg80211_ssid));
			}
		}

		if (request->ie_len) {
			memcpy(dram->md.cfg.scan.data.ie, request->ie, request->ie_len);
		}

		/* copy channel data */
		for (i=0; i<request->n_channels; i++) {
			memcpy(&dram->md.cfg.scan.data.channels_data[i], request->channels[i],
				sizeof(struct ieee80211_channel));
		}
	}
	#else /* !WFO_SHRINK_CFG_SIZE */
		memcpy(&cfg->scan.data.source, request, len);

		if (request->n_ssids) {
			for (i=0;i<request->n_ssids;i++) {
				memcpy(&cfg->scan.data.ssids[i] , &request->ssids[i],
					sizeof(struct cfg80211_ssid));
			}
		}

		if (request->ie_len) {
			memcpy(cfg->scan.data.ie, request->ie, request->ie_len);
		}

		/* copy channel data */
		for (i=0; i<request->n_channels; i++) {
			memcpy(&cfg->scan.data.channels_data[i], request->channels[i],
				sizeof(struct ieee80211_channel));
		}
	#endif /* WFO_SHRINK_CFG_SIZE */
	}

	if (info) {
		cfg->scan.mflags |= (1<<WFO_SCAN_HAS_INFO);
		memcpy(&cfg->scan.info, info, sizeof(struct cfg80211_scan_info));
	}

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_scan_done));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_scan_done(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct wfo_ndev_priv *ndev_priv =(struct wfo_ndev_priv *)netdev_priv(virt_ndev);
	struct cfg80211_scan_request *request = ndev_priv->scan_req; /* backup at wfo_virt_cmd_sender_scan */
	struct cfg80211_scan_info *info = NULL;
	unsigned int i, len = 0U;
	wfo_dram_t *dram = RTK_SHM_DRAM;

#ifdef WFO_K510
	request = ndev_priv->int_scan_req ? ndev_priv->int_scan_req : ndev_priv->scan_req;
	ndev_priv->int_scan_req = NULL;
#endif /* WFO_K510 */

	ndev_priv->scan_req = NULL;
	if(!request) {
		WFO_PRINT("dev=%s, request is null\n", virt_ndev->name);
		return ret;
	}

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_scan_done));

	if (cfg->scan.mflags & (1<<WFO_SCAN_HAS_REQUEST)) {
	#if (WFO_SHRINK_CFG_SIZE==1)
		memcpy(request, &dram->md.cfg.scan.data.source,
			sizeof(struct cfg80211_scan_request));

		len = sizeof(*request) +
			sizeof(*request->ssids) * (u32)request->n_ssids +
			sizeof(*request->channels) * request->n_channels +
			request->ie_len;

		memcpy(request, &dram->md.cfg.scan.data, len);

		if (request->n_ssids)
			request->ssids = (void *)&dram->md.cfg.scan.data.ssids[0];

		if (request->ie_len)
			request->ie = dram->md.cfg.scan.data.ie;

		for (i=0U; i<request->n_channels; i++) {
			request->channels[i] = &dram->md.cfg.scan.data.channels_data[i];
		}
	#else /* !WFO_SHRINK_CFG_SIZE */
		memcpy(request, &cfg->scan.data.source,
			sizeof(struct cfg80211_scan_request));

		len = sizeof(*request) +
			sizeof(*request->ssids) * (u32)request->n_ssids +
			sizeof(*request->channels) * request->n_channels +
			request->ie_len;

		memcpy(request, &cfg->scan.data, len);

		if (request->n_ssids)
			request->ssids = (void *)&cfg->scan.data.ssids[0];

		if (request->ie_len)
			request->ie = cfg->scan.data.ie;

		for (i=0U; i<request->n_channels; i++) {
			request->channels[i] = &cfg->scan.data.channels_data[i];
		}
	#endif /* WFO_SHRINK_CFG_SIZE */
		request->wdev = ndev_to_wdev(virt_ndev);
		request->wiphy = wiphy;
	}

	//if (cfg->scan.mflags & (1<<WFO_SCAN_HAS_INFO)) {
		info = &request->info;
	//}

	cfg80211_scan_done(request, info);
	cfg->ret_val = ret;

	dram->is_scanning = 0U;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_scan_done));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_inform_bss_frame(wiphy, rx_channel, mgmt, len, signal, gfp)

#if defined(WFO_RADIO_SENDER)
struct cfg80211_bss *wfo_radio_cmd_sender_cfg80211_inform_bss_frame(struct wiphy *wiphy,
			  struct ieee80211_channel *rx_channel,
			  struct ieee80211_mgmt *mgmt, size_t len,
			  s32 signal, gfp_t gfp, u32 freq)
{
	struct cfg80211_bss *ret = NULL;
	struct net_device *radio_ndev = wiphy_to_ndev(wiphy);
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_inform_bss_frame));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		ret = cfg80211_inform_bss_frame(wiphy, rx_channel, mgmt, len, signal, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return ret;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_inform_bss_frame));

	WFO_DBGP_CLIENT("freq[%d]\n", freq);

	if (mgmt) {
		/* copy mgmt & mgmt->u.probe_resp.variable*/
		memcpy(&cfg->cfg80211_inform_bss_frame.mgmt, mgmt, len);
	}

	cfg->cfg80211_inform_bss_frame.len = len;
	cfg->cfg80211_inform_bss_frame.signal = signal;
	cfg->cfg80211_inform_bss_frame.gfp = gfp;
	cfg->cfg80211_inform_bss_frame.freq = freq;

	wfo_radio_cmd_sender(cfg);
#if defined(WFO_ASYNC_INFORM_BSS)
	ret = (struct cfg80211_bss *)0xFFFFFFFF;
#else /* !WFO_ASYNC_INFORM_BSS */
	ret = (struct cfg80211_bss *)cfg->ret_val;
#endif /* WFO_ASYNC_INFORM_BSS */

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_inform_bss_frame));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_inform_bss_frame(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct cfg80211_bss *bss = NULL;
	struct ieee80211_channel *channel = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_inform_bss_frame));

	if (cfg->cfg80211_inform_bss_frame.freq)
		channel = ieee80211_get_channel(wiphy, (s32)cfg->cfg80211_inform_bss_frame.freq);

	WFO_DBGP_CLIENT("wiphy[0x%p], channel[0x%p], freq[%d]\n",
		wiphy, channel, cfg->cfg80211_inform_bss_frame.freq);

	if (unlikely(!channel)) {
		if (net_ratelimit()) WFO_PRINT("BSS NULL\n");
		goto exit;
	}

	WFO_DBGP_CLIENT("channel[0x%p], band[%d], center_freq[%d]\n",
		channel, channel->band, channel->center_freq);

	bss = cfg80211_inform_bss_frame(wiphy,
		channel,
		&cfg->cfg80211_inform_bss_frame.mgmt,
		cfg->cfg80211_inform_bss_frame.len,
		cfg->cfg80211_inform_bss_frame.signal,
		cfg->cfg80211_inform_bss_frame.gfp);
	cfg->ret_val = ret = (int)bss;

#if defined(WFO_ASYNC_INFORM_BSS)
	if (unlikely(!bss)) {
		if (net_ratelimit()) {
			WFO_PRINT("BSS NULL\n");
		}
		goto exit;
	}
	cfg80211_put_bss(wiphy, bss);
#endif /* WFO_ASYNC_INFORM_BSS */

exit:
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_inform_bss_frame));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_put_bss(wiphy, pub)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_put_bss(struct wiphy *wiphy, struct cfg80211_bss *bss)
{
	struct net_device *radio_ndev = wiphy_to_ndev(wiphy);
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_put_bss));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_put_bss(wiphy, bss);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_put_bss));

	cfg->cfg80211_put_bss.bss = bss;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_put_bss));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_put_bss(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_put_bss));

	cfg80211_put_bss(wiphy, (struct cfg80211_bss *)cfg->cfg80211_put_bss.bss);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_put_bss));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
/* void cfg80211_ch_switch_started_notify(struct net_device *dev,
				       struct cfg80211_chan_def *chandef,
				       u8 count); */


#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_ch_switch_started_notify(struct net_device *dev,
				       struct cfg80211_chan_def *chandef,
				       u8 count)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(cfg80211_ch_switch_started_notify));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_ch_switch_started_notify(dev, chandef, count);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_ch_switch_started_notify));

	wfo_copy_struct_data(wfo_ipc_sender,
		WFO_STRUCT_TYPE(cfg80211_chan_def),
		&cfg->cfg80211_ch_switch_started_notify.data.source,
		chandef);

	cfg->cfg80211_ch_switch_started_notify.count = count;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_ch_switch_started_notify));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_ch_switch_started_notify(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct cfg80211_chan_def _chandef, *chandef = &_chandef;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_ch_switch_started_notify));

	wfo_copy_struct_data(wfo_ipc_receiver,
		WFO_STRUCT_TYPE(cfg80211_chan_def),
		chandef,
		&cfg->cfg80211_ch_switch_started_notify.data.source);
	chandef->chan =
		ieee80211_get_channel(wiphy, (s32)chandef->chan->center_freq);

	cfg80211_ch_switch_started_notify(virt_ndev, chandef,
		cfg->cfg80211_ch_switch_started_notify.count);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_ch_switch_started_notify));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
/* struct cfg80211_bss *cfg80211_get_bss(struct wiphy *wiphy,
				      struct ieee80211_channel *channel,
				      const u8 *bssid,
				      const u8 *ssid, size_t ssid_len,
				      enum ieee80211_bss_type bss_type,
				      enum ieee80211_privacy privacy); */

#if defined(WFO_RADIO_SENDER)
struct cfg80211_bss *wfo_radio_cmd_sender_cfg80211_get_bss(struct wiphy *wiphy,
				      struct ieee80211_channel *channel,
				      const u8 *bssid,
				      const u8 *ssid, size_t ssid_len,
				      enum ieee80211_bss_type bss_type,
				      enum ieee80211_privacy privacy)
{
	struct cfg80211_bss *ret = NULL;
	struct net_device *radio_ndev = wiphy_to_ndev(wiphy);
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_get_bss));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		ret = cfg80211_get_bss(wiphy, channel, bssid, ssid, ssid_len, bss_type, privacy);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return ret;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_get_bss));

	if (channel)
		cfg->cfg80211_get_bss.freq = channel->center_freq;
	if (bssid)
		memcpy(&cfg->cfg80211_get_bss.bssid, bssid, ETH_ALEN);
	if (ssid)
		memcpy(&cfg->cfg80211_get_bss.ssid, ssid, ssid_len);

	cfg->cfg80211_get_bss.ssid_len = ssid_len;
	cfg->cfg80211_get_bss.bss_type = bss_type;
	cfg->cfg80211_get_bss.privacy = privacy;

	wfo_radio_cmd_sender(cfg);
	ret = (struct cfg80211_bss *)cfg->ret_val;

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_get_bss));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_get_bss(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct ieee80211_channel *channel = NULL;
	u8 *bssid = NULL;
	u8 *ssid = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_get_bss));

	if (cfg->cfg80211_get_bss.freq)
		channel = ieee80211_get_channel(wiphy, (s32)cfg->cfg80211_get_bss.freq);
	bssid = (u8 *)&cfg->cfg80211_get_bss.bssid;
	ssid = (u8 *)&cfg->cfg80211_get_bss.ssid;

	ret = (int)cfg80211_get_bss(wiphy, channel, bssid, ssid,
			cfg->cfg80211_get_bss.ssid_len,
			cfg->cfg80211_get_bss.bss_type,
			cfg->cfg80211_get_bss.privacy);
	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_get_bss));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
/* void cfg80211_connect_result(struct net_device *dev, const u8 *bssid,
			const u8 *req_ie, size_t req_ie_len,
			const u8 *resp_ie, size_t resp_ie_len,
			u16 status, gfp_t gfp) */

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_connect_result(struct net_device *dev, const u8 *bssid,
			const u8 *req_ie, size_t req_ie_len,
			const u8 *resp_ie, size_t resp_ie_len,
			u16 status, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(cfg80211_connect_result));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_connect_result(dev, bssid, req_ie, req_ie_len,
			resp_ie, resp_ie_len, status, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_connect_result));

	if (bssid) {
		cfg->cfg80211_connect_result.mflags |= (1<<WFO_CFG80211_CONNECT_RESULT_HAS_BSSID);
		memcpy(&cfg->cfg80211_connect_result.bssid, bssid, ETH_ALEN);
	}
	if (req_ie) {
		cfg->cfg80211_connect_result.mflags |= (1<<WFO_CFG80211_CONNECT_RESULT_HAS_REQ_IE);
		memcpy(&cfg->cfg80211_connect_result.req_ie, req_ie, req_ie_len);
		cfg->cfg80211_connect_result.req_ie_len= req_ie_len;
	}
	if (resp_ie) {
		cfg->cfg80211_connect_result.mflags |= (1<<WFO_CFG80211_CONNECT_RESULT_HAS_RESP_IE);
		memcpy(&cfg->cfg80211_connect_result.resp_ie, resp_ie, resp_ie_len);
		cfg->cfg80211_connect_result.resp_ie_len = resp_ie_len;
	}

	cfg->cfg80211_connect_result.status = status;
	cfg->cfg80211_connect_result.gfp = gfp;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_connect_result));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_connect_result(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	u8 *bssid = NULL, *req_ie = NULL, *resp_ie = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_connect_result));

	if (cfg->cfg80211_connect_result.mflags & (1<<WFO_CFG80211_CONNECT_RESULT_HAS_BSSID))
		bssid = (u8 *)&cfg->cfg80211_connect_result.bssid;

	if (cfg->cfg80211_connect_result.mflags & (1<<WFO_CFG80211_CONNECT_RESULT_HAS_REQ_IE))
		req_ie = (u8 *)&cfg->cfg80211_connect_result.req_ie;

	if (cfg->cfg80211_connect_result.mflags & (1<<WFO_CFG80211_CONNECT_RESULT_HAS_RESP_IE))
		resp_ie = (u8 *)&cfg->cfg80211_connect_result.resp_ie;

	cfg80211_connect_result(virt_ndev, bssid,
		req_ie, cfg->cfg80211_connect_result.req_ie_len,
		resp_ie, cfg->cfg80211_connect_result.resp_ie_len,
		cfg->cfg80211_connect_result.status,
		cfg->cfg80211_connect_result.gfp);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_connect_result));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_rx_mgmt(wdev , freq , sig_dbm , buf , len , gfp)

#if defined(WFO_RADIO_SENDER)
bool wfo_radio_cmd_sender_cfg80211_rx_mgmt(struct wireless_dev *wdev, int freq, int sig_dbm,
		const u8 *buf, size_t len, u32 flags)
{
	bool ret = 0;
	struct net_device *radio_ndev = wdev_to_ndev(wdev);
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(radio_ndev, RUN_REMOTELY, WFO_CMD(cfg80211_rx_mgmt));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		ret = cfg80211_rx_mgmt(wdev, freq, sig_dbm, buf, len, flags);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return ret;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_rx_mgmt));

	cfg->cfg80211_rx_mgmt.freq = freq;
	cfg->cfg80211_rx_mgmt.sig_dbm = sig_dbm;
	if (len < RX_MGMT_MAX_LEN) {
		cfg->cfg80211_rx_mgmt.len = len;
	} else {
		WFO_WARN("len[%d] > MAX_LEN[%d]! Please check!!\n", len, RX_MGMT_MAX_LEN);
		cfg->cfg80211_rx_mgmt.len = RX_MGMT_MAX_LEN;
	}
	cfg->cfg80211_rx_mgmt.flags = flags;
	if (buf)
		memcpy(&cfg->cfg80211_rx_mgmt.buf, buf, cfg->cfg80211_rx_mgmt.len);

	wfo_radio_cmd_sender(cfg);
	//ret = cfg->ret_val;

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_rx_mgmt));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_rx_mgmt(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct wireless_dev *wdev = ndev_to_wdev(virt_ndev);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_rx_mgmt));

	ret = cfg80211_rx_mgmt(wdev,
		cfg->cfg80211_rx_mgmt.freq,
		cfg->cfg80211_rx_mgmt.sig_dbm,
		cfg->cfg80211_rx_mgmt.buf,
		cfg->cfg80211_rx_mgmt.len,
		cfg->cfg80211_rx_mgmt.flags);
	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_rx_mgmt));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_rx_unprot_mlme_mgmt

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_rx_unprot_mlme_mgmt(struct net_device *dev, const u8 *buf, size_t len)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(cfg80211_rx_unprot_mlme_mgmt));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_rx_unprot_mlme_mgmt(dev, buf, len);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_rx_unprot_mlme_mgmt));

	if (len < RX_MGMT_MAX_LEN) {
		cfg->cfg80211_rx_unprot_mlme_mgmt.len = len;
	} else {
		WFO_WARN("len[%d] > MAX_LEN[%d]! Please check!!\n", len, RX_MGMT_MAX_LEN);
		cfg->cfg80211_rx_unprot_mlme_mgmt.len = RX_MGMT_MAX_LEN;
	}

	if (buf)
		memcpy(&cfg->cfg80211_rx_unprot_mlme_mgmt.buf, buf, cfg->cfg80211_rx_unprot_mlme_mgmt.len);

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_rx_unprot_mlme_mgmt));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_rx_unprot_mlme_mgmt(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_rx_unprot_mlme_mgmt));

	cfg80211_rx_unprot_mlme_mgmt(virt_ndev,
		cfg->cfg80211_rx_unprot_mlme_mgmt.buf,
		cfg->cfg80211_rx_unprot_mlme_mgmt.len);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_rx_unprot_mlme_mgmt));
	return 0;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_external_auth_request(dev, params, gfp)

#if defined(WFO_RADIO_SENDER)
int wfo_radio_cmd_sender_cfg80211_external_auth_request(struct net_device *dev,
		 struct cfg80211_external_auth_params *params, gfp_t gfp)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(cfg80211_external_auth_request));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		ret = cfg80211_external_auth_request(dev, params, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return ret;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_external_auth_request));

	memcpy(&cfg->cfg80211_external_auth_request.params, params, sizeof(struct cfg80211_external_auth_params));
	cfg->cfg80211_external_auth_request.gfp = gfp;

	wfo_radio_cmd_sender(cfg);
	ret = cfg->ret_val;

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_external_auth_request));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_external_auth_request(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_external_auth_request));

	ret = cfg80211_external_auth_request(virt_ndev, &cfg->cfg80211_external_auth_request.params, cfg->cfg80211_external_auth_request.gfp);

	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_external_auth_request));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
//cfg80211_disconnected(wdev_to_ndev(wdev), reason, ie, ie_len, locally_generated, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_disconnected(struct net_device *ndev, u16 reason,
		const u8 *ie, size_t ie_len,
		bool locally_generated, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(ndev, RUN_REMOTELY, WFO_CMD(cfg80211_disconnected));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_disconnected(ndev, reason, ie, ie_len, locally_generated, gfp);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_disconnected));

	cfg->cfg80211_disconnected.reason = reason;
	cfg->cfg80211_disconnected.ie_len = ie_len;
	cfg->cfg80211_disconnected.locally_generated = locally_generated;
	cfg->cfg80211_disconnected.gfp = gfp;
	if (ie_len)
		memcpy(&cfg->cfg80211_disconnected.ie, ie, ie_len);

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_disconnected));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_disconnected(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_disconnected));

	cfg80211_disconnected(virt_ndev,
		cfg->cfg80211_disconnected.reason,
		cfg->cfg80211_disconnected.ie,
		cfg->cfg80211_disconnected.ie_len,
		cfg->cfg80211_disconnected.locally_generated,
		cfg->cfg80211_disconnected.gfp);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_disconnected));
	return 0;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
// for rtk_eventd_netlink_send

#if defined(CONFIG_RTK_WLAN_EVENT_INDICATE)
#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_rtk_eventd_netlink_send(int rtk_eventd_pid, struct sock *nl_eventd_sk, int eventID, char *ifname, char *data, int data_len)
{
	struct net_device *ndev = wfo_dev_get_by_name(&init_net, WLAN_WFOVIRT);
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(ndev, RUN_REMOTELY, WFO_CMD(rtk_eventd_netlink_send));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(rtk_eventd_netlink_send));

	if (data != NULL)
		memcpy(&cfg->rtk_eventd_netlink_send.data, data, data_len);
	if (ifname != NULL)
		memcpy(&cfg->rtk_eventd_netlink_send.ifname, ifname, strlen(ifname)+1);

	cfg->rtk_eventd_netlink_send.data_len = data_len;
	cfg->rtk_eventd_netlink_send.eventID = eventID;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(rtk_eventd_netlink_send));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_rtk_eventd_netlink_send(struct wiphy *wiphy,
	struct net_device *virt_ndev,
	wfo_cfg80211_t *cfg)
{
	int ret = 0;
	extern int get_nl_eventd_pid(void);
	extern struct sock *get_nl_eventd_sk(void);
	extern void rtk_eventd_netlink_send(int pid, struct sock *nl_sk, int eventID, char *ifname, char *data, int data_len);
	int rtk_eventd_pid = get_nl_eventd_pid();
	struct sock* nl_eventd_sk = get_nl_eventd_sk();

	WFO_RECEIVER_TRACE_IN(WFO_CMD(rtk_eventd_netlink_send));

	if (cfg->rtk_eventd_netlink_send.ifname[0] == '\0') {
		rtk_eventd_netlink_send(rtk_eventd_pid, nl_eventd_sk, cfg->rtk_eventd_netlink_send.eventID,
					NULL, &cfg->rtk_eventd_netlink_send.data, cfg->rtk_eventd_netlink_send.data_len);
	}
	else {
		rtk_eventd_netlink_send(rtk_eventd_pid, nl_eventd_sk, cfg->rtk_eventd_netlink_send.eventID,
					&cfg->rtk_eventd_netlink_send.ifname, &cfg->rtk_eventd_netlink_send.data, cfg->rtk_eventd_netlink_send.data_len);
	}


	WFO_RECEIVER_TRACE_OUT(WFO_CMD(rtk_eventd_netlink_send));

	return ret;
}
#endif /* WFO_RADIO_RECEIVER */
#endif /* CONFIG_RTK_WLAN_EVENT_INDICATE */

//----------------------------------------------------------------------------
// for rtw_netlink_send_msg

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_netlink_send_msg(struct net_device *ndev, void *msg, u32 msg_len)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(ndev, RUN_REMOTELY, WFO_CMD(netlink_send_msg));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(netlink_send_msg));

	memcpy(&cfg->netlink_send_msg.msg, msg, msg_len);
	cfg->netlink_send_msg.msg_len = msg_len;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(netlink_send_msg));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_netlink_send_msg(struct wiphy *wiphy,
	struct net_device *virt_ndev,
	wfo_cfg80211_t *cfg)
{
	int ret = 0;
	extern void rtw_netlink_send_msg(void *msg, u32 msg_len);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(netlink_send_msg));

#ifdef CONFIG_WLAN_MANAGER
	rtw_netlink_send_msg((void *)&cfg->netlink_send_msg.msg, cfg->netlink_send_msg.msg_len);
#endif

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(netlink_send_msg));

	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
// for _core_map_nl_event_send_fragment

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_core_map_nl_event_send_fragment(struct net_device *ndev, void *data, u32 msg_len)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(ndev, RUN_REMOTELY, WFO_CMD(core_map_nl_event_send_fragment));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(core_map_nl_event_send_fragment));

	memcpy(&cfg->core_map_nl_event_send_fragment.data, data, msg_len);
	cfg->core_map_nl_event_send_fragment.msg_len = msg_len;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(core_map_nl_event_send_fragment));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_core_map_nl_event_send_fragment(struct wiphy *wiphy,
                struct net_device *virt_ndev,
                wfo_cfg80211_t *cfg)
{
	int ret = 0;
	extern void _core_map_nl_event_send_fragment(void *data, u32 msg_len);
	WFO_RECEIVER_TRACE_IN(WFO_CMD(core_map_nl_event_send_fragment));

#ifdef CONFIG_RTW_MULTI_AP
	_core_map_nl_event_send_fragment((void *)&cfg->core_map_nl_event_send_fragment.data, cfg->core_map_nl_event_send_fragment.msg_len);
#endif

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(core_map_nl_event_send_fragment));

	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


#if !defined(CPTCFG_WFO_VIRT_SAME_CPU)
//----------------------------------------------------------------------------
//open rfe txt

#if defined(WFO_RADIO_SENDER)
u32 wfo_radio_cmd_sender_open_file(struct net_device *dev, const char *str, u32 para_size, u8 *para_buf)
{
	u32 ret = 0U;
	wfo_dram_t *dram = RTK_SHM_DRAM;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(open_file));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(open_file));

	strcpy(cfg->open_file.str, str);
	wfo_radio_cmd_sender(cfg);
	memcpy(para_buf, dram->md.cfg.open_file.sz, cfg->open_file.size);
	ret = cfg->open_file.size;

	WFO_SENDER_TRACE_OUT(WFO_CMD(open_file));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_open_file(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	wfo_dram_t *dram = RTK_SHM_DRAM;
	extern int rtw_retrieve_from_file(const char *path, u8 *buf, u32 sz);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(open_file));

	cfg->open_file.size = (u32)rtw_retrieve_from_file(
		cfg->open_file.str, dram->md.cfg.open_file.sz, (u32)MD_SZ);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(open_file));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */
#endif /* !CPTCFG_WFO_VIRT_SAME_CPU */

#if !defined(CPTCFG_WFO_VIRT_SAME_CPU)
//----------------------------------------------------------------------------
//is file readable

#if defined(WFO_RADIO_SENDER)
u32 wfo_radio_cmd_sender_is_file_readable(struct net_device *dev, const char *str)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(is_file_readable));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(is_file_readable));

	strcpy(cfg->open_file.str, str);

	wfo_radio_cmd_sender(cfg);
	ret = cfg->ret_val;

	WFO_SENDER_TRACE_OUT(WFO_CMD(is_file_readable));

	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_is_file_readable(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	extern int rtw_is_file_readable(const char *path);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(is_file_readable));

	cfg->ret_val = rtw_is_file_readable(cfg->open_file.str);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(is_file_readable));
	return 0;
}
#endif /* WFO_RADIO_RECEIVER */
#endif /* !CPTCFG_WFO_VIRT_SAME_CPU */

#if !defined(CPTCFG_WFO_VIRT_SAME_CPU)
//----------------------------------------------------------------------------
// wlan proc

#if defined(WFO_RADIO_SENDER)
u32 wfo_radio_cmd_sender_pass_log(wfo_cfg80211_t *cfg)
{
	if(cfg->pass_log.str[0] == '\0')
		return 0;
  
	WFO_SENDER_TRACE_IN(WFO_CMD(pass_log));

	wfo_radio_cmd_sender((wfo_cfg80211_t *)cfg);

	memset(cfg->pass_log.str, 0, MAX_PROC_LOG_LEN);

	WFO_SENDER_TRACE_OUT(WFO_CMD(pass_log));

	return 0;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_pass_log(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	char *tmp;
	char *token;
	char *print_ptr;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(pass_log));

	if (cfg->pass_log.printk_format) {
		print_ptr = token = cfg->pass_log.str;
		while(*token != '\0') {
			if(*token == '\n') {
				*token = '\0';
				printk("%s\n", print_ptr);
				print_ptr = token + 1;
			}
			token++;
		}
		printk("%s", print_ptr);
	}
	else {
		if(strlen(cfg->pass_log.str) + strlen(seq_file_ptr->buf) >= seq_file_ptr->size) {
			if (unlikely((seq_file_ptr->size + (u32)MAX_PROC_LOG_LEN) > (INT_MAX & PAGE_MASK))) {
				WFO_WARN("alloc seq_buff fail: size %d > (INT_MAX & PAGE_MASK)!!\n", seq_file_ptr->size + MAX_PROC_LOG_LEN);
				WFO_RECEIVER_TRACE_OUT(WFO_CMD(pass_log));
				return ret;
			}

			tmp = kvmalloc(seq_file_ptr->size + (u32)MAX_PROC_LOG_LEN, GFP_KERNEL_ACCOUNT);

			if (tmp == NULL) {
				WFO_WARN("kvmalloc failed!!\n");
				WFO_RECEIVER_TRACE_OUT(WFO_CMD(pass_log));
				return ret;
			}

			memset(tmp, 0, seq_file_ptr->size + (u32)MAX_PROC_LOG_LEN);
			memcpy(tmp, seq_file_ptr->buf, strlen(seq_file_ptr->buf) );
			kvfree(seq_file_ptr->buf);
			seq_file_ptr->buf = tmp;
			seq_file_ptr->size = seq_file_ptr->size + (u32)MAX_PROC_LOG_LEN;
		}

		seq_printf(seq_file_ptr ,"%s", cfg->pass_log.str);
	}

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(pass_log));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */
#endif /* !CPTCFG_WFO_VIRT_SAME_CPU */

#endif /* !CONFIG_RTK_WFO_NO_VIRT */

//----------------------------------------------------------------------------
//cfg80211_vendor_cmd_reply(skb)

#if defined(WFO_RADIO_SENDER)
int wfo_radio_cmd_sender_cfg80211_vendor_cmd_reply(struct sk_buff *skb)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(skb->dev, RUN_LOCALLY, WFO_CMD(cfg80211_vendor_cmd_reply));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_vendor_cmd_reply));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	ret = cfg80211_vendor_cmd_reply(skb);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_vendor_cmd_reply));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_vendor_cmd_reply(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_vendor_cmd_reply));
	//ret = cfg80211_vendor_cmd_reply(skb);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_vendor_cmd_reply));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_ibss_joined(dev, bssid, channel, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_ibss_joined(struct net_device *dev, const u8 *bssid,
		struct ieee80211_channel *channel, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_LOCALLY, WFO_CMD(cfg80211_ibss_joined));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_ibss_joined));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_ibss_joined(dev, bssid, channel, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_ibss_joined));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_ibss_joined(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_ibss_joined));
	//cfg80211_ibss_joined(dev, bssid, channel, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_ibss_joined));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_ch_switch_notify(dev, chandef)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_ch_switch_notify(struct net_device *dev,
		struct cfg80211_chan_def *chandef)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(cfg80211_ch_switch_notify));

	if (cfg == NULL) {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		cfg80211_ch_switch_notify(dev, chandef);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
		return;
	}

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_ch_switch_notify));

	wfo_copy_struct_data(wfo_ipc_sender,
		WFO_STRUCT_TYPE(cfg80211_chan_def),
		&cfg->cfg80211_ch_switch_notify.data.source,
		chandef);

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_ch_switch_notify));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_ch_switch_notify(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct cfg80211_chan_def _chandef, *chandef = &_chandef;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_ch_switch_notify));

	wfo_copy_struct_data(wfo_ipc_receiver,
		WFO_STRUCT_TYPE(cfg80211_chan_def),
		chandef,
		&cfg->cfg80211_ch_switch_notify.data.source);
	chandef->chan =
		ieee80211_get_channel(wiphy, (s32)chandef->chan->center_freq);

	cfg80211_ch_switch_notify(virt_ndev, chandef);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_ch_switch_notify));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_ready_on_channel(wdev, cookie, chan, duration, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_ready_on_channel(struct wireless_dev *wdev, u64 cookie,
		struct ieee80211_channel *chan,
		unsigned int duration, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wdev->netdev, RUN_LOCALLY, WFO_CMD(cfg80211_ready_on_channel));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_ready_on_channel));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_ready_on_channel(wdev, cookie, chan, duration, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_ready_on_channel));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_ready_on_channel(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_ready_on_channel));
	//cfg80211_ready_on_channel(wdev, cookie, chan, duration, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_ready_on_channel));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_vendor_cmd_alloc_reply_skb(wiphy, approxlen)

#if defined(WFO_RADIO_SENDER)
struct sk_buff *wfo_radio_cmd_sender_cfg80211_vendor_cmd_alloc_reply_skb(struct wiphy *wiphy,
		int approxlen)
{
	struct sk_buff *ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wiphy_to_ndev(wiphy), RUN_LOCALLY, WFO_CMD(cfg80211_vendor_cmd_alloc_reply_skb));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_vendor_cmd_alloc_reply_skb));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	ret = cfg80211_vendor_cmd_alloc_reply_skb(wiphy, approxlen);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_vendor_cmd_alloc_reply_skb));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_vendor_cmd_alloc_reply_skb(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_vendor_cmd_alloc_reply_skb));
	//ret = cfg80211_vendor_cmd_alloc_reply_skb(wiphy, approxlen);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_vendor_cmd_alloc_reply_skb));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_roamed(dev, info, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_roamed(struct net_device *dev, struct cfg80211_roam_info *info,
		gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_LOCALLY, WFO_CMD(cfg80211_roamed));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_roamed));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_roamed(dev, info, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_roamed));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_roamed(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_roamed));
	//cfg80211_roamed(dev, info, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_roamed));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_mgmt_tx_status(wdev, cookie, buf, len, ack, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_mgmt_tx_status(struct wireless_dev *wdev, u64 cookie,
		const u8 *buf, size_t len, bool ack, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wdev->netdev, RUN_REMOTELY, WFO_CMD(cfg80211_mgmt_tx_status));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_mgmt_tx_status));

#ifndef __ECOS
	cfg80211_mgmt_tx_status(wdev, cookie, buf, len, ack, gfp);
#else
	cfg->cfg80211_mgmt_tx_status.cookie = cookie;
	cfg->cfg80211_mgmt_tx_status.len = len;
	cfg->cfg80211_mgmt_tx_status.ack = ack;
	cfg->cfg80211_mgmt_tx_status.gfp = gfp;

	if (buf)
		memcpy(&cfg->cfg80211_mgmt_tx_status.buf, buf, len);

	wfo_radio_cmd_sender(cfg);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_mgmt_tx_status));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_mgmt_tx_status(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	struct wireless_dev *wdev = ndev_to_wdev(virt_ndev);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_mgmt_tx_status));

	cfg80211_mgmt_tx_status(wdev,
				cfg->cfg80211_mgmt_tx_status.cookie,
				cfg->cfg80211_mgmt_tx_status.buf,
				cfg->cfg80211_mgmt_tx_status.len,
				cfg->cfg80211_mgmt_tx_status.ack,
				cfg->cfg80211_mgmt_tx_status.gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_mgmt_tx_status));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_remain_on_channel_expired(wdev, cookie, chan, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_remain_on_channel_expired(struct wireless_dev *wdev,
		u64 cookie,
		struct ieee80211_channel *chan,
		gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wdev->netdev, RUN_LOCALLY, WFO_CMD(cfg80211_remain_on_channel_expired));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_remain_on_channel_expired));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_remain_on_channel_expired(wdev, cookie, chan, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_remain_on_channel_expired));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_remain_on_channel_expired(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_remain_on_channel_expired));
	//cfg80211_remain_on_channel_expired(wdev, cookie, chan, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_remain_on_channel_expired));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_unlink_bss(wiphy, pub)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_unlink_bss(struct wiphy *wiphy, struct cfg80211_bss *bss)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wiphy_to_ndev(wiphy), RUN_REMOTELY, WFO_CMD(cfg80211_unlink_bss));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_unlink_bss));

	cfg->cfg80211_unlink_bss.bss = bss;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_unlink_bss));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_unlink_bss(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_unlink_bss));

	cfg80211_unlink_bss(wiphy, (struct cfg80211_bss *)cfg->cfg80211_unlink_bss.bss);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_unlink_bss));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_vendor_event(skb, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_vendor_event(struct sk_buff *skb, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(skb->dev, RUN_LOCALLY, WFO_CMD(cfg80211_vendor_event));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_vendor_event));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_vendor_event(skb, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_vendor_event));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_vendor_event(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_vendor_event));
	//cfg80211_vendor_event(skb, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_vendor_event));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_michael_mic_failure(dev, addr, key_type, key_id, tsc, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_michael_mic_failure(struct net_device *dev, const u8 *addr,
		enum nl80211_key_type key_type, int key_id,
		const u8 *tsc, gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_LOCALLY, WFO_CMD(cfg80211_michael_mic_failure));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_michael_mic_failure));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_michael_mic_failure(dev, addr, key_type, key_id, tsc, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_michael_mic_failure));
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_michael_mic_failure(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_michael_mic_failure));
	//cfg80211_michael_mic_failure(dev, addr, key_type, key_id, tsc, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_michael_mic_failure));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//cfg80211_vendor_event_alloc(wiphy, wdev, approxlen, event_idx, gfp)

#if defined(WFO_RADIO_SENDER)
struct sk_buff *wfo_radio_cmd_sender_cfg80211_vendor_event_alloc(struct wiphy *wiphy,
		struct wireless_dev *wdev, int approxlen, int event_idx, gfp_t gfp)
{
	struct sk_buff *ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(wdev->netdev, RUN_LOCALLY, WFO_CMD(cfg80211_vendor_event_alloc));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_vendor_event_alloc));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	ret = cfg80211_vendor_event_alloc(wiphy, wdev, approxlen, event_idx, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_vendor_cmd_reply));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_vendor_event_alloc(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_vendor_event_alloc));
	//ret = cfg80211_vendor_event_alloc(wiphy, wdev, approxlen, event_idx, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_vendor_event_alloc));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
//cfg80211_connect_done(dev, params, gfp)

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_connect_done(struct net_device *dev,
		struct cfg80211_connect_resp_params *params,
		gfp_t gfp)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_LOCALLY, WFO_CMD(cfg80211_connect_done));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_connect_done));

	WFO_WARN("not support!!\n");
#ifndef __ECOS
	cfg80211_connect_done(dev, params, gfp);
#endif

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_connect_done));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_connect_done(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_connect_done));
	//ret = cfg80211_connect_done(wiphy, wdev, approxlen, event_idx, gfp);
	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_connect_done));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
//check_srcmac_in_fdb_for_ax_driver(dev, addr)

#if defined(WFO_RADIO_SENDER)
int wfo_radio_cmd_sender_check_srcmac_in_fdb_for_ax_driver(struct net_device *dev, const unsigned char *addr)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(check_srcmac_in_fdb_for_ax_driver));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(check_srcmac_in_fdb_for_ax_driver));

	memcpy(cfg->check_srcmac_in_fdb_for_ax_driver.addr, addr, ETH_ALEN);

	wfo_radio_cmd_sender(cfg);
	ret = cfg->ret_val;

	WFO_SENDER_TRACE_OUT(WFO_CMD(check_srcmac_in_fdb_for_ax_driver));
	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_check_srcmac_in_fdb_for_ax_driver(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg)
{
	int ret = 0;
	extern int check_srcmac_in_fdb_for_ax_driver(struct net_device *dev, const unsigned char *addr);
	WFO_RECEIVER_TRACE_IN(WFO_CMD(check_srcmac_in_fdb_for_ax_driver));

	ret = check_srcmac_in_fdb_for_ax_driver(virt_ndev, cfg->check_srcmac_in_fdb_for_ax_driver.addr);
	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(check_srcmac_in_fdb_for_ax_driver));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */

//----------------------------------------------------------------------------
// for regulatory_hint

#if defined(WFO_RADIO_SENDER)
int wfo_radio_cmd_sender_regulatory_hint(struct wiphy *wiphy,
			const char *alpha2)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = NULL;
	_adapter *adapter  = NULL;

	// we set default value "00" in virtual drvier, see wfo_virt_regd_init() in wfo_virt.c
	if (!strcmp("00", alpha2))
		return 0;

	if (wiphy)
		adapter = wiphy_to_adapter(wiphy);

	if (adapter && adapter->rtw_wdev && adapter->rtw_wdev->netdev)
		cfg = get_ra_cfg_addr(adapter->rtw_wdev->netdev,
					RUN_REMOTELY, WFO_CMD(regulatory_hint));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(regulatory_hint));

	memcpy(&cfg->regulatory_hint.alpha2, (void *)alpha2, 3);

	wfo_radio_cmd_sender(cfg);
	ret = cfg->ret_val;	// Don't care the return value

	WFO_SENDER_TRACE_OUT(WFO_CMD(regulatory_hint));

	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_regulatory_hint(struct wiphy *wiphy,
                struct net_device *radio_ndev,
                wfo_cfg80211_t *cfg)
{
	extern int regulatory_hint(struct wiphy *wiphy, const char *alpha2);

	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(regulatory_hint));

	ret = regulatory_hint(wiphy, (char *)&cfg->regulatory_hint.alpha2);

	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(regulatory_hint));

	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


#ifdef CONFIG_RTW_80211R
//----------------------------------------------------------------------------
// for cfg80211_ft_event

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_cfg80211_ft_event(struct net_device *netdev,
	       struct cfg80211_ft_event_params *ft_event)
{
	wfo_cfg80211_t *cfg =
		get_ra_cfg_addr(netdev, RUN_REMOTELY, WFO_CMD(cfg80211_ft_event));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(cfg80211_ft_event));

	if (ft_event) {
		struct cfg80211_ft_event_params *ft_event2 =
				&cfg->ft_event.source;
		memcpy(ft_event2, ft_event, sizeof(struct cfg80211_ft_event_params));

		ft_event2->ies = &cfg->ft_event.ies;
		ft_event2->target_ap = &cfg->ft_event.target_ap;
		ft_event2->ric_ies = &cfg->ft_event.ric_ies;

		if (ft_event->ies)
			memcpy(ft_event2->ies, ft_event->ies, ft_event->ies_len);
		if (ft_event->target_ap)
			memcpy(ft_event2->target_ap, ft_event->target_ap, ETH_ALEN);
		if (ft_event->ric_ies)
			memcpy(ft_event2->ric_ies, ft_event->ric_ies, ft_event->ric_ies_len);
	}

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(cfg80211_ft_event));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_cfg80211_ft_event(struct wiphy *wiphy,
                struct net_device *radio_ndev,
                wfo_cfg80211_t *cfg)
{
	int ret = 0;
	struct cfg80211_ft_event_params *ft_event = &cfg->ft_event.source;
	extern void cfg80211_ft_event(struct net_device *netdev,
		       struct cfg80211_ft_event_params *ft_event);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(cfg80211_ft_event));

	ft_event->ies = &cfg->ft_event.ies;
	ft_event->target_ap = &cfg->ft_event.target_ap;
	ft_event->ric_ies = &cfg->ft_event.ric_ies;

	cfg80211_ft_event(radio_ndev, ft_event);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(cfg80211_ft_event));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */
#endif /* CONFIG_RTW_80211R */


//----------------------------------------------------------------------------
// for wireless_send_event

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_wireless_send_event(struct net_device *netdev,
			 unsigned int		cmd,
			 union iwreq_data *	wrqu,
			 const char *		extra)
{
	wfo_cfg80211_t *cfg =
		get_ra_cfg_addr(netdev, RUN_REMOTELY, WFO_CMD(wireless_send_event));

	if (cfg == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(wireless_send_event));

	cfg->wireless_send_event.cmd = cmd;
	if (wrqu) {
		if (wrqu->data.length > MAX_EXTRA_LEN) {
			printk("%s(): set length from %d to %d\n",
				__func__, wrqu->data.length, MAX_EXTRA_LEN);
			wrqu->data.length = MAX_EXTRA_LEN;
		}

		cfg->wireless_send_event.mflags |= (1<<WFO_WIRELESS_SEND_EVENT_HAS_WRQU);
		memcpy(&cfg->wireless_send_event.wrqu, wrqu, sizeof(union iwreq_data));

		if (extra) {
			cfg->wireless_send_event.mflags |= (1<<WFO_WIRELESS_SEND_EVENT_HAS_EXTRA);

			memcpy(&cfg->wireless_send_event.extra, extra, wrqu->data.length);
		}
	}

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(wireless_send_event));
	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_wireless_send_event(struct wiphy *wiphy,
                struct net_device *radio_ndev,
                wfo_cfg80211_t *cfg)
{
	int ret = 0;
	union iwreq_data *wrqu = NULL;
	char *extra = NULL;

	extern void wireless_send_event(struct net_device *	dev,
			 unsigned int		cmd,
			 union iwreq_data *	wrqu,
			 const char *		extra);

	WFO_RECEIVER_TRACE_IN(WFO_CMD(wireless_send_event));

	if (cfg->wireless_send_event.mflags & (1<<WFO_WIRELESS_SEND_EVENT_HAS_WRQU))
		wrqu = &cfg->wireless_send_event.wrqu;

	if (cfg->wireless_send_event.mflags & (1<<WFO_WIRELESS_SEND_EVENT_HAS_EXTRA))
		extra = (char *)&cfg->wireless_send_event.extra;

	wireless_send_event(radio_ndev,
		cfg->wireless_send_event.cmd, wrqu, extra);

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(wireless_send_event));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
// for mac_to_fc

#if defined(WFO_RADIO_SENDER)
void wfo_radio_cmd_sender_mac_to_fc(struct net_device *dev, u8 *mac_addr, u8 flag)
{
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(mac_to_fc));

	if (cfg == NULL || dev == NULL || mac_addr == NULL)
		return;

	WFO_SENDER_TRACE_IN(WFO_CMD(mac_to_fc));

	if (mac_addr) {
		cfg->mac_to_fc.mflags |= (1<<WFO_MAC_TO_FC_HAS_MAC);
		memcpy(&cfg->mac_to_fc.mac_addr, mac_addr, ETH_ALEN);
	}
	cfg->mac_to_fc.flag = flag;

	wfo_radio_cmd_sender(cfg);

	WFO_SENDER_TRACE_OUT(WFO_CMD(mac_to_fc));
}
#endif /* WFO_RADIO_SENDER */

#if defined(CONFIG_FC_WIFI_TX_GMAC_TRUNKING_SUPPORT) || defined(CONFIG_FC_WIFI_TRAP_HASH_SUPPORT)
extern int rtk_fc_external_lut_process(bool add, char *wlan_devname, char *sta_mac);
#endif
#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_mac_to_fc(struct wiphy *wiphy,
                struct net_device *radio_ndev,
                wfo_cfg80211_t *cfg)
{
	int ret = 0;
	u8 *mac_addr = NULL;
	u8 *dev_name = NULL;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(mac_to_fc));

	if (radio_ndev)
		dev_name = radio_ndev->name;
	if (cfg->mac_to_fc.mflags & (1<<WFO_MAC_TO_FC_HAS_MAC))
		mac_addr = cfg->mac_to_fc.mac_addr;

	//todo, pass dev_name/mac_addr to FC
#if defined(CONFIG_FC_WIFI_TX_GMAC_TRUNKING_SUPPORT) || defined(CONFIG_FC_WIFI_TRAP_HASH_SUPPORT)
	rtk_fc_external_lut_process(cfg->mac_to_fc.flag, dev_name, mac_addr);
#endif

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(mac_to_fc));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//----------------------------------------------------------------------------
// for new_radio_cmd_xxx

#if defined(WFO_RADIO_SENDER)
int wfo_radio_cmd_sender_new_radio_cmd_xxx(struct net_device *dev)
{
	int ret = 0;
	wfo_cfg80211_t *cfg = get_ra_cfg_addr(dev, RUN_REMOTELY, WFO_CMD(new_radio_cmd_xxx));

	if (cfg == NULL)
		return ret;

	WFO_SENDER_TRACE_IN(WFO_CMD(new_radio_cmd_xxx));

	//todo, copy parameter to cfg

	wfo_radio_cmd_sender(cfg);
	ret = cfg->ret_val;

	//todo, copy parameter in cfg for caller (if need)

	WFO_SENDER_TRACE_OUT(WFO_CMD(new_radio_cmd_xxx));

	return ret;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)
static int wfo_radio_cmd_receiver_new_radio_cmd_xxx(struct wiphy *wiphy,
                struct net_device *radio_ndev,
                wfo_cfg80211_t *cfg)
{
	int ret = 0;

	WFO_RECEIVER_TRACE_IN(WFO_CMD(new_radio_cmd_xxx));

	//todo, copy parameter for cfg, and call related function
	//ret = xxxx(....);

	cfg->ret_val = ret;

	WFO_RECEIVER_TRACE_OUT(WFO_CMD(new_radio_cmd_xxx));
	return ret;
}
#endif /* WFO_RADIO_RECEIVER */


//==============================================================================
//	radio IPC
//==============================================================================

#define WFO_RADIO_CMD_NUM (WFO_CMD(max)-WFO_CMD(cfg80211_new_sta))

#if defined(WFO_RADIO_RECEIVER)
#define WFO_RADIO_CMD(type) \
	wfo_cmd_##type, wfo_radio_cmd_receiver_##type
#else
#define WFO_RADIO_CMD(type) \
	wfo_cmd_##type, NULL
#endif

#if defined(WFO_RADIO_SENDER)
static int wfo_radio_cmd_ipc_type[WFO_RADIO_CMD_NUM] = {0};
#endif /* WFO_RADIO_SENDER */

static struct wfo_cmd_recv_array_s ra_recv_cmd[WFO_RADIO_CMD_NUM] = {
//----- nl80211: radio -> virt -------------------------------------------------
	{ WFO_RADIO_CMD(cfg80211_new_sta), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_del_sta), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_scan_done), ASYNC },
#if defined(WFO_ASYNC_INFORM_BSS)
	{ WFO_RADIO_CMD(cfg80211_inform_bss_frame), ASYNC },
#else /* !WFO_ASYNC_INFORM_BSS */
	{ WFO_RADIO_CMD(cfg80211_inform_bss_frame), SYNC },
#endif /* WFO_ASYNC_INFORM_BSS */
	{ WFO_RADIO_CMD(cfg80211_put_bss), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_ch_switch_started_notify), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_get_bss), SYNC },
	{ WFO_RADIO_CMD(cfg80211_connect_result), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_vendor_cmd_reply), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_ibss_joined), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_ch_switch_notify), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_ready_on_channel), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_vendor_cmd_alloc_reply_skb), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_roamed), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_mgmt_tx_status), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_rx_mgmt), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_rx_unprot_mlme_mgmt), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_remain_on_channel_expired), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_unlink_bss), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_disconnected), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_vendor_event), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_michael_mic_failure), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_vendor_event_alloc), ASYNC },
	{ WFO_RADIO_CMD(cfg80211_external_auth_request), SYNC },
	{ WFO_RADIO_CMD(cfg80211_connect_done), ASYNC },
	#if !defined(CPTCFG_WFO_VIRT_SAME_CPU)
	{ WFO_RADIO_CMD(open_file), SYNC },
	{ WFO_RADIO_CMD(is_file_readable), SYNC },
	{ WFO_RADIO_CMD(pass_log), ASYNC },
	#endif /* !CPTCFG_WFO_VIRT_SAME_CPU */
	{ WFO_RADIO_CMD(check_srcmac_in_fdb_for_ax_driver), SYNC},
#if defined(CONFIG_RTK_WLAN_EVENT_INDICATE)
	{ WFO_RADIO_CMD(rtk_eventd_netlink_send), ASYNC},
#endif
	{ WFO_RADIO_CMD(netlink_send_msg), ASYNC },
	{ WFO_RADIO_CMD(core_map_nl_event_send_fragment), ASYNC },
	{ WFO_RADIO_CMD(regulatory_hint), ASYNC },
#ifdef CONFIG_RTW_80211R
	{ WFO_RADIO_CMD(cfg80211_ft_event), ASYNC },
#endif /* CONFIG_RTW_80211R */
	{ WFO_RADIO_CMD(wireless_send_event), ASYNC },
	{ WFO_RADIO_CMD(mac_to_fc), ASYNC },

	{ WFO_RADIO_CMD(new_radio_cmd_xxx), SYNC },

//----- nl80211: radio -> virt -------------------------------------------------
};

#if defined(WFO_RADIO_SENDER)
void init_wfo_radio_cmd_sender_sync_type(void)
{
	int i = 0, j = 0;

	for (i=0;i<WFO_RADIO_CMD_NUM;i++) {
		for (j=0;j<WFO_RADIO_CMD_NUM;j++) {
			if ((i+WFO_CMD(cfg80211_new_sta)) == ra_recv_cmd[j].cmd) {
				wfo_radio_cmd_ipc_type[i] = ra_recv_cmd[j].sync;
			}
		}
	}
}

void wfo_radio_cmd_sender(wfo_cfg80211_t *cfg)
{
	int ret = 0;
	cfg->ret_val = -1;

	if (cfg->side == RUN_REMOTELY) {
		if (cfg->cmd == WFO_CMD(pass_log) && cfg->pass_log.ipc_type == 1)
			ret = wfo_radio_cmd_ipc_sender((void *)cfg, sizeof(wfo_cfg80211_t), SYNC);
		else
			ret = wfo_radio_cmd_ipc_sender((void *)cfg, sizeof(wfo_cfg80211_t),
				wfo_radio_cmd_ipc_type[cfg->cmd - WFO_CMD(cfg80211_new_sta)]);
	} else {
		WFO_PRINT("run on local?!!\n");
	}

	if (ret) {
		show_ipc_fail_cmd(cfg, "radio");
	}

	return;
}
#endif /* WFO_RADIO_SENDER */

#if defined(WFO_RADIO_RECEIVER)

static int (*wfo_radio_cmd_receiver_fp[WFO_RADIO_CMD_NUM])(struct wiphy *wiphy,
		struct net_device *virt_ndev,
		wfo_cfg80211_t *cfg) = { NULL };

void init_wfo_radio_cmd_receiver_fp(void)
{
	unsigned int i = 0U, j = 0U;

	for (i=0U;i<WFO_RADIO_CMD_NUM;i++) {
		for (j=0U;j<WFO_RADIO_CMD_NUM;j++) {
			if ((i+WFO_CMD(cfg80211_new_sta)) == ra_recv_cmd[j].cmd) {
				wfo_radio_cmd_receiver_fp[i] = ra_recv_cmd[j].cmd_fp;
			}
		}
	}

	register_wfo_ipc_show_cmd_name();
}

void wfo_radio_cmd_receiver(wfo_cfg80211_t *cfg)
{
	struct net_device *virt_ndev = NULL;
	struct wireless_dev *virt_wdev = NULL;
	struct wiphy *virt_wiphy = NULL;
	unsigned int fp_idx = 0U;
#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
	char dev_name[IFNAMSIZ];
#endif /* CPTCFG_WFO_VIRT_SAME_CPU */

	if (virt_map_tbl[cfg->dev_idx].ndev) {
		virt_ndev = virt_map_tbl[cfg->dev_idx].ndev;
		virt_wiphy = virt_map_tbl[cfg->dev_idx].wiphy;
	} else {
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		snprintf(dev_name, 1+strlen(cfg->dev_name)-strlen(WFO_IFACE_POSTFIX),
			"%s", cfg->dev_name);
		virt_ndev = wfo_dev_get_by_name(&init_net, dev_name);
	#else
		virt_ndev = wfo_dev_get_by_name(&init_net, cfg->dev_name);
	#endif /* CPTCFG_WFO_VIRT_SAME_CPU */

		if (virt_ndev) {
			virt_wdev = ndev_to_wdev(virt_ndev);
			if (virt_wdev)
				virt_wiphy = wdev_to_wiphy(virt_wdev);

			if (virt_wiphy) {
				virt_map_tbl[cfg->dev_idx].ndev  = virt_ndev;
				virt_map_tbl[cfg->dev_idx].wiphy = virt_wiphy;
			}
		}
	}

#if (WFO_CFG_DBG_TRACE_OPS==1)
	if (virt_ndev && (wfo_trace[TRACE_RECEIVER][cfg->cmd])) {
		WFO_PRINT("on %s, cmd[%d:%s], %s -> %s\n",
			cmd_run_mode[cfg->side],
			cfg->cmd, wfo_cmd_str[cfg->cmd],
			cfg->dev_name, virt_ndev->name);
	}
#endif /* WFO_CFG_DBG_TRACE_OPS */

	if (!virt_ndev) {
#if (WFO_CFG_DBG_TRACE_OPS==1)
	#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
		WFO_PRINT("can't get dev[%s]!! cmd[%d][%s]\n",
			dev_name, cfg->cmd, wfo_cmd_str[cfg->cmd]);
	#else
		WFO_PRINT("can't get dev[%s]!! cmd[%d:%s]\n",
			cfg->dev_name, cfg->cmd, wfo_cmd_str[cfg->cmd]);
	#endif
#endif /* WFO_CFG_DBG_TRACE_OPS */

		cfg->ret_val = 0;
		return;
	}

	fp_idx = cfg->cmd-WFO_CMD(cfg80211_new_sta);
	if (wfo_radio_cmd_receiver_fp[fp_idx]) {
		(*wfo_radio_cmd_receiver_fp[fp_idx])(virt_wiphy, virt_ndev, cfg);
	} else if (!virt_ndev) {
		WFO_PRINT("cmd[%d][%s] function pointer can't run when dev null! virt_wiphy=%p, virt_ndev=%p\n",
			cfg->cmd, wfo_cmd_str[cfg->cmd], virt_wiphy, virt_ndev);
	} else {
		WFO_PRINT("cmd[%d][%s] function pointer is not assigned! virt_wiphy=%p, virt_ndev=%p\n",
			cfg->cmd, wfo_cmd_str[cfg->cmd], virt_wiphy, virt_ndev);
	}

	return;
}
#endif /* WFO_RADIO_RECEIVER */


#if defined(WFO_RADIO_SENDER)
#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
#define MASK_U32(VAL) ((VAL) & 0xffffffff)

int is_ipc_meme(void *_addr)
{
#ifdef USE_RTK_TAROKO_IPC
	u32 beg = MASK_U32((u32)RTK_SHM_DRAM_BASE);
	u32 end = MASK_U32((u32)(RTK_SHM_DRAM_BASE+CONFIG_RTL8686_IPC_MEM_SIZE));
	u32 addr = MASK_U32((u32)(_addr));

	WFO_DBGP_IOCTL("[0x%x-0x%x]=[%d], addr[0x%x]\n", beg, end, end-beg, addr);

	if (addr>=beg && addr<=end)
		return 1;
	else
		return 0;
#else /* !USE_RTK_TAROKO_IPC */
	return 1;
#endif /* USE_RTK_TAROKO_IPC */
}

long _wfo_copy_from_user(void *to,
		const void __user * from, unsigned long n)
{
	if (is_ipc_meme((void *)from)) {
		WFO_DBGP_IOCTL("memcpy: to[0x%x], from[0x%x], size[%d]\n", to, from, n);
		memcpy(to, from, n);
		return 0;
	} else {
		unsigned long res = n;
		WFO_DBGP_IOCTL("copy_from_user: to[0x%x], from[0x%x], size[%d]\n", to, from, n);
		might_fault();
		if (likely(access_ok(VERIFY_READ, from, n)))
			res = __copy_from_user(to, from, n);
		if (unlikely(res))
			memset((void *)(to + (n - res)), 0, res);
		return res;
	}
}

long _wfo_copy_to_user(void __user *to,
		const void *from, unsigned long n)
{
	if (is_ipc_meme(to)) {
		WFO_DBGP_IOCTL("memcpy: to[0x%x], from[0x%x], size[%d]\n", to, from, n);
		memcpy(to, from, n);
		return 0;
	} else {
		WFO_DBGP_IOCTL("copy_to_user: to[0x%x], from[0x%x], size[%d]\n", to, from, n);
		might_fault();
		if (access_ok(VERIFY_WRITE, to, n))
			return __copy_to_user(to, from, n);
		else
			return n;
	}
}
#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
#endif /* WFO_RADIO_SENDER */

