#include "./wfo_virt.h"
#include "wfo/wfo_cmd_virt.h"

struct wiphy *g_wiphy[WLAN_ROOT_IFACE_NUM] = { 0 };

static int virt_iface[WLAN_ROOT_IFACE_NUM] = { 0 };
static int n_virt_iface = WLAN_ROOT_IFACE_NUM;
module_param_array(virt_iface , int , &n_virt_iface , S_IRUGO);

#include "wfo/wfo_cmd_virt.c"
#include "wfo/wfo_proc.c"
#if defined(USE_WFO_NIC)
#include "trx/wfo_pseudo_trx.c"
#endif /* USE_WFO_NIC */

#define VIRT_DEVNAME WLAN_IFACE_NAME"%d"

//=============================================================================
//  for rtw_get_wireless_stats
//=============================================================================
struct iw_statistics *wfo_get_wireless_stats(struct net_device *virt_ndev)
{
	struct wfo_ndev_priv *ndev_priv = (struct wfo_ndev_priv *)netdev_priv(virt_ndev);
	wfo_virt_cmd_sender_get_wireless_stats(virt_ndev);
	return &ndev_priv->iwstats;
}

//==============================================================================
//  netdev_ops
//==============================================================================
#if defined(WFO_CMD_VIRT_NDEV_OPS)
static int wfo_virt_set_address(struct net_device *ndev, void *p)
{
	struct sockaddr *sa = p;

	if (!is_valid_ether_addr(sa->sa_data)) {
		return -EADDRNOTAVAIL;
	}

	memcpy(ndev->dev_addr, sa->sa_data, ETH_ALEN);
	wfo_virt_cmd_sender_ndo_set_mac_address(ndev, sa);

	return 0;
}

static int wfo_virt_ioctl(struct net_device *ndev, struct ifreq *ifr, int cmd)
{
	int ret = 0;

	struct iwreq *wrq = (struct iwreq *)ifr;
	unsigned short subcmd = 0U;
	unsigned char dir = 0U;

	if (cmd == SIOCDEVPRIVATEAXEXT) {
		extern int __rtw_ioctl_private(struct net_device *dev, struct iwreq *wrq, int exec);

		subcmd = wrq->u.data.flags;
		dir = (u8)__rtw_ioctl_private(ndev, wrq, 0U);
		if (dir) {
			ret = wfo_virt_cmd_sender_ndo_do_ioctl(ndev, ifr, cmd, dir, subcmd);
		} else {
			ret = -EIO;
		}
	} else if (cmd == SIOCDEVPRIVATE) {
		ret = wfo_virt_cmd_sender_ndo_do_ioctl(ndev, ifr, cmd, 0U, 0U);
	} else {
		WFO_PRINT("cmd[0x%x] isn't supported(ret=%d)! (%s support 0x%x & 0x%x only)\n",
			cmd, ret, ndev->name, SIOCDEVPRIVATE, SIOCDEVPRIVATEAXEXT);
	}

	return ret;
}

struct net_device_stats *wfo_virt_get_stats(struct net_device *ndev)
{
#if defined(RTK_SHM_DRAM)
	//int idx = 0;
	wfo_dram_t *dram = RTK_SHM_DRAM;
	struct wfo_ndev_priv *ndev_priv = (struct wfo_ndev_priv *)netdev_priv(ndev);

	return &dram->md.dev_st[ndev_priv->iface_id];
#else /* !RTK_SHM_DRAM */
	return wfo_virt_cmd_sender_ndo_get_stats(ndev);
#endif /* RTK_SHM_DRAM */
}

static int wfo_virt_open(struct net_device *ndev)
{
	return wfo_virt_cmd_sender_ndo_open(ndev);
}

static int wfo_virt_close(struct net_device *ndev)
{
	return wfo_virt_cmd_sender_ndo_stop(ndev);
}

#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
static int wfo_virt_xmit(struct sk_buff *skb, struct net_device *dev)
{
	int ret = 0;
	struct wfo_ndev_priv *ndev_priv =
		(struct wfo_ndev_priv *)netdev_priv(dev);
	struct net_device *radio_ndev = ndev_priv->radio_ndev;

	if (ndev_priv->radio_ndev == NULL && ndev_priv) {
		radio_ndev = ndev_priv->radio_ndev = wfo_dev_get_by_name(&init_net, ndev_priv->radio_name);
	}

	if (radio_ndev && netif_carrier_ok(radio_ndev)) {
		dev->stats.tx_packets++;
		dev->stats.tx_bytes += skb->len;

		ret = radio_ndev->netdev_ops->ndo_start_xmit(skb, radio_ndev);
	}
	else {
		dev_kfree_skb(skb);
	}

#if (WFO_DBG_TX==1)
	WFO_DBGP("dev[%s] ndev_priv->radio_name[%s], radio_ndev=%p/[%s]\n",
		dev->name,
		ndev_priv->radio_name,
		radio_ndev,
		radio_ndev?radio_ndev->name:"null");

	if (net_ratelimit()) {
		static int count = 0;
		if (count == 0) {
			if (radio_ndev && netif_carrier_ok(radio_ndev))
				WFO_DBGP("to radio_ndev=%s\n",
					radio_ndev?radio_ndev->name:"null");
			else
				WFO_DBGP("drop radio_ndev=%s\n",
					radio_ndev?radio_ndev->name:"null");
		}
		if (count++ > 30)
			count = 0;
	}
#endif /* WFO_DBG_TX */

	return ret;
}
#endif /* CPTCFG_WFO_VIRT_SAME_CPU */

static const struct net_device_ops wfo_virt_netdev_ops = {
#if defined(CPTCFG_WFO_VIRT_SAME_CPU)
	.ndo_start_xmit		= wfo_virt_xmit,
#else /* !CPTCFG_WFO_VIRT_SAME_CPU */
	.ndo_start_xmit		= rtk_wfo_pseudo_wlan_tx,
#endif /* CPTCFG_WFO_VIRT_SAME_CPU */
	.ndo_validate_addr	= eth_validate_addr,
	.ndo_set_mac_address	= wfo_virt_set_address,

	.ndo_open			= wfo_virt_open,
	.ndo_stop			= wfo_virt_close,
	.ndo_get_stats		= wfo_virt_get_stats,
	.ndo_do_ioctl		= wfo_virt_ioctl,
};
#endif /* WFO_CMD_VIRT_NDEV_OPS */


//==============================================================================
//  wiphy parameter
//==============================================================================
#if defined(WFO_CMD_VIRT_CFG_OPS)
#define RTW_SCAN_IE_LEN_MAX 	 2304
#define RTW_MAX_REMAIN_ON_CHANNEL_DURATION 5000 /* ms */
#define RTW_MAX_NUM_PMKIDS 4

enum band_type {
		BAND_ON_24G 	= 0,
		BAND_ON_5G		= 1,
		BAND_ON_6G		= 2,
		BAND_MAX,
};

#define CENTER_CH_2G_40M_NUM	9
#define CENTER_CH_2G_NUM		14
#define CENTER_CH_5G_20M_NUM	28	/* 20M center channels */
#define CENTER_CH_5G_40M_NUM	14	/* 40M center channels */
#define CENTER_CH_5G_80M_NUM	7	/* 80M center channels */
#define CENTER_CH_5G_160M_NUM	3	/* 160M center channels */
#define CENTER_CH_5G_ALL_NUM	(CENTER_CH_5G_20M_NUM + CENTER_CH_5G_40M_NUM + CENTER_CH_5G_80M_NUM)

#define	MAX_CHANNEL_NUM_2G	CENTER_CH_2G_NUM
#define	MAX_CHANNEL_NUM_5G	CENTER_CH_5G_20M_NUM
#define	MAX_CHANNEL_NUM		(MAX_CHANNEL_NUM_2G + MAX_CHANNEL_NUM_5G)

#define rtw_band_to_nl80211_band(band) \
	((u32)band == (u32)BAND_ON_24G) ? (u32)NL80211_BAND_2GHZ : \
	((u32)band == (u32)BAND_ON_5G) ? (u32)NL80211_BAND_5GHZ : (u32)NUM_NL80211_BANDS

#define WIFI_CIPHER_SUITE_GCMP		0x000FAC08
#define WIFI_CIPHER_SUITE_GCMP_256	0x000FAC09
#define WIFI_CIPHER_SUITE_CCMP_256	0x000FAC0A
#define WIFI_CIPHER_SUITE_BIP_GMAC_128	0x000FAC0B
#define WIFI_CIPHER_SUITE_BIP_GMAC_256	0x000FAC0C
#define WIFI_CIPHER_SUITE_BIP_CMAC_256	0x000FAC0D

static const u32 rtw_cipher_suites[] = {
	WLAN_CIPHER_SUITE_WEP40,
	WLAN_CIPHER_SUITE_WEP104,
	WLAN_CIPHER_SUITE_TKIP,
	WLAN_CIPHER_SUITE_CCMP,
#ifdef CONFIG_WAPI_SUPPORT
	WLAN_CIPHER_SUITE_SMS4,
#endif /* CONFIG_WAPI_SUPPORT */
#ifdef CONFIG_IEEE80211W
	WLAN_CIPHER_SUITE_AES_CMAC,
	WIFI_CIPHER_SUITE_GCMP,
	WIFI_CIPHER_SUITE_GCMP_256,
	WIFI_CIPHER_SUITE_CCMP_256,
	WIFI_CIPHER_SUITE_BIP_GMAC_128,
	WIFI_CIPHER_SUITE_BIP_GMAC_256,
	WIFI_CIPHER_SUITE_BIP_CMAC_256,
#endif /* CONFIG_IEEE80211W */
};

#define RATETAB_ENT(_rate, _rateid, _flags) \
	{								\
		.bitrate	= (_rate),				\
		.hw_value	= (_rateid),				\
		.flags		= (_flags),				\
	}

#define CHAN2G(_channel, _freq, _flags) {			\
		.band			= NL80211_BAND_2GHZ,		\
		.center_freq		= (_freq),			\
		.hw_value		= (_channel),			\
		.flags			= (_flags),			\
		.max_antenna_gain	= 0,				\
		.max_power		= 30,				\
	}

#define CHAN5G(_channel, _flags) {				\
		.band			= NL80211_BAND_5GHZ,		\
		.center_freq		= 5000 + (5 * (_channel)),	\
		.hw_value		= (_channel),			\
		.flags			= (_flags),			\
		.max_antenna_gain	= 0,				\
		.max_power		= 30,				\
	}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0))
/* if wowlan is not supported, kernel generate a disconnect at each suspend
 * cf: /net/wireless/sysfs.c, so register a stub wowlan.
 * Moreover wowlan has to be enabled via a the nl80211_set_wowlan callback.
 * (from user space, e.g. iw phy0 wowlan enable)
 */
static const struct wiphy_wowlan_support wowlan_stub = {
	.flags = WIPHY_WOWLAN_ANY,
	.n_patterns = 0,
	.pattern_max_len = 0,
	.pattern_min_len = 0,
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 10, 0))
	.max_pkt_offset = 0,
#endif
};
#endif

static const struct ieee80211_rate rtw_rates[] = {
	RATETAB_ENT(10,  0x1,	0),
	RATETAB_ENT(20,  0x2,	0),
	RATETAB_ENT(55,  0x4,	0),
	RATETAB_ENT(110, 0x8,	0),
	RATETAB_ENT(60,  0x10,	0),
	RATETAB_ENT(90,  0x20,	0),
	RATETAB_ENT(120, 0x40,	0),
	RATETAB_ENT(180, 0x80,	0),
	RATETAB_ENT(240, 0x100, 0),
	RATETAB_ENT(360, 0x200, 0),
	RATETAB_ENT(480, 0x400, 0),
	RATETAB_ENT(540, 0x800, 0),
};

#define rtw_a_rates		(rtw_rates + 4)
#define RTW_A_RATES_NUM	8
#define rtw_g_rates		(rtw_rates + 0)
#define RTW_G_RATES_NUM	12

/* from center_ch_2g */
static const struct ieee80211_channel rtw_2ghz_channels[MAX_CHANNEL_NUM_2G] = {
	CHAN2G(1, 2412, 0),
	CHAN2G(2, 2417, 0),
	CHAN2G(3, 2422, 0),
	CHAN2G(4, 2427, 0),
	CHAN2G(5, 2432, 0),
	CHAN2G(6, 2437, 0),
	CHAN2G(7, 2442, 0),
	CHAN2G(8, 2447, 0),
	CHAN2G(9, 2452, 0),
	CHAN2G(10, 2457, 0),
	CHAN2G(11, 2462, 0),
	CHAN2G(12, 2467, 0),
	CHAN2G(13, 2472, 0),
	CHAN2G(14, 2484, 0),
};

/* from center_ch_5g_20m */
static const struct ieee80211_channel rtw_5ghz_a_channels[MAX_CHANNEL_NUM_5G] = {
	CHAN5G(36, 0),	CHAN5G(40, 0),	CHAN5G(44, 0),	CHAN5G(48, 0),

	CHAN5G(52, 0),	CHAN5G(56, 0),	CHAN5G(60, 0),	CHAN5G(64, 0),

	CHAN5G(100, 0),	CHAN5G(104, 0),	CHAN5G(108, 0),	CHAN5G(112, 0),
	CHAN5G(116, 0),	CHAN5G(120, 0),	CHAN5G(124, 0),	CHAN5G(128, 0),
	CHAN5G(132, 0),	CHAN5G(136, 0),	CHAN5G(140, 0),	CHAN5G(144, 0),

	CHAN5G(149, 0),	CHAN5G(153, 0),	CHAN5G(157, 0),	CHAN5G(161, 0),
	CHAN5G(165, 0),	CHAN5G(169, 0),	CHAN5G(173, 0),	CHAN5G(177, 0),
};

void rtw_2g_channels_init(struct ieee80211_channel *channels)
{
	_rtw_memcpy((void *)channels, (void *)rtw_2ghz_channels, sizeof(rtw_2ghz_channels));
}

void rtw_5g_channels_init(struct ieee80211_channel *channels)
{
	_rtw_memcpy((void *)channels, (void *)rtw_5ghz_a_channels, sizeof(rtw_5ghz_a_channels));
}

void rtw_2g_rates_init(struct ieee80211_rate *rates)
{
	_rtw_memcpy(rates, rtw_g_rates,
		sizeof(struct ieee80211_rate) * RTW_G_RATES_NUM
	);
}

void rtw_5g_rates_init(struct ieee80211_rate *rates)
{
	_rtw_memcpy(rates, rtw_a_rates,
		sizeof(struct ieee80211_rate) * RTW_A_RATES_NUM
	);
}

#if 1/* for coverity, copy from linux-5.10.x/include/linux/preempt.h */
#define _in_interrupt()	(_irq_count())
#define _irq_count()	((unsigned long)preempt_count() & (HARDIRQ_MASK | SOFTIRQ_MASK \
				 | NMI_MASK))
#endif

struct ieee80211_supported_band *rtw_spt_band_alloc(enum band_type band)
{
	struct ieee80211_supported_band *spt_band = NULL;
	u32 n_channels, n_bitrates;

	if ((u32)band == (u32)BAND_ON_24G) {
		n_channels = (u32)MAX_CHANNEL_NUM_2G;
		n_bitrates = (u32)RTW_G_RATES_NUM;
	} else if ((u32)band == (u32)BAND_ON_5G) {
		n_channels = (u32)MAX_CHANNEL_NUM_5G;
		n_bitrates = (u32)RTW_A_RATES_NUM;
	} else
		goto exit;

	spt_band = (struct ieee80211_supported_band *)rtw_zmalloc(
		sizeof(struct ieee80211_supported_band)
		+ sizeof(struct ieee80211_channel) * n_channels
		+ sizeof(struct ieee80211_rate) * n_bitrates
#if defined(CONFIG_80211AX_HE) && (defined(CPTCFG_VERSION) || (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)))
		+ sizeof(struct ieee80211_sband_iftype_data) * 2
#endif /* defined(CONFIG_80211AX_HE) && LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0) */
		, _in_interrupt() ? (u32)GFP_ATOMIC : (u32)GFP_KERNEL);

	if (!spt_band)
		goto exit;

	spt_band->channels = (struct ieee80211_channel *)(((u8 *)spt_band) + sizeof(struct ieee80211_supported_band));
	spt_band->bitrates = (struct ieee80211_rate *)(((u8 *)spt_band->channels) + sizeof(struct ieee80211_channel) * n_channels);
	spt_band->band = rtw_band_to_nl80211_band(band);
	spt_band->n_channels = (int)n_channels;
	spt_band->n_bitrates = (int)n_bitrates;
#if defined(CONFIG_80211AX_HE) && (defined(CPTCFG_VERSION) || (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0)))
	spt_band->iftype_data = (struct ieee80211_sband_iftype_data *)(((u8 *)spt_band->bitrates)
							+ sizeof(struct ieee80211_rate) * n_bitrates);
	spt_band->n_iftype_data = 0x0000U;
#endif /* defined(CONFIG_80211AX_HE) && LINUX_VERSION_CODE >= KERNEL_VERSION(4, 20, 0) */

	if ((u32)band == (u32)BAND_ON_24G) {
		rtw_2g_channels_init(spt_band->channels);
		rtw_2g_rates_init(spt_band->bitrates);
	} else if ((u32)band == (u32)BAND_ON_5G) {
		rtw_5g_channels_init(spt_band->channels);
		rtw_5g_rates_init(spt_band->bitrates);
	}

	/* spt_band.ht_cap */

exit:

	return spt_band;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)) || defined(COMPAT_KERNEL_RELEASE)
static const struct ieee80211_txrx_stypes
	rtw_cfg80211_default_mgmt_stypes[NUM_NL80211_IFTYPES] = {
	[NL80211_IFTYPE_ADHOC] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ACTION >> 4)
	},
	[NL80211_IFTYPE_STATION] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ACTION >> 4) |
		BIT(IEEE80211_STYPE_AUTH >> 4) |
		BIT(IEEE80211_STYPE_PROBE_REQ >> 4)
	},
	[NL80211_IFTYPE_AP] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_REASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_PROBE_REQ >> 4) |
		BIT(IEEE80211_STYPE_DISASSOC >> 4) |
		BIT(IEEE80211_STYPE_AUTH >> 4) |
		BIT(IEEE80211_STYPE_DEAUTH >> 4) |
		BIT(IEEE80211_STYPE_ACTION >> 4)
	},
	[NL80211_IFTYPE_AP_VLAN] = {
		/* copy AP */
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_REASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_PROBE_REQ >> 4) |
		BIT(IEEE80211_STYPE_DISASSOC >> 4) |
		BIT(IEEE80211_STYPE_AUTH >> 4) |
		BIT(IEEE80211_STYPE_DEAUTH >> 4) |
		BIT(IEEE80211_STYPE_ACTION >> 4)
	},
	[NL80211_IFTYPE_P2P_CLIENT] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ACTION >> 4) |
		BIT(IEEE80211_STYPE_PROBE_REQ >> 4)
	},
	[NL80211_IFTYPE_P2P_GO] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_REASSOC_REQ >> 4) |
		BIT(IEEE80211_STYPE_PROBE_REQ >> 4) |
		BIT(IEEE80211_STYPE_DISASSOC >> 4) |
		BIT(IEEE80211_STYPE_AUTH >> 4) |
		BIT(IEEE80211_STYPE_DEAUTH >> 4) |
		BIT(IEEE80211_STYPE_ACTION >> 4)
	},
#if defined(RTW_DEDICATED_P2P_DEVICE)
	[NL80211_IFTYPE_P2P_DEVICE] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ACTION >> 4) |
			BIT(IEEE80211_STYPE_PROBE_REQ >> 4)
	},
#endif
#if defined(CONFIG_RTW_MESH)
	[NL80211_IFTYPE_MESH_POINT] = {
		.tx = 0xffff,
		.rx = BIT(IEEE80211_STYPE_ACTION >> 4)
			| BIT(IEEE80211_STYPE_AUTH >> 4)
	},
#endif

};
#endif

static void rtw_cfg80211_preinit_wiphy(struct wiphy *wiphy, int idx)
{
	struct wfo_wiphy_data *wiphy_data = wiphy_priv(wiphy);
	wiphy_data->idx = (u8)idx;

	/* copy mac_addr to wiphy */
	//_rtw_memcpy(wiphy->perm_addr, adapter_mac_addr(adapter), ETH_ALEN);

	wiphy->signal_type = CFG80211_SIGNAL_TYPE_MBM;

	wiphy->max_scan_ssids = (u8)RTW_SSID_SCAN_AMOUNT;
	wiphy->max_scan_ie_len = (u16)RTW_SCAN_IE_LEN_MAX;
	wiphy->max_num_pmkids = (u8)RTW_MAX_NUM_PMKIDS;

#if 0//CONFIG_RTW_MACADDR_ACL && (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 9, 0))
	wiphy->max_acl_mac_addrs = NUM_ACL;
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 38)) || defined(COMPAT_KERNEL_RELEASE)
	wiphy->max_remain_on_channel_duration = (u16)RTW_MAX_REMAIN_ON_CHANNEL_DURATION;
#endif

{
	u32 interface_modes = BIT(NL80211_IFTYPE_STATION)
		| BIT(NL80211_IFTYPE_ADHOC)
		#ifdef CONFIG_AP_MODE
		| BIT(NL80211_IFTYPE_AP)
		#ifdef CONFIG_WIFI_MONITOR
		| BIT(NL80211_IFTYPE_MONITOR)
		#endif
		#endif
		#if defined(CONFIG_P2P) && ((LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)) || defined(COMPAT_KERNEL_RELEASE))
		| BIT(NL80211_IFTYPE_P2P_CLIENT)
		| BIT(NL80211_IFTYPE_P2P_GO)
		#if defined(RTW_DEDICATED_P2P_DEVICE)
		| BIT(NL80211_IFTYPE_P2P_DEVICE)
		#endif
		#endif

		#ifdef CONFIG_RTW_MESH
		| BIT(NL80211_IFTYPE_MESH_POINT)	/* 2.6.26 */
		#endif
		;
	wiphy->interface_modes = (u16)interface_modes;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37)) || defined(COMPAT_KERNEL_RELEASE)
#ifdef CONFIG_AP_MODE
	wiphy->mgmt_stypes = rtw_cfg80211_default_mgmt_stypes;
#endif /* CONFIG_AP_MODE */
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0))
	#ifdef CONFIG_WIFI_MONITOR
	wiphy->software_iftypes |= BIT(NL80211_IFTYPE_MONITOR);
	#endif
#endif

#if 0//(LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0)) && (CONFIG_IFACE_NUMBER >= 2)
	wiphy->iface_combinations = rtw_combinations;
	wiphy->n_iface_combinations = ARRAY_SIZE(rtw_combinations);
#endif

	wiphy->cipher_suites = rtw_cipher_suites;
	wiphy->n_cipher_suites = (int)ARRAY_SIZE(rtw_cipher_suites);

{
	u32 band = (wiphy_data->idx == 1)?(u32)NL80211_BAND_2GHZ:(u32)NL80211_BAND_5GHZ;
	wiphy_data->band = (u8)band;
}

	if (wiphy_data->band == NL80211_BAND_2GHZ)
		wiphy->bands[NL80211_BAND_2GHZ] = rtw_spt_band_alloc(BAND_ON_24G);
	else if (wiphy_data->band == NL80211_BAND_5GHZ)
		wiphy->bands[NL80211_BAND_5GHZ] = rtw_spt_band_alloc(BAND_ON_5G);
	else
		WFO_PRINT("unknow band[%d]!!!\n", wiphy_data->band);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 38) && LINUX_VERSION_CODE < KERNEL_VERSION(3, 0, 0))
	wiphy->flags |= (u32)WIPHY_FLAG_SUPPORTS_SEPARATE_DEFAULT_KEYS;
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 3, 0))
	wiphy->flags |= (u32)WIPHY_FLAG_HAS_REMAIN_ON_CHANNEL;
	wiphy->flags |= (u32)WIPHY_FLAG_HAVE_AP_SME;
	/* remove WIPHY_FLAG_OFFCHAN_TX, because we not support this feature */
	/* wiphy->flags |= WIPHY_FLAG_OFFCHAN_TX | WIPHY_FLAG_HAVE_AP_SME; */
#endif

#if !defined(CPTCFG_VERSION) && defined(CONFIG_PM) && (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0) && \
			   LINUX_VERSION_CODE < KERNEL_VERSION(4, 12, 0))
	wiphy->flags |= (u32)WIPHY_FLAG_SUPPORTS_SCHED_SCAN;
#ifdef CONFIG_PNO_SUPPORT
	wiphy->max_sched_scan_ssids = MAX_PNO_LIST_COUNT;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 2, 0)
	wiphy->max_match_sets = MAX_PNO_LIST_COUNT;
#endif
#endif
#endif

#if defined(CONFIG_PM) && (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0))
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3, 11, 0))
	wiphy->wowlan = wowlan_stub;
#else
	wiphy->wowlan = &wowlan_stub;
#endif
#endif

#if defined(CONFIG_TDLS) && (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 2, 0))
	wiphy->flags |= (u32)WIPHY_FLAG_SUPPORTS_TDLS;
#ifndef CONFIG_TDLS_DRIVER_SETUP
	wiphy->flags |= (u32)WIPHY_FLAG_TDLS_EXTERNAL_SETUP;	/* Driver handles key exchange */
	wiphy->flags |= (u32)NL80211_ATTR_HT_CAPABILITY;
#endif /* CONFIG_TDLS_DRIVER_SETUP */
#endif /* CONFIG_TDLS */

#if 0
	if (regsty->power_mgnt != PM_PS_MODE_ACTIVE)
		wiphy->flags |= WIPHY_FLAG_PS_ON_BY_DEFAULT;
	else
		wiphy->flags &= ~WIPHY_FLAG_PS_ON_BY_DEFAULT;
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 2, 0))
	/* wiphy->flags |= WIPHY_FLAG_SUPPORTS_FW_ROAM; */
#endif

#ifdef CONFIG_RTW_MESH
	wiphy->flags |= (u32)(0
		#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 37))
		| WIPHY_FLAG_IBSS_RSN
		#endif
		#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 0, 0))
		| WIPHY_FLAG_MESH_AUTH
		#endif
		);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 3, 0))
	wiphy->features |= (u32)(0
		#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 10, 0))
		| NL80211_FEATURE_USERSPACE_MPM
		#endif
		);
#endif /* (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 3, 0)) */
#endif /* CONFIG_RTW_MESH */

#ifdef CONFIG_RTW_MULTI_AP
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 33))
	DBGP("enable A4 \n");
	wiphy->flags |= (u32)WIPHY_FLAG_4ADDR_AP;
	wiphy->flags |= (u32)WIPHY_FLAG_4ADDR_STATION;
#endif
#endif

#if defined(CONFIG_RTW_80211K) && (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 6, 0))
	wiphy_ext_feature_set(wiphy, NL80211_EXT_FEATURE_RRM);
#endif

#if (KERNEL_VERSION(3, 8, 0) <= LINUX_VERSION_CODE)
	wiphy->features |= (u32)NL80211_FEATURE_SAE;
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 11, 0))
#ifdef CONFIG_WIFI_MONITOR
	/* Currently only for Monitor debugging */
	wiphy->flags |= (u32)WIPHY_FLAG_SUPPORTS_5_10_MHZ;
#endif
#endif /* (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 11, 0)) */

#ifdef CONFIG_CSA_IE
	wiphy->flags |= WIPHY_FLAG_HAS_CHANNEL_SWITCH;
#ifdef CONFIG_ECSA_IE
	wiphy->max_num_csa_counters = 2;
#else
	wiphy->max_num_csa_counters = 1;
#endif
#endif
}
#endif /* WFO_CMD_VIRT_CFG_OPS */



//==============================================================================
//  wireless_handlers support
//==============================================================================
#if defined(CONFIG_WIRELESS_EXT)
static iw_handler wfo_std_handler[56] = {0};

static int wfo_priv_std_handler(struct net_device *dev,
		struct iw_request_info *info,
		union iwreq_data *wdata, char *extra)
{
	return wfo_virt_cmd_sender_std_handler(dev, info, wdata, extra);
}

static iw_handler wfo_private_handler[0x1D+1] = {0};

static int wfo_priv_handler(struct net_device *dev,
		struct iw_request_info *info,
		union iwreq_data *wdata, char *extra)
{
	struct iw_point *p = (struct iw_point *)wdata;
	int ret = 0;

	if (extra == NULL) {
		p->length = 0U;
		return -EIO;
	}

	ret = wfo_virt_cmd_sender_iwpriv(dev, info, wdata, extra);

	return ret;
}

struct iw_handler_def wfo_iw_handler_defs = { /* in g6 is rtw_handlers_def */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 33)) || defined(CONFIG_WEXT_PRIV)
	.standard = wfo_std_handler,
	.num_standard = sizeof(wfo_std_handler) / sizeof(iw_handler),

	.private = wfo_private_handler,
	.private_args = NULL, /* assign it at init_iw_handler */
	.num_private = sizeof(wfo_private_handler) / sizeof(iw_handler),
	.num_private_args = 0, /* assign it at init_iw_handler */
#endif
#if WIRELESS_EXT >= 17
	.get_wireless_stats = wfo_get_wireless_stats,
#endif
};
#endif /* CONFIG_WIRELESS_EXT */


//==============================================================================
//  init wiphy/ndev
//==============================================================================
#if defined(WFO_CMD_VIRT_CFG_OPS)
static struct device_type wfo_wiphy_type = {
	.name	= WLAN_IFACE_NAME,
};

void wfo_virt_iface_random_ether_addr(u8 *mac_addr)
{
	u8 RTW_MAC_OUI[3] = {0x00U, 0xe0U, 0x4cU};

	random_ether_addr(mac_addr);
	memcpy(mac_addr, RTW_MAC_OUI, sizeof(RTW_MAC_OUI));
}

#if 1 /* copy from g6_wifi_driver/os_dep/linux/wifi_regd.h */
#define RTL819x_2GHZ_CH01_11    REG_RULE(2412-10, 2462+10, 40, 0, 20, 0)
#define RTL819x_2GHZ_CH12_13    REG_RULE(2467-30, 2472+10, 40, 0, 20, 0)
#define RTL819x_2GHZ_CH14       REG_RULE(2484-10, 2484+10, 20, 0, 20, 0)
/* 5G */
#define RTL819x_5GHZ_5150_5350  REG_RULE(5150-10, 5350+10, 80, 0, 30, 0)
#define RTL819x_5GHZ_5470_5850  REG_RULE(5470-10, 5850+10, 80, 0, 30, 0)

#define RTL_2GHZ_ALL            RTL819x_2GHZ_CH01_11, \
								RTL819x_2GHZ_CH12_13, \
								RTL819x_2GHZ_CH14

#define RTL_5GHZ_ALL            RTL819x_5GHZ_5150_5350, \
								RTL819x_5GHZ_5470_5850

static const struct ieee80211_regdomain rtl_regdom_all = {
		.n_reg_rules = 5,
		.alpha2 =  "00",
		.reg_rules = {
				RTL_2GHZ_ALL,
				RTL_5GHZ_ALL,
		}
};
#endif

void wfo_virt_regd_init(struct wiphy *wiphy)
{
	const struct ieee80211_regdomain *regd = NULL;
	unsigned int size = sizeof(struct ieee80211_regdomain) +
		((&rtl_regdom_all)->n_reg_rules) * sizeof(struct ieee80211_reg_rule);

	if (!wiphy->regd) {
		regd = kzalloc(size, GFP_KERNEL);
		memcpy((void *)regd, (void *)&rtl_regdom_all, size);
		rcu_assign_pointer(wiphy->regd, regd);
	}

	wiphy->reg_notifier = wfo_virt_cmd_sender_reg_notifier;
	wiphy->regulatory_flags = (u32)REGULATORY_WIPHY_SELF_MANAGED;
	regulatory_hint(wiphy, "00");
}

int wfo_virt_iface_init_wiphy_parameter(struct wiphy *wiphy, int idx)
{
	int ret = 0;
	struct wfo_wiphy_data *wiphy_data = wiphy_priv(wiphy);

	rtw_cfg80211_preinit_wiphy(wiphy, idx);
	//set_wiphy_dev(wiphy, wiphy_data->ndev);

	memcpy(wiphy->perm_addr, wiphy_data->mac_addr, ETH_ALEN);

	/* init reg_notifier */
	wfo_virt_regd_init(wiphy);

	return ret;
}

void wfo_virt_iface_set_dev_name(unsigned char *dev_name, unsigned char idx)
{
	idx = wfo_get_ndev_id(idx);

	if (dev_name)
		snprintf(dev_name, (u32)IFNAMSIZ, VIRT_DEVNAME, idx);
	else
		WFO_PRINT("dev_name is null!!\n");
}

struct wiphy *wfo_virt_iface_create_wiphy(unsigned char idx)
{
	struct wfo_wiphy_data *wiphy_data = NULL;
	struct wiphy *wiphy = NULL;

	wiphy = wiphy_new(&wfo_virt_cfg80211_ops, (s32)sizeof(struct wfo_wiphy_data));
	if (wiphy) {
		wiphy_data = wiphy_priv(wiphy);
		wiphy_data->wiphy = wiphy;
	} else {
		WFO_PRINT("allocate wiphy fail!!\n");
		return NULL;
	}

	wfo_virt_iface_set_dev_name(wiphy_data->dev_name, idx);
	wfo_virt_iface_random_ether_addr(wiphy_data->mac_addr);

	return wiphy;
}

#ifdef CONFIG_WIRELESS_EXT
extern iw_handler rtw_private_handler[];

void init_iw_handler(void)
{
	int i = 0;
	struct net_device *ndev = NULL;
	ndev = wfo_dev_get_by_name(&init_net, WLAN_G6);

	if (ndev) {
		for (i=0; i<wfo_iw_handler_defs.num_standard; i++) {
			if (ndev->wireless_handlers->standard[i])
				wfo_std_handler[i] = wfo_priv_std_handler;
		}

		for (i=0; i<ndev->wireless_handlers->num_private; i++) {
			if (ndev->wireless_handlers->private[i])
				wfo_private_handler[i] = wfo_priv_handler;
		}

		wfo_iw_handler_defs.private_args = ndev->wireless_handlers->private_args;
		wfo_iw_handler_defs.num_private_args = ndev->wireless_handlers->num_private_args;
	} else {
		WFO_PRINT("can't find G6 device!!!\n\n\n");
	}
}
#endif /* CONFIG_WIRELESS_EXT */

#ifdef WIFI6_THER_CTRL
static void therctl_sync_mib_wifi6(struct net_device *dev, struct ther_info_s *info)
{
	wfo_virt_cmd_sender_therctl_ops(dev, THERCTL_GETVAL, (void *) info);
}

static void therctl_set_limit_tp_wifi6(struct net_device *dev, int level)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_LIMIT_TP, (void *) &level);
}

static void therctl_set_bandwidth_wifi6(struct net_device *dev, int bw)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_BANDWIDTH, (void *) &bw);
}

static void therctl_set_txduty_wifi6(struct net_device *dev, int level)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_TX_DUTY, (void *) &level);
}

static void therctl_set_power_wifi6(struct net_device *dev, int low_power)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_POWER, (void *) &low_power);
}

static void therctl_set_path_wifi6(struct net_device *dev, int path)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_SET_PATH, (void *) &path);
}

static void therctl_set_funcoff_wifi6(struct net_device *dev, int enable)
{
	wfo_virt_cmd_sender_therctl_ops_async(dev, THERCTL_FUNC_OFF, (void *) &enable);
}

static struct therctl_ops wfo_therctl_wifi6 = {
	.sync_mib	= therctl_sync_mib_wifi6,
	.set_limit_tp	= therctl_set_limit_tp_wifi6,
	.set_bandwidth	= therctl_set_bandwidth_wifi6,
	.set_txduty	= therctl_set_txduty_wifi6,
	.set_power	= therctl_set_power_wifi6,
	.set_path	= therctl_set_path_wifi6,
	.set_funcoff	= therctl_set_funcoff_wifi6,
};

void wfo_reg_wifi6_ther_support(struct net_device *dev)
{
	struct therctl_ops *ops = &wfo_therctl_wifi6;

	ops->dev = dev;
	reg_therctl_ops(ops);
}
EXPORT_SYMBOL(wfo_reg_wifi6_ther_support);
#endif /* WIFI6_THER_CTRL */

int wfo_virt_iface_register_netdev(struct net_device *ndev,
		struct wiphy *wiphy,
		enum nl80211_iftype type,
		unsigned int vap_idx)
{
	struct wfo_ndev_priv *ndev_priv = (struct wfo_ndev_priv *)netdev_priv(ndev);
	struct wfo_wiphy_data *wiphy_data = wiphy_priv(wiphy);
	int ret = 0;

	unsigned char *dev_name = rtl8198d_wfo_wlandevname(vap_idx);
	if (vap_idx == 0U)
		wiphy_data->ndev = ndev;
	if (dev_name) {
		memset(ndev->name, 0U, (u32)IFNAMSIZ);
		memcpy(ndev->name, dev_name, (u32)IFNAMSIZ);
	}

	memcpy(ndev->dev_addr, wiphy_data->wiphy->perm_addr, ETH_ALEN);
	ndev->dev_addr[ETH_ALEN - 1] += vap_idx;
	dev_net_set(ndev, wiphy_net(wiphy));

	ndev_priv->wdev.wiphy = wiphy;

	ndev->ieee80211_ptr = &ndev_priv->wdev;

	snprintf(ndev_priv->radio_name, (u32)IFNAMSIZ, "%s%s", ndev->name, WFO_IFACE_POSTFIX);

#ifdef CONFIG_WIRELESS_EXT
	if (vap_idx == 0U)
		init_iw_handler();
	ndev->wireless_handlers = (struct iw_handler_def *)&wfo_iw_handler_defs;
#endif /* CONFIG_WIRELESS_EXT */

	SET_NETDEV_DEV(ndev, wiphy_dev(wiphy));

	ndev_priv->wdev.netdev = ndev;
	ndev_priv->wdev.iftype = type;
	ndev_priv->iface_id = (u8)vap_idx;

	SET_NETDEV_DEVTYPE(ndev, &wfo_wiphy_type);

#if !defined(USE_GLOBAL_CFG)
#if (WFO_CFG_SMP==1)
{
	int i;
	wfo_cfg80211_t *cfg = NULL;
	for (i=0; i<NR_CPUS; i++) {
		cfg = &ndev_priv->cfg[i];
		atomic_set(&cfg->busy, 0);
	}
}
#endif /* WFO_CFG_SMP */
#endif /* !USE_GLOBAL_CFG */

	if (vap_idx == 0U)
		ret = register_netdev(ndev);
	else
		ret = register_netdevice(ndev);

#if defined(WIFI6_THER_CTRL)
	if (vap_idx==0U) {
		wfo_reg_wifi6_ther_support(ndev);
	}
#endif /* WIFI6_THER_CTRL */

	return ret;
}

void wfo_virt_iface_unregister_netdev(struct net_device *ndev)
{
	unregister_netdevice(ndev);
#if !defined(CPTCFG_WFO_VIRT_SAME_CPU)
	wfo_virt_cmd_sender_unregister_ndev(ndev);
#endif
}

struct net_device *wfo_virt_iface_alloc_etherdev(void)
{
	struct net_device *ndev = alloc_etherdev_mq((int)sizeof(struct wfo_ndev_priv), 1U);
	struct wfo_ndev_priv *ndev_priv = NULL;

	if (ndev) {
		strcpy(ndev->name, VIRT_DEVNAME);
		ndev->netdev_ops = &wfo_virt_netdev_ops;
	} else {
		WFO_PRINT("allocate net_device fail!!\n");
		return NULL;
	}

	netif_carrier_off(ndev);

	ndev_priv = (struct wfo_ndev_priv *)netdev_priv(ndev);
	ndev_priv->ndev = ndev;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0))
	ndev->rtk_priv_flags |= (u32)RTK_IFF_DOMAIN_WLAN;
#else
	ndev->priv_flags |= (u32)IFF_DOMAIN_WLAN;
#endif

	return ndev;
}

int wfo_virt_iface_init_cfg80211(void)
{
	unsigned char i = 0U;
	struct net_device *ndev = NULL;
	struct wiphy *wiphy = NULL;

	for(i = 0U; i < n_virt_iface; i++) {
		int ret;
		if (virt_iface[i]) {
			wiphy = wfo_virt_iface_create_wiphy(i);
			if (!wiphy)
				return -1;

			g_wiphy[i] = wiphy;

			ndev = wfo_virt_iface_alloc_etherdev();
			if (!ndev)
				return -1;

			wfo_virt_iface_init_wiphy_parameter(wiphy, (s32)i); /* root only */
			wfo_virt_iface_register_netdev(ndev, wiphy, NL80211_IFTYPE_STATION, 0U);
			rtw_virt_adapter_proc_init(ndev);

			ret = wiphy_register(wiphy);
			if (ret) {
				WFO_PRINT("couldn't register wiphy device, ret=%d\n", ret);
				return ret;
			}
		}
	}

	return 0;
}

void wfo_virt_iface_post_init_cfg80211(void)
{
	unsigned char i = 0U;

	wfo_dram_t *dram = RTK_SHM_DRAM;
	memset(&dram->md, 0, sizeof(struct middleware));

	for(i = 0U; i < n_virt_iface; i++) {
		if (virt_iface[i]) {
			struct wiphy *wiphy = g_wiphy[i];
			if (wiphy) {
				wfo_virt_cmd_sender_band_cap(wiphy, wiphy_to_ndev(wiphy));
			}
		}
	}
}
#endif /* WFO_CMD_VIRT_CFG_OPS */


//==============================================================================
//  init module
//==============================================================================
int check_module_parameter(void)
{
	int ret = 0;

	if (n_virt_iface != WLAN_ROOT_IFACE_NUM) {
		WFO_PRINT("n_virt_iface[%d] != WLAN_ROOT_IFACE_NUM[%d]\n",
			n_virt_iface, WLAN_ROOT_IFACE_NUM);
		ret = 1;
	}

	#if defined(CPTCFG_WFO_OFFLOAD_2G)
		virt_iface[0] = 0;
		virt_iface[1] = 1;
	#elif defined(CPTCFG_WFO_OFFLOAD_5G)
		virt_iface[0] = 1;
		virt_iface[1] = 0;
	#else
		#error "Please enable CPTCFG_WFO_OFFLOAD_2G or CPTCFG_WFO_OFFLOAD_5G!"
	#endif

	return ret;
}

extern int (*wfo_sender_netlink_recv_msg)(struct sk_buff *skb);
void register_g6_netlink_recv_msg_ops(void)
{
#if defined(CONFIG_WLAN_MANAGER) && !defined(RTW_COMMON_NETLINK)
	wfo_sender_netlink_recv_msg = &wfo_virt_cmd_sender_netlink_recv_msg;
#endif
}

#if defined(WFO_SYNC_STA_STATUS) && defined(WFO_VIRT_SENDER) && defined(CONFIG_RTK_L34_FLEETCONNTRACK_ENABLE)
extern void (*rtk_fc_wfo_deliver_sta_status)(struct wfo_sta_st_s *sta_st, u32 sta_st_sz);

void wfo_virt_deliver_sta_status(struct wfo_sta_st_s *sta_st, u32 sta_st_sz)
{
	return wfo_virt_cmd_sender_sync_sta_status(sta_st, sta_st_sz);
}
#endif

#if (WFO_IFACE_NUM != CONFIG_IFACE_NUMBER)
#error "PLEASE check CONFIG_IFACE_NUMBER & WFO_IFACE_NUM!!"
#endif

static int __init wfo_virt_init(void)
{
	wfo_dram_t *dram = RTK_SHM_DRAM;
	WFO_PRINT("load %s\n", MODULE_NAME);

	if (check_module_parameter())
		return -1;

#if defined(WFO_RADIO_RECEIVER)
	init_wfo_radio_cmd_receiver_fp();
	register_wfo_radio_ipc_cb();
#endif /* WFO_RADIO_RECEIVER */

#if defined(WFO_VIRT_SENDER)
	init_wfo_virt_cmd_sender_sync_type();
#endif /* WFO_VIRT_SENDER */

	if (wfo_virt_iface_init_cfg80211())
		return -1;

#if (WFO_CFG_DBG_TRACE_OPS==1)
	dbg_trace_ops();
#endif /* WFO_CFG_DBG_TRACE_OPS */

	wfo_virt_iface_post_init_cfg80211();

//#if !defined(CONFIG_HWNAT_FLEETCONNTRACK) && !defined(CONFIG_RTL_8198D_TAROKO)
	rtk_wfo_pseudo_wlan_init();
//#endif

#ifdef CONFIG_RTW_A4_STA
	reg_check_srcmac_for_ax_driver();
#endif /* CONFIG_RTW_A4_STA */
	register_g6_netlink_recv_msg_ops();

	dram->is_scanning = 0U;

#if defined(WFO_SYNC_STA_STATUS) && defined(WFO_VIRT_SENDER) && defined(CONFIG_RTK_L34_FLEETCONNTRACK_ENABLE)
	rtk_fc_wfo_deliver_sta_status = wfo_virt_deliver_sta_status;
#endif

#if defined(CONFIG_RTK_TAROKO_IPC )
{
	extern void (*show_wfo_trace_count_fp)(void);
	show_wfo_trace_count_fp = show_wfo_trace_count;
}
#endif /* CONFIG_RTK_TAROKO_IPC */

	WFO_PRINT("load %s: ok\n", MODULE_NAME);

	return 0;
}

static void __exit wfo_virt_cleanup(void)
{
	//not implement
}

module_init(wfo_virt_init);
module_exit(wfo_virt_cleanup);
MODULE_LICENSE("GPL");

