#include <soc.h>
#include <symb_define.h>
#include <cg/cg.h>
#include <bamt.h>

#define SECTION_FT __attribute__ ((section (".text")))

#define FT_SYMB_STATUS 0xF7E57F16
#define FT_SYMB_PUTC   0xF7E57C5L

u32_t ft_plr_status;

SECTION_UART void uart_putc(char c);
SECTION_UART void silent_putc(char c) {
	return;
}

symb_pdefine(ft_plr_status, FT_SYMB_STATUS, &ft_plr_status);
symb_fdefine(FT_SYMB_PUTC, uart_putc);

SECTION_FT
static u32_t ft_ocp_freq(void) {
#define IO_MODE_EN *((volatile u32_t *)0xbb023004)
#define PABCD_DAT  *((volatile u32_t *)0xb800330c)
	u32_t io_mode_en_bak, val;

	/* Disable JTAG to enable GPIO2 */
	io_mode_en_bak = IO_MODE_EN;
	IO_MODE_EN = io_mode_en_bak | 2;

	if (PABCD_DAT & 0x0004) {
		val = 400;
	} else {
		val = 100;
	}

	IO_MODE_EN = io_mode_en_bak;

	return val;
}

SECTION_FT
void ft_disable_putc(void) {
	_bios.uart_putc = silent_putc;
	return;
}
REG_INIT_FUNC(ft_disable_putc, 2);

SECTION_FT
void ft_enable_putc(void) {
	_bios.uart_putc = uart_putc;
	return;
}

SECTION_FT
void ft_init(void) {
	cg_info_query.dev_freq.ocp_mhz = ft_ocp_freq();
	return;
}
REG_INIT_FUNC(ft_init, 16);

SECTION_FT
u32_t ft_dram_addr_pin_walking_1(const u32_t dram_size) {
	u32_t a = 2;
	u32_t *addr;
	volatile u16_t *saddr = (u16_t *)0xa0000002;
	volatile u8_t  *caddr = (u8_t *)0xa0000000;
	u32_t erraddr = 0, errcnt = 0;

	*caddr++ = 0xff;
	*caddr++ = 0x5a;
	*saddr   = 0xa503;

	while ((1 << a) < dram_size) {
		addr = (u32_t *)(0xa0000000 + (1 << a));
		*addr = (u32_t)addr;
		a++;
	}

	addr = (u32_t *)0xa0000000;
	if (*addr != 0xff5aa503) {
		errcnt++;
		FT_PRINTF("EE: %08x@%08x != %08x\n", *addr, addr, 0xff5aa503);
	}
	addr++;
	while ((u32_t)addr <= (0xa0000000 + (dram_size/2))) {
		a = *addr;
		if ((a != 0) && (a != ((u32_t)addr))) {
			erraddr = (u32_t)addr;
			if ((errcnt++) < 16) {
				FT_PRINTF("EE: %08x@%08x != %08x\n", a, addr, addr);
			}
		}
		addr++;
	}

	if (errcnt) {
		FT_PRINTF("EE: ... ~ %08x, total: %d W\n", erraddr, errcnt);
	}

	return (errcnt > 0);
}

SECTION_FT
void ft_dram_test(void) {
	bonding_desc_t dram_side_dq_rel[] = {
		{9,  14},
		{9,  15},
		{15, 8},
		{15, 9},
		{12, 11},
		{12, 10},
		{10, 13},
		{10, 12},
		{01, 6},
		{1,  7},
		{7,  0},
		{7,  1},
		{3,  4},
		{3,  5},
		{5,  2},
		{5,  3},
		{-1, -1},
	};

	bonding_desc_t soc_side_dq_rel[] = {
		{14, 15},
		{9,  8},
		{11, 10},
		{13, 12},
		{7,  6},
		{7,  0},
		{0,  7},
		{0,  1},
		{5,  4},
		{5,  2},
		{3,  3},
		{-1, -1},
	};

	/* Commented out before addr. bonding test completed. */
	/* bonding_desc_t soc_side_addr_rel[] = { */
	/* 	{10, 10}, */
	/* 	{1, 0}, */
	/* 	{0, 1}, */
	/* 	{0, 2}, */
	/* 	{2, 0}, */
	/* 	{2, 3}, */
	/* 	{3, 2}, */
	/* 	{3, 5}, */
	/* 	{5, 3}, */
	/* 	{5, 4}, */
	/* 	{5, 7}, */
	/* 	{4, 5}, */
	/* 	{4, 7}, */
	/* 	{4, 6}, */
	/* 	{7, 5}, */
	/* 	{7, 4}, */
	/* 	{7, 6}, */
	/* 	{7, 9}, */
	/* 	{6, 4}, */
	/* 	{6, 7}, */
	/* 	{6, 9}, */
	/* 	{6, 8}, */
	/* 	{9, 7}, */
	/* 	{9, 6}, */
	/* 	{9, 8}, */
	/* 	{9, 12}, */
	/* 	{8, 6}, */
	/* 	{8, 9}, */
	/* 	{8, 12}, */
	/* 	{8, 11}, */
	/* 	{12, 9}, */
	/* 	{12, 8}, */
	/* 	{12, 11}, */
	/* 	{11, 8}, */
	/* 	{11, 12}, */
	/* 	{-1, -1}, */
	/* }; */

	mc_addr_info_t mc_ap_info;
	u32_t dram_size;
	u32_t cycle[5];

	ft_plr_status = 0xF7E57000;

	ft_enable_putc();
	cg_result_decode();
	ft_disable_putc();

	if (cg_info_query.dev_freq.ocp_mhz != 400) {
		/* Only apply DRAM test when OCP is 400MHz. */
		return;
	}

	/* ft_enable_putc(); */

	bamt_get_addr_pin_info(&mc_ap_info);
	dram_size = (1 << mc_ap_info.bank_offset) * mc_ap_info.bank_num * (1 << RFLD_DCR(dbuswid));

	cycle[0] = CPU_GET_CP0_CYCLE_COUNT();

	/* Set and check each bit for 0 */
	bamt_dram_fill_with_cache((u32_t *)0x80000000, 0x00000000, dram_size, 16*1024);
	ft_plr_status |= bamt_dram_verify((u32_t *)(0x80000000 + (dram_size/2)),
	                                  0x00000000,
	                                  (dram_size/2));

	cycle[1] = CPU_GET_CP0_CYCLE_COUNT();

	/* Toggle each address pins */
	ft_plr_status |= ft_dram_addr_pin_walking_1(dram_size) << 1;

	cycle[2] = CPU_GET_CP0_CYCLE_COUNT();

	/* Apply bonding-aware test */
	ft_plr_status |= bonding_aware_memory_test(dram_side_dq_rel, &mc_ap_info, bamt_data) << 2;
	ft_plr_status |= bonding_aware_memory_test(soc_side_dq_rel,  &mc_ap_info, bamt_data) << 3;
	/* ft_plr_status |= bonding_aware_memory_test(soc_side_addr_rel, &mc_ap_info, bamt_addr); */

	cycle[3] = CPU_GET_CP0_CYCLE_COUNT();

	/* Set and check each bit for 1 */
	bamt_dram_fill_with_cache((u32_t *)0x80000000, 0xffffffff, dram_size, 16*1024);
	ft_plr_status |= bamt_dram_verify((u32_t *)0x80000000, 0xffffffff, dram_size) << 4;

	cycle[4] = CPU_GET_CP0_CYCLE_COUNT();

	FT_PRINTF("DD: %d/%d/%d/%d ms -> Total: %d ms\n",
	          (cycle[1] - cycle[0])/cg_info_query.dev_freq.ocp_mhz/1000,
	          (cycle[2] - cycle[1])/cg_info_query.dev_freq.ocp_mhz/1000,
	          (cycle[3] - cycle[2])/cg_info_query.dev_freq.ocp_mhz/1000,
	          (cycle[4] - cycle[3])/cg_info_query.dev_freq.ocp_mhz/1000,

	          (cycle[4] - cycle[0])/cg_info_query.dev_freq.ocp_mhz/1000);

	ft_enable_putc();

	if (ft_plr_status & 0xfff) {
		printf("memory test fail: %03x\n", ft_plr_status & 0xfff);
	} else {
		printf("memory test pass\n");
	}

	ft_disable_putc();

	return;
}
REG_INIT_FUNC(ft_dram_test, 30);
