#include <common.h>
#include <mpcntl.h>

typedef struct {
	char core_num;
	char vpe_num;
	char tc_num;
} mp_status_t;

static mp_status_t mp_status = {0};

static void mpcntl_count_mp(void) {
	C0_MVPCONF0_T c0_mvpconf0;
	c0_mvpconf0.v = GET_CP0(C0_MVPCONF0);
	mp_status.tc_num = c0_mvpconf0.f.ptc + 1;
	mp_status.vpe_num = c0_mvpconf0.f.pvpe + 1;
	mp_status.core_num = RFLD_GCR_CONFIG(pcores) + 1;
	return;
}

static int mp_count_core(void) {
	if (mp_status.core_num == 0) {
		mpcntl_count_mp();
	}
	return mp_status.core_num;
}

static int mp_count_vpe(void) {
	if (mp_status.vpe_num == 0) {
		mpcntl_count_mp();
	}
	return mp_status.vpe_num;
}

static int mp_count_tc(void) {
	if (mp_status.tc_num == 0) {
		mpcntl_count_mp();
	}
	return mp_status.core_num;
}

static void mpcntl_info_core(void) {
	int core_num, cid;

	const char *pwup_policy[] = {
		"Power Down",
		"Clock-Off",
		"Power Up"
	};

	const char *seq_state[] = {
		"D0 - PwrDwn",
		"U0 - VddOK",
		"U1 - UpDelay",
		"U2 - UClkOff",
		"U3 - Reset",
		"U4 - ResetDly",
		"U5 - nonCoherent execution",
		"U6 - Coherent execution",
		"D1 - Isolate",
		"D3 - ClrBus",
		"D2 - DClkOff",
	};

	const char *cmd[] = {
		"Reserved",
		"ClockOff",
		"PwrDown",
		"PwrUp",
		"Reset"
	};

	if (GCR_CPC_BASErv & 1) {
		core_num = mp_count_core();

		cid = 0;
		while (cid < core_num) {
			RMOD_CPC_CL_OTHER(corenum, cid);
			RMOD_GCR_CL_OTHER(corenum, cid);
			mp_printf("Core %d: PWUP_POLICY: %s, SEQ_STATE: %s, latest CMD: %s, BEVEXCBASE: %x\n",
			       cid,
			       pwup_policy[RFLD_CPC_CO_STAT_CONF(pwup_policy)],
			       seq_state[RFLD_CPC_CO_STAT_CONF(seq_state)],
			       cmd[RFLD_CPC_CO_STAT_CONF(cmd)],
			       GCR_CO_RESET_BASErv);
			cid++;
		}
	} else {
		printf("GCR_CPC_BASE is not enabled: %08x\n", GCR_CPC_BASErv);
	}

	return;
}

static void mpcntl_info_system(void) {
	mp_printf("Global Control Register (GCR) Base expected: %08x\n", GCR_CONFIG_ADDR);
	mp_printf("GCR Base Reg. Val.: %08x\n", GCR_BASErv);
	mp_printf("Global Interrupt Controller (GIC) Base Reg. Val.: %08x\n", GCR_GIC_BASErv);
	mp_printf("Cluster Power Controller (CPC) Base Reg. Val.: %08x\n", GCR_CPC_BASErv);
	mp_printf("CP0 PrID Val.: %08x\n", GET_CP0(C0_PRID));

	mp_printf("Total TC/VPE/Core num.: %d/%d/%d, IOCU num.: %d\n",
	          mp_count_tc(), mp_count_vpe(), mp_count_core(), RFLD_GCR_CONFIG(numiocu));

	return;
}

void mpcntl_init_other_vpes(const int vpe_num) {
	int vid = 1;
	int tid;

	/* enter VPE configure state */
	SET_CP0_F(1, C0_MVPCTL, vpc);

	while (vid < vpe_num) {
		/* select target TC */
		tid = vid;
		SET_CP0_F(tid, C0_VPECTL, targtc);

		/* halt the TC */
		MTSET_CP0_F(1, C0_TCHALT, h);

		/* ignore interrupt of the TC */
		MTSET_CP0((1 << 10), C0_TCSTATUS);

		/* bind TC to VPE */
		MTSET_CP0_F(vid, C0_TCBIND, curvpe);

		/* restrict VPE to the selected TC only. */
		MTSET_CP0_F(tid, C0_VPECONF0, xtc);

		/* disable threading; only one TC can execute on a VPE. */
		MTSET_CP0_F(0, C0_VPECTL, te);

		/* set to be master so the VPE can config other VPEs */
		MTSET_CP0_F(1, C0_VPECONF0, mvp);

		/* deactivate the VPE */
		MTSET_CP0_F(0, C0_VPECONF0, vpa);

		/* init CP0 regs */
		MTSET_CP0(GET_CP0(C0_STATUS), C0_STATUS);
		MTSET_CP0(0x12345678, C0_EPC);
		MTSET_CP0(0, C0_CAUSE);
		MTSET_CP0(GET_CP0(C0_CONFIG), C0_CONFIG);

		/* set start point */
		MTSET_CP0(MPCNTL_CPU_START_ADDR, C0_TCRESTART);

		/* activate TC */
		MTSET_CP0_F(1, C0_TCSTATUS, a);

		/* unhalt TC */
		MTSET_CP0_F(0, C0_TCHALT, h);

		/* activate VPE */
		MTSET_CP0_F(1, C0_VPECONF0, vpa);

		vid++;
	}

	/* enable all VPE to fetch and execute */
	SET_CP0_F(1, C0_MVPCTL, evp);

	/* exit VPE configuring state */
	SET_CP0_F(0, C0_MVPCTL, vpc);

	return;
}

static void mpcntl_init_other_cores(const int core_num) {
	int cid = 1;
	while (cid < core_num) {
		RMOD_GCR_CL_OTHER(corenum, cid);
		GCR_CO_RESET_BASErv = MPCNTL_CPU_START_ADDR;
		while (GCR_CO_RESET_BASErv != MPCNTL_CPU_START_ADDR) {
			;
		}
		RMOD_CPC_CL_OTHER(corenum, cid);
		RMOD_CPC_CO_CMD(cmd, CPC_CMD_RESET);
		cid++;
	}
	return;
}

void mpcntl_join_domain(void) {
	int core_num, vpe_num, cid;

	core_num = mp_count_core();
	vpe_num = mp_count_vpe();

	/* allow other CPUs to access GCR reg. */
	GCR_CL_COHERENCErv = (1 << (vpe_num * core_num)) - 1;

	cid = 0;
	while (cid < core_num) {
		RMOD_GCR_CL_OTHER(corenum, cid);
		while (GCR_CO_COHERENCErv == 0x0) {
			;
		}
		cid++;
	}

	return;
}
#if 0
static void mpcntl_info_tc(void) {
	int tc_num, tid;
	C0_TCBIND_T c0_tcbind;
	C0_TCSTATUS_T c0_tcstatus;
	C0_TCRESTART_T c0_tcrestart;
	C0_TCHALT_T c0_tchalt;
	C0_VPECTL_T c0_vpectl;

	tc_num = mp_count_tc();

	c0_tcbind.v = GET_CP0(C0_TCBIND);
	mp_printf("Cur. TC/VPE/Core: %d/%d/%d\n",
	          c0_tcbind.f.curtc, c0_tcbind.f.curvpe, RFLD_GCR_CL_ID(corenum));

	tid = 0;
	while (tid < tc_num) {
		/* Select target TC */
		c0_vpectl.v = GET_CP0(C0_VPECTL);
		c0_vpectl.f.targtc = tid;
		SET_CP0(c0_vpectl.v, C0_VPECTL);

		/* TCHALT for TC halt state */
		c0_tchalt.v = MTGET_CP0(C0_TCHALT);
		c0_tcstatus.v = MTGET_CP0(C0_TCSTATUS);
		c0_tcbind.v = MTGET_CP0(C0_TCBIND);
		c0_tcrestart.v = MTGET_CP0(C0_TCRESTART);
		mp_printf("TC %d @ vpe %d: halt = %d, C0_TCSTATUS = %08x, C0_TCRESTART = %08x\n",
		          tid,
		          c0_tcbind.f.curvpe,
		          c0_tchalt.f.h,
		          c0_tcstatus.v,
		          c0_tcrestart.v);
		tid++;
	}

	return;
}
#endif
void mp_init_mipsia(void) {
	int core_num, vpe_num;
	uint32_t c0_config = GET_CP0(C0_CONFIG);

	/* common GCR init */
	GCR_CPC_BASErv = PADDR(CPC_CONFIG_ADDR) | 1;
	GCR_GIC_BASErv = PADDR(GIC_CONFIG_ADDR) | 1;
	GCR_REG0_BASErv = 0xffff0000; /* clear CM2 regions for IOCU */
	GCR_REG1_BASErv = 0xffff0000;
	GCR_REG2_BASErv = 0xffff0000;
	GCR_REG3_BASErv = 0xffff0000;
	GCR_REG0_MASKrv = 0xffff0000;
	GCR_REG1_MASKrv = 0xffff0000;
	GCR_REG2_MASKrv = 0xffff0000;
	GCR_REG3_MASKrv = 0xffff0000;

	vpe_num = mp_count_vpe();
	core_num = mp_count_core();

	/* allow other CPUs to access GCR reg. */
	GCR_ACCESSrv = (1 << (vpe_num * core_num)) - 1;

	mpcntl_info_system();

	mp_printf("Reset default core status\n");
	mpcntl_info_core();

	/* Enable CCA of CPU0 */
	flush_cache(0, 0);
	c0_config &= (~0x7);
	c0_config |= 0x5;
	SET_CP0(c0_config, C0_CONFIG);
	mpcntl_init_other_cores(core_num);

	mpcntl_join_domain();

	mpcntl_init_other_vpes(vpe_num);

	/* mpcntl_info_tc(); */

	/* mp_printf("Current MIPS interAptiv core status\n"); */
	/* mpcntl_info_core(); */

	return;
}

void mp_kill_mipsia(void) {
	__asm__ __volatile__("wait");
	mp_printf("EE: CPU%d is still alive\n", mp_cpuid());
	while (1) {
		;
	}
	return;
}
