#include <linux/ctype.h>
#include <common.h>
#include <command.h>
#include "dramtest.h"

#define PLAT_BACKUP_SIZE_B (128)

static char ps[4] = {'|', '/', '-', '\\'};
static char *pstr, *fstr;

/*
 * Do RAM test.
 */
static void dtest_assign_default(dramtest_info_t *info) {
	memset(info, 0, sizeof(dramtest_info_t));
	info->loop = 1;
	info->stop = 1;
	info->dbg = 0;
	info->taddr = 0x0;          // physical address
	info->tsize = 0x1000000;    // 16MB
	info->maddr[0] = VA2PA(CONFIG_SYS_TEXT_BASE) - 0x400000;
	info->msize[0] = 0x800000;  // 8MB
	info->sp_case = 0xBEEF;
	/*  info->maddr[1] = 0x1F000000;
			info->msize[1] = 0x10000;   //64KB */
}

#define SWAP(x, y) ({uint32_t __x; __x=x; x=y; y=__x;})
static int dtest_process_segment(dramtest_info_t *info) {
	uint32_t i, j, done=0;
	uint32_t start = info->taddr;
	uint32_t end = info->taddr + info->tsize;
	uint32_t _start, _end;

	for (i=0; i<MASK_ENT_MAX; i++) {
		for (j=0; j<(MASK_ENT_MAX-i); j++) {
			if (0 == info->msize[j+1]) break;
			if (info->maddr[j] > info->maddr[j+1]) {
				SWAP(info->maddr[j], info->maddr[j+1]);
				SWAP(info->msize[j], info->msize[j+1]);
			}
		}
	}

	j = 0;

	for (i=0; i<MASK_ENT_MAX; i++) {
		if (0 == info->msize[i]) break;
		_start = info->maddr[i];
		_end = info->maddr[i] + info->msize[i];

		if (end<=_start || start>=_end) {
			;// no overlap
		} else {    // end>_start && start<_end
			if (start < _start) {
				info->seg[j] = start;
				info->ssize[j] = _start-start;
				j++;
			}
			if(end > _end) {
				start = _end;
			} else {
				done = 1;
				break;
			}
		}
	}
	if (!done) {
		// record the last one
		info->seg[j] = start;
		info->ssize[j] = end - start;
	}

	return 0;
}

static void dtest_dump_info(const dramtest_info_t *info) {
	uint32_t i;
	printf("II: DCache Size: %dKB\n", plat_dcache_size_b()>>10);
	puts("II: Test Info:\n");
	printf("\t Loops: %d, Block: %c\n", info->loop, info->stop?'Y':'N');
	printf("\t Test Range: 0x%08x ~ 0x%08x\n", info->taddr, info->taddr+info->tsize);
	printf("\t Test Segment:\n");
	for (i=0; i<SEG_ENT_MAX; i++) {
		if (0 == info->ssize[i]) break;
		printf("\t   [%d] 0x%08x~0x%08x\n", i, info->seg[i], info->seg[i]+info->ssize[i]);
	}
	return;
}

static void dtest_process_cpu_info(dramtest_info_t *info) {
	info->tunit = plat_dcache_size_b()<<2;
	return;
}

int do_wdtest (cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	dramtest_info_t info;
	uint32_t i, j=0, l, c, r;
	dtest_func_ptr_t *dtest_func_ptr;
	uint32_t ssize, start, bsize, base;
	void *plat_backup;

	i=1;

	// assign default
	dtest_assign_default(&info);
	while (i < argc) {
		if('-' == *argv[i]) {
			switch(*(argv[i]+1)) {
			case 'b': // test block or not
				if((i+1)>=argc) { goto INCOMP_CMD; }
				info.stop = simple_strtoul(argv[i+1], NULL, 10);
				i+=2;
				break;
			case 'c': // specific test case
				if((i+1)>=argc) { goto INCOMP_CMD; }
				info.sp_case = simple_strtoul(argv[i+1], NULL, 10);
				i+=2;
				break;
			case 'd': // debug enable
				if((i+1)>=argc) { goto INCOMP_CMD; }
				info.dbg = simple_strtoul(argv[i+1], NULL, 10);
				i+=2;
				break;
			case 'l': // loop count
				if((i+1)>=argc) { goto INCOMP_CMD; }
				info.loop = simple_strtoul(argv[i+1], NULL, 10);
				if(0==info.loop || -1<info.loop) {
					printf("Incorrect loop count = %d\n", info.loop); return 1;
				}
				i+=2;
				break;
			case 'm': // mask address and size
				if((i+2)>=argc) { goto INCOMP_CMD; }
				if (j >= MASK_ENT_MAX) {
					printf("Too many mask entries! (Max.= %d)\n", MASK_ENT_MAX); break;
				}
				info.maddr[j] = simple_strtoul(argv[i+1], NULL, 16);
				info.msize[j] = simple_strtoul(argv[i+2], NULL, 16);
				if (0 == info.msize[j]) {
					printf("Incorrect mask range 0x%x %d\n", info.maddr[j], info.msize[j]);
					return 1;
				}
				i += 3;
				j++;
				break;
			case 'r': // test range and size
				if((i+2)>=argc) { goto INCOMP_CMD; }
				info.taddr = simple_strtoul(argv[i+1], NULL, 16);
				info.tsize = simple_strtoul(argv[i+2], NULL, 16);
				if (0 == info.tsize) {
					printf("Incorrect test size = %d\n", info.tsize);
					return 1;
				}
				i+=3;
				break;
			case 's': // customized pass/fail message
				if ((i+2) >= argc) {
					goto INCOMP_CMD;
				}
				pstr = argv[i+1];
				fstr = argv[i+2];
				i += 3;
				//printf("II: pstr=%s, fstr=%s\n", pstr, fstr);
				break;
			case 'z': // dump test cases
				for(r=0; NULL!=(dtest_func_ptr=dtest[r].ptr); r++) {
					printf("  [%2d] - %s \n", r, dtest[r].name);
				}
				return 0;
				break;
			default:
				goto UNKNOWN_CMD;
				break;
			}
		} else {
			goto UNKNOWN_CMD;
		}
	}

	plat_backup = __builtin_alloca(PLAT_BACKUP_SIZE_B);
	plat_backup = plat_dram_space_serialize(plat_backup);

	dtest_process_segment(&info);
	dtest_process_cpu_info(&info);

	dtest_dump_info(&info);

	// dram test function
	for (l=1; l <= info.loop; l++) {
		if (l>1) printf("II: Loop %d:\n", l);
		for (r=0, c=0; NULL!=(dtest_func_ptr=dtest[r].ptr); r++) {
			if (info.sp_case != 0xBEEF) {
				if (r != info.sp_case) continue;
			}
			printf("  [%2d] - %s ... ", r, dtest[r].name);
			for (j=0; j<SEG_ENT_MAX; j++) {
				if (0 == (ssize = info.ssize[j])) break;
				//printf("\t SEG_%d, 0x%08x ~ 0x%08x ", j, info.seg[j], info.seg[j]+info.ssize[j]);
				start = info.seg[j];
				while (ssize>0) {
					if (ctrlc()) { putc ('\n'); goto NORMAL_OUT; }

					base = start & ~(info.tunit-1);
					if (ssize >= info.tunit) {
						bsize = base + info.tunit - start;
					} else {
						bsize = ssize;
					}

					printf("%c\b", ps[(c++>>3) & 0x3]);
					//printf("\t test from 0x%x ~ 0x%x\n", start, start+bsize);
					if ((*dtest_func_ptr)(&info, start, bsize)) {
						if (fstr) {
							puts(fstr);
						}
						goto NORMAL_OUT;
					}
					ssize -= bsize;
					start += bsize;
				}
			}
			puts("OK\n");
		}
	}

	puts(pstr);
NORMAL_OUT:
	putc('\n');
	plat_dram_space_restore(plat_backup);
	return 0;
INCOMP_CMD:
	printf("Incomlete ARG \"%s\"\n", argv[i]);
	return 10;
UNKNOWN_CMD:
	printf("Unknown ARG \"%s\"\n", argv[i]);
	return 11;
}

U_BOOT_CMD(
	wdtest,    20,    0,     do_wdtest,
	"do whole DRAM test",
	"-r <start physical addr.> <size>"
	"[-m <masked addr.> <size>] [-l <loops>] [-s <pass msg.> <fail msg.>] [-b <block>]\n"
	"\t i.e. dramtest -r 0 0x10000 -l 5\n"
	"\t r: test range from <start physical addr.> size is <size>\n"
	"\t m: ignore <masked addr.>, masked size is <size>\n"
	"\t l: test with loop <loops>\n"
	"\t s: customized pass/fail message <pass msg.> and <fail msg.>\n"
	"\t b: block test or not\n"
	"\t c: specific test case\n"
	"\t z: memory test cases list\n"
	);
