#include <linux/ctype.h>
#include <common.h>
#include "dramtest.h"

static const unsigned int dram_pat[] =    {
	0x01234567,     // comp => 0xFDECBA98
	0x76543210,     // comp => 0x89ABCDEF
	0x00000000,     // comp => 0xFFFFFFFF
	0x11111111,     // comp => 0xEEEEEEEE
	0x22222222,     // comp => 0xDDDDDDDD
	0x33333333,     // comp => 0xCCCCCCCC
	0x44444444,     // comp => 0xBBBBBBBB
	0x55555555,     // comp => 0xAAAAAAAA
	0x66666666,     // comp => 0x99999999
	0x77777777,     // comp => 0x88888888
	0xA5A5A5A5,     // comp => 0x5A5A5A5A
	0xF0F0F0F0,     // comp => 0x0F0F0F0F
	0xFF00FF00,     // comp => 0x00FF00FF
	0x0000FFFF,     // comp => 0xFFFF0000
	0x00FFFF00,     // comp => 0xFF0000FF
};

#define ADR_DATA_ERR(addr, pat) ({														\
			uint32_t __r;																						\
			printf("\nEE:(%s, %d)\n", __FUNCTION__, __LINE__);			\
			for(__r=1; __r<=3; __r++) {															\
				printf("\t0x%x(ADR)=0x%x(Bad)!=0x%x(Good)...(R%d)\n",	\
							 (uint32_t)(addr), *(addr), pat, __r);					\
			}																												\
			if (0!=info->stop) return 1;														\
		})

#define PATTERN_LEN (sizeof(dram_pat)/sizeof(uint32_t))

/*Cases dependent parameters*/
#define GET_SEED 1
#define SET_SEED 0
/*
  get_or_set = GET_SEED: get seed
  get_or_set = SET_SEED: set seed
*/
static void __srandom32(uint32_t *a1, uint32_t *a2, uint32_t *a3, uint32_t get_or_set) {
	static int s1, s2, s3;
	if (GET_SEED == get_or_set) {
		*a1 = s1;
		*a2 = s2;
		*a3 = s3;
	} else {
		s1 = *a1;
		s2 = *a2;
		s3 = *a3;
	}
	return;
}

static unsigned int __random32(void) {
#define TAUSWORTHE(s,a,b,c,d) ((s&c)<<d)^(((s<<a)^s)>>b)
	unsigned int s1, s2, s3;
	__srandom32(&s1, &s2, &s3, GET_SEED);

	s1 = TAUSWORTHE(s1, 13, 19, 4294967294UL, 12);
	s2 = TAUSWORTHE(s2, 2, 25, 4294967288UL, 4);
	s3 = TAUSWORTHE(s3, 3, 11, 4294967280UL, 17);

	__srandom32(&s1, &s2, &s3, SET_SEED);

	return (s1^s2^s3);
}

static int dtest_word_access(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=PATTERN_LEN;
	volatile uint32_t *p;

	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		v = dram_pat[i];

		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (uint32_t *)_e;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p<=_s) break;
			p--;
		}

		/* write compliment pattern from end*/
		v = ~v;
		p = (uint32_t *)_e;
		while(1) {
			*p = v;
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		p = (uint32_t *)_s;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

static int dtest_half_word_access(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=PATTERN_LEN;
	uint16_t v1, v2;
	volatile uint16_t *p;

	_s = start;
	_e = _s + size - 2;

	for(i=0; i<l; i++) {
		v = dram_pat[i];
		v1 = (uint16_t)v;
		v2 = (uint16_t)(v>>16);

		/* write pattern from start*/
		p = (uint16_t *)_s;
		while(1) {
			*p = v1;
			*(++p) = v2;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (uint16_t *)_e;
		while(1) {
			if(v2 != *p) { ADR_DATA_ERR(p, v1); }
			if(v1 != *(--p)) { ADR_DATA_ERR(p, v2); }
			if((uint32_t)p<=_s) break;
			p--;
		}

		/* write compliment pattern from end*/
		v = ~v;
		v1 = (uint16_t)v;
		v2 = (uint16_t)(v>>16);

		p = (uint16_t *)_e;
		while(1) {
			*p = v1;
			*(--p) = v2;
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		p = (uint16_t *)_s;
		while(1) {
			if(v2 != *p) {  ADR_DATA_ERR(p, v1); }
			if(v1 != *(++p)) {  ADR_DATA_ERR(p, v2); }
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

static int dtest_byte_access(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=PATTERN_LEN;
	uint8_t v1, v2, v3, v4;
	volatile uint8_t *p;

	_s = start;
	_e = _s + size - 1;

	for(i=0; i<l; i++) {
		v = dram_pat[i];
		v1 = (uint8_t)v;
		v2 = (uint8_t)(v>>8);
		v3 = (uint8_t)(v>>16);
		v4 = (uint8_t)(v>>24);

		/* write pattern from start*/
		p = (uint8_t *)_s;
		while(1) {
			*(p++) = v1;
			*(p++) = v2;
			*(p++) = v3;
			*(p)   = v4;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (uint8_t *)_e;
		while(1) {
			if(v4 != *p)     { ADR_DATA_ERR(p, v4); }
			if(v3 != *(--p)) { ADR_DATA_ERR(p, v3); }
			if(v2 != *(--p)) { ADR_DATA_ERR(p, v2); }
			if(v1 != *(--p)) { ADR_DATA_ERR(p, v1); }
			if((uint32_t)p<=_s) break;
			p--;
		}

		/* write compliment pattern from end*/
		v = ~v;
		v1 = (uint8_t)v;
		v2 = (uint8_t)(v>>8);
		v3 = (uint8_t)(v>>16);
		v4 = (uint8_t)(v>>24);

		p = (uint8_t *)_e;
		while(1) {
			*(p--) = v1;
			*(p--) = v2;
			*(p--) = v3;
			*(p)   = v4;
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		p = (uint8_t *)_s;
		while(1) {
			if(v4 != *p)     { ADR_DATA_ERR(p, v4); }
			if(v3 != *(++p)) { ADR_DATA_ERR(p, v3); }
			if(v2 != *(++p)) { ADR_DATA_ERR(p, v2); }
			if(v1 != *(++p)) { ADR_DATA_ERR(p, v1); }
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

/* this test case is reference cache_flush_dispersed*/
static int dtest_cache_line_access(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v1, v2, _s, _e, l=PATTERN_LEN;
	volatile uint32_t *p;

	uint32_t a=0x13243, b=0xaaa0bdd, c=0xfffbda0;
	__srandom32(&a, &b, &c, SET_SEED);

	_s = start;
	_e = _s + size;

	for(i=0; i<l; i++) {
		v1 = __random32();
		v2 = __random32();

		dcache_flush_range(_s, _e);

		plat_dram_space_mod_ca(info, CCAE_UC_NCO);
		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			*p++ = v1;
			*p++ = v1;
			*p++ = v2;
			*p++ = v2;
			if((uint32_t)p>=_e) break;
		}

		plat_dram_space_mod_ca(info, CCAE_CA_NCO_WB_WA);
		/* check data from end*/
		p = (uint32_t *)_e;
		while(1) {
			if(v2 != *--p) { ADR_DATA_ERR(p, v2); }
			if(v2 != *--p) { ADR_DATA_ERR(p, v2); }
			if(v1 != *--p) { ADR_DATA_ERR(p, v1); }
			if(v1 != *--p) { ADR_DATA_ERR(p, v1); }
			if((uint32_t)p<=_s) break;
		}

		/* write compliment pattern from end*/
		dcache_flush_range(_s, _e);
		plat_dram_space_mod_ca(info, CCAE_UC_NCO);
		v1 = ~v1;
		v2 = ~v2;
		p = (uint32_t *)_e;
		while(1) {
			*--p = v2;
			*--p = v2;
			*--p = v1;
			*--p = v1;
			if((uint32_t)p<=_s) break;
		}

		/* check data from start */
		plat_dram_space_mod_ca(info, CCAE_CA_NCO_WB_WA);
		p = (uint32_t *)_s;
		while(1) {
			if(v1 != *p)   { ADR_DATA_ERR(p, v1); }
			if(v1 != *++p) { ADR_DATA_ERR(p, v1); }
			if(v2 != *++p) { ADR_DATA_ERR(p, v2); }
			if(v2 != *++p) { ADR_DATA_ERR(p, v2); }
			if((uint32_t)++p>=_e) break;
		}
	}
	return 0;
}

static int dtest_walking_of_0_and_1(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=32;
	volatile uint32_t *p;

	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		v = 0x1 << i;

		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (uint32_t *)_e;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p<=_s) break;
			p--;
		}

		/* write compliment pattern from end*/
		v = ~v;
		p = (uint32_t *)_e;
		while(1) {
			*p = v;
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		p = (uint32_t *)_s;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

#define ROTATE_N(v, n)  (((v)>>n) | ((v)&((1<<n)-1))<<(31-n))
static int dtest_address_rotate(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=32;
	volatile uint32_t *p;

	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			v = ROTATE_N((u32)p, i);
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (uint32_t *)_e;
		while(1) {
			v = ROTATE_N((u32)p, i);
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p<=_s) break;
			p--;
		}

		/* write compliment pattern from end*/
		p = (uint32_t *)_e;
		while(1) {
			v = ~ROTATE_N((u32)p, i);
			*p = v;
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		p = (uint32_t *)_s;
		while(1) {
			v = ~ROTATE_N((u32)p, i);
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

#pragma pack(1)
/* Data structures used for testing unaligned load/store operations. */
typedef struct{
	uint8_t  c1;
	uint32_t  w;
} off_1_t;

typedef struct{
	uint8_t  c1;
	uint8_t  c2;
	uint32_t w;
} off_2_t;
typedef struct{
	uint8_t  c1;
	uint8_t  c2;
	uint8_t  c3;
	uint32_t w;
} off_3_t;
typedef struct{
	uint8_t  c;
	uint16_t s;
	uint32_t w;
} off_4_t;
#pragma pack(4)

static int dtest_unalg_test_off1(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t _s, _e, l=PATTERN_LEN;
	uint32_t off = sizeof(off_1_t);
	uint32_t w;
	uint8_t c1;
	volatile off_1_t *p;

	_s = start;
	_e = _s + size;

	for(i=0; i<l; i++) {
		c1 = CPU_GET_CP0_CYCLE_COUNT()&0xFF;
		w = dram_pat[i];

		/* write pattern from start*/
		p = (off_1_t *)_s;
		while(1) {
			p->c1 = c1;
			p->w = w;
			p++;
			if((uint32_t)p>(_e-off)) break;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (off_1_t *)_s;
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			p++;
			if((uint32_t)p>(_e-off)) break;
		}

		/* write compliment pattern from end*/
		w = ~w;
		p = (off_1_t *)(_e-off);
		while(1) {
			p->c1 = c1;
			p->w  = w;
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from end */
		p = (off_1_t *)(_e-off);
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
	}
	return 0;
}

static int dtest_unalg_test_off2(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t _s, _e, l=PATTERN_LEN;
	uint32_t off = sizeof(off_2_t);
	uint32_t w;
	uint8_t c1, c2;
	volatile off_2_t *p;

	_s = start;
	_e = _s + size;

	for(i=0; i<l; i++) {
		w = dram_pat[i];
		c1 = CPU_GET_CP0_CYCLE_COUNT()&0xFF;
		c2 = (CPU_GET_CP0_CYCLE_COUNT()>>8)&0xFF;

		/* write pattern from start*/
		p = (off_2_t *)_s;
		while(1) {
			p->c1 = c1;
			p->c2 = c2;
			p->w = w;
			p++;
			if((uint32_t)p>(_e-off)) break;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (off_2_t *)_s;
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(c2!=p->c2) { ADR_DATA_ERR(&(p->c2), c2); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			p++;
			if((uint32_t)p>(_e-off)) break;
		}

		/* write compliment pattern from end*/
		w = ~w;
		p = (off_2_t *)(_e-off);
		while(1) {
			p->c1 = c1;
			p->c2 = c2;
			p->w  = w;
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from end */
		p = (off_2_t *)(_e-off);
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(c2!=p->c2) { ADR_DATA_ERR(&(p->c2), c2); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
	}
	return 0;
}

static int dtest_unalg_test_off3(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t _s, _e, l=PATTERN_LEN;
	uint32_t off = sizeof(off_3_t);
	uint32_t w;
	uint8_t c1, c2, c3;
	volatile off_3_t *p;

	_s = start;
	_e = _s + size;

	for(i=0; i<l; i++) {
		w = dram_pat[i];
		c1 = CPU_GET_CP0_CYCLE_COUNT()&0xFF;
		c2 = (CPU_GET_CP0_CYCLE_COUNT()>>8)&0xFF;
		c3 = (CPU_GET_CP0_CYCLE_COUNT()>>16)&0xFF;

		/* write pattern from start*/
		p = (off_3_t *)_s;
		while(1) {
			p->c1 = c1;
			p->c2 = c2;
			p->c3 = c3;
			p->w  = w;
			p++;
			if((uint32_t)p>(_e-off)) break;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (off_3_t *)_s;
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(c2!=p->c2) { ADR_DATA_ERR(&(p->c2), c2); }
			if(c3!=p->c3) { ADR_DATA_ERR(&(p->c3), c3); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			p++;
			if((uint32_t)p>(_e-off)) break;
		}

		/* write compliment pattern from end*/
		w = ~w;
		c1 = ~c1;
		c2 = ~c2;
		c3 = ~c3;
		p = (off_3_t *)(_e-off);
		while(1) {
			p->c1 = c1;
			p->c2 = c2;
			p->c3 = c3;
			p->w  = w;
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from end */
		p = (off_3_t *)(_e-off);
		while(1) {
			if(c1!=p->c1) { ADR_DATA_ERR(&(p->c1), c1); }
			if(c2!=p->c2) { ADR_DATA_ERR(&(p->c2), c2); }
			if(c3!=p->c3) { ADR_DATA_ERR(&(p->c3), c3); }
			if(w !=p->w ) { ADR_DATA_ERR(&(p->w), w); }
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
	}
	return 0;
}

static int dtest_unalg_test_off4(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t _s, _e, l=PATTERN_LEN;
	uint32_t off = sizeof(off_4_t);
	uint8_t c;
	uint16_t s;
	uint32_t w;
	volatile off_4_t *p;

	_s = start;
	_e = _s + size;

	for(i=0; i<l; i++) {
		w = dram_pat[i];
		c = CPU_GET_CP0_CYCLE_COUNT()&0xFF;
		s = (CPU_GET_CP0_CYCLE_COUNT()>>8)&0xFFFF;

		/* write pattern from start*/
		p = (off_4_t *)_s;
		while(1) {
			p->w = w;
			p->s = s;
			p->c = c;
			p++;
			if((uint32_t)p>(_e-off)) break;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		p = (off_4_t *)_s;
		while(1) {
			if(w!=p->w) { ADR_DATA_ERR(&(p->w), w); }
			if(s!=p->s) { ADR_DATA_ERR(&(p->s), s); }
			if(c!=p->c) { ADR_DATA_ERR(&(p->c), c); }
			p++;
			if((uint32_t)p>(_e-off)) break;
		}

		/* write compliment pattern from end*/
		w = ~w;
		s = ~s;
		c = ~c;
		p = (off_4_t *)(_e-off);
		while(1) {
			p->w = w;
			p->s = s;
			p->c = c;
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from end */
		p = (off_4_t *)(_e-off);
		while(1) {
			if(w!=p->w) { ADR_DATA_ERR(&(p->w), w); }
			if(s!=p->s) { ADR_DATA_ERR(&(p->s), s); }
			if(c!=p->c) { ADR_DATA_ERR(&(p->c), c); }
			if((uint32_t)p<(_s+off)) break;
			p--;
		}
	}
	return 0;
}

#define GEN_SEED(v, a, b, c)    ({  *a = (v>>8)&0xFFFF;									\
			*b = (v>>16)&(v&0xFFFF)<<16;																			\
			*c = ((v&0xFF)<<24) | (v>>24) | (((v>>16)&0xFF)<<8) | (((v>>8)&0xFF)<<16); })

static int dtest_random_pattern(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=10;
	volatile uint32_t *p;
	uint32_t a, b, c;
	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		v = CPU_GET_CP0_CYCLE_COUNT();
		GEN_SEED(v, &a, &b, &c);
		/* write pattern from start*/
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_s;
		while(1) {
			v = __random32();
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data from start */
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_s;
		while(1) {
			v = __random32();
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* write compliment pattern from end*/
		v = ~v;
		GEN_SEED(v, &a, &b, &c);
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_e;
		while(1) {
			*p = __random32();
			if((uint32_t)p<=_s) break;
			p--;
		}
		dcache_flush_range(_s, _e);

		/* check data from end*/
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_e;
		while(1) {
			v = __random32();
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p<=_s) break;
			p--;
		}
	}
	return 0;
}

static int dtest_moving_inversions(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=10;
	volatile uint32_t *p;
	uint32_t a, b, c;
	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		v = CPU_GET_CP0_CYCLE_COUNT();
		GEN_SEED(v, &a, &b, &c);
		/* write pattern from start*/
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_s;
		while(1) {
			v = __random32();
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}
		dcache_flush_range(_s, _e);

		/* check data and write the complement*/
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_s;
		while(1) {
			v = __random32();
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			*p = ~v;
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* check the complement data*/
		__srandom32(&a, &b, &c, SET_SEED);
		p = (uint32_t *)_s;
		while(1) {
			v = __random32();
			if(~v != *p) {
				ADR_DATA_ERR(p, ~v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

static int dtest_moving_inversions_pat(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i;
	uint32_t v, _s, _e, l=PATTERN_LEN;
	volatile uint32_t *p;
	_s = start;
	_e = _s + size - 4;

	for(i=0; i<l; i++) {
		v = dram_pat[i];
		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* check data and write the complement*/
		p = (uint32_t *)_s;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			*p = ~v;
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* check data and write original data*/
		p = (uint32_t *)_s;
		while(1) {
			if(~v != *p) {
				ADR_DATA_ERR(p, ~v);
			}
			*p = v;
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* check data and write original data*/
		p = (uint32_t *)_s;
		while(1) {
			if(v != *p) {
				ADR_DATA_ERR(p, v);
			}
			if((uint32_t)p>=_e) break;
			p++;
		}
	}
	return 0;
}

static int dtest_moving_inversions_32(dramtest_info_t *info, uint32_t start, uint32_t size) {
	uint32_t _s, _e;
	uint32_t i, k, v, p1, p3, lb, hb, sval;
	volatile uint32_t *p;
	_s = start;
	_e = _s + size - 4;

	for(i=0, v=1; v; v=v<<1, i++) {
		p1 = v;
		lb = 1;
		hb = 0x80000000;
		sval = 0;
		p3 = sval << 31;
		k = i;

		/* write pattern from start*/
		p = (uint32_t *)_s;
		while(1) {
			*p = p1;
			if(++k > 32) {
				p1 = lb;
				k = 0;
			} else {
				p1 = p1 << 1;
				p1 |= sval;
			}
			if((uint32_t)p>=_e) break;
			p++;
		}

		/* check data from top*/
		p1 = v;
		k = i;
		p = (uint32_t *)_s;
		while(1) {
			if(p1 != *p) {
				ADR_DATA_ERR(p, p1);
			}
			*p = ~p1;
			if((uint32_t)p>=_e) break;
			p++;

			if(++k > 32) {
				p1 = lb;
				k = 0;
			} else {
				p1 = p1 << 1;
				p1 |= sval;
			}
		}

		/*  uint32_t n;
        if (--k < 0) {
				k = 31;
        }
        for (p1 = lb, n = 0; n < k; n++) {
				p1 = p1 << 1;
				p1 |= sval;
        }
        k++; */

		p = (uint32_t *)_e;
		while(1) {
			if(~p1 != *p) {
				ADR_DATA_ERR(p, ~p1);
			}
			*p = p1;
			if((uint32_t)p>=_e) break;
			p++;

			if(--k <= 0) {
				p1 = hb;
				k = 32;
			} else {
				p1 = p1 >> 1;
				p1 |= p3;
			}
		}
	}
	return 0;
}

#define MOD_SZ  (23)
static int dtest_modulo_x(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i, j;
	uint32_t p1, p2, k, _s, _e;
	uint32_t a, b, c, off;
	volatile uint32_t *p;
	_s = start;
	_e = _s + size - 4;

	for(i=0; i<MOD_SZ; i++) {
		p1 = CPU_GET_CP0_CYCLE_COUNT();
		GEN_SEED(p1, &a, &b, &c);
		/* write pattern from start*/
		__srandom32(&a, &b, &c, SET_SEED);

		off = i;
		p1 = __random32();
		p2 = ~p1;
		/* write pattern with offset MOD_SZ */
		p = (uint32_t *)_s + off;
		while(1) {
			*p = p1;
			p+=MOD_SZ;
			if((uint32_t)p>=_e) break;
		}

		/* write rest j times with complement*/
		for(j=0; j<10; j++) {
			p = (uint32_t *)_s;
			k = 0;
			while(1) {
				if(k != off) {
					*p = p2;
				}
				if(++k > MOD_SZ-1) {
					k = 0;
				}
				if((uint32_t)p>=_e) break;
				p++;
			}
		}

		p = (uint32_t *)_s + off;
		while(1) {
			if(p1 != *p) {
				ADR_DATA_ERR(p, p1);
			}
			p+=MOD_SZ;
			if((uint32_t)p>=_e) break;
		}
	}
	return 0;
}

#define MOD_SZ  (23)
static int dtest_modulo_x_hword(dramtest_info_t *info, uint32_t start, uint32_t size) {
	int i, j;
	uint32_t k, _s, _e;
	uint16_t p1, p2;
	uint32_t a, b, c, off;
	volatile uint16_t *p;
	_s = start;
	_e = _s + size - 4;

	for(i=0; i<MOD_SZ; i++) {
		p1 = CPU_GET_CP0_CYCLE_COUNT();
		GEN_SEED(p1, &a, &b, &c);
		/* write pattern from start*/
		__srandom32(&a, &b, &c, SET_SEED);

		off = i;
		p1 = __random32();
		p2 = ~p1;
		/* write pattern with offset MOD_SZ */
		p = (uint16_t *)_s + off;
		while(1) {
			*p = p1;
			p+=MOD_SZ;
			if((uint32_t)p>=_e) break;
		}

		/* write rest j times with complement*/
		for(j=0; j<10; j++) {
			p = (uint16_t *)_s;
			k = 0;
			while(1) {
				if(k != off) {
					*p = p2;
				}
				if(++k > MOD_SZ-1) {
					k = 0;
				}
				if((uint32_t)p>=_e) break;
				p++;
			}
		}

		p = (uint16_t *)_s + off;
		while(1) {
			if(p1 != *p) {
				ADR_DATA_ERR(p, p1);
			}
			p+=MOD_SZ;
			if((uint32_t)p>=_e) break;
		}
	}
	return 0;
}

#define BLK_SIZE    (64)
static int dtest_block_move(dramtest_info_t *info, uint32_t start, uint32_t size)
{
	uint32_t _s, _e;
	uint32_t p1, p2;
	uint32_t bsize = BLK_SIZE/4;

	volatile uint32_t *p, *mp, *ep;
	_s = start;
	_e = _s + size - BLK_SIZE;

	/* write pattern from start with unit is block*/
	p1 = 1;
	while(1) {
		p2 = ~p1;
		p = (uint32_t *)_s;
		while((uint32_t)p <= _e) {
			__asm__ __volatile__ ( \
				"sw %1,  0(%0)"  ASM_NL \
				"sw %1,  4(%0)"  ASM_NL \
				"sw %1,  8(%0)"  ASM_NL\
				"sw %1, 12(%0)"  ASM_NL\
				"sw %2, 16(%0)"  ASM_NL\
				"sw %2, 20(%0)"  ASM_NL\
				"sw %1, 24(%0)"  ASM_NL\
				"sw %1, 28(%0)"  ASM_NL\
				"sw %1, 32(%0)"  ASM_NL\
				"sw %1, 36(%0)"  ASM_NL\
				"sw %2, 40(%0)"  ASM_NL\
				"sw %2, 44(%0)"  ASM_NL\
				"sw %1, 48(%0)"  ASM_NL\
				"sw %1, 52(%0)"  ASM_NL\
				"sw %2, 56(%0)"  ASM_NL\
				"sw %2, 60(%0)"  ASM_NL\
				: /* output */ \
				: "r" (p), "r" (p1), "r" (p2)\
				);

			p+=bsize;
		}

		/* move the block */
		p = (uint32_t *)_s;
		while(((uint32_t)p+(BLK_SIZE/2))<= _e) {
			mp = p + (bsize)/2;
			ep = p + bsize;

			__asm__ __volatile__ ( \
				"lw $8,  0(%1)"  ASM_NL \
				"sw $8,  0(%2)"  ASM_NL \
				"lw $8,  4(%1)"  ASM_NL \
				"sw $8,  4(%2)"  ASM_NL \
				"lw $8,  8(%1)"  ASM_NL \
				"sw $8,  8(%2)"  ASM_NL \
				"lw $8, 12(%1)"  ASM_NL \
				"sw $8, 12(%2)"  ASM_NL \
				"lw $8, 16(%1)"  ASM_NL \
				"sw $8, 16(%2)"  ASM_NL \
				"lw $8, 20(%1)"  ASM_NL \
				"sw $8, 20(%2)"  ASM_NL \
				"lw $8, 24(%1)"  ASM_NL \
				"sw $8, 24(%2)"  ASM_NL \
				"lw $8, 28(%1)"  ASM_NL \
				"sw $8, 28(%2)"  ASM_NL \
				"lw $8,  0(%0)"  ASM_NL \
				"sw $8,  0(%1)"  ASM_NL \
				"lw $8,  4(%0)"  ASM_NL \
				"sw $8,  4(%1)"  ASM_NL \
				"lw $8,  8(%0)"  ASM_NL \
				"sw $8,  8(%1)"  ASM_NL \
				"lw $8, 12(%0)"  ASM_NL \
				"sw $8, 12(%1)"  ASM_NL \
				"lw $8, 16(%0)"  ASM_NL \
				"sw $8, 16(%1)"  ASM_NL \
				"lw $8, 20(%0)"  ASM_NL \
				"sw $8, 20(%1)"  ASM_NL \
				"lw $8, 24(%0)"  ASM_NL \
				"sw $8, 24(%1)"  ASM_NL \
				"lw $8, 28(%0)"  ASM_NL \
				"sw $8, 28(%1)"  ASM_NL \
				: /* output */ \
				: "r" (p), "r" (mp), "r" (ep)\
				);

			p += (bsize)/2;
		}

		/* check the adjacent DWORDS */
		p = (uint32_t *)_s;
		while(1) {
			if(*p != *(p+1)) {
				ADR_DATA_ERR(p, *(p+1));
				ADR_DATA_ERR((p+1), *p);
			}
			p+=2;
			if((uint32_t)p>=_e) break;
		}
		if(p1==(1<<31)) break;
		p1 = p1 << 1;
	}
	return 0;
}

__attribute__ ((weak))
dtest_func_t dtest[] = {
	DTEST_FUNC(dtest_word_access),
	DTEST_FUNC(dtest_half_word_access),
	DTEST_FUNC(dtest_byte_access),
	DTEST_FUNC(dtest_cache_line_access),
	DTEST_FUNC(dtest_walking_of_0_and_1),
	DTEST_FUNC(dtest_address_rotate),
	DTEST_FUNC(dtest_random_pattern),
	DTEST_FUNC(dtest_unalg_test_off1),
	DTEST_FUNC(dtest_unalg_test_off2),
	DTEST_FUNC(dtest_unalg_test_off3),
	DTEST_FUNC(dtest_unalg_test_off4),
	DTEST_FUNC(dtest_moving_inversions),
	DTEST_FUNC(dtest_moving_inversions_pat),
	DTEST_FUNC(dtest_moving_inversions_32),
	DTEST_FUNC(dtest_modulo_x),
	DTEST_FUNC(dtest_modulo_x_hword),
	DTEST_FUNC(dtest_block_move),
	DTEST_FUNC(NULL),
};
