#include <common.h>
#include <command.h>
#include <cpu.h>
#include <cross_env.h>

#if defined(CACHE_TYPE_I)
#	if defined(MIPS_CONFIG1_ICACHE_OFF)
#		undef MIPS_CONFIG1_ICACHE_OFF
#	endif
#	define MIPS_CONFIG1_ICACHE_OFF CACHE_TYPE_I
#endif

#if defined(CACHE_TYPE_D)
#	if defined(MIPS_CONFIG1_DCACHE_OFF)
#		undef MIPS_CONFIG1_DCACHE_OFF
#	endif
#	define MIPS_CONFIG1_DCACHE_OFF CACHE_TYPE_D
#endif

int do_reset_sw(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	void (*f)(void) = (void *)0xbfc00000;
	f();
	fprintf(stderr, "*** system s/w reset failed ***\n");
	return 0;
}

U_BOOT_CMD(
	reset_sw, 1, 0,    do_reset_sw,
	"Perform software RESET",
	""
	);

extern int do_reset_all(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]);

int do_reset(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	return do_reset_all(cmdtp, flag, argc, argv);
}

void flush_cache(ulong start_addr, ulong size) {
	writeback_invalidate_dcache_all();
	invalidate_icache_all();
}

static int otto_cpu_id(void) {
	return asm_mfc0(CP0_PRID);
}

static int otto_cpu_is_rlx5x81(void) {
	return ((otto_cpu_id()>>14) & 0x1);
}

static char *otto_cpu_name_string[] = {
	[0] = "MIPS interAptiv UP",
	[1] = "MIPS interAptiv",
	"N/A",
	"N/A",
	[4] = "MIPS 34Kc",
	[5] = "RLX 5281",
	"N/A",
	[7] = "RLX 5181",
	"N/A",
	[9] = "MIPS 4Kec"
};

char* otto_cpu_name(void) {
	/* hash(prid) = (((id>>8)&0xf) + ((id>>12)&0xf)) % 10;
		5281: 0x0???_dc02 => hash() => 5
		5181: 0x0???_cf01 => hash() => 7
		4KEc: 0x??01_90?? => hash() => 9
		34Kc: 0x??01_95?? => hash() => 4
		IA:   0x??01_a1?? => hash() => 1
		IAUP: 0x??01_a0?? => hash() => 0
	 */
	uint32_t cpu_hash = otto_cpu_id();
	cpu_hash = (((cpu_hash>>8)&0xf) + ((cpu_hash>>12)&0xf)) % 10;
	return otto_cpu_name_string[cpu_hash];
}

int otto_cache_line_size(int cache_type) {
	int cache_line_size_b;

	if (otto_cpu_is_rlx5x81()) {
		cache_line_size_b = CONFIG_SYS_CACHELINE_SIZE;
	} else {
		cache_line_size_b = __asm_mfc0(CP0_CONFIG, 1) >> (cache_type + 3);
		cache_line_size_b &= 0x7;
		cache_line_size_b = 1 << (cache_line_size_b + 1);
	}

	return cache_line_size_b;
}

static int otto_mips_cache_way_num(int cache_type) {
	int cache_way_num;

	cache_way_num = __asm_mfc0(CP0_CONFIG, 1) >> (cache_type + 0);
	cache_way_num &= 0x7;
	cache_way_num += 1;

	return cache_way_num;
}

static int otto_mips_cache_set_num(int cache_type) {
	int cache_set_num;

	cache_set_num = __asm_mfc0(CP0_CONFIG, 1) >> (cache_type + 6);
	cache_set_num &= 0x7;
	cache_set_num = 1 << (cache_set_num + 6);

	return cache_set_num;
}

void otto_cache_dump_info(void) {
	int way, set, ls;
	if (otto_cpu_is_rlx5x81()) {
		printf("ICACHE: %dKB (%d-byte)\n", CFG_ICACHE_SIZE/1024, CONFIG_SYS_CACHELINE_SIZE);
		printf("DCACHE: %dKB (%d-byte)\n", CFG_DCACHE_SIZE/1024, CONFIG_SYS_CACHELINE_SIZE);
	} else {
		way = otto_mips_cache_way_num(MIPS_CONFIG1_ICACHE_OFF);
		set = otto_mips_cache_set_num(MIPS_CONFIG1_ICACHE_OFF);
		ls = otto_cache_line_size(MIPS_CONFIG1_ICACHE_OFF);
		printf("ICACHE: %dKB (%d-byte/%d-way/%d-set)\n",
		       way*set*ls/1024,
		       ls, way, set);
		way = otto_mips_cache_way_num(MIPS_CONFIG1_DCACHE_OFF);
		set = otto_mips_cache_set_num(MIPS_CONFIG1_DCACHE_OFF);
		ls = otto_cache_line_size(MIPS_CONFIG1_DCACHE_OFF);
		printf("DCACHE: %dKB (%d-byte/%d-way/%d-set)\n",
		       way*set*ls/1024,
		       ls, way, set);
	}
	return;
}
