#include <common.h>
#include <soc.h>
#include <asm/symb_define.h>
#include <asm/otto_util.h>
#include <asm/otto_onfi_flash.h>
#include <onfi/onfi_uboot_util.h>

#define MAX_BLOCKS (1024<<3)


#ifndef USE_BBT_SKIP
    #define USE_BBT_SKIP 1
#endif

#if USE_BBT_SKIP
uint8_t bb_skip_table[MAX_BLOCKS];
#else
u32_t _bbt_table[NUM_WORD(MAX_BLOCKS)];
#endif


#define EBUF_SIZE                   (MAX_ECC_BUF_SIZE)
#define PBUF_SIZE                   (MAX_PAGE_BUF_SIZE+MAX_OOB_BUF_SIZE)

u8_t _ecc_buf[EBUF_SIZE] DATA_ALIGNED32;
u8_t _pge_buf[PBUF_SIZE] DATA_ALIGNED32;

_onfi_info_t _ub_onfi_info;


u32_t onfi_chip_size(void)
{
    return _ub_onfi_info.chip_size;
}

u32_t onfi_block_size(void)
{
    return _ub_onfi_info.block_size;
}

u32_t onfi_page_size(void)
{
    return _ub_onfi_info.page_size;
}

u32_t onfi_spare_size(void)
{
    return _ub_onfi_info.spare_size;
}

/* inline function */
inline static void
_set_flags(u32_t *arr, u32_t i)
{
    u32_t idx=i/(8*sizeof(u32_t));
    i &= (8*sizeof(u32_t))-1;
    arr[idx] |= 1UL << i;
}

inline static int
_get_flags(u32_t *arr, u32_t i)
{
    u32_t idx=i/(8*sizeof(u32_t));
    i &= (8*sizeof(u32_t))-1;
    return (arr[idx] & (1UL << i)) != 0;
}

static void create_bbt(u32_t *bbt_table)
{
    u8_t bb_tag[4];
    u32_t blk_num;
    u32_t block_count = _ub_onfi_info.block_count;
    u8_t is_1st_err=0, prt_cnt = 0;
    int ret;

    for( blk_num = 1; blk_num < block_count; blk_num++){
        ofu_pio_read(bb_tag, 4, blk_num<<6, _ub_onfi_info.page_size);
        if(0xFF == bb_tag[0]){
            continue;
        }
        ret=ofu_page_read_ecc(_pge_buf, blk_num<<6, _ecc_buf);
        if (ECC_CTRL_ERR==(ret&0xFFFFFF00)) {
            // ecc error, mark it as a bad block
        } else {
            if( 0xFF == _pge_buf[_ub_onfi_info.page_size]) {
                continue;
            }
        }


        if(!is_1st_err){
            printf("ONFI BBT:");
            is_1st_err = 1;
        }
        printf("%4d  ", blk_num);
        if(((++prt_cnt%8)==0)&&(!is_1st_err)){
            puts("\n         ");
        }

        _set_flags(bbt_table, blk_num);
    }
    puts("\n");
#if !USE_BBT_SKIP
    puts("Don't use bad block skip table!\n");
#endif
}

#if USE_BBT_SKIP
static void create_skip_table(u32_t *bbt_table){
    u32_t good_num; 
    int j;
    u32_t bbi_block;
    u32_t skip_block;
    u8_t *skip_table = bb_skip_table;

    skip_table[0] = 0;
    skip_block = good_num = bbi_block = 1;
    for (;bbi_block<_ub_onfi_info.block_count;bbi_block++) {
        j=_get_flags(bbt_table, bbi_block);

        if(j==0){ //good block

            skip_table[skip_block]=bbi_block-good_num;
            good_num++;	
            skip_block++;			
        }else{
            //printf("detect block %d is bad \n",bbi_block);
            if(bbi_block-good_num == 0xFF){
                break;
            }
        }
    }
    for (;skip_block<_ub_onfi_info.block_count;skip_block++){
        skip_table[skip_block]=0xff;		
    }	
}
#endif

static int get_good_block(u32_t blk_num, u32_t *mapped_blk_num)
{
#if USE_BBT_SKIP
    u32_t skip_num = bb_skip_table[blk_num];
    if( skip_num == 0xFF){
        return -1;
    }
    *mapped_blk_num = blk_num + skip_num;
#else
    *mapped_blk_num = blk_num;
#endif
    //printf("get good block %d => %d\n", blk_num, *mapped_blk_num);
    return 0;
}

#if !USE_BBT_SKIP
static int go_to_good_block(u32_t *blk_num)
{
    int j;
    for (;*blk_num<_ub_onfi_info.block_count;(*blk_num)++) {
        j = _get_flags(_bbt_table, *blk_num);
        if ( j == 0 ) { //good block
            return 0;
        }
        printf("Skip Bad Block %d\n", *blk_num);
    }

    return -1;
}
#endif

static int32_t
is_blank_data(uint32_t *buf, int32_t data_len)
{
    while (data_len--) {
        if (*buf != 0xFFFFFFFF) {
            return 0;
        }
        buf++;
    }
    return 1;
}

#if defined(CONFIG_MTD_DEVICE) && defined(CONFIG_MTD_PARTITIONS)
#ifdef CONFIG_ONFI_FLASH
extern int rtk_onfi_nand_init (void);
struct onfi_info_s *get_onfi_info(void)
{
	return (struct onfi_info_s *)_plr_onfi_info;
}
#endif
#endif
void onfi_init(void)
{
#if USE_BBT_SKIP
    u32_t _bbt_table[NUM_WORD(MAX_BLOCKS)];
#endif
    onfi_info_t *fi;
    u32_t max=1;
  
    /* fill onfi info */
    fi = _plr_onfi_info;

    _ub_onfi_info.id=fi->id_code;
    _ub_onfi_info.cs_count=max;
    _ub_onfi_info.chip_size=ONFI_NUM_OF_BLOCK(fi)
                   *ONFI_NUM_OF_PAGE_PER_BLK(fi)
                   *ONFI_PAGE_SIZE(fi);
    _ub_onfi_info.block_size=ONFI_NUM_OF_PAGE_PER_BLK(fi)
                   *ONFI_PAGE_SIZE(fi);
    _ub_onfi_info.page_size=ONFI_PAGE_SIZE(fi);
    _ub_onfi_info.oob_size=ONFI_OOB_SIZE(fi);
    _ub_onfi_info.spare_size=ONFI_SPARE_SIZE(fi);
    _ub_onfi_info.block_count=ONFI_NUM_OF_BLOCK(fi);
    _ub_onfi_info.page_count=ONFI_NUM_OF_PAGE_PER_BLK(fi);

    
    printf("ONFI: %X, %uMB*%d\n", fi->id_code, onfi_chip_size()>>20, max);
    memset(_bbt_table, '\0', sizeof(_bbt_table));
    if(_ub_onfi_info.block_count > MAX_BLOCKS){
        printf("ONFI Flash: Create Bad Block Table Error!\n Block Number is More than %d!\n",  MAX_BLOCKS);
    }else{
        create_bbt(_bbt_table);
#if USE_BBT_SKIP
        create_skip_table(_bbt_table);
#endif
#if defined(CONFIG_MTD_DEVICE) && defined(CONFIG_MTD_PARTITIONS)
#ifdef CONFIG_ONFI_FLASH
        rtk_onfi_nand_init ();
#endif
#endif
    }
}

void onfi_info(void) {
    puts("Device:\n");
    printf(" ONFI_ID  = %x\n",_ub_onfi_info.id);
    printf(" Capacity = %uMB *%d \n", onfi_chip_size()>>20, _ub_onfi_info.cs_count);
    printf("   each chip  has %8d block(s)\n", _ub_onfi_info.block_count);
    printf("   each block has %8d page(s)\n", _ub_onfi_info.page_count);
    printf("    page size     %8d byte(s)\n", _ub_onfi_info.page_size);
    printf("     oob size     %8d byte(s)\n", _ub_onfi_info.oob_size);
    printf("   spare size     %8d byte(s)\n", _ub_onfi_info.spare_size);
    printf("   erase size     %8d byte(s)\n", _ub_onfi_info.block_size);
    printf("       others     %8d byte(s)\n", _ub_onfi_info.spare_size-_ub_onfi_info.oob_size);
}

#define BLK_IDX(off, bsize)         ((off)/(bsize))
#define PAG_IDX(off, bsize, psize)  (((off)%(bsize))/(psize))
#define COL_OFF(off, psize)         ((off)%(psize))

#define BSIZE                       (_ub_onfi_info.block_size)
#define PSIZE                       (_ub_onfi_info.page_size)
#define SSIZE                       (_ub_onfi_info.spare_size)
#define BCNT                        (_ub_onfi_info.block_count)
#define PCNT                        (_ub_onfi_info.page_count)


int _onfi_check_ecc(int r)
{
    if (ECC_CTRL_ERR==(r&0xFFFFFF00)) {
        printf("\nWW: ecc check failed (ret 0x%x)\n", r);
        return -1;
    } else {
        return 0;
    }
}
    
int onfi_read_write(u32_t offset, u32_t length, u8_t *buffer, char opt) {
    u32_t bi, pi, co, r_bi;
    u32_t size = length, l;
    
    // start addr
    bi = BLK_IDX(offset, PCNT*(PSIZE+SSIZE));
    pi = PAG_IDX(offset, PCNT*(PSIZE+SSIZE), (PSIZE+SSIZE));
    co = COL_OFF(offset, (PSIZE+SSIZE));
        
		
    //printf("DD: start from blk_0x%x, pg_0x%x, col_0x%x, size %d to buffer(0x%x)\n", 
    //    bi, pi, co, size, buffer);

    while (size>0) {
        if(-1 == get_good_block(bi, &r_bi)){
            puts("\nEE: block error!\n");
            return -1;
        }
#if !USE_BBT_SKIP
        if (-1 == go_to_good_block(&r_bi)) {
            puts("\nEE: go good block error!\n");
            return -1;
        }
        bi = r_bi;
#endif
        // check column offset
        if (0<co) {
            puts("\nEE: column address incorrect!\n");
            return -1;
        } else if (0==co && size>(PSIZE+SSIZE)) {
            l=PSIZE+SSIZE;
            (opt)?ofu_page_write(buffer, BLOCK_PAGE_ADDR(r_bi, pi))
                :ofu_page_read(buffer, BLOCK_PAGE_ADDR(r_bi, pi));
        } else {
            //l=(size>((PSIZE+SSIZE)-co))?((PSIZE+SSIZE)-co):size;
            l=size;
            (opt)?ofu_pio_write(buffer, l, BLOCK_PAGE_ADDR(r_bi, pi), co)
                :ofu_pio_read(buffer, l, BLOCK_PAGE_ADDR(r_bi, pi), co);            
        }
        //printf("DD: nasu_pio_%s buf 0x%x, len %d, 0x%x (b %d(%d), p %d), col 0x%x)\n", 
        //    (opt)?"write":"read", buffer, l, BLOCK_PAGE_ADDR(r_bi, pi), r_bi, bi, pi, co);
        size-=l;
        co=0;
        buffer+=l;
        if (++pi >= PCNT) {  // block idx increased
            bi++;
            pi=0;
        }
        printf("%03d%%\b\b\b\b", 100-(100*size)/length);
    }
    return 0;   
}

int onfi_write_ecc(u32_t offset, u32_t length, u8_t *buffer)
{
    u32_t bi, pi, co, r_bi;
    u32_t size = length, l;
    
    // start addr
    bi = BLK_IDX(offset, BSIZE);
    pi = PAG_IDX(offset, BSIZE, PSIZE);
    co = COL_OFF(offset, PSIZE);

    while (size>0) {
        if(-1 == get_good_block(bi, &r_bi)){
            puts("\nEE: block error!\n");
            return -1;
        }
#if !USE_BBT_SKIP
        if (-1 == go_to_good_block(&r_bi)) {
            puts("\nEE: go good block error!\n");
            return -1;
        }
        bi = r_bi;
#endif
        memset(_pge_buf+_ub_onfi_info.page_size, 0xFF, PBUF_SIZE-_ub_onfi_info.page_size);                
        // check column offset
        if (0<co) {              
            l=(size>(PSIZE-co))?(PSIZE-co):size;
        } else if (0==co) {
            if (size>=PSIZE) {
                l=PSIZE;
            } else {
                l=size;
            }
        } else {
            puts("\nEE: column address incorrect!\n");
            return -1;
        }
        memcpy(_pge_buf+co, buffer, l);
        if (is_blank_data((uint32_t*)_pge_buf, _ub_onfi_info.page_size>>2)) {
            printf("Ignore to write blank data in %x\n", BLOCK_PAGE_ADDR(r_bi, pi));
        } else {
            ofu_page_write_ecc(_pge_buf, BLOCK_PAGE_ADDR(r_bi, pi), _ecc_buf);
        }
        size-=l;
        co=0;
        buffer+=l;
        if (++pi >= PCNT) {  // block idx increased
            bi++;
            pi=0;
        }
    }
    return 0;
}

int onfi_read_ecc(u32_t offset, u32_t length, u8_t *buffer)
{
    u32_t bi, pi, co, r_bi;
    u32_t size = length, l;
    int ret;
    
    // start addr
    bi = BLK_IDX(offset, BSIZE);
    pi = PAG_IDX(offset, BSIZE, PSIZE);
    co = COL_OFF(offset, PSIZE);

    while (size>0) {
        if(-1 == get_good_block(bi, &r_bi)){
            puts("\nEE: block error!\n");
            return -1;
        }
        memset(_pge_buf, 0, PBUF_SIZE);
        ret=ofu_page_read_ecc(_pge_buf, BLOCK_PAGE_ADDR(r_bi, pi), _ecc_buf);
        if (_onfi_check_ecc(ret)) return -1;
        
        // check column offset
        if(0<co) {
            l=(size>(PSIZE-co))?(PSIZE-co):size;
        }else if (0==co) {
            l=(size>PSIZE)?PSIZE:size;
        }else {
            puts("\nEE: column address incorrect!\n");
            return -1;
        }
        memcpy(buffer, _pge_buf+co, l);        
        size-=l;
        co=0;
        buffer+=l;
        if (++pi >= PCNT) {  // block idx increased
            bi++;
            pi=0;
        }
    }
    return 0;
}

int onfi_erase(u32_t offset, u32_t length)
{
    u32_t bi, cnt=0, r_bi;
    int size = length;
    
    bi = BLK_IDX(offset, BSIZE);

    //printf("DD: block erase from 0x%x (block %d).\n", offset, blk);
    while (size>0) {
        if(-1 == get_good_block(bi, &r_bi)){
            puts("\nEE: block error!\n");
            return -1;
        }
#if !USE_BBT_SKIP
        if (-1 == go_to_good_block(&r_bi)) {
            puts("\nEE: go good block error!\n");
            return -1;
        }
        bi = r_bi;
#endif

        if (-1==ofu_block_erase(BLOCK_PAGE_ADDR(r_bi, 0))) {
            puts("\nEE: erase failed!\n");
            return -1;
        }
        size-=BSIZE;
        cnt++;
        bi++; 
    }
    //printf("from blk #%d size 0x%x is erased\n", BLK_IDX(offset, BSIZE), length);
    printf("%d block(s) erased \n", cnt);
    return 0;
}

