#include <common.h>
#include <soc.h>
#include <asm/otto_uart.h>
#include <asm/arch/cpu.h>
#include <asm/otto_pll.h>

#define UID (0)

#ifndef SECTION_OTTO_UART
	#define SECTION_OTTO_UART
#endif

uint32_t otto_uart_quiet = 0;

typedef struct {
	uint32_t frame_period_c;
	uint32_t base_clk_mhz;
	uint32_t r_to_cnt;
	uint32_t t_to_cnt;
} uart_stat_t;

static uart_stat_t ustat;

DECLARE_GLOBAL_DATA_PTR;

SECTION_OTTO_UART
int _uart_tstc(void) {
	return (UART_REG(UID, O_LSR) & (LSR_RFE | LSR_FE | LSR_PE | LSR_OE | LSR_DR)) == LSR_DR;
}

SECTION_OTTO_UART
int _uart_getc(void) {
	while(!_uart_tstc());
	return UART_REG(UID, O_RBR);
}

SECTION_OTTO_UART
void _uart_putc(char c) {
	if (otto_uart_quiet) return;

	while( !(UART_REG(UID, O_LSR) & LSR_THRE));
	UART_REG(UID, O_THR) = c;
	return;
}

int serial_init (void) {
	if (gd->baudrate == 0x0) {
	} else {
		serial_setbrg();
	}
	return (0);
}

void serial_putc(const char c) {
	if (c == '\n')
		_uart_putc('\r');

	_uart_putc(c);
	return;
}

SECTION_OTTO_UART
void serial_putc_raw(const char c) {
	_uart_putc(c);
	return;
}

SECTION_OTTO_UART
void serial_puts(const char *s) {
	while (*s) {
		serial_putc(*s++);
	}
	return;
}

int serial_getc() {
	return _uart_getc();
}

int serial_tstc() {
	return _uart_tstc();
}

static void
uart_set_baud(uint32_t uid, uint32_t baudrate, uint32_t base_clk_mhz) {
	uint32_t bclk_hz = base_clk_mhz * 1000 * 1000;
	uint32_t div, add, is_hs_uart;
	uint8_t lcr_bak;

	lcr_bak = UART_REG(uid, O_LCR);
	UART_REG(uid, O_LCR) = (lcr_bak | LCR_DLAB_EN);
	is_hs_uart = ((UART_REG(uid, O_IIR) & IIR_FIFO16_MASK) == 0);

	if (is_hs_uart) {
		div = bclk_hz / (16 * baudrate);
		add = (bclk_hz + (baudrate / 2)) / baudrate - (16 * div);
		UART_REG(uid, O_ADD) = add;
	} else {
		div = (bclk_hz + (8 * baudrate)) / (16 * baudrate);
	}

	UART_REG(uid, O_DLL) = (div) & 0xff;
	UART_REG(uid, O_DLH) = (div >> 8) & 0xff;
	UART_REG(uid, O_LCR) = lcr_bak;
	return;
}

static uint32_t
_uart_init(uint32_t uid, uint32_t baudrate, uint32_t base_clk_mhz) {
	ustat.r_to_cnt = 0;
	ustat.t_to_cnt = 0;
	ustat.base_clk_mhz = base_clk_mhz;

	UART_REG(uid, O_IER) = 0;

	uart_set_baud(uid, baudrate, base_clk_mhz);

	UART_REG(uid, O_LCR) = (LCR_8B | LCR_1S | LCR_NO_PARITY);
	UART_REG(uid, O_MCR) = (MCR_DTR | MCR_RTS | MCR_CLK_LX);
	UART_REG(uid, O_FCR) = (FCR_RBR_TRIG_HI | FCR_FIFO_EN);
	UART_REG(uid, O_FCR) = (FCR_RBR_TRIG_HI | FCR_THR_RESET | FCR_RBR_RESET | FCR_FIFO_EN);
	return 0;
}

SECTION_OTTO_UART
void serial_setbrg(void) {
	u32_t bus_mhz = cg_query_freq(CG_DEV_LX);
	u32_t brate = gd->baudrate;
	_uart_init(UID, brate, bus_mhz);
	return;
}
