#include <config.h>
#include <common.h>
#include <command.h>
#include "nor_spi_gen3/nor_spif_core.h"

#ifndef SECTION_XMODEM_TO_FLASH
	#define SECTION_XMODEM_TO_FLASH
#endif

#define SOH  0x01
#define STX  0x02
#define EOT  0x04
#define ACK  0x06
#define NAK  0x15
#define CAN  0x18
#define CTRLZ 0x1A

#define DLY_1S 1000000
#define MAXRETRANS 25

#define ERASE_UNIT_B (64*1024)

void serial_puts(const char *s);
void _uart_putc(const char c);
int _uart_getc(void);
int _uart_tstc(void);
uint16_t cyg_crc16(const unsigned char *buf, int len);

SECTION_XMODEM_TO_FLASH
static int check(int crc, const unsigned char *buf, int sz) {
	if (crc) {
		unsigned short crc = cyg_crc16(buf, sz);
		unsigned short tcrc = (buf[sz]<<8)+buf[sz+1];
		if (crc == tcrc)
			return 1;
	} else {
		int i;
		unsigned char cks = 0;
		for (i = 0; i < sz; ++i) {
			cks += buf[i];
		}
		if (cks == buf[sz])
			return 1;
	}

	return 0;
}

SECTION_XMODEM_TO_FLASH
static void _outbyte(uint8_t c) {
	return _uart_putc(c);
}

SECTION_XMODEM_TO_FLASH
static int32_t _inbyte(int32_t to_ms) {
	while (!_uart_tstc()) {
		udelay(1);
		if ((--to_ms) == 0) {
			return -2;
		}
	};
	return _uart_getc();
}

SECTION_XMODEM_TO_FLASH
static void flushinput(void) {
	while (_inbyte(((DLY_1S)*3)>>1) >= 0)
		;
}

SECTION_XMODEM_TO_FLASH
static int xmodemReceive(uint32_t flash_dest) {
	uint8_t xbuff[1030]; /* 1024 for XModem 1k + 3 head chars + 2 crc + nul */
	uint8_t *p;
	int bufsz, crc = 0;
	unsigned char trychar = 'C';
	unsigned char packetno = 1;
	int i, c, len = 0;
	int retry, retrans = MAXRETRANS;

	uint32_t erased_sz_b;
	erased_sz_b = 0;

	for(;;) {
		for( retry = 0; retry < 16; ++retry) {
			if (trychar) _outbyte(trychar);
			if ((c = _inbyte((DLY_1S)<<1)) >= 0) {
				switch (c) {
				case SOH:
					bufsz = 128;
					goto start_recv;
				case STX:
					bufsz = 1024;
					goto start_recv;
				case EOT:
					flushinput();
					_outbyte(ACK);

					return len; /* normal end */
				case CAN:
					if ((c = _inbyte(DLY_1S)) == CAN) {
						flushinput();
						_outbyte(ACK);
						return -1; /* canceled by remote */
					}
					break;
				default:
					break;
				}
			}
		}
		if (trychar == 'C') { trychar = NAK; continue; }

		flushinput();
		_outbyte(CAN);
		_outbyte(CAN);
		_outbyte(CAN);
		return -2; /* sync error */

	start_recv:
		if (trychar == 'C') crc = 1;
		trychar = 0;
		p = xbuff;
		*p++ = c;
		for (i = 0;  i < (bufsz+(crc?1:0)+3); ++i) {
			if ((c = _inbyte(DLY_1S)) < 0) goto reject;
			*p++ = c;
		}

		if ((xbuff[1] == (unsigned char)(~xbuff[2])) &&
		    (xbuff[1] == packetno || xbuff[1] == (unsigned char)packetno-1) &&
		    check(crc, &xbuff[3], bufsz)) {
			if (xbuff[1] == packetno)	{
				register int count = bufsz;

				/* Just-in-time erase. */
				if (count > 0) {
					len += count;
					if (flash_dest < 0x80000000) {
						if (len > erased_sz_b) {
							NORSF_ERASE(flash_dest, ERASE_UNIT_B, 0, 0);
							erased_sz_b += ERASE_UNIT_B;
						}
						NORSF_PROG(flash_dest, count, &xbuff[3], 0);
					} else {
						memcpy((void *)flash_dest, &xbuff[3], count);
					}
					flash_dest += count;
				}

				++packetno;
				retrans = MAXRETRANS+1;
			}
			if (--retrans <= 0) {
				flushinput();
				_outbyte(CAN);
				_outbyte(CAN);
				_outbyte(CAN);
				return -3; /* too many retry error */
			}
			_outbyte(ACK);
			continue;
		}
	reject:
		flushinput();
		_outbyte(NAK);
	}
}

SECTION_XMODEM_TO_FLASH
int xmodem_2_sf(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	int st;
	uint32_t dest_addr;
	char retry = 'n';

	/* These messages must not on flash. */
	char ask_msg[] = "\n(y/n)";
	char fail_msg[] = ":(\n\0";

	if (argc != 2) {
		printf("xf [offset]\n");
		return 1;
	}

	dest_addr = simple_strtoul(argv[1], NULL, 16);

	printf("II: XMODEM -> %s:%08x...\n", (dest_addr<0x80000000)?"FLASH":"MEM", dest_addr);
	do {
		st = xmodemReceive(dest_addr);
		if (st < 0) {
			serial_puts(fail_msg);
		}
		serial_puts(ask_msg);
	} while ((retry = _uart_getc()) == 'y');

	if (dest_addr < 0x80000000) {
		SYSTEM_RESET();
	}

	return 0;
}

U_BOOT_CMD(
	xf, 2, 0, xmodem_2_sf,
	"xmodem to flash",
	""
);
