#include <common.h>
#include <soc.h>
#include <asm/otto_onfi_flash.h>
#include <onfi/onfi_uboot_util.h>


#define SIZE_5KB              (5*1024)
#define ONAF_SRC_CHUNK_BUF    (0xA0000000)
#define ONAF_CHK_CHUNK_BUF    (ONAF_SRC_CHUNK_BUF + SIZE_5KB)
#define ONAF_ECC_TAG_BUF      (ONAF_CHK_CHUNK_BUF + SIZE_5KB)
#define wbuf ((u32_t *)(ONAF_SRC_CHUNK_BUF))
#define rbuf ((u32_t *)(ONAF_CHK_CHUNK_BUF))
#define eccb ((u32_t *)(ONAF_ECC_TAG_BUF))

#define PATTERN1 (ROW_ADDR(b, p))
#define PATTERN2 (((ROW_ADDR(b, p)&0xFF)<<24)|ROW_ADDR(b, p))
#define PATTERN3 ((pattern[(i&0xF)]<<((b+p)%13))|(pattern[(i&0xF)]>>(32-((b+p)%13))))

#define BLOCK_OFFSET    (6)
#define ROW_ADDR(b, p)  ((b<<BLOCK_OFFSET)|p)
#define BUF_RST_VAL     (0xCAFEBEEF)
#define START_BLOCK     (0)

#define CHK_N_SKIP_BAD_BLK() ({\
    printf("BLK%04d\b\b\b\b\b\b\b", b);\
    if (1==ti->bbi[b]){\
        printf("(BBI:%d)\n", b);\
        continue;\
    }\
})

#define PRINT_BUF_CONTENT() ({\
    for(i=0; i<psize; i++) {\
        if((i%8) ==0){\
            printf("\n0x%04x: ",i<<3);\
        }\
        printf("%08x ",*(wbuf+i));\
    }\
})


u32_t pattern[] = {
    0x5a5aa5a5,
    0xff00ff00,
    0xa55aa55a,
    0x00ff00ff,
    0xa5a5a5a5,
    0x13572468,
    0x5a5a5a5a,
    0x24683579,
    0xace1ace1,
    0xffff0000, 
    0x0000ffff,
    0x5555aaaa,
    0xaaaa5555,
    0x0f1e2d3c,
    0x01010101,
    0xFEFEFEFE
};

typedef struct {
	u32_t loops;
    u32_t bbi[8192];
	u32_t start_blk;
	u32_t blk_count;
    u8_t f_all_case;
    u8_t f_reset;
    u8_t f_bad;
} onfi_test_info_t;

extern _onfi_info_t _ub_onfi_info;
extern int _onfi_check_ecc(int r);
#define _cache_flush	(((soc_t *)(0x9fc00020))->bios).dcache_writeback_invalidate_all
#define SET_SEED 0
#define GET_SEED 1
/*
  get_or_set = GET_SEED: get seed
  get_or_set = SET_SEED: set seed
*/
static void __srandom32(u32_t *a1, u32_t *a2, u32_t *a3, u32_t get_or_set)
{
    static int s1, s2, s3;
    if(GET_SEED==get_or_set){
        *a1=s1;
        *a2=s2;
        *a3=s3;
    }else{
        s1 = *a1;
        s2 = *a2;
        s3 = *a3;
    }
}

static u32_t __random32(void)
{
#define TAUSWORTHE(s,a,b,c,d) ((s&c)<<d) ^ (((s <<a) ^ s)>>b)
    u32_t s1, s2, s3;
    __srandom32(&s1, &s2, &s3, GET_SEED);

    s1 = TAUSWORTHE(s1, 13, 19, 4294967294UL, 12);
    s2 = TAUSWORTHE(s2, 2, 25, 4294967288UL, 4);
    s3 = TAUSWORTHE(s3, 3, 11, 4294967280UL, 17);

    __srandom32(&s1, &s2, &s3, SET_SEED);

    return (s1 ^ s2 ^ s3);
}

u32_t onfi_pio_with_ecc_test(onfi_info_t *fi, onfi_test_info_t *ti)
{
    if (VZERO == fi) return 1;
    u32_t sb = ti->start_blk;
    u32_t nb = ti->start_blk+ti->blk_count;
    u32_t np = ONFI_NUM_OF_PAGE_PER_BLK(fi);
    u32_t psize = (ONFI_PAGE_SIZE(fi) + ONFI_SPARE_SIZE(fi)) >> 2;
    u32_t vsize = (ONFI_PAGE_SIZE(fi) + ONFI_OOB_SIZE(fi)) >> 2;   //The verify size without syndrome
    u32_t b, p, i;
    int ret;  
    
    printf("II: do %s ... ", __FUNCTION__);
    
    _cache_flush();
    for(b=sb; b<nb; b++){
        CHK_N_SKIP_BAD_BLK();
        
        ofu_block_erase(ROW_ADDR(b, 0));
        for(p=0; p<np; p++){
            //Generate & Verify pattern
            for(i=0; i<psize; i++) *(wbuf+i) = PATTERN3;             
            for(i=0; i<psize; i++) {
               if(PATTERN3 != *(wbuf+i)) {puts("\nEE: pattern check failed!\n"); return 1;} 
            }
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;

            // pattern write
            ofu_ecc_encode(wbuf, eccb);
            ofu_pio_write(wbuf, psize*4, ROW_ADDR(b, p), 0);
        }
    }    
    puts("verify... ");
    _cache_flush();
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        for(p=0; p<np; p++) {
            //Generate pattern & Reset read buf to a known value
            for(i=0; i<psize; i++){
                *(wbuf+i) = PATTERN3;
                *(rbuf+i) = BUF_RST_VAL;
            }            
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;

            // read page/chunk/column
            ofu_pio_read(rbuf, psize*4, ROW_ADDR(b, p), 0);
            ret = ofu_ecc_decode(rbuf, eccb);
            if(IS_ECC_DECODE_FAIL(ret)){
                printf("\nEE: BLK%d,PAGE%d,COL%d, ecc_decode fail(0x%x)\n", b, p, i<<2, ret);
                continue;
            }
 
            for(i=0; i<vsize; i++) {
                if( *(wbuf+i) != *(rbuf+i) ) { 
                    printf("\nEE: BLK%d,PAGE%d,COL%d,data:0x%08x != pattern:0x%08x\n", b,p,i<<2,*(wbuf+i),*(rbuf+i));
                    return 1;
                }
            }
        }
    }
    puts("done       \n");
    return 0;
}

u32_t onfi_dma_with_ecc_test(onfi_info_t *fi, onfi_test_info_t *ti)
{
    if (VZERO == fi) return 1;
    u32_t sb = ti->start_blk;
    u32_t nb = ti->start_blk+ti->blk_count;
    u32_t np = ONFI_NUM_OF_PAGE_PER_BLK(fi);
    u32_t psize = (ONFI_PAGE_SIZE(fi) + ONFI_SPARE_SIZE(fi)) >> 2;
    u32_t vsize = (ONFI_PAGE_SIZE(fi) + ONFI_OOB_SIZE(fi)) >> 2;   //The verify size without syndrome
    u32_t b, p, i, ra, rb, rc, v;
    int ret;  
    #define SEED1   ((0x13243*(b+1))&0xffffff)
    #define SEED2   (0xaaa0bdd+b)
    #define SEED3   (0xfffbda0-b)
    
    printf("II: do %s ... ", __FUNCTION__);
 
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        // target block erase
        ofu_block_erase(ROW_ADDR(b, 0));
    
        for(p=0; p<np; p++) {
            // Generate random patterns
            ra=SEED1;rb=SEED2;rc=SEED3;
            __srandom32(&ra, &rb, &rc, SET_SEED);
            for(i=0; i<psize; i++)  *(wbuf+i) = __random32();      
        
            // Verify random patterns
            ra=SEED1;rb=SEED2;rc=SEED3;
            __srandom32(&ra, &rb, &rc, SET_SEED);
            for(i=0; i<psize; i++) {
                v = __random32();
                if(v!= *(wbuf+i)){printf("EE: Pattern to addr(0x%x):0x%x!=0x%x\n", (u32_t)(wbuf+i), *(wbuf+i), v); return 1;}
            }
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;

            ofu_page_write_ecc(wbuf, ROW_ADDR(b, p), eccb);
        }
    }
    puts("verify... ");

    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        for(p=0; p<np; p++) {
            // Generate random patterns
            ra=SEED1;rb=SEED2;rc=SEED3;
            __srandom32(&ra, &rb, &rc, SET_SEED);
            for(i=0; i<psize; i++){
                *(wbuf+i) = __random32();  
                *(rbuf+i) = BUF_RST_VAL;
            }
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;
        
            // dma read
            ret = ofu_page_read_ecc(rbuf, ROW_ADDR(b, p), eccb);
            if(IS_ECC_DECODE_FAIL(ret)){
                printf("\nEE: BLK%d, PAGE%d,COL%d, ecc_decode fail(0x%x)\n", b, p, i<<2, ret);
                continue;
            }
    
            // data verify
            for(i=0; i<vsize; i++) {
                if(*(rbuf+i)!= *(wbuf+i)) {
                    printf("\nEE: double confirm, BLK%d,PAGE%d,COL%d,dst(addr:0x%x)0x%x != src(addr:0x%x)0x%x\n", 
                            b, p, i<<2, (u32_t)(rbuf+i), *(rbuf+i), (u32_t)(wbuf+i), *(wbuf+i));
                    return 1;
                }
            }
        }
    }    
    puts("done       \n");
    return 0;
}

u32_t onfi_scan_block_wr_with_ecc_test(onfi_info_t *fi, onfi_test_info_t *ti)
{
    if (VZERO == fi) return 1;
    u32_t sb = ti->start_blk;
    u32_t nb = ti->start_blk+ti->blk_count;
    u32_t np = ONFI_NUM_OF_PAGE_PER_BLK(fi);
    u32_t psize = (ONFI_PAGE_SIZE(fi) + ONFI_SPARE_SIZE(fi)) >> 2;
    u32_t vsize = (ONFI_PAGE_SIZE(fi) + ONFI_OOB_SIZE(fi)) >> 2;   //The verify size without syndrome
    u32_t b, p, i;
    int ret;

    printf("II: do %s ... ", __FUNCTION__);

    //1. DMA Wirte whole chip
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        ofu_block_erase(ROW_ADDR(b, 0)); 
        for(p=0; p < np; p++) {
            //Generate pattern
            for(i=0; i<psize; i++) *(wbuf+i) = PATTERN1; 
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;
            
            ofu_page_write_ecc(wbuf, ROW_ADDR(b, p), eccb);
        }
    }

    //2. PIO Read whole chip
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        for(p=0; p < np; p++) {
            //Generate pattern & Reset read buf to a known value
            for(i=0; i<psize; i++){
                *(wbuf+i) = PATTERN1;
                *(rbuf+i) = BUF_RST_VAL;
            }            
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;

            // pio read
            ofu_pio_read(rbuf, psize*4, ROW_ADDR(b, p), 0);
            ret= ofu_ecc_decode(rbuf, eccb);
            if(IS_ECC_DECODE_FAIL(ret)){
                printf("\nEE: dw-pr, BLK%d,PAGE%d: ecc_decode fail (0x%x)\n", b,p,ret);
                continue;
            }
 
            for(i=0; i<vsize; i++) {
                if( *(wbuf+i) != *(rbuf+i) ) { 
                    printf("\nEE: dw-pr, PAGE%d,COL%d,data(0x%08x):0x%08x != pattern(0x%08x):0x%08x\n",p,i<<2,(u32_t)(rbuf+i),*(rbuf+i),(u32_t)(wbuf+i),*(wbuf+i));
                    return 1;
                }
            }
        }
    }

    //3. ECC & PIO Write whole chip
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        ofu_block_erase(ROW_ADDR(b, 0));    
        for(p=0; p < np; p++) {
            //Generate pattern
            for(i=0; i<psize; i++) *(wbuf+i) = PATTERN2; 
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;
            //printf("\nII: addr=0x%x: blk(%d), pge(%d), pat(0x%x)", ROW_ADDR(b, p),b,p, pattern);
           
            ofu_ecc_encode(wbuf, eccb);            
            ofu_pio_write(wbuf, psize*4, ROW_ADDR(b, p), 0);
        }
    }

    //4. DMA Read whole chip
    for(b=sb; b<nb; b++) {
        CHK_N_SKIP_BAD_BLK();

        for(p=0; p < np; p++) {
            //Generate pattern & //Reset read buf to a known value
            for(i=0; i<psize; i++){
                *(wbuf+i) = PATTERN2;
                *(rbuf+i) = BUF_RST_VAL;
            } 
            wbuf[(ONFI_PAGE_SIZE(fi)/4)] = 0xFFFFFFFF;

            //printf("\nII: addr=0x%x: blk(%d), pge(%d), pat(0x%x)", ROW_ADDR(b, p),b,p, pattern);
           
            ret = ofu_page_read_ecc(rbuf, ROW_ADDR(b, p), eccb);
            if(IS_ECC_DECODE_FAIL(ret)){
                printf("\nEE: pw-dr, BLK%d,PAGE%d: ecc_decode fail (0x%x)\n", b,p,ret);
                continue;
            }

            // Check page read pattern
            for(i=0; i<vsize; i++) {
                if(*(wbuf+i) != *(rbuf+i)) {
                    printf("\nEE: pw-dr, PAGE%d,COL%d ,dst(0x%x):0x%x != pat(0x%x):0x%x\n", p, i<<2,(u32_t)(rbuf+i),*(rbuf+i),(u32_t)(wbuf+i),*(wbuf+i));
                    return 1;
                }
            }
        }
    }        

    puts("done       \n");
    return 0;
}


u32_t onfi_bad_block_search(onfi_info_t *fi, onfi_test_info_t *ti) 
{
    u32_t b, nb, sb;
    u32_t buf[2];
    sb = ti->start_blk; // start block
    nb = ti->start_blk+ti->blk_count;
    u32_t bbi_cnt = 0;
    
    puts("II: do bad block search ...");
    
    for(b=sb;b<nb;b++){ //block_num
        // performing erase
        if(0!=ofu_block_erase(ROW_ADDR(b, 0))) {
            ti->bbi[b]=1;
        } else ti->bbi[b]=0;
	}
    
    for(b=sb; b<nb; b++) {
        printf("BLK%04d\b\b\b\b\b\b\b", b);
        ofu_pio_read(buf, 4, ROW_ADDR(b, 0), 0x800);
        if(0xFFFFFFFF != buf[0]) {
            ti->bbi[b]=1; printf("BLK%04d ", b);
            bbi_cnt++;
        } else ti->bbi[b]=0;
    }

    puts(" done\n");
    return bbi_cnt;
}

enum cmd_error {
    ERR_INVALD_CMD,
    ERR_LOOP_CMD,
    ERR_RANGE_CMD,
    ERR_START_ADDR,
    ERR_SIZE,
};

enum reset_type {
    RESET_NONE=0,
    RESET_UBOOT,
    RESET_CHIP,
};

int cmd_parsing(int argc, char * const argv[], onfi_test_info_t *ti) 
{
	u32_t i=1, err;

    /* init value */
	while(i<argc) {
		if ('-' == argv[i][0]) {
            if( 0==(strcmp(argv[i],"-l")) || 0==(strcmp(argv[i],"-loops")) ) {
                u32_t loop;
                if((i+1) >= argc) { err=ERR_LOOP_CMD; goto error; }
                loop = simple_strtoul(argv[i+1], NULL, 10);
                if (0==loop) { puts("WW: invalid loop count(reset to 1)\n"); loop=1; }
                ti->loops = loop;
                i+=2;
            } else if (0==(strcmp(argv[i],"-r")) || 0==(strcmp(argv[i],"-range")) ) {
                u32_t blk, cnt;
                if((i+2) >= argc) { err=ERR_RANGE_CMD; goto error; }
                blk = simple_strtoul(argv[i+1], NULL, 10);
                cnt = simple_strtoul(argv[i+2], NULL, 10);
                printf("DD: input range %d %d\n", blk, cnt);
                if(0==blk || blk >= _ub_onfi_info.block_size ) { err=ERR_START_ADDR; goto error; }
                if(0==cnt || cnt >= _ub_onfi_info.block_size ) { err=ERR_SIZE; goto error; }
                ti->start_blk=blk;
                ti->blk_count=cnt;
                i+=3;
            } else if(0==strcmp(argv[i],"-reset")) {
                ti->f_reset = RESET_UBOOT;
                i++;
            } else if(0==strcmp(argv[i],"-reset_all")){
                ti->f_reset = RESET_CHIP;
                i++;
            } else if(0==strcmp(argv[i],"-bad")){
                ti->f_bad = 1;
                i++;
            } else if( 0==(strcmp(argv[i],"-ac")) || 0==(0==strcmp(argv[i],"-all_case")) ){
                ti->f_all_case = 1;
                i++;
            } else {
                printf("WW: unknown command \"%s\" ignored.\n", argv[i]);
                i++;
            }
        } else {
            err = ERR_INVALD_CMD; goto error;
        }
	}

    if (0==ti->blk_count) { puts("EE: input error\n"); return 1;}
    
    return 0;
error:
    printf("EE: incomplete commands (type: %d)\n", err);
    return -1;	
}

int onfi_mt_test(onfi_info_t *fi, onfi_test_info_t *ti)
{
    if(ti->f_all_case){
        if((0!=onfi_pio_with_ecc_test(fi, ti)) || (0!=onfi_dma_with_ecc_test(fi, ti))){
            puts("EE: ONFI flash test failed\n"); 
            return 1;
        }
    }
    if(0!=onfi_scan_block_wr_with_ecc_test(fi, ti)){
        puts("EE: onfi_scan_block_wr_with_ecc_test failed\n");
        return 1;
    }  
    return 0;
    
}

int do_onfi_test (cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
    u32_t i=1;
    onfi_test_info_t ti={0};
    onfi_info_t *fi = _plr_onfi_info;    
    ti.loops = 1;
    
    if (0!=cmd_parsing(argc, argv, &ti)) return 1;
    printf("II: ONFI test info:\n\tstart: BLK%04d, end: BLK%04d\n", 
            ti.start_blk, ti.start_blk+ti.blk_count-1);
    printf("\tloop: %d, reset type: %s\n\tbad block search only: %c\n",
            ti.loops, ti.f_reset?(RESET_UBOOT==ti.f_reset?"reset":"reset_all"):"none", ti.f_bad?'y':'n');
    printf("\tall_testcase: %c\n\tOnly block scan(w/ ecc): %c\n",
            ti.f_all_case?'y':'n', ti.f_all_case?'n':'y');

    if((ti.start_blk+ti.blk_count-1) == onfi_bad_block_search(fi, &ti)){
        printf("EE: All of the assigned blocks are bad block\n");
        return -1;
    }
    if (ti.f_bad) return 0;
    
    while(i<=ti.loops) {
        printf("II: #%d test\n", i++);
        if(onfi_mt_test(fi, &ti)) {return -1;}
    }
    if(RESET_UBOOT == ti.f_reset){
        do_reset (NULL, 0, 0, NULL);
    } else if (RESET_CHIP ==  ti.f_reset){
        SYSTEM_RESET();
    } 
    return 0;
}

U_BOOT_CMD(
        monfi_test, 10, 1, do_onfi_test,
        "monfi_test  - do ONFI flash test. ",
        "-r/-range <start block> <block count> [-l/-loops <test loops>] [-ac/-all_case] [-reset/-reset_all] [-bad]\n"\
        "    => do ONFI flash test.\n" \
        "    => [-ac/-all_case]: Set '1' to do all test cases, or only do 'Block Scan Pattern' case."
);
