#include <common.h>
#include <command.h>
#include <soc.h>
#include <cpu.h>
#include "snaf_simple_driver.h"

typedef struct {
	u8_t cmd;
	u8_t addr_io_type;
	u8_t addr_dummy_b;
	u8_t data_io_type;
} snaf_cmd_t;

static uint32_t _snaf_oip(void) {
	uint32_t oip;
	snafsd_cs('0', '0');
	snafsd_write(SNAF_SIO, 2, 0x0fc0);
	oip = snafsd_read(SNAF_SIO, 1) >> 24;
	snafsd_cs('0', '1');

	return (oip & 1);
}

static void _snaf_read_page_to_cache(uint32_t page_id) {
	snafsd_cs('0', '0');
	snafsd_write(SNAF_SIO, 1, 0x13);
	snafsd_write(SNAF_SIO, 1, 0);
	snafsd_write(SNAF_SIO, 2, page_id);
	snafsd_cs('0', '1');

	while (_snaf_oip());

	return;
}

static uint32_t
_snaf_read_cache_to_reg(const snaf_cmd_t *cmd, const uint32_t off, const uint32_t len) {
	uint32_t data;
	snafsd_cs('0', '0');
	snafsd_write(SNAF_SIO, 1, cmd->cmd);
	snafsd_write(cmd->addr_io_type, 2, off);
	snafsd_write(cmd->addr_io_type, cmd->addr_dummy_b, 0);
	data = snafsd_read(cmd->data_io_type, len) >> ((4-len) * 8);
	snafsd_cs('0', '1');
	return data;
}

static uint8_t *
_snaf_read_cache_to_mem(const snaf_cmd_t *cmd, const uint32_t off, const uint32_t len,
                        uint8_t *addr) {
	snafsd_cs('0', '0');
	snafsd_write(SNAF_SIO, 1, cmd->cmd);
	snafsd_write(cmd->addr_io_type, 2, off);
	snafsd_write(cmd->addr_io_type, cmd->addr_dummy_b, 0);
	snafsd_dma_read(cmd->data_io_type, (uint32_t)addr, len);
	snafsd_cs('0', '1');
	return addr;
}

static uint32_t _min(uint32_t a, uint32_t b) {
	return ((a>b)?b:a);
}

static void _snaf_read_page_to_mem_pio(
	const snaf_cmd_t *cmd,
	uint8_t *addr,
	const uint32_t psize_b) {

	uint32_t cmd_len, off, val;

	off = 0;
	while (off < psize_b) {
		cmd_len = _min((psize_b - off), 4);

		val = _snaf_read_cache_to_reg(cmd, off, cmd_len);
		memcpy(addr+off, &val, cmd_len);

		off += cmd_len;
	}
	return;
}

static void _snafut_cmds_lens_read(
	const snaf_cmd_t *cmds,
	uint8_t *addr,
	const uint32_t psize_b) {

	uint32_t cmd_len, off, val;
	const snaf_cmd_t *cmd;

	off = 0;
	cmd_len = 0;
	cmd = cmds;

	while (off < psize_b) {
		cmd_len = (cmd_len % 4) + 1;
		cmd_len = _min((psize_b - off), cmd_len);

		(*((volatile uint32_t *)&val)) =
			_snaf_read_cache_to_reg(cmd, off, cmd_len) << ((4-cmd_len)*8);
		memcpy(addr + off, &val, cmd_len);

		off += cmd_len;
		if ((++cmd)->cmd == 0) cmd = cmds;
	}
	return;
}

static void
_snafut_cmds_lens_dma_pio_mix_read(const snaf_cmd_t *cmds,
                                   uint8_t *addr,
                                   const uint32_t psize_b) {
	const snaf_cmd_t *cmd;
	uint32_t cmd_len, off;

	off = 0;
	cmd_len = 0;
	cmd = cmds;

	while (off < psize_b) {
		cmd_len = _min((psize_b - off), 64);

		_snaf_read_cache_to_mem(cmd, off, cmd_len, addr + off);
		_snaf_read_cache_to_reg(cmd, off, ((off/64)%3+1));

		off += cmd_len;
		if ((++cmd)->cmd == 0) cmd = cmds;
	}
	return;
}

static void _snafut_cmd_info(const snaf_cmd_t *cmds) {
	const snaf_cmd_t *cmd;
	const char io_type_char[] = {'S', 'D', 'Q'};

	cmd = cmds;
	puts("II: Utilized Commands:\n");
	while (cmd->cmd) {
		printf(
			"  %02x(S, 1B) + Addr.(%c, 2B) + Dummy(%c, %dB) + Data(%c)\n",
			cmd->cmd,
			io_type_char[cmd->addr_io_type],
			io_type_char[cmd->addr_io_type],
			cmd->addr_dummy_b,
			io_type_char[cmd->data_io_type]);
		cmd++;
	}
	return;
}

static int
do_snaf_cntlr_test(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	uint8_t *sio_in;
	uint8_t *var_in;
	const uint32_t page_size_b = 2048 + 64;
	const snaf_cmd_t snaf_read_cmds[] = {
		{0x03, SNAF_SIO, 1, SNAF_SIO},
		{0x0b, SNAF_SIO, 1, SNAF_SIO},
		{0x3b, SNAF_SIO, 1, SNAF_DIO},
		{0x6b, SNAF_SIO, 1, SNAF_QIO},

		{0xbb, SNAF_DIO, 1, SNAF_DIO},
		{0xeb, SNAF_QIO, 2, SNAF_QIO},

		{0, 0, 0, 0}
	};

	sio_in = (uint8_t *)__builtin_alloca(page_size_b*6);
	var_in = (uint8_t *)__builtin_alloca(page_size_b*6);

	memset(sio_in, 0, page_size_b);
	memset(var_in, 0, page_size_b);

	_snafut_cmd_info(snaf_read_cmds);

	/* fill base data */
	_snaf_read_page_to_cache(0);
	_snaf_read_page_to_mem_pio(&snaf_read_cmds[0], sio_in, page_size_b);

	puts("II: CMDs x 1/2/3/4 bytes PIO reads... ");
	_snafut_cmds_lens_read(snaf_read_cmds, var_in, page_size_b);
	if (memcmp(var_in, sio_in, page_size_b)) {
		printf("fail: dst:%p != src:%p\n", var_in, sio_in);
	} else {
		puts("ok\n");
	}

	puts("II: CMDs DMA reads + CMDs x 1/2/3/4 bytes PIO reads... ");
	memset(var_in, 0, page_size_b);
	_snafut_cmds_lens_dma_pio_mix_read(snaf_read_cmds, var_in, page_size_b);
	if (memcmp(var_in, sio_in, page_size_b)) {
		printf("fail: dst:%p != src:%p\n", var_in, sio_in);
	} else {
		puts("ok\n");
	}


	return 0;
}

U_BOOT_CMD(
	msnaf_cntlr_test, 1, 0, do_snaf_cntlr_test,
	"SPI NAND flash cntlr. test",
	""
	);
