#include <common.h>
#include <command.h>
#include <soc.h>
#include <cpu.h>
#include "snaf_simple_driver.h"

static uint32_t verbose;

static void snafreg_till_ready(void) {
	uint32_t start_c;
	if (verbose) puts("II: check ready... ");
	start_c = CPU_GET_CP0_CYCLE_COUNT();
	while (REG32(0xb801a440) & (0x28)) {
		if ((CPU_GET_CP0_CYCLE_COUNT() - start_c) > 1*1000*1000*1000) {
			puts("time out... ");
			start_c = CPU_GET_CP0_CYCLE_COUNT();
		}
	}
	return;
}

static void snafreg_set(uint32_t addr, uint32_t val) {
	snafreg_till_ready();
	if (verbose) printf("mw 0x%08x 0x%08x...", addr, val);
	REG32(addr) = val;
	if (verbose) putc('\n');
	return;
}

static uint32_t snafreg_get(uint32_t addr) {
	uint32_t val;
	snafreg_till_ready();
	if (verbose) printf("md 0x%08x 1         => ", addr);
	val = REG32(addr);
	if (verbose) printf("0x%08x\n", val);
	return val;
}

void snafsd_cs(const char _cs, const char _cs_lvl) {
	const int cs = _cs & 0x1;
	const uint32_t cs_lvl = _cs_lvl & 0x1;
	uint32_t reg = 0x11;

	snafreg_set(0xb801a404, reg);
	while ((REG32(0xb801a440) & 0x11) != reg) {
		;
	}

	if (cs_lvl) return;
	/* '0': 0011_0000: & 0x1 = 0
	   '1': 0011_0001: & 0x1 = 1
		 for cs0, 0x10 >> (0*4) = 0x10;
		 for cs1, 0x10 >> (1*4) = 0x01. Both are the exact masks for CS0 and CS1. */
	reg = (0x10 >> (cs*4)) | (cs_lvl << (cs * 4));

	snafreg_set(0xb801a404, reg);
	while ((REG32(0xb801a440) & 0x11) != reg) {
		;
	}

	return;
}

void snafsd_write(uint32_t _io, uint32_t _len, uint32_t _data) {
	const uint32_t cmd_reg = (_io << 28) | (_len - 1);

	snafreg_set(0xb801a408, cmd_reg);
	snafreg_set(0xb801a414, _data << ((4 - _len) * 8));

	return;
}

uint32_t snafsd_read(uint32_t _io, uint32_t _len) {
	const uint32_t cmd_reg = (_io << 28) | (_len - 1);

	snafreg_set(0xb801a40c, cmd_reg);
	return snafreg_get(0xb801a410);
}

uint32_t snafsd_dma_read(uint32_t _io, uint32_t _addr, uint32_t _len) {
	const uint32_t leng_reg = (_io << 28) | (_len - 1);
	const uint32_t phy_addr = _addr & 0x1fffffff;

	writeback_invalidate_dcache_range(_addr, _addr+_len);

	snafreg_set(0xb801a424, leng_reg);
	snafreg_set(0xb801a41c, phy_addr);
	snafreg_set(0xb801a418, 0);
	snafreg_till_ready();
	if (verbose) puts("dma done\n");
	return _addr;
}

static uint32_t snafcli_char_2_io_type(char _io) {
	/* _io could be
	   s/S: 01x1_0011: [6] + [1:0] = 0
	   d/D: 01x0_0100: [6] + [1:0] = 1
	   q/Q: 01x1_0001: [6] + [1:0] = 2
		 0x0: 0000_0000: [6] + [1:0] = 0 (no input, default to SIO)
	   ==> 0/1/2 = SNAF[W|R]CMR.IO_WIDTH
		 note: no shit protection */
	return ((_io + (_io >> 6)) & 0x3);
}

static void snafcli_write(char _io, char *_data, char *_len) {
	uint32_t data, len, io;

	len = strlen(_data);
	if ((_data[1] & 'X') == 'X') {
		len -= 2;
		_data[1] = 'x';
	}
	len = (len + 1)/2;
	data = simple_strtoul(_data, NULL, 16);
	io = snafcli_char_2_io_type(_io);

	snafsd_write(io, len, data);

	return;
}

static uint32_t snafcli_read(char _io, char *_addr, char *_len) {
	const uint32_t io = snafcli_char_2_io_type(_io);
	const uint32_t len = simple_strtoul(_len, NULL, 10);

	if (_addr) {
		const uint32_t addr = simple_strtoul(_addr, NULL, 16);
		return snafsd_dma_read(io, addr, len);
	} else {
		return snafsd_read(io, len);
	}
}

static int do_snafcli(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[]) {
	char input, io, cs, val;
	char *data_string, *len_string;

	if ((argc != 3) && (argc != 4)) return 1;

	input = argv[1][0];
	data_string = NULL;
	len_string = NULL;

	switch(input) {
	case 'v':
		verbose = argv[2][0] & 1; break;
	case '0':
	case '1':
		cs = input;
		val = argv[2][0];
		snafsd_cs(cs, val); break;
	case 'w':
		io = argv[1][1];
		data_string = argv[2];
		if (argc == 4) len_string = argv[3];
		snafcli_write(io, data_string, len_string); break;
	case 'r':
		io = argv[1][1];
		if (argc == 4) {
			data_string = argv[2];
			len_string = argv[3];
		} else {
			len_string = argv[2];
		}
		snafcli_read(io, data_string, len_string); break;
	default:
		return -1;
	}

	return 0;
}

U_BOOT_CMD(
	snafcli, 4, 0, do_snafcli,
	"Low-level reg. access of SNAF cntlr.",
	"snafcli v [1|0]\n"
	"snafcli [cs] [level]\n"
	"snafcli w[s|d|q] [data]\n"
	"snafcli r[s|d|q] [len]\n"
	"snafcli r[s|d|q] [addr] [len]\n"
	"\n"
	"Examples:\n"
	"  snafcli v 1: dump more message\n"
	"  snafcli 0 0: pull cs0 low\n"
	"  snafcli ws 0x03: SIO write 0x03 to SNAF cntlr.\n"
	"  snafcli rq 4: QIO read 4 bytes of data from SNAF cntlr.\n"
	"  snafcli rd 0x80000000 128: QIO read 128-byte data and DMA write to 0x80000000\n"
	"\n"
	"  read ID:\n"
	"    snafcli 0 0; snafcli w 0X9f00; snafcli r 3; snafcli 0 1\n"
	"  read page 0 to cache:\n"
	"    snafcli 0 0; snafcli ws 0x13; snafcli ws 0x00; snafcli ws 0x0000; snafcli 0 1\n"
	"  read page 0 from cache:\n"
	"    snafcli 0 0; snafcli ws 0x03; snafcli ws 0x0000; snafcli ws 0x00; snafcli rs 0x80000000 2112; snafreg 0 1\n"
	);
