/*
 * (C) Copyright 2000
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

/*
 * Cache support: switch on or off, get status
 */
#include <asm/mipsregs.h>
#include <common.h>
#include <command.h>
#include <linux/compiler.h>
#define CKSEG0 0x80000000
#if 0
#define ICACHE_SIZE 0x10000
#define DCACHE_SIZE 0x8000
#define LINE_SIZE 0x20
#endif
/*
 * Cache Operations available on all MIPS processors with R4000-style caches
 */
#define Index_Invalidate_I		0x00
#define Index_Writeback_Inv_D		0x01
#define Index_Load_Tag_I		0x04
#define Index_Load_Tag_D		0x05
#define Index_Store_Tag_I		0x08
#define Index_Store_Tag_D		0x09

#define Index_Store_Data_I      0x0c
#define Index_Store_Data_D       0x0d

#define Hit_Invalidate_I		0x10
#define Hit_Invalidate_D		0x11
#define Hit_Writeback_Inv_D		0x15

/*
 * R4000-specific cacheops
 */
#define Create_Dirty_Excl_D		0x0d
#define Fill				0x14
#define Hit_Writeback_I			0x18
#define Hit_Writeback_D			0x19

/******************/
#define MIPS_ISA_ARCH_LEVEL "arch=r4000"
#define cache_op(op,addr)						\
	__asm__ __volatile__(						\
	"	.set	push					\n"	\
	"	.set	noreorder				\n"	\
	"	.set "MIPS_ISA_ARCH_LEVEL"			\n"	\
	"	cache	%0, %1					\n"	\
	"	.set	pop					\n"	\
	:								\
	: "i" (op), "R" (*(unsigned char *)(addr)))


/******************/
#define MIPS_ISA_LEVEL "mips64r2"
#define instruction_hazard()						\
do {									\
	unsigned long tmp;						\
									\
	__asm__ __volatile__(						\
	"	.set "MIPS_ISA_LEVEL"				\n"	\
	"	dla	%0, 1f					\n"	\
	"	jr.hb	%0					\n"	\
	"	.set	mips0					\n"	\
	"1:							\n"	\
	: "=r" (tmp));							\
} while (0)



#define __stringify_1(x...)	#x
#define __stringify(x...)	__stringify_1(x)


#define ___ehb								\
	sll	$0, $0, 3

#define __back_to_back_c0_hazard					\
	___ehb

#define back_to_back_c0_hazard() 					\
do {									\
	__asm__ __volatile__(						\
	__stringify(__back_to_back_c0_hazard)				\
	);								\
} while (0)

#define _mips_sync() __asm__ __volatile__ ("sync  0x3" : : : "memory")
#define _mips_sync2() __asm__ __volatile__ ("sync  0x2" : : : "memory")



#define read_c0_idatalo()		__read_32bit_c0_register($28, 1)
#define write_c0_idatalo(val)	__write_32bit_c0_register($28, 1, val)

#define read_c0_idatahi()	__read_32bit_c0_register($29, 1)
#define write_c0_idatahi(val)	__write_32bit_c0_register($29, 1, val)

#define read_c0_ddatalo()	__read_32bit_c0_register($28, 3)
#define write_c0_ddatalo(val)	__write_32bit_c0_register($28, 3, val)



#define CACHE_WAY  4
/***********************************/
#define PATTERN_NUM 5
int pi = 0;
u32 patterns[PATTERN_NUM][3] = { 
{0, 0, 0},
{0xffffffff, 0xffffffff, 0xffffffff},
{0xffff0000, 0xffff0000, 0xffff0000},
{0x0000FFFF, 0x0000FFFF, 0x0000FFFF},
{0xAAAAAAAA, 0xAAAAAAAA, 0xAAAAAAAA}
};

/********************************/
void (*my_func_ptr) (int);

static int parse_argv(const char *);
static int parse_argv_int(const char *);


void icache_disable(void);
void dcache_disable(void);

static inline void __flush_icache(void)
{
    int i;

    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Invalidate_I, i);
    }
    _mips_sync2();
}

static inline void __flush_dcache(void)
{

    int i;

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Writeback_Inv_D, i);
    }
    _mips_sync2();
}

void flush_l1cache(void)
{
    __flush_icache();
    __flush_dcache();
}

void flush_icache(void)
{
    __flush_icache();
}

void flush_dcache(void)
{
    __flush_dcache();
}



#define IWAY_VERIFY 1
static void icache_w(unsigned int tag, unsigned int way, unsigned int data)
{
    int i, j;
    int dw = 0;
    write_c0_idatalo(data);
    write_c0_idatahi(data);
    back_to_back_c0_hazard();
//      printf("datalo: 0x%x, datahi: 0x%x\n", read_c0_idatalo(), read_c0_idatahi());
    write_c0_taglo(tag);
    back_to_back_c0_hazard();
    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        write_c0_ecc(read_c0_ecc() | (1 << 29));        //PCO 27 Precode override. Used for diagnostic/test of the instruction cache.
        back_to_back_c0_hazard();
        for (j = 0; j < 4; j++) {
            dw = (j << 3);
            //i add DW to cache line 
            cache_op(Index_Store_Data_I, i | dw);
        }
        write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
        back_to_back_c0_hazard();
        cache_op(Index_Store_Tag_I, i);
    }
#if IWAY_VERIFY
    //way array     
    write_c0_taglo(way);
    write_c0_ecc(read_c0_ecc() | (1 << 29));
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Store_Tag_I, i);
    }
    write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
#endif
    back_to_back_c0_hazard();
    _mips_sync();


    return;
}

int ICACHE_SETS = 512;
int cache_error = 0;
static void icache_data_verify(u32 index, u32 dw, u32 expected_data)
{
    unsigned int datalo, datahi;
    unsigned long clinze = 0;
    unsigned long ori_dw = 0;

    //DATA
    datalo = read_c0_idatalo();
    datahi = read_c0_idatahi();
    clinze = (index - CKSEG0) / CACHELINE_SIZE;
    ori_dw = dw >> 3;
    if (expected_data != datalo) {
        printf
            ("ICacheLine(%lu), way/index/dw: %lu / %lu / %lu , [ DataLo ]: 0x%08x, expected: 0x%08x\n",
             clinze, clinze / ICACHE_SETS, clinze % ICACHE_SETS, ori_dw,
             datalo, expected_data);
    }
    if (expected_data != datahi) {
        printf
            ("ICacheLine(%lu), way/index/dw: %lu / %lu / %lu , [ DataHi ]: 0x%08x, expected: 0x%08x\n",
             clinze, clinze / ICACHE_SETS, clinze % ICACHE_SETS, ori_dw,
             datahi, expected_data);
    }



}

#define ITAG_VALID_MASK  0xfffff0a0
static void icache_tag_verify(u32 index, u32 expected_tag)
{
    unsigned int tag = 0;
    unsigned int vailed_tag_pn;
    unsigned long clinze = (index - CKSEG0) / CACHELINE_SIZE;


    write_c0_taglo(0);
    back_to_back_c0_hazard();
    vailed_tag_pn = (expected_tag & ITAG_VALID_MASK);

    cache_op(Index_Load_Tag_I, index);  //i add DW to cache line  index
    back_to_back_c0_hazard();

    tag = read_c0_taglo();

    if (vailed_tag_pn != tag) {
        printf
            ("ICacheLine(%lu), way/index: %lu / %lu , [ Tag      ]: 0x%08x, expected: 0x%08x, pattern: 0x%08x\n",
             clinze, clinze / ICACHE_SETS, clinze % ICACHE_SETS, tag,
             vailed_tag_pn, expected_tag);
    }
}

#if IWAY_VERIFY
#define IWAY_VALID_MASK 0xfffffca0
static void icache_way_verify(u32 expected_way)
{

    int i;
    unsigned int way_r;
    unsigned int vailed_way_pn;
    unsigned long clinze;

    vailed_way_pn = (expected_way & IWAY_VALID_MASK);
    // WAY ARRAY
    write_c0_ecc(read_c0_ecc() | (1 << 29));
    back_to_back_c0_hazard();
    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Load_Tag_I, i);
        back_to_back_c0_hazard();
        way_r = read_c0_taglo();
        clinze = (i - CKSEG0) / CACHELINE_SIZE;
        if (way_r != vailed_way_pn) {
            printf
                ("ICacheLine(%lu), way/index: %lu / %lu , [ Way      ]: 0x%08x, expected: 0x%08x, pattern: 0x%08x\n",
                 clinze, clinze / ICACHE_SETS, clinze % ICACHE_SETS, way_r,
                 vailed_way_pn, expected_way);
        }
    }
    write_c0_ecc(read_c0_ecc() & ~(1 << 29));
    back_to_back_c0_hazard();



}
#endif
static void icache_verify(unsigned int tag_data, unsigned int data,
                          unsigned int way_data)
{
    unsigned int i, j;
    int dw = 0;

    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        for (j = 0; j < 4; j++) {
            dw = (j << 3);
            cache_op(Index_Load_Tag_I, i | dw); //i add DW to cache line  index
            back_to_back_c0_hazard();

            icache_data_verify(i, dw, data);
        }
        //TAG
        icache_tag_verify(i, tag_data);
    }
#if IWAY_VERIFY
    icache_way_verify(way_data);
#endif
    return;

}


/*******************/
#define PCI_BIT 13
#define PCI_MASK 0x3f
#define PRECODE_PATS  4
unsigned int precode_pattern[PRECODE_PATS] = {0x0, 0x3f, 0x15,0x2a};
static void icache_precode_test(int loops)
{
    unsigned int i, j, runs;
#if 1
    int data;
	unsigned long clinze;
#endif
	unsigned long tmp;
	 printf("--- precode_pattern: \n");
	for(j=0;j<PRECODE_PATS;j++){
		  printf(" 0x%08x, ",  precode_pattern[j]);
	}
	printf("\n   (Test Loops=%d) Start...............\n", loops);
	
for(runs=0;runs<loops;runs++) {
 
	for(j=0;j<PRECODE_PATS;j++){
//		 data = 0;
		 write_c0_ecc(read_c0_ecc() | (1 << 29) | (1<<27));        //waybit 29, PCO27
		 back_to_back_c0_hazard();	
		
#if 0		
		 write_c0_ecc(read_c0_ecc() | (precode_pattern[j]&PCI_MASK)<<PCI_BIT);       
		 back_to_back_c0_hazard();	
		 printf("read_c0_ecc()=0x%08x\n", read_c0_ecc());
#endif
 //        printf("--- precode_pattern[%d]=0x%08x\n", j , precode_pattern[j]);
		for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
	//	for (i = CKSEG0; i < (CKSEG0 + 0x80); i += CACHELINE_SIZE) {
	       tmp = read_c0_ecc() & ~(PCI_MASK<<PCI_BIT);
		   write_c0_ecc(tmp);
		   back_to_back_c0_hazard();
		   back_to_back_c0_hazard();
	       write_c0_ecc(read_c0_ecc() | ( precode_pattern[j]&PCI_MASK ) << PCI_BIT);       
		   back_to_back_c0_hazard();
           
		   //printf("(%u)write: read_c0_ecc()=0x%08x, pat=0x%08x\n", i, read_c0_ecc() ,( precode_pattern[j]&PCI_MASK ) << PCI_BIT );
	       
		   cache_op(Index_Store_Data_I, i);
		   back_to_back_c0_hazard();
		   back_to_back_c0_hazard();
		   
		   cache_op(Index_Load_Tag_I, i);
		   back_to_back_c0_hazard();
		   data= read_c0_ecc();
		   data = (data >> PCI_BIT) & PCI_MASK;
		   if(data != precode_pattern[j]){
				 clinze = (i - CKSEG0) / CACHELINE_SIZE;
				 printf("ICacheLine(%lu), way/index: %lu / %lu , [ Precode     ]: 0x%08x, expected: 0x%08x\n",
				 clinze, clinze / ICACHE_SETS, clinze % ICACHE_SETS, data, precode_pattern[j]);
			}
		   
		  // printf("(%u)read : read_c0_ecc()=0x%08x\n", i, read_c0_ecc());
		
		}

		write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
		back_to_back_c0_hazard();

       
	}	
}	
    return;

}
/*****************************************/

static void icache_test(int loops)
{
    int i;
    int n = 0;
	icache_disable();
	printf("---------   ICache Precode RAM Test---------------------------\n");
	icache_precode_test(loops);
	printf("---------   ICache Precode RAM Test END!\n\n");
    printf("---------   ICache Tag/Way/Data RAMs Test --------------------\n");
    for (i = 0; i < PATTERN_NUM; i++) {
        printf("<pattern(%d):0x%08x,0x%08x,0x%08x>\n", i, patterns[i][0],
               patterns[i][1], patterns[i][2]);
    }


    for (n = 0; n < loops; n++) {
        printf("ICACHE_TEST LOOP(%d):\n", n);
        cache_error = 0;
        for (i = 0; i < PATTERN_NUM; i++) {
            icache_w(patterns[i][0], patterns[i][1], patterns[i][2]);
            icache_verify(patterns[i][0], patterns[i][1], patterns[i][2]);
        }
        printf("\n");
    }
}


 /***********************************************/
static void icache_write(int data)
{
    int i;
//      int j;
    //  int dw=0;
    write_c0_idatalo(data);
    back_to_back_c0_hazard();
    write_c0_idatahi(data);

    back_to_back_c0_hazard();
    back_to_back_c0_hazard();
    //printf("[%s]  1: lo: 0x%x, hi: 0x%x , input: 0x%08x\n", __func__, read_c0_idatalo(), read_c0_idatahi(), data);
    //write_c0_ecc(read_c0_ecc()| (1<<27) | (1<<30) ); //PCO 27 Precode override. Used for diagnostic/test of the instruction cache.
    write_c0_ecc(read_c0_ecc() | (1 << 29));    //bit 29 , way bit to enable.
    back_to_back_c0_hazard();

    printf("  read_c0_ecc(): 0x%08x\n", read_c0_ecc());
#if 1
    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE / 16); i += CACHELINE_SIZE) {
#if 0
        for (j = 0; j < 4; j++) {
            dw = (j << 3);
            //i add DW to cache line 
            cache_op(Index_Store_Data_I, i | dw);
        }
#else
        //cache_op(Index_Store_Data_I, i );
        cache_op(0xc, i);
#endif
    }
    write_c0_ecc(read_c0_ecc() | 0xDFFFFFFF);
#else
    /* index store data for icache */
    cache_op(0xc, 0x80000000);
    back_to_back_c0_hazard();
    back_to_back_c0_hazard();
    //index load tag
    instruction_hazard();
    /* index load tag(data) for icache */
    cache_op(0x4, 0x80000000);
    back_to_back_c0_hazard();
    back_to_back_c0_hazard();
    back_to_back_c0_hazard();
//      printf("    2:lo: 0x%x, hi: 0x%x\n", read_c0_idatalo(), read_c0_idatahi());
#endif
}

static void icache_read(void)
{
    unsigned int i;
    int j = 0;
    // int dw = 0;
    int cnt = 0;
    printf("CACHELINE_SIZE : %d ,  ICACHE_SIZE: %d\n", CACHELINE_SIZE,
           ICACHE_SIZE);
    printf("read_c0_ecc()=0x%x\n", read_c0_ecc());
    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE / 16);
         i += (CACHELINE_SIZE)) {
#if 0
        for (j = 0; j < 4; j++) {
            dw = (j << 3);
            cache_op(Index_Load_Tag_I, i | dw); //i add DW to cache line  index
            back_to_back_c0_hazard();
            printf("[Line %d][dw %d]datahi: 0x%x, datalo: 0x%x\n", cnt, j,
                   read_c0_idatalo(), read_c0_idatahi());

        }
#else
        cache_op(Index_Load_Tag_I, i);  //i add DW to cache line  index
        back_to_back_c0_hazard();
        printf("[Line %d][dw %d]datahi: 0x%x, datalo: 0x%x\n", cnt, j,
               read_c0_idatalo(), read_c0_idatahi());
#endif
        cnt++;
    }



    return;
}


/** DCACHE ***************************************************DCACHE **************/
/** DCACHE ***************************************************DCACHE **************/
int DCACHE_SETS = 256;

#define DWAY_VERIFY 1
#define DTAG_VERIFY 1

static void dcache_data_verify(u32 index, u32 dw, u32 expected_data);
#if 0
static void dcache_w(unsigned int tag, unsigned int way, unsigned int data)
{
    int i, j;
    int dw = 0;
    write_c0_ddatalo(data);
    back_to_back_c0_hazard();

    write_c0_dtaglo(tag);
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        write_c0_ecc(read_c0_ecc() | (1 << 29));        //waybit 29 
        back_to_back_c0_hazard();
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            cache_op(Index_Store_Data_D, i | dw);
        }
#ifdef DTAG_VERIFY
        write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
        back_to_back_c0_hazard();
        cache_op(Index_Store_Tag_D, i);
#endif
    }
#if DWAY_VERIFY
    //way array     
    write_c0_dtaglo(way);
    write_c0_ecc(read_c0_ecc() | (1 << 29));
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Store_Tag_D, i);
    }
    write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
#endif
    back_to_back_c0_hazard();
    _mips_sync();


    return;
}

#endif
static void dcache_data_w(unsigned int data)
{
    int i, j;
    int dw = 0;
    write_c0_ddatalo(data);
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        write_c0_ecc(read_c0_ecc() | (1 << 29));        //waybit 29 
        back_to_back_c0_hazard();
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            cache_op(Index_Store_Data_D, i | dw);
        }
    }

    back_to_back_c0_hazard();
    _mips_sync2();


    return;
}

static void dcache_data_read_compare(unsigned int data)
{
    unsigned int i, j;
    int dw = 0;

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            cache_op(Index_Load_Tag_D, i | dw); //i add DW to cache line  index
            back_to_back_c0_hazard();

            dcache_data_verify(i, dw, data);
        }
    }
    return;

}

static void dcache_data_test(int loops)
{
    int i;
    int n = 0;
    printf("Dache DataRAM Test\n");

    for (n = 0; n < loops; n++) {
        printf("   DCACHE: DataRAM TEST(%d):\n", n);
        cache_error = 0;
        for (i = 0; i < PATTERN_NUM; i++) {
            dcache_data_w(patterns[i][2]);
            dcache_data_read_compare(patterns[i][2]);
        }
    }
}



static void dcache_data_verify(u32 index, u32 dw, u32 expected_data)
{
    unsigned int datalo;
    unsigned long clinze = 0;
    unsigned long ori_dw = 0;

    //DATA
    datalo = read_c0_ddatalo();

    clinze = (index - CKSEG0) / CACHELINE_SIZE;
    ori_dw = dw >> 2;
    if (expected_data != datalo) {
        printf
            ("DCacheLine(%lu), way/index/dw: %lu / %lu / %lu , [ DataLo ]: 0x%08x, expected: 0x%08x\n",
             clinze, clinze / DCACHE_SETS, clinze % DCACHE_SETS, ori_dw,
             datalo, expected_data);
    }
}

#ifdef DTAG_VERIFY
#define DTAG_VALID_MASK  0xfffff0e0
static void dcache_tag_verify(u32 index, u32 expected_tag)
{
    unsigned int tag = 0;
    unsigned int vailed_tag_pn;
    unsigned long clinze = (index - CKSEG0) / CACHELINE_SIZE;


    write_c0_dtaglo(0);
    back_to_back_c0_hazard();
    vailed_tag_pn = (expected_tag & DTAG_VALID_MASK);

    cache_op(Index_Load_Tag_D, index);  //i add DW to cache line  index
    back_to_back_c0_hazard();

    tag = read_c0_dtaglo();

    if (vailed_tag_pn != tag) {
        printf
            ("DCacheLine(%lu), way/index: %lu / %lu , [ DTag      ]: 0x%08x, expected: 0x%08x, pattern: 0x%08x\n",
             clinze, clinze / DCACHE_SETS, clinze % DCACHE_SETS, tag,
             vailed_tag_pn, expected_tag);
    }
}

static void dcache_tag_w(u32 tag_data)
{
    int i;

    write_c0_dtaglo(tag_data);
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF);
        back_to_back_c0_hazard();
        cache_op(Index_Store_Tag_D, i);
    }

    back_to_back_c0_hazard();
    _mips_sync2();


    return;

}

static void dcache_tag_read_compare(u32 tag_data)
{
    unsigned int i;

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        //TAG
        dcache_tag_verify(i, tag_data);
    }

    return;

}

static void dcache_tag_test(int loops)
{
    int i;
    int n = 0;
	
    printf("-------- Dache TagRAM Test -------------------\n");

    for (n = 0; n < loops; n++) {
        printf("   DCACHE:  TagRAM TEST(%d):\n", n);
        cache_error = 0;
        for (i = 0; i < PATTERN_NUM; i++) {
            dcache_tag_w(patterns[i][0]);
            dcache_tag_read_compare(patterns[i][0]);
        }
    }
}


#endif
#if DWAY_VERIFY
#define DWAY_VALID_MASK 0x000ffce0
static void dcache_way_verify(u32 expected_way)
{

    int i;
    unsigned int way_r;
    unsigned int vailed_way_pn;
    unsigned long clinze;

    vailed_way_pn = (expected_way & DWAY_VALID_MASK);
    // WAY ARRAY
    write_c0_ecc(read_c0_ecc() | (1 << 29));
    back_to_back_c0_hazard();
    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Load_Tag_D, i);
        back_to_back_c0_hazard();
        way_r = read_c0_dtaglo();
        clinze = (i - CKSEG0) / CACHELINE_SIZE;
        if (way_r != vailed_way_pn) {
            printf
                ("DCacheLine(%lu), way/index: %lu / %lu , [ Way      ]: 0x%08x, expected: 0x%08x, pattern: 0x%08x\n",
                 clinze, clinze / DCACHE_SETS, clinze % DCACHE_SETS, way_r,
                 vailed_way_pn, expected_way);
        }
    }
    write_c0_ecc(read_c0_ecc() & ~(1 << 29));
    back_to_back_c0_hazard();



}



static void dcache_way_w(u32 way_data)
{
    int i;

    write_c0_dtaglo(way_data);
    back_to_back_c0_hazard();

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        write_c0_ecc(read_c0_ecc() | (1 << 29));        //waybit 1
        back_to_back_c0_hazard();
        cache_op(Index_Store_Tag_D, i);
    }

    back_to_back_c0_hazard();
    _mips_sync2();


    return;

}

static void dcache_way_read_compare(u32 way_data)
{

    dcache_way_verify(way_data);

    return;

}

static void dcache_way_test(int loops)
{
    int i;
    int n = 0;
    printf("Dache WayRAM Test\n");

    for (n = 0; n < loops; n++) {
        printf("   DCACHE:  WayRAM TEST(%d):\n", n);
        cache_error = 0;
        for (i = 0; i < PATTERN_NUM; i++) {
            dcache_way_w(patterns[i][1]);
            dcache_way_read_compare(patterns[i][1]);
        }
    }
}






#endif
#if 0
static void dcache_verify(unsigned int tag_data, unsigned int data,
                          unsigned int way_data)
{
    unsigned int i, j;
    int dw = 0;

    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            cache_op(Index_Load_Tag_D, i | dw); //i add DW to cache line  index
            back_to_back_c0_hazard();

            dcache_data_verify(i, dw, data);
        }
        //TAG
#ifdef DTAG_VERIFY
        dcache_tag_verify(i, tag_data);
#endif
    }
#if DWAY_VERIFY
    dcache_way_verify(way_data);
#endif
    return;

}
#endif


static void dcache_test(int loops)
{
    int i;
    printf("Dache DCACHE_SETS=%d\n", DCACHE_SETS);
    printf(" Dache Test : ");
	printf(" DataRAM ");
#ifdef DTAG_VERIFY
    printf(" TagRAM ");
#endif
#ifdef DWAY_VERIFY
    printf(" WayRAM");
#endif
   printf("\n");
   dcache_disable();
   
    for (i = 0; i < PATTERN_NUM; i++) {
        printf("<pattern(%d):0x%08x,0x%08x,0x%08x>\n", i, patterns[i][0],
               patterns[i][1], patterns[i][2]);
    }


    dcache_data_test(loops);
#ifdef DTAG_VERIFY
    dcache_tag_test(loops);
#endif
#ifdef DWAY_VERIFY
    dcache_way_test(loops);
#endif

#if 0
    int n = 0;
    for (n = 0; n < loops; n++) {
        printf("DCACHE_TEST LOOP(%d):\n", n);
        cache_error = 0;
        for (i = 0; i < PATTERN_NUM; i++) {
            dcache_w(patterns[i][0], patterns[i][1], patterns[i][2]);
            dcache_verify(patterns[i][0], patterns[i][1], patterns[i][2]);
        }
    }
#endif
}


static void dcache_write(int data)
{
    int i;
    int j;
    int dw = 0;
    write_c0_ddatalo(data);
    back_to_back_c0_hazard();
    printf("0x%x, 0x%x, 0x%x\n", read_c0_ddatalo(), read_c0_ddatalo(),
           CKSEG0);
    write_c0_ecc(read_c0_ecc() | (1 << 29));    //bit 29 , way bit to enable.
    back_to_back_c0_hazard();
    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            //i add DW to cache line 
            cache_op(Index_Store_Data_D, i | dw);
        }

    }
    write_c0_ecc(read_c0_ecc() | 0xDFFFFFFF);
    back_to_back_c0_hazard();
    _mips_sync();
}

static void dcache_read(void)
{
    unsigned int i;
    int j = 0;
    int dw = 0;
    int cnt = 0;
    printf("CACHELINE_SIZE : %d ,  DCACHE_SIZE: %d\n", CACHELINE_SIZE,
           DCACHE_SIZE);
    printf("read_c0_ecc()=0x%x\n", read_c0_ecc());
    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += (CACHELINE_SIZE)) {
        printf("[Line %d]:", cnt);
        for (j = 0; j < 8; j++) {
            dw = (j << 2);
            cache_op(Index_Load_Tag_D, i | dw); //i add DW to cache line  index
            back_to_back_c0_hazard();
            printf("< [dw %d]datalo: 0x%x >", j, read_c0_ddatalo());

        }
        printf("\n");
        cnt++;
    }



    return;
}




/*******************************Common*********************/

static void show_icache_tag(void)
{
    unsigned int res;
    int count = 0, i;
    for (i = CKSEG0; i < (CKSEG0 + ICACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Load_Tag_I, i);
        back_to_back_c0_hazard();
        res = read_c0_taglo();
        printf("IcaheLine(%d) - Tag:0x%x\n", count, res);
        count++;
    }
}

static void show_dcache_tag(void)
{
    unsigned int res;
    int count = 0, i;
    for (i = CKSEG0; i < (CKSEG0 + DCACHE_SIZE); i += CACHELINE_SIZE) {
        cache_op(Index_Load_Tag_D, i);
        back_to_back_c0_hazard();
        res = read_c0_dtaglo();
        printf("DcaheLine(%d) - Tag:0x%x\n", count, res);
        count++;
    }




}

static int cache_status(void)
{
    unsigned int res;

    res = read_c0_config();
    printf("config: 0x%x\n", res);
    res &= 0x7;
    if (res <= 0x2 || res == 0x7) {
        return 0;
    } else {
        return 1;
    }
}

int icache_status(void)
{
    return cache_status();
}

int dcache_status(void)
{
    return cache_status();
}

#if 0
void __weak flush_icache(void)
{
    /* please define arch specific flush_icache */
    puts("No arch specific flush_icache available!\n");

}
#endif

static void conf_k0_switch(int flag)
{
    unsigned int res;
    res = read_c0_config();
    printf("config: 0x%x\n", res);
    if ((read_c0_config() & 0x7) > 0x2) {


        __flush_icache();
        __flush_dcache();



    }


    switch (flag) {
        case 0:
            change_c0_config(0x7, 0x2);
            break;
        case 1:
            change_c0_config(0x7, 0x3);
            break;
        default:
            break;
    }
    return;
}




void icache_disable(void)
{
    unsigned long tmp;
    tmp = (unsigned long) conf_k0_switch;
    //my_func_ptr = conf_k0_switch ;
    printf(" ------ 0x%lx\n", tmp | 0x20000000);
    my_func_ptr = (void *) (tmp | 0x20000000);


    printf("%s\n", __func__);
    my_func_ptr(0);
}

void icache_enable(void)
{
    unsigned long tmp;
    tmp = (unsigned long) conf_k0_switch;
    //my_func_ptr = conf_k0_switch ;
    printf(" ------ 0x%lx\n", tmp | 0x20000000);
    my_func_ptr = (void *) (tmp | 0x20000000);
    printf("%s\n", __func__);
    my_func_ptr(1);
}

void dcache_disable(void)
{
    unsigned long tmp;
    tmp = (unsigned long) conf_k0_switch;
    //my_func_ptr = conf_k0_switch ;
    printf(" ------ 0x%lx\n", tmp | 0x20000000);
    my_func_ptr = (void *) (tmp | 0x20000000);
    printf("%s\n", __func__);
    my_func_ptr(0);


}

void dcache_enable(void)
{
    unsigned long tmp;
    tmp = (unsigned long) conf_k0_switch;
    //my_func_ptr = conf_k0_switch ;
    printf(" ------ 0x%lx\n", tmp | 0x20000000);
    my_func_ptr = (void *) (tmp | 0x20000000);

    printf("%s\n", __func__);
    my_func_ptr(1);
}



#define inline_ascii_value(c, base) ({    \
                         char __ascii_c=c; \
                         int __ascii_base=base; \
                         int __ascii_r; \
                         if ((__ascii_c<='9') && (__ascii_c>='0')) { \
                                 __ascii_r=__ascii_c-'0'; \
                         } else { \
                                 __ascii_c|=32; \
                                 __ascii_r=((__ascii_base==16)&&(__ascii_c<='f')&&(__ascii_c>='a'))? \
                                         (__ascii_c-'a'+10): \
                                         -1; \
                         } \
                         __ascii_r; \
                 })



static s32_t atoi(const char *v)
{
    const char *orig_v = v;
    int base = 10, d, r = 0;
    if (*v == '\0')
        return 0;
    if ((v[0] == '0') && ((v[1] & ~32) == 'X')) {
        v += 2;
        base = 16;
    }
    while ((d = inline_ascii_value(*(v++), base)) >= 0) {
        r *= base;
        r += d;
    }

    if ((orig_v[0] != '0') && (r == 0)) {
        r = -1;
    }
    return r;
}

int do_icache(cmd_tbl_t * cmdtp, int flag, int argc, char *const argv[])
{

    unsigned long tmp;
    ICACHE_SETS = (ICACHE_SIZE / CACHELINE_SIZE) / CACHE_WAY;
    switch (argc) {

        case 3:                /* test xxxx */
            switch (parse_argv(argv[1])) {
                case 4:
                    icache_test(parse_argv_int(argv[2]));
                    break;
                default:
                    break;
            }

            break;
        case 2:                /* on / off     */
            switch (parse_argv(argv[1])) {
                case 0:
                    icache_disable();
                    break;
                case 1:
                    icache_enable();
                    break;
                case 2:
                    flush_icache();
                    break;
                case 3:
                    show_icache_tag();
                    break;
                case 4:
                    icache_test(10);
                    break;
                case 5:
                    tmp = (unsigned long) icache_write;
                    my_func_ptr = (void *) (tmp | 0x20000000);
                    my_func_ptr(0);
                    break;
                case 6:
                    icache_read();
                    break;
            }
            /* FALL TROUGH */
        case 1:                /* get status */
            printf("Instruction Cache is %s\n",
                   icache_status()? "ON" : "OFF");
            return 0;
        default:
            return cmd_usage(cmdtp);
    }
    return 0;
}

#if 0
void __weak flush_dcache(void)
{
    puts("No arch specific flush_dcache available!\n");
    /* please define arch specific flush_dcache */
}
#endif
int do_dcache(cmd_tbl_t * cmdtp, int flag, int argc, char *const argv[])
{
    unsigned long tmp;
    DCACHE_SETS = (DCACHE_SIZE / CACHELINE_SIZE) / CACHE_WAY;
    switch (argc) {

        case 3:                /* test xxxx */
            switch (parse_argv(argv[1])) {
                case 4:
                    dcache_test(parse_argv_int(argv[2]));
                    break;
                default:
                    break;
            }

            break;
        case 2:                /* on / off     */
            switch (parse_argv(argv[1])) {
                case 0:
                    dcache_disable();
                    break;
                case 1:
                    dcache_enable();
                    break;
                case 2:
                    flush_dcache();
                    break;
                case 3:
                    show_dcache_tag();
                    break;
                case 4:
                    dcache_test(10);
                    break;
                case 5:        //dcache_write
                    tmp = (unsigned long) dcache_write;
                    my_func_ptr = (void *) (tmp | 0x20000000);
                    my_func_ptr(0);
                    break;
                case 6:
                    dcache_read();
                    break;
            }
            /* FALL TROUGH */
        case 1:                /* get status */
            printf("Instruction Cache is %s\n",
                   icache_status()? "ON" : "OFF");
            return 0;
        default:
            return cmd_usage(cmdtp);
    }
    return 0;

}

static int parse_argv(const char *s)
{
    if (strcmp(s, "flush") == 0) {
        return (2);
    } else if (strcmp(s, "on") == 0) {
        return (1);
    } else if (strcmp(s, "off") == 0) {
        return (0);
    } else if (strcmp(s, "showtag") == 0) {
        return (3);
    } else if (strcmp(s, "test") == 0) {
        return (4);
    } else if (strcmp(s, "write") == 0) {
        return (5);
    } else if (strcmp(s, "read") == 0) {
        return (6);
    }
    return (-1);
}

static int parse_argv_int(const char *s)
{
    int input;
    input = (atoi(s));
    printf("Input: %d\n", input);
    return input;
}

/*********************************************************************************/
//For l2cache
/******************************************************************************/
/*
 * R4000SC and R4400SC-specific cacheops
 */
#define Index_Invalidate_SI		0x02
#define Index_Writeback_Inv_SD		0x03
#define Index_Load_Tag_SI		0x06
#define Index_Load_Tag_SD		0x07
#define Index_Store_Tag_SI		0x0A
#define Index_Store_Tag_SD		0x0B
#define Create_Dirty_Excl_SD		0x0f
#define Hit_Invalidate_SI		0x12
#define Hit_Invalidate_SD		0x13
#define Hit_Writeback_Inv_SD		0x17
#define Hit_Writeback_SD		0x1b
#define Hit_Set_Virtual_SI		0x1e
#define Hit_Set_Virtual_SD		0x1f




#define _mips_sync() __asm__ __volatile__ ("sync  0x3" : : : "memory")

#define read_c0_staglo()        __read_32bit_c0_register($28, 4)
#define write_c0_staglo(val)    __write_32bit_c0_register($28, 4, val)


#define read_c0_sdatalo()        __read_32bit_c0_register($28, 5)
#define write_c0_sdatalo(val)    __write_32bit_c0_register($28, 5, val)

#define read_c0_sdatahi()        __read_32bit_c0_register($29, 5)
#define write_c0_sdatahi(val)    __write_32bit_c0_register($29, 5, val)


//#define TAG_RESERVED_BITS_MASK (0x1F <<9) //bit14 is reserved , MIPS_IA Spec issue?
#define TAG_RESERVED_BITS_MASK (0x3F <<9)
#define TAG_PARITY_BITS_MASK 0x1F
#define TAG_TOTAL_PARITY_MASK (0x1<<8)
#define TAG_VALID_MASK   (~(TAG_RESERVED_BITS_MASK) & ~(TAG_PARITY_BITS_MASK)) & ~(TAG_TOTAL_PARITY_MASK)

#define WAY_RESERVED_BITS 0x1FF
#define WAY_DP_BITS_MASK 0xFF000000
#define WAY_VALID_MASK    ~(WAY_RESERVED_BITS) & ~(WAY_DP_BITS_MASK)


#define DO_TAG_WAY_DATA 1 
#define WAY_VERIFY 0

#define L2_PATTERN_NUM 5

u32 l2_patterns[L2_PATTERN_NUM][3]= {{0,0,0}, 
	                          {0xffffffff, 0xffffffff,0xffffffff}, 
	                          {0xffff0000, 0xffff0000,0xffff0000},
  	    					  {0x0000FFFF, 0x0000FFFF,0x0000FFFF},
                              {0xAAAAAAAA, 0xAAAAAAAA,0xAAAAAAAA}};

unsigned int scache_size = 0;
unsigned int scache_line_sz = 0;
unsigned int scache_ways = 0;
unsigned int scache_sets = 1024;

unsigned int scache_loop =0;

#define L2PRINTF(fmt, args...) printf( "\n L2 LOOP(%u): " fmt, scache_loop, ## args)

static int get_l2cache_inf(void){

	unsigned int config2;
	unsigned int tmp;
	config2 = read_c0_config2();
	if (config2 & (1 << 12))
			return 0;
				      
       /* L2-Cache */
	tmp = (config2 >> 4) & 0x0f;
	if (0 < tmp && tmp <= 7)
		scache_line_sz = 2 << tmp;
	else
		return 0;		
		
	tmp = (config2 >> 8) & 0x0f;
	if (0 <= tmp && tmp <= 7)
		scache_sets = 64 << tmp;
	else
		return 0;

	tmp = (config2 >> 0) & 0x0f;
	if (0 <= tmp && tmp <= 7)
		scache_ways = tmp + 1;
	else
		return 0;
	

	scache_size = scache_sets *  scache_ways * scache_line_sz;

	printf("L2cache LineSize=%u, Sets=%u, Ways=%u, CacheSize=%u\n", scache_line_sz, scache_sets, scache_ways, scache_size);
	return 0;
}



static void l2cache_write(u32 tag, u32 data){
	int i,j;
    int dw=0;
	write_c0_staglo(tag);
	write_c0_sdatalo(data);
	write_c0_sdatahi(data);
	back_to_back_c0_hazard();
	for(i=CKSEG0;  i < (CKSEG0 + scache_size); i +=  scache_line_sz){
		for(j=0;j<4;j++){
			dw = (j<<3);
			 //i add DW to cache line 
			cache_op(Create_Dirty_Excl_SD, i | dw);
        }
			cache_op(Index_Store_Tag_SD, i );
	}

	back_to_back_c0_hazard();
	_mips_sync();
	

	return;
}

/***   Verify ***************************/
/*******************Verify **************************
*
*
**************************************************/
static void l2cache_tag_verify(u32 index, u32 expected_tag){
	unsigned int tag = 0;
	unsigned int vailed_tag_pn ;
	unsigned long	clinze = (index - CKSEG0)/scache_line_sz;
	
	
	write_c0_staglo(0);
	back_to_back_c0_hazard();
	vailed_tag_pn = (expected_tag & TAG_VALID_MASK);
	
	cache_op(Index_Load_Tag_SD, index );  //i add DW to cache line  index
	back_to_back_c0_hazard();
	
	tag = read_c0_staglo();
	
	if(tag & TAG_PARITY_BITS_MASK){
			L2PRINTF("L2CacheLine(%lu), way/index: %lu / %lu ,Tag: 0x%x, TAG Parity bit0~4 != 0.\n", clinze, clinze/scache_sets, clinze%scache_sets,  tag );
	}
	if(tag & TAG_RESERVED_BITS_MASK){
			L2PRINTF("L2CacheLine(%lu), way/index: %lu / %lu ,Tag: 0x%08x,  TAG Reserved bit9~14 !=0.\n", clinze, clinze/scache_sets, clinze%scache_sets,  tag );	
	}
	if(tag & TAG_TOTAL_PARITY_MASK){
			L2PRINTF("L2CacheLine(%lu), way/index: %lu / %lu ,Tag: 0x%08x,  TAG TP bit8 !=0.\n", clinze, clinze/scache_sets, clinze%scache_sets,  tag );	
	}
	tag = tag & TAG_VALID_MASK;
	if( vailed_tag_pn != tag ){
			L2PRINTF("CacheLine(%lu), way/index: %lu / %lu , Tag: 0x%08x, expected: 0x%08x, pattern: 0x%08x\n",  clinze, clinze/scache_sets, clinze%scache_sets,  tag , vailed_tag_pn, expected_tag);
	}

}

static void l2cache_data_verify(u32 index, u32 dw, u32 expected_data){
	unsigned int datalo, datahi;
	unsigned long clinze = 0;
	unsigned long  ori_dw = 0;
	
			//DATA
	datalo = read_c0_sdatalo();
	datahi = read_c0_sdatahi();
	clinze = (index - CKSEG0)/scache_line_sz;
	ori_dw =  dw>>3;
	if(expected_data != datalo){
		//DEBUG_data("LOOP(%llu), index:0x%lx, dw:0x%lx ,datalo: 0x%08x, expected: 0x%08x\n", atomic64_read(&loop_count), index, dw, datalo, expected_data);
		L2PRINTF("L2Cacheline(%lu), way/index/dw: %lu / %lu / %lu , datalo: 0x%08x, expected: 0x%08x\n", clinze, clinze/scache_sets, clinze%scache_sets, ori_dw, datalo, expected_data);
	}
	if(expected_data != datahi){
		L2PRINTF("L2Cacheline(%lu), way/index/dw: %lu / %lu / %lu , datahi: 0x%08x, expected: 0x%08x\n", clinze, clinze/scache_sets, clinze%scache_sets, ori_dw, datahi, expected_data);
	}
	
	
	
}

static void l2cache_way_verify(u32 expected_way){

	int i;
	unsigned int way_r;
	unsigned int vailed_way_pn ;
	unsigned int	clinze = 0;

	vailed_way_pn = (expected_way & WAY_VALID_MASK);
	// WAY ARRAY
	write_c0_ecc(read_c0_ecc()| (1<<29));
	back_to_back_c0_hazard();
	for(i=CKSEG0;  i < (CKSEG0 + scache_size); i +=  scache_line_sz){	
		cache_op(Index_Load_Tag_SD,i);
		back_to_back_c0_hazard();
		way_r = read_c0_staglo();
		
		clinze = (i - CKSEG0)/scache_line_sz;
		
		if(way_r & WAY_DP_BITS_MASK){
				L2PRINTF("L2CacheLine(%u) Way: 0x%08x, WAY DP bits != 0.\n", clinze, way_r );
		}
		if(way_r & WAY_RESERVED_BITS){
				L2PRINTF("L2CacheLine(%u) Way: 0x%08x, WAY RESERVED bits != 0.\n", clinze, way_r );
		}
		way_r = way_r & WAY_VALID_MASK;
        if( way_r !=  vailed_way_pn){
			L2PRINTF("L2 Way array(%u): 0x%08x, expected: 0x%08x\n", clinze, way_r , vailed_way_pn);
		}
	}
	write_c0_ecc(read_c0_ecc()& ~(1<<29));
	back_to_back_c0_hazard();
	
}

static void l2cache_tag_data_verify( unsigned int tag_data, unsigned int data){
	unsigned int i,j;
    int dw = 0;
	
	for(i=CKSEG0;  i < (CKSEG0 + scache_size); i +=  scache_line_sz){
		for(j=0; j<4; j++){
			dw = (j<<3);
			cache_op(Index_Load_Tag_SD, i | dw);  //i add DW to cache line  index
			back_to_back_c0_hazard();

			l2cache_data_verify(i, dw, data);
		}
		//TAG
		l2cache_tag_verify(i, tag_data);
	}

	return;
}

static void l2_way_write(unsigned int waydata){
	int i;
	write_c0_staglo(waydata);
	write_c0_ecc(read_c0_ecc()| (1<<29));
	back_to_back_c0_hazard();
    
	for(i=CKSEG0;  i < (CKSEG0 + scache_size); i +=  scache_line_sz){
		cache_op(Index_Store_Tag_SD,i);
	}
	write_c0_ecc(read_c0_ecc() & 0xDFFFFFFF );
	
	
}

static void l2_way_test(int patnum){
	
    l2_way_write(l2_patterns[patnum][2]);

	l2cache_way_verify(l2_patterns[patnum][2]);

	
}



/* TAG and Data */
static void l2_tag_data_test(int patnum){
	l2cache_write(l2_patterns[patnum][0]  , l2_patterns[patnum][1]);
	l2cache_tag_data_verify(l2_patterns[patnum][0] , l2_patterns[patnum][1]);	
}


/*********Main *************************/

static void l2cache_test(int loops){
	int j;
	printf("L2Cache RAMs Test %d:\n", loops);
	for(scache_loop=0; scache_loop<loops; scache_loop++){
	printf(".");
		 for(j=0;j<L2_PATTERN_NUM;j++){
			l2_tag_data_test(j);
			l2_way_test(j);
		 }
	}
}

static int l2cache_status(void){
	printf("GCR_BASE: 0x%08x\n", *(int *)0xBFBF8008);
	printf("Config2: 0x%08x\n", read_c0_config2());
	
	printf(" - L2_Bypass: %s\n",  read_c0_config2() & (1<<12) ? "Enable" : "Disable");
	
	return 0;
}

int do_l2cache(cmd_tbl_t * cmdtp, int flag, int argc, char *const argv[]){
    get_l2cache_inf();
    switch (argc) {

        case 3:                /* test xxxx */
            switch (parse_argv(argv[1])) {
                case 4:
                    l2cache_test(parse_argv_int(argv[2]));
                    break;
                default:
                    break;
            }

            break;
        case 2:                /* on / off     */
            switch (parse_argv(argv[1])) {
                case 0:
                    //l2cache_disable();
                    break;
                case 1:
                    //l2cache_enable();
                    break;
                case 2:
                    //flush_l2cache();
                    break;
                case 3:
                    //show_l2cache_tag();
                    break;
                case 4:
                    l2cache_test(10);
                    break;
                case 5:      
                     break;
                case 6:
                    break;
            }
            /* FALL TROUGH */
        case 1:                /* get status */
                   l2cache_status();
            return 0;
        default:
            return cmd_usage(cmdtp);
    }

    return 0;

	
}


/***************************************************************************/
U_BOOT_CMD(icache, 3, 1, do_icache,
           "enable or disable instruction cache",
           "[on, off, flush, test]\n"
           "    - enable, disable, or flush instruction cache");

U_BOOT_CMD(dcache, 3, 1, do_dcache,
           "enable or disable data cache",
           "[on, off, flush, test]\n"
           "    - enable, disable, or flush data (writethrough) cache");

U_BOOT_CMD(l2cache, 3, 1, do_l2cache,
           "Test l2data cache",
           "[test]\n"
           " test L2 cache");
		   
		   