/*
 * Driver for NAND support, Rick Bronson
 * borrowed heavily from:
 * (c) 1999 Machine Vision Holdings, Inc.
 * (c) 1999, 2000 David Woodhouse <dwmw2@infradead.org>
 *
 * Ported 'dynenv' to 'nand env.oob' command
 * (C) 2010 Nanometrics, Inc.
 * 'dynenv' -- Dynamic environment offset in NAND OOB
 * (C) Copyright 2006-2007 OpenMoko, Inc.
 * Added 16-bit nand support
 * (C) 2004 Texas Instruments
 *
 * Copyright 2010 Freescale Semiconductor
 * The portions of this file whose copyright is held by Freescale and which
 * are not considered a derived work of GPL v2-only code may be distributed
 * and/or modified under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 */

#include <common.h>
#if defined(CONFIG_CMD_MTDPARTS)
#include <linux/mtd/mtd.h>
#include <jffs2/load_kernel.h>
#endif
#include <command.h>
//#include <watchdog.h>
//#include <malloc.h>
//#include <asm/byteorder.h>
//#include <jffs2/jffs2.h>
//#include <nand.h>

extern void spi_nand_info(void) ;
extern int spi_nand_read_write(uint32_t offset, uint32_t length, u_char *buffer, char opt);
extern int spi_nand_write_ecc(uint32_t offset, uint32_t length, u_char *buffer);
extern int spi_nand_read_ecc(uint32_t offset, uint32_t length, u_char *buffer);
extern int spi_nand_erase(uint32_t offset, uint32_t length);

#if defined(CONFIG_CMD_MTDPARTS)

/* partition handling routines */
int mtdparts_init(void);
int id_parse(const char *id, const char **ret_id, u8 *dev_type, u8 *dev_num);
int find_dev_and_part(const char *id, struct mtd_device **dev,
		      u8 *part_num, struct part_info **part);
#endif

#ifdef CONFIG_CMD_MTDPARTS
static int get_part(const char *partname, int *idx, loff_t *off, loff_t *size)
{
	struct mtd_device *dev;
	struct part_info *part;
	u8 pnum;
	int ret;

	ret = mtdparts_init();
	if (ret)
		return ret;

	ret = find_dev_and_part(partname, &dev, &pnum, (struct part_info **)&part);
	if (ret)
		return ret;

	if (dev->id->type != MTD_DEV_TYPE_NAND) {
		puts("not a NAND device\n");
		return -1;
	}

	*off = part->offset;
	*size = part->size;
	*idx = dev->id->num;

	return 0;
}
#endif

#if defined(CONFIG_CMD_MTDPARTS)
static int
do_read_part(int argc, char * const argv[])
{
	loff_t off, size;
	uint32_t arg_addr;
	uint32_t arg_size;
	int idx;
	int ret = 0;
	if (argc < 4) {
		return -1;
	}

	/* spi_nand read.part address partition [size] */
	/* 0        1         2       3          4 */
	arg_addr = (ulong)simple_strtoul(argv[2], NULL, 16);
	if (arg_addr<0x80000000 || arg_addr>0xC0000000) { // FIXME?
		puts("EE: incorrect addr.\n");
		return 0;
	}

	ret = get_part(argv[3], &idx, &off, &size);
	if (ret) {
		return ret;
	}
	printf("Partition: %s(%d), offset: %lx, size: %lx\n", argv[3], idx, (unsigned long)off, (unsigned long)size);
	if (argc > 4) {
		arg_size = (ulong)simple_strtoul(argv[4], NULL, 16);
		if (arg_size > size) {
			printf("WARN: Read size bigger than partition size\n");
		}
	} else {
		arg_size = (uint32_t)size;
	}

	ret = spi_nand_read_ecc(off, arg_size, (u_char *)arg_addr);

	return ret;
}

static int
do_write_part(int argc, char * const argv[])
{
	loff_t off, size;
	int idx;
	uint32_t arg_addr;
	uint32_t arg_size;
	int ret = 0;
	if (argc < 5) {
		return -1;
	}

	/* spi_nand write.part address partition size */
	/* 0        1          2       3         4*/
	arg_addr = (ulong)simple_strtoul(argv[2], NULL, 16);
	if (arg_addr<0x80000000 || arg_addr>0xC0000000) { // FIXME?
		puts("EE: incorrect addr.\n");
		return 0;
	}

	ret = get_part(argv[3], &idx, &off, &size);
	if (ret) {
		return ret;
	}
	printf("Partition: %s(%d), offset: %lx, size: %lx\n", argv[3], idx, (unsigned long)off, (unsigned long)size);
	arg_size = (ulong)simple_strtoul(argv[4], NULL, 16);
	if (arg_size > size) {
		printf("EE: Write size bigger than partition size\n");
		return -1;
	}
	if (0 == arg_size && '0' != argv[4][0]) {
		printf("EE: Size Error\n");
		return -1;
	}

	ret = spi_nand_write_ecc(off, arg_size, (u_char *)arg_addr);

	return ret;
}

static int
do_erase_part(int argc, char * const argv[])
{
	loff_t off, size;
	int idx;
	uint32_t arg_size;
	int ret = 0;
	if (argc < 3) {
		return -1;
	}

	ret = get_part(argv[2], &idx, &off, &size);
	if (ret) {
		return ret;
	}
	printf("Partition: %s(%d), offset: %lx, size: %lx\n", argv[2], idx, (unsigned long)off, (unsigned long)size);

	if (argc > 3) {
		arg_size = (ulong)simple_strtoul(argv[3], NULL, 16);
		if (arg_size > size) {
			printf("EE: Erase size bigger than partition size\n");
			return -1;
		}
    if (0 == arg_size && '0' != argv[3][0]) {
			printf("EE: Size Error\n");
      return -1;
    }
	} else {
		arg_size = (uint32_t)size;
	}

	ret = spi_nand_erase(off, arg_size);

	return ret;
}
#endif

int do_spi_nand(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	int ret = 0;
	uint32_t addr;
	uint32_t off, size;
	char *cmd, *s;

	/* at least two arguments please */
	if (argc<2)
		goto usage;

	cmd = argv[1];
	if (0==strcmp(cmd, "info")) {
		spi_nand_info();
		return 0;
#if defined(CONFIG_CMD_MTDPARTS)
	} else if (0 == strncmp(cmd, "read.part", strlen("read.part"))) {
		ret = do_read_part(argc, argv);
		if(0==ret) puts("done\n");
		return ret;
	} else if (0 == strncmp(cmd, "write.part", strlen("write.part"))) {
		ret = do_write_part(argc, argv);
		if(0==ret) puts("done\n");
		return ret;
	} else if (0 == strncmp(cmd, "erase.part", strlen("erase.part"))) {
		ret = do_erase_part(argc, argv);
		if(0==ret) puts("done\n");
		return ret;
#endif
	} else if ((0==strncmp(cmd, "read", 4)) || 0==strncmp(cmd, "write", 5)) {
		char opt;   // 0: read, 1: write
		if (argc<5) goto usage;
        
		opt=(0==strncmp(cmd, "read", 4))?0:1; 
		addr = (ulong)simple_strtoul(argv[2], NULL, 16);
		if (addr<0x80000000 || addr>0xC0000000) { // FIXME?
			puts("EE: incorrect addr.\n");
			return 0;
		}
		off = (ulong)simple_strtoul(argv[3], NULL, 16);
		size = (ulong)simple_strtoul(argv[4], NULL, 16);
		//printf("DD: addr 0x%x, off 0x%x, size 0x%x\n", addr, off, size);
		s = strchr(cmd, '.');
		if (s && !strcmp(s, ".raw")) {            
			ret = spi_nand_read_write(off, size, (u_char *)addr, opt);
		} else {
			ret = (opt)?spi_nand_write_ecc(off, size, (u_char *)addr)
				:spi_nand_read_ecc(off, size, (u_char *)addr);
		}
	
		if(0==ret) puts("done\n");
		return ret;
	} else if (0==strncmp(cmd, "erase", 5)) {
		if (argc < 4) {
			goto usage;
		}
		off = (ulong)simple_strtoul(argv[2], NULL, 16);
		size = (ulong)simple_strtoul(argv[3], NULL, 16);
		if (0 == size) {
			goto usage;
		}
		ret = spi_nand_erase(off, size);
		return ret;
	}
usage:
	return cmd_usage(cmdtp);
}

U_BOOT_CMD(
	spi_nand, CONFIG_SYS_MAXARGS, 0, do_spi_nand,
	"SPI-NAND sub-system",
	"info - show available NAND devices\n"
	"spi_nand read - addr off size\n"
	"spi_nand read.raw - addr off size\n"	
#if defined(CONFIG_CMD_MTDPARTS)
	"spi_nand read.part address partition [size]\n"
#endif
	"spi_nand write - addr off size\n"
#if defined(CONFIG_CMD_MTDPARTS)
	"spi_nand write.part address partition size - write to MTD partition\n"
#endif
	"spi_nand write.raw - addr off size\n"	
	"    read/write 'size' bytes starting at offset 'off'\n"
	"               offset ignores 'spare' "
	"    to/from memory address 'addr'.\n"
	"    read.raw/write.raw skip ECC\n"
	"spi_nand erase off size - erase 'size' bytes\n"
#if defined(CONFIG_CMD_MTDPARTS)
	"spi_nand erase.part partition [size] - erase MTD partition\n"
#endif
	"    from offset 'off'\n"
	"    'size' and 'off' are block size alignment\n"
);

// FIXME
// do_spi_nand_boot
//U_BOOT_CMD(snboot, 4, 1, do_nandboot,
//	"boot from NAND device",
//	"[partition] | [[[loadAddr] dev] offset]"
//);

