/*
 * Command for updating vm.img format image (containing uImage and rootfs) into corresponding partition on luna platform.
 * Author: bohungwu@realtek.com
 */

#include <common.h>
#include <malloc.h>
#include <asm/io.h>

enum { NAME_SIZE = 100 }; /* because gcc won't let me use 'static const int' */

/* POSIX tar Header Block, from POSIX 1003.1-1990  */
struct TarHeader
{
                                /* byte offset */
	char name[NAME_SIZE];         /*   0-99 */
	char mode[8];                 /* 100-107 */
	char uid[8];                  /* 108-115 */
	char gid[8];                  /* 116-123 */
	char size[12];                /* 124-135 */
	char mtime[12];               /* 136-147 */
	char chksum[8];               /* 148-155 */
	char typeflag;                /* 156-156 */
	char linkname[NAME_SIZE];     /* 157-256 */
	char magic[6];                /* 257-262 */
	char version[2];              /* 263-264 */
	char uname[32];               /* 265-296 */
	char gname[32];               /* 297-328 */
	char devmajor[8];             /* 329-336 */
	char devminor[8];             /* 337-344 */
	char prefix[155];             /* 345-499 */
	char padding[12];             /* 500-512 (pad to exactly the TAR_BLOCK_SIZE) */
};
typedef struct TarHeader TarHeader;

/* A few useful constants */
#define TAR_MAGIC          "ustar "        /* ustar and a null */
#define TAR_VERSION        " "           /* Be compatable with GNU tar format */
static const int TAR_MAGIC_LEN = 6;
static const int TAR_VERSION_LEN = 2;
static const int TAR_BLOCK_SIZE = 512;

#define TAR_KERNEL_NAME		"uImage"
#define TAR_KERNEL_NAME_SIZE	6

#define TAR_ROOTFS_NAME		"rootfs"
#define TAR_ROOTFS_NAME_SIZE	6

#define TAR_OSGI_NAME		"osgi.img"
#define TAR_OSGI_NAME_SIZE	8

#define TAR_FWUVER_NAME		"fwu_ver"
#define TAR_FWUVER_NAME_SIZE	6
#define TAR_FWUVER_FILE_SIZE	32


#if CONFIG_RTK_USE_ONE_UBI_DEVICE
#define UBI_K0_NAME "ubi_k0"
#define UBI_K1_NAME "ubi_k1"
#define UBI_R0_NAME "ubi_r0"
#define UBI_R1_NAME "ubi_r1"
#define UBI_OSGI_NAME "ubi_osgi"
#endif

int kernel_location = 0;		/* The uImage location in tar file */
int kernel_size = 0;
int rootfs_location = 0;		/* The rootfs location in tar file */
int rootfs_size = 0;
int osgi_location = 0;			/* The osgi location in tar file */
int osgi_size = 0;
char fwu_ver[TAR_FWUVER_FILE_SIZE];	/* fwu_ver content */

#ifdef CONFIG_SINGLE_IMAGE
char *dual_image = NULL;
int image_type=0;
#define DUAL_IMAGE 1
#endif

#define ENV_SW_VERSION0		"sw_version0"
#define ENV_SW_VERSION1		"sw_version1"
#define ENV_SW_CRC0		"sw_crc0"
#define ENV_SW_CRC1		"sw_crc1"

//#define ENV_KERNEL0_CRC	"kernel0_crc"
//#define ENV_KERNEL1_CRC	"kernel1_crc"
//#define ENV_ROOTFS0_CRC	"rootfs0_crc"
//#define ENV_ROOTFS1_CRC	"rootfs1_crc"

#define CMD_BUF_SZ (256)
#define CRC32_STRING_SIZE	32

int check_img_tar(unsigned char *buffer, unsigned int size)
{
	TarHeader *hdr;

	//printf("Is this a img.tar? (size: 0x%08x)\n", size);
	if (size >= TAR_BLOCK_SIZE) {
		hdr = (TarHeader *)buffer;

		//printf("Magic/Version: [%s]/[%x]/[%x]\n", hdr->magic, hdr->version[0], hdr->version[1]);
		//if((strncmp(hdr->magic, TAR_MAGIC, TAR_MAGIC_LEN) == 0) && hdr->version[0] == 0x20 && hdr->version[1] == 0x0)
		//if(strncmp(hdr->magic, TAR_MAGIC TAR_VERSION, TAR_MAGIC_LEN + TAR_VERSION_LEN) == 0) 
		if(strncmp(hdr->magic, TAR_MAGIC TAR_VERSION, TAR_MAGIC_LEN + TAR_VERSION_LEN) == 0)
			return 1;	/* This is a tar header */
	}

	return 0;
}

void scan_img_tar(unsigned char *buffer, unsigned int size)
{
	TarHeader *hdr;
	unsigned int location = 0,  img_size = 0;

	while((location + TAR_BLOCK_SIZE) < size) {
		hdr = (TarHeader *)(buffer + location);

		if(check_img_tar((unsigned char *)hdr, TAR_BLOCK_SIZE)){
			location += sizeof(TarHeader);
			img_size = simple_strtoul(hdr->size, 0, 8);
			printf("File in Tar: %s at 0x%08x (size: 0x%08x)\n", hdr->name, location, img_size);

			/* Find uImage, rootfs and fwu_ver */
			if (strncmp(hdr->name, TAR_KERNEL_NAME, TAR_KERNEL_NAME_SIZE) == 0) {
				kernel_location = location;
				kernel_size = img_size;
			}
			else if (strncmp(hdr->name, TAR_ROOTFS_NAME, TAR_ROOTFS_NAME_SIZE) == 0) {
				rootfs_location = location;
				rootfs_size = img_size;
			}
			else if (strncmp(hdr->name, TAR_FWUVER_NAME, TAR_FWUVER_NAME_SIZE) == 0) {
				//printf("Version Strings Size = %d\n", min((img_size - 1), TAR_FWUVER_FILE_SIZE));
				strncpy(fwu_ver, (char *)(buffer + location), min((img_size - 1), TAR_FWUVER_FILE_SIZE));	/* The last char. is '\n' */
			}
			else if (strncmp(hdr->name, TAR_OSGI_NAME, TAR_OSGI_NAME_SIZE) == 0) {
				osgi_location = location;
				osgi_size = img_size;
			}

			location += img_size;
			location = (location + TAR_BLOCK_SIZE - 1) & ~0x1FFU;	/* 512 Bytes alignment  */
			//printf("Next File at 0x%08x\n", location);
		}
		else {
			return;
		}
	}

	return;
}

void save_fwu_version_env(void)
{
	char *old_ver;
	int dirty = 0;		/* 1: The env is modified */

	printf("Update SW Version: %s\n", fwu_ver);
	
	old_ver = getenv(ENV_SW_VERSION0);
	if(old_ver == NULL){
		printf("%s empty, set %s\n",ENV_SW_VERSION0, fwu_ver);
		setenv(ENV_SW_VERSION0, fwu_ver);
		dirty = 1;
	} else {
		if(strcmp(old_ver, fwu_ver)!=0){
			printf("old %s [%s] is different to new [%s], set new version\n", ENV_SW_VERSION0, old_ver, fwu_ver);
			setenv(ENV_SW_VERSION0, fwu_ver);
			dirty = 1;
		} //else 
			//old and new version are the same, don't save env!	
	}

#ifdef CONFIG_LUNA_MULTI_BOOT
#ifdef CONFIG_SINGLE_IMAGE
	if(image_type==DUAL_IMAGE)
#endif
	{
		old_ver = getenv(ENV_SW_VERSION1);
		if(old_ver == NULL){
			printf("%s empty, set %s\n",ENV_SW_VERSION1, fwu_ver);
			setenv(ENV_SW_VERSION1, fwu_ver);
			dirty = 1;
		} else {
			if(strcmp(old_ver, fwu_ver) != 0){
				printf("old %s [%s] is different to new [%s], set new version\n", ENV_SW_VERSION1, old_ver, fwu_ver);
				setenv(ENV_SW_VERSION1, fwu_ver);
				dirty = 1;
			} //else
				//old and new version are the same, don't save env!
		}
	}
#endif

	if (dirty)	saveenv();
	return;
}

void save_images_crc_env(unsigned char *img, unsigned int size)
{
	char *old;
	char new[CRC32_STRING_SIZE];
	ulong crc;
	int dirty = 0;		/* 1: The env is modified */

	printf("Update Image CRC32\n");

	/* Caculate img.tar crc32 */
	old = getenv(ENV_SW_CRC0);
	crc = crc32_wd(0, img, size, CHUNKSZ_CRC32);
	sprintf(new, "%08lx", crc);
	
	if(old == NULL){
		printf("%s empty, set %s\n", ENV_SW_CRC0, new);
		setenv(ENV_SW_CRC0, new);
		dirty = 1;
	} else {
		if(strcmp(old, new) != 0){
			printf("old %s [%s] is different to new [%s], set new crc\n", ENV_SW_CRC0, old, new);
			setenv(ENV_SW_CRC0, new);
			dirty = 1;
		} //else
			//old and new crc are the same, don't save env!
	}

#ifdef CONFIG_LUNA_MULTI_BOOT
#ifdef CONFIG_SINGLE_IMAGE
	if(image_type==DUAL_IMAGE)
#endif
	{
		old = getenv(ENV_SW_CRC1);
	
		if(old == NULL){
			printf("%s empty, set %s\n", ENV_SW_CRC1, new);
			setenv(ENV_SW_CRC1, new);
			dirty = 1;
		} else {
			if(strcmp(old, new) != 0){
				printf("old %s [%s] is different to new [%s], set new crc\n", ENV_SW_CRC1, old, new);
				setenv(ENV_SW_CRC1, new);
				dirty = 1;
			} //else
				//old and new crc are the same, don't save env!
		}
	}
#endif

#if 0
	/* Caculate uImage crc32 */
	old = getenv(ENV_KERNEL0_CRC);
	crc = crc32_wd(0, (const unsigned char *)(img + kernel_location), kernel_size, CHUNKSZ_CRC32);
	sprintf(new, "%08lx", crc);
	
	if(old == NULL){
		printf("%s empty, set %s\n", ENV_KERNEL0_CRC, new);
		setenv(ENV_KERNEL0_CRC, new);
		dirty = 1;
	} else {
		if(strcmp(old, new) != 0){
			printf("old %s [%s] is different to new [%s], set new crc\n", ENV_KERNEL0_CRC, old, new);
			setenv(ENV_KERNEL0_CRC, new);
			dirty = 1;
		} //else 
			//old and new crc are the same, don't save env!	
	}

#ifdef CONFIG_LUNA_MULTI_BOOT
	old = getenv(ENV_KERNEL1_CRC);
	
	if(old == NULL){
		printf("%s empty, set %s\n", ENV_KERNEL1_CRC, new);
		setenv(ENV_KERNEL1_CRC, new);
		dirty = 1;
	} else {
		if(strcmp(old, new) != 0){
			printf("old %s [%s] is different to new [%s], set new crc\n", ENV_KERNEL1_CRC, old, new);
			setenv(ENV_KERNEL1_CRC, new);
			dirty = 1;
		} //else 
			//old and new crc are the same, don't save env!	
	}
#endif

	/* Caculate rootfs crc32 */
	old = getenv(ENV_ROOTFS0_CRC);
	crc = crc32_wd(0, (const unsigned char *)(img + rootfs_location), rootfs_size, CHUNKSZ_CRC32);
	sprintf(new, "%08lx", crc);
	
	if(old == NULL){
		printf("%s empty, set %s\n", ENV_ROOTFS0_CRC, new);
		setenv(ENV_ROOTFS0_CRC, new);
		dirty = 1;
	} else {
		if(strcmp(old, new) != 0){
			printf("old %s [%s] is different to new [%s], set new crc\n", ENV_ROOTFS0_CRC, old, new);
			setenv(ENV_ROOTFS0_CRC, new);
			dirty = 1;
		} //else 
			//old and new crc are the same, don't save env!	
	}

#ifdef CONFIG_LUNA_MULTI_BOOT
	old = getenv(ENV_ROOTFS1_CRC);
	
	if(old == NULL){
		printf("%s empty, set %s\n", ENV_ROOTFS1_CRC, new);
		setenv(ENV_ROOTFS1_CRC, new);
		dirty = 1;
	} else {
		if(strcmp(old, new) != 0){
			printf("old %s [%s] is different to new [%s], set new crc\n", ENV_ROOTFS1_CRC, old, new);
			setenv(ENV_ROOTFS1_CRC, new);
			dirty = 1;
		} //else 
			//old and new crc are the same, don't save env!	
	}
#endif
#endif

	if (dirty)	saveenv();
	return;
}


#if CONFIG_RTK_USE_ONE_UBI_DEVICE
static int
do_ubi_update(char *vol_name,  unsigned int src, unsigned int src_size)
{
	char cmd_buf[CMD_BUF_SZ] = {0};
	int ret;
	sprintf(cmd_buf, "set current_vol %s", vol_name);
	ret = run_command(cmd_buf, 0);
	sprintf(cmd_buf, "run check_vol");
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	sprintf(cmd_buf, "ubi write %x %s %x", src, vol_name, src_size);
	printf("CMD = %s\n", cmd_buf);
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	return 0;
}
#endif


static int do_upimgtar(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret = 0;
	char *endp;
	u8 *img;
	u32 size;
	char cmd_buf[CMD_BUF_SZ] = {0};
	u32 part_base, part_sz;

	/* need at least three arguments */
	if (argc < 3)
		goto usage;

	/* Retrieve imgi.tar location */
	img = (u8 *)simple_strtoul(argv[1], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
		return -1;

	/* Retrieve img.tar size */
	size = (u32)simple_strtoul(argv[2], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
		return -1;

	printf("img.tar is located at %p (size = 0x%08x)\n", img, size);
	scan_img_tar(img, size);

	if (kernel_location && rootfs_location) {
		printf("kernel image is at 0x%08x (size: 0x%08x)\n", kernel_location, kernel_size);
		printf("rootfs image is at 0x%08x (size: 0x%08x)\n", rootfs_location, rootfs_size);
		if (osgi_location)
			printf("osgi image is at 0x%08x (size: 0x%08x)\n", osgi_location, osgi_size);
		printf("Firmware Version: [%s]\n\n", fwu_ver);
	}

	/* Update kernel image */
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
	ret = do_ubi_update(UBI_K0_NAME, (unsigned int)(img + kernel_location), kernel_size);
	if (ret) {
		goto done;
	}
	(void)cmd_buf;
	(void)part_base;
	(void)part_sz;
#else
	part_base = simple_strtoul(getenv("fl_kernel1"), &endp, 16);
	part_sz = simple_strtoul(getenv("fl_kernel1_sz"), &endp, 16);
	printf("kernel partition at 0x%08x, size=0x%08x\n", part_base, part_sz);

	#ifdef CONFIG_CMD_SF
	sprintf(cmd_buf, "sf erase %x +%x; sf write %x %x %x", part_base, part_sz, (unsigned int)(img + kernel_location), part_base, kernel_size);
	#elif defined (CONFIG_CMD_SPI_NAND)
	sprintf(cmd_buf, "spi_nand erase %x %x; spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + kernel_location), part_base, kernel_size);
	#else
	printf("ERROR: UNKNOWN flash type, please check %s!!!\n", __FILE__);
	#endif
	if(kernel_size > part_sz) {
		printf("Error: image size is larger than the partition, operation aborted\n");
		goto done;
	}
	printf("CMD = %s\n", cmd_buf);
	run_command(cmd_buf, 0);
#endif

#ifdef CONFIG_LUNA_MULTI_BOOT
	/* Update 2nd kernel image */
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
	ret = do_ubi_update(UBI_K1_NAME, (unsigned int)(img + kernel_location), kernel_size);
	if (ret) {
		goto done;
	}
#else
#ifdef CONFIG_SINGLE_IMAGE
	dual_image=getenv("dual_image");
	image_type= dual_image ? simple_strtoul (dual_image, NULL, 10) : 0;

	if(image_type!=DUAL_IMAGE)
	    printf("Single image, skip 2nd kernel.\n");
	else
#endif
	{	part_base = simple_strtoul(getenv("fl_kernel2"), &endp, 16);
		part_sz = simple_strtoul(getenv("fl_kernel2_sz"), &endp, 16);
		printf("2nd kernel partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
		#ifdef CONFIG_CMD_SF
		sprintf(cmd_buf, "sf erase %x +%x; sf write %x %x %x", part_base, part_sz, (unsigned int)(img + kernel_location), part_base, kernel_size);
		#elif defined (CONFIG_CMD_SPI_NAND)
		sprintf(cmd_buf, "spi_nand erase %x %x; spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + kernel_location), part_base, kernel_size);
		#else
		printf("ERROR: UNKNOWN flash type, please check %s!!!\n", __FILE__);
		#endif
		if(kernel_size > part_sz) {
			printf("Error: image size is larger than the partition, operation aborted\n");
			goto done;
		}
		printf("CMD = %s\n", cmd_buf);
		run_command(cmd_buf, 0);
	}
#endif
	
#endif
	/* Rootfs image */
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
	ret = do_ubi_update(UBI_R0_NAME, (u32)(img + rootfs_location), rootfs_size);
	if (ret) {
		goto done;
	}
#else
	part_base = simple_strtoul(getenv("fl_rootfs1"), &endp, 16);
	part_sz = simple_strtoul(getenv("fl_rootfs1_sz"), &endp, 16);
	printf("rootfs partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
	#ifdef CONFIG_SPINOR_FLASH
	sprintf(cmd_buf, "sf erase %x +%x; sf write %x %x %x", part_base, part_sz, (u32)(img + rootfs_location), part_base, rootfs_size);
	#else
	sprintf(cmd_buf, "spi_nand erase %x %x; spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + rootfs_location), part_base, rootfs_size);
	#endif
	if(rootfs_size > part_sz) {
		printf("Error: image size is larger than the partition, operation aborted\n");
		goto done;
	}
	printf("CMD = %s\n", cmd_buf);
	run_command(cmd_buf, 0);
#endif

#ifdef CONFIG_LUNA_MULTI_BOOT
	/* 2nd Rootfs image */
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
	ret = do_ubi_update(UBI_R1_NAME, (u32)(img + rootfs_location), rootfs_size);
	if (ret) {
		goto done;
	}
#else
#ifdef CONFIG_SINGLE_IMAGE
	if(image_type!=DUAL_IMAGE)
		printf("Single image, skip 2nd rootfs.\n");
	else
#endif
	{
		part_base = simple_strtoul(getenv("fl_rootfs2"), &endp, 16);
		part_sz = simple_strtoul(getenv("fl_rootfs2_sz"), &endp, 16);
		printf("2nd rootfs partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
		#ifdef CONFIG_SPINOR_FLASH
		sprintf(cmd_buf, "sf erase %x +%x; sf write %x %x %x", part_base, part_sz, (u32)(img + rootfs_location), part_base, rootfs_size);
		#else
		sprintf(cmd_buf, "spi_nand erase %x %x; spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + rootfs_location), part_base, rootfs_size);
		#endif
		if(rootfs_size > part_sz) {
			printf("Error: image size is larger than the partition, operation aborted\n");
			goto done;
		}
		printf("CMD = %s\n", cmd_buf);
		run_command(cmd_buf, 0);
	}
#endif
#endif
	/* osgi image */
	if (osgi_location) {
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
		ret = do_ubi_update(UBI_OSGI_NAME, (u32)(img + osgi_location), osgi_size);
		if (ret) {
			goto done;
		}
#else
		part_base = simple_strtoul(getenv("fl_osgi"), &endp, 16);
		part_sz = simple_strtoul(getenv("fl_osgi_sz"), &endp, 16);
		printf("osgi partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
		#ifdef CONFIG_SPINOR_FLASH
		sprintf(cmd_buf, "sf erase %x +%x; sf write %x %x %x", part_base, part_sz, (u32)(img + osgi_location), part_base, osgi_size);
		#else
		sprintf(cmd_buf, "spi_nand erase %x %x; spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + osgi_location), part_base, osgi_size);
		#endif
		if(osgi_size > part_sz) {
			printf("Error: image size is larger than the partition, operation aborted\n");
			goto done;
		}
		printf("CMD = %s\n", cmd_buf);
		run_command(cmd_buf, 0);
#endif
	}

	save_fwu_version_env();
	save_images_crc_env(img, size);

done:
	printf("Update img.tar Done\n");
	if (ret != -1)
		return ret;

usage:
	return cmd_usage(cmdtp);
}

U_BOOT_CMD(
	upimgtar,	3,	1,	do_upimgtar,
	"update kernel and rootfs by tar format on luna platform",
	"ADDRESS\n"
	"SIZE\n"
	"\n"
	"Description:\n"
	"Extract kernel and rootfs in tar file located at ADDRESS and write them into corresponding partitions.\n"
);
