/*
 * Command for updating vm.img format image (containing uImage and rootfs) into corresponding partition on luna platform.
 * Author: bohungwu@realtek.com
 */

#include <common.h>
#include <malloc.h>

#include <asm/io.h>

//#define VMIMG_DEBUG

#if CONFIG_RTK_USE_ONE_UBI_DEVICE
#define UBI_K0_NAME "ubi_k0"
#define UBI_K1_NAME "ubi_k1"
#define UBI_R0_NAME "ubi_r0"
#define UBI_R1_NAME "ubi_r1"
#define UBI_FRAMEWORK1_NAME "ubi_framework1"
#define UBI_FRAMEWORK2_NAME "ubi_framework2"
#define UBI_APP_NAME "ubi_apps"
#define UBI_OSGI_NAME "ubi_osgi"
#endif

struct vmimg_hdr {
	u32 key;
	u32 load_addr;
	u32 img_sz;
	u32 entry_addr;
	u32 padding[11];
	u32 flash_base;
};

#ifdef VMIMG_DEBUG
static void dump_vmimg_hdr(struct vmimg_hdr *vmimg_hdr_p) {

	printf("key=0x%x\n", vmimg_hdr_p->key);
	printf("load_addr=0x%x\n", vmimg_hdr_p->load_addr);
	printf("img_sz=0x%x(%u)\n", vmimg_hdr_p->img_sz, vmimg_hdr_p->img_sz);
	printf("entry_addr=0x%x\n", vmimg_hdr_p->entry_addr);
	printf("flash_base=0x%x\n", vmimg_hdr_p->flash_base);
}
#else
#define dump_vmimg_hdr(x)
#endif

#define VMIMG_BOOT_KEY		(0xB0010001)
#define VMIMG_KERNEL_KEY	(0xa0000203)
#define VMIMG_ROOTFS_KEY	(0xa0000403)
#define VMIMG_CONFIG_KEY	(0xCF010002)
#define VMIMG_CFGPROV_KEY	(0xCF010202)
#define VMIMG_APPLICATION_KEY	(0xA0000003)
#define VMIMG_OSGI_KEY		(0xCAFEBABE)

#define ENV_CONFIG_PROVINCE	"cfg_province"
#define ENV_RESET_DEFAULT	"rst2dfl"
#define ENV_SYNC_DEFAULT	"sync_default_env"
#define CMD_BUF_SZ		(256)		/* Maximum U-Boot Command Buffer Size */
#define CFG_PROV_SZ		(8)		/* Maximum Config Province Code */

int check_vm_img(unsigned char *buffer, unsigned int size)
{
	struct vmimg_hdr *hdr;

	//printf("Is this a vm.img?\n");
	if (size < sizeof(struct vmimg_hdr))	return 0;

	hdr = (struct vmimg_hdr *)buffer;
	
	switch (hdr->key) {
	case VMIMG_BOOT_KEY:
	case VMIMG_KERNEL_KEY:
	case VMIMG_ROOTFS_KEY:
	case VMIMG_CONFIG_KEY:
	case VMIMG_CFGPROV_KEY:
	case VMIMG_APPLICATION_KEY:
	case VMIMG_OSGI_KEY:
		return 1;
		break;
	}

	return 0;
}

#if CONFIG_RTK_USE_ONE_UBI_DEVICE
static int
do_ubi_update(char *vol_name,  unsigned int src, unsigned int src_size)
{
	char cmd_buf[CMD_BUF_SZ] = {0};
	int ret;
	sprintf(cmd_buf, "set current_vol %s", vol_name);
	ret = run_command(cmd_buf, 0);
	sprintf(cmd_buf, "run check_vol");
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	sprintf(cmd_buf, "ubi write %x %s %x", src, vol_name, src_size);
	printf("CMD = %s\n", cmd_buf);
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	return 0;
}

static int
do_ubi_clean(char *vol_name)
{
	char cmd_buf[CMD_BUF_SZ] = {0};
	int ret;
	sprintf(cmd_buf, "set current_vol %s", vol_name);
	ret = run_command(cmd_buf, 0);
	sprintf(cmd_buf, "run check_vol");
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	sprintf(cmd_buf, "ubi clean %s", vol_name);
	printf("CMD = %s\n", cmd_buf);
	ret = run_command(cmd_buf, 0);
	if (ret == -1) {
		return ret;
	}
	return 0;
}

#endif // if CONFIG_RTK_USE_ONE_UBI_DEVICE


static int do_upvmimg(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	int ret = 0, dirty = 0;
	char *endp;
	char *old_cfg_prov;
	u8 *img;
	char cmd_buf[CMD_BUF_SZ] = {0};
	char new_cfg_prov[CFG_PROV_SZ] = {0};
	struct vmimg_hdr hdr;
	u32 part_base, part_sz;

	/* need at least two arguments */
	if (argc < 2)
		goto usage;

	/* Retrieve vmimg location */
	img = (u8 *)simple_strtoul(argv[1], &endp, 16);
	if (*argv[1] == 0 || *endp != 0)
	{
		#ifdef CONFIG_MULTICAST_UPGRADE
		led_multicastUP_error_hang();
		#endif
		return -1;
	}
	printf("vmimg is located at %p\n", img);

	while(1) {

		int unknownImg = 0;
		memcpy((void *)&hdr, (void *)img, sizeof(hdr));
		dump_vmimg_hdr(&hdr);

		switch (hdr.key) {
		case VMIMG_BOOT_KEY:
			printf("\n\n====== [BOOT] Image is Loacted at %p ======\n", img);

			part_base = 0;
			part_sz = simple_strtoul(getenv("fl_boot_sz"), &endp, 16);
			printf("boot partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("boot image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write.raw %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write.raw %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);

			// Upgrage boot also reset default cs
			setenv(ENV_RESET_DEFAULT, "1");
			
			// Sync new bootloader default env in next boot up
			setenv(ENV_SYNC_DEFAULT, "1");
			saveenv();
			
			break;

		case VMIMG_KERNEL_KEY:
			printf("\n\n====== [Kernel] Image is Loacted at %p ======\n", img);
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
			ret = do_ubi_update(UBI_K0_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
#ifdef CONFIG_LUNA_MULTI_BOOT
			ret = do_ubi_update(UBI_K1_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
#endif //end ifdef CONFIG_LUNA_MULTI_BOOT
#else // if CONFIG_RTK_USE_ONE_UBI_DEVICE

			part_base = simple_strtoul(getenv("fl_kernel1"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_kernel1_sz"), &endp, 16);
			printf("kernel partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("kernel image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);

#ifdef CONFIG_LUNA_MULTI_BOOT
				/* Update 2nd kernel image */
				part_base = simple_strtoul(getenv("fl_kernel2"), &endp, 16);
				part_sz = simple_strtoul(getenv("fl_kernel2_sz"), &endp, 16);
				printf("kernel2 partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
				printf("kernel2 image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
				if(hdr.img_sz > part_sz) {
					printf("Error: image size is larger than the partition, operation aborted\n");
					ret = -1;
					goto done;
				}

#ifdef CONFIG_SPINOR_FLASH
				sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
				sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
				printf("CMD=%s\n", cmd_buf);
				run_command(cmd_buf, 0);
#endif
#endif // else if CONFIG_RTK_USE_ONE_UBI_DEVICE
			break;

		case VMIMG_ROOTFS_KEY:
			printf("\n\n====== [Rootfs] Image is Loacted at %p ======\n", img);

#if CONFIG_RTK_USE_ONE_UBI_DEVICE
			ret = do_ubi_update(UBI_R0_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
#ifdef CONFIG_LUNA_MULTI_BOOT
			ret = do_ubi_update(UBI_R1_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
#endif
#else // if CONFIG_RTK_USE_ONE_UBI_DEVICE
			part_base = simple_strtoul(getenv("fl_rootfs1"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_rootfs1_sz"), &endp, 16);
			printf("rootfs partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("rootfs image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);

			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (u32)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);

#ifdef CONFIG_LUNA_MULTI_BOOT
			/* 2nd Rootfs image */
			part_base = simple_strtoul(getenv("fl_rootfs2"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_rootfs2_sz"), &endp, 16);
			printf("rootfs2 partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("rootfs2 image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (u32)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);
#endif
#endif // else if CONFIG_RTK_USE_ONE_UBI_DEVICE
			break;

		case VMIMG_CONFIG_KEY:
			printf("\n\n====== [Config] Image is Loacted at %p ======\n", img);
			printf("Config Image is NOT support!!!\n");

			break;

		case VMIMG_CFGPROV_KEY:
			printf("\n\n====== [Config Province] Image is Loacted at %p (%d) ======\n", img, hdr.img_sz);
			strncpy(new_cfg_prov, (char *)(img + sizeof(hdr)), min(hdr.img_sz, CFG_PROV_SZ));
			printf("Update cfg_province: [%s]\n", new_cfg_prov);

			old_cfg_prov = getenv(ENV_CONFIG_PROVINCE);
			if(old_cfg_prov == NULL){
				printf("%s empty, set %s\n",ENV_CONFIG_PROVINCE, new_cfg_prov);
				setenv(ENV_CONFIG_PROVINCE, new_cfg_prov);
				dirty = 1;
			} else {
				if(strcmp(old_cfg_prov, new_cfg_prov)!=0){
					printf("old %s [%s] is different to new [%s], set new version\n", ENV_CONFIG_PROVINCE, old_cfg_prov, new_cfg_prov);
					setenv(ENV_CONFIG_PROVINCE, new_cfg_prov);
					dirty = 1;
				} //else 
					//old and new version are the same, don't save env!	
			}

			if (dirty)	saveenv();

			break;

		case VMIMG_APPLICATION_KEY:
			printf("\n\n====== [App] Image is Loacted at %p ======\n", img);
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
			ret = do_ubi_update(UBI_FRAMEWORK1_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
			ret = do_ubi_update(UBI_FRAMEWORK2_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
			ret = do_ubi_clean(UBI_APP_NAME);
			if (ret) goto done;
#else  // if CONFIG_RTK_USE_ONE_UBI_DEVICE

			/* Updating framework1 */
			part_base = simple_strtoul(getenv("fl_framework1"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_framework1_sz"), &endp, 16);
			printf("framework1 partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("framework1 image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (u32)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);

			/* Updating framework2 */
			part_base = simple_strtoul(getenv("fl_framework2"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_framework2_sz"), &endp, 16);
			printf("framework2 partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("framework2 image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (u32)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);

			/* Erasing apps */
			part_base = simple_strtoul(getenv("fl_apps"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_apps_sz"), &endp, 16);
			printf("apps partition at 0x%08x, size=0x%08x\n", part_base, part_sz);

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x", part_base, part_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x", part_base, part_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);
#endif // else if CONFIG_RTK_USE_ONE_UBI_DEVICE

			break;

		case VMIMG_OSGI_KEY:
			printf("\n\n====== [OSGI] Image is Loacted at %p ======\n", img);
#if CONFIG_RTK_USE_ONE_UBI_DEVICE
			ret = do_ubi_update(UBI_OSGI_NAME, (unsigned int)(img + sizeof(hdr)), hdr.img_sz);
			if (ret) goto done;
#else  // if CONFIG_RTK_USE_ONE_UBI_DEVICE

			/* Updating osgi */
			part_base = simple_strtoul(getenv("fl_osgi"), &endp, 16);
			part_sz = simple_strtoul(getenv("fl_osgi_sz"), &endp, 16);
			printf("osgi partition at 0x%08x, size=0x%08x\n", part_base, part_sz);
			printf("osgi image size=0x%08x(%u)\n", hdr.img_sz, hdr.img_sz);
			if(hdr.img_sz > part_sz) {
				printf("Error: image size is larger than the partition, operation aborted\n");
				ret = -1;
				goto done;
			}

#ifdef CONFIG_SPINOR_FLASH
			sprintf(cmd_buf, "sf erase %x +%x;sf write %x %x %x", part_base, part_sz, (u32)(img + sizeof(hdr)), part_base, hdr.img_sz);
#else
			sprintf(cmd_buf, "spi_nand erase %x %x;spi_nand write %x %x %x", part_base, part_sz, (unsigned int)(img + sizeof(hdr)), part_base, hdr.img_sz);
#endif
			printf("CMD=%s\n", cmd_buf);
			run_command(cmd_buf, 0);
#endif // else if CONFIG_RTK_USE_ONE_UBI_DEVICE

			break;

		default:
			printf("Unknown Key (%08X) is Loacted at %p\n", hdr.key, img);
			unknownImg = 1;

			break;
		}
	
		if (unknownImg) break;
	
		img = (u8 *)(img + sizeof(struct vmimg_hdr) + hdr.img_sz);
	}

done:
	
	printf("Update vm.img Done\n");
        if (ret != 0)
        {
		#ifdef CONFIG_MULTICAST_UPGRADE
		led_multicastUP_error_hang();
		#endif
                return ret;
        }
        else return ret;


usage:
	return cmd_usage(cmdtp);
}

U_BOOT_CMD(
	upvmimg,	2,	1,	do_upvmimg,
	"update kernel and rootfs by vmimg format on luna platform",
	"ADDRESS\n"
	"\n"
	"Description:\n"
	"Extract kernel and rootfs in vmimg image located at ADDRESS and write them into corresponding partitions.\n"
);
