
#include <common.h>
#include <ubi_uboot.h>

#define CMD_BUF_SZ (256)
#define VOL_UPDATED "VOL_UPDATED"


/* Private own data */
static struct ubi_device *ubi;

struct volume_size_info {
	char *vol_name;
	uint32_t vol_size;
};

static struct volume_size_info s128_vol_predef[] = {
	{
		"ubi_Config",
		10752*1024
	},
	{
		"ubi_k0",
		10*1024*1024
	},
	{
		"ubi_r0",
		25*1024*1024
	},
	{
		"ubi_k1",
		10*1024*1024
	},
	{
		"ubi_r1",
		25*1024*1024
	},
	{ "", 0} // end of list
};

static struct volume_size_info s256_vol_predef[] = {
	{
		"ubi_Config",
		10752*1024
	},
	{
		"ubi_k0",
		10*1024*1024
	},
	{
		"ubi_r0",
		25*1024*1024
	},
	{
		"ubi_k1",
		10*1024*1024
	},
	{
		"ubi_r1",
		25*1024*1024
	},
	{
		"ubi_framework1",
		8*1024*1024
	},
	{
		"ubi_framework2",
		8*1024*1024
	},
	{
		"ubi_apps",
		36*1024*1024
	},
	{ "", 0} // end of list
};

static struct volume_size_info s512_vol_predef[] = {
	{
	},
	{ "", 0} //end of list
};


enum flash_size_id{
	FLS_128MB = (128*1024*1024),
	FLS_256MB = (256*1024*1024),
	FLS_512MB = (512*1024*1024),
	FLS_NONE = 0,
}; // flash_size

struct flash_volume_info{
	enum flash_size_id size;
	struct volume_size_info * volume_list;
};

static struct flash_volume_info flash_volumes[] = {
	{ FLS_128MB,
		s128_vol_predef},
	{ FLS_256MB,
		s256_vol_predef},
	{ FLS_512MB,
		s512_vol_predef},
	{ FLS_NONE,
		NULL} // end of list
};

static uint32_t
get_flash_size(void)
{
	uint32_t ret_val;
#if defined (CONFIG_CMD_SPI_NAND)
	extern uint32_t spi_nand_chip_size(uint32_t idx);
	ret_val=spi_nand_chip_size(0);
	/* ToDo , multiple chip to be considered */
#elif defined (CONFIG_CMD_ONFI)
	extern uint32_t onfi_chip_size(uint32_t idx);
	ret_val=onfi_chip_size(0);
	/* ToDo , multiple chip to be considered */
#else
	ret_val = SWP_FLASHI.num_chips * SWP_FLASHI.size_per_chip_b;
#endif
	return ret_val;
}




static struct ubi_volume *
ubi_find_volume(char *volume)
{
	struct ubi_volume *vol = NULL;
	int i;

	for (i = 0; i < ubi->vtbl_slots; i++) {
		vol = ubi->volumes[i];
		if (vol && !strcmp(vol->name, volume))
			return vol;
	}

	printf("Volume %s not found!\n", volume);
	return NULL;
}

static int
ubi_volume_resize(char *volume, int32_t size)
{
	struct ubi_volume_desc desc;
	struct ubi_volume *vol = NULL;
	int err;
	int peb_num;
	vol = ubi_find_volume(volume);
	if (vol == NULL)
		return ENODEV;

	printf("Resize UBI volume %s (id %d) to %X(%d) \n", vol->name, vol->vol_id, size, size);
	desc.vol = vol;

	peb_num = size / vol->usable_leb_size;
	peb_num += (size % vol->usable_leb_size) ? 1 : 0;
	if (!peb_num) {
		printf("cannot resize volume %s %d to zero PEB!\n", volume, vol->vol_id );
		return -1;
	}
	if (vol->reserved_pebs == peb_num) {
		return 0;
	}
	err = ubi_resize_volume(&desc, peb_num);
	if (err) {
		printf("cannot resize volume %s %d (error %d)\n", volume, vol->vol_id, err );
		return err;
	}

	return 0;
}

static int
do_ubi_resize(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	int32_t size;
	ubi = ubi_devices[0];
	if (!ubi) {
		printf("No ubi device selected!\n");
		return  -1;
	}
	if (argc < 3) {
		printf("Not enough argument!\n");
		return -1;
	}
	size = simple_strtoul(argv[2], NULL, 16);
	if (0 == size) {
		printf("Can't resize volume as 0!\n");
		return -1;
	}
	return ubi_volume_resize(argv[1], size);
}

U_BOOT_CMD(
	ubi_resize, 6, 1, do_ubi_resize,
	"ubi resize command",
	"volum_name size\n"
	"size is hex\n"
);

int
checkVolumeExist(char *cmd_buf, int buf_size, char *volume_name)
{
		sprintf(cmd_buf, "ubi check %s", volume_name);
		printf("CMD: %s\n", cmd_buf);
		return run_command(cmd_buf, 0);
}

int
createVolume(char *cmd_buf, int buf_size, char* volume_name, uint32_t volume_size)
{
	int ret;
	sprintf(cmd_buf, "ubi create %s %x dynamic", volume_name, volume_size);
	printf("CMD: %s\n", cmd_buf);
	ret = run_command(cmd_buf, 0);
	if (-1 == ret) {
		printf("Create volume %s Fail!\n", volume_name);
	}
	return ret;
}

int
createOrModifyVolume(char *cmd_buf, int buf_size, char* vol_name, uint32_t vol_size)
{
	int ret = 0;
	ret = checkVolumeExist(cmd_buf, buf_size, vol_name);
	if (-1 == ret) {
		printf("Volume %s not exist!\n", vol_name);
		ret = createVolume(cmd_buf, buf_size, vol_name, vol_size );
		return ret;
	}
	sprintf(cmd_buf, "ubi_resize %s %x", vol_name, vol_size);
	printf("CMD: %s\n", cmd_buf);
	ret = run_command(cmd_buf, 0);
	return ret;
}




#define UBI_VOLUME_LAYOUT_ENV "ubi_layout"
int
volume_resize_from_var(void)
{
	int ret = 1;
	int func_ret = 1;
	char *value;
	char *vol_name;
	uint32_t vol_size;
	char cmd_buf[CMD_BUF_SZ] = {0};
	char *split_ptr;
	int layout_str_len;

	value = getenv(UBI_VOLUME_LAYOUT_ENV);
	if (!value) {
		return func_ret;
	}
	layout_str_len = strlen(value);
	char new_buf[layout_str_len+1];
	strcpy( new_buf, value);
	value = new_buf;
	split_ptr = strchr(value, ':');
	while (split_ptr && layout_str_len > 0) {
		vol_name = value;

		if ((split_ptr-value) <= 0) {
			break;
		}
		layout_str_len -= (split_ptr - value);
		layout_str_len -= 1;
		*split_ptr = '\0';

		value = split_ptr+1;

		split_ptr = strchr(value, ';');
		if (split_ptr) {
			*split_ptr = '\0';
		}
		vol_size = simple_strtoul(value, NULL, 16);
		if (0 == vol_size) {
			break;
		}

		ret = createOrModifyVolume(cmd_buf, sizeof(cmd_buf), vol_name, vol_size);
		if (-1 == ret) {
			break;
		}
		func_ret = 0;

		if (split_ptr) {
			layout_str_len -= (split_ptr - value);
			layout_str_len -= 1;
			if (layout_str_len <= 0) {
				break;
			}
			*split_ptr = '\0';

			value = split_ptr+1;

			split_ptr = strchr(value, ':');
		} else {
			break;
		}
	}

	return func_ret;
}


void
swp_otto_ubi_volume_resize_init(void)
{
	int ret;
	uint32_t fl_size;
	struct flash_volume_info *vol_info = flash_volumes;
	struct volume_size_info *size_info = NULL;
	char cmd_buf[CMD_BUF_SZ] = {0};
	char *value = NULL;
	value = getenv(VOL_UPDATED);
	if (NULL != value) {
		return;
	}

	ubi = ubi_devices[0];
	if (!ubi) {
		printf("No ubi device selected!\n");
		return  ;
	}

	if (!volume_resize_from_var()) {
		setenv(VOL_UPDATED, "1");
		saveenv();
		return;
	}

	fl_size = get_flash_size();
	printf("flash size is %d\n", fl_size);
	while(vol_info->size != FLS_NONE){
		if (vol_info->size == fl_size) {
			break;
		}
		vol_info += 1;
	}

	if (vol_info->size == FLS_NONE) {
		return;
	}

	setenv(VOL_UPDATED, "1");
	saveenv();
	size_info = vol_info->volume_list;
	while(size_info->vol_name && 0 != strcmp(size_info->vol_name, "")) {
		if (0 == size_info->vol_size) {
			return;
		}
		ret = createOrModifyVolume(cmd_buf, sizeof(cmd_buf), size_info->vol_name, size_info->vol_size);
		if (-1 == ret) {
			return;
		}
		size_info += 1;
	}


}

static int
do_ubi_resize_init(cmd_tbl_t * cmdtp, int flag, int argc, char * const argv[])
{
	swp_otto_ubi_volume_resize_init();
	return 0;
}

U_BOOT_CMD(
	ubi_resize_init, 6, 1, do_ubi_resize_init,
	"do ubi resize command",
	"\n"
);



