/*
 * Copyright (C) 2022, MediaTek Inc. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <errno.h>
#include <stdbool.h>
#include <lib/mmio.h>
#include <drivers/delay_timer.h>
#include <platform_def.h>
#include <mt7987_def.h>
#include "fwdl-internal.h"

#define TOPS_HDR_MAGIC				0x53504f54

#define PART_FL_LOAD_ADDR_OVERRIDE		(1 << 0)
#define PART_FL_BOOT_PART_OVERRIDE		(1 << 1)

/* Key hash */
extern const uint8_t tops_keyhash[];

enum tops_part_type {
	TOPS_PT_IRAM0,
	TOPS_PT_DRAM0,
	TOPS_PT_L2SRAM,
	TOPS_PT_METADATA,

	__TOPS_PT_MAX
};

enum tops_platform {
	TOPS_PLAT_MT7988,

	__TOPS_PLAT_MAX
};

enum tops_role_type {
	TOPS_ROLE_MGMT,
	TOPS_ROLE_CLUSTER,

	__TOPS_ROLE_MAX
};

struct tops_loadable_part {
	enum tops_part_type type;
	uintptr_t addr;
};

struct tops_core {
	uintptr_t core_reset_reg;
	uintptr_t core_halt_reg;
	uintptr_t start_vector_sel_reg;
	uintptr_t alt_vector_addr_reg;

	const struct tops_loadable_part *parts;
	uint32_t num_parts;
};

struct tops_part {
	enum tops_part_type type;
	uint32_t base;
	uint32_t size;
};

struct tops_role {
	enum tops_role_type type;

	const struct tops_part *parts;
	uint32_t num_parts;

	const struct tops_core *cores;
	uint32_t num_cores;

	uintptr_t l2sram_addr;

	uint32_t default_reset_vector_base;
};

static const struct tops_part tops_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.base = 0x40000000,
		.size = 0x8000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.base = 0x40020000,
		.size = 0x8000,
	},
	{
		.type = TOPS_PT_L2SRAM,
		.base = 0x4e100000,
		.size = 0x40000,
	},
};

static const struct tops_loadable_part tops_mgmt_loadable_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.addr = 0x09400000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.addr = 0x09410000,
	},
};

static const struct tops_core tops_mgmt_core = {
	.core_reset_reg = 0x09101000,
	.core_halt_reg = 0x09101004,
	.start_vector_sel_reg = 0x0910A068,
	.alt_vector_addr_reg = 0x0910A06C,

	.parts = tops_mgmt_loadable_parts,
	.num_parts = ARRAY_SIZE(tops_mgmt_loadable_parts),
};

static const struct tops_loadable_part tops_cluster_0_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.addr = 0x09900000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.addr = 0x09910000,
	},
};

static const struct tops_loadable_part tops_cluster_1_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.addr = 0x09920000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.addr = 0x09930000,
	},
};

static const struct tops_loadable_part tops_cluster_2_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.addr = 0x09940000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.addr = 0x09950000,
	},
};

static const struct tops_loadable_part tops_cluster_3_parts[] = {
	{
		.type = TOPS_PT_DRAM0,
		.addr = 0x09960000,
	},
	{
		.type = TOPS_PT_IRAM0,
		.addr = 0x09970000,
	},
};

static const struct tops_core tops_cluster_cores[] = {
	{
		.core_reset_reg = 0x09601000,
		.core_halt_reg = 0x09601004,
		.start_vector_sel_reg = 0x0960A068,
		.alt_vector_addr_reg = 0x0960A06C,

		.parts = tops_cluster_0_parts,
		.num_parts = ARRAY_SIZE(tops_cluster_0_parts),
	},
	{
		.core_reset_reg = 0x09602000,
		.core_halt_reg = 0x09602004,
		.start_vector_sel_reg = 0x0960A074,
		.alt_vector_addr_reg = 0x0960A078,

		.parts = tops_cluster_1_parts,
		.num_parts = ARRAY_SIZE(tops_cluster_1_parts),
	},
	{
		.core_reset_reg = 0x09603000,
		.core_halt_reg = 0x09603004,
		.start_vector_sel_reg = 0x0960A080,
		.alt_vector_addr_reg = 0x0960A084,

		.parts = tops_cluster_2_parts,
		.num_parts = ARRAY_SIZE(tops_cluster_2_parts),
	},
	{
		.core_reset_reg = 0x09604000,
		.core_halt_reg = 0x09604004,
		.start_vector_sel_reg = 0x0960A08C,
		.alt_vector_addr_reg = 0x0960A090,

		.parts = tops_cluster_3_parts,
		.num_parts = ARRAY_SIZE(tops_cluster_3_parts),
	},
};

static const struct tops_role tops_roles[] = {
	{
		.type = TOPS_ROLE_MGMT,

		.parts = tops_parts,
		.num_parts = ARRAY_SIZE(tops_parts),

		.cores = &tops_mgmt_core,
		.num_cores = 1,

		.l2sram_addr = 0x09200000,

		.default_reset_vector_base = 0x40020000,
	},
	{
		.type = TOPS_ROLE_CLUSTER,

		.parts = tops_parts,
		.num_parts = ARRAY_SIZE(tops_parts),

		.cores = tops_cluster_cores,
		.num_cores = ARRAY_SIZE(tops_cluster_cores),

		.l2sram_addr = 0x09800000,

		.default_reset_vector_base = 0x40020000,
	},
};

static const uint16_t tops_fw_plat_ids[] = { TOPS_PLAT_MT7988 };
static const uint8_t tops_fw_role_ids[] = { TOPS_ROLE_MGMT, TOPS_ROLE_CLUSTER };
static const uint8_t tops_fw_part_types[] = {
	TOPS_PT_IRAM0,
	TOPS_PT_DRAM0,
	TOPS_PT_L2SRAM,
};

#ifdef FWDL_DEBUG
#define TOPS_PWR_CON_0			0x11D10040
#define TOPS_PWR_CON_1			0x11D10044
#define TOPS_SRAM_PDN			BIT(8)
#define TOPS_SRAM_CKISO			BIT(5)
#define TOPS_PWR_ISO			BIT(4)
#define TOPS_PWR_CLK_DIS		BIT(3)
#define TOPS_PWR_ON_2ND			BIT(2)
#define TOPS_PWR_ON			BIT(1)
#define TOPS_PWR_RST_N			BIT(0)

#define TOPS_PWR_CON_2			0x11D10048
#define TOPS_C0_SRAM_PDN		BIT(6)
#define TOPS_C0_SRAM_CKISO		BIT(4)
#define TOPS_TOP_SRAM_PDN		BIT(2)
#define TOPS_TOP_SRAM_CKISO		BIT(0)

#define TOP_COREM_OCD_CTRL		0x09101018
#define OCDHaltOnReset			BIT(0)

#define TOP_CORE_DBG_CTRL		0x0910A064

#define TOPS_CLUST0_CORE0_OCD_CTRL	0x09601018
#define TOPS_CLUST0_CORE1_OCD_CTRL	0x09602018
#define TOPS_CLUST0_CORE2_OCD_CTRL	0x09603018
#define TOPS_CLUST0_CORE3_OCD_CTRL	0x09604018
#define CLUST_CORE_DBG_CTRL		0x0960A064

static void tops_power_up_one(uintptr_t con_reg, uint32_t sram_pdn_mask,
			      uint32_t sram_ckiso_mask)
{
	mmio_setbits_32(con_reg, TOPS_PWR_ON);
	mdelay(1);

	mmio_setbits_32(con_reg, TOPS_PWR_ON_2ND);
	mdelay(1);

	mmio_clrbits_32(con_reg, TOPS_PWR_ISO);
	mdelay(1);

	mmio_setbits_32(con_reg, TOPS_PWR_RST_N);
	mdelay(1);

	mmio_clrbits_32(TOPS_PWR_CON_2, sram_pdn_mask);
	mdelay(1);

	mmio_clrbits_32(TOPS_PWR_CON_2, sram_ckiso_mask);
	mdelay(1);

	mmio_clrbits_32(con_reg, TOPS_SRAM_PDN);
	mdelay(1);

	mmio_clrbits_32(con_reg, TOPS_SRAM_CKISO);
	mdelay(1);

	mmio_clrbits_32(con_reg, TOPS_PWR_CLK_DIS);
	mdelay(1);
}

static void tops_power_up(void)
{
	static bool powered = false;

	if (powered)
		return;

	tops_power_up_one(TOPS_PWR_CON_0, TOPS_TOP_SRAM_PDN, TOPS_TOP_SRAM_CKISO);
	tops_power_up_one(TOPS_PWR_CON_1, TOPS_C0_SRAM_PDN, TOPS_C0_SRAM_CKISO);

	powered = true;
}

static void tops_enable_uart(void)
{
	mmio_clrsetbits_32(0x1001F330, 0xffff0000, 0x44440000);
}

static void tops_enable_debug(void)
{
	mmio_clrsetbits_32(0x1001F370, 0x0fffff00, 0x04444400);

	mmio_write_32(TOP_CORE_DBG_CTRL, 0x3f);
	mmio_write_32(CLUST_CORE_DBG_CTRL, 0x1f);

#if 0
	mmio_setbits_32(TOP_COREM_OCD_CTRL, OCDHaltOnReset);
	mmio_setbits_32(TOPS_CLUST0_CORE0_OCD_CTRL, OCDHaltOnReset);
	mmio_setbits_32(TOPS_CLUST0_CORE1_OCD_CTRL, OCDHaltOnReset);
	mmio_setbits_32(TOPS_CLUST0_CORE2_OCD_CTRL, OCDHaltOnReset);
	mmio_setbits_32(TOPS_CLUST0_CORE3_OCD_CTRL, OCDHaltOnReset);
#endif
}
#endif /* FWDL_DEBUG */

static void tops_core_prepare(const struct tops_core *core, uint32_t reset_addr)
{
	/* De-assert core reset */
	mmio_write_32(core->core_reset_reg, 0);

	/* Halt core */
	mmio_write_32(core->core_halt_reg, 1);

	/* Set core reset base */
	mmio_write_32(core->start_vector_sel_reg, 1);
	mmio_write_32(core->alt_vector_addr_reg, reset_addr);

	/* Reset core */
	mmio_write_32(core->core_reset_reg, 1);
	mmio_write_32(core->core_reset_reg, 0);
}

static void tops_core_release(const struct tops_core *core)
{
	/* Release core */
	mmio_write_32(core->core_halt_reg, 0);
}

static void tops_load_data(uintptr_t addr, const void *data, size_t size)
{
	const uint32_t *buf = data;

	size = (size + sizeof(uint32_t) - 1) & ~(sizeof(uint32_t) - 1);

	while (size) {
		mmio_write_32(addr, *buf);
		size -= sizeof(uint32_t);
		addr += sizeof(uint32_t);
		buf++;
	}
}

static const struct tops_loadable_part *tops_find_loadable_part(
					const struct tops_loadable_part *parts,
					uint32_t nparts,
					enum tops_part_type type)
{
	uint32_t i;

	for (i = 0; i < nparts; i++) {
		if (parts[i].type == type)
			return &parts[i];
	}

	return NULL;
}

static const struct tops_part *tops_find_part(const struct tops_part *parts,
					      uint32_t nparts,
					      enum tops_part_type type)
{
	uint32_t i;

	for (i = 0; i < nparts; i++) {
		if (parts[i].type == type)
			return &parts[i];
	}

	return NULL;
}

static int tops_fw_get_reset_vector_addr(struct fw_image_register *fwir,
					 const struct tops_role *role,
					 uint32_t *addr)
{
	bool boot_part_override = false;
	const struct tops_part *part;
	uint32_t boot_addr = 0;
	uint32_t i;

	for (i = 0; i < fwir->num_parts; i++) {
		if (!(fwir->part_data[i].flags & PART_FL_BOOT_PART_OVERRIDE))
			continue;

		FWDL_N("TOPS: Firmware defined boot part override\n");

		if (boot_part_override) {
			FWDL_E("TOPS: Multiple bootable parts defined\n");
			return -ERANGE;
		}

		if (fwir->part_data[i].flags & PART_FL_LOAD_ADDR_OVERRIDE) {
			boot_addr = fwir->part_data[i].value[0];
			FWDL_N("TOPS: Boot addr overrided with load address 0x%08x\n", boot_addr);
		} else {
			part = tops_find_part(role->parts, role->num_parts,
					      tops_fw_part_types[i]);
			if (!part) {
				FWDL_E("TOPS: Invalid bootable part\n");
				return -ENOENT;
			}

			boot_addr = part->base;
			FWDL_N("TOPS: Boot addr overrided with part base 0x%08x\n", boot_addr);
		}

		if (!boot_addr) {
			FWDL_E("TOPS: Invalid bootable part\n");
			return -EINVAL;
		}

		boot_part_override = true;
	}

	if (!boot_part_override) {
		boot_addr = role->default_reset_vector_base;
		FWDL_N("TOPS: Boot addr default 0x%08x\n", boot_addr);
	}

	if (addr)
		*addr = boot_addr;

	return 0;
}

static enum fwdl_status tops_do_fw_load(struct fw_image_register *fwir,
					uint16_t plat_id, uint8_t role_id,
					uint64_t flags)
{
	const struct tops_loadable_part *bpart;
	uint32_t i, c, boot_addr, load_addr;
	const struct tops_role *role = NULL;
	const struct tops_part *part;
	int ret;

	for (i = 0; i < ARRAY_SIZE(tops_roles); i++) {
		if (tops_roles[i].type == role_id) {
			role = &tops_roles[i];
			break;
		}
	}

	if (!role) {
		FWDL_E("TOPS: Unregistered firmware role\n");
		return FWDL_STATUS_INVALID_FW_ROLE;
	}

#ifdef FWDL_DEBUG
	if (flags == BIT(63)) {
		FWDL_N("TOPS: Power up\n");
		tops_power_up();
		tops_enable_uart();
		tops_enable_debug();
	}
#endif

	FWDL_N("TOPS: Loading fw for role %u\n", role->type);
	FWDL_N("TOPS:     %u core(s)\n", role->num_cores);
	FWDL_N("TOPS:     %u parts\n", role->num_parts);

	for (i = 0; i < fwir->num_parts; i++) {
		part = tops_find_part(role->parts, role->num_parts,
				      tops_fw_part_types[i]);
		if (!part) {
			FWDL_E("TOPS: Invalid part type\n");
			return FWDL_STATUS_INVALID_PART_TYPE;
		}

		if (fwir->part_data[i].size > part->size) {
			FWDL_E("TOPS: Data size of part %u is too large\n",
			       part->type);
			return FWDL_STATUS_FW_PART_TOO_LARGE;
		}
	}

	ret = tops_fw_get_reset_vector_addr(fwir, role, &boot_addr);
	if (ret)
		return FWDL_STATUS_INVALID_BOOT_PART;

	for (c = 0; c < role->num_cores; c++) {
		FWDL_N("TOPS: Reset and halt core %u\n", c);
		tops_core_prepare(&role->cores[c], boot_addr);
	}

	for (i = 0; i < fwir->num_parts; i++) {
		if (!fwir->part_data[i].data) {
			FWDL_N("TOPS: Skipping unused part %u\n",
			       tops_fw_part_types[i]);
			continue;
		}

		if (!fwir->part_data[i].size) {
			FWDL_N("TOPS: Skipping empty-data part %u\n",
			       tops_fw_part_types[i]);
			continue;
		}

		switch (tops_fw_part_types[i]) {
		case TOPS_PT_L2SRAM:
			tops_load_data(role->l2sram_addr,
				       fwir->part_data[i].data,
				       fwir->part_data[i].size);

			FWDL_N("TOPS: Loaded L2SRAM at 0x%08zx\n",
			       role->l2sram_addr);

			break;

		case TOPS_PT_IRAM0:
		case TOPS_PT_DRAM0:
			for (c = 0; c < role->num_cores; c++) {
				const struct tops_core *core;

				core = &role->cores[c];

				if (fwir->part_data[i].flags & PART_FL_LOAD_ADDR_OVERRIDE) {
					load_addr = fwir->part_data[i].value[0];
				} else {
					bpart = tops_find_loadable_part(core->parts,
						core->num_parts, tops_fw_part_types[i]);

					if (!bpart) {
						FWDL_E("TOPS: Internal error\n");
						return FWDL_STATUS_INTERNAL_ERROR;
					}

					load_addr = bpart->addr;
				}

				tops_load_data(load_addr,
					       fwir->part_data[i].data,
					       fwir->part_data[i].size);

				FWDL_N("TOPS: Load core %u part %u at 0x%08x\n",
				       c, i, load_addr);
			}
		}
	}

	for (c = 0; c < role->num_cores; c++) {
		FWDL_N("TOPS: Release core %u\n", c);
		tops_core_release(&role->cores[c]);
	}

	return FWDL_STATUS_OK;
}

static struct fw_image_part_data tops_part_data[ARRAY_SIZE(tops_fw_part_types)];

struct fw_image_register tops_fwimg_reg = {
	.magic = TOPS_HDR_MAGIC,

#ifdef TOPS_KEY_HASH_VERIFY
	.keyhash = tops_keyhash,
#endif

	.plat_ids = tops_fw_plat_ids,
	.num_plats = ARRAY_SIZE(tops_fw_plat_ids),

	.role_ids = tops_fw_role_ids,
	.num_roles = ARRAY_SIZE(tops_fw_role_ids),

	.required_part_types = tops_fw_part_types,
	.num_parts = ARRAY_SIZE(tops_fw_part_types),

	.part_data = tops_part_data,
	.do_fw_load = tops_do_fw_load,
};
