// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2021 MediaTek Inc. All Rights Reserved.
 *
 * Author: Weijie Gao <weijie.gao@mediatek.com>
 *
 * Generic image boot helper
 */

#include <errno.h>
#include <image.h>
#include <malloc.h>
#include <linux/types.h>
#include <linux/sizes.h>
#include <linux/ctype.h>
#include <linux/string.h>

#include "upgrade_helper.h"
#include "image_helper.h"
#include "boot_helper.h"
#include "dual_boot.h"
#include "dm_parser.h"

#define ARGLIST_INCR				20

static struct arg_list bootargs = { .name = "bootargs" };
static struct arg_list fdtargs = { .name = "fdtargs" };

int boot_from_mem(ulong data_load_addr)
{
	const char *bootconf, *stock_bootconf, *bootconf_extra;
	bool use_stock_bootconf = false;
	size_t bootconf_base_len;
	char cmd[512], *p;

	bootconf = env_get("bootconf");
	if (!bootconf)
		bootconf = CONFIG_MTK_DEFAULT_FIT_BOOT_CONF;

	bootconf_extra = env_get("bootconf_extra");

	p = strchr(bootconf, '#');
	if (p)
		bootconf_base_len = p - bootconf;
	else
		bootconf_base_len = strlen(bootconf);

	snprintf(cmd, sizeof(cmd), "%.*s", (int)bootconf_base_len, bootconf);

	if (!fit_image_conf_exists((void *)data_load_addr, cmd))
		bootconf = "";

	if (!bootconf[0]) {
		if (bootconf_extra) {
			snprintf(cmd, sizeof(cmd), "bootm 0x%lx#%s",
				 data_load_addr, bootconf_extra);
		} else {
			snprintf(cmd, sizeof(cmd), "bootm 0x%lx",
				 data_load_addr);
		}
	} else {
		stock_bootconf = fit_image_conf_def((void *)data_load_addr);
		if (stock_bootconf) {
			if (bootconf_base_len != strlen(stock_bootconf) ||
			    strncmp(stock_bootconf, bootconf, bootconf_base_len))
				use_stock_bootconf = true;
		}

		if (use_stock_bootconf) {
			if (bootconf_extra) {
				snprintf(cmd, sizeof(cmd),
					 "bootm 0x%lx#%s#%s#%s", data_load_addr,
					 stock_bootconf, bootconf,
					 bootconf_extra);
			} else {
				snprintf(cmd, sizeof(cmd), "bootm 0x%lx#%s#%s",
					 data_load_addr, stock_bootconf,
					 bootconf);
			}
		} else {
			if (bootconf_extra) {
				snprintf(cmd, sizeof(cmd), "bootm 0x%lx#%s#%s",
					 data_load_addr, bootconf,
					 bootconf_extra);
			} else {
				snprintf(cmd, sizeof(cmd), "bootm 0x%lx#%s",
					 data_load_addr, bootconf);
			}
		}
	}

	strtoul(cmd + 6, &p, 16);
	if (p[0] == '#')
		printf("bootconf: %s\n", p + 1);

	return run_command(cmd, 0);
}

const char *get_arg_next(const char *args, const char **param,
			 size_t *keylen)
{
	unsigned int i, equals = 0;
	int in_quote = 0;

	args = skip_spaces(args);
	if (!*args)
		return NULL;

	if (*args == '"') {
		args++;
		in_quote = 1;
	}

	for (i = 0; args[i]; i++) {
		if (isspace(args[i]) && !in_quote)
			break;

		if (equals == 0) {
			if (args[i] == '=')
				equals = i;
		}

		if (args[i] == '"')
			in_quote = !in_quote;
	}

	*param = args;

	if (equals)
		*keylen = equals;
	else
		*keylen = i;

	return args + i;
}

void list_all_args(const char *args)
{
	const char *n = args, *p;
	size_t len, keylen;

	while (1) {
		n = get_arg_next(n, &p, &keylen);
		if (!n)
			break;

		len = n - p;

		printf("[%zu], <%.*s> \"%.*s\"\n", len, (u32)keylen, p,
		       (u32)len, p);
	}
}

static bool compare_key(const char *key, size_t keylen, const char *str)
{
	size_t ckeylen;

	ckeylen = strlen(str);
	if (ckeylen == keylen) {
		if (!strncmp(str, key, keylen))
			return true;
	}

	return false;
}

static bool bootargs_find(const struct arg_pair *bootargs, u32 count,
			  const char *key, size_t keylen)
{
	u32 i;

	for (i = 0; i < count; i++) {
		if (bootargs[i].key) {
			if (compare_key(key, keylen, bootargs[i].key))
				return true;
		}
	}

	return false;
}

static int arg_list_expand(struct arg_list *arglist)
{
	struct arg_pair *newptr;

	if (!arglist->ap) {
		newptr = malloc(ARGLIST_INCR * sizeof(*arglist->ap));
		arglist->used = 0;
		arglist->max = 0;
	} else {
		newptr = realloc(arglist->ap,
				 (arglist->max + ARGLIST_INCR) *
				 sizeof(*arglist->ap));
	}

	if (!newptr)
		return -ENOMEM;

	arglist->ap = newptr;
	arglist->max += ARGLIST_INCR;

	return 0;
}

static int arg_list_set(struct arg_list *arglist, const char *key,
			const char *value)
{
	if (arglist->used == arglist->max) {
		if (arg_list_expand(arglist)) {
			panic("Error: No space for %s\n", arglist->name);
			return -1;
		}
	}

	arglist->ap[arglist->used].key = key;
	arglist->ap[arglist->used].value = value;
	arglist->used++;

	return 0;
}

static void arg_list_remove(struct arg_list *arglist, const char *key)
{
	u32 i;

	for (i = 0; i < arglist->used; i++) {
		if (strcmp(arglist->ap[i].key, key))
			continue;

		if (i < arglist->used - 1) {
			memmove(&arglist->ap[i], &arglist->ap[i + 1],
				(arglist->used - i - 1) * sizeof(*arglist->ap));
		}

		arglist->used--;
		break;
	}
}

void bootargs_reset(void)
{
	bootargs.used = 0;
}

int bootargs_set(const char *key, const char *value)
{
	return arg_list_set(&bootargs, key, value);
}

void bootargs_unset(const char *key)
{
	arg_list_remove(&bootargs, key);
}

void fdtargs_reset(void)
{
	fdtargs.used = 0;
}

int fdtargs_set(const char *prop, const char *value)
{
	return arg_list_set(&fdtargs, prop, value);
}

void fdtargs_unset(const char *prop)
{
	arg_list_remove(&fdtargs, prop);
}

#ifdef CONFIG_MTK_SECURE_BOOT
static char *strconcat(char *dst, const char *src)
{
	while (*src)
		*dst++ = *src++;

	return dst;
}

static char *strnconcat(char *dst, const char *src, size_t n)
{
	while (n-- && *src)
		*dst++ = *src++;

	return dst;
}
#endif /* CONFIG_MTK_SECURE_BOOT */

static int cmdline_merge(const char *cmdline, const struct arg_pair *bootargs,
			 u32 count, char **result)
{
	size_t cmdline_len = 0, newlen = 0, plen, keylen;
	const char *n = cmdline, *p;
	char *buff, *b;
	bool matched;
	u32 i;

#ifdef CONFIG_MTK_SECURE_BOOT
	const char *rootdev = NULL;
	struct dm_verity_info dvi;
	char *dm_mod = NULL;
	int ret;
#endif

	if (count && !bootargs)
		return -EINVAL;

	for (i = 0; i < count; i++) {
		if (bootargs[i].key) {
			newlen += strlen(bootargs[i].key);
			if (bootargs[i].value)
				newlen += strlen(bootargs[i].value) + 1;
			newlen++;

#ifdef CONFIG_MTK_SECURE_BOOT
			if (!strcmp(bootargs[i].key, "root") && bootargs[i].value) {
				debug("%s: new rootdev '%s' will be used for dm-verity arg replacing\n",
				       __func__, bootargs[i].value);
				rootdev = bootargs[i].value;
			}
#endif
		}
	}

	if (!newlen)
		return 0;

	if (cmdline)
		cmdline_len = strlen(cmdline);

#ifdef CONFIG_MTK_SECURE_BOOT
	if (rootdev)
		newlen += 2 * strlen(rootdev);
#endif

	buff = malloc(cmdline_len + newlen + 1);
	if (!buff) {
		printf("No memory for new cmdline\n");
		return -ENOMEM;
	}

	b = buff;

	if (cmdline_len) {
		while (1) {
			n = get_arg_next(n, &p, &keylen);
			if (!n)
				break;

			plen = n - p;

			matched = bootargs_find(bootargs, count, p, keylen);

#ifdef CONFIG_MTK_SECURE_BOOT
			if (compare_key(p, keylen, "dm-mod.create")) {
				debug("%s: found dm-mod.create in bootargs, copy and remove it\n",
				       __func__);
				if (p[keylen + 1] == '\"')
					dm_mod = strndup(p + keylen + 2, plen - keylen - 3);
				else
					dm_mod = strndup(p + keylen + 1, plen - keylen - 1);
				matched = true;
			}

			if (compare_key(p, keylen, "root")) {
				debug("%s: found root in bootargs, keep it\n",
				       __func__);
				matched = false;
			}
#endif

			if (matched) {
				debug("%s: matched key '%.*s', removing ...\n",
				      __func__, (u32)keylen, p);
			} else {
				memcpy(b, p, plen);
				b += plen;
				*b++ = ' ';
			}
		}
	}

#ifdef CONFIG_MTK_SECURE_BOOT
	if (!dm_mod) {
		panic("Error: dm-mod.create not found in original bootargs!\n");
		return -EINVAL;
	}
#endif

	for (i = 0; i < count; i++) {
		if (bootargs[i].key) {
#ifdef CONFIG_MTK_SECURE_BOOT
			if (dm_mod && rootdev && !strcmp(bootargs[i].key, "root")) {
				debug("%s: skipping adding new root parameter\n",
				       __func__);
				continue;
			}
#endif
			keylen = strlen(bootargs[i].key);
			memcpy(b, bootargs[i].key, keylen);
			b += keylen;

			if (bootargs[i].value) {
				*b++ = '=';

				plen = strlen(bootargs[i].value);
				memcpy(b, bootargs[i].value, plen);
				b += plen;
			}

			*b++ = ' ';
		}
	}

#ifdef CONFIG_MTK_SECURE_BOOT
	if (dm_mod && rootdev) {
		ret = dm_mod_create_arg_parse(dm_mod, "verity", (struct dm_info *)&dvi);
		if (ret < 0) {
			panic("Failed to parse dm-mod.create\n");
			return ret;
		}

		b = strconcat(b, "dm-mod.create=\"");
		b = strnconcat(b, dm_mod, dvi.datadev_pos);
		b = strconcat(b, rootdev);
		b = strnconcat(b, dm_mod + dvi.datadev_pos + dvi.datadev_len,
			       dvi.hashdev_pos - dvi.datadev_pos - dvi.datadev_len);

		if (!strcmp(dvi.datadev, dvi.hashdev))
			b = strconcat(b, rootdev);
		else
			b = strconcat(b, dvi.hashdev);

		b = strconcat(b, dm_mod + dvi.hashdev_pos + dvi.hashdev_len);
		b = strconcat(b, "\" "); /* trailing space is needed */
	}
#endif

	if (b > buff)
		b[-1] = 0;

	*result = buff;
	return 1;
}

static int fdt_root_prop_merge(void *fdt)
{
	int ret, np, len;
	u32 i;

	for (i = 0; i < fdtargs.used; i++) {
		np = fdt_path_offset(fdt, "/");
		if (np < 0)
			return -ENOENT;

		if (fdtargs.ap[i].value)
			len = strlen(fdtargs.ap[i].value) + 1;
		else
			len = 0;

		ret = fdt_setprop(fdt, np, fdtargs.ap[i].key,
				  fdtargs.ap[i].value, len);
		if (ret < 0) {
			if (len) {
				printf("Failed to set prop '%s = \"%s\"' in FDT",
				       fdtargs.ap[i].key, fdtargs.ap[i].value);
			} else {
				printf("Failed to set prop '%s' in FDT",
				       fdtargs.ap[i].key);
			}

			return -1;
		}
	}

	return 0;
}

void board_prep_linux(struct bootm_headers *images)
{
	void *fdt = images->ft_addr;
	const char *orig_bootargs;
	int nodeoffset, len;
	char *new_cmdline;
	u32 newlen;
	int ret;

	if (!(CONFIG_IS_ENABLED(OF_LIBFDT) && images->ft_len)) {
		printf("Warning: no FDT present for current image!\n");
		return;
	}

	if (IS_ENABLED(CONFIG_MTK_DUAL_BOOT))
		dual_boot_set_defaults(fdt);

	if (!bootargs.used && !fdtargs.used)
		return;

	fdt_increase_size(fdt, 0x1000);

	if (bootargs.used) {
		/* find or create "/chosen" node. */
		nodeoffset = fdt_find_or_add_subnode(fdt, 0, "chosen");
		if (nodeoffset < 0)
			return;

		orig_bootargs = fdt_getprop(fdt, nodeoffset, "bootargs", &len);

		debug("%s: orig cmdline = \"%s\"\n", __func__, orig_bootargs);

		/* setup bootargs */
		ret = cmdline_merge(orig_bootargs, bootargs.ap, bootargs.used,
				    &new_cmdline);
		if (!ret) {
			panic("Error: failed to generate new kernel cmdline\n");
			return;
		}

		debug("%s: new cmdline = \"%s\"\n", __func__, new_cmdline);

		newlen = strlen(new_cmdline) + 1;

		ret = fdt_setprop(fdt, nodeoffset, "bootargs", new_cmdline,
				  newlen);
		if (ret < 0) {
			panic("Error: failed to set new kernel cmdline\n");
			return;
		}
	}

	ret = fdt_root_prop_merge(fdt);
	if (ret) {
		panic("Error: failed to set FDT root props\n");
		return;
	}

	fdt_shrink_to_minimum(fdt, 0);
}
