
#ifndef _NETIF_H_
#define _NETIF_H_

#include <asm/byteorder.h>

#ifndef NULL
#define NULL 0
#endif
/*
 * Minimum ethernet packet length.
 */
#define ETH_MIN_PKTLEN  60
#define ETH_MAX_PKTLEN  (1540-14)
#define ETH_HDR_SIZE    14

typedef unsigned char enet_addr_t[6];
typedef unsigned char ip_addr_t[4];

typedef unsigned char  octet;
typedef unsigned short word;
typedef unsigned int   dword;



/*
 * Ethernet header
 */
typedef struct {
	enet_addr_t destination;
	enet_addr_t source;
	word        type;
#define ETH_TYPE_IP  	0x0800
#define ETH_TYPE_ARP 	0x0806
#define ETH_TYPE_RARP 	0x8053
} eth_header_t;


/*
 * A IP<->ethernet address mapping.
 */
typedef struct {
	ip_addr_t    ip_addr;
	enet_addr_t  enet_addr;
} ip_route_t;

/*
 *  UDP header
 */
typedef struct {
	word src_port;
	word dest_port;
	word length;
	word checksum;
} udp_header_t;

/*
 * TCP header.
 */
typedef struct {
	word	src_port;
	word	dest_port;
	dword	seqnum;
	dword	acknum;
#if defined(__LITTLE_ENDIAN)
	octet       reserved:4,
	            hdr_len:4;
#elif defined(__BIG_ENDIAN)
	octet       hdr_len:4,
	            reserved:4;
#else
#error invalid endian
#endif
	octet	flags;
#define TCP_FLAG_FIN  1
#define TCP_FLAG_SYN  2
#define TCP_FLAG_RST  4
#define TCP_FLAG_PSH  8
#define TCP_FLAG_ACK 16
#define TCP_FLAG_URG 32
	word	window;
	word	checksum;
	word	urgent;
} tcp_header_t;

/*
 * Internet Protocol header.
 */
typedef struct
{
#if defined(__LITTLE_ENDIAN)
	octet       hdr_len:4,
	            version:4;
#elif defined(__BIG_ENDIAN)
	octet       version:4,
	            hdr_len:4;
#else
#error invalid endian
#endif
	octet       tos;
	word        length;
	word        ident;
	word        fragment;
	octet       ttl;
	octet       protocol;
#define IP_PROTO_ICMP  1
#define IP_PROTO_TCP   6
#define IP_PROTO_UDP  17
	word        checksum;
	ip_addr_t   source;
	ip_addr_t   destination;
} ip_header_t;


/*
 * Simple timer support structure.
 */
typedef void (*tmr_handler_t)(void *user_data);

/*
 * Timer structure.
 * When expiration time is met or exceeded, the handler is
 * called and the timer struct is removed from the list.
 */
typedef struct _timer {
	struct _timer *next;        /* next timer in list */
	unsigned long delay;	/* expiration time relative to start time */
	unsigned long start;	/* when the timer was set */
	tmr_handler_t handler;      /* user procedure to call when timer 'fires' */
	void          *user_data;   /* user pointer passed to above procedure */
} timer_t;


/*
 * Packet buffer
 */
typedef struct _pktbuf
{
	struct _pktbuf *next;
	union {
		ip_header_t *__iphdr;		/* pointer to IP header */
	} u1;
	union {
		udp_header_t *__udphdr;
		tcp_header_t *__tcphdr;		/* pointer to TCP header */
	} u2;
	word	pkt_bytes;		/* number of data bytes in buf */
	word        bufsize;                /* size of buf */
	word	*buf;
} pktbuf_t;

static inline ip_header_t * pkt_iphdr(pktbuf_t *pkt) {
	return (ip_header_t *)pkt->u1.__iphdr;
}

static inline tcp_header_t * pkt_tcphdr(pktbuf_t *pkt) {
	return (tcp_header_t *)pkt->u2.__tcphdr;
}

static inline udp_header_t * pkt_udphdr(pktbuf_t *pkt) {
	return (udp_header_t *)pkt->u2.__udphdr;
}

/*
 * TCP socket
 */
typedef struct _tcp_socket {
	struct _tcp_socket *next;
	int		       state;       /* connection state */
#define _CLOSED      0
#define _LISTEN      1
#define _SYN_RCVD    2
#define _SYN_SENT    3
#define _ESTABLISHED 4
#define _CLOSE_WAIT  5
#define _LAST_ACK    6
#define _FIN_WAIT_1  7
#define _FIN_WAIT_2  8
#define _CLOSING     9
#define _TIME_WAIT  10
	ip_route_t         his_addr;    /* address of other end of connection */
	word               our_port;
	word               his_port;
	word               data_bytes;   /* number of data bytes in pkt */
	char               reuse;        /* SO_REUSEADDR, no 2MSL */
	timer_t            timer;
	pktbuf_t           pkt;         /* dedicated xmit packet */
	pktbuf_t           *rxlist;     /* list of unread incoming data packets */
	char               *rxptr;      /* pointer to next byte to read */
	int                rxcnt;       /* bytes left in current read packet */
	dword              ack;
	dword              seq;
	char               pktbuf[ETH_MAX_PKTLEN];
} tcp_socket_t;

extern unsigned char bRunTftp;

//declare for httpd.c
int httpd_isRunning(void);
void httpd_init(void);
void httpd_accept(int is_idle);

//declare for pktbuf.c
void 		__pktbuf_init(void);
pktbuf_t *  	__pktbuf_alloc(int nbytes);
void 		__pktbuf_free(pktbuf_t *pkt);

//declare for cksum.c
unsigned short __sum(word *w,int len,int init_sum);
int __pseudo_sum(ip_header_t *ip);

//declare for timers.c
void __timer_set(timer_t *t, unsigned long delay,tmr_handler_t handler, void *user_data);
void __timer_cancel(timer_t *t);
void __timer_poll(void);

//declare for tcp.c
void	 __tcp_handler(pktbuf_t *pkt,ip_route_t *r);
int 	__ip_send(pktbuf_t *pkt,int protocol,ip_route_t *dest);
int 	__tcp_read(tcp_socket_t *s,char *buf,int len);
void 	__tcp_poll(void);
void	__tcp_close(tcp_socket_t *s);
int	__tcp_write(tcp_socket_t *s,char *buf,int len);
int 	__tcp_listen(tcp_socket_t *s,word port);
void 	__tcp_reset(tcp_socket_t *s);
#endif
